/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "nl-sidebar-item.h"

#include <glib.h>
#include <glib/gi18n.h>
#include <math.h>

#include <gtk/gtk.h>

#include "nl-config.h"
#include "nl-defines.h"
#include "nl-private.h"
#include "nl-texture-frame.h"

G_DEFINE_TYPE (NlSidebarItem, nl_sidebar_item, CTK_TYPE_BUTTON);

#define NL_SIDEBAR_ITEM_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  NL_TYPE_SIDEBAR_ITEM, \
  NlSidebarItemPrivate))

struct _NlSidebarItemPrivate
{
  ClutterActor *normal_bg;
  ClutterActor *normal_prelight;

  ClutterActor *active_bg;

  gint     pos;
  gboolean active;
  gfloat   last_width;
};

enum
{
  PROP_0,

  PROP_POS,
  PROP_ACTIVE
};


/* Globals */
static ClutterActor *prelight_texture = NULL;
static ClutterActor *normal_texture   = NULL;
static ClutterActor *active_texture   = NULL;

/* Forwards */
static void nl_sidebar_item_focus_in  (ClutterActor    *actor);
static void nl_sidebar_item_focus_out (ClutterActor    *actor);

/* GObject stuff */
static void
set_property (GObject      *object,
              guint         prop_id,
              const GValue *value,
              GParamSpec   *pspec)
{
  switch (prop_id)
    {
    case PROP_POS:
      nl_sidebar_item_set_pos (NL_SIDEBAR_ITEM (object),
                               g_value_get_int (value));
      break;

    case PROP_ACTIVE:
      nl_sidebar_item_set_active (NL_SIDEBAR_ITEM (object),
                                  g_value_get_boolean (value));

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
get_property (GObject      *object,
              guint         prop_id,
              GValue       *value,
              GParamSpec   *pspec)
{
  NlSidebarItemPrivate *priv;

  g_return_if_fail (NL_IS_SIDEBAR_ITEM (object));
  priv = NL_SIDEBAR_ITEM_GET_PRIVATE (object);

  switch (prop_id)
    {
    case PROP_POS:
      g_value_set_int (value, priv->pos);
      break;

    case PROP_ACTIVE:
      g_value_set_boolean (value, priv->active);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
nl_sidebar_item_finalize (GObject *object)
{
  NlSidebarItemPrivate *priv;

  priv = NL_SIDEBAR_ITEM_GET_PRIVATE (object);

  if (CLUTTER_IS_ACTOR (priv->normal_bg))
    {
      g_object_unref (priv->normal_bg);
      priv->normal_bg = NULL;
    }
  if (CLUTTER_IS_ACTOR (priv->normal_prelight))
    {
      g_object_unref (priv->normal_prelight);
      priv->normal_prelight = NULL;
    }
  if (CLUTTER_IS_ACTOR (priv->active_bg))
    {
      g_object_unref (priv->active_bg);
      priv->active_bg = NULL;
    }

  G_OBJECT_CLASS (nl_sidebar_item_parent_class)->finalize (object);
}

static void
nl_sidebar_item_get_preferred_width (ClutterActor *actor,
                                     gfloat        for_height,
                                     gfloat       *min_width,
                                     gfloat       *nat_width)
{
  NlSidebarItemPrivate *priv = NL_SIDEBAR_ITEM (actor)->priv;
  ClutterActorClass *klass;

  klass = CLUTTER_ACTOR_CLASS (nl_sidebar_item_parent_class);

  if (priv->active)
    {
      if (min_width)
        *min_width = priv->last_width;
      if (nat_width)
        *nat_width = priv->last_width;
      return;
    }

  klass->get_preferred_width (actor, for_height, min_width, nat_width);

  if (min_width)
    priv->last_width = *min_width;
  if (nat_width)
    priv->last_width = *nat_width;
}

static void
nl_sidebar_item_get_preferred_height (ClutterActor *actor,
                                      gfloat        for_width,
                                      gfloat       *min_height,
                                      gfloat       *nat_height)
{
  if (min_height)
    *min_height = 50.0;
  if (nat_height)
    *nat_height = 50.0;
}

static void
nl_sidebar_item_constructed (GObject *object)
{
  NlSidebarItem        *item = NL_SIDEBAR_ITEM (object);
  NlSidebarItemPrivate *priv = item->priv;
  CtkPadding            padding     = { 1.0, 28.0, 1.0, 12.0 };
  CtkPadding            padding_rtl = { 1.0, 12.0, 1.0, 28.0 };
  GtkTextDirection      direction;

  direction = ctk_actor_get_default_direction ();

  if (G_OBJECT_CLASS (nl_sidebar_item_parent_class)->constructed)
    G_OBJECT_CLASS (nl_sidebar_item_parent_class)->constructed (object);

  if (direction == GTK_TEXT_DIR_RTL)
    ctk_actor_set_padding (CTK_ACTOR (object), &padding_rtl);
  else
    ctk_actor_set_padding (CTK_ACTOR (object), &padding);

  if (!CLUTTER_IS_ACTOR (normal_texture))
    {
      if (direction == GTK_TEXT_DIR_RTL)
        normal_texture = clutter_texture_new_from_file
                         (PKGDATADIR"/sidebar_item_normal_rtl.png", NULL);
      else
        normal_texture = clutter_texture_new_from_file
                         (PKGDATADIR"/sidebar_item_normal.png", NULL);
    }
  if (!CLUTTER_IS_ACTOR (prelight_texture))
    {
      if (direction == GTK_TEXT_DIR_RTL)
        prelight_texture = clutter_texture_new_from_file
                           (PKGDATADIR"/sidebar_item_prelight_rtl.png", NULL);
      else
        prelight_texture = clutter_texture_new_from_file
                           (PKGDATADIR"/sidebar_item_prelight.png", NULL);
    }
  if (!CLUTTER_IS_ACTOR (active_texture))
    {
      if (direction == GTK_TEXT_DIR_RTL)
        active_texture = clutter_texture_new_from_file
                         (PKGDATADIR"/sidebar_item_active_rtl.png", NULL);
      else
        active_texture = clutter_texture_new_from_file
                         (PKGDATADIR"/sidebar_item_active.png", NULL);
    }

  gfloat top = 15, right = 30, bottom = 15, left = 15;
  if (direction == GTK_TEXT_DIR_RTL)
    {
      right = 15; left = 30;
    }
  priv->normal_bg = nl_texture_frame_new (CLUTTER_TEXTURE (normal_texture),
                                          top, right, bottom, left);
  g_object_ref_sink (priv->normal_bg);

  priv->normal_prelight = nl_texture_frame_new
                          (CLUTTER_TEXTURE (prelight_texture),
                           top, right, bottom, left);
  g_object_ref_sink (priv->normal_prelight);

  priv->active_bg = nl_texture_frame_new (CLUTTER_TEXTURE (active_texture),
                                          top, right, bottom, left);
  g_object_ref_sink (priv->active_bg);

  priv->active = TRUE;
  nl_sidebar_item_set_active (NL_SIDEBAR_ITEM (object), FALSE);
}

static void
nl_sidebar_item_class_init (NlSidebarItemClass *klass)
{
  GObjectClass      *obj_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *act_class = CLUTTER_ACTOR_CLASS (klass);
  GParamSpec        *pspec;

  obj_class->finalize            = nl_sidebar_item_finalize;
  obj_class->set_property        = set_property;
  obj_class->get_property        = get_property;
  obj_class->constructed         = nl_sidebar_item_constructed;

  act_class->get_preferred_width  = nl_sidebar_item_get_preferred_width;
  act_class->get_preferred_height = nl_sidebar_item_get_preferred_height;
  act_class->key_focus_in         = nl_sidebar_item_focus_in;
  act_class->key_focus_out        = nl_sidebar_item_focus_out;

  /* Install properties */
  pspec = g_param_spec_int ("position", "position", "position",
                            G_MININT, G_MAXINT, 100, NL_PARAM_READWRITE);
  g_object_class_install_property (obj_class, PROP_POS, pspec);

  pspec = g_param_spec_boolean ("active", "active", "active",
                                FALSE, NL_PARAM_READWRITE);
  g_object_class_install_property (obj_class, PROP_ACTIVE, pspec);

  g_type_class_add_private (obj_class, sizeof (NlSidebarItemPrivate));
}

static void
nl_sidebar_item_init (NlSidebarItem *item)
{
  NlSidebarItemPrivate *priv;

  priv = item->priv = NL_SIDEBAR_ITEM_GET_PRIVATE (item);
}


/*
 * Private methods
 */

static void
nl_sidebar_item_focus_in  (ClutterActor    *actor)
{
  CLUTTER_ACTOR_CLASS (nl_sidebar_item_parent_class)->key_focus_in (actor);

  ctk_actor_set_state (CTK_ACTOR (actor), CTK_STATE_PRELIGHT);
}

static void
nl_sidebar_item_focus_out (ClutterActor    *actor)
{
  CLUTTER_ACTOR_CLASS (nl_sidebar_item_parent_class)->key_focus_out (actor);

  ctk_actor_set_state (CTK_ACTOR (actor), CTK_STATE_NORMAL);
}

/*
 * Public methods
 */

ClutterActor *
nl_sidebar_item_new (const gchar *title,
                     const gchar *tooltip,
                     GdkPixbuf   *icon,
                     gint         pos)
{
  CtkImage *image;
  ClutterActor *b;

  b = g_object_new (NL_TYPE_SIDEBAR_ITEM,
                    "orientation", CTK_ORIENTATION_HORIZONTAL,
                    "label", title,
                    "position", pos,
                    NULL);

  image = ctk_button_get_image (CTK_BUTTON (b));

  ctk_image_set_size (image, 44);
  ctk_image_set_from_pixbuf (image, icon);

  return b;
}

gint
nl_sidebar_item_get_pos  (NlSidebarItem *item)
{
  g_return_val_if_fail (NL_IS_SIDEBAR_ITEM (item), 100);
  return item->priv->pos;
}

void
nl_sidebar_item_set_pos  (NlSidebarItem *item,
                          gint                 pos)
{
  g_return_if_fail (NL_IS_SIDEBAR_ITEM (item));
  item->priv->pos = pos;
}

static void
make_bold (ClutterText *label, gboolean bold)
{
  PangoAttrList *list = clutter_text_get_attributes (label);
  PangoAttribute *attr;

  if (!list)
    list = pango_attr_list_new ();

  if (bold)
    attr = pango_attr_weight_new (PANGO_WEIGHT_BOLD);
  else
    attr = pango_attr_weight_new (PANGO_WEIGHT_NORMAL);

  attr->start_index = 0;
  attr->end_index = strlen (clutter_text_get_text (label));

  pango_attr_list_change (list, attr);

  clutter_text_set_attributes (label, list);
}

void
nl_sidebar_item_set_active (NlSidebarItem *item,
                            gboolean       is_active)
{
  NlSidebarItemPrivate *priv;
  CtkText              *text;

  g_return_if_fail (NL_IS_SIDEBAR_ITEM (item));
  priv = item->priv;

  if (priv->active == is_active)
    return;

  priv->active = is_active;

  text = ctk_button_get_text (CTK_BUTTON (item));

  if (is_active)
    {
      clutter_actor_set_opacity (priv->active_bg, 255);
      ctk_actor_set_background (CTK_ACTOR (item), priv->active_bg);
      ctk_actor_set_background_for_state (CTK_ACTOR (item), CTK_STATE_PRELIGHT,
                                          NULL);
      clutter_actor_animate (CLUTTER_ACTOR (item), CLUTTER_EASE_OUT_SINE, 120,
                             "opacity", 255, NULL);
      make_bold (CLUTTER_TEXT (text), TRUE);
    }
  else
    {
      clutter_actor_set_opacity (priv->normal_bg, 255);
      ctk_actor_set_background (CTK_ACTOR (item), priv->normal_bg);
      ctk_actor_set_background_for_state (CTK_ACTOR (item), CTK_STATE_PRELIGHT,
                                          priv->normal_prelight);
      clutter_actor_animate (CLUTTER_ACTOR (item), CLUTTER_EASE_OUT_SINE, 120,
                             "opacity", 255, NULL);
      make_bold (CLUTTER_TEXT (text), FALSE);
    }

  if (CLUTTER_ACTOR_IS_VISIBLE (item))
    clutter_actor_queue_redraw (CLUTTER_ACTOR (item));
}

gboolean
nl_sidebar_item_get_active (NlSidebarItem *item)
{
  g_return_val_if_fail (NL_IS_SIDEBAR_ITEM (item), FALSE);

  return item->priv->active;
}


