/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "nl-menu-items.h"

#include <glib.h>
#include <glib/gi18n.h>
#include <launcher/launcher.h>
#include <netbook-launcher/netbook-launcher.h>
#include <clutk/clutk.h>
#include <clutter-gtk/clutter-gtk.h>
#include <launcher/launcher.h>

G_DEFINE_TYPE (NlMenuItems, nl_menu_items, G_TYPE_OBJECT);

#define NL_MENU_ITEMS_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  NL_TYPE_MENU_ITEMS, \
  NlMenuItemsPrivate))

#define PREFS

struct _NlMenuItemsPrivate
{
  NlShell           *shell;
  NlPixbufCache     *cache;
  LauncherMenu      *menu;
  LauncherFavorites *favorites;

  GSList        *categories;

  GdkPixbuf     *add_pixbuf;
  ClutterActor  *bg_texture;

  NlSidebarItem *last_active_item;
  ClutterActor  *last_active_view;

  LauncherApplication *drag_app;
};

enum
{
  PROP_0,
  PROP_SHELL
};

/* Forwards */
static void on_menu_changed (LauncherMenu *menu, NlMenuItems *items);
static void on_content_view_child_changed (NlContentView *content_view,
                                           ClutterActor  *new_view,
                                           NlMenuItems   *items);
/* GObject stuff */
static void
set_property (GObject      *object,
              guint         prop_id,
              const GValue *value,
              GParamSpec   *pspec)
{
  NlMenuItemsPrivate *priv;

  g_return_if_fail (NL_IS_MENU_ITEMS (object));
  priv = NL_MENU_ITEMS_GET_PRIVATE (object);

  switch (prop_id)
    {
    case PROP_SHELL:
      priv->shell = g_value_get_pointer (value);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
get_property (GObject      *object,
              guint         prop_id,
              GValue       *value,
              GParamSpec   *pspec)
{
  NlMenuItemsPrivate *priv;

  g_return_if_fail (NL_IS_MENU_ITEMS (object));
  priv = NL_MENU_ITEMS_GET_PRIVATE (object);

  switch (prop_id)
    {
    case PROP_SHELL:
      g_value_set_pointer (value, priv->shell);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}


static void
nl_menu_items_finalize (GObject *object)
{
  NlMenuItemsPrivate *priv;

  priv = NL_MENU_ITEMS_GET_PRIVATE (object);

  if (priv->menu)
    {
      g_object_unref (priv->menu);
      priv->menu = NULL;
    }

  if (priv->cache)
    {
      g_object_unref (priv->cache);
      priv->cache = NULL;
    }

  if (priv->favorites)
    {
      g_object_unref (priv->favorites);
      priv->favorites = NULL;
    }

  if (CLUTTER_IS_ACTOR (priv->bg_texture))
    {
      clutter_actor_destroy (priv->bg_texture);
      priv->bg_texture = NULL;
    }

  if (priv->add_pixbuf)
    {
      g_object_unref (priv->add_pixbuf);
      priv->add_pixbuf = NULL;
    }

  G_OBJECT_CLASS (nl_menu_items_parent_class)->finalize (object);
}

static void
nl_menu_items_constructed (GObject *object)
{
  NlMenuItemsPrivate *priv = NL_MENU_ITEMS (object)->priv;

  priv->menu = launcher_menu_get_default ();
  priv->bg_texture = clutter_texture_new_from_file
                     (PKGDATADIR"/iconview_normal.png", NULL);
  g_object_ref_sink (priv->bg_texture);

  g_signal_connect (priv->menu, "menu-changed",
                    G_CALLBACK (on_menu_changed), object);

  on_menu_changed (priv->menu, NL_MENU_ITEMS (object));

  g_signal_connect (nl_shell_get_content_view (priv->shell), "changed",
                    G_CALLBACK (on_content_view_child_changed), object);
}

static void
nl_menu_items_class_init (NlMenuItemsClass *klass)
{
  GObjectClass *obj_class = G_OBJECT_CLASS (klass);
  GParamSpec   *pspec;

  obj_class->finalize     = nl_menu_items_finalize;
  obj_class->constructed  = nl_menu_items_constructed;
  obj_class->set_property = set_property;
  obj_class->get_property = get_property;

  /* Install properties */
  pspec = g_param_spec_pointer ("shell", "shell", "shell",
                                G_PARAM_READWRITE | G_PARAM_CONSTRUCT);
  g_object_class_install_property (obj_class, PROP_SHELL, pspec);

  g_type_class_add_private (obj_class, sizeof (NlMenuItemsPrivate));
}

static void
nl_menu_items_init (NlMenuItems *menu_items)
{
  NlMenuItemsPrivate *priv;

  priv = menu_items->priv = NL_MENU_ITEMS_GET_PRIVATE (menu_items);

  priv->cache = nl_pixbuf_cache_get_default ();
  priv->favorites = launcher_favorites_get_default ();

  priv->add_pixbuf = nl_pixbuf_cache_icon_for_name (priv->cache,
                     PKGDATADIR"/add.png", -1);
}

/*
 * Public methods
 */
NlMenuItems *
nl_menu_items_new (NlShell *shell)

{
  NlMenuItems *menu_items = NULL;


  menu_items = g_object_new (NL_TYPE_MENU_ITEMS,
                             "shell", shell,
                             NULL);

  return menu_items;
}

/*
 * Private methods
 */
static void
on_app_clicked (ClutterActor *actor, gpointer null)
{
  LauncherApplication *app;
  CtkImage            *image;
  GError              *error = NULL;

  app = g_object_get_data (G_OBJECT (actor), "launcher-application");
  g_return_if_fail (app);

  image = ctk_button_get_image (CTK_BUTTON (actor));

  launcher_application_launch (app, &error);

  if (error)
    {
      GtkWidget *dialog;
      gchar     *title;

      dialog = gtk_message_dialog_new (NULL,
                                       0,
                                       GTK_MESSAGE_ERROR,
                                       GTK_BUTTONS_CLOSE,
                                       "%s",
                                       error->message);

      title = g_strdup_printf (_("Unable to open %s"),
                               launcher_application_get_name (app));
      gtk_window_set_title (GTK_WINDOW (dialog), title);
      g_free (title);

      g_signal_connect (dialog, "close", G_CALLBACK (gtk_widget_destroy), NULL);
      g_signal_connect (dialog, "response", G_CALLBACK(gtk_widget_destroy),NULL);

      gtk_widget_show (dialog);

      g_error_free (error);
    }
  else
    {
      ClutterActor *icon;

      icon = ctk_image_new_from_pixbuf (ctk_image_get_size (image),
                                        ctk_image_get_pixbuf (image));
      nl_notify_popup (nl_notify_get_default (),
                       launcher_application_get_name (app),
                       CTK_IMAGE (icon),
                       0);
    }
}


static void
fav_app (GtkMenuItem *item, ClutterActor *actor)
{
  LauncherApplication *app;
  NlMenuItems         *items;
  NlMenuItemsPrivate  *priv;
  gchar               *uid = NULL;
  GSList              *favs, *f;
  gint                 i;

  items = g_object_get_data (G_OBJECT (actor), "nl-menu-items");
  app = g_object_get_data (G_OBJECT (actor), "launcher-application");
  g_return_if_fail (app);
  if (!NL_IS_MENU_ITEMS (items))
    return;

  priv = items->priv;

  /* Get list of favs and try and create a new one with a normal name */
  favs = launcher_favorites_get_favorites (priv->favorites);
  for (i = 0; i < 50; i++)
    {
      gchar *temp_uid = g_strdup_printf ("app-%d", i);
      gboolean existing = FALSE;

      for (f = favs; f; f = f->next)
        {
          gchar *path = launcher_favorites_get_string (priv->favorites, f->data, "desktop_file");

          if (path)
            {
              if (g_strcmp0 (path, launcher_application_get_desktop_file (app)) == 0)
                {
                  g_free (path);
                  return;
                }

              g_free (path);
            }

          if (g_strcmp0 (f->data, temp_uid) == 0)
            {
              existing = TRUE;
              break;
            }
        }
      if (!existing)
        {
          uid = temp_uid;
          break;
        }
      else
        g_free (temp_uid);
    }
  g_slist_foreach (favs, (GFunc)g_free, NULL);
  g_slist_free (favs);
  if (!uid)
    uid = g_strdup_printf ("application-%d", (gint)time (NULL));

  /* Set the properties of the favorite */
  launcher_favorites_set_string (priv->favorites, uid, "type", "application");
  launcher_favorites_set_string (priv->favorites, uid, "desktop_file",
                                 launcher_application_get_desktop_file (app));
  launcher_favorites_add_favorite (priv->favorites, uid);

  nl_shell_add_favorite (priv->shell, uid);

  g_free (uid);
}

static void
show_menu (ClutterActor *actor, guint32 event_time, LauncherApplication *app)
{
  GtkWidget *menu, *item;

  g_return_if_fail (app);

  menu = gtk_menu_new ();

  item = gtk_image_menu_item_new_from_stock (GTK_STOCK_OPEN, NULL);
  gtk_widget_show (item);
  g_signal_connect_swapped (item, "activate",
                            G_CALLBACK (on_app_clicked), actor);
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);

  item = gtk_image_menu_item_new_with_label (_("Add to Favorites"));
  g_object_set (item,
                "image", gtk_image_new_from_stock (GTK_STOCK_ADD,
                                                   GTK_ICON_SIZE_MENU),
                NULL);
  gtk_widget_show (item);
  g_signal_connect (item, "activate", G_CALLBACK (fav_app), actor);
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);

  gtk_menu_popup (GTK_MENU (menu),
                  NULL, NULL,
                  NULL, NULL,
                  3, event_time);
}

static gchar *
drag_data_func (GObject *items)
{
  LauncherApplication *app;
  NlMenuItemsPrivate  *priv;
  gchar               *uid = NULL;
  GSList              *favs, *f;
  gint                 i;
  if (!NL_IS_MENU_ITEMS (items))
    return NULL;
  priv = NL_MENU_ITEMS (items)->priv;
  app = priv->drag_app;
  if (!app)
    return NULL;

  /* Get list of favs and try and create a new one with a normal name */
  favs = launcher_favorites_get_favorites (priv->favorites);
  for (i = 0; i < 50; i++)
    {
      gchar *temp_uid = g_strdup_printf ("app-%d", i);
      gboolean existing = FALSE;

      for (f = favs; f; f = f->next)
        {
          gchar *path = launcher_favorites_get_string (priv->favorites, f->data, "desktop_file");

          if (path)
            {
              if (g_strcmp0 (path, launcher_application_get_desktop_file (app)) == 0)
                {
                  g_free (path);
                  g_free (temp_uid);
                  return NULL;
                }

              g_free (path);
            }

          if (g_strcmp0 (f->data, temp_uid) == 0)
            {
              existing = TRUE;
              break;
            }
        }
      if (!existing)
        {
          uid = temp_uid;
          break;
        }
      else
        g_free (temp_uid);
    }
  g_slist_foreach (favs, (GFunc)g_free, NULL);
  g_slist_free (favs);
  if (!uid)
    uid = g_strdup_printf ("application-%d", (gint)time (NULL));

  /* Set the properties of the favorite */
  launcher_favorites_set_string (priv->favorites, uid, "type", "application");
  launcher_favorites_set_string (priv->favorites, uid, "desktop_file",
                                 launcher_application_get_desktop_file (app));
  launcher_favorites_add_favorite (priv->favorites, uid);

  return uid;
}

static void
on_drag_move (NlIconTile *tile, NlMenuItems *items)
{
  NlMenuItemsPrivate  *priv;
  LauncherApplication *app;
  ClutterActor        *actor, *stage;
  CtkImage            *image;
  gfloat               x=0, y=0;

  g_return_if_fail (NL_IS_MENU_ITEMS (items));
  priv = items->priv;

  app = (LauncherApplication*)g_object_get_data (G_OBJECT (tile),
        "launcher-application");

  priv->drag_app = app;

  clutter_actor_get_transformed_position (CLUTTER_ACTOR (tile), &x, &y);

  stage = nl_shell_get_stage (priv->shell);
  image = ctk_button_get_image (CTK_BUTTON (tile));
  actor = ctk_image_new_from_pixbuf (64, ctk_image_get_pixbuf (image));
  clutter_container_add_actor (CLUTTER_CONTAINER (stage), actor);
  clutter_actor_set_position (actor, x, y);
  clutter_actor_raise_top (actor);
  clutter_actor_show (actor);

  nl_shell_begin_favorite_drag (priv->shell,
                                G_OBJECT (items),
                                actor,
                                drag_data_func,
                                NULL);
}

static void
on_emblem_clicked (NlIconTile *tile)
{
  fav_app (NULL, CLUTTER_ACTOR (tile));
}

static void
on_clicked (NlSidebarItem *item, NlShell *shell)
{
  NlMenuItemsPrivate *priv;
  ClutterActor     *scroll;
  ClutterActor     *bg;
  ClutterActor     *view;
  CtkPadding        vpadding = { 15, 15, 15, 15 };
  LauncherCategory *cat;
  NlMenuItems      *items;
  GSList           *a;

  cat = (LauncherCategory *)g_object_get_data (G_OBJECT (item), "category");
  items = (NlMenuItems *)g_object_get_data (G_OBJECT (item), "nl-menu-items");

  if (!cat || !NL_IS_MENU_ITEMS (items))
    return;
  priv = items->priv;

  nl_sidebar_item_set_active (item, TRUE);

  scroll = nl_scroll_view_new ();

  bg = nl_texture_frame_new (CLUTTER_TEXTURE (priv->bg_texture),
                             25, 25, 25, 25);
  view = ctk_icon_view_new ();
  ctk_actor_set_padding (CTK_ACTOR (view), &vpadding);
  ctk_actor_set_background_for_state (CTK_ACTOR (view), CTK_STATE_NORMAL, bg);
  clutter_container_add_actor (CLUTTER_CONTAINER (scroll), view);
  clutter_actor_show (view);

  for (a = launcher_category_get_children (cat); a; a = a->next)
    {
      LauncherApplication *app = a->data;
      ClutterActor *tile, *emblem;
      GdkPixbuf *pixbuf;

      if (!a)
        continue;

      pixbuf = nl_pixbuf_cache_icon_for_name (items->priv->cache,
                                      launcher_application_get_icon_name (app),
                                              64);
      if (!pixbuf)
        pixbuf = nl_pixbuf_cache_icon_for_name (items->priv->cache,
                                                "application-x-executable",
                                                64);
      tile = nl_icon_tile_new (launcher_application_get_name (app),
                               launcher_application_get_comment (app),
                               pixbuf);
      clutter_container_add_actor (CLUTTER_CONTAINER (view), tile);
      clutter_actor_show (tile);

      emblem = ctk_image_new_from_pixbuf (24, priv->add_pixbuf);
      nl_icon_tile_set_emblem (NL_ICON_TILE (tile),
                               NL_ICON_TILE_EMBLEM_TYPE_HOVER,
                               emblem);


      g_object_set_data (G_OBJECT (tile), "nl-menu-items", items);
      g_object_set_data (G_OBJECT (tile), "launcher-application", app);

      g_signal_connect (tile, "clicked", G_CALLBACK (on_app_clicked), app);
      g_signal_connect (tile, "show-context-menu", G_CALLBACK (show_menu), app);
      g_signal_connect (tile, "begin-drag-move", G_CALLBACK (on_drag_move),
                        items);
      g_signal_connect (tile, "emblem-clicked",
                        G_CALLBACK (on_emblem_clicked), NULL);

      if (pixbuf)
        g_object_unref (pixbuf);
    }

  priv->last_active_item = item;
  priv->last_active_view = scroll;
  nl_content_view_set_child (nl_shell_get_content_view (shell), scroll);
}

static void
on_child_focus_changed (ClutterActor    *icon_view,
                        ClutterActorBox *box,
                        ClutterContainer *main_box)
{
  ClutterActor *parent;
  GList *c, *children;
  gfloat y1 = 0, y2 = 0;
  ClutterActorBox focus_box = { 0 };

  parent = clutter_actor_get_parent (icon_view);

  children = clutter_container_get_children (main_box);
  for (c = children; c; c = c->next)
    {
      ClutterActor *a = c->data;
      
      if (a != parent)
        y1 += clutter_actor_get_height (a);
      else
        break;
    }
  y1 += box->y1 - 100;
  y2 = y1 + (box->y2 - box->y1);
  focus_box.y1 = y1;
  focus_box.y2 = y2;

  g_signal_emit_by_name (main_box, "child-focus-changed", &focus_box);

  g_list_free (children);
}

static void
on_system_clicked (NlSidebarItem *item, NlMenuItems *items)
{
  NlMenuItemsPrivate *priv;
  GSList             *c;
  ClutterActor       *scroll, *main_box;

  g_return_if_fail (NL_IS_MENU_ITEMS (items));
  priv = items->priv;

  nl_sidebar_item_set_active (item, TRUE);

  /* for (n categories that have 'preferences' in the icon)
   *   create icon view
   *     populate with vbox with label and apps
   *   add to main vbox
   * add vbox to current view
   */

  scroll = nl_scroll_view_new ();

  main_box = ctk_vbox_new (FALSE);
  clutter_container_add_actor (CLUTTER_CONTAINER (scroll), main_box);
  clutter_actor_show (main_box);

  for (c = launcher_menu_get_categories (priv->menu); c; c = c->next)
    {
      LauncherCategory *cat = c->data;
      ClutterActor     *vbox, *text, *view, *bg;
      CtkPadding        padding = { 15, 15, 15, 15 };
      gchar            *markup;
      GSList           *a;

      if (!g_strstr_len (launcher_category_get_icon_name (cat),
                         -1,
                         "preferences"))
        continue;

      vbox = ctk_vbox_new (6);
      clutter_container_add_actor (CLUTTER_CONTAINER (main_box), vbox);
      clutter_actor_show (vbox);

      bg = nl_texture_frame_new (CLUTTER_TEXTURE (priv->bg_texture),
                                 25, 25, 25, 25);
      ctk_actor_set_background_for_state (CTK_ACTOR (vbox),
                                          CTK_STATE_NORMAL,
                                          bg);
      ctk_actor_set_padding (CTK_ACTOR (vbox), &padding);

      markup = g_strdup_printf ("<big><b>%s</b></big>",
                                launcher_category_get_name (cat));
      text = ctk_text_new (markup);
      ctk_text_set_alignment (CTK_TEXT (text), PANGO_ALIGN_CENTER);
      ctk_box_pack (CTK_BOX (vbox), text, FALSE,FALSE);
      clutter_actor_show (text);

      view = ctk_icon_view_new ();
      ctk_box_pack (CTK_BOX (vbox), view, FALSE,FALSE);
      clutter_actor_show (view);

      g_signal_connect (view, "child-focus-changed",
                        G_CALLBACK (on_child_focus_changed), main_box);

      /* Add all the applications to the view */
      for (a = launcher_category_get_children (cat); a; a = a->next)
        {
          LauncherApplication *app = a->data;
          ClutterActor *tile, *emblem;
          GdkPixbuf *pixbuf;

          if (!a)
            continue;

          pixbuf = nl_pixbuf_cache_icon_for_name (items->priv->cache,
                                      launcher_application_get_icon_name (app),
                                                 64);
          tile = nl_icon_tile_new (launcher_application_get_name (app),
                                   launcher_application_get_comment (app),
                                   pixbuf);
          clutter_container_add_actor (CLUTTER_CONTAINER (view), tile);
          clutter_actor_show (tile);

          emblem = ctk_image_new_from_pixbuf (24, priv->add_pixbuf);
          nl_icon_tile_set_emblem (NL_ICON_TILE (tile),
                                   NL_ICON_TILE_EMBLEM_TYPE_HOVER,
                                   emblem);


          g_object_set_data (G_OBJECT (tile), "nl-menu-items", items);
          g_object_set_data (G_OBJECT (tile), "launcher-application", app);

          g_signal_connect (tile, "clicked", G_CALLBACK (on_app_clicked), app);
          g_signal_connect (tile, "show-context-menu",
                            G_CALLBACK (show_menu), app);
          g_signal_connect (tile, "begin-drag-move", G_CALLBACK (on_drag_move),
                            items);
          g_signal_connect (tile, "emblem-clicked",
                            G_CALLBACK (on_emblem_clicked), NULL);

          if (pixbuf)
            g_object_unref (pixbuf);
        }

      g_free (markup);
    }

  priv->last_active_item = item;
  priv->last_active_view = scroll;
  nl_content_view_set_child (nl_shell_get_content_view (priv->shell), scroll);
}

static void
on_menu_changed (LauncherMenu *menu, NlMenuItems *items)
{
  NlMenuItemsPrivate *priv;
  ClutterContainer *container;
  GSList *cats, *c;
  gint i = 300; /* High-ish order number */
  gboolean add_settings = FALSE;

  g_return_if_fail (LAUNCHER_IS_MENU (menu));
  g_return_if_fail (NL_IS_MENU_ITEMS (items));
  priv = items->priv;

  container = CLUTTER_CONTAINER (nl_shell_get_sidebar (priv->shell));

  for (c = priv->categories; c; c = c->next)
    {
      ClutterActor *category = c->data;

      if (category)
        clutter_container_remove_actor (CLUTTER_CONTAINER (container),
                                        category);
    }
  g_slist_free (priv->categories);
  priv->categories = NULL;

  cats = launcher_menu_get_categories (menu);
  for (c = cats; c; c = c->next)
    {
      LauncherCategory *category = c->data;
      ClutterActor *item;
      GdkPixbuf *pixbuf;
      gchar     *icon_name;

      if (!category)
        continue;

      if (g_strstr_len (launcher_category_get_icon_name (category),
                        -1,
                        "preferences"))
        {
          add_settings = TRUE;
          continue;
        }

      icon_name = g_strconcat ("unr-",
                               launcher_category_get_icon_name (category),
                               NULL);
      pixbuf = nl_pixbuf_cache_icon_for_name (priv->cache,
                                              icon_name,
                                              48);
      if (!pixbuf)
        {
          pixbuf = nl_pixbuf_cache_icon_for_name (priv->cache,
                                 launcher_category_get_icon_name (category),
                                 48);
        }
      g_free (icon_name);

      item = nl_sidebar_item_new (launcher_category_get_name (category),
                                  launcher_category_get_comment (category),
                                  pixbuf,
                                  i++);
      clutter_container_add_actor (container, item);
      clutter_actor_show (item);
      priv->categories = g_slist_append (priv->categories, item);

      g_object_set_data (G_OBJECT (item), "category", category);
      g_object_set_data (G_OBJECT (item), "nl-menu-items", items);
      g_signal_connect (item, "clicked", G_CALLBACK (on_clicked), priv->shell);

      if (pixbuf)
        g_object_unref (pixbuf);
    }

  if (add_settings)
    {
      GKeyFile     *file;
      ClutterActor *item;
      gchar        *name=NULL, *icon=NULL, *icon_name=NULL;
      GdkPixbuf    *pixbuf;

      file = g_key_file_new ();
      if (g_key_file_load_from_file (file,
                                 "/usr/share/desktop-directories/System.directory",
                                 G_KEY_FILE_NONE,
                                 NULL))
        {
          name = g_key_file_get_locale_string (file,
                                               G_KEY_FILE_DESKTOP_GROUP,
                                               G_KEY_FILE_DESKTOP_KEY_NAME,
                                               NULL,
                                               NULL);
          icon = g_key_file_get_string (file,
                                        G_KEY_FILE_DESKTOP_GROUP,
                                        G_KEY_FILE_DESKTOP_KEY_ICON,
                                        NULL);
        }
      g_key_file_free (file);

      if (!name)
        name = g_strdup ("Settings");
      if (!icon)
        icon = g_strdup ("preferences-desktop");

      icon_name = g_strconcat ("unr-", icon, NULL);
      pixbuf = nl_pixbuf_cache_icon_for_name (priv->cache,
                                              icon_name,
                                              48);
      if (!pixbuf)
        {
          pixbuf = nl_pixbuf_cache_icon_for_name (priv->cache, icon, 48);
        }

      item = nl_sidebar_item_new (name, NULL, pixbuf, i++);
      clutter_container_add_actor (container, item);
      clutter_actor_show (item);
      priv->categories = g_slist_append (priv->categories, item);

      g_signal_connect (item, "clicked", G_CALLBACK (on_system_clicked), items);

      if (pixbuf)
        g_object_unref (pixbuf);

      g_free (name);
      g_free (icon);
      g_free (icon_name);
    }
}


static void
on_content_view_child_changed (NlContentView *content_view,
                               ClutterActor  *new_view,
                               NlMenuItems   *items)
{
  NlMenuItemsPrivate *priv;
  GSList *c;

  g_return_if_fail (NL_IS_MENU_ITEMS (items));
  priv = items->priv;

  if (priv->last_active_view != new_view)
    priv->last_active_item = NULL;

  for (c = priv->categories; c; c = c->next)
    {
      NlSidebarItem *item = c->data;

      if (item != priv->last_active_item)
        nl_sidebar_item_set_active (item, FALSE);
    }
}
