/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/* ***** BEGIN LICENSE BLOCK *****
 *
 * Copyright (C) 2007 Andris Pavenis <andris.pavenis@iki.fi>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * ***** END OF LICENSE BLOCK ***** */

#include "mozVoikko.hxx"
#ifdef SYSTEM_LIBVOIKKO
#include <libvoikko/voikko.h>
#else
#include "voikko-dist.h"
#endif
#include "mozVoikkoSpell.hxx"
#include "mozVoikkoUtils.hxx"

#include <nsStringAPI.h>
#include <nsServiceManagerUtils.h>
#include <nsICategoryManager.h>
#include <nsISimpleEnumerator.h>
#include <nsIConsoleService.h>
#include <nsILocalFile.h>
#include <prlock.h>

#include <ctype.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>

namespace
{

    const char *encoding = "UTF-8";

    typedef const char * (*initvoikko_with_path_t)(int*, const char *, int, const char *);

    typedef const char * (*initvoikko_t)(int *, const char *, int);

    typedef int (*setsopt_t)(int, int, const char *);

    typedef int (*setbopt_t)(int, int, int);
 
    typedef int (*spell_t)(int, const char *);

    typedef char ** (*suggest_t)(int, const char *);

    typedef int (*terminate_voikko_t)(int);
	
    typedef void (*free_suggest_cstr_t)(char **);

    // Only one user can use voikko at the time, so we have only one copy and locking
    PRLock *voikko_lock = NULL;

    bool	libvoikko_init_tried = false;

    bool	libvoikko_initialized = false;

    PRLibrary *voikko_lib = NULL;

    int		voikko_users = 0;

    int		voikkohandle;

    // The initialization function of the loaded library
    initvoikko_with_path_t	init_func_;

    // The initialization function without explicit dictionary path
    initvoikko_t	init_system_func_;

    // The termination function of the loaded library
    terminate_voikko_t	terminate_func_;

    // The spell checking function of the loaded library
    spell_t	check_func_;

    // The suggestion generation function of the loaded library
    suggest_t	suggest_func_;

    // The procedure for releasing memory allocated by voikko_suggest_cstr().
    free_suggest_cstr_t free_suggest_cstr_func_;

    // The option setting functions of the loaded library
    setsopt_t	string_option_func_;
    setbopt_t	boolean_option_func_;

    // Preloaded libraries
    PreloadedLibraries *preloadedLibraries = NULL;

    struct VoikkoLock 
    {
        VoikkoLock(void)
        {
            PR_Lock(voikko_lock);
        }

        ~VoikkoLock()
        {
            PR_Unlock(voikko_lock);
        }
    };

    struct VoikkoPluginInit
    {
        VoikkoPluginInit(void)
        {

            voikko_lock = PR_NewLock();
            voikko_users = 0;
        }

        ~VoikkoPluginInit()
        {
            if (libvoikko_initialized)
            {
                (*terminate_func_)(voikkohandle);
            }
            else if (voikko_lib)
            {
                PR_UnloadLibrary (voikko_lib);
            }

            delete preloadedLibraries;

            PR_DestroyLock(voikko_lock);
        }
    };

    VoikkoPluginInit voikko_init_0_;

    bool LibvoikkoInit(void)
    {
        VoikkoLock lock;
        if (libvoikko_init_tried)
            return libvoikko_initialized;

        libvoikko_init_tried = true;

        #ifndef SYSTEM_LIBVOIKKO
        nsresult rv;
        nsCOMPtr<nsIFile> libDir, dataDir;
        rv = getMozVoikkoBaseDirs(getter_AddRefs(libDir), getter_AddRefs(dataDir));

        if (NS_FAILED(rv))
        {
            logMessage("Failed to get directories where Voikko libraries and data are installed");
            return false;
        }

        nsCString libDirPath, dataDirPath;
        libDir->GetNativePath(libDirPath);
        dataDir->GetNativePath(dataDirPath);

        rv = dataDir->GetNativePath(dataDirPath);
        NS_ENSURE_SUCCESS(rv, false);

        logMessage("Using Voikko data directory %s", dataDirPath.get());

        preloadedLibraries = new PreloadedLibraries(libDir, preloadLibNames, numPreloadedLibs);
        if (!preloadedLibraries)
            return false;

        if (!*preloadedLibraries)
        {
            return false;
        }
        #endif

        voikko_lib = PR_LoadLibrary(libvoikkoName);

        if (voikko_lib == 0) {
            logMessage("%s is not available: %s", libvoikkoName, prGetErrorText().get());
            return false;
        }

        bool symbols_ok = LoadSymbol(voikko_lib, &init_func_, "voikko_init_with_path")
            && LoadSymbol(voikko_lib, &init_system_func_, "voikko_init")
            && LoadSymbol(voikko_lib, &terminate_func_, "voikko_terminate")
            && LoadSymbol(voikko_lib, &check_func_, "voikko_spell_cstr")
            && LoadSymbol(voikko_lib, &suggest_func_, "voikko_suggest_cstr")
            && LoadSymbol(voikko_lib, &string_option_func_, "voikko_set_string_option")
            && LoadSymbol(voikko_lib, &boolean_option_func_, "voikko_set_bool_option")
            && LoadSymbol(voikko_lib, &free_suggest_cstr_func_, "voikko_free_suggest_cstr");

        if (!symbols_ok) {
            logMessage("Failed to find at least one required symbol in %s.", libvoikkoName);
            PR_UnloadLibrary (voikko_lib);
            voikko_lib = NULL;
            return false;
        }

        #ifdef SYSTEM_LIBVOIKKO
        const char* error = init_system_func_(&voikkohandle, "fi_FI", 0);
        #else
        const char* error = init_func_(&voikkohandle, "fi_FI", 0, dataDirPath.get());
        #endif
        if (error != 0) {
            logMessage("Failed to initialize libvoikko: %s.", error);
            PR_UnloadLibrary (voikko_lib);
            voikko_lib = NULL;
            return false;
        }

        boolean_option_func_(voikkohandle, VOIKKO_OPT_IGNORE_DOT, 1);
        boolean_option_func_(voikkohandle, VOIKKO_OPT_IGNORE_NUMBERS, 1);
        boolean_option_func_(voikkohandle, VOIKKO_OPT_IGNORE_UPPERCASE, 1);
        #ifdef VOIKKO_OPT_ACCEPT_MISSING_HYPHENS
        boolean_option_func_(voikkohandle, VOIKKO_OPT_ACCEPT_MISSING_HYPHENS, 1);
        #endif

        int status = string_option_func_(voikkohandle, VOIKKO_OPT_ENCODING, encoding);
        if (!status) {
            logMessage("Failed to set encoding %s for libvoikko.", encoding);
            return false;
        }
	
        logMessage("%s is successfully initialized.", libvoikkoName);

        libvoikko_initialized = true;
        return true;
    }

    int check (const char * str)
    {
        int status;
        VoikkoLock lock;
        // Always return spellcheck failure if not initialized properly (should never happen...).
        if (!libvoikko_initialized) return 0;

        status = (*check_func_)(voikkohandle, str);
        return status ? 1 : 0;
    }

    char **get_suggestions (const char * word)
    {
        char **vsuggestions = NULL;
        VoikkoLock lock;
        if (!libvoikko_initialized) return NULL;
        vsuggestions = (*suggest_func_) (voikkohandle, word);
        return vsuggestions;
    }
}

MozVoikko::MozVoikko (void)
    : is_ok (false)
{
    if (LibvoikkoInit()) {
        is_ok = true;
    }

    initialized_ = true;
}

MozVoikko::~MozVoikko()
{
}

int MozVoikko::suggest (char *** slst, const char * word)
{
    int ns = 0;
  
    *slst = get_suggestions(word);
    if (*slst != NULL) 
    {
        while ((*slst)[ns] != NULL) ns++;
    }
  
    return ns;
}

void MozVoikko::freeSuggestions(char **slst)
{
    free_suggest_cstr_func_(slst);
}

int MozVoikko::spell (const char * str)
{
    int status;

    // Always return spellcheck failure if not initialized properly.
    if (!is_ok) return 0;
    status = check(str);
    return status ? 1 : 0;
}

char * MozVoikko::get_dic_encoding()
{
    return (char*) encoding;
}
