# Copyright (c) 2009 Benjamin Drung <bdrung@debian.org>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.

# Reference: https://developer.mozilla.org/en/Toolkit_version_format

import sys

def decode_part(part):
    """Decodes a version part (like 5pre4) to
       <number-a><string-b><number-c><string-d>"""
    subpart = [0, "", 0, ""]

    # Split <number-a>
    length = 0
    for i in xrange(len(part)):
        if part[i].isdigit() or part[i] in ("-"):
            length += 1
        else:
            break
    if length > 0:
        subpart[0] = int(part[0:length])
    part = part[length:]

    # Split <string-b>
    length = 0
    for i in xrange(len(part)):
        if not (part[i].isdigit() or part[i] in ("-")):
            length += 1
        else:
            break
    subpart[1] = part[0:length]
    part = part[length:]

    # Split <number-c>
    length = 0
    for i in xrange(len(part)):
        if part[i].isdigit() or part[i] in ("-"):
            length += 1
        else:
            break
    if length > 0:
        subpart[2] = int(part[0:length])
    subpart[3] = part[length:]

    # if string-b is a plus sign, number-a is incremented to be compatible with
    # the Firefox 1.0.x version format: 1.0+ is the same as 1.1pre
    if subpart[1] == "+":
        subpart[0] += 1
        subpart[1] = "pre"

    # if the version part is a single asterisk, it is interpreted as an
    # infinitely-large number: 1.5.0.* is the same as 1.5.0.(infinity)
    if subpart[1] == "*":
        subpart[0] = sys.maxint
        subpart[1] = ""

    return subpart

def decode_version(version, verbose=False):
    """Decodes a version string like 1.1pre1a"""
    parts = version.split(".")
    decoded_parts = map(decode_part, parts)
    if verbose:
        print "I: Split %s up into %s." % (version, decoded_parts)
    return decoded_parts

def compare_subpart((a, b)):
    # A string-part that exists is always less-then a nonexisting string-part
    if a == "":
        if b == "":
            return 0
        else:
            return 1
    elif b == "":
        if a == "":
            return 0
        else:
            return -1
    else:
        return cmp(a, b)

def compare_part((x, y)):
    compared_subparts = filter(lambda x: x != 0,
                               map(compare_subpart, zip(x, y)))
    if compared_subparts:
        return compared_subparts[0]
    else:
        return 0

def compare_versions(version1, version2, verbose=False):
    a = decode_version(version1, verbose)
    b = decode_version(version2, verbose)

    if len(a) < len(b):
        a.extend((len(b) - len(a)) * [[0, "", 0, ""]])
    if len(b) < len(a):
        b.extend((len(a) - len(b)) * [[0, "", 0, ""]])

    result = filter(lambda x: x != 0, map(compare_part, zip(a, b)))
    if result:
        return result[0]
    else:
        return 0

def extract_upstream_version(debian_version):
    # remove last part separated by a dash (1.0-2 -> 1.0)
    parts = debian_version.split('-')
    if len(parts) > 1:
        del parts[-1]
    upstream_version = '-'.join(parts)

    # remove epoch
    parts = upstream_version.split(':')
    if len(parts) > 1:
        del parts[0]
    upstream_version = ':'.join(parts)

    return upstream_version

def convert_debian_to_moz_version(debian_version, verbose=False):
    upstream_version = extract_upstream_version(debian_version)

    # compatibility: strip +nobinonly and +build
    parts = upstream_version.split('+')
    if len(parts) > 1 and parts[-1] == "nobinonly":
        del parts[-1]
    if len(parts) > 1 and parts[-1].startswith("build"):
        del parts[-1]
    upstream_version = '+'.join(parts)

    moz_version = upstream_version.replace("~", "")
    return moz_version

def convert_moz_to_debian_version(moz_version, epoch=0, verbose=False):
    parts = decode_version(moz_version, verbose)
    # tranform parts
    for i in xrange(len(parts)):
        (number_a, string_b, number_c, string_d) = parts[i]
        part = ""
        if string_d != "":
            part = "~" + string_d
        if number_c != 0 or string_d != "":
            part = str(number_c) + part
        if string_b != "":
            part = "~" + string_b + part
        parts[i] = str(number_a) + part
    debian_version = ".".join(parts)
    if epoch != 0:
        debian_version = str(epoch) + ":" + debian_version
    return debian_version
