//
// Gendarme.Rules.Smells.AvoidLongMethodsRule class
//
// Authors:
//	Néstor Salceda <nestor.salceda@gmail.com>
//
// 	(C) 2007-2008 Néstor Salceda
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
//
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//

using System;
using System.Collections.Generic;

using Mono.Cecil;
using Mono.Cecil.Cil;

using Gendarme.Framework;
using Gendarme.Framework.Rocks;

namespace Gendarme.Rules.Smells {
	[Problem ("Long methods are usually hard to understand and maintain.  The method can cause problems because contains more code than the maximum allowed.")]
	[Solution ("You should apply an Extract Method refactoring, but there are other solutions.")]
	public class AvoidLongMethodsRule : Rule,IMethodRule {

		const int AssignationRatio = 7;
		const int DefaultAmountOfElements = 13;
		static Dictionary<string, string> typeMethodDictionary;

		static AvoidLongMethodsRule ()
		{
			typeMethodDictionary = new Dictionary<string,string> (4);
			typeMethodDictionary.Add ("Gtk.Bin", "Build");
			typeMethodDictionary.Add ("Gtk.Window", "Build");
			typeMethodDictionary.Add ("Gtk.Dialog", "Build");
			typeMethodDictionary.Add ("System.Windows.Forms.Form", "InitializeComponent");
		}

		public AvoidLongMethodsRule ()
		{
			MaxInstructions = 165;
			MaxSourceLineOfCode = 40;
		}

		public int MaxInstructions { get; set; }

		public int MaxSourceLineOfCode { get; set; }

		/// <summary>
		/// Use IL to approximate the size of the method. By default logical
		/// source lines are used (when debugging symbols are available).
		/// </summary>
		public bool UseIlApproximation { get; set; }


		private static bool IsAutogeneratedByTools (MethodDefinition method)
		{
			if (method.Parameters.Count != 0)
				return false;

			TypeDefinition type = method.DeclaringType.Resolve ();
			if ((type != null) && (type.BaseType != null)) {
				string method_name;
				if (typeMethodDictionary.TryGetValue (type.BaseType.FullName, out method_name)) {
					return (method_name == method.Name);
				}
			}
			return false;
		}

		private static int CountStaticFields (TypeDefinition type)
		{
			int counter = 0;
			foreach (FieldDefinition field in type.Fields) {
				if (field.IsStatic || field.HasConstant)
					counter++;
				//if the field is an array, we should take care
				//about their elements.
				ArrayType array = field.FieldType as ArrayType;
				if (array != null) {
					for (int index = 0; index < array.Dimensions.Count; index++)  
						//I can't calculate the array
						//length, then, i add a
						//default amount of elements 
						//TODO: Perhaps we can do other
						//approach with random nunbers?
						counter+= DefaultAmountOfElements;
				}
			}
			return counter;
		}

		private static int CountInstanceFields (TypeDefinition type) 
		{
			int counter = 0;
			foreach (FieldDefinition field in type.Fields) {
				if (!(field.IsStatic || field.HasConstant))
					counter++;
				//I not take care about arrays here.
			}
			return counter;
		}

		private static int GetFieldCount (TypeDefinition type, bool staticFields)
		{
			return staticFields ? CountStaticFields (type) : CountInstanceFields (type);
		}

		private static int CountSourceLinesOfCode (MethodDefinition method)
		{
			int sloc = 0;
			int current_line = -1;
			foreach (Instruction ins in method.Body.Instructions) {
				SequencePoint sp = ins.SequencePoint;
				if (sp == null)
					continue;

				int line = sp.StartLine;
				// special value for PDB (so that debuggers can ignore a line)
				if (line == 0xFEEFEE)
					continue;

				// lines numbers may not be ordered (loops) or reused several times
				if ((current_line == -1) || (line > current_line))
					sloc++;
				current_line = line;
			}
			return sloc;
		}

		private static int CountInstructions (MethodDefinition method)
		{
			int count = 0;
			foreach (Instruction ins in method.Body.Instructions) {
				switch (ins.OpCode.Code) {
				case Code.Nop:
				case Code.Box:
				case Code.Unbox:
					break;
				default:
					count++;
					break;
				}
			}
			return count;
		}

		public RuleResult CheckMethod (MethodDefinition method)
		{
			// rule does not apply if method as no code (e.g. abstract, p/invoke)
			// rule does not apply to code outside the developer's control
			// rule does not apply to autogenerated code from some tools
			if (!method.HasBody || method.IsGeneratedCode () || IsAutogeneratedByTools (method))
				return RuleResult.DoesNotApply;

			// rule applies!

			int field_count = 0;
			if (method.IsConstructor)
				field_count = GetFieldCount ((method.DeclaringType as TypeDefinition), method.IsStatic);

			// if we have debugging information available and we're not asked to use IL approximation
			if (!UseIlApproximation && method.DeclaringType.Module.HasDebuggingInformation ()) {
				// add a few extra lines to let the constructors initialize the fields
				int max = MaxSourceLineOfCode + field_count;
				int sloc = CountSourceLinesOfCode (method);
				if (sloc <= max)
					return RuleResult.Success;

				string message = String.Format ("Logical SLOC: {0}. Maximum : {1}", sloc, max);
				Runner.Report (method, Severity.High, Confidence.High, message);
			} else {
				// success if the instruction count is below the defined threshold
				// add a few extra lines to let the constructors initialize the fields
				int max = MaxInstructions + field_count * AssignationRatio;
				int count = CountInstructions (method);
				if (count <= max)
					return RuleResult.Success;

				string message = String.Format ("Method IL Size: {0}. Maximum Size: {1}", count, max);
				Runner.Report (method, Severity.High, Confidence.Normal, message);
			}

			return RuleResult.Failure;
		}
	}
}
