// 
// Unit tests for EnumsShouldUseInt32Rule
//
// Authors:
//	Sebastien Pouliot  <sebastien@ximian.com>
//
// Copyright (C) 2008 Novell, Inc (http://www.novell.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

using System;
using System.Reflection;

using Mono.Cecil;

using Gendarme.Framework;
using Gendarme.Rules.Design;

using NUnit.Framework;
using Test.Rules.Helpers;

namespace Test.Rules.Design {

	[TestFixture]
	public class EnumsShouldUseInt32Test {

		public enum DefaultEnum {
		}

		public enum ByteEnum : byte {
		}

		public enum SignedByteEnum : sbyte {
		}

		public enum ShortEnum : short {
		}

		public enum UnsignedShortEnum : ushort {
		}

		public enum IntEnum : int {
		}

		public enum UnsignedIntEnum : uint {
		}

		public enum LongEnum : long {
		}

		public enum UnsignedLongEnum : ulong {
		}

		private ITypeRule rule;
		private AssemblyDefinition assembly;
		private TestRunner runner;

		[TestFixtureSetUp]
		public void FixtureSetUp ()
		{
			string unit = Assembly.GetExecutingAssembly ().Location;
			assembly = AssemblyFactory.GetAssembly (unit);
			rule = new EnumsShouldUseInt32Rule ();
			runner = new TestRunner (rule);
		}

		private TypeDefinition GetTest (string name)
		{
			string fullname = "Test.Rules.Design.EnumsShouldUseInt32Test" + name;
			return assembly.MainModule.Types [fullname];
		}

		[Test]
		public void NotAnEnum ()
		{
			TypeDefinition type = GetTest (String.Empty);
			Assert.AreEqual (RuleResult.DoesNotApply, runner.CheckType (type), "RuleResult");
			Assert.AreEqual (0, runner.Defects.Count, "Count");
		}

		[Test]
		public void Ok ()
		{
			TypeDefinition type = GetTest ("/DefaultEnum");
			Assert.AreEqual (RuleResult.Success, runner.CheckType (type), "RuleResult1");
			Assert.AreEqual (0, runner.Defects.Count, "Count1");

			type = GetTest ("/IntEnum");
			Assert.AreEqual (RuleResult.Success, runner.CheckType (type), "RuleResult2");
			Assert.AreEqual (0, runner.Defects.Count, "Count2");
		}

		[Test]
		public void Bad ()
		{
			// CLS compliant types: Byte, Int16 or Int64 (Int32 is CLS but Ok)

			TypeDefinition type = GetTest ("/ByteEnum");
			Assert.AreEqual (RuleResult.Failure, runner.CheckType (type), "RuleResult1");
			Assert.AreEqual (1, runner.Defects.Count, "Count1");

			type = GetTest ("/ShortEnum");
			Assert.AreEqual (RuleResult.Failure, runner.CheckType (type), "RuleResult2");
			Assert.AreEqual (1, runner.Defects.Count, "Count2");

			type = GetTest ("/LongEnum");
			Assert.AreEqual (RuleResult.Failure, runner.CheckType (type), "RuleResult3");
			Assert.AreEqual (1, runner.Defects.Count, "Count3");
		}

		[Test]
		public void ReallyBad ()
		{
			// i.e. using non-CLS compliant types, SByte, UInt16, UInt32 or UInt64

			TypeDefinition type = GetTest ("/SignedByteEnum");
			Assert.AreEqual (RuleResult.Failure, runner.CheckType (type), "RuleResult1");
			Assert.AreEqual (1, runner.Defects.Count, "Count1");

			type = GetTest ("/UnsignedShortEnum");
			Assert.AreEqual (RuleResult.Failure, runner.CheckType (type), "RuleResult2");
			Assert.AreEqual (1, runner.Defects.Count, "Count2");

			type = GetTest ("/UnsignedIntEnum");
			Assert.AreEqual (RuleResult.Failure, runner.CheckType (type), "RuleResult3");
			Assert.AreEqual (1, runner.Defects.Count, "Count3");

			type = GetTest ("/UnsignedLongEnum");
			Assert.AreEqual (RuleResult.Failure, runner.CheckType (type), "RuleResult4");
			Assert.AreEqual (1, runner.Defects.Count, "Count4");
		}
	}
}
