# Copyright 2016 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

"""Tests for the Discovery model."""

__all__ = []

from maasserver.dbviews import register_view
from maasserver.models import (
    Discovery,
    discovery as discovery_module,
    MDNS,
    Neighbour,
)
from maasserver.testing.factory import factory
from maasserver.testing.testcase import MAASServerTestCase
from maastesting.matchers import (
    DocTestMatches,
    Matches,
    MockCalledOnceWith,
    MockNotCalled,
)
from testtools.matchers import (
    Equals,
    Is,
)


class TestDiscoveryModel(MAASServerTestCase):

    def setUp(self):
        super().setUp()
        register_view("maasserver_discovery")

    def test_mac_organization(self):
        discovery = factory.make_Discovery(mac_address="48:51:b7:00:00:00")
        self.assertThat(discovery.mac_organization, Equals("Intel Corporate"))

    def test__ignores_duplicate_macs(self):
        rack1 = factory.make_RackController()
        rack2 = factory.make_RackController()
        iface1 = factory.make_Interface(node=rack1)
        iface2 = factory.make_Interface(node=rack2)
        # Simulate a single device being observed from two different rack
        # interfaces.
        d1 = factory.make_Discovery(interface=iface1)
        factory.make_Discovery(
            interface=iface2, mac_address=d1.mac_address, ip=d1.ip)
        # ... the Discovery view should only display one entry.
        self.assertThat(Discovery.objects.count(), Equals(1))

    def test__query_by_unknown_mac(self):
        rack = factory.make_RackController()
        iface = factory.make_Interface(node=rack)
        discovery = factory.make_Discovery(interface=iface)
        self.assertThat(Discovery.objects.by_unknown_mac().count(), Equals(1))
        factory.make_Interface(mac_address=discovery.mac_address)
        # Now that we have a known interface with the same MAC, the discovery
        # should disappear from this query.
        self.assertThat(Discovery.objects.by_unknown_mac().count(), Equals(0))

    def test__query_by_unknown_ip(self):
        rack = factory.make_RackController()
        iface = factory.make_Interface(node=rack)
        discovery = factory.make_Discovery(interface=iface, ip="10.0.0.1")
        self.assertThat(Discovery.objects.by_unknown_ip().count(), Equals(1))
        factory.make_StaticIPAddress(ip=discovery.ip, cidr="10.0.0.0/8")
        # Now that we have a known IP address that matches, the discovery
        # should disappear from this query.
        self.assertThat(Discovery.objects.by_unknown_ip().count(), Equals(0))

    def test__query_by_unknown_ip_and_mac__known_ip(self):
        rack = factory.make_RackController()
        iface = factory.make_Interface(node=rack)
        discovery = factory.make_Discovery(interface=iface, ip="10.0.0.1")
        self.assertThat(
            Discovery.objects.by_unknown_ip_and_mac().count(), Equals(1))
        factory.make_StaticIPAddress(ip=discovery.ip, cidr="10.0.0.0/8")
        # Known IP address, unexpected MAC.
        self.assertThat(
            Discovery.objects.by_unknown_ip_and_mac().count(), Equals(0))

    def test__query_by_unknown_ip_and_mac__known_mac(self):
        rack = factory.make_RackController()
        iface = factory.make_Interface(node=rack)
        discovery = factory.make_Discovery(interface=iface)
        self.assertThat(
            Discovery.objects.by_unknown_ip_and_mac().count(), Equals(1))
        # Known MAC, unknown IP.
        factory.make_Interface(mac_address=discovery.mac_address)
        self.assertThat(
            Discovery.objects.by_unknown_ip_and_mac().count(), Equals(0))

    def test__does_not_fail_if_cannot_find_subnet(self):
        rack = factory.make_RackController()
        iface = factory.make_Interface(node=rack)
        factory.make_Discovery(interface=iface, ip="10.0.0.1")
        self.assertThat(Discovery.objects.first().subnet, Is(None))

    def test__associates_known_subnet(self):
        rack = factory.make_RackController()
        iface = factory.make_Interface(node=rack)
        subnet = factory.make_Subnet(cidr="10.0.0.0/8", vlan=iface.vlan)
        factory.make_Discovery(interface=iface, ip="10.0.0.1")
        self.assertThat(Discovery.objects.first().subnet, Equals(subnet))

    def test__associates_best_subnet(self):
        rack = factory.make_RackController()
        iface = factory.make_Interface(node=rack)
        # Seems unlikely, but we'll test it anyway. ;-)
        subnet = factory.make_Subnet(cidr="10.0.0.0/24", vlan=iface.vlan)
        factory.make_Subnet(cidr="10.0.0.0/8", vlan=iface.vlan)
        factory.make_Discovery(interface=iface, ip="10.0.0.1")
        self.assertThat(Discovery.objects.first().subnet, Equals(subnet))
        self.assertThat(Discovery.objects.count(), Equals(1))

    def test__is_external_dhcp(self):
        rack = factory.make_RackController()
        iface = factory.make_Interface(node=rack)
        factory.make_Subnet(cidr="10.0.0.0/8", vlan=iface.vlan)
        factory.make_Discovery(interface=iface, ip="10.0.0.1")
        discovery = Discovery.objects.first()
        self.assertThat(discovery.is_external_dhcp, Equals(False))
        iface.vlan.external_dhcp = '10.0.0.1'
        iface.vlan.save()
        discovery = Discovery.objects.first()
        self.assertThat(discovery.is_external_dhcp, Equals(True))


class TestDiscoveryManagerClear(MAASServerTestCase):
    """Tests for `DiscoveryManager.clear` """

    def test__clear_mdns_entries(self):
        maaslog = self.patch(discovery_module.maaslog, 'info')
        factory.make_MDNS()
        factory.make_MDNS()
        factory.make_Neighbour()
        factory.make_Neighbour()
        self.assertThat(MDNS.objects.count(), Equals(2))
        self.assertThat(Neighbour.objects.count(), Equals(2))
        Discovery.objects.clear(mdns=True)
        self.assertThat(MDNS.objects.count(), Equals(0))
        self.assertThat(Neighbour.objects.count(), Equals(2))
        self.assertThat(maaslog, MockCalledOnceWith(
            Matches(DocTestMatches('Cleared all mDNS entries.'))))

    def test__clear_neighbour_entries(self):
        maaslog = self.patch(discovery_module.maaslog, 'info')
        factory.make_MDNS()
        factory.make_MDNS()
        factory.make_Neighbour()
        factory.make_Neighbour()
        self.assertThat(MDNS.objects.count(), Equals(2))
        self.assertThat(Neighbour.objects.count(), Equals(2))
        Discovery.objects.clear(neighbours=True)
        self.assertThat(MDNS.objects.count(), Equals(2))
        self.assertThat(Neighbour.objects.count(), Equals(0))
        self.assertThat(maaslog, MockCalledOnceWith(
            Matches(DocTestMatches('Cleared all neighbour entries.'))
        ))

    def test__clear_all_entries(self):
        maaslog = self.patch(discovery_module.maaslog, 'info')
        factory.make_MDNS()
        factory.make_MDNS()
        factory.make_Neighbour()
        factory.make_Neighbour()
        self.assertThat(MDNS.objects.count(), Equals(2))
        self.assertThat(Neighbour.objects.count(), Equals(2))
        Discovery.objects.clear(all=True)
        self.assertThat(MDNS.objects.count(), Equals(0))
        self.assertThat(Neighbour.objects.count(), Equals(0))
        self.assertThat(maaslog, MockCalledOnceWith(
            Matches(DocTestMatches('Cleared all mDNS and neighbour entries.'))
        ))

    def test__clear_mdns_entries_is_noop_if_what_to_clear_is_unspecified(self):
        maaslog = self.patch(discovery_module.maaslog, 'info')
        factory.make_MDNS()
        factory.make_MDNS()
        factory.make_Neighbour()
        factory.make_Neighbour()
        self.assertThat(MDNS.objects.count(), Equals(2))
        self.assertThat(Neighbour.objects.count(), Equals(2))
        # clear() is a no-op if what to clear isn't specified.
        Discovery.objects.clear()
        self.assertThat(MDNS.objects.count(), Equals(2))
        self.assertThat(Neighbour.objects.count(), Equals(2))
        self.assertThat(maaslog, MockNotCalled())

    def test__clear_logs_username_if_given(self):
        user = factory.make_admin()
        maaslog = self.patch(discovery_module.maaslog, 'info')
        factory.make_MDNS()
        factory.make_Neighbour()
        Discovery.objects.clear(user=user, all=True)
        self.assertThat(maaslog, MockCalledOnceWith(
            Matches(DocTestMatches("User '%s' cleared..." % user.username))
        ))
