/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2011 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gio/gio.h>
#include <glib/gi18n-lib.h>
#include <libsoup/soup-uri.h>

#include "u1-request-chrome.h"

static GInputStream *u1_request_chrome_send (SoupRequest *request,
                                             GCancellable *cancellable,
                                             GError **error);
static goffset u1_request_chrome_get_content_length (SoupRequest *request);
static const gchar *u1_request_chrome_get_content_type (SoupRequest *request);

G_DEFINE_TYPE (U1RequestChrome, u1_request_chrome, SOUP_TYPE_REQUEST)

/* The URI schemes we handle */
static const gchar *chrome_schemes[] = { "u1chrome", NULL };

struct _U1RequestChromePrivate {
};

static void
u1_request_chrome_init (U1RequestChrome *chrome)
{
    chrome->priv = G_TYPE_INSTANCE_GET_PRIVATE (chrome, U1_TYPE_REQUEST_CHROME, U1RequestChromePrivate);
}

static void
u1_request_chrome_class_init (U1RequestChromeClass *klass)
{
    SoupRequestClass *request_class = SOUP_REQUEST_CLASS (klass);

    request_class->schemes = chrome_schemes;

    g_type_class_add_private (klass, sizeof (U1RequestChromeClass));

    request_class->send = u1_request_chrome_send;
    request_class->get_content_length = u1_request_chrome_get_content_length;
    request_class->get_content_type = u1_request_chrome_get_content_type;
}

static GInputStream *
u1_request_chrome_send (SoupRequest *request,
                        GCancellable *cancellable,
                        GError **error)
{
    SoupURI *uri = soup_request_get_uri (request);
    gchar *data;

    if (uri->path && !g_ascii_strcasecmp (uri->path, "empty"))
        data = g_strdup ("");
    else
        data = g_strdup ("");

    return g_memory_input_stream_new_from_data (data, strlen (data), g_free);
}

static goffset
u1_request_chrome_get_content_length (SoupRequest *request)
{
    return -1;
}

static const gchar *
u1_request_chrome_get_content_type (SoupRequest *request)
{
    return "text/plain";
}
