/*
 * validate_x86.c - validate event tables + encodings
 *
 * Copyright (c) 2010 Google, Inc
 * Contributed by Stephane Eranian <eranian@gmail.com>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * This file is part of libpfm, a performance monitoring support library for
 * applications on Linux.
 */
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <inttypes.h>
#include <stdarg.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>

#include <perfmon/pfmlib.h>

#define MAX_ENCODING	8
#define SRC_LINE	.line = __LINE__

typedef struct {
	const char *name;
	const char *fstr;
	uint64_t codes[MAX_ENCODING];
	int ret, count;
	int line;
} test_event_t;

static const test_event_t x86_test_events[]={
	{ SRC_LINE,
	  .name = "core::INST_RETIRED:ANY_P",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5300c0ull,
	},
	{ SRC_LINE,
	  .name = "core::INST_RETIRED:ANY_P:ANY_P",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5300c0ull,
	},
	{ SRC_LINE,
	  .name = "core::INST_RETIRED:ANY_P:DEAD",
	  .ret  = PFM_ERR_ATTR, /* cannot know if it is umask or mod */
	  .count = 0,
	  .codes[0] = 0ull,
	},
	{ SRC_LINE,
	  .name = "core::INST_RETIRED:ANY_P:u:u",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5100c0ull,
	},
	{ SRC_LINE,
	  .name = "core::INST_RETIRED:ANY_P:u=0:k=1:u=1",
	  .ret  = PFM_ERR_ATTR_SET,
	  .count = 0,
	  .codes[0] = 0ull,
	},
	{ SRC_LINE,
	  .name = "core::INST_RETIRED:ANY_P:c=1:i",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1d300c0ull,
	},
	{ SRC_LINE,
	  .name = "core::INST_RETIRED:ANY_P:c=1:i=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1d300c0ull,
	},
	{ SRC_LINE,
	  .name = "core::INST_RETIRED:ANY_P:c=2",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x25300c0ull,
	},
	{ SRC_LINE,
	  .name = "core::INST_RETIRED:ANY_P:c=320",
	  .ret  = PFM_ERR_ATTR_VAL,
	  .count = 0,
	  .codes[0] = 0ull,
	},
	{ SRC_LINE,
	  .name = "core::INST_RETIRED:ANY_P:t=1",
	  .ret  = PFM_ERR_ATTR,
	  .count = 0,
	  .codes[0] = 0ull,
	},
	{ SRC_LINE,
	  .name = "core::L2_LINES_IN",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x537024ull,
	},
	{ SRC_LINE,
	  .name = "core::L2_LINES_IN:SELF",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x537024ull,
	  .fstr = "core::L2_LINES_IN:SELF:ANY:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "core::L2_LINES_IN:SELF:BOTH_CORES",
	  .ret  = PFM_ERR_FEATCOMB,
	  .count = 0,
	  .codes[0] = 0ull,
	},
	{ SRC_LINE,
	  .name = "core::L2_LINES_IN:SELF:PREFETCH",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x535024ull,
	},
	{ SRC_LINE,
	  .name = "core::L2_LINES_IN:SELF:PREFETCH:ANY",
	  .ret  = PFM_ERR_FEATCOMB,
	  .count = 0,
	  .codes[0] = 0ull,
	},
	{ SRC_LINE,
	  .name = "core::RS_UOPS_DISPATCHED_NONE",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1d300a0ull,
	},
	{ SRC_LINE,
	  .name = "core::RS_UOPS_DISPATCHED_NONE:c=2",
	  .ret  = PFM_ERR_ATTR_SET,
	  .count = 1,
	  .codes[0] = 0ull,
	},
	{ SRC_LINE,
	  .name = "core::branch_instructions_retired",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5300c4ull,
	  .fstr = "core::BR_INST_RETIRED:ANY:k=1:u=1:e=0:i=0:c=0"
	},
	{ SRC_LINE,
	  .name = "nhm::branch_instructions_retired",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5300c4ull,
	  .fstr = "nhm::BR_INST_RETIRED:ALL_BRANCHES:k=1:u=1:e=0:i=0:c=0:t=0"
	},
	{ SRC_LINE,
	  .name = "wsm::BRANCH_INSTRUCTIONS_RETIRED",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5300c4ull, /* architected encoding, guaranteed to exist */
	  .fstr = "wsm::BR_INST_RETIRED:ALL_BRANCHES:k=1:u=1:e=0:i=0:c=0:t=0"
	},
	{ SRC_LINE,
	  .name = "nhm::ARITH:DIV:k",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1d60114ull,
	  .fstr = "nhm::ARITH:CYCLES_DIV_BUSY:k=1:u=0:e=1:i=1:c=1:t=0",
	},
	{ SRC_LINE,
	  .name = "nhm::ARITH:CYCLES_DIV_BUSY:k=1:u=1:e=1:i=1:c=1:t=0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1d70114ull,
	  .fstr = "nhm::ARITH:CYCLES_DIV_BUSY:k=1:u=1:e=1:i=1:c=1:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm::UOPS_EXECUTED:CORE_STALL_COUNT:u",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1f53fb1ull,
	  .fstr = "wsm::UOPS_EXECUTED:CORE_STALL_CYCLES:k=0:u=1:e=1:i=1:c=1:t=1",
	},
	{ SRC_LINE,
	  .name = "wsm::UOPS_EXECUTED:CORE_STALL_COUNT:u:t=0",
	  .ret  = PFM_ERR_ATTR_SET,
	  .count = 0,
	  .codes[0] = 0ull,
	},
	{ SRC_LINE,
	  .name = "wsm_unc::unc_qmc_writes:full_any:partial_any",
	  .ret  = PFM_ERR_FEATCOMB,
	  .count = 0,
	  .codes[0] = 0ull,
	},
	{ SRC_LINE,
	  .name = "wsm_unc::unc_qmc_writes",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x50072full,
	  .fstr = "wsm_unc::UNC_QMC_WRITES:FULL_ANY:e=0:i=0:c=0:o=0",
	},
	{ SRC_LINE,
	  .name = "wsm_unc::unc_qmc_writes:full_any",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x50072full,
	  .fstr = "wsm_unc::UNC_QMC_WRITES:FULL_ANY:e=0:i=0:c=0:o=0",
	},
	{ SRC_LINE,
	  .name = "wsm_unc::unc_qmc_writes:full_ch0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x50012full,
	  .fstr = "wsm_unc::UNC_QMC_WRITES:FULL_CH0:e=0:i=0:c=0:o=0",
	},
	{ SRC_LINE,
	  .name = "wsm_unc::unc_qmc_writes:partial_any",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x50382full,
	  .fstr = "wsm_unc::UNC_QMC_WRITES:PARTIAL_ANY:e=0:i=0:c=0:o=0",
	},
	{ SRC_LINE,
	  .name = "wsm_unc::unc_qmc_writes:partial_ch0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x50082full,
	  .fstr = "wsm_unc::UNC_QMC_WRITES:PARTIAL_CH0:e=0:i=0:c=0:o=0",
	},
	{ SRC_LINE,
	  .name = "wsm_unc::unc_qmc_writes:partial_ch0:partial_ch1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x50182full,
	  .fstr = "wsm_unc::UNC_QMC_WRITES:PARTIAL_CH0:PARTIAL_CH1:e=0:i=0:c=0:o=0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam10h_barcelona::DISPATCHED_FPU",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x533f00ull,
	  .fstr = "amd64_fam10h_barcelona::DISPATCHED_FPU:ALL:k=1:u=1:e=0:i=0:c=0:h=0:g=0"
	},
	{ SRC_LINE,
	  .name = "amd64_fam10h_barcelona::DISPATCHED_FPU:k:u=0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x523f00ull,
	  .fstr = "amd64_fam10h_barcelona::DISPATCHED_FPU:ALL:k=1:u=0:e=0:i=0:c=0:h=0:g=0"
	},
	{ SRC_LINE,
	  .name = "amd64_fam10h_barcelona::DISPATCHED_FPU:OPS_ADD:OPS_MULTIPLY",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x530300ull,
	  .fstr = "amd64_fam10h_barcelona::DISPATCHED_FPU:OPS_ADD:OPS_MULTIPLY:k=1:u=1:e=0:i=0:c=0:h=0:g=0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam10h_barcelona::L2_CACHE_MISS:ALL:DATA",
	  .ret  = PFM_ERR_FEATCOMB,
	  .count = 0,
	  .codes[0] = 0ull,
	},
	{ SRC_LINE,
	  .name = "amd64_fam10h_barcelona::MEMORY_CONTROLLER_REQUESTS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x10053fff0ull,
	  .fstr = "amd64_fam10h_barcelona::MEMORY_CONTROLLER_REQUESTS:ALL:k=1:u=1:e=0:i=0:c=0:h=0:g=0",
	},
	{ SRC_LINE,
	  .name = "amd64_k8_revb::RETURN_STACK_OVERFLOWS:g=1:u",
	  .ret  = PFM_ERR_ATTR,
	  .count = 0,
	  .codes[0] = 0ull,
	},
	{ SRC_LINE,
	  .name = "amd64_k8_revb::RETURN_STACK_HITS:e=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x570088ull,
	  .fstr = "amd64_k8_revb::RETURN_STACK_HITS:k=1:u=1:e=1:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "amd64_k8_revb::PROBE:ALL",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x533fecull,
	  .fstr = "amd64_k8_revb::PROBE:ALL:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "amd64_k8_revc::PROBE:ALL",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x533fecull,
	  .fstr = "amd64_k8_revc::PROBE:ALL:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "amd64_k8_revd::PROBE:ALL",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x537fecull,
	  .fstr = "amd64_k8_revd::PROBE:ALL:k=1:u=1:e=0:i=0:c=0"
	},
	{ SRC_LINE,
	  .name = "amd64_k8_reve::PROBE:ALL",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x537fecull,
	  .fstr = "amd64_k8_reve::PROBE:ALL:k=1:u=1:e=0:i=0:c=0"
	},
	{ SRC_LINE,
	  .name = "amd64_k8_revf::PROBE:ALL",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x537fecull,
	  .fstr = "amd64_k8_revf::PROBE:ALL:k=1:u=1:e=0:i=0:c=0"
	},
	{ SRC_LINE,
	  .name = "amd64_k8_revg::PROBE:ALL",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x537fecull,
	  .fstr = "amd64_k8_revg::PROBE:ALL:k=1:u=1:e=0:i=0:c=0"
	},
	{ SRC_LINE,
	  .name = "amd64_fam10h_barcelona::L1_DTLB_MISS_AND_L2_DTLB_HIT:L2_1G_TLB_HIT",
	  .ret  = PFM_ERR_ATTR,
	  .count = 0,
	  .codes[0] = 0ull,
	},
	{ SRC_LINE,
	  .name = "amd64_fam10h_barcelona::L1_DTLB_MISS_AND_L2_DTLB_HIT:ALL",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x530345ull,
	  .fstr = "amd64_fam10h_barcelona::L1_DTLB_MISS_AND_L2_DTLB_HIT:ALL:k=1:u=1:e=0:i=0:c=0:h=0:g=0"
	},
	{ SRC_LINE,
	  .name = "amd64_fam10h_shanghai::L1_DTLB_MISS_AND_L2_DTLB_HIT:ALL",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x530745ull,
	  .fstr = "amd64_fam10h_shanghai::L1_DTLB_MISS_AND_L2_DTLB_HIT:ALL:k=1:u=1:e=0:i=0:c=0:h=0:g=0"
	},
	{ SRC_LINE,
	  .name = "amd64_fam10h_istanbul::L1_DTLB_MISS_AND_L2_DTLB_HIT:ALL",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x530745ull,
	  .fstr = "amd64_fam10h_istanbul::L1_DTLB_MISS_AND_L2_DTLB_HIT:ALL:k=1:u=1:e=0:i=0:c=0:h=0:g=0"
	},
	{ SRC_LINE,
	  .name = "amd64_fam10h_barcelona::READ_REQUEST_TO_L3_CACHE",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x40053f7e0ull,
	  .fstr = "amd64_fam10h_barcelona::READ_REQUEST_TO_L3_CACHE:ANY_READ:ALL_CORES:k=1:u=1:e=0:i=0:c=0:h=0:g=0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam10h_shanghai::READ_REQUEST_TO_L3_CACHE",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x40053f7e0ull,
	  .fstr = "amd64_fam10h_shanghai::READ_REQUEST_TO_L3_CACHE:ANY_READ:ALL_CORES:k=1:u=1:e=0:i=0:c=0:h=0:g=0",
	},
	{ SRC_LINE,
	  .name = "core::RAT_STALLS:ANY:u:c=1,cycles", /* must cut at comma */
	  .ret  = PFM_ERR_INVAL,
	},
	{ SRC_LINE,
	  .name = "wsm::mem_uncore_retired:remote_dram",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x53200f,
	  .fstr = "wsm::MEM_UNCORE_RETIRED:REMOTE_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm_dp::mem_uncore_retired:remote_dram",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x53100f,
	  .fstr = "wsm_dp::MEM_UNCORE_RETIRED:REMOTE_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm::mem_uncore_retired:local_dram",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x53100f,
	  .fstr = "wsm::MEM_UNCORE_RETIRED:LOCAL_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm_dp::mem_uncore_retired:local_dram",
	  .ret  = PFM_ERR_ATTR,
	  .count = 1,
	  .codes[0] = 0,
	},
	{ SRC_LINE,
	  .name = "nhm::mem_uncore_retired:uncacheable",
	  .ret  = PFM_ERR_ATTR,
	  .count = 1,
	  .codes[0] = 0,
	},
	{ SRC_LINE,
	  .name = "nhm::mem_uncore_retired:l3_data_miss_unknown",
	  .ret  = PFM_ERR_ATTR,
	  .count = 1,
	  .codes[0] = 0,
	},
	{ SRC_LINE,
	  .name = "nhm_ex::mem_uncore_retired:uncacheable",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x53800f,
	  .fstr = "nhm_ex::MEM_UNCORE_RETIRED:UNCACHEABLE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "nhm_ex::mem_uncore_retired:l3_data_miss_unknown",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x53010f,
	  .fstr = "nhm_ex::MEM_UNCORE_RETIRED:L3_DATA_MISS_UNKNOWN:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "nhm::mem_uncore_retired:local_dram",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x53200f,
	  .fstr = "nhm::MEM_UNCORE_RETIRED:LOCAL_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "nhm_ex::mem_uncore_retired:local_dram",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x53200f,
	  .fstr = "nhm_ex::MEM_UNCORE_RETIRED:LOCAL_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm::offcore_response_0:k",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5201b7,
	  .codes[1] = 0xffff,
	  .fstr = "wsm::OFFCORE_RESPONSE_0:ANY_REQUEST:ANY_RESPONSE:k=1:u=0:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm::offcore_response_0:local_dram",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x20ff,
	  .fstr = "wsm::OFFCORE_RESPONSE_0:ANY_REQUEST:LOCAL_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm::offcore_response_0:PF_IFETCH",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0xff40,
	  .fstr = "wsm::OFFCORE_RESPONSE_0:PF_IFETCH:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm::offcore_response_0:ANY_DATA:LOCAL_DRAM",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x2033,
	  .fstr = "wsm::OFFCORE_RESPONSE_0:DMND_DATA_RD:DMND_RFO:PF_DATA_RD:PF_RFO:LOCAL_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm::offcore_response_0:DMND_RFO:DMND_DATA_RD:LOCAL_DRAM:REMOTE_DRAM",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x6003,
	  .fstr = "wsm::OFFCORE_RESPONSE_0:DMND_DATA_RD:DMND_RFO:LOCAL_DRAM:REMOTE_DRAM:k=1:u=1:e=0:i=0:c=0:t=0"
	},
	{ SRC_LINE,
	  .name = "wsm::offcore_response_1:local_dram",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x20ff,
	  .fstr = "wsm::OFFCORE_RESPONSE_1:ANY_REQUEST:LOCAL_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm::offcore_response_1:PF_IFETCH",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0xff40,
	  .fstr = "wsm::OFFCORE_RESPONSE_1:PF_IFETCH:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm::offcore_response_1:ANY_DATA:LOCAL_DRAM",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x2033,
	  .fstr = "wsm::OFFCORE_RESPONSE_1:DMND_DATA_RD:DMND_RFO:PF_DATA_RD:PF_RFO:LOCAL_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm::offcore_response_1:DMND_RFO:DMND_DATA_RD:LOCAL_DRAM:REMOTE_DRAM",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x6003,
	  .fstr = "wsm::OFFCORE_RESPONSE_1:DMND_DATA_RD:DMND_RFO:LOCAL_DRAM:REMOTE_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm::offcore_response_0:ANY_LLC_MISS",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0xf8ff,
	  .fstr = "wsm::OFFCORE_RESPONSE_0:ANY_REQUEST:REMOTE_CACHE_HITM:REMOTE_CACHE_FWD:LOCAL_DRAM:REMOTE_DRAM:NON_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm_dp::offcore_response_0:ANY_LLC_MISS",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0xf8ff,
	  .fstr = "wsm_dp::OFFCORE_RESPONSE_0:ANY_REQUEST:REMOTE_CACHE_HITM:LOCAL_DRAM_AND_REMOTE_CACHE_HIT:REMOTE_DRAM:OTHER_LLC_MISS:NON_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm_dp::offcore_response_0:LOCAL_CACHE",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x7ff,
	  .fstr = "wsm_dp::OFFCORE_RESPONSE_0:ANY_REQUEST:UNCORE_HIT:OTHER_CORE_HIT_SNP:OTHER_CORE_HITM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm_dp::offcore_response_0:ANY_CACHE_DRAM",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x7fff,
	  .fstr = "wsm_dp::OFFCORE_RESPONSE_0:ANY_REQUEST:UNCORE_HIT:OTHER_CORE_HIT_SNP:OTHER_CORE_HITM:REMOTE_CACHE_HITM:LOCAL_DRAM_AND_REMOTE_CACHE_HIT:REMOTE_DRAM:OTHER_LLC_MISS:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "nhm::offcore_response_0:k",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5201b7,
	  .codes[1] = 0xffff,
	  .fstr = "nhm::OFFCORE_RESPONSE_0:ANY_REQUEST:ANY_RESPONSE:k=1:u=0:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "nhm::offcore_response_0:local_dram",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x40ff,
	  .fstr = "nhm::OFFCORE_RESPONSE_0:ANY_REQUEST:LOCAL_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "nhm::offcore_response_0:any_llc_miss",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0xf8ff,
	  .fstr = "nhm::OFFCORE_RESPONSE_0:ANY_REQUEST:REMOTE_CACHE_HITM:REMOTE_CACHE_FWD:REMOTE_DRAM:LOCAL_DRAM:NON_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "nhm::offcore_response_0:any_dram",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x60ff,
	  .fstr = "nhm::OFFCORE_RESPONSE_0:ANY_REQUEST:REMOTE_DRAM:LOCAL_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "nhm::offcore_response_0:PF_IFETCH",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0xff40,
	  .fstr = "nhm::OFFCORE_RESPONSE_0:PF_IFETCH:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "nhm::offcore_response_0:ANY_DATA:LOCAL_DRAM",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x4033,
	  .fstr = "nhm::OFFCORE_RESPONSE_0:DMND_DATA_RD:DMND_RFO:PF_DATA_RD:PF_RFO:LOCAL_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "nhm::offcore_response_0:DMND_RFO:DMND_DATA_RD:LOCAL_DRAM:REMOTE_DRAM",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x6003,
	  .fstr = "nhm::OFFCORE_RESPONSE_0:DMND_DATA_RD:DMND_RFO:REMOTE_DRAM:LOCAL_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "amd64_k8_revg::DISPATCHED_FPU:0xff:k",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x52ff00ull,
	  .fstr = "amd64_k8_revg::DISPATCHED_FPU:0xff:k=1:u=0:e=0:i=0:c=0"
	},
	{ SRC_LINE,
	  .name = "amd64_k8_revg::DISPATCHED_FPU:0x4ff",
	  .ret  = PFM_ERR_ATTR,
	  .count = 0,
	},
	{ SRC_LINE,
	  .name = "amd64_fam10h_barcelona::DISPATCHED_FPU:0x4ff:u",
	  .ret  = PFM_ERR_ATTR
	},
	{ SRC_LINE,
	  .name = "amd64_fam10h_barcelona::DISPATCHED_FPU:0xff:u",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x51ff00ull,
	  .fstr = "amd64_fam10h_barcelona::DISPATCHED_FPU:0xff:k=0:u=1:e=0:i=0:c=0:h=0:g=0"
	},
	{ SRC_LINE,
	  .name = "wsm::inst_retired:0xff:k",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x52ffc0,
	  .fstr = "wsm::INST_RETIRED:0xff:k=1:u=0:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm::uops_issued:0xff:stall_cycles",
	  .ret  = PFM_ERR_ATTR,
	  .count = 0,
	},
	{ SRC_LINE,
	  .name = "wsm::uops_issued:0xff:0xf1",
	  .ret  = PFM_ERR_ATTR,
	  .count = 0,
	},
	{ SRC_LINE,
	  .name = "wsm::uops_issued:0xff=",
	  .ret  = PFM_ERR_ATTR_VAL,
	  .count = 0,
	},
	{ SRC_LINE,
	  .name = "wsm::uops_issued:123",
	  .ret  = PFM_ERR_ATTR,
	  .count = 0,
	},
	{ SRC_LINE,
	  .name = "wsm::uops_issued:0xfff",
	  .ret  = PFM_ERR_ATTR,
	  .count = 0,
	},
	{ SRC_LINE,
	  .name = "netburst::global_power_events",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x2600020f,
	  .codes[1] = 0x3d000,
	  .fstr = "netburst::global_power_events:RUNNING:k=1:u=1:e=0:cmpl=0:thr=0",
	},
	{ SRC_LINE,
	  .name = "netburst::global_power_events:RUNNING:u:k",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x2600020f,
	  .codes[1] = 0x3d000,
	  .fstr = "netburst::global_power_events:RUNNING:k=1:u=1:e=0:cmpl=0:thr=0",
	},
	{ SRC_LINE,
	  .name = "netburst::global_power_events:RUNNING:e",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x2600020f,
	  .codes[1] = 0x107d000,
	  .fstr = "netburst::global_power_events:RUNNING:k=1:u=1:e=1:cmpl=0:thr=0",
	},
	{ SRC_LINE,
	  .name = "netburst::global_power_events:RUNNING:cmpl:e:u",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x26000205,
	  .codes[1] = 0x10fd000,
	  .fstr = "netburst::global_power_events:RUNNING:k=0:u=1:e=1:cmpl=1:thr=0",
	},
	{ SRC_LINE,
	  .name = "netburst::global_power_events:RUNNING:cmpl:thr=8:u",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x26000205,
	  .codes[1] = 0x8fd000,
	  .fstr = "netburst::global_power_events:RUNNING:k=0:u=1:e=0:cmpl=1:thr=8",
	},
	{ SRC_LINE,
	  .name = "netburst::global_power_events:RUNNING:cmpl:thr=32:u",
	  .ret  = PFM_ERR_ATTR_VAL,
	  .count = 0,
	},
	{ SRC_LINE,
	  .name = "netburst::instr_completed:nbogus",
	  .ret  = PFM_ERR_NOTFOUND,
	  .count = 0,
	},
	{ SRC_LINE,
	  .name = "netburst_p::instr_completed:nbogus",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0xe00020f,
	  .codes[1] = 0x39000,
	  .fstr = "netburst_p::instr_completed:NBOGUS:k=1:u=1:e=0:cmpl=0:thr=0",
	},
	{ SRC_LINE,
	  .name = "snb::cpl_cycles:ring0_trans:u",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x155015c,
	  .fstr = "snb::CPL_CYCLES:RING0:k=0:u=1:e=1:i=0:c=1:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::cpl_cycles:ring0_trans:e=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x157015cull,
	},
	{ SRC_LINE,
	  .name = "snb::OFFCORE_REQUESTS_OUTSTanding:ALL_DATA_RD_CYCLES",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1530860,
	  .fstr = "snb::OFFCORE_REQUESTS_OUTSTANDING:ALL_DATA_RD:k=1:u=1:e=0:i=0:c=1:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::uops_issued:core_stall_cycles:u:k",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1f3010e,
	  .fstr = "snb::UOPS_ISSUED:ANY:k=1:u=1:e=0:i=1:c=1:t=1",
	},
	{ SRC_LINE,
	  .name = "snb::LLC_REFERences:k:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x724f2e,
	  .fstr = "snb::LAST_LEVEL_CACHE_REFERENCES:k=1:u=0:e=0:i=0:c=0:t=1",
	},
	{ SRC_LINE,
	  .name = "snb::ITLB:0x1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5301ae,
	  .fstr = "snb::ITLB:0x1:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:DMND_RFO:ANY_RESPONSE",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x10002,
	  .fstr = "snb::OFFCORE_RESPONSE_0:DMND_RFO:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:DMND_RFO:ANY_REQUEST",
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x18fff,
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:DMND_RFO",
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x10002,
	  .fstr = "snb::OFFCORE_RESPONSE_0:DMND_RFO:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:any_response",
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x18fff,
	  .fstr = "snb::OFFCORE_RESPONSE_0:ANY_REQUEST:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:NO_SUPP:SNP_NONE:PF_RFO:PF_IFETCH",
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x80020060,
	  .fstr = "snb::OFFCORE_RESPONSE_0:PF_RFO:PF_IFETCH:NO_SUPP:SNP_NONE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_1:DMND_RFO:ANY_RESPONSE",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x10002,
	  .fstr = "snb::OFFCORE_RESPONSE_1:DMND_RFO:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_1:DMND_RFO:ANY_REQUEST",
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x18fff,
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_1:DMND_RFO",
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x10002,
	  .fstr = "snb::OFFCORE_RESPONSE_1:DMND_RFO:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_1:any_response",
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x18fff,
	  .fstr = "snb::OFFCORE_RESPONSE_1:ANY_REQUEST:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_1:NO_SUPP:SNP_NONE:PF_RFO:PF_IFETCH",
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x80020060,
	  .fstr = "snb::OFFCORE_RESPONSE_1:PF_RFO:PF_IFETCH:NO_SUPP:SNP_NONE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_1:ANY_REQUEST:LLC_MISS_LOCAL_DRAM",
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x3f80408fffull,
	  .fstr = "snb::OFFCORE_RESPONSE_1:DMND_DATA_RD:DMND_RFO:DMND_IFETCH:WB:PF_DATA_RD:PF_RFO:PF_IFETCH:PF_LLC_DATA_RD:PF_LLC_RFO:PF_LLC_IFETCH:BUS_LOCKS:STRM_ST:OTHER:LLC_MISS_LOCAL_DRAM:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam14h_bobcat::MAB_REQUESTS:DC_BUFFER_0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x530068,
	  .fstr = "amd64_fam14h_bobcat::MAB_REQUESTS:DC_BUFFER_0:k=1:u=1:e=0:i=0:c=0:h=0:g=0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam14h_bobcat::MAB_REQUESTS:DC_BUFFER_0:DC_BUFFER_1",
	  .ret  = PFM_ERR_FEATCOMB,
	},
	{ SRC_LINE,
	  .name = "amd64_fam14h_bobcat::MAB_REQUESTS:DC_BUFFER_0:IC_BUFFER_0",
	  .ret  = PFM_ERR_FEATCOMB,
	},
	{ SRC_LINE,
	  .name = "amd64_fam14h_bobcat::MAB_REQUESTS:ANY_DC_BUFFER",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x530b68,
	  .fstr = "amd64_fam14h_bobcat::MAB_REQUESTS:ANY_DC_BUFFER:k=1:u=1:e=0:i=0:c=0:h=0:g=0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam14h_bobcat::MAB_REQUESTS:ANY_IC_BUFFER",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x530a68,
	  .fstr = "amd64_fam14h_bobcat::MAB_REQUESTS:ANY_IC_BUFFER:k=1:u=1:e=0:i=0:c=0:h=0:g=0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam14h_bobcat::MAB_REQUESTS:ANY_IC_BUFFER:IC_BUFFER_1",
	  .ret  = PFM_ERR_FEATCOMB,
	},

	{ SRC_LINE,
	  .name = "core::INST_RETIRED:ANY_P:e",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "core::INST_RETIRED:ANY_P:e:c=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x15700c0ull,
	},
	{ SRC_LINE,
	  .name = "atom::INST_RETIRED:ANY_P:e",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5700c0ull,
	},
	{ SRC_LINE,
	  .name = "atom::INST_RETIRED:ANY_P:e:c=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x15700c0ull,
	},
	{ SRC_LINE,
	  .name = "nhm::INST_RETIRED:ANY_P:e",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "nhm::INST_RETIRED:ANY_P:e:c=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x15700c0ull,
	},
	{ SRC_LINE,
	  .name = "wsm::INST_RETIRED:ANY_P:e",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "wsm::INST_RETIRED:ANY_P:e:c=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x15701c0ull,
	},
	{ SRC_LINE,
	  .name = "snb::INST_RETIRED:ANY_P:e",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "snb::INST_RETIRED:ANY_P:e:c=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x15700c0ull,
	},
	{ SRC_LINE,
	  .name = "snb::INST_RETIRED:ANY_P:e:c=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x15700c0ull,
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:any_request",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7ull,
	  .codes[1]= 0x18fffull,
	  .fstr = "snb::OFFCORE_RESPONSE_0:DMND_DATA_RD:DMND_RFO:DMND_IFETCH:WB:PF_DATA_RD:PF_RFO:PF_IFETCH:PF_LLC_DATA_RD:PF_LLC_RFO:PF_LLC_IFETCH:BUS_LOCKS:STRM_ST:OTHER:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:dmnd_data_rd",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7ull,
	  .codes[1]=0x10001ull,
	  .fstr = "snb::OFFCORE_RESPONSE_0:DMND_DATA_RD:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:dmnd_data_rd:llc_hite",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7ull,
	  .codes[1]=0x3f80080001ull,
	  .fstr = "snb::OFFCORE_RESPONSE_0:DMND_DATA_RD:LLC_HITE:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:dmnd_data_rd:llc_hite:snp_any",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7ull,
	  .codes[1]=0x3f80080001ull,
	  .fstr = "snb::OFFCORE_RESPONSE_0:DMND_DATA_RD:LLC_HITE:SNP_NONE:SNP_NOT_NEEDED:SNP_MISS:SNP_NO_FWD:SNP_FWD:HITM:NON_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:dmnd_data_rd:llc_hite:hitm",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7ull,
	  .codes[1]=0x1000080001ull,
	  .fstr = "snb::OFFCORE_RESPONSE_0:DMND_DATA_RD:LLC_HITE:HITM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:dmnd_data_rd:any_response",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7ull,
	  .codes[1]=0x10001ull,
	  .fstr = "snb::OFFCORE_RESPONSE_0:DMND_DATA_RD:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:dmnd_data_rd:any_response:snp_any",
	  .ret  = PFM_ERR_FEATCOMB,
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:dmnd_data_rd:any_response:llc_hitmesf",
	  .ret  = PFM_ERR_FEATCOMB,
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:any_response",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7ull,
	  .codes[1]=0x18fffull,
	  .fstr = "snb::OFFCORE_RESPONSE_0:ANY_REQUEST:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:l3_miss",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] =0x5301b7,
	  .codes[1] = 0x3f80408fffull,
	  .fstr = "snb::OFFCORE_RESPONSE_0:ANY_REQUEST:LLC_MISS_LOCAL_DRAM:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam11h_turion::MAB_REQUESTS:DC_BUFFER_0",
	  .ret  = PFM_ERR_NOTFOUND,
	},
	{ SRC_LINE,
	  .name = "amd64_fam11h_turion::RETIRED_INSTRUCTIONS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5300c0,
	  .fstr = "amd64_fam11h_turion::RETIRED_INSTRUCTIONS:k=1:u=1:e=0:i=0:c=0:h=0:g=0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam11h_turion::RETIRED_UOPS:k",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5200c1,
	  .fstr = "amd64_fam11h_turion::RETIRED_UOPS:k=1:u=0:e=0:i=0:c=0:h=0:g=0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam11h_turion::CPU_CLK_UNHALTED",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x530076,
	  .fstr = "amd64_fam11h_turion::CPU_CLK_UNHALTED:k=1:u=1:e=0:i=0:c=0:h=0:g=0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam11h_turion::RETIRED_UOPS:c=1:i",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1d300c1,
	  .fstr = "amd64_fam11h_turion::RETIRED_UOPS:k=1:u=1:e=0:i=1:c=1:h=0:g=0",
	},
	{ SRC_LINE,
	  .name = "ivb::ARITH:FPU_DIV",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1570414,
	  .fstr = "ivb::ARITH:FPU_DIV:k=1:u=1:e=1:i=0:c=1:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::INST_RETIRED:ALL",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5301c0,
	  .fstr = "ivb::INST_RETIRED:ALL:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::INST_RETIRED:ALL:k",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5201c0,
	  .fstr = "ivb::INST_RETIRED:ALL:k=1:u=0:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::INST_RETIRED:ALL:u",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5101c0,
	  .fstr = "ivb::INST_RETIRED:ALL:k=0:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::TLB_ACCESS:LOAD_STLB_HIT",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x53045f,
	  .fstr = "ivb::DTLB_LOAD_ACCESS:STLB_HIT:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::TLB_ACCESS:STLB_HIT",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x53045f,
	  .fstr = "ivb::DTLB_LOAD_ACCESS:STLB_HIT:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::DTLB_LOAD_ACCESS:STLB_HIT",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x53045f,
	  .fstr = "ivb::DTLB_LOAD_ACCESS:STLB_HIT:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::MOVE_ELIMINATION:INT_NOT_ELIMINATED",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x530158,
	  .fstr = "ivb::MOVE_ELIMINATION:INT_NOT_ELIMINATED:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::RESOURCE_STALLS:SB:RS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x530ca2,
	  .fstr = "ivb::RESOURCE_STALLS:RS:SB:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::RESOURCE_STALLS:ROB:RS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5314a2,
	  .fstr = "ivb::RESOURCE_STALLS:RS:ROB:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::UOPS_EXECUTED:THREAD",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5301b1,
	  .fstr = "ivb::UOPS_EXECUTED:THREAD:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::UOPS_EXECUTED:THREAD:e:c=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x15701b1,
	  .fstr = "ivb::UOPS_EXECUTED:THREAD:k=1:u=1:e=1:i=0:c=1:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::UOPS_EXECUTED:THREAD:e",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "ivb::UOPS_EXECUTED:THREAD:c=1:i",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1d301b1,
	  .fstr = "ivb::UOPS_EXECUTED:THREAD:k=1:u=1:e=0:i=1:c=1:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::CPU_CLK_UNHALTED:REF_P",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x53013c,
	  .fstr = "ivb::CPU_CLK_UNHALTED:REF_XCLK:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::DTLB_LOAD_MISSES:DEMAND_LD_MISS_CAUSES_A_WALK",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x538108,
	  .fstr = "ivb::DTLB_LOAD_MISSES:MISS_CAUSES_A_WALK:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::offcore_response_0:k",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5201b7,
	  .codes[1] = 0x18fff,
	  .fstr = "ivb::OFFCORE_RESPONSE_0:ANY_REQUEST:ANY_RESPONSE:k=1:u=0:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::offcore_response_0:LLC_MISS_LOCAL",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x3f80408fffull,
	  .fstr = "ivb::OFFCORE_RESPONSE_0:ANY_REQUEST:LLC_MISS_LOCAL:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::offcore_response_0:l3_miss",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] =0x5301b7,
	  .codes[1] = 0x3f80408fffull,
	  .fstr = "ivb::OFFCORE_RESPONSE_0:ANY_REQUEST:LLC_MISS_LOCAL:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::DTLB_LOAD_MISSES:STLB_HIT",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x53045f,
	  .fstr = "ivb::DTLB_LOAD_MISSES:STLB_HIT:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::DTLB_LOAD_MISSES:LARGE_WALK_COMPLETED:u",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x518808,
	  .fstr = "ivb::DTLB_LOAD_MISSES:LARGE_WALK_COMPLETED:k=0:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::l2_lines_in:i:i=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xd301f1,
	  .fstr = "snb::L2_LINES_IN:I:k=1:u=1:e=0:i=1:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::l2_lines_in:i=1:i",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xd301f1,
	  .fstr = "snb::L2_LINES_IN:I:k=1:u=1:e=0:i=1:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::l2_lines_in:i:i=0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5301f1,
	  .fstr = "snb::L2_LINES_IN:I:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::l2_lines_in:e:e=0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5304f1,
	  .fstr = "snb::L2_LINES_IN:E:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::l2_lines_in:e:e=1",
	  .ret  = PFM_ERR_ATTR,
	  .count = 0,
	},
	{ SRC_LINE,
	  .name = "snb::l2_lines_in:e:e=1:c=10",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xa5704f1,
	  .fstr = "snb::L2_LINES_IN:E:k=1:u=1:e=1:i=0:c=10:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_unc_cbo0::unc_clockticks",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5000ff,
	  .fstr = "snb_unc_cbo0::UNC_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "snb_unc_cbo1::unc_clockticks",
	  .ret  = PFM_ERR_NOTFOUND
	},
	{ SRC_LINE,
	  .name = "snb_unc_cbo2::unc_clockticks",
	  .ret  = PFM_ERR_NOTFOUND
	},
	{ SRC_LINE,
	  .name = "snb_unc_cbo3::unc_clockticks",
	  .ret  = PFM_ERR_NOTFOUND
	},
	{ SRC_LINE,
	  .name = "snb_unc_cbo1::UNC_CBO_CACHE_LOOKUP:STATE_MESI:READ_FILTER:c=1:i",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1d01f34,
	  .fstr = "snb_unc_cbo1::UNC_CBO_CACHE_LOOKUP:STATE_MESI:READ_FILTER:e=0:i=1:c=1",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo1::UNC_C_CLOCKTICKS:u",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo0::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "snbep_unc_cbo0::UNC_C_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo0::UNC_C_LLC_LOOKUP:DATA_READ",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x334,
	  .codes[1] = 0x7c0000,
	  .fstr = "snbep_unc_cbo0::UNC_C_LLC_LOOKUP:DATA_READ:STATE_MESIF:e=0:i=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo0::UNC_C_LLC_LOOKUP:ANY",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x1f34,
	  .codes[1] = 0x7c0000,
	  .fstr = "snbep_unc_cbo0::UNC_C_LLC_LOOKUP:ANY:STATE_MESIF:e=0:i=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo0::UNC_C_LLC_LOOKUP:DATA_READ:nf=1",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo0::UNC_C_LLC_LOOKUP:DATA_READ:tid=1",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo0::UNC_C_LLC_LOOKUP:NID",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo0::UNC_C_LLC_LOOKUP:NID:STATE_M",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo0::UNC_C_LLC_LOOKUP:NID:nf=3",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x4334,
	  .codes[1] = 0x7c0c00,
	  .fstr = "snbep_unc_cbo0::UNC_C_LLC_LOOKUP:DATA_READ:NID:STATE_MESIF:e=0:i=0:t=0:tf=0:cf=0:nf=3",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo0::UNC_C_LLC_LOOKUP:NID:STATE_M:nf=3",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x4334,
	  .codes[1] = 0x200c00,
	  .fstr = "snbep_unc_cbo0::UNC_C_LLC_LOOKUP:DATA_READ:NID:STATE_M:e=0:i=0:t=0:tf=0:cf=0:nf=3",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo0::UNC_C_LLC_LOOKUP:NID:nf=3:tid=1",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo0::UNC_C_TOR_INSERTS:OPCODE",
	  .ret  = PFM_ERR_UMASK,
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo0::UNC_C_TOR_INSERTS:WB",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1035,
	  .fstr = "snbep_unc_cbo0::UNC_C_TOR_INSERTS:WB:e=0:i=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo0::UNC_C_TOR_INSERTS:OPCODE:OPC_PCIWILF",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x135,
	  .codes[1] = 0xca000000,
	  .fstr = "snbep_unc_cbo0::UNC_C_TOR_INSERTS:OPCODE:OPC_PCIWILF:e=0:i=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo0::UNC_C_TOR_INSERTS:OPCODE:OPC_PCIWILF:nf=1",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo0::UNC_C_TOR_INSERTS:NID_OPCODE:OPC_PCIRDCUR:nf=1",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x4135,
	  .codes[1] = 0xcf000400,
	  .fstr = "snbep_unc_cbo0::UNC_C_TOR_INSERTS:NID_OPCODE:OPC_PCIRDCUR:e=0:i=0:t=0:tf=0:cf=0:nf=1",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo0::UNC_C_TOR_INSERTS:OPC_RFO:NID_OPCODE:nf=1",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x4135,
	  .codes[1] = 0xc0000400,
	  .fstr = "snbep_unc_cbo0::UNC_C_TOR_INSERTS:NID_OPCODE:OPC_RFO:e=0:i=0:t=0:tf=0:cf=0:nf=1",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_ha::UNC_H_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "snbep_unc_ha::UNC_H_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_ha::UNC_H_REQUESTS:READS:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1000301,
	  .fstr = "snbep_unc_ha::UNC_H_REQUESTS:READS:e=0:i=0:t=1",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_imc0::UNC_M_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xff,
	  .fstr = "snbep_unc_imc0::UNC_M_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_imc0::UNC_M_CLOCKTICKS:t=1",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "snbep_unc_imc0::UNC_M_CAS_COUNT:RD",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0304,
	  .fstr = "snbep_unc_imc0::UNC_M_CAS_COUNT:RD:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_imc0::UNC_M_POWER_CKE_CYCLES:RANK0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x183,
	  .fstr = "snbep_unc_imc0::UNC_M_POWER_CKE_CYCLES:RANK0:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_imc0::UNC_M_CAS_COUNT:WR",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xc04,
	  .fstr = "snbep_unc_imc0::UNC_M_CAS_COUNT:WR:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_pcu::UNC_P_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "snbep_unc_pcu::UNC_P_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_pcu::UNC_P_CLOCKTICKS:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1000000,
	  .fstr = "snbep_unc_pcu::UNC_P_CLOCKTICKS:e=0:i=0:t=1",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_pcu::UNC_P_CORE0_TRANSITION_CYCLES",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x200003,
	  .fstr = "snbep_unc_pcu::UNC_P_CORE0_TRANSITION_CYCLES:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "snbep_unc_pcu::UNC_P_FREQ_BAND1_CYCLES",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "snbep_unc_pcu::UNC_P_FREQ_BAND2_CYCLES",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "snbep_unc_pcu::UNC_P_FREQ_BAND3_CYCLES",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "snbep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:ff=32",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0xb,
	  .codes[1] = 0x20,
	  .fstr = "snbep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:e=0:i=0:t=0:ff=32",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_pcu::UNC_P_FREQ_BAND1_CYCLES:ff=16",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0xc,
	  .codes[1] = 0x1000,
	  .fstr = "snbep_unc_pcu::UNC_P_FREQ_BAND1_CYCLES:e=0:i=0:t=0:ff=16",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_pcu::UNC_P_FREQ_BAND2_CYCLES:ff=8",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0xd,
	  .codes[1] = 0x80000,
	  .fstr = "snbep_unc_pcu::UNC_P_FREQ_BAND2_CYCLES:e=0:i=0:t=0:ff=8",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_pcu::UNC_P_FREQ_BAND3_CYCLES:ff=40",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0xe,
	  .codes[1] = 0x28000000,
	  .fstr = "snbep_unc_pcu::UNC_P_FREQ_BAND3_CYCLES:e=0:i=0:t=0:ff=40",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:ff=32:e",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x4000b,
	  .codes[1] = 0x20,
	  .fstr = "snbep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:e=1:i=0:t=0:ff=32",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:ff=32:i",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x80000b,
	  .codes[1] = 0x20,
	  .fstr = "snbep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:e=0:i=1:t=0:ff=32",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:ff=32:e:i",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x84000b,
	  .codes[1] = 0x20,
	  .fstr = "snbep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:e=1:i=1:t=0:ff=32",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:ff=32:e:i:t=4",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x484000b,
	  .codes[1] = 0x20,
	  .fstr = "snbep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:e=1:i=1:t=4:ff=32",
	},
	{ SRC_LINE,
	  .name = "SNBEP_UNC_PCU::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x4080,
	  .fstr = "snbep_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0:e=0:i=0:t=0"
	},
	{ SRC_LINE,
	  .name = "SNBEP_UNC_PCU::UNC_P_POWER_STATE_OCCUPANCY:CORES_C3",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x8080,
	  .fstr = "snbep_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C3:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "SNBEP_UNC_PCU::UNC_P_POWER_STATE_OCCUPANCY:CORES_C6",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xc080,
	  .fstr = "snbep_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C6:e=0:i=0:t=0"
	},
	{ SRC_LINE,
	  .name = "SNBEP_UNC_PCU::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0:i",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x40004080,
	  .fstr = "snbep_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0:e=0:i=1:t=0"
	},
	{ SRC_LINE,
	  .name = "SNBEP_UNC_PCU::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0:i:e",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xc0004080,
	  .fstr = "snbep_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0:e=1:i=1:t=0"
	},
	{ SRC_LINE,
	  .name = "snbep_unc_qpi0::UNC_Q_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x14,
	  .fstr = "snbep_unc_qpi0::UNC_Q_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_qpi0::UNC_Q_RXL_FLITS_G0:DATA",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x201,
	  .fstr = "snbep_unc_qpi0::UNC_Q_RXL_FLITS_G0:DATA:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_qpi0::UNC_Q_RXL_FLITS_G0:IDLE:t=1:i",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1800101,
	  .fstr = "snbep_unc_qpi0::UNC_Q_RXL_FLITS_G0:IDLE:e=0:i=1:t=1",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_qpi0::UNC_Q_TXL_FLITS_G0:DATA",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x200,
	  .fstr = "snbep_unc_qpi0::UNC_Q_TXL_FLITS_G0:DATA:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_qpi0::UNC_Q_RXL_FLITS_G1:HOM",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x200602,
	  .fstr = "snbep_unc_qpi0::UNC_Q_RXL_FLITS_G1:HOM:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_qpi0::UNC_Q_TXL_FLITS_G1:HOM",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x200600,
	  .fstr = "snbep_unc_qpi0::UNC_Q_TXL_FLITS_G1:HOM:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_ubo::UNC_U_LOCK_CYCLES",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x44,
	  .fstr = "snbep_unc_ubo::UNC_U_LOCK_CYCLES:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_r2pcie::UNC_R2_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1,
	  .fstr = "snbep_unc_r2pcie::UNC_R2_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_r2pcie::UNC_R2_RING_AD_USED:ANY",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xf07,
	  .fstr = "snbep_unc_r2pcie::UNC_R2_RING_AD_USED:ANY:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_r3qpi0::UNC_R3_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1,
	  .fstr = "snbep_unc_r3qpi0::UNC_R3_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_r3qpi0::UNC_R3_TXR_CYCLES_FULL:e=0:i=0:t=0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x25,
	  .fstr = "snbep_unc_r3qpi0::UNC_R3_TXR_CYCLES_FULL:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_r3qpi1::UNC_R3_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1,
	  .fstr = "snbep_unc_r3qpi1::UNC_R3_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_r3qpi1::UNC_R3_TXR_CYCLES_FULL:e=0:i=0:t=0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x25,
	  .fstr = "snbep_unc_r3qpi1::UNC_R3_TXR_CYCLES_FULL:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "knc::cpu_clk_unhalted",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x53002a,
	  .fstr = "knc::CPU_CLK_UNHALTED:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "knc::instructions_executed",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x530016,
	  .fstr = "knc::INSTRUCTIONS_EXECUTED:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "knc::vpu_data_read",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x532000,
	  .fstr = "knc::VPU_DATA_READ:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "knc::vpu_data_read:t:c=1:i",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1f32000,
	  .fstr = "knc::VPU_DATA_READ:k=1:u=1:e=0:i=1:c=1:t=1",
	},
	{ SRC_LINE,
	  .name = "snb_ep::cpl_cycles:ring0_trans:u",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x155015c,
	  .fstr = "snb_ep::CPL_CYCLES:RING0:k=0:u=1:e=1:i=0:c=1:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_ep::cycle_activity:0x6:c=6",
	  .count = 1,
	  .codes[0] = 0x65306a3,
	  .fstr = "snb_ep::CYCLE_ACTIVITY:0x6:k=1:u=1:e=0:i=0:c=6:t=0",
	},

	{ SRC_LINE,
	  .name = "snb_ep::cpl_cycles:ring0_trans:e=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x157015cull,
	},
	{ SRC_LINE,
	  .name = "snb_ep::OFFCORE_REQUESTS_OUTSTanding:ALL_DATA_RD_CYCLES",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1530860,
	  .fstr = "snb_ep::OFFCORE_REQUESTS_OUTSTANDING:ALL_DATA_RD:k=1:u=1:e=0:i=0:c=1:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_ep::uops_issued:core_stall_cycles:u:k",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1f3010e,
	  .fstr = "snb_ep::UOPS_ISSUED:ANY:k=1:u=1:e=0:i=1:c=1:t=1",
	},
	{ SRC_LINE,
	  .name = "snb_ep::LLC_REFERences:k:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x724f2e,
	  .fstr = "snb_ep::LAST_LEVEL_CACHE_REFERENCES:k=1:u=0:e=0:i=0:c=0:t=1",
	},
	{ SRC_LINE,
	  .name = "snb_ep::ITLB:0x1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5301ae,
	  .fstr = "snb_ep::ITLB:0x1:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_ep::mem_load_uops_llc_miss_retired:local_dram",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5301d3,
	  .fstr = "snb_ep::MEM_LOAD_UOPS_LLC_MISS_RETIRED:LOCAL_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_ep::mem_load_uops_llc_miss_retired:remote_dram",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5304d3,
	  .fstr = "snb_ep::MEM_LOAD_UOPS_LLC_MISS_RETIRED:REMOTE_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_ep::offcore_response_0:DMND_RFO:ANY_RESPONSE",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x10002,
	  .fstr = "snb_ep::OFFCORE_RESPONSE_0:DMND_RFO:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_ep::offcore_response_0:DMND_RFO:ANY_REQUEST",
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x18fff,
	},
	{ SRC_LINE,
	  .name = "snb_ep::offcore_response_0:DMND_RFO",
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x10002,
	  .fstr = "snb_ep::OFFCORE_RESPONSE_0:DMND_RFO:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_ep::offcore_response_0:any_response",
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x18fff,
	  .fstr = "snb_ep::OFFCORE_RESPONSE_0:ANY_REQUEST:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_ep::offcore_response_0:NO_SUPP:SNP_NONE:PF_RFO:PF_IFETCH",
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x80020060,
	  .fstr = "snb_ep::OFFCORE_RESPONSE_0:PF_RFO:PF_IFETCH:NO_SUPP:SNP_NONE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_ep::offcore_response_1:DMND_RFO:ANY_RESPONSE",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x10002,
	  .fstr = "snb_ep::OFFCORE_RESPONSE_1:DMND_RFO:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_ep::offcore_response_1:DMND_RFO:ANY_REQUEST",
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x18fff,
	},
	{ SRC_LINE,
	  .name = "snb_ep::offcore_response_1:DMND_RFO",
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x10002,
	  .fstr = "snb_ep::OFFCORE_RESPONSE_1:DMND_RFO:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_ep::offcore_response_1:any_response",
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x18fff,
	  .fstr = "snb_ep::OFFCORE_RESPONSE_1:ANY_REQUEST:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_ep::offcore_response_1:NO_SUPP:SNP_NONE:PF_RFO:PF_IFETCH",
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x80020060,
	  .fstr = "snb_ep::OFFCORE_RESPONSE_1:PF_RFO:PF_IFETCH:NO_SUPP:SNP_NONE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_ep::offcore_response_1:ANY_REQUEST:LLC_MISS_LOCAL_DRAM",
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x3f80408fffull,
	  .fstr = "snb_ep::OFFCORE_RESPONSE_1:DMND_DATA_RD:DMND_RFO:DMND_IFETCH:WB:PF_DATA_RD:PF_RFO:PF_IFETCH:PF_LLC_DATA_RD:PF_LLC_RFO:PF_LLC_IFETCH:BUS_LOCKS:STRM_ST:OTHER:LLC_MISS_LOCAL_DRAM:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_ep::offcore_response_1:ANY_REQUEST:LLC_MISS_REMOTE_DRAM",
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x3fff808fffull,
	  .fstr = "snb_ep::OFFCORE_RESPONSE_1:DMND_DATA_RD:DMND_RFO:DMND_IFETCH:WB:PF_DATA_RD:PF_RFO:PF_IFETCH:PF_LLC_DATA_RD:PF_LLC_RFO:PF_LLC_IFETCH:BUS_LOCKS:STRM_ST:OTHER:LLC_MISS_REMOTE_DRAM:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_ep::offcore_response_0:l3_miss",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] =0x5301b7,
	  .codes[1] = 0x3fffc08fffull,
	  .fstr = "snb_ep::OFFCORE_RESPONSE_0:ANY_REQUEST:LLC_MISS_LOCAL_DRAM:LLC_MISS_REMOTE_DRAM:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0",
	},

	{ SRC_LINE,
	  .name = "snb_ep::mem_trans_retired:latency_above_threshold",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0]=0x5301cd,
	  .codes[1] = 3,
	  .fstr = "snb_ep::MEM_TRANS_RETIRED:LATENCY_ABOVE_THRESHOLD:k=1:u=1:e=0:i=0:c=0:t=0:ldlat=3",
	},
	{ SRC_LINE,
	  .name = "snb_ep::mem_trans_retired:latency_above_threshold:ldlat=2",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "snb_ep::mem_trans_retired:latency_above_threshold:ldlat=3",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0]=0x5301cd,
	  .codes[1] = 3,
	  .fstr = "snb_ep::MEM_TRANS_RETIRED:LATENCY_ABOVE_THRESHOLD:k=1:u=1:e=0:i=0:c=0:t=0:ldlat=3",
	},
	{ SRC_LINE,
	  .name = "snb_ep::mem_trans_retired:latency_above_threshold:ldlat=1000000",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "snb::mem_trans_retired:latency_above_threshold",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301cd,
	  .codes[1] = 3,
	  .fstr = "snb::MEM_TRANS_RETIRED:LATENCY_ABOVE_THRESHOLD:k=1:u=1:e=0:i=0:c=0:t=0:ldlat=3",
	},
	{ SRC_LINE,
	  .name = "snb::mem_trans_retired:latency_above_threshold:ldlat=2",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "snb::mem_trans_retired:latency_above_threshold:ldlat=3",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301cd,
	  .codes[1] = 3,
	  .fstr = "snb::MEM_TRANS_RETIRED:LATENCY_ABOVE_THRESHOLD:k=1:u=1:e=0:i=0:c=0:t=0:ldlat=3",
	},
	{ SRC_LINE,
	  .name = "snb::mem_trans_retired:latency_above_threshold:ldlat=1000000",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "snb::cycle_activity:0x6:c=6",
	  .count = 1,
	  .codes[0] = 0x65306a3,
	  .fstr = "snb::CYCLE_ACTIVITY:0x6:k=1:u=1:e=0:i=0:c=6:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::mem_trans_retired:latency_above_threshold",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301cd,
	  .codes[1] = 3,
	  .fstr = "ivb::MEM_TRANS_RETIRED:LATENCY_ABOVE_THRESHOLD:k=1:u=1:e=0:i=0:c=0:t=0:ldlat=3",
	},
	{ SRC_LINE,
	  .name = "ivb::mem_trans_retired:latency_above_threshold:ldlat=2",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "ivb::mem_trans_retired:latency_above_threshold:ldlat=3:u",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5101cd,
	  .codes[1] = 3,
	  .fstr = "ivb::MEM_TRANS_RETIRED:LATENCY_ABOVE_THRESHOLD:k=0:u=1:e=0:i=0:c=0:t=0:ldlat=3",
	},
	{ SRC_LINE,
	  .name = "ivb::mem_trans_retired:latency_above_threshold:ldlat=1000000",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "nhm::mem_inst_retired:latency_above_threshold",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x53100b,
	  .codes[1] = 3,
	  .fstr = "nhm::MEM_INST_RETIRED:LATENCY_ABOVE_THRESHOLD:k=1:u=1:e=0:i=0:c=0:t=0:ldlat=3",
	},
	{ SRC_LINE,
	  .name = "nhm::mem_inst_retired:latency_above_threshold:ldlat=2",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "nhm::mem_inst_retired:latency_above_threshold:ldlat=3",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x53100b,
	  .codes[1] = 3,
	  .fstr = "nhm::MEM_INST_RETIRED:LATENCY_ABOVE_THRESHOLD:k=1:u=1:e=0:i=0:c=0:t=0:ldlat=3",
	},
	{ SRC_LINE,
	  .name = "nhm::mem_inst_retired:latency_above_threshold:ldlat=1000000",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "wsm::mem_inst_retired:latency_above_threshold",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x53100b,
	  .codes[1] = 3,
	  .fstr = "wsm::MEM_INST_RETIRED:LATENCY_ABOVE_THRESHOLD:k=1:u=1:e=0:i=0:c=0:t=0:ldlat=3",
	},
	{ SRC_LINE,
	  .name = "wsm::mem_inst_retired:latency_above_threshold:ldlat=2",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "wsm::mem_inst_retired:latency_above_threshold:ldlat=3",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x53100b,
	  .codes[1] = 3,
	  .fstr = "wsm::MEM_INST_RETIRED:LATENCY_ABOVE_THRESHOLD:k=1:u=1:e=0:i=0:c=0:t=0:ldlat=3",
	},
	{ SRC_LINE,
	  .name = "wsm::mem_inst_retired:latency_above_threshold:ldlat=1000000",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "amd64_fam15h_interlagos::LINK_TRANSMIT_BANDWIDTH_LINK_0:NOP_DW_SENT",
	  .ret = PFM_ERR_NOTFOUND, /* event in Northbridge PMU */
	},
	{ SRC_LINE,
	  .name = "amd64_fam15h_nb::LINK_TRANSMIT_BANDWIDTH_LINK_0:NOP_DW_SENT",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5308f6,
	  .fstr = "amd64_fam15h_nb::LINK_TRANSMIT_BANDWIDTH_LINK_0:NOP_DW_SENT:SUBLINK_0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam15h_nb::LINK_TRANSMIT_BANDWIDTH_LINK_0:ALL",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x533ff6,
	  .fstr = "amd64_fam15h_nb::LINK_TRANSMIT_BANDWIDTH_LINK_0:ALL:SUBLINK_0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam15h_nb::LINK_TRANSMIT_BANDWIDTH_LINK_0:ALL:SUBLINK_1",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x53bff6,
	  .fstr = "amd64_fam15h_nb::LINK_TRANSMIT_BANDWIDTH_LINK_0:ALL:SUBLINK_1",
	},
	{ SRC_LINE,
	  .name = "amd64_fam15h_nb::LINK_TRANSMIT_BANDWIDTH_LINK_0:COMMAND_DW_SENT:DATA_DW_SENT",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5303f6,
	  .fstr = "amd64_fam15h_nb::LINK_TRANSMIT_BANDWIDTH_LINK_0:COMMAND_DW_SENT:DATA_DW_SENT:SUBLINK_0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam15h_interlagos::DISPATCHED_FPU_OPS:0x4ff:u",
	  .ret  = PFM_ERR_ATTR
	},
	{ SRC_LINE,
	  .name = "amd64_fam15h_interlagos::DISPATCHED_FPU_OPS:0xff:u",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x51ff00ull,
	  .fstr = "amd64_fam15h_interlagos::DISPATCHED_FPU_OPS:0xff:k=0:u=1:e=0:i=0:c=0:h=0:g=0"
	},
	{ SRC_LINE,
	  .name = "amd64_fam15h_nb::READ_REQUEST_TO_L3_CACHE:read_block_modify:core_3",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x4005334e0ull,
	  .fstr = "amd64_fam15h_nb::READ_REQUEST_TO_L3_CACHE:READ_BLOCK_MODIFY:CORE_3",
	},
	{ SRC_LINE,
	  .name = "amd64_fam15h_nb::READ_REQUEST_TO_L3_CACHE",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x40053f7e0ull,
	  .fstr = "amd64_fam15h_nb::READ_REQUEST_TO_L3_CACHE:READ_BLOCK_ANY:ANY_CORE",
	},
	{ SRC_LINE,
	  .name = "amd64_fam15h_nb::READ_REQUEST_TO_L3_CACHE:READ_BLOCK_EXCLUSIVE:PREFETCH:READ_BLOCK_MODIFY:core_4",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x400534de0ull,
	  .fstr = "amd64_fam15h_nb::READ_REQUEST_TO_L3_CACHE:READ_BLOCK_EXCLUSIVE:READ_BLOCK_MODIFY:PREFETCH:CORE_4",
	},
	{ SRC_LINE,
	  .name = "amd64_fam15h_nb::READ_REQUEST_TO_L3_CACHE:read_block_any:prefetch:core_1",
	  .ret  = PFM_ERR_FEATCOMB, /* must use individual umasks to combine with prefetch */
	},
	{ SRC_LINE,
	  .name = "amd64_fam15h_nb::READ_REQUEST_TO_L3_CACHE:read_block_any:prefetch:core_1:core_3",
	  .ret  = PFM_ERR_FEATCOMB, /* core umasks cannot be combined */
	},
	{ SRC_LINE,
	  .name = "amd64_fam15h_nb::READ_REQUEST_TO_L3_CACHE:prefetch:core_0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x4005308e0ull,
	  .fstr = "amd64_fam15h_nb::READ_REQUEST_TO_L3_CACHE:PREFETCH:CORE_0",
	},
	{ SRC_LINE,
	  .name = "ivb_ep::mem_load_uops_llc_miss_retired:local_dram",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5301d3,
	  .fstr = "ivb_ep::MEM_LOAD_UOPS_LLC_MISS_RETIRED:LOCAL_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb_ep::mem_load_uops_llc_miss_retired:remote_dram",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x530cd3,
	  .fstr = "ivb_ep::MEM_LOAD_UOPS_LLC_MISS_RETIRED:REMOTE_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb_ep::mem_load_uops_llc_miss_retired:remote_hitm",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5310d3,
	  .fstr = "ivb_ep::MEM_LOAD_UOPS_LLC_MISS_RETIRED:REMOTE_HITM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb_ep::mem_load_uops_llc_miss_retired:remote_fwd",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5320d3,
	  .fstr = "ivb_ep::MEM_LOAD_UOPS_LLC_MISS_RETIRED:REMOTE_FWD:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::mem_load_uops_llc_miss_retired:remote_dram",
	  .ret = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "ivb::cycle_activity:0x6:c=6",
	  .count = 1,
	  .codes[0] = 0x65306a3,
	  .fstr = "ivb::CYCLE_ACTIVITY:0x6:k=1:u=1:e=0:i=0:c=6:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb_ep::offcore_response_0:any_request:LLC_MISS_REMOTE",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x3fff808fffULL,
	  .fstr = "ivb_ep::OFFCORE_RESPONSE_0:DMND_DATA_RD:DMND_RFO:DMND_IFETCH:WB:PF_DATA_RD:PF_RFO:PF_IFETCH:PF_LLC_DATA_RD:PF_LLC_RFO:PF_LLC_IFETCH:BUS_LOCKS:STRM_ST:OTHER:LLC_MISS_REMOTE_DRAM:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0"
	},
	{ SRC_LINE,
	  .name = "ivb_ep::offcore_response_0:l3_miss",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] =0x5301b7,
	  .codes[1] = 0x3fffc08fffull,
	  .fstr = "ivb_ep::OFFCORE_RESPONSE_0:ANY_REQUEST:LLC_MISS_LOCAL:LLC_MISS_REMOTE_DRAM:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hsw::mem_trans_retired:latency_above_threshold:ldlat=3:u",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5101cd,
	  .codes[1] = 3,
	  .fstr = "hsw::MEM_TRANS_RETIRED:LOAD_LATENCY:k=0:u=1:e=0:i=0:c=0:t=0:ldlat=3:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw::mem_trans_retired:latency_above_threshold:ldlat=1000000",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "hsw::mem_trans_retired:load_latency",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301cd,
	  .codes[1] = 3,
	  .fstr = "hsw::MEM_TRANS_RETIRED:LOAD_LATENCY:k=1:u=1:e=0:i=0:c=0:t=0:ldlat=3:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw::mem_trans_retired:load_latency:ldlat=1000000",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "hsw::mem_trans_retired:latency_above_threshold:ldlat=2:intx=0:intxcp=0",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "hsw::inst_Retired:any_p:intx",
	  .count = 1,
	  .codes[0] = 0x1005300c0ull,
	  .fstr = "hsw::INST_RETIRED:ANY_P:k=1:u=1:e=0:i=0:c=0:t=0:intx=1:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw::inst_Retired:any_p:intx:intxcp",
	  .count = 1,
	  .codes[0] = 0x3005300c0ull,
	  .fstr = "hsw::INST_RETIRED:ANY_P:k=1:u=1:e=0:i=0:c=0:t=0:intx=1:intxcp=1",
	},
	{ SRC_LINE,
	  .name = "hsw::inst_Retired:any_p:intx=0:intxcp",
	  .count = 1,
	  .codes[0] = 0x2005300c0ull,
	  .fstr = "hsw::INST_RETIRED:ANY_P:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=1",
	},
	{ SRC_LINE,
	  .name = "hsw::cycle_activity:cycles_l2_pending",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x15301a3,
	  .fstr = "hsw::CYCLE_ACTIVITY:CYCLES_L2_PENDING:k=1:u=1:e=0:i=0:c=1:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw::cycle_activity:cycles_l2_pending:c=8",
	  .ret = PFM_ERR_ATTR_SET
	},
	{ SRC_LINE,
	  .name = "hsw::hle_retired:aborted",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5304c8,
	  .fstr = "hsw::HLE_RETIRED:ABORTED:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw::rtm_retired:aborted",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5304c9,
	  .fstr = "hsw::RTM_RETIRED:ABORTED:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw::offcore_response_0:k:intx=1",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x1005201b7ull,
	  .codes[1] = 0x18fff,
	  .fstr = "hsw::OFFCORE_RESPONSE_0:ANY_REQUEST:ANY_RESPONSE:k=1:u=0:e=0:i=0:c=0:t=0:intx=1:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw::offcore_response_0:any_request",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x18fff,
	  .fstr = "hsw::OFFCORE_RESPONSE_0:DMND_DATA_RD:DMND_RFO:DMND_CODE_RD:WB:PF_DATA_RD:PF_RFO:PF_CODE_RD:PF_L3_DATA_RD:PF_L3_RFO:PF_L3_CODE_RD:SPLIT_LOCK_UC_LOCK:STRM_ST:OTHER:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw::offcore_response_0:any_request:any_response:L3_MISS_LOCAL",
	  .ret  = PFM_ERR_FEATCOMB,
	},
	{ SRC_LINE,
	  .name = "hsw::offcore_response_0:split_lock_uc_lock",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x10400,
	  .fstr = "hsw::OFFCORE_RESPONSE_0:SPLIT_LOCK_UC_LOCK:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw::offcore_response_0:any_ifetch",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x10240,
	  .fstr = "hsw::OFFCORE_RESPONSE_0:ANY_IFETCH:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw::offcore_response_0:L3_HITF",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "hsw::offcore_response_0:LLC_MISS_LOCAL",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "hsw::offcore_response_0:L3_HIT:u",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5101b7,
	  .codes[1] = 0x3f801c8fffull,
	  .fstr = "hsw::OFFCORE_RESPONSE_0:ANY_REQUEST:L3_HITM:L3_HITE:L3_HITS:SNP_ANY:k=0:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw::offcore_response_0:ANY_DATA",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x10091,
	  .fstr = "hsw::OFFCORE_RESPONSE_0:DMND_DATA_RD:PF_DATA_RD:PF_L3_DATA_RD:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw::offcore_response_0:DMND_DATA_RD:L3_HITS:SNP_FWD",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x800100001ull,
	  .fstr = "hsw::OFFCORE_RESPONSE_0:DMND_DATA_RD:L3_HITS:SNP_FWD:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw::offcore_response_0:l3_miss",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] =0x5301b7,
	  .codes[1] = 0x3f80408fffull,
	  .fstr = "hsw::OFFCORE_RESPONSE_0:ANY_REQUEST:L3_MISS_LOCAL:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "ivb_unc_cbo0::unc_clockticks",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5000ff,
	  .fstr = "ivb_unc_cbo0::UNC_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "ivb_unc_cbo1::unc_clockticks",
	  .ret  = PFM_ERR_NOTFOUND
	},
	/*
	 * RAPL note:
	 * we can only use the PKG event because it is the only one available
	 * on all processors. The GPU is client only, the CORES is only certain early CPUs
	 */
	{ SRC_LINE,
	  .name = "rapl::rapl_energy_pkg",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x2,
	  .fstr = "rapl::RAPL_ENERGY_PKG",
	},
	{ SRC_LINE,
	  .name = "rapl::rapl_energy_pkg:u",
	  .ret = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "slm::offcore_response_0:snp_hitm",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1]=0x100001ffffull,
	  .fstr = "slm::OFFCORE_RESPONSE_0:ANY_REQUEST:ANY_RESPONSE:SNP_HITM:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "slm::offcore_response_0:any_data",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1]=0x12011,
	  .fstr = "slm::OFFCORE_RESPONSE_0:DMND_DATA_RD:PF_L2_DATA_RD:PF_L1_DATA_RD:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "slm::offcore_response_0:uc_ifetch",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1]=0x10200,
	  .fstr = "slm::OFFCORE_RESPONSE_0:UC_IFETCH:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "slm::offcore_response_0:any_ifetch",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1]=0x10244,
	  .fstr = "slm::OFFCORE_RESPONSE_0:DMND_IFETCH:PF_IFETCH:UC_IFETCH:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "slm::offcore_response_1:snp_hitm",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5302b7,
	  .codes[1]=0x100001ffffull,
	  .fstr = "slm::OFFCORE_RESPONSE_1:ANY_REQUEST:ANY_RESPONSE:SNP_HITM:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "slm::offcore_response_1:any_data",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5302b7,
	  .codes[1]=0x12011,
	  .fstr = "slm::OFFCORE_RESPONSE_1:DMND_DATA_RD:PF_L2_DATA_RD:PF_L1_DATA_RD:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "slm::offcore_response_1:uc_ifetch",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5302b7,
	  .codes[1]=0x10200,
	  .fstr = "slm::OFFCORE_RESPONSE_1:UC_IFETCH:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "slm::offcore_response_1:any_ifetch",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5302b7,
	  .codes[1]=0x10244,
	  .fstr = "slm::OFFCORE_RESPONSE_1:DMND_IFETCH:PF_IFETCH:UC_IFETCH:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "slm::decode_restriction:predecode_wrong",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5301e9,
	  .fstr = "slm::DECODE_RESTRICTION:PREDECODE_WRONG:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "slm::rs_full_stall:any",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x531fcb,
	  .fstr = "slm::RS_FULL_STALL:ALL:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "slm::no_alloc_cycles:any",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x533fca,
	  .fstr = "slm::NO_ALLOC_CYCLES:ALL:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "slm::no_alloc_cycles:any:t=1",
	  .ret = PFM_ERR_ATTR
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_irp::unc_i_clockticks",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "ivbep_unc_irp::UNC_I_CLOCKTICKS:e=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_irp::unc_i_transactions:reads",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x115,
	  .fstr = "ivbep_unc_irp::UNC_I_TRANSACTIONS:READS:e=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_irp::unc_i_transactions:reads:c=1:i",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_irp::unc_i_transactions:reads:t=6",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x6000115,
	  .fstr = "ivbep_unc_irp::UNC_I_TRANSACTIONS:READS:e=0:t=6",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_cbo1::UNC_C_CLOCKTICKS:u",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_cbo0::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "ivbep_unc_cbo0::UNC_C_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_cbo0::UNC_C_LLC_LOOKUP:DATA_READ",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x334,
	  .codes[1] = 0x7e0000,
	  .fstr = "ivbep_unc_cbo0::UNC_C_LLC_LOOKUP:DATA_READ:STATE_MESIF:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_cbo0::UNC_C_LLC_LOOKUP:DATA_READ:nf=1",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_cbo0::UNC_C_LLC_LOOKUP",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x1134,
	  .codes[1] = 0x7e0000,
	  .fstr = "ivbep_unc_cbo0::UNC_C_LLC_LOOKUP:ANY:STATE_MESIF:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_cbo0::UNC_C_LLC_LOOKUP:NID:STATE_M",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_cbo0::UNC_C_LLC_LOOKUP:NID:nf=3",
	  .ret  = PFM_SUCCESS,
	  .count = 3,
	  .codes[0] = 0x5134,
	  .codes[1] = 0x7e0000,
	  .codes[2] = 0x3,
	  .fstr = "ivbep_unc_cbo0::UNC_C_LLC_LOOKUP:ANY:NID:STATE_MESIF:e=0:t=0:tf=0:cf=0:nf=3",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_cbo0::UNC_C_LLC_LOOKUP:NID:STATE_M:tid=1",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_cbo0::UNC_C_LLC_LOOKUP:DATA_READ:WRITE",
	  .ret  = PFM_ERR_FEATCOMB,
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_cbo0::UNC_C_LLC_LOOKUP:WRITE:NID:nf=3:tf=1:e:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 3,
	  .codes[0] = 0x10c4534,
	  .codes[1] = 0x7e0001,
	  .codes[2] = 0x3,
	  .fstr = "ivbep_unc_cbo0::UNC_C_LLC_LOOKUP:WRITE:NID:STATE_MESIF:e=1:t=1:tf=1:cf=0:nf=3",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_cbo0::UNC_C_LLC_VICTIMS",
	  .ret  = PFM_ERR_UMASK,
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_cbo0::UNC_C_LLC_VICTIMS:NID",
	  .ret  = PFM_ERR_UMASK,
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_cbo0::UNC_C_LLC_VICTIMS:NID:nf=1",
	  .ret  = PFM_ERR_UMASK,
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_cbo0::UNC_C_LLC_VICTIMS:STATE_M",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x137,
	  .fstr = "ivbep_unc_cbo0::UNC_C_LLC_VICTIMS:STATE_M:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_cbo0::UNC_C_LLC_VICTIMS:STATE_M:STATE_S",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x537,
	  .fstr = "ivbep_unc_cbo0::UNC_C_LLC_VICTIMS:STATE_M:STATE_S:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_cbo0::UNC_C_LLC_VICTIMS:STATE_M:STATE_S:NID:nf=1",
	  .ret  = PFM_SUCCESS,
	  .count = 3,
	  .codes[0] = 0x4537,
	  .codes[1] = 0x0,
	  .codes[2] = 0x1,
	  .fstr = "ivbep_unc_cbo0::UNC_C_LLC_VICTIMS:STATE_M:STATE_S:NID:e=0:t=0:tf=0:cf=0:nf=1",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_cbo0::UNC_C_TOR_INSERTS:OPCODE",
	  .ret  = PFM_ERR_UMASK,
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_cbo0::UNC_C_TOR_INSERTS:WB",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1035,
	  .fstr = "ivbep_unc_cbo0::UNC_C_TOR_INSERTS:WB:e=0:t=0:tf=0:cf=0:isoc=0:nc=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_cbo0::UNC_C_TOR_INSERTS:OPCODE:OPC_PCIWILF",
	  .ret  = PFM_SUCCESS,
	  .count = 3,
	  .codes[0] = 0x135,
	  .codes[1] = 0x0,
	  .codes[2] = 0x19400000ull,
	  .fstr = "ivbep_unc_cbo0::UNC_C_TOR_INSERTS:OPCODE:OPC_PCIWILF:e=0:t=0:tf=0:cf=0:isoc=0:nc=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_cbo0::UNC_C_TOR_INSERTS:OPCODE:OPC_PCIWILF:isoc=1",
	  .ret  = PFM_SUCCESS,
	  .count = 3,
	  .codes[0] = 0x135,
	  .codes[1] = 0x0,
	  .codes[2] = 0x99400000ull,
	  .fstr = "ivbep_unc_cbo0::UNC_C_TOR_INSERTS:OPCODE:OPC_PCIWILF:e=0:t=0:tf=0:cf=0:isoc=1:nc=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_cbo0::UNC_C_TOR_INSERTS:OPCODE:OPC_PCIWILF:nf=1",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_cbo0::UNC_C_TOR_INSERTS:NID_OPCODE:OPC_PCIRDCUR:nf=1",
	  .ret  = PFM_SUCCESS,
	  .count = 3,
	  .codes[0] = 0x4135,
	  .codes[1] = 0x0,
	  .codes[2] = 0x19e00001ull,
	  .fstr = "ivbep_unc_cbo0::UNC_C_TOR_INSERTS:NID_OPCODE:OPC_PCIRDCUR:e=0:t=0:tf=0:cf=0:nf=1:isoc=0:nc=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_cbo0::UNC_C_TOR_INSERTS:OPC_RFO:NID_OPCODE:nf=1",
	  .ret  = PFM_SUCCESS,
	  .count = 3,
	  .codes[0] = 0x4135,
	  .codes[1] = 0x0,
	  .codes[2] = 0x18000001ull,
	  .fstr = "ivbep_unc_cbo0::UNC_C_TOR_INSERTS:NID_OPCODE:OPC_RFO:e=0:t=0:tf=0:cf=0:nf=1:isoc=0:nc=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_cbo0::UNC_C_TOR_OCCUPANCY:MISS_REMOTE",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x8a36,
	  .fstr = "ivbep_unc_cbo0::UNC_C_TOR_OCCUPANCY:MISS_REMOTE:e=0:t=0:tf=0:cf=0:isoc=0:nc=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_pcu::UNC_P_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "ivbep_unc_pcu::UNC_P_CLOCKTICKS:e=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_pcu::UNC_P_CLOCKTICKS:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1000000,
	  .fstr = "ivbep_unc_pcu::UNC_P_CLOCKTICKS:e=0:t=1",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_pcu::UNC_P_CORE0_TRANSITION_CYCLES",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x70,
	  .fstr = "ivbep_unc_pcu::UNC_P_CORE0_TRANSITION_CYCLES:e=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_pcu::UNC_P_FREQ_BAND1_CYCLES",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_pcu::UNC_P_FREQ_BAND2_CYCLES",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_pcu::UNC_P_FREQ_BAND3_CYCLES",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:ff=32",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0xb,
	  .codes[1] = 0x20,
	  .fstr = "ivbep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:e=0:t=0:ff=32",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_pcu::UNC_P_FREQ_BAND1_CYCLES:ff=16",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0xc,
	  .codes[1] = 0x1000,
	  .fstr = "ivbep_unc_pcu::UNC_P_FREQ_BAND1_CYCLES:e=0:t=0:ff=16",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_pcu::UNC_P_FREQ_BAND2_CYCLES:ff=8",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0xd,
	  .codes[1] = 0x80000,
	  .fstr = "ivbep_unc_pcu::UNC_P_FREQ_BAND2_CYCLES:e=0:t=0:ff=8",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_pcu::UNC_P_FREQ_BAND3_CYCLES:ff=40",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0xe,
	  .codes[1] = 0x28000000,
	  .fstr = "ivbep_unc_pcu::UNC_P_FREQ_BAND3_CYCLES:e=0:t=0:ff=40",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:ff=32:e",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x4000b,
	  .codes[1] = 0x20,
	  .fstr = "ivbep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:e=1:t=0:ff=32",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:ff=32:t=24",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x1800000b,
	  .codes[1] = 0x20,
	  .fstr = "ivbep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:e=0:t=24:ff=32",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:ff=32:e:t=4",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x404000b,
	  .codes[1] = 0x20,
	  .fstr = "ivbep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:e=1:t=4:ff=32",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x4080,
	  .fstr = "ivbep_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0:e=0:i=0:t=0"
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C3",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x8080,
	  .fstr = "ivbep_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C3:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "IVBEP_UNC_PCU::UNC_P_POWER_STATE_OCCUPANCY:CORES_C6",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xc080,
	  .fstr = "ivbep_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C6:e=0:i=0:t=0"
	},
	{ SRC_LINE,
	  .name = "IVBEP_UNC_PCU::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0:t=6",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x6004080,
	  .fstr = "ivbep_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0:e=0:i=0:t=6"
	},
	{ SRC_LINE,
	  .name = "IVBEP_UNC_PCU::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0:t=6:i",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x46004080,
	  .fstr = "ivbep_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0:e=0:i=1:t=6"
	},
	{ SRC_LINE,
	  .name = "IVBEP_UNC_PCU::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0:t=6:i:e",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xc6004080,
	  .fstr = "ivbep_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0:e=1:i=1:t=6"
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_pcu::UNC_P_DEMOTIONS_CORE10",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x42,
	  .fstr = "ivbep_unc_pcu::UNC_P_DEMOTIONS_CORE10:e=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_pcu::UNC_P_DEMOTIONS_CORE14",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x46,
	  .fstr = "ivbep_unc_pcu::UNC_P_DEMOTIONS_CORE14:e=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_ha0::UNC_H_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "ivbep_unc_ha0::UNC_H_CLOCKTICKS:e=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_ha1::UNC_H_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "ivbep_unc_ha1::UNC_H_CLOCKTICKS:e=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_ha1::UNC_H_REQUESTS:READS:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1000301,
	  .fstr = "ivbep_unc_ha1::UNC_H_REQUESTS:READS:e=0:t=1",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_ha0::UNC_H_IMC_WRITES:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1000f1a,
	  .fstr = "ivbep_unc_ha0::UNC_H_IMC_WRITES:ALL:e=0:t=1",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_ha0::UNC_H_IMC_READS:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1000117,
	  .fstr = "ivbep_unc_ha0::UNC_H_IMC_READS:NORMAL:e=0:t=1",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_imc0::UNC_M_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xff,
	  .fstr = "ivbep_unc_imc0::UNC_M_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_imc0::UNC_M_CLOCKTICKS:t=1",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_imc0::UNC_M_DCLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "ivbep_unc_imc0::UNC_M_DCLOCKTICKS:e=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_imc4::UNC_M_DCLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "ivbep_unc_imc4::UNC_M_DCLOCKTICKS:e=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_imc0::UNC_M_CAS_COUNT:RD",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0304,
	  .fstr = "ivbep_unc_imc0::UNC_M_CAS_COUNT:RD:e=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_imc0::UNC_M_PRE_COUNT:WR",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0802,
	  .fstr = "ivbep_unc_imc0::UNC_M_PRE_COUNT:WR:e=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_imc0::UNC_M_POWER_CKE_CYCLES:RANK0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x183,
	  .fstr = "ivbep_unc_imc0::UNC_M_POWER_CKE_CYCLES:RANK0:e=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_imc0::UNC_M_CAS_COUNT:WR",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xc04,
	  .fstr = "ivbep_unc_imc0::UNC_M_CAS_COUNT:WR:e=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_imc0::UNC_M_RD_CAS_RANK0:BANK0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1b0,
	  .fstr = "ivbep_unc_imc0::UNC_M_RD_CAS_RANK0:BANK0:e=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_imc0::UNC_M_RD_CAS_RANK4:BANK7",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x80b4,
	  .fstr = "ivbep_unc_imc0::UNC_M_RD_CAS_RANK4:BANK7:e=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_imc0::UNC_M_RD_CAS_RANK4:BANK7:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x10080b4,
	  .fstr = "ivbep_unc_imc0::UNC_M_RD_CAS_RANK4:BANK7:e=0:t=1",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_qpi0::UNC_Q_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x14,
	  .fstr = "ivbep_unc_qpi0::UNC_Q_CLOCKTICKS:e=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_qpi0::UNC_Q_RXL_FLITS_G0:DATA",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x201,
	  .fstr = "ivbep_unc_qpi0::UNC_Q_RXL_FLITS_G0:DATA:e=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_qpi0::UNC_Q_RXL_FLITS_G0:IDLE:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1000101,
	  .fstr = "ivbep_unc_qpi0::UNC_Q_RXL_FLITS_G0:IDLE:e=0:t=1",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_qpi0::UNC_Q_TXL_FLITS_G0:DATA",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x200,
	  .fstr = "ivbep_unc_qpi0::UNC_Q_TXL_FLITS_G0:DATA:e=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_qpi0::UNC_Q_RXL_FLITS_G1:HOM",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x200602,
	  .fstr = "ivbep_unc_qpi0::UNC_Q_RXL_FLITS_G1:HOM:e=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_qpi0::UNC_Q_TXL_FLITS_G1:HOM",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x200600,
	  .fstr = "ivbep_unc_qpi0::UNC_Q_TXL_FLITS_G1:HOM:e=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_ubo::UNC_U_LOCK_CYCLES",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x44,
	  .fstr = "ivbep_unc_ubo::UNC_U_LOCK_CYCLES:e=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_r2pcie::UNC_R2_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1,
	  .fstr = "ivbep_unc_r2pcie::UNC_R2_CLOCKTICKS:e=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_r2pcie::UNC_R2_RING_AD_USED:CW",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x3307,
	  .fstr = "ivbep_unc_r2pcie::UNC_R2_RING_AD_USED:CW:e=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_r3qpi0::UNC_R3_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1,
	  .fstr = "ivbep_unc_r3qpi0::UNC_R3_CLOCKTICKS:e=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_r3qpi0::UNC_R3_TXR_CYCLES_FULL:e=0:t=0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x25,
	  .fstr = "ivbep_unc_r3qpi0::UNC_R3_TXR_CYCLES_FULL:e=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_r3qpi1::UNC_R3_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1,
	  .fstr = "ivbep_unc_r3qpi1::UNC_R3_CLOCKTICKS:e=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivbep_unc_r3qpi1::UNC_R3_TXR_CYCLES_FULL:e=0:t=0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x25,
	  .fstr = "ivbep_unc_r3qpi1::UNC_R3_TXR_CYCLES_FULL:e=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hsw_ep::mem_trans_retired:latency_above_threshold:ldlat=3:u",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5101cd,
	  .codes[1] = 3,
	  .fstr = "hsw_ep::MEM_TRANS_RETIRED:LOAD_LATENCY:k=0:u=1:e=0:i=0:c=0:t=0:ldlat=3:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw_ep::mem_trans_retired:latency_above_threshold:ldlat=1000000",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "hsw_ep::mem_trans_retired:load_latency",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301cd,
	  .codes[1] = 3,
	  .fstr = "hsw_ep::MEM_TRANS_RETIRED:LOAD_LATENCY:k=1:u=1:e=0:i=0:c=0:t=0:ldlat=3:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw_ep::cycle_activity:0x6:c=6",
	  .count = 1,
	  .codes[0] = 0x65306a3,
	  .fstr = "hsw_ep::CYCLE_ACTIVITY:0x6:k=1:u=1:e=0:i=0:c=6:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw_ep::mem_trans_retired:load_latency:ldlat=1000000",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "hsw_ep::mem_trans_retired:latency_above_threshold:ldlat=2:intx=0:intxcp=0",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "hsw_ep::inst_Retired:any_p:intx",
	  .count = 1,
	  .codes[0] = 0x1005300c0ull,
	  .fstr = "hsw_ep::INST_RETIRED:ANY_P:k=1:u=1:e=0:i=0:c=0:t=0:intx=1:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw_ep::inst_Retired:any_p:intx:intxcp",
	  .count = 1,
	  .codes[0] = 0x3005300c0ull,
	  .fstr = "hsw_ep::INST_RETIRED:ANY_P:k=1:u=1:e=0:i=0:c=0:t=0:intx=1:intxcp=1",
	},
	{ SRC_LINE,
	  .name = "hsw_ep::inst_Retired:any_p:intx=0:intxcp",
	  .count = 1,
	  .codes[0] = 0x2005300c0ull,
	  .fstr = "hsw_ep::INST_RETIRED:ANY_P:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=1",
	},
	{ SRC_LINE,
	  .name = "hsw_ep::cycle_activity:cycles_l2_pending",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x15301a3,
	  .fstr = "hsw_ep::CYCLE_ACTIVITY:CYCLES_L2_PENDING:k=1:u=1:e=0:i=0:c=1:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw_ep::cycle_activity:cycles_l2_pending:c=8",
	  .ret = PFM_ERR_ATTR_SET,
	},
	{ SRC_LINE,
	  .name = "hsw_ep::hle_retired:aborted",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5304c8,
	  .fstr = "hsw_ep::HLE_RETIRED:ABORTED:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw_ep::mem_load_uops_l3_miss_retired:remote_dram",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5304d3,
	  .fstr = "hsw_ep::MEM_LOAD_UOPS_L3_MISS_RETIRED:REMOTE_DRAM:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw_ep::offcore_response_0:any_data:L3_miss_local",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x3f80400091ull,
	  .fstr = "hsw_ep::OFFCORE_RESPONSE_0:DMND_DATA_RD:PF_DATA_RD:PF_L3_DATA_RD:L3_MISS_LOCAL:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw_ep::offcore_response_0:any_data:L3_miss_remote",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x3fb8000091ull,
	  .fstr = "hsw_ep::OFFCORE_RESPONSE_0:DMND_DATA_RD:PF_DATA_RD:PF_L3_DATA_RD:L3_MISS_REMOTE_HOP0:L3_MISS_REMOTE_HOP1:L3_MISS_REMOTE_HOP2P:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE, /* here SNP_ANY gets expanded when passed on the cmdline, but not when added automatically by library */
	  .name = "hsw_ep::OFFCORE_RESPONSE_0:DMND_DATA_RD:PF_DATA_RD:PF_L3_DATA_RD:L3_MISS_LOCAL:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x3f80400091ull,
	  .fstr = "hsw_ep::OFFCORE_RESPONSE_0:DMND_DATA_RD:PF_DATA_RD:PF_L3_DATA_RD:L3_MISS_LOCAL:SNP_NONE:SNP_NOT_NEEDED:SNP_MISS:SNP_NO_FWD:SNP_FWD:SNP_HITM:SNP_NON_DRAM:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0"
	},
	{ SRC_LINE,
	  .name = "hsw_ep::offcore_response_0:any_data:LLC_miss_local",
	  .ret = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "hsw_ep::offcore_response_0:any_data:LLC_miss_remote",
	  .ret = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "hsw_ep::offcore_response_0:any_data:L3_HIT",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] =0x5301b7,
	  .codes[1] =0x3f803c0091ull,
	  .fstr = "hsw_ep::OFFCORE_RESPONSE_0:DMND_DATA_RD:PF_DATA_RD:PF_L3_DATA_RD:L3_HITM:L3_HITE:L3_HITS:L3_HITF:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw_ep::offcore_response_0:l3_miss",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] =0x5301b7,
	  .codes[1] = 0x3fb8408fffull,
	  .fstr = "hsw_ep::OFFCORE_RESPONSE_0:ANY_REQUEST:L3_MISS_LOCAL:L3_MISS_REMOTE_HOP0:L3_MISS_REMOTE_HOP1:L3_MISS_REMOTE_HOP2P:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "bdw::mem_trans_retired:latency_above_threshold:ldlat=3:u",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5101cd,
	  .codes[1] = 3,
	  .fstr = "bdw::MEM_TRANS_RETIRED:LOAD_LATENCY:k=0:u=1:e=0:i=0:c=0:t=0:ldlat=3:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "bdw::mem_trans_retired:latency_above_threshold:ldlat=1000000",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "bdw::mem_trans_retired:load_latency",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301cd,
	  .codes[1] = 3,
	  .fstr = "bdw::MEM_TRANS_RETIRED:LOAD_LATENCY:k=1:u=1:e=0:i=0:c=0:t=0:ldlat=3:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "bdw::mem_trans_retired:load_latency:ldlat=1000000",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "bdw::mem_trans_retired:latency_above_threshold:ldlat=2:intx=0:intxcp=0",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "bdw::inst_Retired:any_p:intx",
	  .count = 1,
	  .codes[0] = 0x1005300c0ull,
	  .fstr = "bdw::INST_RETIRED:ANY_P:k=1:u=1:e=0:i=0:c=0:t=0:intx=1:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "bdw::inst_Retired:any_p:intx:intxcp",
	  .count = 1,
	  .codes[0] = 0x3005300c0ull,
	  .fstr = "bdw::INST_RETIRED:ANY_P:k=1:u=1:e=0:i=0:c=0:t=0:intx=1:intxcp=1",
	},
	{ SRC_LINE,
	  .name = "bdw::cycle_activity:0x6:c=6",
	  .count = 1,
	  .codes[0] = 0x65306a3,
	  .fstr = "bdw::CYCLE_ACTIVITY:0x6:k=1:u=1:e=0:i=0:c=6:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "bdw::inst_Retired:any_p:intx=0:intxcp",
	  .count = 1,
	  .codes[0] = 0x2005300c0ull,
	  .fstr = "bdw::INST_RETIRED:ANY_P:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=1",
	},
	{ SRC_LINE,
	  .name = "bdw::cycle_activity:cycles_l2_pending",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x15301a3,
	  .fstr = "bdw::CYCLE_ACTIVITY:CYCLES_L2_PENDING:k=1:u=1:e=0:i=0:c=1:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "bdw::cycle_activity:stalls_ldm_pending",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x65306a3,
	  .fstr = "bdw::CYCLE_ACTIVITY:STALLS_LDM_PENDING:k=1:u=1:e=0:i=0:c=6:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "bdw::CYCLE_ACTIVITY:STALLS_LDM_PENDING:k=1:u=1:e=0:i=0:c=6:t=0:intx=0:intxcp=0",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x65306a3,
	  .fstr = "bdw::CYCLE_ACTIVITY:STALLS_LDM_PENDING:k=1:u=1:e=0:i=0:c=6:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "bdw::cycle_activity:cycles_l2_pending:c=8",
	  .ret = PFM_ERR_ATTR_SET,
	},
	{ SRC_LINE,
	  .name = "bdw::hle_retired:aborted",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5304c8,
	  .fstr = "bdw::HLE_RETIRED:ABORTED:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "bdw::rtm_retired:aborted",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5304c9,
	  .fstr = "bdw::RTM_RETIRED:ABORTED:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "bdw::arith:fpu_div_active",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x530114,
	  .fstr = "bdw::ARITH:FPU_DIV_ACTIVE:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "bdw::inst_retired:prec_dist",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5301c0,
	  .fstr = "bdw::INST_RETIRED:PREC_DIST:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "bdw::rs_events:empty_end",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1d7015e,
	  .fstr = "bdw::RS_EVENTS:EMPTY_END:k=1:u=1:e=1:i=1:c=1:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "bdw::offcore_response_0:llc_hit",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x3f803c8fffull,
	  .fstr = "bdw::OFFCORE_RESPONSE_0:ANY_REQUEST:L3_HITM:L3_HITE:L3_HITS:L3_HITF:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "bdw::offcore_response_0:llc_miss_local",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x3f84008fffull,
	  .fstr = "bdw::OFFCORE_RESPONSE_0:ANY_REQUEST:L3_MISS_LOCAL:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "bdw::offcore_response_0:l3_miss_local",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x3f84008fffull,
	  .fstr = "bdw::OFFCORE_RESPONSE_0:ANY_REQUEST:L3_MISS_LOCAL:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "bdw::offcore_response_0:l3_miss",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] =0x5301b7,
	  .codes[1] = 0x3f84008fffull,
	  .fstr = "bdw::OFFCORE_RESPONSE_0:ANY_REQUEST:L3_MISS_LOCAL:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "bdw::offcore_response_1:any_data",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] =0x5301bb,
	  .codes[1] = 0x10091,
	  .fstr = "bdw::OFFCORE_RESPONSE_1:DMND_DATA_RD:PF_DATA_RD:PF_LLC_DATA_RD:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "bdw_ep::offcore_response_0:l3_miss",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x3fbc008fffull,
	  .fstr = "bdw_ep::OFFCORE_RESPONSE_0:ANY_REQUEST:L3_MISS_LOCAL:L3_MISS_REMOTE_HOP0:L3_MISS_REMOTE_HOP1:L3_MISS_REMOTE_HOP2P:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "bdw_ep::offcore_response_1:l3_miss_remote",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x3fb8008fffull,
	  .fstr = "bdw_ep::OFFCORE_RESPONSE_1:ANY_REQUEST:L3_MISS_REMOTE_HOP0:L3_MISS_REMOTE_HOP1:L3_MISS_REMOTE_HOP2P:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "bdw_ep::offcore_response_0:L3_MISS_REMOTE_HOP0_DRAM",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] =0x5301b7,
	  .codes[1] = 0x3f88008fffull,
	  .fstr = "bdw_ep::OFFCORE_RESPONSE_0:ANY_REQUEST:L3_MISS_REMOTE_HOP0:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "bdw::offcore_response_0:L3_MISS_REMOTE_HOP0_DRAM",
	  .ret = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo1::UNC_C_CLOCKTICKS:u",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo0::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "hswep_unc_cbo0::UNC_C_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo1::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "hswep_unc_cbo1::UNC_C_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo2::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "hswep_unc_cbo2::UNC_C_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo3::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "hswep_unc_cbo3::UNC_C_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo4::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "hswep_unc_cbo4::UNC_C_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo5::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "hswep_unc_cbo5::UNC_C_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo6::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "hswep_unc_cbo6::UNC_C_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo7::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "hswep_unc_cbo7::UNC_C_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo8::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "hswep_unc_cbo8::UNC_C_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo9::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "hswep_unc_cbo9::UNC_C_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo10::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "hswep_unc_cbo10::UNC_C_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo11::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "hswep_unc_cbo11::UNC_C_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo12::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "hswep_unc_cbo12::UNC_C_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo13::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "hswep_unc_cbo13::UNC_C_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo14::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "hswep_unc_cbo14::UNC_C_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo15::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "hswep_unc_cbo15::UNC_C_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo16::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "hswep_unc_cbo16::UNC_C_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo17::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "hswep_unc_cbo17::UNC_C_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo0::UNC_C_LLC_LOOKUP:DATA_READ",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x334,
	  .codes[1] = 0xfe0000,
	  .fstr = "hswep_unc_cbo0::UNC_C_LLC_LOOKUP:DATA_READ:STATE_MESIFD:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo0::UNC_C_LLC_LOOKUP:DATA_READ:nf=1",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo0::UNC_C_LLC_LOOKUP",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x1134,
	  .codes[1] = 0xfe0000,
	  .fstr = "hswep_unc_cbo0::UNC_C_LLC_LOOKUP:ANY:STATE_MESIFD:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo0::UNC_C_LLC_LOOKUP:NID:STATE_M",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo0::UNC_C_LLC_LOOKUP:NID:nf=3",
	  .ret  = PFM_SUCCESS,
	  .count = 3,
	  .codes[0] = 0x5134,
	  .codes[1] = 0xfe0000,
	  .codes[2] = 0x3,
	  .fstr = "hswep_unc_cbo0::UNC_C_LLC_LOOKUP:ANY:NID:STATE_MESIFD:e=0:t=0:tf=0:nf=3:cf=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo0::UNC_C_LLC_LOOKUP:NID:STATE_M:tid=1",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo0::UNC_C_LLC_LOOKUP:DATA_READ:WRITE",
	  .ret  = PFM_ERR_FEATCOMB,
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo0::UNC_C_LLC_LOOKUP:WRITE:NID:nf=3:tf=1:e:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 3,
	  .codes[0] = 0x10c4534,
	  .codes[1] = 0xfe0001,
	  .codes[2] = 0x3,
	  .fstr = "hswep_unc_cbo0::UNC_C_LLC_LOOKUP:WRITE:NID:STATE_MESIFD:e=1:t=1:tf=1:nf=3:cf=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo0::UNC_C_LLC_VICTIMS",
	  .ret  = PFM_ERR_UMASK,
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo0::UNC_C_LLC_VICTIMS:NID",
	  .ret  = PFM_ERR_UMASK,
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo0::UNC_C_LLC_VICTIMS:NID:nf=1",
	  .ret  = PFM_ERR_UMASK,
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo0::UNC_C_LLC_VICTIMS:STATE_M",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x137,
	  .fstr = "hswep_unc_cbo0::UNC_C_LLC_VICTIMS:STATE_M:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo0::UNC_C_LLC_VICTIMS:STATE_M:STATE_S",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x537,
	  .fstr = "hswep_unc_cbo0::UNC_C_LLC_VICTIMS:STATE_M:STATE_S:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo0::UNC_C_LLC_VICTIMS:STATE_M:STATE_S:NID:nf=1",
	  .ret  = PFM_SUCCESS,
	  .count = 3,
	  .codes[0] = 0x4537,
	  .codes[1] = 0x0,
	  .codes[2] = 0x1,
	  .fstr = "hswep_unc_cbo0::UNC_C_LLC_VICTIMS:STATE_M:STATE_S:NID:e=0:t=0:tf=0:nf=1:cf=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo0::UNC_C_TOR_INSERTS:OPCODE",
	  .ret  = PFM_ERR_UMASK,
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo0::UNC_C_TOR_INSERTS:WB",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1035,
	  .fstr = "hswep_unc_cbo0::UNC_C_TOR_INSERTS:WB:e=0:t=0:tf=0:isoc=0:nc=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo0::UNC_C_TOR_INSERTS:OPCODE:OPC_ITOM",
	  .ret  = PFM_SUCCESS,
	  .count = 3,
	  .codes[0] = 0x135,
	  .codes[1] = 0x0,
	  .codes[2] = 0x1c800000ull,
	  .fstr = "hswep_unc_cbo0::UNC_C_TOR_INSERTS:OPCODE:OPC_ITOM:e=0:t=0:tf=0:isoc=0:nc=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo0::UNC_C_TOR_INSERTS:OPCODE:OPC_ITOM:isoc=1",
	  .ret  = PFM_SUCCESS,
	  .count = 3,
	  .codes[0] = 0x135,
	  .codes[1] = 0x0,
	  .codes[2] = 0x9c800000ull,
	  .fstr = "hswep_unc_cbo0::UNC_C_TOR_INSERTS:OPCODE:OPC_ITOM:e=0:t=0:tf=0:isoc=1:nc=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo0::UNC_C_TOR_INSERTS:OPCODE:OPC_PCIWILF:nf=1",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo0::UNC_C_TOR_INSERTS:NID_OPCODE:OPC_PCIRDCUR:nf=1",
	  .ret  = PFM_SUCCESS,
	  .count = 3,
	  .codes[0] = 0x4135,
	  .codes[1] = 0x0,
	  .codes[2] = 0x19e00001ull,
	  .fstr = "hswep_unc_cbo0::UNC_C_TOR_INSERTS:NID_OPCODE:OPC_PCIRDCUR:e=0:t=0:tf=0:nf=1:isoc=0:nc=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo0::UNC_C_TOR_INSERTS:OPC_RFO:NID_OPCODE:nf=1",
	  .ret  = PFM_SUCCESS,
	  .count = 3,
	  .codes[0] = 0x4135,
	  .codes[1] = 0x0,
	  .codes[2] = 0x18000001ull,
	  .fstr = "hswep_unc_cbo0::UNC_C_TOR_INSERTS:NID_OPCODE:OPC_RFO:e=0:t=0:tf=0:nf=1:isoc=0:nc=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_cbo0::UNC_C_TOR_OCCUPANCY:MISS_REMOTE",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x8a36,
	  .fstr = "hswep_unc_cbo0::UNC_C_TOR_OCCUPANCY:MISS_REMOTE:e=0:t=0:tf=0:isoc=0:nc=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_irp::unc_i_clockticks",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "hswep_unc_irp::UNC_I_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_irp::unc_i_coherent_ops:RFO",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x813,
	  .fstr = "hswep_unc_irp::UNC_I_COHERENT_OPS:RFO:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_irp::unc_i_transactions:reads",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x116,
	  .fstr = "hswep_unc_irp::UNC_I_TRANSACTIONS:READS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_irp::unc_i_transactions:reads:c=1:i",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "hswep_unc_irp::unc_i_transactions:reads:t=6",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x6000116,
	  .fstr = "hswep_unc_irp::UNC_I_TRANSACTIONS:READS:e=0:i=0:t=6",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_sbo0::unc_s_clockticks",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "hswep_unc_sbo0::UNC_S_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_sbo1::unc_s_clockticks",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "hswep_unc_sbo1::UNC_S_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_sbo2::unc_s_clockticks",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "hswep_unc_sbo2::UNC_S_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_sbo3::unc_s_clockticks",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "hswep_unc_sbo3::UNC_S_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_pcu::UNC_P_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "hswep_unc_pcu::UNC_P_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_pcu::UNC_P_CLOCKTICKS:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1000000,
	  .fstr = "hswep_unc_pcu::UNC_P_CLOCKTICKS:e=0:i=0:t=1",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_pcu::UNC_P_CORE0_TRANSITION_CYCLES",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x60,
	  .fstr = "hswep_unc_pcu::UNC_P_CORE0_TRANSITION_CYCLES:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_pcu::UNC_P_CORE17_TRANSITION_CYCLES",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x71,
	  .fstr = "hswep_unc_pcu::UNC_P_CORE17_TRANSITION_CYCLES:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "hswep_unc_pcu::UNC_P_FREQ_BAND1_CYCLES",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "hswep_unc_pcu::UNC_P_FREQ_BAND2_CYCLES",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "hswep_unc_pcu::UNC_P_FREQ_BAND3_CYCLES",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "hswep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:ff=32",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0xb,
	  .codes[1] = 0x20,
	  .fstr = "hswep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:e=0:i=0:t=0:ff=32",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_pcu::UNC_P_FREQ_BAND1_CYCLES:ff=16",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0xc,
	  .codes[1] = 0x1000,
	  .fstr = "hswep_unc_pcu::UNC_P_FREQ_BAND1_CYCLES:e=0:i=0:t=0:ff=16",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_pcu::UNC_P_FREQ_BAND2_CYCLES:ff=8",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0xd,
	  .codes[1] = 0x80000,
	  .fstr = "hswep_unc_pcu::UNC_P_FREQ_BAND2_CYCLES:e=0:i=0:t=0:ff=8",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_pcu::UNC_P_FREQ_BAND3_CYCLES:ff=40",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0xe,
	  .codes[1] = 0x28000000,
	  .fstr = "hswep_unc_pcu::UNC_P_FREQ_BAND3_CYCLES:e=0:i=0:t=0:ff=40",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:ff=32:e",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x4000b,
	  .codes[1] = 0x20,
	  .fstr = "hswep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:e=1:i=0:t=0:ff=32",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:ff=32:t=24",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x1800000b,
	  .codes[1] = 0x20,
	  .fstr = "hswep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:e=0:i=0:t=24:ff=32",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:ff=32:e:t=4",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x404000b,
	  .codes[1] = 0x20,
	  .fstr = "hswep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:e=1:i=0:t=4:ff=32",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x4080,
	  .fstr = "hswep_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0:e=0:i=0:t=0"
	},
	{ SRC_LINE,
	  .name = "hswep_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C3",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x8080,
	  .fstr = "hswep_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C3:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C6",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xc080,
	  .fstr = "hswep_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C6:e=0:i=0:t=0"
	},
	{ SRC_LINE,
	  .name = "hswep_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0:t=6:i",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x46004080,
	  .fstr = "hswep_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0:e=0:i=1:t=6"
	},
	{ SRC_LINE,
	  .name = "hswep_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0:t=6",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x6004080,
	  .fstr = "hswep_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0:e=0:i=0:t=6"
	},
	{ SRC_LINE,
	  .name = "hswep_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0:t=6:i:e",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xc6004080,
	  .fstr = "hswep_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0:e=1:i=1:t=6"
	},
	{ SRC_LINE,
	  .name = "hswep_unc_pcu::UNC_P_DEMOTIONS_CORE10",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x3a,
	  .fstr = "hswep_unc_pcu::UNC_P_DEMOTIONS_CORE10:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_pcu::UNC_P_DEMOTIONS_CORE14",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x3e,
	  .fstr = "hswep_unc_pcu::UNC_P_DEMOTIONS_CORE14:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_pcu::UNC_P_DEMOTIONS_CORE17",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x41,
	  .fstr = "hswep_unc_pcu::UNC_P_DEMOTIONS_CORE17:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_ha0::UNC_H_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "hswep_unc_ha0::UNC_H_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_ha1::UNC_H_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "hswep_unc_ha1::UNC_H_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_ha1::UNC_H_REQUESTS:READS:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1000301,
	  .fstr = "hswep_unc_ha1::UNC_H_REQUESTS:READS:e=0:i=0:t=1",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_ha0::UNC_H_IMC_WRITES:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1000f1a,
	  .fstr = "hswep_unc_ha0::UNC_H_IMC_WRITES:ALL:e=0:i=0:t=1",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_ha0::UNC_H_IMC_READS:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1000117,
	  .fstr = "hswep_unc_ha0::UNC_H_IMC_READS:NORMAL:e=0:i=0:t=1",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_imc0::UNC_M_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xff,
	  .fstr = "hswep_unc_imc0::UNC_M_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_imc0::UNC_M_CLOCKTICKS:t=1",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "hswep_unc_imc0::UNC_M_DCLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "hswep_unc_imc0::UNC_M_DCLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_imc4::UNC_M_DCLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "hswep_unc_imc4::UNC_M_DCLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_imc0::UNC_M_CAS_COUNT:RD",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0304,
	  .fstr = "hswep_unc_imc0::UNC_M_CAS_COUNT:RD:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_imc0::UNC_M_PRE_COUNT:WR",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0802,
	  .fstr = "hswep_unc_imc0::UNC_M_PRE_COUNT:WR:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_imc0::UNC_M_POWER_CKE_CYCLES:RANK0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x183,
	  .fstr = "hswep_unc_imc0::UNC_M_POWER_CKE_CYCLES:RANK0:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_imc0::UNC_M_CAS_COUNT:WR",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xc04,
	  .fstr = "hswep_unc_imc0::UNC_M_CAS_COUNT:WR:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_imc0::UNC_M_RD_CAS_RANK0:BANK0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xb0,
	  .fstr = "hswep_unc_imc0::UNC_M_RD_CAS_RANK0:BANK0:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_imc0::UNC_M_RD_CAS_RANK0:ALLBANKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x10b0,
	  .fstr = "hswep_unc_imc0::UNC_M_RD_CAS_RANK0:ALLBANKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_imc0::UNC_M_RD_CAS_RANK0:BANKG0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x11b0,
	  .fstr = "hswep_unc_imc0::UNC_M_RD_CAS_RANK0:BANKG0:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_imc0::UNC_M_RD_CAS_RANK4:BANK7",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x7b4,
	  .fstr = "hswep_unc_imc0::UNC_M_RD_CAS_RANK4:BANK7:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_imc0::UNC_M_RD_CAS_RANK4:BANK7:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x10007b4,
	  .fstr = "hswep_unc_imc0::UNC_M_RD_CAS_RANK4:BANK7:e=0:i=0:t=1",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_imc0::UNC_M_RD_CAS_RANK7:BANK7:t=1:i",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x18007b7,
	  .fstr = "hswep_unc_imc0::UNC_M_RD_CAS_RANK7:BANK7:e=0:i=1:t=1",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_sbo0::UNC_S_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "hswep_unc_sbo0::UNC_S_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_sbo0::UNC_S_FAST_ASSERTED:t=1:i",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1800009,
	  .fstr = "hswep_unc_sbo0::UNC_S_FAST_ASSERTED:e=0:i=1:t=1",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_sbo3::UNC_S_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "hswep_unc_sbo3::UNC_S_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_sbo3::UNC_S_FAST_ASSERTED:t=1:i",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1800009,
	  .fstr = "hswep_unc_sbo3::UNC_S_FAST_ASSERTED:e=0:i=1:t=1",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_ubo::UNC_U_EVENT_MSG",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x842,
	  .fstr = "hswep_unc_ubo::UNC_U_EVENT_MSG:DOORBELL_RCVD:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_qpi0::UNC_Q_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x14,
	  .fstr = "hswep_unc_qpi0::UNC_Q_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_qpi0::UNC_Q_DIRECT2CORE:SUCCESS_RBT_HIT",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x113,
	  .fstr = "hswep_unc_qpi0::UNC_Q_DIRECT2CORE:SUCCESS_RBT_HIT:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_qpi0::UNC_Q_RXL_FLITS_G1:DRS:i:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1a01802,
	  .fstr = "hswep_unc_qpi0::UNC_Q_RXL_FLITS_G1:DRS:e=0:i=1:t=1",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_qpi0::UNC_Q_TXL_FLITS_G2:NDR_AD",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x200101,
	  .fstr = "hswep_unc_qpi0::UNC_Q_TXL_FLITS_G2:NDR_AD:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_qpi0::UNC_Q_RXL_OCCUPANCY",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xb,
	  .fstr = "hswep_unc_qpi0::UNC_Q_RXL_OCCUPANCY:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_qpi0::UNC_Q_TXL_INSERTS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x4,
	  .fstr = "hswep_unc_qpi0::UNC_Q_TXL_INSERTS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_r2pcie::UNC_R2_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1,
	  .fstr = "hswep_unc_r2pcie::UNC_R2_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_r2pcie::UNC_R2_RING_AD_USED:CW",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x307,
	  .fstr = "hswep_unc_r2pcie::UNC_R2_RING_AD_USED:CW:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_r3qpi0::UNC_R3_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1,
	  .fstr = "hswep_unc_r3qpi0::UNC_R3_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_r3qpi0::UNC_R3_RXR_CYCLES_NE:SNP:e=0:t=0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x210,
	  .fstr = "hswep_unc_r3qpi0::UNC_R3_RXR_CYCLES_NE:SNP:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_r3qpi1::UNC_R3_RING_SINK_STARVED",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x20e,
	  .fstr = "hswep_unc_r3qpi1::UNC_R3_RING_SINK_STARVED:AK:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "hswep_unc_r3qpi1::UNC_R3_HA_R2_BL_CREDITS_EMPTY:HA1:i:t=2",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x280022d,
	  .fstr = "hswep_unc_r3qpi1::UNC_R3_HA_R2_BL_CREDITS_EMPTY:HA1:e=0:i=1:t=2",
	},
	{ SRC_LINE,
	  .name = "skl::FRONTEND_RETIRED:DSB_MISS",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301c6,
	  .codes[1] = 0x11,
	  .fstr = "skl::FRONTEND_RETIRED:DSB_MISS:k=1:u=1:e=0:i=0:c=0:intx=0:intxcp=0:fe_thres=0",
	},
	{ SRC_LINE,
	  .name = "skl::MEM_LOAD_RETIRED:L3_MISS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5320d1,
	  .fstr = "skl::MEM_LOAD_RETIRED:L3_MISS:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "skl::FRONTEND_RETIRED:ITLB_MISS",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301c6,
	  .codes[1] = 0x14,
	  .fstr = "skl::FRONTEND_RETIRED:ITLB_MISS:k=1:u=1:e=0:i=0:c=0:intx=0:intxcp=0:fe_thres=0",
	},
	{ SRC_LINE,
	  .name = "skl::FRONTEND_RETIRED:L1I_MISS",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301c6,
	  .codes[1] = 0x12,
	  .fstr = "skl::FRONTEND_RETIRED:L1I_MISS:k=1:u=1:e=0:i=0:c=0:intx=0:intxcp=0:fe_thres=0",
	},
	{ SRC_LINE,
	  .name = "skl::FRONTEND_RETIRED:L2_MISS:u",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5101c6,
	  .codes[1] = 0x13,
	  .fstr = "skl::FRONTEND_RETIRED:L2_MISS:k=0:u=1:e=0:i=0:c=0:intx=0:intxcp=0:fe_thres=0",
	},
	{ SRC_LINE,
	  .name = "skl::FRONTEND_RETIRED:STLB_MISS:c=1:i",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x1d301c6,
	  .codes[1] = 0x15,
	  .fstr = "skl::FRONTEND_RETIRED:STLB_MISS:k=1:u=1:e=0:i=1:c=1:intx=0:intxcp=0:fe_thres=0",
	},
	{ SRC_LINE,
	  .name = "skl::FRONTEND_RETIRED:IDQ_4_BUBBLES",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301c6,
	  .codes[1] = 0x400106,
	  .fstr = "skl::FRONTEND_RETIRED:IDQ_4_BUBBLES:k=1:u=1:e=0:i=0:c=0:intx=0:intxcp=0:fe_thres=1",
	},
	{ SRC_LINE,
	  .name = "skl::FRONTEND_RETIRED:IDQ_3_BUBBLES",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301c6,
	  .codes[1] = 0x300106,
	  .fstr = "skl::FRONTEND_RETIRED:IDQ_3_BUBBLES:k=1:u=1:e=0:i=0:c=0:intx=0:intxcp=0:fe_thres=1",
	},
	{ SRC_LINE,
	  .name = "skl::FRONTEND_RETIRED:IDQ_3_BUBBLES:fe_thres=8",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301c6,
	  .codes[1] = 0x300806,
	  .fstr = "skl::FRONTEND_RETIRED:IDQ_3_BUBBLES:k=1:u=1:e=0:i=0:c=0:intx=0:intxcp=0:fe_thres=8",
	},
	{ SRC_LINE,
	  .name = "skl::FRONTEND_RETIRED:IDQ_3_BUBBLES:fe_thres=4095",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301c6,
	  .codes[1] = 0x3fff06,
	  .fstr = "skl::FRONTEND_RETIRED:IDQ_3_BUBBLES:k=1:u=1:e=0:i=0:c=0:intx=0:intxcp=0:fe_thres=4095",
	},
	{ SRC_LINE,
	  .name = "skl::FRONTEND_RETIRED:IDQ_3_BUBBLES:fe_thres=4096",
	  .ret  = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "skl::FRONTEND_RETIRED:DSB_MISS:ITLB_MISS",
	  .ret  = PFM_ERR_FEATCOMB,
	},
	{ SRC_LINE,
	  .name = "skl::offcore_response_0:any_request",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x18007,
	  .fstr = "skl::OFFCORE_RESPONSE_0:DMND_DATA_RD:DMND_RFO:DMND_CODE_RD:OTHER:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "skl::offcore_response_0:l3_hitmes",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x3f801d8007ull,
	  .fstr = "skl::OFFCORE_RESPONSE_0:ANY_REQUEST:L3_HITM:L3_HITE:L3_HITS:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "skl::offcore_response_0:L4_HIT_LOCAL_L4",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x3f80418007ull,
	  .fstr = "skl::OFFCORE_RESPONSE_0:ANY_REQUEST:L4_HIT_LOCAL_L4:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "skl::offcore_response_0:L3_MISS_LOCAL",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x3f84018007ull,
	  .fstr = "skl::OFFCORE_RESPONSE_0:ANY_REQUEST:L3_MISS_LOCAL:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "skl::offcore_response_0:l3_miss",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] =0x5301b7,
	  .codes[1] = 0x3f84018007ull,
	  .fstr = "skl::OFFCORE_RESPONSE_0:ANY_REQUEST:L3_MISS_LOCAL:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},

	{ SRC_LINE,
	  .name = "skl::cycle_activity:0x6:c=6",
	  .count = 1,
	  .codes[0] = 0x65306a3,
	  .fstr = "skl::CYCLE_ACTIVITY:0x6:k=1:u=1:e=0:i=0:c=6:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "skl::dtlb_store_misses:walk_completed_2m_4m:c=1",
	  .count = 1,
	  .codes[0] = 0x1530449,
	  .fstr = "skl::DTLB_STORE_MISSES:WALK_COMPLETED_2M_4M:k=1:u=1:e=0:i=0:c=1:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "skl::rob_misc_events:lbr_inserts",
	  .count = 1,
	  .codes[0] = 0x5320cc,
	  .fstr = "skl::ROB_MISC_EVENTS:LBR_INSERTS:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "skl::cycle_activity:stalls_mem_any:c=6",
	  .ret = PFM_ERR_ATTR_SET,
	},
	{ SRC_LINE,
	  .name = "skl::uops_dispatched_port:port_0",
	  .count = 1,
	  .codes[0] = 0x5301a1,
	  .fstr = "skl::UOPS_DISPATCHED_PORT:PORT_0:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "skl::uops_dispatched:port_0",
	  .count = 1,
	  .codes[0] = 0x5301a1,
	  .fstr = "skl::UOPS_DISPATCHED_PORT:PORT_0:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw::CYCLE_ACTIVITY:CYCLES_L2_PENDING:k=1:u=1:e=0:i=0:c=1:t=0:intx=0:intxcp=0",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x15301a3,
	  .fstr = "hsw::CYCLE_ACTIVITY:CYCLES_L2_PENDING:k=1:u=1:e=0:i=0:c=1:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw::CYCLE_ACTIVITY:CYCLES_L2_PENDING:c=1",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x15301a3,
	  .fstr = "hsw::CYCLE_ACTIVITY:CYCLES_L2_PENDING:k=1:u=1:e=0:i=0:c=1:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "glm::offcore_response_1:any_request",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5302b7,
	  .codes[1] = 0x18000,
	  .fstr = "glm::OFFCORE_RESPONSE_1:ANY_REQUEST:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "glm::offcore_response_1:any_rfo",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5302b7,
	  .codes[1] = 0x10022,
	  .fstr = "glm::OFFCORE_RESPONSE_1:DMND_RFO:PF_RFO:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "glm::offcore_response_1:any_rfo:l2_miss_snp_miss_or_no_snoop_needed",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5302b7,
	  .codes[1] = 0x200010022ull,
	  .fstr = "glm::OFFCORE_RESPONSE_1:DMND_RFO:PF_RFO:ANY_RESPONSE:L2_MISS_SNP_MISS_OR_NO_SNOOP_NEEDED:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "glm::offcore_response_0:strm_st",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x14800,
	  .fstr = "glm::OFFCORE_RESPONSE_0:FULL_STRM_ST:PARTIAL_STRM_ST:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "glm::offcore_response_1:dmnd_data_rd:outstanding",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "glm::offcore_response_1:dmnd_data_rd:l2_hit:outstanding",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "glm::offcore_response_0:strm_st:outstanding",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x4000004800ull,
	  .fstr = "glm::OFFCORE_RESPONSE_0:FULL_STRM_ST:PARTIAL_STRM_ST:OUTSTANDING:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "glm::offcore_response_0:outstanding:dmnd_data_rd:u",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5101b7,
	  .codes[1] = 0x4000000001ull,
	  .fstr = "glm::OFFCORE_RESPONSE_0:DMND_DATA_RD:OUTSTANDING:k=0:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "glm::offcore_response_0:strm_st:l2_hit:outstanding",
	  .ret  = PFM_ERR_FEATCOMB,
	},
	{ SRC_LINE,
	  .name = "glm::ISSUE_SLOTS_NOT_CONSUMED:RESOURCE_FULL",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5301ca,
	  .fstr = "glm::ISSUE_SLOTS_NOT_CONSUMED:RESOURCE_FULL:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "glm::ISSUE_SLOTS_NOT_CONSUMED:RESOURCE_FULL:k:c=1:i",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1d201ca,
	  .fstr = "glm::ISSUE_SLOTS_NOT_CONSUMED:RESOURCE_FULL:k=1:u=0:e=0:i=1:c=1",
	},
	{ SRC_LINE,
	  .name = "glm::ISSUE_SLOTS_NOT_CONSUMED:RESOURCE_FULL:u:t",
	  .ret = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "glm::ISSUE_SLOTS_NOT_CONSUMED:RESOURCE_FULL:u:intxcp",
	  .ret = PFM_ERR_ATTR,
	},
	/*
	 * test delimiter options
	 */
	{ SRC_LINE,
	  .name = "glm::ISSUE_SLOTS_NOT_CONSUMED.RESOURCE_FULL.k=1.u=0.e=0.i=0.c=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x15201ca,
	  .fstr = "glm::ISSUE_SLOTS_NOT_CONSUMED:RESOURCE_FULL:k=1:u=0:e=0:i=0:c=1",
	},
	{ SRC_LINE,
	  .name = "glm::ISSUE_SLOTS_NOT_CONSUMED.RESOURCE_FULL:k=1:u=1:e=0:i=0:c=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x15301ca,
	  .fstr = "glm::ISSUE_SLOTS_NOT_CONSUMED:RESOURCE_FULL:k=1:u=1:e=0:i=0:c=1",
	},
	{ SRC_LINE,
	  .name = "glm::ISSUE_SLOTS_NOT_CONSUMED.RESOURCE_FULL:k=1:u=1:e=0.i=0.c=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x15301ca,
	  .fstr = "glm::ISSUE_SLOTS_NOT_CONSUMED:RESOURCE_FULL:k=1:u=1:e=0:i=0:c=1",
	},
	{ SRC_LINE,
	  .name = "knl::no_alloc_cycles:all",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x537fca,
	  .fstr = "knl::NO_ALLOC_CYCLES:ALL:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "knl::MEM_UOPS_RETIRED:DTLB_MISS_LOADS",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x530804,
	  .fstr = "knl::MEM_UOPS_RETIRED:DTLB_MISS_LOADS:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "knl::uops_retired:any:t",
	  .ret = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "knl::unhalted_reference_cycles:u:t",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x710300,
	  .fstr = "knl::UNHALTED_REFERENCE_CYCLES:k=0:u=1:t=1",
	},
	{ SRC_LINE,
	  .name = "knl::instructions_retired:k:t",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] =0x7200c0,
	  .fstr = "knl::INSTRUCTION_RETIRED:k=1:u=0:e=0:i=0:c=0:t=1",
	},
	{ SRC_LINE,
	  .name = "knl::unhalted_core_cycles:k:t",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x72003c,
	  .fstr = "knl::UNHALTED_CORE_CYCLES:k=1:u=0:e=0:i=0:c=0:t=1",
	},
	{ SRC_LINE,
	  .name = "knl::offcore_response_1:any_request",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5302b7,
	  .codes[1] = 0x18000,
	  .fstr = "knl::OFFCORE_RESPONSE_1:ANY_REQUEST:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "knl::offcore_response_0:any_read",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x132e7,
	  .fstr = "knl::OFFCORE_RESPONSE_0:DMND_DATA_RD:DMND_RFO:DMND_CODE_RD:PF_L2_RFO:PF_L2_CODE_RD:PARTIAL_READS:UC_CODE_READS:PF_SOFTWARE:PF_L1_DATA_RD:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "knl::offcore_response_1:any_read",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5302b7,
	  .codes[1] = 0x132e7,
	  .fstr = "knl::OFFCORE_RESPONSE_1:DMND_DATA_RD:DMND_RFO:DMND_CODE_RD:PF_L2_RFO:PF_L2_CODE_RD:PARTIAL_READS:UC_CODE_READS:PF_SOFTWARE:PF_L1_DATA_RD:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "knl::offcore_response_0:any_request:ddr_near",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x80808000ull,
	  .fstr = "knl::OFFCORE_RESPONSE_0:ANY_REQUEST:DDR_NEAR:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "knl::offcore_response_0:any_request:L2_HIT_NEAR_TILE:L2_HIT_FAR_TILE",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x1800588000ull,
	  .fstr = "knl::OFFCORE_RESPONSE_0:ANY_REQUEST:L2_HIT_NEAR_TILE:L2_HIT_FAR_TILE:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "knl::offcore_response_0:dmnd_data_rd:outstanding",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x4000000001ull,
	  .fstr = "knl::OFFCORE_RESPONSE_0:DMND_DATA_RD:OUTSTANDING:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "knl::offcore_response_0:dmnd_data_rd:ddr_near:outstanding",
	  .ret  = PFM_ERR_FEATCOMB,
	},
	{ SRC_LINE,
	  .name = "knl::offcore_response_1:dmnd_data_rd:outstanding",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "knl_unc_imc0::UNC_M_D_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "knl_unc_imc0::UNC_M_D_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "knl_unc_imc0::UNC_M_CAS_COUNT:RD",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0103,
	  .fstr = "knl_unc_imc0::UNC_M_CAS_COUNT:RD",
	},
	{ SRC_LINE,
	  .name = "knl_unc_imc0::UNC_M_CAS_COUNT:WR",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0203,
	  .fstr = "knl_unc_imc0::UNC_M_CAS_COUNT:WR",
	},
	{ SRC_LINE,
	  .name = "knl_unc_imc0::UNC_M_CAS_COUNT:ALL",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0303,
	  .fstr = "knl_unc_imc0::UNC_M_CAS_COUNT:ALL",
        },
	{ SRC_LINE,
	  .name = "knl_unc_imc_uclk0::UNC_M_U_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "knl_unc_imc_uclk0::UNC_M_U_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "knl_unc_edc_uclk0::UNC_E_U_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "knl_unc_edc_uclk0::UNC_E_U_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "knl_unc_edc_uclk0::UNC_E_EDC_ACCESS:HIT_CLEAN",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0102,
	  .fstr = "knl_unc_edc_uclk0::UNC_E_EDC_ACCESS:HIT_CLEAN",
	},
	{ SRC_LINE,
	  .name = "knl_unc_edc_uclk0::UNC_E_EDC_ACCESS:HIT_DIRTY",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0202,
	  .fstr = "knl_unc_edc_uclk0::UNC_E_EDC_ACCESS:HIT_DIRTY",
	},
	{ SRC_LINE,
	  .name = "knl_unc_edc_uclk0::UNC_E_EDC_ACCESS:MISS_CLEAN",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0402,
	  .fstr = "knl_unc_edc_uclk0::UNC_E_EDC_ACCESS:MISS_CLEAN",
	},
	{ SRC_LINE,
	  .name = "knl_unc_edc_uclk0::UNC_E_EDC_ACCESS:MISS_DIRTY",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0802,
	  .fstr = "knl_unc_edc_uclk0::UNC_E_EDC_ACCESS:MISS_DIRTY",
	},
	{ SRC_LINE,
	  .name = "knl_unc_edc_uclk0::UNC_E_EDC_ACCESS:MISS_INVALID",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1002,
	  .fstr = "knl_unc_edc_uclk0::UNC_E_EDC_ACCESS:MISS_INVALID",
        },
	{ SRC_LINE,
	  .name = "knl_unc_edc_eclk0::UNC_E_E_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "knl_unc_edc_eclk0::UNC_E_E_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "knl_unc_edc_eclk0::UNC_E_RPQ_INSERTS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0101,
	  .fstr = "knl_unc_edc_eclk0::UNC_E_RPQ_INSERTS",
	},
	{ SRC_LINE,
	  .name = "knl_unc_cha0::UNC_H_U_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "knl_unc_cha0::UNC_H_U_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "knl_unc_cha1::UNC_H_U_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "knl_unc_cha1::UNC_H_U_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "knl_unc_cha10::UNC_H_U_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "knl_unc_cha10::UNC_H_U_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "knl_unc_cha20::UNC_H_U_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "knl_unc_cha20::UNC_H_U_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "knl_unc_cha25::UNC_H_U_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "knl_unc_cha25::UNC_H_U_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "knl_unc_cha30::UNC_H_U_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "knl_unc_cha30::UNC_H_U_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "knl_unc_cha37::UNC_H_U_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "knl_unc_cha37::UNC_H_U_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "knl_unc_cha0::UNC_H_INGRESS_OCCUPANCY:IRQ",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0111,
	  .fstr = "knl_unc_cha0::UNC_H_INGRESS_OCCUPANCY:IRQ",
	},
	{ SRC_LINE,
	  .name = "knl_unc_cha0::UNC_H_INGRESS_OCCUPANCY:IRQ_REJ",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0211,
	  .fstr = "knl_unc_cha0::UNC_H_INGRESS_OCCUPANCY:IRQ_REJ",
	},
	{ SRC_LINE,
	  .name = "knl_unc_cha0::UNC_H_INGRESS_OCCUPANCY:IPQ",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0411,
	  .fstr = "knl_unc_cha0::UNC_H_INGRESS_OCCUPANCY:IPQ",
	},
	{ SRC_LINE,
	  .name = "knl_unc_cha0::UNC_H_INGRESS_OCCUPANCY:PRQ",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1011,
	  .fstr = "knl_unc_cha0::UNC_H_INGRESS_OCCUPANCY:PRQ",
	},
	{ SRC_LINE,
	  .name = "knl_unc_cha0::UNC_H_INGRESS_OCCUPANCY:PRQ_REJ",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x2011,
	  .fstr = "knl_unc_cha0::UNC_H_INGRESS_OCCUPANCY:PRQ_REJ",
	},
	{ SRC_LINE,
	  .name = "knl_unc_cha0::UNC_H_INGRESS_INSERTS:IRQ",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0113,
	  .fstr = "knl_unc_cha0::UNC_H_INGRESS_INSERTS:IRQ",
	},
	{ SRC_LINE,
	  .name = "knl_unc_cha0::UNC_H_INGRESS_INSERTS:IRQ_REJ",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0213,
	  .fstr = "knl_unc_cha0::UNC_H_INGRESS_INSERTS:IRQ_REJ",
	},
	{ SRC_LINE,
	  .name = "knl_unc_cha0::UNC_H_INGRESS_INSERTS:IPQ",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0413,
	  .fstr = "knl_unc_cha0::UNC_H_INGRESS_INSERTS:IPQ",
	},
	{ SRC_LINE,
	  .name = "knl_unc_cha0::UNC_H_INGRESS_INSERTS:PRQ",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1013,
	  .fstr = "knl_unc_cha0::UNC_H_INGRESS_INSERTS:PRQ",
	},
	{ SRC_LINE,
	  .name = "knl_unc_cha0::UNC_H_INGRESS_INSERTS:PRQ_REJ",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x2013,
	  .fstr = "knl_unc_cha0::UNC_H_INGRESS_INSERTS:PRQ_REJ",
        },
	{ SRC_LINE,
	  .name = "knl_unc_cha0::UNC_H_INGRESS_RETRY_IRQ0_REJECT:AD_RSP_VN0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0218,
	  .fstr = "knl_unc_cha0::UNC_H_INGRESS_RETRY_IRQ0_REJECT:AD_RSP_VN0",
	},
	{ SRC_LINE,
	  .name = "knl_unc_m2pcie::UNC_M2P_INGRESS_CYCLES_NE:ALL",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0810,
	  .fstr = "knl_unc_m2pcie::UNC_M2P_INGRESS_CYCLES_NE:ALL",
	},
	{ SRC_LINE,
	  .name = "knl_unc_m2pcie::UNC_M2P_EGRESS_CYCLES_NE:AD_0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0123,
	  .fstr = "knl_unc_m2pcie::UNC_M2P_EGRESS_CYCLES_NE:AD_0",
	},
	{ SRC_LINE,
	  .name = "knl_unc_m2pcie::UNC_M2P_EGRESS_CYCLES_NE:AD_1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0823,
	  .fstr = "knl_unc_m2pcie::UNC_M2P_EGRESS_CYCLES_NE:AD_1",
	},
	{ SRC_LINE,
	  .name = "knl_unc_m2pcie::UNC_M2P_EGRESS_INSERTS:AD_0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0124,
	  .fstr = "knl_unc_m2pcie::UNC_M2P_EGRESS_INSERTS:AD_0",
	},
	{ SRC_LINE,
	  .name = "knl_unc_m2pcie::UNC_M2P_EGRESS_INSERTS:AD_1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1024,
	  .fstr = "knl_unc_m2pcie::UNC_M2P_EGRESS_INSERTS:AD_1",
	},
	{ SRC_LINE,
	  .name = "knl_unc_m2pcie::UNC_M2P_EGRESS_CYCLES_FULL:AD_0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0125,
	  .fstr = "knl_unc_m2pcie::UNC_M2P_EGRESS_CYCLES_FULL:AD_0",
	},
	{ SRC_LINE,
	  .name = "knl_unc_m2pcie::UNC_M2P_EGRESS_CYCLES_FULL:AD_1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0825,
	  .fstr = "knl_unc_m2pcie::UNC_M2P_EGRESS_CYCLES_FULL:AD_1",
	},
	{ SRC_LINE,
	  .name = "wsm::offcore_response_0:0xf",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0xf,
	  .fstr = "wsm::OFFCORE_RESPONSE_0:0xf:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm::offcore_response_0:0xffff",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0xffff,
	  .fstr = "wsm::OFFCORE_RESPONSE_0:0xffff:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm::offcore_response_0:0x7fffffffff",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:0xf",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0xf,
	  .fstr = "snb::OFFCORE_RESPONSE_0:0xf:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:0xfffffffff",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0xfffffffff,
	  .fstr = "snb::OFFCORE_RESPONSE_0:0xffffffff:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:0x7fffffffff",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "ivb_ep::offcore_response_0:0xf",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0xf,
	  .fstr = "ivb_ep::OFFCORE_RESPONSE_0:0xf:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb_ep::offcore_response_0:0xfffffffff",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0xfffffffff,
	  .fstr = "ivb_ep::OFFCORE_RESPONSE_0:0xffffffff:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb_ep::offcore_response_0:0x7fffffffff",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "hsw::offcore_response_0:0xf",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0xf,
	  .fstr = "hsw::OFFCORE_RESPONSE_0:0xf:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw::offcore_response_0:0xfffffffff",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0xfffffffff,
	  .fstr = "hsw::OFFCORE_RESPONSE_0:0xffffffff:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw::offcore_response_0:0x7fffffffff",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "bdw_ep::offcore_response_0:0xf",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0xf,
	  .fstr = "bdw_ep::OFFCORE_RESPONSE_0:0xf:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "bdw_ep::offcore_response_0:0xfffffffff",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0xfffffffff,
	  .fstr = "bdw_ep::OFFCORE_RESPONSE_0:0xffffffff:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "bdw_ep::offcore_response_0:0x7fffffffff",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "skl::offcore_response_0:0xf",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0xf,
	  .fstr = "skl::OFFCORE_RESPONSE_0:0xf:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "skl::offcore_response_0:0xfffffffff",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0xfffffffff,
	  .fstr = "skl::OFFCORE_RESPONSE_0:0xffffffff:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "skl::offcore_response_0:0x7fffffffff",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "wsm::offcore_response_1:0xfff",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0xfff,
	  .fstr = "wsm::OFFCORE_RESPONSE_1:0xfff:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm::offcore_response_1:0x7fffffffff",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_1:0xf",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0xf,
	  .fstr = "snb::OFFCORE_RESPONSE_1:0xf:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_1:0xfffffffff",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0xfffffffff,
	  .fstr = "snb::OFFCORE_RESPONSE_1:0xffffffff:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_1:0x7fffffffff",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "ivb_ep::offcore_response_1:0xf",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0xf,
	  .fstr = "ivb_ep::OFFCORE_RESPONSE_1:0xf:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb_ep::offcore_response_1:0xfffffffff",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0xfffffffff,
	  .fstr = "ivb_ep::OFFCORE_RESPONSE_1:0xffffffff:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb_ep::offcore_response_1:0x7fffffffff",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "hsw::offcore_response_1:0xf",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0xf,
	  .fstr = "hsw::OFFCORE_RESPONSE_1:0xf:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw::offcore_response_1:0xfffffffff",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0xfffffffff,
	  .fstr = "hsw::OFFCORE_RESPONSE_1:0xffffffff:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw::offcore_response_1:0x7fffffffff",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "bdw_ep::offcore_response_1:0xf",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0xf,
	  .fstr = "bdw_ep::OFFCORE_RESPONSE_1:0xf:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "bdw_ep::offcore_response_1:0xfffffffff",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0xfffffffff,
	  .fstr = "bdw_ep::OFFCORE_RESPONSE_1:0xffffffff:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "bdw_ep::offcore_response_1:0x7fffffffff",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "skl::offcore_response_1:0xf",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0xf,
	  .fstr = "skl::OFFCORE_RESPONSE_1:0xf:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "skl::offcore_response_1:0xfffffffff",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0xfffffffff,
	  .fstr = "skl::OFFCORE_RESPONSE_1:0xffffffff:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "skl::offcore_response_1:0x7fffffffff",
	  .ret  = PFM_ERR_ATTR,
	},

	{ SRC_LINE,
	  .name = "bdx_unc_cbo1::UNC_C_CLOCKTICKS:u",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo0::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "bdx_unc_cbo0::UNC_C_CLOCKTICKS:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo1::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "bdx_unc_cbo1::UNC_C_CLOCKTICKS:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo2::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "bdx_unc_cbo2::UNC_C_CLOCKTICKS:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo3::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "bdx_unc_cbo3::UNC_C_CLOCKTICKS:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo4::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "bdx_unc_cbo4::UNC_C_CLOCKTICKS:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo5::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "bdx_unc_cbo5::UNC_C_CLOCKTICKS:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo6::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "bdx_unc_cbo6::UNC_C_CLOCKTICKS:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo7::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "bdx_unc_cbo7::UNC_C_CLOCKTICKS:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo8::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "bdx_unc_cbo8::UNC_C_CLOCKTICKS:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo9::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "bdx_unc_cbo9::UNC_C_CLOCKTICKS:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo10::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "bdx_unc_cbo10::UNC_C_CLOCKTICKS:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo11::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "bdx_unc_cbo11::UNC_C_CLOCKTICKS:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo12::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "bdx_unc_cbo12::UNC_C_CLOCKTICKS:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo13::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "bdx_unc_cbo13::UNC_C_CLOCKTICKS:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo14::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "bdx_unc_cbo14::UNC_C_CLOCKTICKS:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo15::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "bdx_unc_cbo15::UNC_C_CLOCKTICKS:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo16::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "bdx_unc_cbo16::UNC_C_CLOCKTICKS:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo17::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "bdx_unc_cbo17::UNC_C_CLOCKTICKS:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo18::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "bdx_unc_cbo18::UNC_C_CLOCKTICKS:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo19::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "bdx_unc_cbo19::UNC_C_CLOCKTICKS:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo20::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "bdx_unc_cbo20::UNC_C_CLOCKTICKS:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo21::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "bdx_unc_cbo21::UNC_C_CLOCKTICKS:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo0::UNC_C_LLC_LOOKUP:DATA_READ",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x334,
	  .codes[1] = 0xfe0000,
	  .fstr = "bdx_unc_cbo0::UNC_C_LLC_LOOKUP:DATA_READ:STATE_MESIFD:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo0::UNC_C_LLC_LOOKUP:DATA_READ:nf=1",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo0::UNC_C_LLC_LOOKUP",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x1134,
	  .codes[1] = 0xfe0000,
	  .fstr = "bdx_unc_cbo0::UNC_C_LLC_LOOKUP:ANY:STATE_MESIFD:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo0::UNC_C_LLC_LOOKUP:NID:STATE_M",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo0::UNC_C_LLC_LOOKUP:NID:nf=3",
	  .ret  = PFM_SUCCESS,
	  .count = 3,
	  .codes[0] = 0x5134,
	  .codes[1] = 0xfe0000,
	  .codes[2] = 0x3,
	  .fstr = "bdx_unc_cbo0::UNC_C_LLC_LOOKUP:ANY:NID:STATE_MESIFD:e=0:t=0:tf=0:nf=3:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo0::UNC_C_LLC_LOOKUP:NID:STATE_M:tid=1",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo0::UNC_C_ring_iv_used:DN:UP",
	  .ret  = PFM_ERR_FEATCOMB,
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo0::UNC_C_LLC_LOOKUP:WRITE:NID:nf=3:tf=1:e:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 3,
	  .codes[0] = 0x10c4534,
	  .codes[1] = 0xfe0001,
	  .codes[2] = 0x3,
	  .fstr = "bdx_unc_cbo0::UNC_C_LLC_LOOKUP:NID:WRITE:STATE_MESIFD:e=1:t=1:tf=1:nf=3:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo0::UNC_C_LLC_VICTIMS",
	  .ret  = PFM_ERR_UMASK,
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo0::UNC_C_LLC_VICTIMS:NID",
	  .ret  = PFM_ERR_UMASK,
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo0::UNC_C_LLC_VICTIMS:NID:nf=1",
	  .ret  = PFM_ERR_UMASK,
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo0::UNC_C_LLC_VICTIMS:M_STATE",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x137,
	  .fstr = "bdx_unc_cbo0::UNC_C_LLC_VICTIMS:M_STATE:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo0::UNC_C_LLC_VICTIMS:M_STATE:S_STATE",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x537,
	  .fstr = "bdx_unc_cbo0::UNC_C_LLC_VICTIMS:S_STATE:M_STATE:e=0:t=0:tf=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo0::UNC_C_LLC_VICTIMS:M_STATE:S_STATE:NID:nf=1",
	  .ret  = PFM_SUCCESS,
	  .count = 3,
	  .codes[0] = 0x4537,
	  .codes[1] = 0x0,
	  .codes[2] = 0x1,
	  .fstr = "bdx_unc_cbo0::UNC_C_LLC_VICTIMS:S_STATE:M_STATE:NID:e=0:t=0:tf=0:nf=1:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo0::UNC_C_TOR_INSERTS:OPCODE",
	  .ret  = PFM_ERR_UMASK,
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo0::UNC_C_TOR_INSERTS:WB",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1035,
	  .fstr = "bdx_unc_cbo0::UNC_C_TOR_INSERTS:WB:e=0:t=0:tf=0:isoc=0:nc=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo0::UNC_C_TOR_INSERTS:OPCODE:OPC_ITOM",
	  .ret  = PFM_SUCCESS,
	  .count = 3,
	  .codes[0] = 0x135,
	  .codes[1] = 0x0,
	  .codes[2] = 0x1c800000ull,
	  .fstr = "bdx_unc_cbo0::UNC_C_TOR_INSERTS:OPCODE:OPC_ITOM:e=0:t=0:tf=0:isoc=0:nc=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo0::UNC_C_TOR_INSERTS:OPCODE:OPC_ITOM:isoc=1",
	  .ret  = PFM_SUCCESS,
	  .count = 3,
	  .codes[0] = 0x135,
	  .codes[1] = 0x0,
	  .codes[2] = 0x9c800000ull,
	  .fstr = "bdx_unc_cbo0::UNC_C_TOR_INSERTS:OPCODE:OPC_ITOM:e=0:t=0:tf=0:isoc=1:nc=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo0::UNC_C_TOR_INSERTS:OPCODE:OPC_PCIWILF:nf=1",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo0::UNC_C_TOR_INSERTS:NID_OPCODE:OPC_PCIRDCUR:nf=1",
	  .ret  = PFM_SUCCESS,
	  .count = 3,
	  .codes[0] = 0x4135,
	  .codes[1] = 0x0,
	  .codes[2] = 0x19e00001ull,
	  .fstr = "bdx_unc_cbo0::UNC_C_TOR_INSERTS:NID_OPCODE:OPC_PCIRDCUR:e=0:t=0:tf=0:nf=1:isoc=0:nc=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo0::UNC_C_TOR_INSERTS:OPC_RFO:NID_OPCODE:nf=1",
	  .ret  = PFM_SUCCESS,
	  .count = 3,
	  .codes[0] = 0x4135,
	  .codes[1] = 0x0,
	  .codes[2] = 0x18000001ull,
	  .fstr = "bdx_unc_cbo0::UNC_C_TOR_INSERTS:NID_OPCODE:OPC_RFO:e=0:t=0:tf=0:nf=1:isoc=0:nc=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_cbo0::UNC_C_TOR_OCCUPANCY:MISS_REMOTE",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x8a36,
	  .fstr = "bdx_unc_cbo0::UNC_C_TOR_OCCUPANCY:MISS_REMOTE:e=0:t=0:tf=0:isoc=0:nc=0:cf=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_irp::unc_i_clockticks",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "bdx_unc_irp::UNC_I_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_irp::unc_i_coherent_ops:RFO",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x813,
	  .fstr = "bdx_unc_irp::UNC_I_COHERENT_OPS:RFO:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_irp::unc_i_transactions:reads",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x116,
	  .fstr = "bdx_unc_irp::UNC_I_TRANSACTIONS:READS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_irp::unc_i_transactions:reads:c=1:i",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "bdx_unc_irp::unc_i_transactions:reads:t=6",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x6000116,
	  .fstr = "bdx_unc_irp::UNC_I_TRANSACTIONS:READS:e=0:i=0:t=6",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_sbo0::unc_s_clockticks",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "bdx_unc_sbo0::UNC_S_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_sbo1::unc_s_clockticks",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "bdx_unc_sbo1::UNC_S_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_sbo2::unc_s_clockticks",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "bdx_unc_sbo2::UNC_S_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_sbo3::unc_s_clockticks",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "bdx_unc_sbo3::UNC_S_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_pcu::UNC_P_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "bdx_unc_pcu::UNC_P_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_pcu::UNC_P_CLOCKTICKS:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1000000,
	  .fstr = "bdx_unc_pcu::UNC_P_CLOCKTICKS:e=0:i=0:t=1",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_pcu::UNC_P_CORE0_TRANSITION_CYCLES",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x60,
	  .fstr = "bdx_unc_pcu::UNC_P_CORE0_TRANSITION_CYCLES:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_pcu::UNC_P_CORE17_TRANSITION_CYCLES",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x71,
	  .fstr = "bdx_unc_pcu::UNC_P_CORE17_TRANSITION_CYCLES:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_pcu::UNC_P_FREQ_BAND0_CYCLES",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "bdx_unc_pcu::UNC_P_FREQ_BAND1_CYCLES",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "bdx_unc_pcu::UNC_P_FREQ_BAND2_CYCLES",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "bdx_unc_pcu::UNC_P_FREQ_BAND3_CYCLES",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "bdx_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:ff=32",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0xb,
	  .codes[1] = 0x20,
	  .fstr = "bdx_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:e=0:i=0:t=0:ff=32",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_pcu::UNC_P_FREQ_BAND1_CYCLES:ff=16",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0xc,
	  .codes[1] = 0x1000,
	  .fstr = "bdx_unc_pcu::UNC_P_FREQ_BAND1_CYCLES:e=0:i=0:t=0:ff=16",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_pcu::UNC_P_FREQ_BAND2_CYCLES:ff=8",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0xd,
	  .codes[1] = 0x80000,
	  .fstr = "bdx_unc_pcu::UNC_P_FREQ_BAND2_CYCLES:e=0:i=0:t=0:ff=8",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_pcu::UNC_P_FREQ_BAND3_CYCLES:ff=40",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0xe,
	  .codes[1] = 0x28000000,
	  .fstr = "bdx_unc_pcu::UNC_P_FREQ_BAND3_CYCLES:e=0:i=0:t=0:ff=40",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:ff=32:e",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x4000b,
	  .codes[1] = 0x20,
	  .fstr = "bdx_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:e=1:i=0:t=0:ff=32",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:ff=32:t=24",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x1800000b,
	  .codes[1] = 0x20,
	  .fstr = "bdx_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:e=0:i=0:t=24:ff=32",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:ff=32:e:t=4",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x404000b,
	  .codes[1] = 0x20,
	  .fstr = "bdx_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:e=1:i=0:t=4:ff=32",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x4080,
	  .fstr = "bdx_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0:e=0:i=0:t=0"
	},
	{ SRC_LINE,
	  .name = "bdx_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C3",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x8080,
	  .fstr = "bdx_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C3:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C6",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xc080,
	  .fstr = "bdx_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C6:e=0:i=0:t=0"
	},
	{ SRC_LINE,
	  .name = "bdx_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0:t=6:i",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x46004080,
	  .fstr = "bdx_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0:e=0:i=1:t=6"
	},
	{ SRC_LINE,
	  .name = "bdx_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0:t=6",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x6004080,
	  .fstr = "bdx_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0:e=0:i=0:t=6"
	},
	{ SRC_LINE,
	  .name = "bdx_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0:t=6:i:e",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xc6004080,
	  .fstr = "bdx_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0:e=1:i=1:t=6"
	},
	{ SRC_LINE,
	  .name = "bdx_unc_pcu::UNC_P_DEMOTIONS_CORE10",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x3a,
	  .fstr = "bdx_unc_pcu::UNC_P_DEMOTIONS_CORE10:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_pcu::UNC_P_DEMOTIONS_CORE14",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x3e,
	  .fstr = "bdx_unc_pcu::UNC_P_DEMOTIONS_CORE14:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_pcu::UNC_P_DEMOTIONS_CORE17",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x41,
	  .fstr = "bdx_unc_pcu::UNC_P_DEMOTIONS_CORE17:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_ha0::UNC_H_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "bdx_unc_ha0::UNC_H_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_ha1::UNC_H_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "bdx_unc_ha1::UNC_H_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_ha1::UNC_H_REQUESTS:READS:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1000301,
	  .fstr = "bdx_unc_ha1::UNC_H_REQUESTS:READS:e=0:i=0:t=1",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_ha0::UNC_H_IMC_WRITES:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1000f1a,
	  .fstr = "bdx_unc_ha0::UNC_H_IMC_WRITES:ALL:e=0:i=0:t=1",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_ha0::UNC_H_IMC_READS:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1000117,
	  .fstr = "bdx_unc_ha0::UNC_H_IMC_READS:NORMAL:e=0:i=0:t=1",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_imc0::UNC_M_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xff,
	  .fstr = "bdx_unc_imc0::UNC_M_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_imc1::UNC_M_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xff,
	  .fstr = "bdx_unc_imc1::UNC_M_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_imc2::UNC_M_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xff,
	  .fstr = "bdx_unc_imc2::UNC_M_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_imc3::UNC_M_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xff,
	  .fstr = "bdx_unc_imc3::UNC_M_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_imc4::UNC_M_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xff,
	  .fstr = "bdx_unc_imc4::UNC_M_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_imc5::UNC_M_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xff,
	  .fstr = "bdx_unc_imc5::UNC_M_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_imc6::UNC_M_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xff,
	  .fstr = "bdx_unc_imc6::UNC_M_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_imc7::UNC_M_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xff,
	  .fstr = "bdx_unc_imc7::UNC_M_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_imc0::UNC_M_CLOCKTICKS:t=1",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "bdx_unc_imc0::UNC_M_DCLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "bdx_unc_imc0::UNC_M_DCLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_imc4::UNC_M_DCLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "bdx_unc_imc4::UNC_M_DCLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_imc0::UNC_M_CAS_COUNT:RD",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0304,
	  .fstr = "bdx_unc_imc0::UNC_M_CAS_COUNT:RD:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_imc0::UNC_M_PRE_COUNT:WR",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0802,
	  .fstr = "bdx_unc_imc0::UNC_M_PRE_COUNT:WR:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_imc0::UNC_M_POWER_CKE_CYCLES:RANK0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x183,
	  .fstr = "bdx_unc_imc0::UNC_M_POWER_CKE_CYCLES:RANK0:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_imc0::UNC_M_CAS_COUNT:WR",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xc04,
	  .fstr = "bdx_unc_imc0::UNC_M_CAS_COUNT:WR:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_imc0::UNC_M_RD_CAS_RANK0:BANK0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xb0,
	  .fstr = "bdx_unc_imc0::UNC_M_RD_CAS_RANK0:BANK0:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_imc0::UNC_M_RD_CAS_RANK0:ALLBANKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x10b0,
	  .fstr = "bdx_unc_imc0::UNC_M_RD_CAS_RANK0:ALLBANKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_imc0::UNC_M_RD_CAS_RANK0:BANKG0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x11b0,
	  .fstr = "bdx_unc_imc0::UNC_M_RD_CAS_RANK0:BANKG0:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_imc0::UNC_M_RD_CAS_RANK4:BANK7",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x7b4,
	  .fstr = "bdx_unc_imc0::UNC_M_RD_CAS_RANK4:BANK7:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_imc0::UNC_M_RD_CAS_RANK4:BANK7:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x10007b4,
	  .fstr = "bdx_unc_imc0::UNC_M_RD_CAS_RANK4:BANK7:e=0:i=0:t=1",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_imc0::UNC_M_RD_CAS_RANK7:BANK7:t=1:i",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x18007b7,
	  .fstr = "bdx_unc_imc0::UNC_M_RD_CAS_RANK7:BANK7:e=0:i=1:t=1",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_sbo0::UNC_S_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "bdx_unc_sbo0::UNC_S_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_sbo0::UNC_S_FAST_ASSERTED:t=1:i",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1800009,
	  .fstr = "bdx_unc_sbo0::UNC_S_FAST_ASSERTED:e=0:i=1:t=1",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_sbo3::UNC_S_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "bdx_unc_sbo3::UNC_S_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_sbo3::UNC_S_FAST_ASSERTED:t=1:i",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1800009,
	  .fstr = "bdx_unc_sbo3::UNC_S_FAST_ASSERTED:e=0:i=1:t=1",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_ubo::UNC_U_EVENT_MSG",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x842,
	  .fstr = "bdx_unc_ubo::UNC_U_EVENT_MSG:DOORBELL_RCVD:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_qpi0::UNC_Q_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x14,
	  .fstr = "bdx_unc_qpi0::UNC_Q_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_qpi1::UNC_Q_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x14,
	  .fstr = "bdx_unc_qpi1::UNC_Q_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_qpi2::UNC_Q_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x14,
	  .fstr = "bdx_unc_qpi2::UNC_Q_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_qpi0::UNC_Q_DIRECT2CORE:SUCCESS_RBT_HIT",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x113,
	  .fstr = "bdx_unc_qpi0::UNC_Q_DIRECT2CORE:SUCCESS_RBT_HIT:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_qpi0::UNC_Q_RXL_FLITS_G1:DRS:i:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1a01802,
	  .fstr = "bdx_unc_qpi0::UNC_Q_RXL_FLITS_G1:DRS:e=0:i=1:t=1",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_qpi0::UNC_Q_TXL_FLITS_G2:NDR_AD",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x200101,
	  .fstr = "bdx_unc_qpi0::UNC_Q_TXL_FLITS_G2:NDR_AD:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_qpi0::UNC_Q_RXL_OCCUPANCY",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xb,
	  .fstr = "bdx_unc_qpi0::UNC_Q_RXL_OCCUPANCY:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_qpi0::UNC_Q_TXL_INSERTS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x4,
	  .fstr = "bdx_unc_qpi0::UNC_Q_TXL_INSERTS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_r2pcie::UNC_R2_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1,
	  .fstr = "bdx_unc_r2pcie::UNC_R2_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_r2pcie::UNC_R2_RING_AD_USED:CW",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x307,
	  .fstr = "bdx_unc_r2pcie::UNC_R2_RING_AD_USED:CW:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_r3qpi0::UNC_R3_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1,
	  .fstr = "bdx_unc_r3qpi0::UNC_R3_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_r3qpi0::UNC_R3_RXR_CYCLES_NE:SNP:e=0:t=0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x210,
	  .fstr = "bdx_unc_r3qpi0::UNC_R3_RXR_CYCLES_NE:SNP:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_r3qpi1::UNC_R3_RING_SINK_STARVED",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x20e,
	  .fstr = "bdx_unc_r3qpi1::UNC_R3_RING_SINK_STARVED:AK:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "bdx_unc_r3qpi1::UNC_R3_HA_R2_BL_CREDITS_EMPTY:HA1:i:t=2",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x280022d,
	  .fstr = "bdx_unc_r3qpi1::UNC_R3_HA_R2_BL_CREDITS_EMPTY:HA1:e=0:i=1:t=2",
	},
	{ SRC_LINE,
	  .name = "amd64_fam17h::retired_uops",
	  .count = 1,
	  .codes[0] = 0x5300c1ull,
	  .fstr = "amd64_fam17h::RETIRED_UOPS:k=1:u=1:e=0:i=0:c=0:h=0:g=0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam17h::cycles_not_in_halt",
	  .count = 1,
	  .codes[0] = 0x530076ull,
	  .fstr = "amd64_fam17h::CYCLES_NOT_IN_HALT:k=1:u=1:e=0:i=0:c=0:h=0:g=0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam17h::locks:spec_lock",
	  .count = 1,
	  .codes[0] = 0x530425ull,
	  .fstr = "amd64_fam17h::LOCKS:SPEC_LOCK:k=1:u=1:e=0:i=0:c=0:h=0:g=0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam17h::L1_DTLB_MISS:TLB_RELOAD_1G_L2_HIT:u",
	  .count = 1,
	  .codes[0] = 0x510845ull,
	  .fstr = "amd64_fam17h::L1_DTLB_MISS:TLB_RELOAD_1G_L2_HIT:k=0:u=1:e=0:i=0:c=0:h=0:g=0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam16h::RETIRED_INSTRUCTIONS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5300c0,
	  .fstr = "amd64_fam16h::RETIRED_INSTRUCTIONS:k=1:u=1:e=0:i=0:c=0:h=0:g=0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam16h::CPU_CLK_UNHALTED:u",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x510076,
	  .fstr = "amd64_fam16h::CPU_CLK_UNHALTED:k=0:u=1:e=0:i=0:c=0:h=0:g=0",
	},
	{ SRC_LINE,
	  .name = "skx::offcore_response_1:pf_l1d_and_sw",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x10400,
	  .fstr = "skx::OFFCORE_RESPONSE_1:PF_L1D_AND_SW:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "skx::offcore_response_0:any_request",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x185b7,
	  .fstr = "skx::OFFCORE_RESPONSE_0:DMND_DATA_RD:DMND_RFO:DMND_CODE_RD:PF_L2_DATA_RD:PF_L2_RFO:PF_L3_DATA_RD:PF_L3_RFO:PF_L1D_AND_SW:OTHER:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "skx::offcore_response_0:snp_any",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x3f800185b7ull,
	  .fstr = "skx::OFFCORE_RESPONSE_0:ANY_REQUEST:ANY_RESPONSE:SNP_NONE:SNP_NOT_NEEDED:SNP_MISS:SNP_HIT_NO_FWD:SNP_HIT_WITH_FWD:SNP_HITM:SNP_NON_DRAM:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "skx::offcore_response_0:l3_hitmesf",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x3f803c85b7ull,
	  .fstr = "skx::OFFCORE_RESPONSE_0:ANY_REQUEST:L3_HITM:L3_HITE:L3_HITS:L3_HITF:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "skx::offcore_response_0:L4_HIT_LOCAL_L4",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "skx::offcore_response_0:L3_MISS_LOCAL",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x3f840085b7ull,
	  .fstr = "skx::OFFCORE_RESPONSE_0:ANY_REQUEST:L3_MISS_LOCAL:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "skx::offcore_response_0:l3_miss",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] =0x5301b7,
	  .codes[1] = 0x3f840085b7ull,
	  .fstr = "skx::OFFCORE_RESPONSE_0:ANY_REQUEST:L3_MISS_LOCAL:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "skx::mem_load_uops_l3_miss_retired:remote_hitm",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] =0x5304d3,
	  .fstr = "skx::MEM_LOAD_L3_MISS_RETIRED:REMOTE_HITM:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "skx::mem_load_uops_l3_miss_retired:local_dram",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5301d3,
	  .fstr = "skx::MEM_LOAD_L3_MISS_RETIRED:LOCAL_DRAM:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "skx::mem_load_uops_l3_hit_retired:xsnp_hit",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] =0x5302d2,
	  .fstr = "skx::MEM_LOAD_L3_HIT_RETIRED:XSNP_HIT:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},

	{ SRC_LINE,
	  .name = "skx_unc_cha1::UNC_C_CLOCKTICKS:u",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha0::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "skx_unc_cha0::UNC_C_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha1::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "skx_unc_cha1::UNC_C_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha2::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "skx_unc_cha2::UNC_C_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha3::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "skx_unc_cha3::UNC_C_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha4::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "skx_unc_cha4::UNC_C_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha5::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "skx_unc_cha5::UNC_C_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha6::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "skx_unc_cha6::UNC_C_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha7::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "skx_unc_cha7::UNC_C_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha8::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "skx_unc_cha8::UNC_C_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha9::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "skx_unc_cha9::UNC_C_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha10::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "skx_unc_cha10::UNC_C_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha11::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "skx_unc_cha11::UNC_C_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha12::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "skx_unc_cha12::UNC_C_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha13::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "skx_unc_cha13::UNC_C_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha14::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "skx_unc_cha14::UNC_C_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha15::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "skx_unc_cha15::UNC_C_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha16::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "skx_unc_cha16::UNC_C_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha17::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "skx_unc_cha17::UNC_C_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha18::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "skx_unc_cha18::UNC_C_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha19::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "skx_unc_cha19::UNC_C_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha20::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "skx_unc_cha20::UNC_C_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha21::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "skx_unc_cha21::UNC_C_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha22::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "skx_unc_cha22::UNC_C_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha23::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "skx_unc_cha23::UNC_C_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha24::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "skx_unc_cha24::UNC_C_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha25::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "skx_unc_cha25::UNC_C_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha26::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "skx_unc_cha26::UNC_C_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha27::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "skx_unc_cha27::UNC_C_CLOCKTICKS:e=0:i=0:t=0",
	},

	{ SRC_LINE,
	  .name = "skx_unc_cha0::UNC_C_LLC_LOOKUP:DATA_READ",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x334l,
	  .codes[1] = 0xfe0000,
	  .fstr = "skx_unc_cha0::UNC_C_LLC_LOOKUP:DATA_READ:STATE_CACHE_ANY:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha0::UNC_C_TOR_INSERTS:IA_MISS",
	  .ret  = PFM_SUCCESS,
	  .count = 3,
	  .codes[0] = 0x2135,
	  .codes[1] = 0x0,
	  .codes[2] = 0x3b,
	  .fstr = "skx_unc_cha0::UNC_C_TOR_INSERTS:IA_MISS:e=0:i=0:t=0:isoc=0:nc=0:loc=1:rem=1:lmem=1:rmem=1",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha0::UNC_C_TOR_INSERTS:IRQ:OPC0_RFO:OPC1_RFO",
	  .ret  = PFM_SUCCESS,
	  .count = 3,
	  .codes[0] = 0x0135,
	  .codes[1] = 0x0,
	  .codes[2] = 0x10040000,
	  .fstr = "skx_unc_cha0::UNC_C_TOR_INSERTS:IRQ:OPC0_RFO:OPC1_RFO:e=0:i=0:t=0:isoc=0:nc=0:loc=0:rem=0:lmem=0:rmem=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha0::UNC_C_TOR_INSERTS:IRQ:OPC0_RFO:OPC1_RFO:lmem=1",
	  .ret  = PFM_SUCCESS,
	  .count = 3,
	  .codes[0] = 0x0135,
	  .codes[1] = 0x0,
	  .codes[2] = 0x10040002,
	  .fstr = "skx_unc_cha0::UNC_C_TOR_INSERTS:IRQ:OPC0_RFO:OPC1_RFO:e=0:i=0:t=0:isoc=0:nc=0:loc=1:rem=0:lmem=1:rmem=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha0::UNC_C_TOR_INSERTS:IPQ:OPC0_RFO:OPC1_RFO",
	  .ret  = PFM_ERR_FEATCOMB,
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha0::UNC_C_TOR_INSERTS:IRQ:OPC0_RFO:OPC1_DRD:lmem=0:rmem=1",
	  .ret  = PFM_SUCCESS,
	  .count = 3,
	  .codes[0] = 0x0135,
	  .codes[1] = 0x0,
	  .codes[2] = 0x10140020,
	  .fstr = "skx_unc_cha0::UNC_C_TOR_INSERTS:IRQ:OPC0_RFO:OPC1_DRD:e=0:i=0:t=0:isoc=0:nc=0:loc=0:rem=0:lmem=0:rmem=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha0::UNC_C_TOR_INSERTS:IRQ:OPC1_RFO:OPC0_DRD:lmem=0:rmem=1",
	  .ret  = PFM_SUCCESS,
	  .count = 3,
	  .codes[0] = 0x0135,
	  .codes[1] = 0x0,
	  .codes[2] = 0x10040420,
	  .fstr = "skx_unc_cha0::UNC_C_TOR_INSERTS:IRQ:OPC0_DRD:OPC1_RFO:e=0:i=0:t=0:isoc=0:nc=0:loc=0:rem=0:lmem=0:rmem=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha0::UNC_C_LLC_LOOKUP:LOCAL:DATA_READ",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x3334,
	  .codes[1] = 0xfe0000,
	  .fstr = "skx_unc_cha0::UNC_C_LLC_LOOKUP:DATA_READ:LOCAL:STATE_CACHE_ANY:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha0::UNC_C_LLC_LOOKUP:ANY",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x1134,
	  .codes[1] = 0xfe0000,
	  .fstr = "skx_unc_cha0::UNC_C_LLC_LOOKUP:ANY:STATE_CACHE_ANY:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha0::UNC_C_LLC_LOOKUP:REMOTE:STATE_LLC_S",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x9134,
	  .codes[1] = 0x200000,
	  .fstr = "skx_unc_cha0::UNC_C_LLC_LOOKUP:REMOTE:STATE_LLC_S:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_cha0::UNC_C_LLC_VICTIMS:local_all",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x2f37,
	  .fstr = "skx_unc_cha0::UNC_C_LLC_VICTIMS:LOCAL_ALL:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_irp::unc_i_clockticks",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1,
	  .fstr = "skx_unc_irp::UNC_I_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_irp::unc_i_coherent_ops:RFO",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x810,
	  .fstr = "skx_unc_irp::UNC_I_COHERENT_OPS:RFO:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_irp::unc_i_transactions:reads",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x111,
	  .fstr = "skx_unc_irp::UNC_I_TRANSACTIONS:READS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_irp::unc_i_transactions:reads:c=1:i",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "skx_unc_irp::unc_i_transactions:reads:t=6",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x6000111,
	  .fstr = "skx_unc_irp::UNC_I_TRANSACTIONS:READS:e=0:i=0:t=6",
	},
	{ SRC_LINE,
	  .name = "skx_unc_pcu::UNC_P_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "skx_unc_pcu::UNC_P_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_pcu::UNC_P_CLOCKTICKS:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1000000,
	  .fstr = "skx_unc_pcu::UNC_P_CLOCKTICKS:e=0:i=0:t=1",
	},
	{ SRC_LINE,
	  .name = "skx_unc_pcu::UNC_P_CORE_TRANSITION_CYCLES",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x60,
	  .fstr = "skx_unc_pcu::UNC_P_CORE_TRANSITION_CYCLES:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_pcu::UNC_P_FREQ_BAND0_CYCLES",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "skx_unc_pcu::UNC_P_FREQ_BAND1_CYCLES",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "skx_unc_pcu::UNC_P_FREQ_BAND2_CYCLES",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "skx_unc_pcu::UNC_P_FREQ_BAND3_CYCLES",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "skx_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:ff=32",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0xb,
	  .codes[1] = 0x20,
	  .fstr = "skx_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:e=0:i=0:t=0:ff=32",
	},
	{ SRC_LINE,
	  .name = "skx_unc_pcu::UNC_P_FREQ_BAND1_CYCLES:ff=16",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0xc,
	  .codes[1] = 0x1000,
	  .fstr = "skx_unc_pcu::UNC_P_FREQ_BAND1_CYCLES:e=0:i=0:t=0:ff=16",
	},
	{ SRC_LINE,
	  .name = "skx_unc_pcu::UNC_P_FREQ_BAND2_CYCLES:ff=8",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0xd,
	  .codes[1] = 0x80000,
	  .fstr = "skx_unc_pcu::UNC_P_FREQ_BAND2_CYCLES:e=0:i=0:t=0:ff=8",
	},
	{ SRC_LINE,
	  .name = "skx_unc_pcu::UNC_P_FREQ_BAND3_CYCLES:ff=40",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0xe,
	  .codes[1] = 0x28000000,
	  .fstr = "skx_unc_pcu::UNC_P_FREQ_BAND3_CYCLES:e=0:i=0:t=0:ff=40",
	},
	{ SRC_LINE,
	  .name = "skx_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:ff=32:e",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x4000b,
	  .codes[1] = 0x20,
	  .fstr = "skx_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:e=1:i=0:t=0:ff=32",
	},
	{ SRC_LINE,
	  .name = "skx_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:ff=32:t=24",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x1800000b,
	  .codes[1] = 0x20,
	  .fstr = "skx_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:e=0:i=0:t=24:ff=32",
	},
	{ SRC_LINE,
	  .name = "skx_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:ff=32:e:t=4",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x404000b,
	  .codes[1] = 0x20,
	  .fstr = "skx_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:e=1:i=0:t=4:ff=32",
	},
	{ SRC_LINE,
	  .name = "skx_unc_pcu::UNC_P_DEMOTIONS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x30,
	  .fstr = "skx_unc_pcu::UNC_P_DEMOTIONS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_imc0::UNC_M_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "skx_unc_imc0::UNC_M_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_imc1::UNC_M_CLOCKTICKS:e=0:i=0:t=0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "skx_unc_imc1::UNC_M_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_imc2::UNC_M_CLOCKTICKS:e=0:i=0:t=0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "skx_unc_imc2::UNC_M_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_imc3::UNC_M_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "skx_unc_imc3::UNC_M_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_imc4::UNC_M_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "skx_unc_imc4::UNC_M_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_imc5::UNC_M_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "skx_unc_imc5::UNC_M_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_imc0::UNC_M_DCLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xff,
	  .fstr = "skx_unc_imc0::UNC_M_DCLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "skx_unc_imc0::UNC_M_DCLOCKTICKS:e=1",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "skx_unc_imc4::UNC_M_DCLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xff,
	  .fstr = "skx_unc_imc4::UNC_M_DCLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "skx_unc_imc0::UNC_M_CAS_COUNT:RD",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0304,
	  .fstr = "skx_unc_imc0::UNC_M_CAS_COUNT:RD:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_imc0::UNC_M_PRE_COUNT:WR",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0802,
	  .fstr = "skx_unc_imc0::UNC_M_PRE_COUNT:WR:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_imc0::UNC_M_POWER_CKE_CYCLES:RANK0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x183,
	  .fstr = "skx_unc_imc0::UNC_M_POWER_CKE_CYCLES:RANK0:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_imc0::UNC_M_CAS_COUNT:WR",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xc04,
	  .fstr = "skx_unc_imc0::UNC_M_CAS_COUNT:WR:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_imc0::UNC_M_RD_CAS_RANK0:BANK0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xb0,
	  .fstr = "skx_unc_imc0::UNC_M_RD_CAS_RANK0:BANK0:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_imc0::UNC_M_RD_CAS_RANK0:ALLBANKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x10b0,
	  .fstr = "skx_unc_imc0::UNC_M_RD_CAS_RANK0:ALLBANKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_imc0::UNC_M_RD_CAS_RANK0:BANKG0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x11b0,
	  .fstr = "skx_unc_imc0::UNC_M_RD_CAS_RANK0:BANKG0:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_imc0::UNC_M_RD_CAS_RANK4:BANK7",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x7b4,
	  .fstr = "skx_unc_imc0::UNC_M_RD_CAS_RANK4:BANK7:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_imc0::UNC_M_RD_CAS_RANK4:BANK7:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x10007b4,
	  .fstr = "skx_unc_imc0::UNC_M_RD_CAS_RANK4:BANK7:e=0:i=0:t=1",
	},
	{ SRC_LINE,
	  .name = "skx_unc_imc0::UNC_M_RD_CAS_RANK7:BANK7:t=1:i",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x18007b7,
	  .fstr = "skx_unc_imc0::UNC_M_RD_CAS_RANK7:BANK7:e=0:i=1:t=1",
	},
	{ SRC_LINE,
	  .name = "skx_unc_ubo::UNC_U_EVENT_MSG:DOORBELL_RCVD",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x842,
	  .fstr = "skx_unc_ubo::UNC_U_EVENT_MSG:DOORBELL_RCVD:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_upi0::UNC_UPI_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1,
	  .fstr = "skx_unc_upi0::UNC_UPI_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_upi1::UNC_UPI_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1,
	  .fstr = "skx_unc_upi1::UNC_UPI_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_upi2::UNC_UPI_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1,
	  .fstr = "skx_unc_upi2::UNC_UPI_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_upi0::UNC_UPI_RXL_BASIC_HDR_MATCH:REQ",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x805,
	  .codes[1] = 0x0,
	  .fstr = "skx_unc_upi0::UNC_UPI_RXL_BASIC_HDR_MATCH:REQ:FILT_NONE:e=0:i=0:t=0:dnid=0:rcsnid=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_upi0::UNC_UPI_RXL_BASIC_HDR_MATCH:REQ_OPC",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "skx_unc_upi0::UNC_UPI_RXL_BASIC_HDR_MATCH:REQ_OPC_RDINV:WB",
	  .ret  = PFM_ERR_FEATCOMB,
	},
	{ SRC_LINE,
	  .name = "skx_unc_upi0::UNC_UPI_RXL_BASIC_HDR_MATCH:REQ_OPC_RDINV",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0xc805ull,
	  .codes[1] = 0x1,
	  .fstr = "skx_unc_upi0::UNC_UPI_RXL_BASIC_HDR_MATCH:REQ_OPC_RDINV:FILT_NONE:e=0:i=0:t=0:dnid=0:rcsnid=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_upi0::UNC_UPI_RXL_BASIC_HDR_MATCH:REQ_OPC_RDINV:FILT_SLOT2",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0xc805,
	  .codes[1] = 0x200001,
	  .fstr = "skx_unc_upi0::UNC_UPI_RXL_BASIC_HDR_MATCH:REQ_OPC_RDINV:FILT_SLOT2:e=0:i=0:t=0:dnid=0:rcsnid=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_upi0::UNC_UPI_RXL_BASIC_HDR_MATCH:REQ_OPC_RDINV:FILT_SLOT2:FILT_SLOT1",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0xc805,
	  .codes[1] = 0x300001,
	  .fstr = "skx_unc_upi0::UNC_UPI_RXL_BASIC_HDR_MATCH:REQ_OPC_RDINV:FILT_SLOT1:FILT_SLOT2:e=0:i=0:t=0:dnid=0:rcsnid=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_upi0::UNC_UPI_RXL_BASIC_HDR_MATCH:NCB_OPC_NCWR:FILT_SLOT2:FILT_IMPL_NULL:dnid=1:FILT_REMOTE",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0xe05,
	  .codes[1] = 0xa02105,
	  .fstr = "skx_unc_upi0::UNC_UPI_RXL_BASIC_HDR_MATCH:NCB_OPC_NCWR:FILT_REMOTE:FILT_SLOT2:FILT_IMPL_NULL:e=0:i=0:t=0:dnid=1:rcsnid=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_upi1::UNC_UPI_RXL_BASIC_HDR_MATCH:WB_OPC_WBMTOI:FILT_SLOT2:t=2:dnid=1:FILT_REMOTE:FILT_LOCAL",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x2000d05,
	  .codes[1] = 0x202107,
	  .fstr = "skx_unc_upi1::UNC_UPI_RXL_BASIC_HDR_MATCH:WB_OPC_WBMTOI:FILT_LOCAL:FILT_REMOTE:FILT_SLOT2:e=0:i=0:t=2:dnid=1:rcsnid=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_upi1::UNC_UPI_RXL_BASIC_HDR_MATCH:WB:t=2:i:e=1:dnid=1:FILT_REMOTE:FILT_LOCAL",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x2840d05,
	  .codes[1] = 0x2106,
	  .fstr = "skx_unc_upi1::UNC_UPI_RXL_BASIC_HDR_MATCH:WB:FILT_LOCAL:FILT_REMOTE:e=1:i=1:t=2:dnid=1:rcsnid=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_iio0::UNC_IO_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1,
	  .fstr = "skx_unc_iio0::UNC_IO_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_iio0::UNC_IO_DATA_REQ_OF_CPU:MEM_READ_PART0:FC_ANY",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x701000000483ull,
	  .fstr = "skx_unc_iio0::UNC_IO_DATA_REQ_OF_CPU:MEM_READ_PART0:FC_POSTED_REQ:FC_NON_POSTED_REQ:FC_CMPL:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_iio0::UNC_IO_DATA_REQ_OF_CPU:MEM_READ_PART0:FC_POSTED_REQ:FC_NON_POSTED_REQ",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x301000000483ull,
	  .fstr = "skx_unc_iio0::UNC_IO_DATA_REQ_OF_CPU:MEM_READ_PART0:FC_POSTED_REQ:FC_NON_POSTED_REQ:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_iio0::UNC_IO_DATA_REQ_OF_CPU:MEM_READ_PART0:MEM_READ_PART1:FC_ANY",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x703000000483ull,
	  .fstr = "skx_unc_iio0::UNC_IO_DATA_REQ_OF_CPU:MEM_READ_PART0:MEM_READ_PART1:FC_POSTED_REQ:FC_NON_POSTED_REQ:FC_CMPL:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_iio0::UNC_IO_DATA_REQ_OF_CPU:MEM_READ_PART0:MEM_READ_PART1:FC_ANY:t=2:i",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x703002800483ull,
	  .fstr = "skx_unc_iio0::UNC_IO_DATA_REQ_OF_CPU:MEM_READ_PART0:MEM_READ_PART1:FC_POSTED_REQ:FC_NON_POSTED_REQ:FC_CMPL:e=0:i=1:t=2",
	},

	{ SRC_LINE,
	  .name = "skx_unc_irp::UNC_I_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1,
	  .fstr = "skx_unc_irp::UNC_I_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_irp::UNC_I_FAF_INSERTS:t=2:i",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x2800018,
	  .fstr = "skx_unc_irp::UNC_I_FAF_INSERTS:e=0:i=1:t=2",
	},

	{ SRC_LINE,
	  .name = "skx_unc_m2m0::UNC_M2_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "skx_unc_m2m0::UNC_M2_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_m2m0::UNC_M2_DIRECTORY_LOOKUP",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x12d,
	  .fstr = "skx_unc_m2m0::UNC_M2_DIRECTORY_LOOKUP:ANY:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_m2m1::UNC_M2_DIRECTORY_LOOKUP:STATE_A:t=2",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x200082d,
	  .fstr = "skx_unc_m2m1::UNC_M2_DIRECTORY_LOOKUP:STATE_A:e=0:i=0:t=2",
	},
	{ SRC_LINE,
	  .name = "skx_unc_m2m1::UNC_M2_CMS_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xc0,
	  .fstr = "skx_unc_m2m1::UNC_M2_CMS_CLOCKTICKS:e=0:i=0:t=0",
	},

	{ SRC_LINE,
	  .name = "skx_unc_m3upi0::UNC_M3_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1,
	  .fstr = "skx_unc_m3upi0::UNC_M3_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_m3upi1::UNC_M3_CMS_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xc0,
	  .fstr = "skx_unc_m3upi1::UNC_M3_CMS_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "skx_unc_m3upi1::UNC_M3_AG1_BL_CREDITS_ACQUIRED:TGR5:t=2:i",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x280208c,
	  .fstr = "skx_unc_m3upi1::UNC_M3_AG1_BL_CREDITS_ACQUIRED:TGR5:e=0:i=1:t=2",
	},

        { SRC_LINE,
          .name = "knm::no_alloc_cycles:all",
          .ret = PFM_SUCCESS,
          .count = 1,
          .codes[0] = 0x537fca,
          .fstr = "knm::NO_ALLOC_CYCLES:ALL:k=1:u=1:e=0:i=0:c=0",
        },
        { SRC_LINE,
          .name = "knm::MEM_UOPS_RETIRED:DTLB_MISS_LOADS",
          .ret = PFM_SUCCESS,
          .count = 1,
          .codes[0] = 0x530804,
          .fstr = "knm::MEM_UOPS_RETIRED:DTLB_MISS_LOADS:k=1:u=1:e=0:i=0:c=0",
        },
        { SRC_LINE,
          .name = "knm::uops_retired:any:t",
          .ret = PFM_ERR_ATTR,
        },
        { SRC_LINE,
          .name = "knm::unhalted_reference_cycles:u:t",
          .ret = PFM_SUCCESS,
          .count = 1,
          .codes[0] = 0x710300,
          .fstr = "knm::UNHALTED_REFERENCE_CYCLES:k=0:u=1:t=1",
        },
        { SRC_LINE,
          .name = "knm::instructions_retired:k:t",
          .ret = PFM_SUCCESS,
          .count = 1,
          .codes[0] =0x7200c0,
          .fstr = "knm::INSTRUCTION_RETIRED:k=1:u=0:e=0:i=0:c=0:t=1",
        },
        { SRC_LINE,
          .name = "knm::unhalted_core_cycles:k:t",
          .ret = PFM_SUCCESS,
          .count = 1,
          .codes[0] = 0x72003c,
          .fstr = "knm::UNHALTED_CORE_CYCLES:k=1:u=0:e=0:i=0:c=0:t=1",
        },
        { SRC_LINE,
          .name = "knm::offcore_response_1:any_request",
          .ret  = PFM_SUCCESS,
          .count = 2,
          .codes[0] = 0x5302b7,
          .codes[1] = 0x18000,
          .fstr = "knm::OFFCORE_RESPONSE_1:ANY_REQUEST:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0",
        },
        { SRC_LINE,
          .name = "knm::offcore_response_0:any_read",
          .ret  = PFM_SUCCESS,
          .count = 2,
          .codes[0] = 0x5301b7,
          .codes[1] = 0x132e7,
          .fstr = "knm::OFFCORE_RESPONSE_0:DMND_DATA_RD:DMND_RFO:DMND_CODE_RD:PF_L2_RFO:PF_L2_CODE_RD:PARTIAL_READS:UC_CODE_READS:PF_SOFTWARE:PF_L1_DATA_RD:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0",
        },
        { SRC_LINE,
          .name = "knm::offcore_response_1:any_read",
          .ret  = PFM_SUCCESS,
          .count = 2,
          .codes[0] = 0x5302b7,
          .codes[1] = 0x132e7,
          .fstr = "knm::OFFCORE_RESPONSE_1:DMND_DATA_RD:DMND_RFO:DMND_CODE_RD:PF_L2_RFO:PF_L2_CODE_RD:PARTIAL_READS:UC_CODE_READS:PF_SOFTWARE:PF_L1_DATA_RD:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0",
        },
        { SRC_LINE,
          .name = "knm::offcore_response_0:any_request:ddr_near",
          .ret  = PFM_SUCCESS,
          .count = 2,
          .codes[0] = 0x5301b7,
          .codes[1] = 0x80808000ull,
          .fstr = "knm::OFFCORE_RESPONSE_0:ANY_REQUEST:DDR_NEAR:k=1:u=1:e=0:i=0:c=0",
        },
        { SRC_LINE,
          .name = "knm::offcore_response_0:any_request:L2_HIT_NEAR_TILE:L2_HIT_FAR_TILE",
          .ret  = PFM_SUCCESS,
          .count = 2,
          .codes[0] = 0x5301b7,
          .codes[1] = 0x1800588000ull,
          .fstr = "knm::OFFCORE_RESPONSE_0:ANY_REQUEST:L2_HIT_NEAR_TILE:L2_HIT_FAR_TILE:k=1:u=1:e=0:i=0:c=0",
        },
        { SRC_LINE,
          .name = "knm::offcore_response_0:dmnd_data_rd:outstanding",
          .ret  = PFM_SUCCESS,
          .count = 2,
          .codes[0] = 0x5301b7,
          .codes[1] = 0x4000000001ull,
          .fstr = "knm::OFFCORE_RESPONSE_0:DMND_DATA_RD:OUTSTANDING:k=1:u=1:e=0:i=0:c=0",
        },
        { SRC_LINE,
          .name = "knm::offcore_response_0:dmnd_data_rd:ddr_near:outstanding",
          .ret  = PFM_ERR_FEATCOMB,
        },
        { SRC_LINE,
          .name = "knm::offcore_response_1:dmnd_data_rd:outstanding",
          .ret  = PFM_ERR_ATTR,
        },
	{ SRC_LINE,
	  .name = "knm_unc_imc0::UNC_M_D_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "knm_unc_imc0::UNC_M_D_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "knm_unc_imc0::UNC_M_CAS_COUNT:RD",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0103,
	  .fstr = "knm_unc_imc0::UNC_M_CAS_COUNT:RD",
	},
	{ SRC_LINE,
	  .name = "knm_unc_imc0::UNC_M_CAS_COUNT:WR",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0203,
	  .fstr = "knm_unc_imc0::UNC_M_CAS_COUNT:WR",
	},
	{ SRC_LINE,
	  .name = "knm_unc_imc0::UNC_M_CAS_COUNT:ALL",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0303,
	  .fstr = "knm_unc_imc0::UNC_M_CAS_COUNT:ALL",
        },
	{ SRC_LINE,
	  .name = "knm_unc_imc_uclk0::UNC_M_U_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "knm_unc_imc_uclk0::UNC_M_U_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "knm_unc_edc_uclk0::UNC_E_U_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "knm_unc_edc_uclk0::UNC_E_U_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "knm_unc_edc_uclk0::UNC_E_EDC_ACCESS:HIT_CLEAN",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0102,
	  .fstr = "knm_unc_edc_uclk0::UNC_E_EDC_ACCESS:HIT_CLEAN",
	},
	{ SRC_LINE,
	  .name = "knm_unc_edc_uclk0::UNC_E_EDC_ACCESS:HIT_DIRTY",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0202,
	  .fstr = "knm_unc_edc_uclk0::UNC_E_EDC_ACCESS:HIT_DIRTY",
	},
	{ SRC_LINE,
	  .name = "knm_unc_edc_uclk0::UNC_E_EDC_ACCESS:MISS_CLEAN",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0402,
	  .fstr = "knm_unc_edc_uclk0::UNC_E_EDC_ACCESS:MISS_CLEAN",
	},
	{ SRC_LINE,
	  .name = "knm_unc_edc_uclk0::UNC_E_EDC_ACCESS:MISS_DIRTY",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0802,
	  .fstr = "knm_unc_edc_uclk0::UNC_E_EDC_ACCESS:MISS_DIRTY",
	},
	{ SRC_LINE,
	  .name = "knm_unc_edc_uclk0::UNC_E_EDC_ACCESS:MISS_INVALID",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1002,
	  .fstr = "knm_unc_edc_uclk0::UNC_E_EDC_ACCESS:MISS_INVALID",
        },
	{ SRC_LINE,
	  .name = "knm_unc_edc_eclk0::UNC_E_E_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "knm_unc_edc_eclk0::UNC_E_E_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "knm_unc_edc_eclk0::UNC_E_RPQ_INSERTS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0101,
	  .fstr = "knm_unc_edc_eclk0::UNC_E_RPQ_INSERTS",
	},
	{ SRC_LINE,
	  .name = "knm_unc_cha0::UNC_H_U_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "knm_unc_cha0::UNC_H_U_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "knm_unc_cha1::UNC_H_U_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "knm_unc_cha1::UNC_H_U_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "knm_unc_cha10::UNC_H_U_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "knm_unc_cha10::UNC_H_U_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "knm_unc_cha20::UNC_H_U_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "knm_unc_cha20::UNC_H_U_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "knm_unc_cha25::UNC_H_U_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "knm_unc_cha25::UNC_H_U_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "knm_unc_cha30::UNC_H_U_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "knm_unc_cha30::UNC_H_U_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "knm_unc_cha37::UNC_H_U_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "knm_unc_cha37::UNC_H_U_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "knm_unc_cha0::UNC_H_INGRESS_OCCUPANCY:IRQ",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0111,
	  .fstr = "knm_unc_cha0::UNC_H_INGRESS_OCCUPANCY:IRQ",
	},
	{ SRC_LINE,
	  .name = "knm_unc_cha0::UNC_H_INGRESS_OCCUPANCY:IRQ_REJ",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0211,
	  .fstr = "knm_unc_cha0::UNC_H_INGRESS_OCCUPANCY:IRQ_REJ",
	},
	{ SRC_LINE,
	  .name = "knm_unc_cha0::UNC_H_INGRESS_OCCUPANCY:IPQ",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0411,
	  .fstr = "knm_unc_cha0::UNC_H_INGRESS_OCCUPANCY:IPQ",
	},
	{ SRC_LINE,
	  .name = "knm_unc_cha0::UNC_H_INGRESS_OCCUPANCY:PRQ",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1011,
	  .fstr = "knm_unc_cha0::UNC_H_INGRESS_OCCUPANCY:PRQ",
	},
	{ SRC_LINE,
	  .name = "knm_unc_cha0::UNC_H_INGRESS_OCCUPANCY:PRQ_REJ",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x2011,
	  .fstr = "knm_unc_cha0::UNC_H_INGRESS_OCCUPANCY:PRQ_REJ",
	},
	{ SRC_LINE,
	  .name = "knm_unc_cha0::UNC_H_INGRESS_INSERTS:IRQ",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0113,
	  .fstr = "knm_unc_cha0::UNC_H_INGRESS_INSERTS:IRQ",
	},
	{ SRC_LINE,
	  .name = "knm_unc_cha0::UNC_H_INGRESS_INSERTS:IRQ_REJ",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0213,
	  .fstr = "knm_unc_cha0::UNC_H_INGRESS_INSERTS:IRQ_REJ",
	},
	{ SRC_LINE,
	  .name = "knm_unc_cha0::UNC_H_INGRESS_INSERTS:IPQ",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0413,
	  .fstr = "knm_unc_cha0::UNC_H_INGRESS_INSERTS:IPQ",
	},
	{ SRC_LINE,
	  .name = "knm_unc_cha0::UNC_H_INGRESS_INSERTS:PRQ",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1013,
	  .fstr = "knm_unc_cha0::UNC_H_INGRESS_INSERTS:PRQ",
	},
	{ SRC_LINE,
	  .name = "knm_unc_cha0::UNC_H_INGRESS_INSERTS:PRQ_REJ",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x2013,
	  .fstr = "knm_unc_cha0::UNC_H_INGRESS_INSERTS:PRQ_REJ",
        },
	{ SRC_LINE,
	  .name = "knm_unc_cha0::UNC_H_INGRESS_RETRY_IRQ0_REJECT:AD_RSP_VN0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0218,
	  .fstr = "knm_unc_cha0::UNC_H_INGRESS_RETRY_IRQ0_REJECT:AD_RSP_VN0",
	},
	{ SRC_LINE,
	  .name = "knm_unc_m2pcie::UNC_M2P_INGRESS_CYCLES_NE:ALL",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0810,
	  .fstr = "knm_unc_m2pcie::UNC_M2P_INGRESS_CYCLES_NE:ALL",
	},
	{ SRC_LINE,
	  .name = "knm_unc_m2pcie::UNC_M2P_EGRESS_CYCLES_NE:AD_0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0123,
	  .fstr = "knm_unc_m2pcie::UNC_M2P_EGRESS_CYCLES_NE:AD_0",
	},
	{ SRC_LINE,
	  .name = "knm_unc_m2pcie::UNC_M2P_EGRESS_CYCLES_NE:AD_1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0823,
	  .fstr = "knm_unc_m2pcie::UNC_M2P_EGRESS_CYCLES_NE:AD_1",
	},
	{ SRC_LINE,
	  .name = "knm_unc_m2pcie::UNC_M2P_EGRESS_INSERTS:AD_0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0124,
	  .fstr = "knm_unc_m2pcie::UNC_M2P_EGRESS_INSERTS:AD_0",
	},
	{ SRC_LINE,
	  .name = "knm_unc_m2pcie::UNC_M2P_EGRESS_INSERTS:AD_1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1024,
	  .fstr = "knm_unc_m2pcie::UNC_M2P_EGRESS_INSERTS:AD_1",
	},
	{ SRC_LINE,
	  .name = "knm_unc_m2pcie::UNC_M2P_EGRESS_CYCLES_FULL:AD_0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0125,
	  .fstr = "knm_unc_m2pcie::UNC_M2P_EGRESS_CYCLES_FULL:AD_0",
	},
	{ SRC_LINE,
	  .name = "knm_unc_m2pcie::UNC_M2P_EGRESS_CYCLES_FULL:AD_1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0825,
	  .fstr = "knm_unc_m2pcie::UNC_M2P_EGRESS_CYCLES_FULL:AD_1",
	},
};

#define NUM_TEST_EVENTS (int)(sizeof(x86_test_events)/sizeof(test_event_t))

static int
check_pmu_supported(const char *evt)
{
	pfm_pmu_info_t info;
	char *p;
	int ret;
	pfm_pmu_t i;

	memset(&info, 0, sizeof(info));
	info.size = sizeof(info);

	/* look for pmu_name::.... */
	p = strchr(evt, ':');
	if (!p)
		return 1;
	if (*(p+1) != ':')
		return 1;

	pfm_for_all_pmus(i) {
		ret = pfm_get_pmu_info(i, &info);
		if (ret != PFM_SUCCESS)
			continue;
		if (!strncmp(info.name, evt, p - evt))
			return 1;
	}
	/* PMU not there */
	return 0;
}

static int check_test_events(FILE *fp)
{
	const test_event_t *e;
	char *fstr;
	uint64_t *codes;
	int count, i, j;
	int ret, errors = 0;

	for (i=0, e = x86_test_events; i < NUM_TEST_EVENTS; i++, e++) {
		codes = NULL;
		count = 0;
		fstr = NULL;
		ret = pfm_get_event_encoding(e->name, PFM_PLM0 | PFM_PLM3, &fstr, NULL, &codes, &count);
		if (ret != e->ret) {
			if (ret == PFM_ERR_NOTFOUND && !check_pmu_supported(e->name)) {
				fprintf(fp,"Line %d, Event%d %s, skipped because no PMU support\n", e->line, i, e->name);
				continue;
			}
			fprintf(fp,"Line %d, Event%d %s, ret=%s(%d) expected %s(%d)\n", e->line, i, e->name, pfm_strerror(ret), ret, pfm_strerror(e->ret), e->ret);
			errors++;
		} else {
			if (ret != PFM_SUCCESS) {
				if (fstr) {
					fprintf(fp,"Line %d, Event%d %s, expected fstr NULL but it is not\n", e->line, i, e->name);
					errors++;
				}
				if (count != 0) {
					fprintf(fp,"Line %d, Event%d %s, expected count=0 instead of %d\n", e->line, i, e->name, count);
					errors++;
				}
				if (codes) {
					fprintf(fp,"Line %d, Event%d %s, expected codes[] NULL but it is not\n", e->line, i, e->name);
					errors++;
				}
			} else {
				if (count != e->count) {
					fprintf(fp,"Line %d, Event%d %s, count=%d expected %d\n", e->line, i, e->name, count, e->count);
					errors++;
				}
				for (j=0; j < count; j++) {
					if (codes[j] != e->codes[j]) {
						fprintf(fp,"Line %d, Event%d %s, codes[%d]=%#"PRIx64" expected %#"PRIx64"\n", e->line, i, e->name, j, codes[j], e->codes[j]);
						errors++;
					}
				}
				if (e->fstr && strcmp(fstr, e->fstr)) {
					fprintf(fp,"Line %d, Event%d %s, fstr=%s expected %s\n", e->line, i, e->name, fstr, e->fstr);
					errors++;
				}
			}
		}
		if (codes)
			free(codes);
		if (fstr)
			free(fstr);
	}
	printf("\t %d x86 events: %d errors\n", i, errors);
	return errors;
}

int
validate_arch(FILE *fp)
{
	return check_test_events(fp);
}
