/*
A library to allow applictions to provide simple indications of
information to be displayed to users of the application through the
interface shell.

Copyright 2009 Canonical Ltd.

Authors:
    Ted Gould <ted@canonical.com>

This program is free software: you can redistribute it and/or modify it 
under the terms of either or both of the following licenses:

1) the GNU Lesser General Public License version 3, as published by the 
Free Software Foundation; and/or
2) the GNU Lesser General Public License version 2.1, as published by 
the Free Software Foundation.

This program is distributed in the hope that it will be useful, but 
WITHOUT ANY WARRANTY; without even the implied warranties of 
MERCHANTABILITY, SATISFACTORY QUALITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the applicable version of the GNU Lesser General Public 
License for more details.

You should have received a copy of both the GNU Lesser General Public 
License version 3 and version 2.1 along with this program.  If not, see 
<http://www.gnu.org/licenses/>
*/

#include <stdlib.h>
#include "listener.h"
#include "indicate-marshal.h"
#include <dbus/dbus-glib-bindings.h>
#include <dbus/dbus-glib-lowlevel.h>
#include "interests-priv.h"
#include "dbus-shared.h"
#include "indicate-interface-client.h"
#include "dbus-properties-client.h"

/* Errors */
enum {
	LAST_ERROR
};

/* Signals */
enum {
	INDICATOR_ADDED,
	INDICATOR_REMOVED,
	INDICATOR_MODIFIED,
	SERVER_ADDED,
	SERVER_REMOVED,
	SERVER_COUNT_CHANGED,
	INDICATOR_SERVERS_REPORT,
	LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0 };

#include "listener-private.h"

typedef struct {
	DBusGProxy * proxy;
	DBusGProxy * property_proxy;
	DBusGConnection * connection;
	gchar * name;
	gchar * path;
	gchar * type;
	IndicateListener * listener;
	GHashTable * indicators;
	guint introspect_level;
	gboolean hidden;

	IndicateListenerServer server;
} proxy_t;

static gint
proxy_t_equal (gconstpointer pa, gconstpointer pb)
{
	proxy_t * a = (proxy_t *)pa; proxy_t * b = (proxy_t *)pb;

	if (a->connection == b->connection) {
		return g_strcmp0(a->name, b->name);
	} else {
		/* we're only using this for equal, not sorting */
		return 1;
	}
}

typedef struct {
	DBusGConnection * bus;
	gchar * name;
	gchar * path;
} proxy_todo_t;

G_DEFINE_TYPE (IndicateListener, indicate_listener, G_TYPE_OBJECT);

/* Prototypes */
static void indicate_listener_finalize (GObject * obj);
static DBusHandlerResult dbus_filter_show_server (DBusConnection * connection, DBusMessage * message, void * user_data);
static void proxy_struct_destroy (gpointer data);
static void todo_list_add (const gchar * name, const gchar * path, IndicateListener * listener);
static gboolean todo_idle (gpointer data);
void set_max_indicators_cb (DBusGProxy * proxy, GError * error, gpointer userdata);
static void get_type_initial_cb (IndicateListener * listener, IndicateListenerServer * server, gchar * type, gpointer data);
static void get_type_cb (IndicateListener * listener, IndicateListenerServer * server, gchar * type, gpointer data);
static void proxy_indicator_added_legacy (DBusGProxy * proxy, guint id, gchar * type, proxy_t * proxyt);
static void proxy_indicator_added (DBusGProxy * proxy, guint id, proxy_t * proxyt);
static void proxy_indicator_removed_legacy (DBusGProxy * proxy, guint id, gchar * type, proxy_t * proxyt);
static void proxy_indicator_removed (DBusGProxy * proxy, guint id, proxy_t * proxyt);
static void proxy_indicator_modified (DBusGProxy * proxy, guint id, const gchar * property, proxy_t * proxyt);
static void proxy_server_count_changed (DBusGProxy * proxy, guint count, proxy_t * proxyt);
static void proxy_get_indicator_list (DBusGProxy * proxy, GArray * indicators, GError * error, gpointer data);
static void proxy_destroyed (GObject * proxy, gpointer user_data);

/* DBus interface */
gboolean _indicate_listener_server_get_indicator_servers (IndicateListener * listener, GList * servers);

/* Need the above prototypes */
#include "indicate-listener-server.h"

/* Code */
static void
indicate_listener_class_init (IndicateListenerClass * class)
{
	/* g_debug("Listener Class Initialized"); */
	GObjectClass * gobj;
	gobj = G_OBJECT_CLASS(class);

	g_type_class_add_private (class, sizeof (IndicateListenerPrivate));

	gobj->finalize = indicate_listener_finalize;

	signals[INDICATOR_ADDED] = g_signal_new(INDICATE_LISTENER_SIGNAL_INDICATOR_ADDED,
	                                        G_TYPE_FROM_CLASS (class),
	                                        G_SIGNAL_RUN_LAST,
	                                        G_STRUCT_OFFSET (IndicateListenerClass, indicator_added),
	                                        NULL, NULL,
	                                        _indicate_marshal_VOID__POINTER_POINTER,
	                                        G_TYPE_NONE, 2, INDICATE_TYPE_LISTENER_SERVER, INDICATE_TYPE_LISTENER_INDICATOR);
	signals[INDICATOR_REMOVED] = g_signal_new(INDICATE_LISTENER_SIGNAL_INDICATOR_REMOVED,
	                                        G_TYPE_FROM_CLASS (class),
	                                        G_SIGNAL_RUN_LAST,
	                                        G_STRUCT_OFFSET (IndicateListenerClass, indicator_removed),
	                                        NULL, NULL,
	                                        _indicate_marshal_VOID__POINTER_POINTER,
	                                        G_TYPE_NONE, 2, INDICATE_TYPE_LISTENER_SERVER, INDICATE_TYPE_LISTENER_INDICATOR);
	signals[INDICATOR_MODIFIED] = g_signal_new(INDICATE_LISTENER_SIGNAL_INDICATOR_MODIFIED,
	                                        G_TYPE_FROM_CLASS (class),
	                                        G_SIGNAL_RUN_LAST,
	                                        G_STRUCT_OFFSET (IndicateListenerClass, indicator_modified),
	                                        NULL, NULL,
	                                        _indicate_marshal_VOID__POINTER_POINTER_STRING,
	                                        G_TYPE_NONE, 3, INDICATE_TYPE_LISTENER_SERVER, INDICATE_TYPE_LISTENER_INDICATOR, G_TYPE_STRING);
	signals[SERVER_ADDED] = g_signal_new(INDICATE_LISTENER_SIGNAL_SERVER_ADDED,
	                                        G_TYPE_FROM_CLASS (class),
	                                        G_SIGNAL_RUN_LAST,
	                                        G_STRUCT_OFFSET (IndicateListenerClass, server_added),
	                                        NULL, NULL,
	                                        _indicate_marshal_VOID__POINTER_STRING,
	                                        G_TYPE_NONE, 2, INDICATE_TYPE_LISTENER_SERVER, G_TYPE_STRING);
	signals[SERVER_REMOVED] = g_signal_new(INDICATE_LISTENER_SIGNAL_SERVER_REMOVED,
	                                        G_TYPE_FROM_CLASS (class),
	                                        G_SIGNAL_RUN_LAST,
	                                        G_STRUCT_OFFSET (IndicateListenerClass, server_removed),
	                                        NULL, NULL,
	                                        _indicate_marshal_VOID__POINTER_STRING,
	                                        G_TYPE_NONE, 2, INDICATE_TYPE_LISTENER_SERVER, G_TYPE_STRING);
	signals[SERVER_COUNT_CHANGED] = g_signal_new(INDICATE_LISTENER_SIGNAL_SERVER_COUNT_CHANGED,
	                                        G_TYPE_FROM_CLASS (class),
	                                        G_SIGNAL_RUN_LAST,
	                                        G_STRUCT_OFFSET (IndicateListenerClass, server_count_changed),
	                                        NULL, NULL,
	                                        _indicate_marshal_VOID__POINTER_UINT,
	                                        G_TYPE_NONE, 2, INDICATE_TYPE_LISTENER_SERVER, G_TYPE_UINT);
	signals[INDICATOR_SERVERS_REPORT] = g_signal_new("indicator-servers-report",
	                                        G_TYPE_FROM_CLASS (class),
	                                        G_SIGNAL_RUN_LAST,
	                                        G_STRUCT_OFFSET (IndicateListenerClass, indicator_servers_report),
	                                        NULL, NULL,
	                                        g_cclosure_marshal_VOID__VOID,
	                                        G_TYPE_NONE, 0, G_TYPE_NONE);

	dbus_g_object_register_marshaller(_indicate_marshal_VOID__UINT_STRING,
	                                  G_TYPE_NONE,
	                                  G_TYPE_UINT,
	                                  G_TYPE_STRING,
	                                  G_TYPE_INVALID);

	dbus_g_object_type_install_info(INDICATE_TYPE_LISTENER,
	                                &dbus_glib__indicate_listener_server_object_info);

	return;
}

static void
indicate_listener_init (IndicateListener * listener)
{
	/* g_debug("Listener Object Initialized"); */
	IndicateListenerPrivate * priv = INDICATE_LISTENER_GET_PRIVATE(listener);
	GError *error = NULL;

	/* Get the buses */
	priv->session_bus = dbus_g_bus_get(DBUS_BUS_SESSION, &error);
	if (error != NULL) {
		g_error("Unable to get session bus: %s", error->message);
		g_error_free(error);
		return;
	}

	/* Initialize Data structures */
	priv->proxies = NULL;

	/* TODO: Look at some common scenarios and find out how to make this sized */
	priv->proxy_todo = g_array_new(FALSE, TRUE, sizeof(proxy_todo_t));
	priv->todo_idle = 0;

	priv->max_indicators = -1;

	dbus_g_connection_register_g_object(priv->session_bus,
	                                    "/org/ayatana/indicate/listener",
	                                    G_OBJECT(listener));

	/*            WARNING              */
	/* Starting massive asynchronisity */
	/*                                 */

	dbus_connection_add_filter(dbus_g_connection_get_connection(priv->session_bus), dbus_filter_show_server, listener, NULL);
	dbus_bus_add_match(dbus_g_connection_get_connection(priv->session_bus), "type='signal',interface='" INDICATE_DBUS_IFACE "',member='ServerShow'", NULL);

	g_signal_emit(G_OBJECT(listener), signals[INDICATOR_SERVERS_REPORT], 0, TRUE);

	return;
}

static void
indicate_listener_finalize (GObject * obj)
{
	IndicateListener * listener = INDICATE_LISTENER(obj);
	IndicateListenerPrivate * priv = INDICATE_LISTENER_GET_PRIVATE(listener);

	if (priv->todo_idle != 0) {
		g_idle_remove_by_data(obj);
	}
	/* Hack: proxy_struct_destroy() lacks a user_data parameter, but since the
	 * caller is responsible for handling params on the stack, it works
	 */
	g_list_foreach(priv->proxies, (GFunc)proxy_struct_destroy, NULL);
	g_list_free(priv->proxies);

	dbus_connection_remove_filter(dbus_g_connection_get_connection(priv->session_bus), dbus_filter_show_server, listener);

	G_OBJECT_CLASS (indicate_listener_parent_class)->finalize (obj);
	return;
}

/**
	indicate_listener_new:

	Creates a new Listener object.  Does not set this to the default
	listener object.

	Return value: A new listener object.
*/
IndicateListener *
indicate_listener_new (void)
{
	IndicateListener * listener;
	listener = g_object_new(INDICATE_TYPE_LISTENER, NULL);
	return listener;
}

/* The pointer to the default listener object */
static IndicateListener * default_indicate_listener = NULL;

/**
	indcate_listener_ref_default:

	Looks for the default listener, and if it doesn't exist it'll
	allocate a new one and mark it the default.  This is what most
	programs should use rather than #indicate_listener_new because
	it is rare that anyone would need more than a single listener.
	This function does increase the ref count on the object so anyone
	calling it should unref thier instance of the default listener.

	Return value: The default instance of #IndicateListener for the
		program.
*/
IndicateListener *
indicate_listener_ref_default (void)
{
	if (default_indicate_listener != NULL) {
		g_object_ref(default_indicate_listener);
	} else {
		default_indicate_listener = g_object_new(INDICATE_TYPE_LISTENER, NULL);
		g_object_add_weak_pointer(G_OBJECT(default_indicate_listener),
		                          (gpointer *)&default_indicate_listener);
	}

	return default_indicate_listener;
}

/* A small filter function that notices when someone sends
   a ServerShow signal and creates an entry for us to investigate
   them more in an idle loop lookup. */
static DBusHandlerResult
dbus_filter_show_server (DBusConnection * connection, DBusMessage * message, void * user_data)
{
	if (!dbus_message_is_signal(message, INDICATE_DBUS_IFACE, "ServerShow")) {
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}

	const char * sender = dbus_message_get_sender(message);
	const char * path = dbus_message_get_path(message);

	todo_list_add(sender, path, INDICATE_LISTENER(user_data));

	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

/* A hashtable for each function to look at all of the indicators
   on a proxy_t object and signal their destruction */
static void
proxy_struct_destroy_indicators (gpointer key, gpointer value, gpointer data)
{
	proxy_t * proxy_data = (proxy_t *)data;

	if (value) {
		g_signal_emit(proxy_data->listener, signals[INDICATOR_REMOVED], 0, &proxy_data->server, GUINT_TO_POINTER(key), TRUE);
	}
	return;
}

/* Cleans up a proxy_t struct after allocation.  It signals that
   all of the indicators are going away and the server itself. */
static void
proxy_struct_destroy (gpointer data)
{
	proxy_t * proxy_data = data;

	if (proxy_data->indicators != NULL) {
		g_hash_table_foreach(proxy_data->indicators,
							 proxy_struct_destroy_indicators,
							 proxy_data);
		g_hash_table_destroy(proxy_data->indicators);

		g_signal_emit(proxy_data->listener, signals[SERVER_REMOVED], 0, &proxy_data->server, proxy_data->type, TRUE);
		proxy_data->indicators = NULL;
	}

	if (DBUS_IS_G_PROXY(proxy_data->property_proxy)) {
		g_object_unref(G_OBJECT(proxy_data->property_proxy));
	}

	if (DBUS_IS_G_PROXY(proxy_data->proxy)) {
		g_signal_handlers_disconnect_by_func(proxy_data->proxy, proxy_destroyed, proxy_data);
		g_object_unref(G_OBJECT(proxy_data->proxy));
	}

	if (proxy_data->name != NULL) {
		g_free(proxy_data->name);
	}

	if (proxy_data->path != NULL) {
		g_free(proxy_data->path);
	}

	if (proxy_data->type != NULL) {
		g_free(proxy_data->type);
	}
	g_free(proxy_data);

	return;
}

/* Creates a todo list item for the particular server and
   path as they've signaled that they're on DBus and like
   talking about indicators. */
static void
todo_list_add (const gchar * name, const gchar * path, IndicateListener * listener)
{
	if (name == NULL || name[0] != ':') {
		return;
	}

	IndicateListenerPrivate * priv = INDICATE_LISTENER_GET_PRIVATE(listener);

	proxy_todo_t todo;
	todo.name = g_strdup(name);
	todo.path = g_strdup(path);
	todo.bus  = priv->session_bus;

	g_array_append_val(priv->proxy_todo, todo);

	if (priv->todo_idle == 0) {
		priv->todo_idle = g_idle_add(todo_idle, listener);
	}

	return;
}

/* This is the callback for when the proxy, which is attached
   to the name owner, is destroy (they fell off the bus) we handle
   that and free up our memory too. */
void
proxy_destroyed (GObject * proxy, gpointer user_data)
{
	proxy_t * proxyt = (proxy_t *)user_data;
	proxyt->proxy = NULL; /* Clear this so we don't get a double free on this guy */
	IndicateListener * listener = proxyt->listener;
	IndicateListenerPrivate * priv = INDICATE_LISTENER_GET_PRIVATE(listener);
	priv->proxies = g_list_remove_all(priv->proxies, proxyt);
	proxy_struct_destroy(proxyt);
	return;
}

gboolean
todo_idle (gpointer data)
{
	IndicateListener * listener = INDICATE_LISTENER(data);
	if (listener == NULL) {
		g_error("Listener got lost in todo_idle");
		return FALSE;
	}

	IndicateListenerPrivate * priv = INDICATE_LISTENER_GET_PRIVATE(listener);

	if (priv->proxy_todo->len == 0) {
		/* Basically if we have no todo, we need to stop running.  This
		 * is done this way to make the function error handling simpler
		 * and results in an extra run */
		priv->todo_idle = 0;
		return FALSE;
	}

	proxy_todo_t * todo = &g_array_index(priv->proxy_todo, proxy_todo_t, priv->proxy_todo->len - 1);
	/* Remove the todo list */
	priv->proxy_todo = g_array_remove_index(priv->proxy_todo, priv->proxy_todo->len - 1);

	/* Check to see if we already have this item, if so,
	   we assume that it's signal handler will handle the
	   ServerShow signal.  We're just going to exit this 
	   function. */
	if (TRUE) {
		proxy_t searchitem;
		searchitem.name = todo->name;
		searchitem.connection = todo->bus;

		GList * proxyitem = g_list_find_custom(priv->proxies, &searchitem, proxy_t_equal);
		if (proxyitem != NULL) {
			g_free(todo->name);
			g_free(todo->path);
			return TRUE;
		}
	}

	proxy_t * proxyt = g_new0(proxy_t, 1);
	proxyt->name = todo->name;
	proxyt->path = todo->path;
	proxyt->type = NULL;
	proxyt->property_proxy = NULL;
	proxyt->proxy = NULL;
	proxyt->listener = listener;
	proxyt->indicators = NULL;
	proxyt->hidden = FALSE;
	proxyt->connection = todo->bus;
	proxyt->server.name = todo->name;
	proxyt->server.proxy = NULL;
	proxyt->server.connection = proxyt->connection;
	proxyt->server.max_indicators = priv->max_indicators;

	/* Build the indicators hash */
	proxyt->indicators = g_hash_table_new(g_direct_hash, g_direct_equal);

	/* Build the proxy and ensure that it gets created.  If
	   it gets created we're all happy. */
	GError * error = NULL;
	proxyt->proxy = dbus_g_proxy_new_for_name_owner(todo->bus,
	                                                proxyt->name,
	                                                todo->path,
	                                                INDICATE_DBUS_IFACE,
	                                                &error);

	if (error != NULL) {
		g_warning("Unable to create proxy for %s", proxyt->name);
		g_error_free(error);
		return TRUE;
	}

	g_signal_connect(G_OBJECT(proxyt->proxy), "destroy", G_CALLBACK(proxy_destroyed), proxyt);

	/* Making sure the server has the proxy as well */
	proxyt->server.proxy = proxyt->proxy;

	/* Adding into the list of proxies */
	priv->proxies = g_list_prepend(priv->proxies, proxyt);

	/* Setup all the signals that we need off of this
	   proxy.  Lot's of fun signals. */
	dbus_g_proxy_add_signal(proxyt->proxy, "IndicatorAdded",
	                            G_TYPE_UINT, G_TYPE_STRING, G_TYPE_INVALID);
	dbus_g_proxy_connect_signal(proxyt->proxy, "IndicatorAdded",
	                            G_CALLBACK(proxy_indicator_added_legacy), proxyt, NULL);
	dbus_g_proxy_add_signal(proxyt->proxy, "IndicatorNew",
	                            G_TYPE_UINT, G_TYPE_INVALID);
	dbus_g_proxy_connect_signal(proxyt->proxy, "IndicatorNew",
	                            G_CALLBACK(proxy_indicator_added), proxyt, NULL);
	dbus_g_proxy_add_signal(proxyt->proxy, "IndicatorRemoved",
	                            G_TYPE_UINT, G_TYPE_STRING, G_TYPE_INVALID);
	dbus_g_proxy_connect_signal(proxyt->proxy, "IndicatorRemoved",
	                            G_CALLBACK(proxy_indicator_removed_legacy), proxyt, NULL);
	dbus_g_proxy_add_signal(proxyt->proxy, "IndicatorDelete",
	                            G_TYPE_UINT, G_TYPE_INVALID);
	dbus_g_proxy_connect_signal(proxyt->proxy, "IndicatorDelete",
	                            G_CALLBACK(proxy_indicator_removed), proxyt, NULL);
	dbus_g_proxy_add_signal(proxyt->proxy, "IndicatorModified",
	                            G_TYPE_UINT, G_TYPE_STRING, G_TYPE_INVALID);
	dbus_g_proxy_connect_signal(proxyt->proxy, "IndicatorModified",
	                            G_CALLBACK(proxy_indicator_modified), proxyt, NULL);
	dbus_g_proxy_add_signal(proxyt->proxy, "ServerCountChanged",
	                            G_TYPE_UINT, G_TYPE_INVALID);
	dbus_g_proxy_connect_signal(proxyt->proxy, "ServerCountChanged",
	                            G_CALLBACK(proxy_server_count_changed), proxyt, NULL);

	/* We're setting the max number of indicators from the default
	   when we detect it.  This should give a reasonable amount of
	   time for listeners to set the default if they want something
	   different from infinite.  Otherwise it'd be easy to miss the
	   first couple. */
	proxyt->server.max_indicators = priv->max_indicators;
	if (proxyt->server.max_indicators != -1) {
		org_ayatana_indicate_set_max_indicators_async(proxyt->proxy, proxyt->server.max_indicators, set_max_indicators_cb, proxyt->name);
	}

	indicate_listener_server_get_type(proxyt->listener, &proxyt->server, get_type_initial_cb, proxyt);
	org_ayatana_indicate_get_indicator_list_async(proxyt->proxy, proxy_get_indicator_list, proxyt);

	return TRUE;
}

/* A callback function for the getting the type when it is looked
   at initially.  So that we'll send a signal that the server has
   been added so that everyone knows. */
static void
get_type_initial_cb (IndicateListener * listener, IndicateListenerServer * server, gchar * type, gpointer data)
{
	get_type_cb(listener, server, type, data);

	proxy_t * proxyt = (proxy_t *)data;

	if (!proxyt->hidden && proxyt->type != NULL) {
		g_signal_emit(proxyt->listener, signals[SERVER_ADDED], 0, &proxyt->server, proxyt->type, TRUE);
	}

	return;
}

/* Callback from getting the type of the server.  We're not using
   this directly now, only through get_type_initial_cb right now
   though we might use it more directly later. */
static void
get_type_cb (IndicateListener * listener, IndicateListenerServer * server, gchar * type, gpointer data)
{
	if (type == NULL) {
		/* This is usually caused by an error getting the type,
		 * which would mean that this isn't an indicator server */
		return;
	}

	proxy_t * proxyt = (proxy_t *)data;

	if (proxyt->type != NULL) {
		g_free(proxyt->type);
		proxyt->type = NULL;
	}
	proxyt->type = g_strdup(type);

	return;
}

/* A call back from setting the max indicators.  We really can't
   do anything about it, so this function is kinda useless. */
void
set_max_indicators_cb (DBusGProxy * proxy, GError * error, gpointer userdata)
{
	if (error != NULL) {
		g_warning("Unable to set the max indicators on '%s': %s", (gchar *)userdata, error->message);
		g_error_free(error);
	}
	return;
}

/* Callback from the call to get the indicator list on new
   servers that we've found through introspection.  It takes
   the list and then calls indicator_added on each one. */
static void
proxy_get_indicator_list (DBusGProxy * proxy, GArray * indicators, GError * error, gpointer data)
{
	if (error != NULL) {
		return;
	}

	proxy_t * proxyt = (proxy_t *)data;

	int i;
	for (i = 0; i < indicators->len; i++) {
		guint id = g_array_index(indicators, guint, i);
		proxy_indicator_added(proxy, id, proxyt);
	}

	return;
}


/* A fun little wrapper so that we can support the
   signals on the indicator v1 interface.  It just drops
   the type and calls the new function. */
static void
proxy_indicator_added_legacy (DBusGProxy * proxy, guint id, gchar * type, proxy_t * proxyt)
{
	return proxy_indicator_added(proxy, id, proxyt);
}

/* Gets called when we get a signal from the server that
   there is a new indicator.  We put it into our list of
   indicators and pass the signal up.  If the server hasn't
   been known about before, we create the appropriate
   structures for it. */
static void
proxy_indicator_added (DBusGProxy * proxy, guint id, proxy_t * proxyt)
{
	g_debug("Proxy Indicator Added");
	g_return_if_fail(proxyt != NULL);

	if (!g_hash_table_lookup(proxyt->indicators, GUINT_TO_POINTER(id))) {
		g_hash_table_insert(proxyt->indicators, GUINT_TO_POINTER(id), GUINT_TO_POINTER(TRUE));
		g_signal_emit(proxyt->listener, signals[INDICATOR_ADDED], 0, &proxyt->server, GUINT_TO_POINTER(id), TRUE);
	}

	return;
}

/* A fun little wrapper so that we can support the
   signals on the indicator v1 interface.  It just drops
   the type and calls the new function. */
static void
proxy_indicator_removed_legacy (DBusGProxy * proxy, guint id, gchar * type, proxy_t * proxyt)
{
	return proxy_indicator_removed(proxy, id, proxyt);
}

/* A server removed an indicator.  This function removes all the
   local data structures and then passes the signal up the stack.
   */
static void
proxy_indicator_removed (DBusGProxy * proxy, guint id, proxy_t * proxyt)
{
	g_debug("Proxy Indicator Removed");
	g_return_if_fail(proxyt != NULL);

	if (proxyt->indicators == NULL) {
		g_warning("Oddly we had an indicator removed from an interface that we didn't think had indicators.");
		return;
	}

	if (!g_hash_table_lookup(proxyt->indicators, GUINT_TO_POINTER(id))) {
		g_warning("No indicator %d on '%s'.", id, proxyt->name);
		return;
	}

	g_hash_table_remove(proxyt->indicators, GUINT_TO_POINTER(id));
	g_signal_emit(proxyt->listener, signals[INDICATOR_REMOVED], 0, &proxyt->server, GUINT_TO_POINTER(id), TRUE);

	return;
}

/* This is a signal from the server that a property on an indicator
   has been modified.  We try and find the indicator, convert all
   the parameters to the local ones, and then pass up the signal. */
static void
proxy_indicator_modified (DBusGProxy * proxy, guint id, const gchar * property, proxy_t * proxyt)
{
	g_debug("Proxy Indicator Modified");
	g_return_if_fail(proxyt != NULL);

	if (proxyt->indicators == NULL) {
		g_warning("Oddly we had an indicator modified from an interface that we didn't think had indicators.");
		return;
	}

	if (!g_hash_table_lookup(proxyt->indicators, GUINT_TO_POINTER(id))) {
		g_warning("Can not modify indicator %d with property '%s' as there are no indicators with that id on %s.", id, property, proxyt->name);
		return;
	}

	g_signal_emit(proxyt->listener, signals[INDICATOR_MODIFIED], 0, &proxyt->server, GUINT_TO_POINTER(id), property, TRUE);

	return;
}

/* This function gets called when the dbus count
   signal comes it.  Basically we're just translating
   it into a local signal with the appropraite parameters
   and structures. */
static void
proxy_server_count_changed (DBusGProxy * proxy, guint count, proxy_t * proxyt)
{
	g_debug("Proxy Server Count Changed");
	g_return_if_fail(proxyt != NULL);
	g_signal_emit(proxyt->listener, signals[SERVER_COUNT_CHANGED], 0, &proxyt->server, count, TRUE);
	return;
}

typedef enum _get_property_type get_property_type;
enum _get_property_type {
	PROPERTY_TYPE_VALUE,
	PROPERTY_TYPE_STRING,
	PROPERTY_TYPE_TIME,
	PROPERTY_TYPE_INT,
	PROPERTY_TYPE_BOOL
};

typedef struct _get_property_t get_property_t;
struct _get_property_t {
	GCallback cb;
	gpointer data;
	IndicateListener * listener;
	IndicateListenerServer * server;
	IndicateListenerIndicator * indicator;
	gchar * property;
	get_property_type type;
};

/* Look at the right align on this comment.  Sweeeeeeeet */
/* A callback from getting a property that takes the string 
   passed across the bus and turning it into something more
   related to what we want on this side.  If it's a time it
   gets converted to a #GTimeVal, if it's an int it goes to
   a gint and if it's a bool we check that too.  This makes
   it nice to work with properties and the listener.
*/
static void
get_property_cb (DBusGProxy *proxy, GValue OUT_value, GError *error, gpointer userdata)
{
	get_property_t * get_property_data = (get_property_t *)userdata;

	if (error != NULL) {
		g_warning("Unable to get property data: %s", error->message);
		g_error_free(error);
		return;
	}

	switch (get_property_data->type) {
	case PROPERTY_TYPE_VALUE: {
		/* Just pass the GValue along. */
		indicate_listener_get_property_value_cb cb =(indicate_listener_get_property_value_cb)get_property_data->cb;
		cb(get_property_data->listener, get_property_data->server, get_property_data->indicator, get_property_data->property, &OUT_value, get_property_data->data);
		break;
	}
	case PROPERTY_TYPE_STRING: {
		/* Just pass the string along. */
		indicate_listener_get_property_cb cb = (indicate_listener_get_property_cb)get_property_data->cb;
		cb(get_property_data->listener, get_property_data->server, get_property_data->indicator, get_property_data->property, g_value_get_string(&OUT_value), get_property_data->data);
		break;
	}
	case PROPERTY_TYPE_TIME: {
		/* Convert it to a time val */
		indicate_listener_get_property_time_cb cb = (indicate_listener_get_property_time_cb)get_property_data->cb;
		GTimeVal time;
		if (g_time_val_from_iso8601(g_value_get_string(&OUT_value), &time)) {
			cb(get_property_data->listener, get_property_data->server, get_property_data->indicator, get_property_data->property, &time, get_property_data->data);
		}
		break;
	}
	case PROPERTY_TYPE_INT: {
		/* Take the string and convert it to an integer */
		indicate_listener_get_property_int_cb cb = (indicate_listener_get_property_int_cb)get_property_data->cb;
		cb(get_property_data->listener, get_property_data->server, get_property_data->indicator, get_property_data->property, g_value_get_int(&OUT_value), get_property_data->data);
		break;
	}
	case PROPERTY_TYPE_BOOL: {
		/* Check to see if it's 'true', if not assume that
		   it's false */
		indicate_listener_get_property_bool_cb cb = (indicate_listener_get_property_bool_cb)get_property_data->cb;
		cb(get_property_data->listener, get_property_data->server, get_property_data->indicator, get_property_data->property, g_value_get_boolean(&OUT_value), get_property_data->data);
		break;
	}
	}

	g_free(get_property_data->property);
	g_free(get_property_data);

	return;
};

/* A small function to take the common list of parameters and
   build a callback structure to hold them all.  Eventually we
   get the data and unwind this structure. */
static void
get_property_helper (IndicateListener * listener, IndicateListenerServer * server, IndicateListenerIndicator * indicator, gchar * property, GCallback callback, gpointer data, get_property_type prop_type)
{
	/* g_debug("get_property_helper: %s %d", property, prop_type); */
	/* TODO: Do we need to somehow refcount the server/indicator while we're waiting on this? */
	get_property_t * get_property_data = g_new0(get_property_t, 1);
	get_property_data->cb = callback;
	get_property_data->data = data;
	get_property_data->listener = listener;
	get_property_data->server = server;
	get_property_data->indicator = indicator;
	get_property_data->property = g_strdup(property);
	get_property_data->type = prop_type;
	
	org_ayatana_indicate_get_indicator_property_async (server->proxy , INDICATE_LISTENER_INDICATOR_ID(indicator), property, get_property_cb, get_property_data);
	return;
}

/**
	indicate_listener_get_property_value:
	@listener: The #IndicateListener representing the connection
	@server: The server that the indicator is on
	@indicator: Which indicator is being queried
	@property: Name of the property to get
	@callback: The callback function to call with the data
	@data: Arbitrary data to give the callback

	A function to get a property from an indicator on a server
	and bring it back locally.  This wraps all the hassle of using
	the DBus API and makes it pretty easy to get properties.

	This function gets the raw gvalue data, without any conversion.
*/
void
indicate_listener_get_property_value (IndicateListener * listener, IndicateListenerServer * server, IndicateListenerIndicator * indicator, gchar * property, indicate_listener_get_property_value_cb callback, gpointer data)
{
	return get_property_helper(listener, server, indicator, property, G_CALLBACK(callback), data, PROPERTY_TYPE_VALUE);
}

/**
	indicate_listener_get_property:
	@listener: The #IndicateListener representing the connection
	@server: The server that the indicator is on
	@indicator: Which indicator is being queried
	@property: Name of the property to get
	@callback: The callback function to call with the data
	@data: Arbitrary data to give the callback

	A function to get a property from an indicator on a server
	and bring it back locally.  This wraps all the hassle of using
	the DBus API and makes it pretty easy to get properties.
*/
void
indicate_listener_get_property (IndicateListener * listener, IndicateListenerServer * server, IndicateListenerIndicator * indicator, gchar * property, indicate_listener_get_property_cb callback, gpointer data)
{
	return get_property_helper(listener, server, indicator, property, G_CALLBACK(callback), data, PROPERTY_TYPE_STRING);
}

/**
	indicate_listener_get_property_time:
	@listener: The #IndicateListener representing the connection
	@server: The server that the indicator is on
	@indicator: Which indicator is being queried
	@property: Name of the property to get
	@callback: The callback function to call with the data
	@data: Arbitrary data to give the callback

	A function to get a property from an indicator on a server
	and bring it back locally.  This wraps all the hassle of using
	the DBus API and makes it pretty easy to get properties.

	Very similar to #indicate_listener_get_property but converts
	the final value into a GTimeVal for easy (and type-safe)
	usage by listeners.
*/
void
indicate_listener_get_property_time (IndicateListener * listener, IndicateListenerServer * server, IndicateListenerIndicator * indicator, gchar * property, indicate_listener_get_property_time_cb callback, gpointer data)
{
	return get_property_helper(listener, server, indicator, property, G_CALLBACK(callback), data, PROPERTY_TYPE_TIME);
}

/**
	indicate_listener_get_property_int:
	@listener: The #IndicateListener representing the connection
	@server: The server that the indicator is on
	@indicator: Which indicator is being queried
	@property: Name of the property to get
	@callback: The callback function to call with the data
	@data: Arbitrary data to give the callback

	A function to get a property from an indicator on a server
	and bring it back locally.  This wraps all the hassle of using
	the DBus API and makes it pretty easy to get properties.

	Very similar to #indicate_listener_get_property but converts
	the final value into a gint for easy (and type-safe)
	usage by listeners.
*/
void
indicate_listener_get_property_int (IndicateListener * listener, IndicateListenerServer * server, IndicateListenerIndicator * indicator, gchar * property, indicate_listener_get_property_int_cb callback, gpointer data)
{
	return get_property_helper(listener, server, indicator, property, G_CALLBACK(callback), data, PROPERTY_TYPE_INT);
}

/**
	indicate_listener_get_property_bool:
	@listener: The #IndicateListener representing the connection
	@server: The server that the indicator is on
	@indicator: Which indicator is being queried
	@property: Name of the property to get
	@callback: The callback function to call with the data
	@data: Arbitrary data to give the callback

	A function to get a property from an indicator on a server
	and bring it back locally.  This wraps all the hassle of using
	the DBus API and makes it pretty easy to get properties.

	Very similar to #indicate_listener_get_property but converts
	the final value into a gboolean for easy (and type-safe)
	usage by listeners.
*/
void
indicate_listener_get_property_bool (IndicateListener * listener, IndicateListenerServer * server, IndicateListenerIndicator * indicator, gchar * property, indicate_listener_get_property_bool_cb callback, gpointer data)
{
	return get_property_helper(listener, server, indicator, property, G_CALLBACK(callback), data, PROPERTY_TYPE_BOOL);
}


gboolean
_indicate_listener_server_get_indicator_servers (IndicateListener * listener, GList * servers)
{

	return TRUE;
}

/* A callback for asking an indicator to be displayed,
   which is unlikely to fail.  So we're throwing a warning. */
static void 
listener_display_cb (DBusGProxy *proxy, GError *error, gpointer userdata)
{
	if (error != NULL) {
		g_warning("Listener display caused an error: %s", error->message);
	}
	return;
}

void
indicate_listener_display (IndicateListener * listener, IndicateListenerServer * server, IndicateListenerIndicator * indicator, guint timestamp)
{
	org_ayatana_indicate_show_indicator_to_user_async (server->proxy, INDICATE_LISTENER_INDICATOR_ID(indicator), timestamp, listener_display_cb, NULL);

	return;
}

/* A callback for saying an indicator is displayed,
   which is unlikely to fail.  So we're throwing a warning. */
static void 
listener_displayed_cb (DBusGProxy *proxy, GError *error, gpointer userdata)
{
	if (error != NULL) {
		g_warning("Listener displayed caused an error: %s", error->message);
	}
	return;
}

/**
	indicate_listener_displayed:
	@listener: The #IndicateListener representing the connection
	@server: The server that the indicator is on
	@indicator: Which indicator is being displayed
	@displayed: Whether it's being displayed or hidden

	This function tells the client whether we're showing this indicator
	to the user.  This doesn't mean that it's necissarilly visible right
	now, but more that there is a way for the user to get to this item
	individually.
*/
void
indicate_listener_displayed (IndicateListener * listener, IndicateListenerServer * server, IndicateListenerIndicator * indicator, gboolean displayed)
{
	org_ayatana_indicate_indicator_displayed_async (server->proxy,
	                                                     INDICATE_LISTENER_INDICATOR_ID(indicator),
	                                                     displayed,
	                                                     listener_displayed_cb,
	                                                     NULL);

	if (!server->interests[INDICATE_INTEREST_INDICATOR_DISPLAY] && displayed) {
		g_warning("It's awful odd that you said in the interest survey you weren't displaying indicators and then you displayed one.  I'm just saying, you've probably confused someone besides me.");
	}

	return;
}

typedef struct {
	IndicateListener * listener;
	IndicateListenerServer * server;
	indicate_listener_get_server_property_cb cb;
	indicate_listener_get_server_uint_property_cb cb_uint;
	gpointer data;
} property_cb_t;

/* A callback from getting the property off of the server
   which unravels the property_cb_t structure that was passed
   as data and calls back the call back that was in it with
   the appropriate data, also unrolled. */
static void
property_cb (DBusGProxy * proxy, GValue property, GError * error, gpointer data)
{
	/* g_debug("Callback for property %s %s %s", dbus_g_proxy_get_bus_name(proxy), dbus_g_proxy_get_path(proxy), dbus_g_proxy_get_interface(proxy)); */
	property_cb_t * propertyt = data;

	if (error != NULL) {
		/* g_warning("Unable to get property: %s", error->message); */
		g_free(propertyt);
		return;
	}

	/* Dup all the values and make them local so that
	   we can free up the data structure to make the
	   rest of the code easier to read. */
	IndicateListener * listener = propertyt->listener;
	IndicateListenerServer * server = propertyt->server;
	indicate_listener_get_server_property_cb cb = propertyt->cb;
	indicate_listener_get_server_uint_property_cb cb_uint = propertyt->cb_uint;
	gpointer cb_data = propertyt->data;

	g_free(propertyt);

	if (G_VALUE_HOLDS_STRING(&property) && cb != NULL) {
		/* If it's got a string, and we have a value for that
		   we'll get the string out and call the call back */
		gchar * propstr = g_value_dup_string(&property);
		return cb(listener, server, propstr, cb_data);
	} else if (G_VALUE_HOLDS_UINT(&property) && cb_uint != NULL) {
		/* If it's got a UINT and we have a callback for that
		   let's grab the value and call the callback. */
		guint val = g_value_get_uint(&property);
		return cb_uint(listener, server, val, cb_data);
	} else if (G_VALUE_TYPE(&property) == DBUS_TYPE_G_OBJECT_PATH && cb != NULL) {
		const gchar * val = (const gchar *)g_value_get_boxed(&property);
		if (val != NULL && val[0] != '\0' && val[1] != '\0') {
			/* If the string is NULL or if there is just a
			   root item we're going to look at that as an
			   error and not call the callback */
			gchar * propstr = g_strdup(val);
			return cb(listener, server, propstr, cb_data);
		}
	} else {
		/* WTF!?!?!?! */
		g_warning("Property back from server that we didn't understand.");
	}

	return;
}

/* This is a helper function for all the functions that
   get properties from the server.  They all need to have
   a callback setup with an intermediary data structure
   and this function builds and populates that, then uses
   a custom callback to call their callback */
static void
get_server_property (IndicateListener * listener, IndicateListenerServer * server, indicate_listener_get_server_property_cb callback, indicate_listener_get_server_uint_property_cb callback_uint, const gchar * property_name, gpointer data)
{
	/* g_debug("Setting up callback for property %s on %s", property_name, INDICATE_LISTENER_SERVER_DBUS_NAME(server)); */
	IndicateListenerPrivate * priv = INDICATE_LISTENER_GET_PRIVATE(listener);

	proxy_t searchitem;
	searchitem.name = server->name;
	searchitem.connection = server->connection;

	GList * proxyitem = g_list_find_custom(priv->proxies, &searchitem, proxy_t_equal);
	if (proxyitem == NULL) {
		g_warning("Can not find a proxy for the server at all.");
		return;
	}

	proxy_t * proxyt = (proxy_t *)proxyitem->data;

	if (proxyt->property_proxy == NULL) {
		proxyt->property_proxy = dbus_g_proxy_new_for_name(proxyt->connection,
		                                                   proxyt->name,
		                                                   proxyt->path,
		                                                   DBUS_INTERFACE_PROPERTIES);
	}

	property_cb_t * localdata = g_new0(property_cb_t, 1);
	localdata->listener = listener;
	localdata->server = server;
	localdata->cb = callback;
	localdata->cb_uint = callback_uint;
	localdata->data = data;

	org_freedesktop_DBus_Properties_get_async(proxyt->property_proxy,
	                                          INDICATE_DBUS_IFACE,
	                                          property_name,
	                                          property_cb,
	                                          localdata);

	return;
}

void
indicate_listener_server_get_type (IndicateListener * listener, IndicateListenerServer * server, indicate_listener_get_server_property_cb callback, gpointer data)
{
	return get_server_property(listener, server, callback, NULL, "type", data);
}

void
indicate_listener_server_get_desktop (IndicateListener * listener, IndicateListenerServer * server, indicate_listener_get_server_property_cb callback, gpointer data)
{
	return get_server_property(listener, server, callback, NULL, "desktop", data);
}

void
indicate_listener_server_get_count (IndicateListener * listener, IndicateListenerServer * server, indicate_listener_get_server_uint_property_cb callback, gpointer data)
{
	return get_server_property(listener, server, NULL, callback, "count", data);
}

void
indicate_listener_server_get_menu (IndicateListener * listener, IndicateListenerServer * server, indicate_listener_get_server_property_cb callback, gpointer data)
{
	return get_server_property(listener, server, callback, NULL, "menu", data);
}


const gchar *
indicate_listener_server_get_dbusname (IndicateListenerServer * server)
{
	if (server == NULL) return NULL;
	return server->name;
}

guint
indicate_listener_indicator_get_id (IndicateListenerIndicator * indicator)
{
	return GPOINTER_TO_UINT(indicator);
}

static const gchar *
interest_to_string (IndicateInterests interest)
{
	switch (interest) {
	case INDICATE_INTEREST_SERVER_DISPLAY:
		return INDICATE_INTEREST_STRING_SERVER_DISPLAY;
	case INDICATE_INTEREST_SERVER_SIGNAL:
		return INDICATE_INTEREST_STRING_SERVER_SIGNAL;
	case INDICATE_INTEREST_INDICATOR_DISPLAY:
		return INDICATE_INTEREST_STRING_INDICATOR_DISPLAY;
	case INDICATE_INTEREST_INDICATOR_SIGNAL:
		return INDICATE_INTEREST_STRING_INDICATOR_SIGNAL;
	case INDICATE_INTEREST_INDICATOR_COUNT:
		return INDICATE_INTEREST_STRING_INDICATOR_COUNT;
	default:
		return "";
	}
}

static void
interest_cb (DBusGProxy *proxy, GError *error, gpointer userdata)
{
	if (error != NULL) {
		g_warning("Unable to configure interest.");
	}

	return;
}

void
indicate_listener_server_show_interest (IndicateListener * listener, IndicateListenerServer * server, IndicateInterests interest)
{
	if (!(interest > INDICATE_INTEREST_NONE && interest < INDICATE_INTEREST_LAST)) {
		return;
	}

	if (!server->interests[interest]) {
		org_ayatana_indicate_show_interest_async (server->proxy, interest_to_string(interest), interest_cb, server);
		server->interests[interest] = TRUE;
	}
	return;
}

void
indicate_listener_server_remove_interest (IndicateListener * listener, IndicateListenerServer * server, IndicateInterests interest)
{
	if (server->interests[interest]) {
		org_ayatana_indicate_remove_interest_async (server->proxy, interest_to_string(interest), interest_cb, server);
		server->interests[interest] = FALSE;
	}
	return;
}

gboolean
indicate_listener_server_check_interest (IndicateListener * listener, IndicateListenerServer * server, IndicateInterests interest)
{
	return server->interests[interest];
}

GType
indicate_listener_server_get_gtype (void)
{
  static GType our_type = 0;
  
  if (our_type == 0)
    our_type = g_pointer_type_register_static ("IndicateListenerServer");

  return our_type;
}

GType
indicate_listener_indicator_get_gtype (void)
{
  static GType our_type = 0;
  
  if (our_type == 0)
    our_type = g_pointer_type_register_static ("IndicateListenerIndicator");

  return our_type;
}

/**
	indicate_listener_set_default_max_indicators:
	@listener: Instance of #IndicateListener to set on.
	@max: The new default number of max indicators.

	This function sets the number that is given to new servers
	when they start for the max number of indicators that you want
	to see.  The client should enforce this number.

	Note: This function WILL NOT reconfigure already recognized
	      servers.  It only affects new servers.
*/
void
indicate_listener_set_default_max_indicators (IndicateListener * listener, gint max)
{
	g_return_if_fail(INDICATE_IS_LISTENER(listener));
	IndicateListenerPrivate * priv = INDICATE_LISTENER_GET_PRIVATE(listener);
	priv->max_indicators = max;
	return;
}

/**
	indicate_listener_set_server_max_indicators:
	@listener: Instance of #IndicateListener to set on.
	@server: Server that the new max should be set on.
	@max: The new number of max indicators.

	This function changes the max number of indicators that
	a server should send.  If the number is different than the
	previous number a signal will be sent to the application to
	adjust then number of indicators that they have.  This is
	sent asynchronously.
*/
void
indicate_listener_set_server_max_indicators (IndicateListener * listener, IndicateListenerServer * server, gint max)
{
	g_return_if_fail(INDICATE_IS_LISTENER(listener));
	g_return_if_fail(server != NULL); /* All we can really check :-/ */

	if (server->max_indicators != max) {
		server->max_indicators = max;
		org_ayatana_indicate_set_max_indicators_async(server->proxy, server->max_indicators, set_max_indicators_cb, server->name);
	}

	return;
}
