/** -*- C++ -*-
 * @file cache/entity/version.h
 * @author Peter Rockai <me@mornfall.net>
 */

#include <apt-front/cache/cache.h>
#include <apt-front/cache/entity/package.h>
#include <apt-front/cache/component/records.h>
#include <string>
#include <stdexcept>

#ifndef APTFRONT_CACHE_ENTITY_VERSION_H
#define APTFRONT_CACHE_ENTITY_VERSION_H

namespace aptFront {
namespace cache {
namespace entity {

class Version : public Implementation<Version> {
public:
    Version() : m_version( 0 ) {}
    Version( const Entity &i ) {
        initFromBase( i.impl() );
    }

    virtual Entity stable() const;

    inline component::Records &recordsCache() const {
        return cache().records();
    }

    /* VersionIterator functionality. */
    bool operator ==( const Version &v ) const {
        return ( !v.valid() && !valid() )
            || ( v.m_version == m_version && ( &v.cache() ) == ( &cache() ) );
    }

    bool valid() const;

    pkgCache::VerFileIterator fileList() const;

    Version nextInCache() const;

    std::string shortDescription( std::string s ) const throw();
    std::string shortDescription() const {
        checkValid(); return shortDescription( "" );
    }

    std::string longDescription( std::string s ) const throw();
    std::string longDescription() const {
        checkValid(); return longDescription( "" );
    }

    std::string maintainer( std::string s ) const throw();
    std::string maintainer() const {
        checkValid(); return maintainer( "" );
    }

    std::string architecture( std::string s ) const throw();
    std::string architecture() const {
        checkValid(); return architecture( "" );
    }

    std::string section( std::string s ) const throw();
    std::string section() const {
        checkValid(); return section( "" );
    }

    std::string versionString( std::string s ) const throw();
    std::string versionString() const {
        checkValid(); return versionString( "" );
    }

    Package package() const throw();

    /**
     * Get the complete package record for this package
     */
    std::string completeRecord();

    Version( pkgCache::VerIterator );
    Version( Cache *c, pkgCache::Version *v )
        : m_version (v)
        {
            setCache( c );
        }
    operator pkgCache::VerIterator();
    bool operator<( const Version &i ) const {
        return m_version < i.m_version;
    } // XXX unit test!
protected:
    pkgCache::Version *m_version;
};

class StableVersion :
        public Implementation<StableVersion, Version>,
        public Observer
{
public:
    StableVersion( const Version &p )
        {
            Version::initFromBase( &p );
            setCache( Version::m_cache );
            observeComponent<component::Packages>();
        }
    StableVersion( Cache *c = 0 ) : Observer (c) {
        setCache( c );
        observeComponent<component::Packages>();
    }
    StableVersion( const Entity &i ) {
        initFromBase( dynamic_cast<Version *>( i.impl() ) );
        observeComponent<component::Packages>();
    }
    virtual ~StableVersion() {}
    void notifyPreRebuild( component::Base *c );
    void notifyPostRebuild( component::Base *c );
    virtual void setCache( Cache *c ) {
        Version::setCache( c );
        Observer::setCache( c );
    }
protected:
    std::string m_pkgName;
    std::string m_pkgVersion;
};

inline Entity Version::stable() const {
    return StableVersion( *this );
}

inline std::string Version::shortDescription( std::string d ) const throw()
{
    if (!valid()) return d;
    return recordsCache().record(*this).shortDescription;
}

inline std::string Version::longDescription( std::string d ) const throw() {
    if (!valid()) return d;
    return recordsCache().record( *this ).longDescription;
}

inline std::string Version::maintainer( std::string d ) const throw() {
    if (!valid()) return d;
    return recordsCache().record( *this ).maintainer;
}

inline std::string Version::architecture( std::string d ) const throw() {
    if (!valid()) return d;
    return packageCache().strPtr( m_version->Arch );
}

inline std::string Version::section( std::string d ) const throw() {
    if (!valid()) return d;
    return packageCache().strPtr( m_version->Section );
}

inline Package Version::package() const throw() {
    if (!valid())
        return Package();
    return Package( m_cache,
                    packageCache().pkgPtr( m_version->ParentPkg ) );
}

inline std::string Version::versionString( std::string d ) const throw() {
    if (!valid()) return d;
    return (packageCache().strPtr( m_version->VerStr ));
}

}
}
}
// vim:set ts=4 sw=4:
#endif
