/****************************************************************************
 * Driver for Solarflare network controllers -
 *          resource management for Xen backend, OpenOnload, etc
 *           (including support for SFE4001 10GBT NIC)
 *
 * This file provides helpers to turn bit shifts into dword shifts and
 * check that the bit fields haven't overflown the dword etc.
 *
 * Copyright 2005-2007: Solarflare Communications Inc,
 *                      9501 Jeronimo Road, Suite 250,
 *                      Irvine, CA 92618, USA
 *
 * Developed and maintained by Solarflare Communications:
 *                      <linux-xen-drivers@solarflare.com>
 *                      <onload-dev@solarflare.com>
 *
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 ****************************************************************************
 */

#ifndef __CI_EFHW_CHECK_H__
#define __CI_EFHW_CHECK_H__

/*----------------------------------------------------------------------------
 *
 * Helpers to turn bit shifts into dword shifts and check that the bit fields
 * haven't overflown the dword etc. Aim is to preserve consistency with the
 * autogenerated headers - once stable we could hard code.
 *
 *---------------------------------------------------------------------------*/

/* mask constructors */
#define __FALCON_MASK(WIDTH, T)	((((T)1) << (WIDTH)) - 1)
#define __FALCON_MASK32(WIDTH)	__FALCON_MASK((WIDTH), uint32_t)
#define __FALCON_MASK64(WIDTH)	__FALCON_MASK((WIDTH), uint64_t)

#define __FALCON_MASKFIELD32(LBN, WIDTH) \
	((uint32_t)(__FALCON_MASK32(WIDTH) << (LBN)))

/* constructors for fields which span the first and second dwords */
#define __LW(LBN)		(32 - LBN)
#define __LOW(v, LBN, WIDTH) \
	((uint32_t)(((v) & __FALCON_MASK64(__LW((LBN)))) << (LBN)))
#define __HIGH(v, LBN, WIDTH) \
	((uint32_t)(((v) >> __LW((LBN))) & \
		    __FALCON_MASK64((WIDTH - __LW((LBN))))))
/* constructors for fields within the second dword */
#define __DW2(LBN)		((LBN) - 32)

/* constructors for fields which span the second and third dwords */
#define __LW2(LBN)		(64 - LBN)
#define __LOW2(v, LBN, WIDTH) \
	((uint32_t)(((v) & __FALCON_MASK64(__LW2((LBN)))) << ((LBN) - 32)))
#define __HIGH2(v, LBN, WIDTH) \
	((uint32_t)(((v) >> __LW2((LBN))) & \
		    __FALCON_MASK64((WIDTH - __LW2((LBN))))))

/* constructors for fields within the third dword */
#define __DW3(LBN)		((LBN) - 64)

/* constructors for fields which span the third and fourth dwords */
#define __LW3(LBN)		(96 - LBN)
#define __LOW3(v, LBN, WIDTH) \
	((uint32_t)(((v) & __FALCON_MASK64(__LW3((LBN)))) << ((LBN) - 64)))
#define __HIGH3(v, LBN, WIDTH) \
	((ci_unit32)(((v) >> __LW3((LBN))) & \
		     __FALCON_MASK64((WIDTH - __LW3((LBN))))))

/* constructors for fields within the fourth dword */
#define __DW4(LBN)		((LBN) - 96)

/* checks that the autogenerated headers are consistent with our model */
#define __WIDTHCHCK(a, b)	EFHW_ASSERT((a) == (b))
#define __RANGECHCK(v, WIDTH) \
	EFHW_ASSERT(((uint64_t)(v) & ~(__FALCON_MASK64((WIDTH)))) == 0)

/* fields within the first dword */
#define __DWCHCK(LBN, WIDTH) \
	EFHW_ASSERT(((LBN) >= 0) && (((LBN)+(WIDTH)) <= 32))

/* fields which span the first and second dwords */
#define __LWCHK(LBN, WIDTH)	EFHW_ASSERT(WIDTH >= __LW(LBN))

/* fields within the second dword */
#define __DW2CHCK(LBN, WIDTH) \
	EFHW_ASSERT(((LBN) >= 32) && (((LBN)+(WIDTH)) <= 64))

/* fields which span the second and third dwords */
#define __LW2CHK(LBN, WIDTH)	EFHW_ASSERT(WIDTH >= __LW2(LBN))

/* fields within the third dword */
#define __DW3CHCK(LBN, WIDTH) \
	EFHW_ASSERT(((LBN) >= 64) && (((LBN)+(WIDTH)) <= 96))

/* fields which span the third and fourth dwords */
#define __LW3CHK(LBN, WIDTH)	EFHW_ASSERT(WIDTH >= __LW3(LBN))

/* fields within the fourth dword */
#define __DW4CHCK(LBN, WIDTH) \
	EFHW_ASSERT(((LBN) >= 96) && (((LBN)+(WIDTH)) <= 128))

/* fields in the first qword */
#define __QWCHCK(LBN, WIDTH) \
	EFHW_ASSERT(((LBN) >= 0) && (((LBN)+(WIDTH)) <= 64))

#endif /* __CI_EFHW_CHECK_H__ */
