/*
 * greeter_iface.c
 * LTSP display manager.
 * Manages spawning a session to a server.
 *
 * (c) Scott Balneaves, sbalneav@ltsp.org
 *
 * This software is licensed under the GPL v2 or later.
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <syslog.h>
#include <string.h>
#include <sys/wait.h>
#include <glib.h>
#include <glib/gi18n.h>

#include "ldm.h"

int
get_userid()
{
    gchar *prompt;
    gchar *isguest;
    gsize l, t;
    gboolean empty;

    empty = TRUE;

    while(empty) {
        prompt = g_strconcat("prompt <b>", _("Username"), "</b>\n", NULL);
        if (g_io_channel_write_chars(ldm.greeterw, prompt, -1, NULL, NULL) !=
            G_IO_STATUS_NORMAL) {
            return 1;
        }
        if (g_io_channel_flush(ldm.greeterw, NULL) != G_IO_STATUS_NORMAL) {
            return 1;
        }
        if (g_io_channel_read_line(ldm.greeterr, &isguest, &l, &t, NULL) !=
            G_IO_STATUS_NORMAL) {
            return 1;
        }
        /* figure out if user is a guest */
        if (ldm.allowguest && !strcmp(isguest, "@GUEST@\n")) {
            ldm.autologin = TRUE;
            empty = FALSE;
        } else {
            ldm.autologin = FALSE;
            if (g_io_channel_read_line(ldm.greeterr, &ldm.username, &l, &t, NULL)
                != G_IO_STATUS_NORMAL) {
                return 1;
            }
            ldm.username[t] = '\0';
            empty = g_regex_match_simple("^\\s*$", ldm.username, 0, 0);
        }
        g_free(isguest);
        g_free(prompt);
    }
    return 0;
}

int
get_passwd()
{
    gchar *prompt;
    gchar *command = "passwd\n";
    gsize l, t;

    prompt = g_strconcat("prompt <b>", _("Password"), "</b>\n", NULL);
    if (g_io_channel_write_chars(ldm.greeterw, prompt, -1, NULL, NULL) !=
            G_IO_STATUS_NORMAL) {
        return 1;
    }
    if (g_io_channel_write_chars(ldm.greeterw, command, -1, NULL, NULL) !=
            G_IO_STATUS_NORMAL) {
        return 1;
    }
    if (g_io_channel_flush(ldm.greeterw, NULL) !=
            G_IO_STATUS_NORMAL) {
        return 1;
    }
    if (g_io_channel_read_line(ldm.greeterr, &ldm.password, &l, &t, NULL) !=
            G_IO_STATUS_NORMAL) {
        return 1;
    }
    ldm.password[t] = '\0';
    g_free(prompt);
    set_message(_("<b>Verifying password.  Please wait.</b>"));
    return 0;
}

int
set_message(gchar *message)
{
    gchar *prompt;

    if (ldm.autologin) {
        return;
    }

    prompt = g_strconcat("msg ", message, "\n", NULL);
    if (g_io_channel_write_chars(ldm.greeterw, prompt, -1, NULL, NULL) !=
            G_IO_STATUS_NORMAL) {
        return 1;
    }
    if (g_io_channel_flush(ldm.greeterw, NULL) !=
            G_IO_STATUS_NORMAL) {
        return 1;
    }
    g_free(prompt);
    return 0;
}

int
get_host()
{
    char *cmd = "hostname\n";
    gsize l, t;

    if (g_io_channel_write_chars(ldm.greeterw, cmd, -1, NULL, NULL) !=
            G_IO_STATUS_NORMAL) {
        return 1;
    }
    if (g_io_channel_flush(ldm.greeterw, NULL) !=
            G_IO_STATUS_NORMAL) {
        return 1;
    }
    if (g_io_channel_read_line(ldm.greeterr, &ldm.server, &l, &t, NULL) !=
            G_IO_STATUS_NORMAL) {
        return 1;
    }
    ldm.server[t] = '\0';
    return 0;
}

int
get_language()
{
    char *cmd = "language\n";
    gchar *lang;
    gsize l, t;

    if (g_io_channel_write_chars(ldm.greeterw, cmd, -1, NULL, NULL) !=
            G_IO_STATUS_NORMAL) {
        return 1;
    }
    if (g_io_channel_flush(ldm.greeterw, NULL) !=
            G_IO_STATUS_NORMAL) {
        return 1;
    }
    if (g_io_channel_read_line(ldm.greeterr, &lang, &l, &t, NULL) !=
            G_IO_STATUS_NORMAL) {
        return 1;
    }
    lang[t] = '\0';
    if (g_ascii_strncasecmp(lang, "None", 4)) {
        if (ldm.lang) {
            g_free(ldm.lang);
        }
        ldm.lang = lang;
    }
    return 0;
}

int
get_session()
{
    char *cmd = "session\n";
    gchar *session;
    gsize l, t;

    if (g_io_channel_write_chars(ldm.greeterw, cmd, -1, NULL, NULL) !=
            G_IO_STATUS_NORMAL) {
        return 1;
    }
    if (g_io_channel_flush(ldm.greeterw, NULL) !=
            G_IO_STATUS_NORMAL) {
        return 1;
    }
    if (g_io_channel_read_line(ldm.greeterr, &session, &l, &t, NULL) !=
            G_IO_STATUS_NORMAL) {
        return 1;
    }
    session[t] = '\0';
    if (g_ascii_strncasecmp(session, "None", 4)) {
        if (ldm.session) {
            g_free(ldm.session);
        }
        ldm.session = session;
    }
    return 0;
}

void
close_greeter()
{
    gint rfd, wfd;
    if (!ldm.greeterpid) {
        return;
    }

    if (kill(ldm.greeterpid, SIGTERM) < 0) {
        fprintf(ldmlog, "Sending SIGTERM to greeter failed\n");
        kill(ldm.greeterpid, SIGKILL);
    }
    ldm_wait(ldm.greeterpid);
    rfd = g_io_channel_unix_get_fd(ldm.greeterr);
    wfd = g_io_channel_unix_get_fd(ldm.greeterw);
    g_io_channel_shutdown(ldm.greeterr, TRUE, NULL);
    g_io_channel_shutdown(ldm.greeterw, TRUE, NULL);
    close(rfd);
    close(wfd);
    ldm.greeterpid = 0;
}
