/*
 * This file is part of the KDE project
 *
 *  Copyright (c) 2007 Cyrille Berger <cberger@cberger.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "FastGaussianBlur.h"

#include <stdlib.h>
#include <vector>

#include <klocale.h>
#include <kiconloader.h>
#include <kinstance.h>
#include <kmessagebox.h>
#include <kstandarddirs.h>
#include <ktempfile.h>
#include <kdebug.h>
#include <kgenericfactory.h>

#include <kis_multi_double_filter_widget.h>
#include <kis_iterators_pixel.h>
#include <kis_progress_display_interface.h>
#include <kis_filter_registry.h>
#include <kis_global.h>
#include <kis_transaction.h>
#include <kis_types.h>
#include <kis_selection.h>

#include <kis_convolution_painter.h>

#include <qimage.h>
#include <qpixmap.h>
#include <qbitmap.h>
#include <qpainter.h>

typedef KGenericFactory<KritaFastGaussianBlur> KritaFastGaussianBlurFactory;
K_EXPORT_COMPONENT_FACTORY( kritaFastGaussianBlur, KritaFastGaussianBlurFactory( "krita" ) )

KritaFastGaussianBlur::KritaFastGaussianBlur(QObject *parent, const char *name, const QStringList &)
: KParts::Plugin(parent, name)
{
    setInstance(KritaFastGaussianBlurFactory::instance());

    kdDebug(41006) << "Fast Gaussian Blur filter plugin. Class: "
    << className()
    << ", Parent: "
    << parent -> className()
    << "\n";

    if(parent->inherits("KisFilterRegistry"))
    {
        KisFilterRegistry * manager = dynamic_cast<KisFilterRegistry *>(parent);
        manager->add(new KisFastGaussianBlurFilter());
    }
}

KritaFastGaussianBlur::~KritaFastGaussianBlur()
{
}

KisFastGaussianBlurFilter::KisFastGaussianBlurFilter() 
    : KisFilter(id(), "FastGaussianBlur", i18n("&Fast Gaussian Blur..."))
{
}

std::list<KisFilterConfiguration*> KisFastGaussianBlurFilter::listOfExamplesConfiguration(KisPaintDeviceSP )
{
    std::list<KisFilterConfiguration*> list;
    list.insert(list.begin(), configuration());
    return list;
}

KisFilterConfiguration* KisFastGaussianBlurFilter::configuration()
{
    KisFilterConfiguration* config = new KisFilterConfiguration(id().id(),1);
    config->setProperty("sigma", 3.0);
    return config;
};

KisFilterConfigWidget * KisFastGaussianBlurFilter::createConfigurationWidget(QWidget* parent, KisPaintDeviceSP dev)
{
    vKisDoubleWidgetParam param;
    param.push_back( KisDoubleWidgetParam( 0.0, 30.0, 3.0, i18n("Sigma"), "sigma" ) );
    KisFilterConfigWidget * w = new KisMultiDoubleFilterWidget(parent, id().id().ascii(), id().id().ascii(), param );
    Q_CHECK_PTR(w);
    return w;
}

KisFilterConfiguration* KisFastGaussianBlurFilter::configuration(QWidget* nwidget)
{
    KisMultiDoubleFilterWidget* widget = (KisMultiDoubleFilterWidget*) nwidget;
    if( widget == 0 )
    {
        return configuration();
    } else {
        KisFilterConfiguration* config = new KisFilterConfiguration(id().id(),1);
        config->setProperty("sigma", widget->valueAt( 0 ));
        return config;
    }
}

inline int gauss(int x, double s2)
{
    return floor( 100 * exp( - (x*x) / (2*s2) ) / sqrt(2*M_PI*s2) );
}

void KisFastGaussianBlurFilter::process(KisPaintDeviceSP src, KisPaintDeviceSP dst, 
                                   KisFilterConfiguration* config, const QRect& rect ) 
{
    Q_ASSERT(src != 0);
    Q_ASSERT(dst != 0);
    KisConvolutionPainter painter( dst );
    if (m_progressDisplay)
    {
        m_progressDisplay->setSubject( &painter, true, true );
    }
    
    double sigma;
    if(config)
    {
        sigma = config->getDouble("sigma", 3.0);
    } else {
        sigma = 3.0;
    }
    double sigma2 = sigma * sigma;
    
//     exp(-x2/ (2s2)) / sqrt(2* pi*s2)
    // Compute the size of the kernel
    int radius = 1;
    while( gauss(radius,sigma2) > 0 )
    {
        radius += 1;
    }
    kdDebug() << "Kernel radius = " << radius << " sigma = " << sigma << endl;
    // Compute the kernel
    KisKernelSP kernel = new KisKernel;
    kernel->width = 2 * radius + 1;
    kernel->height = 1;
    kernel->offset = 0;
    kernel->factor = 0;
    kernel->data = new Q_INT32[kernel->width];
    for(int i = 0; i <radius; i++)
    {
        int coef = gauss(radius - i, sigma2);
        kernel->data[i] = coef;
        kernel->data[kernel->width - 1 - i] = coef;
        kernel->factor += 2*coef;
    }
    kernel->data[radius] = gauss(0, sigma2);
    kernel->factor += kernel->data[radius];

    painter.applyMatrix(kernel, rect.x(), rect.y(), rect.width(), rect.height(), BORDER_REPEAT, KisChannelInfo::FLAG_COLOR_AND_ALPHA);
    KisTransaction("", dst);
    kernel->height = kernel->width;
    kernel->width = 1;
    painter.applyMatrix(kernel, rect.x(), rect.y(), rect.width(), rect.height(), BORDER_REPEAT, KisChannelInfo::FLAG_COLOR_AND_ALPHA);

    if (painter.cancelRequested()) {
        cancel();
    }

    setProgressDone(); // Must be called even if you don't really support progression
}
