/***************************************************************************
 *
 * knetworkmanager-connection_setting_serial.cpp - A NetworkManager frontend for KDE
 *
 * Copyright (C) 2005, 2006 Novell, Inc.
 *
 * Author: Helmut Schaa <hschaa@suse.de>, <helmut.schaa@gmx.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

/* qt headers */
#include <qhostaddress.h>
#include <qvariant.h>

/* kde headers */
#include <kdebug.h>
#include <klocale.h>

/* QDBus headers*/
#include <dbus/qdbusdata.h>
#include <dbus/qdbusdatamap.h>

/* knetworkmanager headers */
#include "knetworkmanager.h"
#include "knetworkmanager-connection.h"
#include "knetworkmanager-connection_setting_serial.h"

using namespace ConnectionSettings;

/*
	class Serial
*/
Serial::Serial(Connection* conn)
	: ConnectionSetting(conn, NM_SETTING_SERIAL_SETTING_NAME),
	_baud( 115200 ),
	_bits( 8 ),
	_parity( PARITY_NONE ),
	_stopBits( 1 ),
    _sendDelay( 0 )
{
}

void Serial::setBaud(Q_UINT32 baud)
{
	_baud = baud;
}

Q_UINT32 Serial::getBaud() const
{
	return _baud;
}

void Serial::setBits(Q_UINT32 bits)
{
	if (bits >= 5 && bits <= 8)
		_bits = bits;
	else
		kdWarning() << k_funcinfo << "bits property not accepted" << endl;
}

Q_UINT32 Serial::getBits() const
{
	return _bits;
}

void Serial::setParity(PARITY_MODE parity)
{
	_parity = parity;
}

Serial::PARITY_MODE Serial::getParity() const
{
	return _parity;
}

void Serial::setStopBits(Q_UINT32 stopBits)
{
	if (stopBits >= 1 && stopBits <= 2)
		_stopBits = stopBits;
	else
		kdWarning() << k_funcinfo << "stopbits property: wrong value" << endl;
}

Q_UINT32 Serial::getStopBits() const
{
	return _stopBits;
}

void Serial::setSendDelay(Q_UINT64 delay)
{
	_sendDelay = delay;
}

Q_UINT64 Serial::getSendDelay() const
{
	return _sendDelay;
}

bool
Serial::isValid() const
{
	// serial setting without ppp setting is not valid
	if (!(getConnection()->getSetting(NM_SETTING_PPP_SETTING_NAME)))
		return false;
	return true;
}

SettingsMap
Serial::toMap() const
{
	SettingsMap map;

	map.insert(NM_SETTING_SERIAL_BAUD, QDBusData::fromUInt32(_baud));
	map.insert(NM_SETTING_SERIAL_BITS, QDBusData::fromUInt32(_bits));

	if (_parity == PARITY_NONE)
		map.insert(NM_SETTING_SERIAL_PARITY, QDBusData::fromByte('n'));
	else if (_parity == PARITY_EVEN)
		map.insert(NM_SETTING_SERIAL_PARITY, QDBusData::fromByte('e'));
	else if (_parity == PARITY_ODD)
		map.insert(NM_SETTING_SERIAL_PARITY, QDBusData::fromByte('o'));

	map.insert(NM_SETTING_SERIAL_STOPBITS, QDBusData::fromUInt32(_stopBits));
	map.insert(NM_SETTING_SERIAL_SEND_DELAY, QDBusData::fromUInt64(_sendDelay));
	
	return map;
}

void
Serial::fromMap(const SettingsMap& map)
{
	for (SettingsMap::ConstIterator it = map.begin(); it != map.end(); ++it)
	{
		if (it.key() == NM_SETTING_SERIAL_BAUD)
			setBaud(it.data().toUInt32());
		else if (it.key() == NM_SETTING_SERIAL_BITS)
			setBits(it.data().toUInt32());
		else if (it.key() == NM_SETTING_SERIAL_PARITY)
                {
			if (it.data().toByte() == 'n')
				_parity = PARITY_NONE;
			else if (it.data().toByte() == 'e')
				_parity = PARITY_EVEN;
			else if (it.data().toByte() == 'o')
				_parity = PARITY_ODD;
                }
		else if (it.key() == NM_SETTING_SERIAL_STOPBITS)
			setStopBits(it.data().toUInt32());
		else if (it.key() == NM_SETTING_SERIAL_SEND_DELAY)
			setSendDelay(it.data().toUInt64());
		else
			kdWarning() << k_funcinfo << " Unknown setting: " << it.key() << endl;
	}
}
