/*
Copyright 2008 Will Stephenson <wstephenson@kde.org>

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation; either version 2 of
the License or (at your option) version 3 or any later version
accepted by the membership of KDE e.V. (or its successor approved
by the membership of KDE e.V.), which shall act as a proxy 
defined in Section 14 of version 3 of the license.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "devicetraycomponent.h"

#include <kiconloader.h>
#include <klocale.h>
#include <ksystemtray.h>

#include "knetworkmanager-device.h"
#include "knetworkmanager-nm_proxy.h"

DeviceTrayComponent::DeviceTrayComponent(Device * device, KSystemTray * parent, const char * name )
    : TrayComponent(parent, name), m_device(device)
{
	connect( m_device, SIGNAL(StateChanged(NMDeviceState)), SLOT(deviceStateChanged(NMDeviceState)));

	m_pixmaps[NM_DEVICE_STATE_UNKNOWN] = SmallIcon("nm_no_connection");
	m_pixmaps[NM_DEVICE_STATE_UNMANAGED] = SmallIcon("nm_no_connection");
	m_pixmaps[NM_DEVICE_STATE_UNAVAILABLE] = SmallIcon("nm_no_connection");
	m_pixmaps[NM_DEVICE_STATE_DISCONNECTED] = SmallIcon("nm_no_connection");

	m_movies[NM_DEVICE_STATE_PREPARE] = QMovie( KGlobal::iconLoader()->moviePath("nm_stage01_connecting", KIcon::Panel));
	m_movies[NM_DEVICE_STATE_PREPARE].pause();

	m_movies[NM_DEVICE_STATE_CONFIG] = QMovie( KGlobal::iconLoader()->moviePath("nm_stage02_connecting", KIcon::Panel));
	m_movies[NM_DEVICE_STATE_CONFIG].pause();

	m_movies[NM_DEVICE_STATE_IP_CONFIG] = QMovie( KGlobal::iconLoader()->moviePath("nm_stage03_connecting", KIcon::Panel));
	m_movies[NM_DEVICE_STATE_IP_CONFIG].pause();

	m_movies[NM_DEVICE_STATE_NEED_AUTH] = m_movies[NM_DEVICE_STATE_CONFIG];
	m_movies[NM_DEVICE_STATE_NEED_AUTH].pause();

	m_pixmaps[NM_DEVICE_STATE_ACTIVATED] = SmallIcon("ok");

	m_pixmaps[NM_DEVICE_STATE_FAILED] = SmallIcon("nm_no_connection");

	m_tooltips[NM_DEVICE_STATE_UNKNOWN] = i18n("Unknown");
	m_tooltips[NM_DEVICE_STATE_UNAVAILABLE] = i18n("Down");
	m_tooltips[NM_DEVICE_STATE_UNMANAGED] = i18n("Unmanaged");
	m_tooltips[NM_DEVICE_STATE_DISCONNECTED] = i18n("Disconnected");
	m_tooltips[NM_DEVICE_STATE_PREPARE] = i18n("Preparing");
	m_tooltips[NM_DEVICE_STATE_CONFIG] = i18n("Configuration");
	m_tooltips[NM_DEVICE_STATE_NEED_AUTH] = i18n("Awaiting authentication");
	m_tooltips[NM_DEVICE_STATE_IP_CONFIG] = i18n("IP configuration");
	m_tooltips[NM_DEVICE_STATE_ACTIVATED] = i18n("Activated");
	m_tooltips[NM_DEVICE_STATE_FAILED] = i18n("Failed");
}

DeviceTrayComponent::~DeviceTrayComponent()
{
}

Device* DeviceTrayComponent::device() const
{
	return m_device;
}

QStringList DeviceTrayComponent::getToolTipText()
{
	QStringList list;
	NMDeviceState state = m_device->getState();
	if (m_tooltips.contains(state)) {
		list.append(i18n("Device: %1").arg(m_device->getInterface()));
		list.append(i18n("State: %1").arg(m_tooltips[state]));
	}
	return list;
}


QPixmap DeviceTrayComponent::pixmapForState(NMDeviceState state)
{
    if (m_pixmaps.contains(state)) {
        return m_pixmaps[state];
    } else {
        return QPixmap();
    }
}

QMovie DeviceTrayComponent::movieForState(NMDeviceState state)
{
    if (m_movies.contains(state)) {
        return m_movies[state];
    } else {
        return QMovie();
    }
}

void DeviceTrayComponent::setMovieForState(NMDeviceState state, QMovie movie)
{
    m_movies[state] = movie;
}

void DeviceTrayComponent::setPixmapForState(NMDeviceState state, QPixmap pixmap)
{
    m_pixmaps[state] = pixmap;
}

void DeviceTrayComponent::deviceStateChanged(NMDeviceState state)
{
    //check if our device now holds the default active connection
    // if it is the default active connection
    // or if it is activating.  what if 2 devices are activating simultaneously?
    NMProxy * nm = 0;
    switch (state) {
        case NM_DEVICE_STATE_PREPARE:
            emit needsCenterStage(this, true);
            break;
        case NM_DEVICE_STATE_ACTIVATED:
            nm = NMProxy::getInstance();
            if ( m_device == nm->getDefaultDevice() ) {
                emit needsCenterStage(this, true);
            }
            break;
        case NM_DEVICE_STATE_UNAVAILABLE:
        case NM_DEVICE_STATE_DISCONNECTED:
        case NM_DEVICE_STATE_FAILED:
            emit needsCenterStage(this, false);
            break;
        default:
            break;
    }
}
#include "devicetraycomponent.moc"
