#
# Copyright 2009 Canonical Ltd.
#
# Written by:
#     Gustavo Niemeyer <gustavo.niemeyer@canonical.com>
#     Sidnei da Silva <sidnei.da.silva@canonical.com>
#
# This file is part of the Image Store Proxy.
#
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
import datetime
import logging
import sys
import os

from twisted.internet import reactor

from imagestore.lib.tests import TestCase
from imagestore.deployment import (
    parseOptions, TooManyArgsError, initializeLogging, buildServiceHub)

from imagestore.lib.service import ServiceHub
from imagestore.proxyservice import ProxyService
from imagestore.eucaservice import EucaService
from imagestore.downloadservice import DownloadService
from imagestore.storageservice import StorageService
from imagestore.signatureservice import SignatureService
from imagestore.installservice import InstallService


class ParseOptionsTest(TestCase):

    def testDefaultValues(self):
        options = parseOptions([])
        self.assertEquals(options.debug, False)
        self.assertEquals(options.log_file, None)
        self.assertEquals(options.api_url,
                          "https://imagestore.canonical.com/api")
        self.assertEquals(options.data_dir,
                          "/var/lib/image-store-proxy")
        self.assertEquals(options.eucalyptus_prefix, "")
        self.assertEquals(options.keyring, None)

    def testCommandLine(self):
        options = parseOptions(["--debug",
                                "--log-file", "log-file",
                                "--api-url", "api-url",
                                "--data-dir", "/data/dir",
                                "--eucalyptus-prefix", "/prefix",
                                "--keyring", "/keyring",
                               ])
        self.assertEquals(options.debug, True)
        self.assertEquals(options.log_file, "log-file")
        self.assertEquals(options.api_url, "api-url")
        self.assertEquals(options.data_dir, "/data/dir")
        self.assertEquals(options.eucalyptus_prefix, "/prefix")
        self.assertEquals(options.keyring, "/keyring")

    def testNoExtraArguments(self):
        self.assertRaises(TooManyArgsError, parseOptions, ["foo"])


class InitializeLoggingTest(TestCase):

    def setUp(self):
        logger = self.getLogger()
        self.oldHandlers = logger.handlers[:]
        self.oldLevel = logger.level
        logger.handlers[:] = []
        logger.setLevel(logging.ERROR)

    def tearDown(self):
        logger = self.getLogger()
        logger.handlers[:] = self.oldHandlers
        logger.setLevel(self.oldLevel)

    def getLogger(self):
        return logging.getLogger()

    def getHandler(self):
        self.assertEquals(len(logging.root.handlers), 1)
        return logging.root.handlers[0]

    def testDefault(self):
        initializeLogging()
        logger = self.getLogger()
        handler = self.getHandler()
        self.assertEquals(logger.level, logging.INFO)
        self.assertTrue(isinstance(handler, logging.StreamHandler))
        self.assertEquals(handler.stream, sys.stdout)

    def testFormatting(self):
        initializeLogging()
        record = logging.LogRecord("name", logging.DEBUG, "func", 42, "msg",
                                   (), None, None)
        record.created = 1234567890.000
        record.msecs = 0
        handler = self.getHandler()

        dt = datetime.datetime.fromtimestamp(record.created)
        formattedTimestamp = dt.strftime("%Y-%m-%d %H:%M:%S")

        self.assertEquals(handler.formatter.format(record),
                          "%s,000 DEBUG    msg" % formattedTimestamp)

    def testDebug(self):
        initializeLogging(debug=True)
        logger = self.getLogger()
        self.assertEquals(logger.level, logging.DEBUG)

    def testLogFile(self):
        path = self.makeFile()
        initializeLogging(logFile=path)
        handler = self.getHandler()
        self.assertTrue(isinstance(handler, logging.FileHandler))
        self.assertEquals(handler.baseFilename, path)


class WhiteBoxBuildServiceHubTest(TestCase):

    # Most of these tests are whiteboxing because we want to ensure
    # that the services are built correctly, but have no reason to
    # expose the actual implementation of the service otherwise.

    def build(self, args=None):
        if args is None:
            args = []
        if "--data-dir" not in args:
            self.dataDir = self.makeDir()
            args.extend(["--data-dir", self.dataDir])
        return buildServiceHub(parseOptions(args))

    def testBasic(self):
        hub = self.build()
        self.assertEquals(type(hub), ServiceHub)

    def testProxyService(self):
        hub = self.build()
        proxyService = hub.getService(ProxyService)
        self.assertEquals(proxyService.endpoint,
                          "https://imagestore.canonical.com/api")
        self.assertEquals(proxyService._reactor, reactor)

    def testProxyServiceWithApiUrl(self):
        hub = self.build(["--api-url", "http://api/url"])
        proxyService = hub.getService(ProxyService)
        self.assertEquals(proxyService.endpoint, "http://api/url")

    def testEucaService(self):
        hub = self.build()
        eucaService = hub.getService(EucaService)
        self.assertEquals(eucaService._reactor, reactor)
        self.assertEquals(eucaService._eucalyptusPrefix, "")
        eucalyptusDir = os.path.join(self.dataDir, "eucalyptus")
        self.assertEquals(eucaService._basePath, eucalyptusDir)
        self.assertTrue(os.path.isdir(eucalyptusDir))

    def testEucaServiceWithEucalyptusPrefix(self):
        hub = self.build(["--eucalyptus-prefix", "/prefix"])
        eucaService = hub.getService(EucaService)
        self.assertEquals(eucaService._eucalyptusPrefix, "/prefix")

    def testFakeEucaService(self):
        hub = self.build(["--fake-eucalyptus", "!KEY!"])
        eucaService = hub.getService(EucaService)
        self.assertEquals(eucaService._fakeMode, True)
        self.assertEquals(eucaService._fakeSecretKey, "!KEY!")

    def testDownloadService(self):
        hub = self.build()
        downloadService = hub.getService(DownloadService)
        self.assertEquals(downloadService._reactor, reactor)
        downloadsDir = os.path.join(self.dataDir, "downloads")
        self.assertEquals(downloadService._basePath, downloadsDir)
        self.assertTrue(os.path.isdir(downloadsDir))

    def testStorageService(self):
        hub = self.build()
        storageService = hub.getService(StorageService)
        self.assertEquals(storageService._reactor, reactor)

        storageFactory = storageService._storageFactory
        storage = storageFactory()
        storage.close()

        self.assertTrue(os.path.isfile(os.path.join(self.dataDir, "database")))

    def testSignatureService(self):
        hub = self.build()
        signatureService = hub.getService(SignatureService)
        self.assertEquals(signatureService._reactor, reactor)
        self.assertEquals(signatureService._keyringPath,
                          os.path.join(self.dataDir, "keyring.gpg"))

    def testSignatureServiceWithKeyring(self):
        keyringPath = self.makeFile()
        hub = self.build(["--keyring", keyringPath])
        signatureService = hub.getService(SignatureService)
        self.assertEquals(signatureService._reactor, reactor)
        self.assertEquals(signatureService._keyringPath,
                          os.path.join(self.dataDir, keyringPath))

    def testInstallService(self):
        hub = self.build()
        installService = hub.getService(InstallService)
        self.assertEquals(installService._serviceHub, hub)

    def testFailToCreateDirectory(self):
        try:
            self.build(["--data-dir", "/proc/woo"])
        except SystemExit, e:
            self.assertStartsWith(str(e),
                                  "Error creating directory /proc/woo: ")
        else:
            self.fail("SystemExit not raised")
