/*

  Broadcom BCM43xx wireless driver

  LO (LocalOscillator) Measuring and Control routines

  Copyright (c) 2005 Martin Langer <martin-langer@gmx.de>,
  Copyright (c) 2005, 2006 Stefano Brivio <st3@riseup.net>
  Copyright (c) 2005-2007 Michael Buesch <mbuesch@freenet.de>
  Copyright (c) 2005, 2006 Danny van Dyk <kugelfang@gentoo.org>
  Copyright (c) 2005, 2006 Andreas Jaggi <andreas.jaggi@waterwave.ch>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; see the file COPYING.  If not, write to
  the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
  Boston, MA 02110-1301, USA.

*/

#include "bcm43xx.h"
#include "bcm43xx_lo.h"
#include "bcm43xx_vstack.h"
#include "bcm43xx_phy.h"
#include "bcm43xx_main.h"

#include <linux/delay.h>


/* Write the LocalOscillator Control (adjust) value-pair. */
static void bcm43xx_lo_write(struct bcm43xx_wldev *dev,
			     struct bcm43xx_loctl *control)
{
	struct bcm43xx_phy *phy = &dev->phy;
	u16 value;
	u16 reg;

	if (BCM43xx_DEBUG) {
		if (unlikely(abs(control->v0) > 16 ||
			     abs(control->v1) > 16)) {
			printk(KERN_ERR PFX "ERROR: Invalid LO control pair "
					    "(first: %d, second: %d)\n",
			       control->v0, control->v1);
			dump_stack();
			return;
		}
	}

	value = (u8)(control->v1);
	value |= ((u8)(control->v0)) << 8;

	reg = (phy->type == BCM43xx_PHYTYPE_B) ? 0x002F : 0x0810;
	bcm43xx_phy_write(dev, reg, value);
}

struct bcm43xx_loctl * bcm43xx_get_loctl(struct bcm43xx_wldev *dev,
					 const struct bcm43xx_rfatt *rfatt,
					 const struct bcm43xx_bbatt *bbatt)
{
	struct bcm43xx_phy *phy = &dev->phy;
	struct bcm43xx_txpower_lo_control *c = phy->lo_control;
	struct bcm43xx_loctl *ret;

	if (unlikely(rfatt->att >= 16)) {
		dprintk(KERN_ERR PFX "ERROR: get_loctl() invalid rf_att: %u\n",
			rfatt->att);
	}
	if (unlikely(bbatt->att >= 9)) {
		dprintk(KERN_ERR PFX "ERROR: get_loctl() invalid bband_att: %u\n",
			bbatt->att);
	}

	if (rfatt->flag)
		ret = &(c->flagged[bbatt->att][rfatt->att]);
	else
		ret = &(c->unflagged[bbatt->att][rfatt->att]);

	return ret;
}

/* Call a function for every possible LO control value-pair. */
static int bcm43xx_call_for_each_loctl(struct bcm43xx_wldev *dev,
				       int (*func)(struct bcm43xx_wldev *,
						   struct bcm43xx_loctl *))
{
	struct bcm43xx_phy *phy = &dev->phy;
	struct bcm43xx_txpower_lo_control *ctl = phy->lo_control;
	int i, j;
	int err;

	for (i = 0; i < BCM43xx_NR_BB; i++) {
		for (j = 0; j < BCM43xx_NR_RF; j++) {
			err = func(dev, &(ctl->flagged[i][j]));
			if (unlikely(err))
				return err;
		}
	}
	for (i = 0; i < BCM43xx_NR_BB; i++) {
		for (j = 0; j < BCM43xx_NR_RF; j++) {
			err = func(dev, &(ctl->unflagged[i][j]));
			if (unlikely(err))
				return err;
		}
	}

	return 0;
}

static u16 lo_b_r15_loop(struct bcm43xx_wldev *dev)
{
	int i;
	u16 ret = 0;

	for (i = 0; i < 10; i++){
		bcm43xx_phy_write(dev, 0x0015, 0xAFA0);
		udelay(1);
		bcm43xx_phy_write(dev, 0x0015, 0xEFA0);
		udelay(10);
		bcm43xx_phy_write(dev, 0x0015, 0xFFA0);
		udelay(40);
		ret += bcm43xx_phy_read(dev, 0x002C);
	}

	return ret;
}

void bcm43xx_lo_b_measure(struct bcm43xx_wldev *dev)
{
	struct bcm43xx_phy *phy = &dev->phy;
	u16 regstack[12] = { 0 };
	u16 mls;
	u16 fval;
	int i, j;

	regstack[0] = bcm43xx_phy_read(dev, 0x0015);
	regstack[1] = bcm43xx_radio_read16(dev, 0x0052) & 0xFFF0;

	if (phy->radio_ver == 0x2053) {
		regstack[2] = bcm43xx_phy_read(dev, 0x000A);
		regstack[3] = bcm43xx_phy_read(dev, 0x002A);
		regstack[4] = bcm43xx_phy_read(dev, 0x0035);
		regstack[5] = bcm43xx_phy_read(dev, 0x0003);
		regstack[6] = bcm43xx_phy_read(dev, 0x0001);
		regstack[7] = bcm43xx_phy_read(dev, 0x0030);

		regstack[8] = bcm43xx_radio_read16(dev, 0x0043);
		regstack[9] = bcm43xx_radio_read16(dev, 0x007A);
		regstack[10] = bcm43xx_read16(dev, 0x03EC);
		regstack[11] = bcm43xx_radio_read16(dev, 0x0052) & 0x00F0;

		bcm43xx_phy_write(dev, 0x0030, 0x00FF);
		bcm43xx_write16(dev, 0x03EC, 0x3F3F);
		bcm43xx_phy_write(dev, 0x0035, regstack[4] & 0xFF7F);
		bcm43xx_radio_write16(dev, 0x007A, regstack[9] & 0xFFF0);
	}
	bcm43xx_phy_write(dev, 0x0015, 0xB000);
	bcm43xx_phy_write(dev, 0x002B, 0x0004);

	if (phy->radio_ver == 0x2053) {
		bcm43xx_phy_write(dev, 0x002B, 0x0203);
		bcm43xx_phy_write(dev, 0x002A, 0x08A3);
	}

	phy->minlowsig[0] = 0xFFFF;

	for (i = 0; i < 4; i++) {
		bcm43xx_radio_write16(dev, 0x0052, regstack[1] | i);
		lo_b_r15_loop(dev);
	}
	for (i = 0; i < 10; i++) {
		bcm43xx_radio_write16(dev, 0x0052, regstack[1] | i);
		mls = lo_b_r15_loop(dev) / 10;
		if (mls < phy->minlowsig[0]) {
			phy->minlowsig[0] = mls;
			phy->minlowsigpos[0] = i;
		}
	}
	bcm43xx_radio_write16(dev, 0x0052, regstack[1] | phy->minlowsigpos[0]);

	phy->minlowsig[1] = 0xFFFF;

	for (i = -4; i < 5; i += 2) {
		for (j = -4; j < 5; j += 2) {
			if (j < 0)
				fval = (0x0100 * i) + j + 0x0100;
			else
				fval = (0x0100 * i) + j;
			bcm43xx_phy_write(dev, 0x002F, fval);
			mls = lo_b_r15_loop(dev) / 10;
			if (mls < phy->minlowsig[1]) {
				phy->minlowsig[1] = mls;
				phy->minlowsigpos[1] = fval;
			}
		}
	}
	phy->minlowsigpos[1] += 0x0101;

	bcm43xx_phy_write(dev, 0x002F, phy->minlowsigpos[1]);
	if (phy->radio_ver == 0x2053) {
		bcm43xx_phy_write(dev, 0x000A, regstack[2]);
		bcm43xx_phy_write(dev, 0x002A, regstack[3]);
		bcm43xx_phy_write(dev, 0x0035, regstack[4]);
		bcm43xx_phy_write(dev, 0x0003, regstack[5]);
		bcm43xx_phy_write(dev, 0x0001, regstack[6]);
		bcm43xx_phy_write(dev, 0x0030, regstack[7]);

		bcm43xx_radio_write16(dev, 0x0043, regstack[8]);
		bcm43xx_radio_write16(dev, 0x007A, regstack[9]);

		bcm43xx_radio_write16(dev, 0x0052,
		                      (bcm43xx_radio_read16(dev, 0x0052) & 0x000F)
				      | regstack[11]);

		bcm43xx_write16(dev, 0x03EC, regstack[10]);
	}
	bcm43xx_phy_write(dev, 0x0015, regstack[0]);
}

static u16 lo_measure_feedthrough(struct bcm43xx_wldev *dev,
				  u16 lna, u16 pga, u16 trsw_rx)
{
	struct bcm43xx_phy *phy = &dev->phy;
	u16 rfover;

	if (phy->gmode) {
		lna <<= BCM43xx_PHY_RFOVERVAL_LNA_SHIFT;
		pga <<= BCM43xx_PHY_RFOVERVAL_PGA_SHIFT;

		assert((lna & ~BCM43xx_PHY_RFOVERVAL_LNA) == 0);
		assert((pga & ~BCM43xx_PHY_RFOVERVAL_PGA) == 0);
/*FIXME This assertion fails		assert((trsw_rx & ~(BCM43xx_PHY_RFOVERVAL_TRSWRX |
				    BCM43xx_PHY_RFOVERVAL_BW)) == 0);
*/
trsw_rx &= (BCM43xx_PHY_RFOVERVAL_TRSWRX | BCM43xx_PHY_RFOVERVAL_BW);

		/* Construct the RF Override Value */
		rfover = BCM43xx_PHY_RFOVERVAL_UNK;
		rfover |= pga;
		rfover |= lna;
		rfover |= trsw_rx;
		if ((dev->dev->bus->sprom.r1.boardflags_lo & BCM43xx_BFL_EXTLNA) &&
		    phy->rev > 6)
			rfover |= BCM43xx_PHY_RFOVERVAL_EXTLNA;

		bcm43xx_phy_write(dev, BCM43xx_PHY_PGACTL, 0xE300);
		bcm43xx_phy_write(dev, BCM43xx_PHY_RFOVERVAL, rfover);
		udelay(10);
		rfover |= BCM43xx_PHY_RFOVERVAL_BW_LBW;
		bcm43xx_phy_write(dev, BCM43xx_PHY_RFOVERVAL, rfover);
		udelay(10);
		rfover |= BCM43xx_PHY_RFOVERVAL_BW_LPF;
		bcm43xx_phy_write(dev, BCM43xx_PHY_RFOVERVAL, rfover);
		udelay(10);
		bcm43xx_phy_write(dev, BCM43xx_PHY_PGACTL, 0xF300);
	} else {
		pga |= BCM43xx_PHY_PGACTL_UNKNOWN;
		bcm43xx_phy_write(dev, BCM43xx_PHY_PGACTL, pga);
		udelay(10);
		pga |= BCM43xx_PHY_PGACTL_LOWBANDW;
		bcm43xx_phy_write(dev, BCM43xx_PHY_PGACTL, pga);
		udelay(10);
		pga |= BCM43xx_PHY_PGACTL_LPF;
		bcm43xx_phy_write(dev, BCM43xx_PHY_PGACTL, pga);
	}
	udelay(21);

	return bcm43xx_phy_read(dev, BCM43xx_PHY_LO_LEAKAGE);
}

/* TXCTL Register and Value Table.
 * Returns the "TXCTL Register".
 * "value" is the "TXCTL Value".
 * "flag" is the "Flag value".
 */
static u16 lo_txctl_register_table(struct bcm43xx_wldev *dev,
				   u16 *value, u16 *flag)
{
	struct bcm43xx_phy *phy = &dev->phy;
	u16 reg, v, f;

	if (phy->type == BCM43xx_PHYTYPE_B) {
		v = 0x30;
		if (phy->radio_rev <= 5) {
			reg = 0x43;
			f = 0;
		} else {
			reg = 0x52;
			f = 5;
		}
	} else {
		if (phy->rev >= 2 && phy->radio_rev == 8) {
			reg = 0x43;
			v = 0x10;
			f = 2;
		} else {
			reg = 0x52;
			v = 0x30;
			f = 5;
		}
	}
	if (value)
		*value = v;
	if (flag)
		*flag = f;

	return reg;
}

static void lo_measure_txctl_values(struct bcm43xx_wldev *dev)
{//TODO review and fix
	struct bcm43xx_phy *phy = &dev->phy;
	struct bcm43xx_txpower_lo_control *lo = phy->lo_control;
	u16 i, j;
	u16 tmp0;
	u16 reg, mask;
	s32 stmp0, stmp1, stmp2;
	s16 loopback_gain, trsw_rx, pga;
	u16 radio_pctl_reg;
	/* LocalOscillator/Measure/TXCTL2Values */
	static const u8 txctl2_values[] = {
		0x09, 0x08, 0x0A, 0x01, 0x00,
		0x02, 0x05, 0x04, 0x06, 0x00,
	};

	if (!has_loopback_gain(phy)) {
		radio_pctl_reg = 6;
		loopback_gain = 0;
		trsw_rx = 2;
		pga = 0;
	} else {
		trsw_rx = 0;
		stmp0 = phy->max_lb_gain;
		if (stmp0 < 0)
			stmp0++;
		stmp0 /= 2;
		loopback_gain = stmp0;
		if (stmp0 > 10) {
			radio_pctl_reg = 0;
			stmp1 = abs(10 - stmp0) / 6;
			pga = limit_value(stmp1, 0, 15);
		} else {
			pga = 0;
			stmp1 = 0x24;
			if ((phy->rev >= 2) &&
			    (phy->radio_ver == 0x2050) &&
			    (phy->radio_rev == 8))
				stmp1 = 0x3C;
			if ((10 - stmp0) < stmp1)
				stmp1 = (10 - stmp0);
			if (stmp1 < -3)
				stmp1 += 6;
			else
				stmp1 += 3;
			stmp2 = 0x24 / 4;
			if ((phy->rev >= 2) &&
			    (phy->radio_ver == 0x2050) &&
			    (phy->radio_rev == 8))
				stmp2 = 0x3C / 4;
			if ((stmp1 / 4) >= stmp2)
				radio_pctl_reg = stmp2;
			else
				radio_pctl_reg = stmp1 / 4;
		}
	}
	bcm43xx_radio_write16(dev, 0x43,
			      (bcm43xx_radio_read16(dev, 0x43)
			       & 0xFFF0) | radio_pctl_reg);
	bcm43xx_phy_set_baseband_attenuation(dev, 2);

	reg = lo_txctl_register_table(dev, &mask, NULL);
	bcm43xx_radio_write16(dev, reg,
			      bcm43xx_radio_read16(dev, reg)
			      & mask);//FIXME

	if (has_tx_magnification(phy)) {
		u32 minimum = 0xFFFFFFFF;
		u16 txctl1;
		u16 txctl2;

		for (i = 0; i < 2; i++) {
			txctl1 = (i == 0) ? 0x7 : 0x4;
			bcm43xx_radio_write16(dev, 0x0052,
					      (bcm43xx_radio_read16(dev, 0x0052)
					       & 0xFF0F) | (txctl1 << 4));
			for (j = 0; j < ARRAY_SIZE(txctl2_values); j++) {
				txctl2 = txctl2_values[j];
				bcm43xx_radio_write16(dev, 0x0052,
						      (bcm43xx_radio_read16(dev, 0x0052)
						       & 0xFFF0) | txctl2);
				tmp0 = lo_measure_feedthrough(dev, 0, pga, trsw_rx);
				if (tmp0 < minimum) {
					lo->txctl1 = txctl1;
					lo->txctl2 = txctl2;
					minimum = tmp0;
				}
				if (lo->txctl2 == 0)
					break;
			}
			bcm43xx_radio_write16(dev, 0x0052,
					      (lo->txctl1 << 4)
					      | lo->txctl2);
		}
	} else {
		lo->txctl2 = 0; /* txctl2 is unused for this radio. */
		bcm43xx_radio_write16(dev, 0x0052,
				      bcm43xx_radio_read16(dev, 0x0052)
				      & 0xFFF0);
	}
}

static void lo_read_power_vector(struct bcm43xx_wldev *dev)
{
	struct bcm43xx_phy *phy = &dev->phy;
	struct bcm43xx_txpower_lo_control *lo = phy->lo_control;
	u16 i;
	u64 tmp;
	u64 power_vector = 0;
	int rf_offset, bb_offset;
	struct bcm43xx_loctl *loctl;

	for (i = 0; i < 8; i += 2) {
		tmp = bcm43xx_shm_read16(dev, BCM43xx_SHM_SHARED,
					 0x89C + i);
		tmp &= 0xFF;//FIXME?
		power_vector |= (tmp << (i * 8));
	}
	for (i = 0; i < 8; i += 2) {
		bcm43xx_shm_write16(dev, BCM43xx_SHM_SHARED,
				    0x89C + i, 0);
	}

	if (power_vector)
		lo->power_vector = power_vector;
	power_vector = lo->power_vector;

	for (i = 0; i < 64; i++) {
		if (power_vector & ((u64)1ULL << i)) {
			/* Now figure out which bcm43xx_loctl corresponds
			 * to this bit.
			 */
			rf_offset = i / lo->rfatt_list.len;
			bb_offset = i % lo->rfatt_list.len;
			loctl = bcm43xx_get_loctl(dev, &lo->rfatt_list.list[rf_offset],
						  &lo->bbatt_list.list[bb_offset]);
			/* And mark it as "used", as the device told us
			 * through the bitmap it is using it.
			 */
			loctl->used = 1;
			lo->some_loctl_is_used = 1;
		}
	}
}

/* LocalOscillator/Measure/LOGainValues */
static void lo_measure_gain_values(struct bcm43xx_wldev *dev,
				   s16 control)
{
	struct bcm43xx_phy *phy = &dev->phy;
	u16 tmp;
	u16 abs_ctl = abs(control);

	if (phy->rev <= 1 || !phy->gmode) {
		phy->lo_gain[0] = 32;
		phy->lo_gain[2] = 0;
		if (abs_ctl >= 20) {
			phy->lo_gain[1] = 1;
			phy->lo_gain[3] = 2;
		} else if (control >= 18) {
			phy->lo_gain[1] = 1;
			phy->lo_gain[3] = 1;
		} else if (control >= 15) {
			phy->lo_gain[1] = 1;
			phy->lo_gain[3] = 0;
		} else {
			phy->lo_gain[1] = 0;
			phy->lo_gain[3] = 0;
		}
	} else {
		phy->lo_gain[0] = 0;
		if (abs_ctl >= phy->trsw_rx_gain / 2) {
			control = abs_ctl - (phy->trsw_rx_gain / 2);
			phy->lo_gain[0] = 32;
		}
		if (control <= 8) {
			phy->lo_gain[1] = 0;
		} else {
			phy->lo_gain[1] = 1;
			control -= 8;
		}
		control = limit_value(control, 0, 45);
		control /= 3;
		if (control >= 5) {
			phy->lo_gain[2] = 2;
			phy->lo_gain[3] = control - 5;
		} else {
			phy->lo_gain[2] = 0;
			phy->lo_gain[3] = control;
		}
	}

	tmp = bcm43xx_radio_read16(dev, 0x007A);
	if (phy->lo_gain[1] == 0)
		tmp &= ~0x0008;
	else
		tmp |= 0x0008;
	bcm43xx_radio_write16(dev, 0x007A, tmp);
}

static void lo_measure_setup(struct bcm43xx_wldev *dev,
			     struct bcm43xx_vstack *stack)
{
	struct ssb_sprom *sprom = &dev->dev->bus->sprom;
	struct bcm43xx_phy *phy = &dev->phy;
	const int is_initializing = (bcm43xx_status(dev) == BCM43xx_STAT_INITIALIZING);
	const int hwpctl = has_hardware_pctl(phy);
	u16 tmp;

	if (hwpctl) {
		bcm43xx_phy_stacksave(stack, BCM43xx_PHY_LO_MASK);
		bcm43xx_phy_stacksave(stack, BCM43xx_PHY_EXTG(0x01));
		bcm43xx_phy_stacksave(stack, BCM43xx_PHY_DACCTL);
		bcm43xx_phy_stacksave(stack, BCM43xx_PHY_BASE(0x14));
		bcm43xx_phy_stacksave(stack, BCM43xx_PHY_HPWR_TSSICTL);

		bcm43xx_phy_write(dev, BCM43xx_PHY_HPWR_TSSICTL, 0x100);
		bcm43xx_phy_write(dev, BCM43xx_PHY_EXTG(0x01), 0x40);
		bcm43xx_phy_write(dev, BCM43xx_PHY_DACCTL, 0x40);
		bcm43xx_phy_write(dev, BCM43xx_PHY_BASE(0x14), 0x200);
	}
	if (phy->type == BCM43xx_PHYTYPE_B &&
	    phy->radio_ver == 0x2050 &&
	    phy->radio_rev < 6) {
		bcm43xx_phy_write(dev, BCM43xx_PHY_BASE(0x16), 0x410);
		bcm43xx_phy_write(dev, BCM43xx_PHY_BASE(0x17), 0x820);
	}
	if (!is_initializing && hwpctl)
		lo_read_power_vector(dev);
	if (phy->gmode) {
		bcm43xx_phy_stacksave(stack, BCM43xx_PHY_ANALOGOVER);
		bcm43xx_phy_stacksave(stack, BCM43xx_PHY_ANALOGOVERVAL);
		bcm43xx_phy_stacksave(stack, BCM43xx_PHY_RFOVER);
		bcm43xx_phy_stacksave(stack, BCM43xx_PHY_RFOVERVAL);
		bcm43xx_phy_stacksave(stack, BCM43xx_PHY_CLASSCTL);
		bcm43xx_phy_stacksave(stack, BCM43xx_PHY_BASE(0x3E));
		bcm43xx_phy_stacksave(stack, BCM43xx_PHY_CRS0);

		bcm43xx_phy_write(dev, BCM43xx_PHY_CLASSCTL,
				  bcm43xx_phy_read(dev, BCM43xx_PHY_CLASSCTL)
				  & 0xFFFC);
		bcm43xx_phy_write(dev, BCM43xx_PHY_CRS0,
				  bcm43xx_phy_read(dev, BCM43xx_PHY_CRS0)
				  & 0x7FFF);
		bcm43xx_phy_write(dev, BCM43xx_PHY_ANALOGOVER,
				  bcm43xx_phy_read(dev, BCM43xx_PHY_ANALOGOVER)
				  | 0x0003);
		bcm43xx_phy_write(dev, BCM43xx_PHY_ANALOGOVERVAL,
				  bcm43xx_phy_read(dev, BCM43xx_PHY_ANALOGOVERVAL)
				  & 0xFFFC);
		if (phy->type == BCM43xx_PHYTYPE_G) {
			if ((phy->rev >= 7) &&
			    (sprom->r1.boardflags_lo & BCM43xx_BFL_EXTLNA)) {
				bcm43xx_phy_write(dev, BCM43xx_PHY_RFOVER, 0x933);
			} else {
				bcm43xx_phy_write(dev, BCM43xx_PHY_RFOVER, 0x133);
			}
		} else {
			bcm43xx_phy_write(dev, BCM43xx_PHY_RFOVER, 0);
		}
		bcm43xx_phy_write(dev, BCM43xx_PHY_BASE(0x3E), 0);
	}
	bcm43xx_mmio_stacksave(stack, 0x3F4);
	bcm43xx_mmio_stacksave(stack, 0x3E2);
	bcm43xx_phy_stacksave(stack, 0x15);//FIXME: New specs say Radio Reg 0x15
	bcm43xx_radio_stacksave(stack, 0x43);
	bcm43xx_radio_stacksave(stack, 0x7A);
	bcm43xx_phy_stacksave(stack, BCM43xx_PHY_BASE(0x2A));
	bcm43xx_phy_stacksave(stack, BCM43xx_PHY_BASE(0x35));
	bcm43xx_phy_stacksave(stack, BCM43xx_PHY_BASE(0x60));

	if (!has_tx_magnification(phy)) {
		tmp = bcm43xx_radio_read16(dev, 0x52);
		tmp &= 0x00F0;
		bcm43xx_vstack_save(stack, BCM43xx_VSTACK_RADIO,
				    0x52, tmp);
	}
	if (phy->type == BCM43xx_PHYTYPE_B) {
		bcm43xx_phy_stacksave(stack, BCM43xx_PHY_BASE(0x30));
		bcm43xx_phy_stacksave(stack, BCM43xx_PHY_BASE(0x06));
		bcm43xx_phy_write(dev, BCM43xx_PHY_BASE(0x30), 0x00FF);
		bcm43xx_phy_write(dev, BCM43xx_PHY_BASE(0x06), 0x3F3F);
	} else {
		bcm43xx_write16(dev, 0x3E2,
				bcm43xx_read16(dev, 0x3E2)
				| 0x8000);
	}
	bcm43xx_write16(dev, 0x3F4,
			bcm43xx_read16(dev, 0x3F4)
			& 0xF000);

	tmp = (phy->type == BCM43xx_PHYTYPE_G) ? BCM43xx_PHY_LO_MASK : BCM43xx_PHY_BASE(0x2E);
	bcm43xx_phy_write(dev, tmp, 0x007F);

	tmp = bcm43xx_vstack_restore(stack, BCM43xx_VSTACK_PHY,
				     BCM43xx_PHY_BASE(0x35));
	bcm43xx_phy_write(dev, 0x0035, tmp & 0xFF7F);
	tmp = bcm43xx_vstack_restore(stack, BCM43xx_VSTACK_RADIO, 0x7A);
	bcm43xx_radio_write16(dev, 0x007A, tmp & 0xFFF0);

	bcm43xx_phy_write(dev, BCM43xx_PHY_BASE(0x2A), 0x8A3);
	if (phy->type == BCM43xx_PHYTYPE_G ||
	    (phy->type == BCM43xx_PHYTYPE_B &&
	     phy->radio_ver == 0x2050 &&
	     phy->radio_rev >= 6)) {
		bcm43xx_phy_write(dev, BCM43xx_PHY_BASE(0x2B), 0x1003);
	} else
		bcm43xx_phy_write(dev, BCM43xx_PHY_BASE(0x2B), 0x0802);
	if (phy->gmode)
		bcm43xx_dummy_transmission(dev);
	bcm43xx_radio_selectchannel(dev, 6, 0);
	bcm43xx_radio_read16(dev, 0x51); /* dummy read */
	if (phy->type == BCM43xx_PHYTYPE_G)
		bcm43xx_phy_write(dev, BCM43xx_PHY_BASE(0x2F), 0);
	if (is_initializing)
		lo_measure_txctl_values(dev);
	if (phy->type == BCM43xx_PHYTYPE_G && phy->rev >= 3) {
		bcm43xx_phy_write(dev, BCM43xx_PHY_LO_MASK, 0xC078);
	} else {
		if (phy->type == BCM43xx_PHYTYPE_B)
			bcm43xx_phy_write(dev, BCM43xx_PHY_BASE(0x2E), 0x8078);
		else
			bcm43xx_phy_write(dev, BCM43xx_PHY_LO_MASK, 0x8078);
	}
}

static void lo_measure_restore(struct bcm43xx_wldev *dev,
			       struct bcm43xx_vstack *stack,
			       u8 old_channel)
{
	struct bcm43xx_phy *phy = &dev->phy;
	const int is_initializing = (bcm43xx_status(dev) == BCM43xx_STAT_INITIALIZING);
	const int hwpctl = has_hardware_pctl(phy);
	u16 tmp;

	if (phy->gmode) {
		bcm43xx_phy_write(dev, 0x0015, 0xE300);
		tmp = (phy->lo_gain[3] << 8);
		bcm43xx_phy_write(dev, 0x0812, tmp | 0x00A0);
		udelay(5);
		bcm43xx_phy_write(dev, 0x0812, tmp | 0x00A2);
		udelay(2);
		bcm43xx_phy_write(dev, 0x0812, tmp | 0x00A3);
	} else {
		tmp = (phy->lo_gain[3] & 0x00FF) | 0xEFA0;
		bcm43xx_phy_write(dev, 0x0015, tmp);
	}
	if (hwpctl) {
		bcm43xx_gphy_dc_lt_init(dev);
	} else {
		if (is_initializing)
			bcm43xx_lo_adjust_to(dev, 3, 2, 0);
		else
			bcm43xx_lo_adjust(dev);
	}
	if (phy->type == BCM43xx_PHYTYPE_G) {
		if (phy->rev >= 3)
			bcm43xx_phy_write(dev, 0x002E, 0xC078);
		else
			bcm43xx_phy_write(dev, 0x002E, 0x8078);
		if (phy->rev >= 2)
			bcm43xx_phy_write(dev, 0x002F, 0x0202);
		else
			bcm43xx_phy_write(dev, 0x002F, 0x0101);
	}
	bcm43xx_mmio_stackrestore(stack, 0x03F4);
	bcm43xx_phy_stackrestore(stack, 0x0015);//FIXME: New specs say Radio Reg 0x15
	bcm43xx_phy_stackrestore(stack, 0x002A);
	bcm43xx_phy_stackrestore(stack, 0x0035);
	bcm43xx_phy_stackrestore(stack, 0x0060);
	bcm43xx_radio_stackrestore(stack, 0x0043);
	bcm43xx_radio_stackrestore(stack, 0x007A);
	if (!(phy->radio_ver == 0x2050 && phy->radio_rev == 8)) {
		tmp = bcm43xx_vstack_restore(stack, BCM43xx_VSTACK_RADIO,
					     0x0052);
		bcm43xx_radio_write16(dev, 0x0052,
				      (bcm43xx_radio_read16(dev, 0x0052)
				       & 0xFF0F) | tmp);
	}
	bcm43xx_mmio_stackrestore(stack, 0x03E2);
	if (phy->type == BCM43xx_PHYTYPE_B &&
	    phy->radio_ver == 0x2050 &&
	    phy->radio_rev <= 5) {
		bcm43xx_phy_stackrestore(stack, 0x0030);
		bcm43xx_phy_stackrestore(stack, 0x0006);
	}
	if (phy->gmode) {
		bcm43xx_phy_stackrestore(stack, 0x0814);
		bcm43xx_phy_stackrestore(stack, 0x0815);
		bcm43xx_phy_stackrestore(stack, 0x0802);
		bcm43xx_phy_stackrestore(stack, 0x0811);
		bcm43xx_phy_stackrestore(stack, 0x0812);
		bcm43xx_phy_stackrestore(stack, 0x003E);
		bcm43xx_phy_stackrestore(stack, 0x0429);
	}
	if (hwpctl) {
		tmp = bcm43xx_vstack_restore(stack, BCM43xx_VSTACK_PHY,
					     0x080F);
		tmp &= 0xBFFF;
		bcm43xx_phy_write(dev, 0x080F, tmp);
		bcm43xx_phy_stackrestore(stack, 0x0801);
		bcm43xx_phy_stackrestore(stack, 0x0060);
		bcm43xx_phy_stackrestore(stack, 0x0014);
		bcm43xx_phy_stackrestore(stack, 0x0478);
	}
	bcm43xx_radio_selectchannel(dev, old_channel, 1);
}

/* Loop over each possible value in this state. */
static int lo_probe_possible_loctls(struct bcm43xx_wldev *dev,
				    struct bcm43xx_loctl *loctl,
				    struct bcm43xx_loctl *last_loctl,
				    int *current_state,
				    u16 *lowest_result)
{
	static const struct bcm43xx_loctl modifiers[] = {
		{ .v0 =  1,  .v1 =  1, },
		{ .v0 =  1,  .v1 =  0, },
		{ .v0 =  1,  .v1 = -1, },
		{ .v0 =  0,  .v1 = -1, },
		{ .v0 = -1,  .v1 = -1, },
		{ .v0 = -1,  .v1 =  0, },
		{ .v0 = -1,  .v1 =  1, },
		{ .v0 =  0,  .v1 =  1, },
	};
	const int is_initializing = (bcm43xx_status(dev) == BCM43xx_STAT_INITIALIZING);
	struct bcm43xx_phy *phy = &dev->phy;
	int multiplier = 1; /* State Value Multiplier. */
	struct bcm43xx_loctl orig_loctl;
	struct bcm43xx_loctl test_loctl;
	struct bcm43xx_loctl prev_loctl = {
		.v0 = -100,
		.v1 = -100,
	};
	int i;
	int begin, end;
	int found_lower = 0;
	u16 tmp;

	if (is_initializing && phy->rev >= 2 && phy->gmode)
		multiplier = 3;

	memcpy(&orig_loctl, loctl, sizeof(orig_loctl));

	if (*current_state == 0) {
		begin = 1;
		end = 8;
	} else if (*current_state % 2 == 0) {
		begin = *current_state - 1;
		end = *current_state + 1;
	} else {
		begin = *current_state - 2;
		end = *current_state + 2;
	}
	if (begin < 1)
		begin += 8;
	if (end > 8)
		end -= 8;

	i = begin;
	while (1) {
		assert(i >= 1 && i <= 8);
		memcpy(&test_loctl, &orig_loctl, sizeof(test_loctl));
		test_loctl.v0 += modifiers[i - 1].v0 * multiplier;
		test_loctl.v1 += modifiers[i - 1].v1 * multiplier;
		if ((test_loctl.v0 != prev_loctl.v0 ||
		     test_loctl.v1 != prev_loctl.v1) &&
		    (abs(test_loctl.v0) <= 16 &&
		     abs(test_loctl.v1) <= 16)) {
			bcm43xx_lo_write(dev, &test_loctl);
			tmp = lo_measure_feedthrough(dev, phy->lo_gain[0],
						  phy->lo_gain[2],
						  phy->lo_gain[3]);
			if (tmp < *lowest_result) {
				found_lower = 1;
				*lowest_result = tmp;
				*current_state = i;

				if (!(phy->rev >= 2 && phy->gmode &&
				      tmp >= 1500)) {
					memcpy(last_loctl, &test_loctl, sizeof(test_loctl));
					break;
				}
			}
		}
		memcpy(&prev_loctl, &test_loctl, sizeof(prev_loctl));
		if (i == end)
			break;
		if (i == 8)
			i = 1;
		else
			i++;
	}

	return found_lower;
}

static void lo_probe_loctls_statemachine(struct bcm43xx_wldev *dev,
					 struct bcm43xx_loctl *loctl,
					 s32 *lo_measure_gain_val)
{
	struct bcm43xx_phy *phy = &dev->phy;
	const int is_initializing = (bcm43xx_status(dev) == BCM43xx_STAT_INITIALIZING);
	u16 tmp;
	u16 lowest_result;
	int state = 0;
	int nr_measured = 0;
	int found_lower;
	struct bcm43xx_loctl last_loctl;

	memcpy(&last_loctl, loctl, sizeof(last_loctl));

	bcm43xx_lo_write(dev, &last_loctl);
	tmp = lo_measure_feedthrough(dev, phy->lo_gain[0],
				  phy->lo_gain[2], phy->lo_gain[3]);
	if (is_initializing || tmp >= 0x258) {
		if (tmp >= 0x12C && !is_initializing)
			*lo_measure_gain_val += 6;
		else if (tmp <= 0x12C)
			*lo_measure_gain_val += 12;
		lo_measure_gain_values(dev, *lo_measure_gain_val);
		tmp = lo_measure_feedthrough(dev, phy->lo_gain[0],
					  phy->lo_gain[2],
					  phy->lo_gain[3]);
	}
	lowest_result = tmp;

	do {
		assert(state >= 0 && state <= 8);
		found_lower = lo_probe_possible_loctls(dev, loctl, &last_loctl,
						       &state, &lowest_result);
		if (loctl->v0 == last_loctl.v0 &&
		    loctl->v1 == last_loctl.v1)
			break;
		memcpy(&last_loctl, loctl, sizeof(last_loctl));
		nr_measured++;
	} while (nr_measured < 24 && found_lower);

	memcpy(loctl, &last_loctl, sizeof(last_loctl));
}

/* LocalOscillator/Measure "Deviation Measuring State Machine" */
static void lo_do_measure(struct bcm43xx_wldev *dev,
			  struct bcm43xx_loctl *loctl,
			  s32 *lo_measure_gain_val)
{
//	struct bcm43xx_phy *phy = &dev->phy;
//	struct bcm43xx_txpower_lo_control *lo = phy->lo_control;
	int i;
	int nr_loops = 1;

//FIXME	if (phy->rev >= 2 && phy->gmode && !lo->some_loctl_is_used)
//		nr_loops = 4;

	for (i = 0; i < nr_loops; i++) {
		lo_probe_loctls_statemachine(dev, loctl,
					     lo_measure_gain_val);
	}
}

static void lo_measure(struct bcm43xx_wldev *dev)
{
	struct bcm43xx_phy *phy = &dev->phy;
	struct bcm43xx_txpower_lo_control *lo = phy->lo_control;
	const int is_initializing = (bcm43xx_status(dev) == BCM43xx_STAT_INITIALIZING);
	struct bcm43xx_loctl loctl = {
		.v0	= 0,
		.v1	= 0,
	};
	struct bcm43xx_loctl *ploctl;
	s32 lo_measure_gain_val;

	u8 rfidx;
	u8 bbidx;

	/* Values from the "TXCTL Register and Value Table" */
	u16 txctl_reg;
	u16 txctl_value;
	u16 txctl_flag;

	txctl_reg = lo_txctl_register_table(dev, &txctl_value, &txctl_flag);

	for (rfidx = 0; rfidx < lo->rfatt_list.len; rfidx++) {

		bcm43xx_radio_write16(dev, 0x0043,
				      (bcm43xx_radio_read16(dev, 0x0043)
				       & 0xFFF0) | lo->rfatt_list.list[rfidx].att);
		bcm43xx_radio_write16(dev, txctl_reg,
				      (bcm43xx_radio_read16(dev, txctl_reg)
				       & ~txctl_value)
				      | (lo->rfatt_list.list[rfidx].flag ? txctl_value : 0));

		for (bbidx = 0; bbidx < lo->bbatt_list.len; bbidx++) {
			int must_respect_flag = 0;

			ploctl = bcm43xx_get_loctl(dev, &lo->rfatt_list.list[rfidx],
						   &lo->bbatt_list.list[bbidx]);
			if (!is_initializing) {
				if (!ploctl->used)
					continue;
				must_respect_flag = 1;
			}
			memcpy(&loctl, ploctl, sizeof(loctl));

			lo_measure_gain_val = (lo->rfatt_list.list[rfidx].att * 2);
			lo_measure_gain_val += (lo->bbatt_list.list[bbidx].att / 2);
			if (must_respect_flag && lo->rfatt_list.list[rfidx].flag)
				lo_measure_gain_val -= txctl_flag;
			if (phy->gmode && phy->rev >= 2)
				lo_measure_gain_val += phy->max_lb_gain & 0xFFFE;
			lo_measure_gain_values(dev, lo_measure_gain_val);

			bcm43xx_phy_set_baseband_attenuation(dev, lo->bbatt_list.list[bbidx].att);
			lo_do_measure(dev, &loctl, &lo_measure_gain_val);
			memcpy(ploctl, &loctl, sizeof(loctl));
			if (phy->type == BCM43xx_PHYTYPE_B) {
				ploctl->v0++;
				ploctl->v1++;
			}
		}
	}
}

#if BCM43xx_DEBUG
static int do_validate_loctl(struct bcm43xx_wldev *dev,
			     struct bcm43xx_loctl *control)
{
	const int is_initializing = (bcm43xx_status(dev) == BCM43xx_STAT_INITIALIZING);

	if (unlikely(abs(control->v0) > 16 ||
		     abs(control->v1) > 16 ||
		     (is_initializing && control->used))) {
		printk(KERN_ERR PFX "ERROR: LO control pair validation failed "
				    "(first: %d, second: %d, used %u)\n",
		       control->v0, control->v1, control->used);
	}
	return 0;
}
static void validate_all_loctls(struct bcm43xx_wldev *dev)
{
	bcm43xx_call_for_each_loctl(dev, do_validate_loctl);
}
#else /* BCM43xx_DEBUG */
static inline void validate_all_loctls(struct bcm43xx_wldev *dev) { }
#endif /* BCM43xx_DEBUG */

void bcm43xx_lo_g_measure(struct bcm43xx_wldev *dev)
{
	struct bcm43xx_phy *phy = &dev->phy;
	u8 old_channel;
	struct bcm43xx_vstack *stack;

	assert(phy->type == BCM43xx_PHYTYPE_B ||
	       phy->type == BCM43xx_PHYTYPE_G);

	/* We use the generic value stack. */
	stack = &dev->genstack;

	old_channel = phy->channel;
	lo_measure_setup(dev, stack);
	lo_measure(dev);
	lo_measure_restore(dev, stack, old_channel);

	bcm43xx_vstack_cleanup(stack);
	validate_all_loctls(dev);

	phy->lo_control->lo_measured = 1;
	return;
}

void bcm43xx_lo_adjust(struct bcm43xx_wldev *dev)
{
	bcm43xx_lo_write(dev, bcm43xx_loctl_current(dev));
}

static inline void fixup_rfatt_for_txctl1(struct bcm43xx_rfatt *rf,
					  u16 txctl1)
{
	if ((rf->att < 5) && (txctl1 & 0x0001))
		rf->att = 4;
}

void bcm43xx_lo_adjust_to(struct bcm43xx_wldev *dev,
			  u16 rfatt, u16 bbatt, u16 txctl1)
{
	struct bcm43xx_rfatt rf;
	struct bcm43xx_bbatt bb;
	struct bcm43xx_loctl *loctl;

	memset(&rf, 0, sizeof(rf));
	memset(&bb, 0, sizeof(bb));
	rf.att = rfatt;
	bb.att = bbatt;
	fixup_rfatt_for_txctl1(&rf, txctl1);
	loctl = bcm43xx_get_loctl(dev, &rf, &bb);
	bcm43xx_lo_write(dev, loctl);
}

struct bcm43xx_loctl * bcm43xx_loctl_current(struct bcm43xx_wldev *dev)
{
	struct bcm43xx_phy *phy = &dev->phy;
	struct bcm43xx_txpower_lo_control *lo = phy->lo_control;
	struct bcm43xx_rfatt rf;

	memcpy(&rf, &lo->rfatt, sizeof(rf));
	fixup_rfatt_for_txctl1(&rf, lo->txctl1);

	return bcm43xx_get_loctl(dev, &rf, &lo->bbatt);
}

static int do_mark_unused(struct bcm43xx_wldev *dev,
			  struct bcm43xx_loctl *control)
{
	control->used = 0;
	return 0;
}

void bcm43xx_loctl_mark_all_unused(struct bcm43xx_wldev *dev)
{
	struct bcm43xx_phy *phy = &dev->phy;
	struct bcm43xx_txpower_lo_control *lo = phy->lo_control;

	bcm43xx_call_for_each_loctl(dev, do_mark_unused);
	lo->some_loctl_is_used = 0;
}

void bcm43xx_loctl_mark_cur_used(struct bcm43xx_wldev *dev)
{
	struct bcm43xx_phy *phy = &dev->phy;
	struct bcm43xx_txpower_lo_control *lo = phy->lo_control;

	bcm43xx_loctl_current(dev)->used = 1;
	lo->some_loctl_is_used = 1;
}
