/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: pixmap.c
 */

#include <frontend.h>

#include <gtk/gtk.h>

#include "pixmap.h"
#include "xpm/unchecked.xpm"
#include "xpm/checkmark.xpm"
#include "xpm/red_checkmark.xpm"
#include "xpm/disk.xpm"
#include "xpm/partition.xpm"
#include "xpm/container.xpm"
#include "xpm/region.xpm"
#include "xpm/object.xpm"
#include "xpm/volume.xpm"
#include "xpm/volume_open.xpm"
#include "xpm/plugin.xpm"
#include "xpm/refresh.xpm"
#include "xpm/info.xpm"
#include "xpm/error.xpm"
#include "xpm/warning.xpm"
#include "xpm/question.xpm"
#include "xpm/topmost.xpm"
#include "xpm/exit.xpm"
#include "xpm/commit.xpm"
#include "xpm/stock_help-16.xpm"
#include "xpm/stock_cancel.xpm"
#include "xpm/stock_left.xpm"
#include "xpm/stock_right.xpm"
#include "xpm/stock_ok.xpm"
#include "xpm/stock_clear.xpm"
#include "xpm/stock_save-16.xpm"
#include "xpm/stock_exit-16.xpm"
#include "xpm/stock_exec-16.xpm"
#include "xpm/stock_search_16.xpm"
#include "logging.h"

static GdkPixmap *unchecked_pixmap;
static GdkBitmap *unchecked_pixmap_mask;
static GdkPixmap *checkmark_pixmap;
static GdkBitmap *checkmark_pixmap_mask;
static GdkPixmap *red_checkmark_pixmap;
static GdkBitmap *red_checkmark_pixmap_mask;
static GdkPixmap *disk_pixmap;
static GdkBitmap *disk_pixmap_mask;
static GdkPixmap *partition_pixmap;
static GdkBitmap *partition_pixmap_mask;
static GdkPixmap *container_pixmap;
static GdkBitmap *container_pixmap_mask;
static GdkPixmap *region_pixmap;
static GdkBitmap *region_pixmap_mask;
static GdkPixmap *object_pixmap;
static GdkBitmap *object_pixmap_mask;
static GdkPixmap *volume_pixmap;
static GdkBitmap *volume_pixmap_mask;
static GdkPixmap *volume_open_pixmap;
static GdkBitmap *volume_open_pixmap_mask;
static GdkPixmap *plugin_pixmap;
static GdkBitmap *plugin_pixmap_mask;
static GdkPixmap *info_pixmap;
static GdkBitmap *info_pixmap_mask;
static GdkPixmap *error_pixmap;
static GdkBitmap *error_pixmap_mask;
static GdkPixmap *warning_pixmap;
static GdkBitmap *warning_pixmap_mask;
static GdkPixmap *question_pixmap;
static GdkBitmap *question_pixmap_mask;
static GdkPixmap *refresh_pixmap;
static GdkBitmap *refresh_pixmap_mask;
static GdkPixmap *topmost_pixmap;
static GdkBitmap *topmost_pixmap_mask;
static GdkPixmap *exit_pixmap;
static GdkBitmap *exit_pixmap_mask;
static GdkPixmap *commit_pixmap;
static GdkBitmap *commit_pixmap_mask;
static GdkPixmap *help_16_pixmap;
static GdkBitmap *help_16_pixmap_mask;
static GdkPixmap *cancel_pixmap;
static GdkBitmap *cancel_pixmap_mask;
static GdkPixmap *left_arrow_pixmap;
static GdkBitmap *left_arrow_pixmap_mask;
static GdkPixmap *right_arrow_pixmap;
static GdkBitmap *right_arrow_pixmap_mask;
static GdkPixmap *ok_pixmap;
static GdkBitmap *ok_pixmap_mask;
static GdkPixmap *commit_16_pixmap;
static GdkBitmap *commit_16_pixmap_mask;
static GdkPixmap *exit_16_pixmap;
static GdkBitmap *exit_16_pixmap_mask;
static GdkPixmap *clear_pixmap;
static GdkBitmap *clear_pixmap_mask;
static GdkPixmap *exec_16_pixmap;
static GdkBitmap *exec_16_pixmap_mask;
static GdkPixmap *search_16_pixmap;
static GdkBitmap *search_16_pixmap_mask;

/*
 *
 *   void create_standard_pixmaps (GtkWidget *)
 *
 *   Description:
 *      This routine creates some of the often seen and used
 *      GdkPixmaps. This routine should be called once from
 *      either main() or an init routine called by main().
 *      The GdkPixmap and GdkBitmap pointers should be accessed
 *      through one of the get functions from this file.
 * 
 *   Entry:
 *      widget - used to provide colormap to
 *               gdk_pixmap_colormap_create_from_xpm_d().
 *
 *   Exit:
 *      Creates the GdkPixmaps and saves the pointers in the file
 *      scope static variables defined above.
 *
 */
void create_standard_pixmaps(GtkWidget * widget)
{
	GdkColormap *colormap;

	colormap = gtk_widget_get_colormap(widget);

	unchecked_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap,
								 &unchecked_pixmap_mask,
								 NULL, unchecked_xpm);

	unchecked_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap,
								 &unchecked_pixmap_mask,
								 NULL, unchecked_xpm);

	checkmark_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap,
								 &checkmark_pixmap_mask,
								 NULL, checkmark_xpm);

	red_checkmark_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap,
								     &red_checkmark_pixmap_mask,
								     NULL, red_checkmark_xpm);

	disk_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap,
							    &disk_pixmap_mask, NULL, disk_xpm);

	partition_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap,
								 &partition_pixmap_mask,
								 NULL, partition_xpm);

	container_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap,
								 &container_pixmap_mask,
								 NULL, container_xpm);

	region_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap,
							      &region_pixmap_mask,
							      NULL, region_xpm);

	object_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap,
							      &object_pixmap_mask,
							      NULL, object_xpm);

	volume_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap,
							      &volume_pixmap_mask,
							      NULL, volume_xpm);

	volume_open_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap,
								   &volume_open_pixmap_mask,
								   NULL, volume_open_xpm);

	plugin_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap,
							      &plugin_pixmap_mask,
							      NULL, plugin_xpm);

	refresh_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap,
							       &refresh_pixmap_mask,
							       NULL, refresh_xpm);

	info_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap,
							    &info_pixmap_mask, NULL, info_xpm);

	error_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap,
							     &error_pixmap_mask, NULL, error_xpm);

	warning_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap,
							       &warning_pixmap_mask,
							       NULL, warning_xpm);

	question_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap,
								&question_pixmap_mask,
								NULL, question_xpm);

	topmost_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap,
							       &topmost_pixmap_mask,
							       NULL, topmost_xpm);

	exit_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap,
							    &exit_pixmap_mask, NULL, exit_xpm);

	commit_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap,
							      &commit_pixmap_mask,
							      NULL, commit_xpm);

	help_16_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap,
							       &help_16_pixmap_mask,
							       NULL, stock_help_16_xpm);

	cancel_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap,
							      &cancel_pixmap_mask,
							      NULL, stock_cancel_xpm);

	left_arrow_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap,
								  &left_arrow_pixmap_mask,
								  NULL, stock_left_xpm);

	right_arrow_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap,
								   &right_arrow_pixmap_mask,
								   NULL, stock_right_xpm);

	ok_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap,
							  &ok_pixmap_mask, NULL, stock_ok_xpm);

	commit_16_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap,
								 &commit_16_pixmap_mask,
								 NULL, stock_save_16_xpm);

	exit_16_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap,
							       &exit_16_pixmap_mask,
							       NULL, stock_exit_16_xpm);

	clear_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap,
							     &clear_pixmap_mask,
							     NULL, stock_clear_xpm);

	exec_16_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap,
							       &exec_16_pixmap_mask,
							       NULL, stock_exec_16_xpm);

	search_16_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap,
								 &search_16_pixmap_mask,
								 NULL, stock_search_16_xpm);
}

/*
 *
 *   void get_pixmap_for_object_type (object_type_t, GdkPixmap **, GdkBitmap **)
 *   
 *   Description:
 *      This routine returns the GdkPixmap * and GdkBitmap *
 *      corresponding to the pixmap and mask for the icon
 *      representing the given object type.
 * 
 *   Entry:
 *      type   - the object type, e.g. DISK, SEGMENT, or REGION
 *      pixmap - the address of where to store the pointer to the pixmap
 *      mask   - the address of where to store the pointer to the mask
 *
 *   Exit:
 *      The pointers are set with the proper pixmap and mask pointers
 *      allocated at init time.
 *
 */
void get_pixmap_for_object_type(object_type_t type, GdkPixmap ** pixmap, GdkBitmap ** mask)
{
	switch (type) {
	case VOLUME:
		*pixmap = volume_pixmap;
		*mask = volume_pixmap_mask;
		break;

	case EVMS_OBJECT:
		*pixmap = object_pixmap;
		*mask = object_pixmap_mask;
		break;

	case REGION:
		*pixmap = region_pixmap;
		*mask = region_pixmap_mask;
		break;

	case CONTAINER:
		*pixmap = container_pixmap;
		*mask = container_pixmap_mask;
		break;

	case SEGMENT:
		*pixmap = partition_pixmap;
		*mask = partition_pixmap_mask;
		break;

	case DISK:
		*pixmap = disk_pixmap;
		*mask = disk_pixmap_mask;
		break;

	case PLUGIN:
		*pixmap = plugin_pixmap;
		*mask = plugin_pixmap_mask;
		break;

	default:
		*pixmap = NULL;
		*mask = NULL;
		log_warning("%s: Unknown type %d.\n", __FUNCTION__, type);
		break;
	}
}

/*
 *
 *   void get_dialog_pixmap (dialog_pixmap_type_t, GdkPixmap **, GdkBitmap **)
 *   
 *   Description:
 *      This routine returns the GdkPixmap * and GdkBitmap *
 *      for a given dialog pixmap type.
 * 
 *   Entry:
 *      type   - INFO_PIXMAP, ERROR_PIXMAP, WARNING_PIXMAP, or QUESTION_PIXMAP
 *      pixmap - the address of where to store the pointer to the pixmap
 *      mask   - the address of where to store the pointer to the mask
 *
 *   Exit:
 *      The pointers are set with the proper pixmap and mask pointers
 *      allocated at init time.
 *
 */
void get_dialog_pixmap(dialog_pixmap_type_t type, GdkPixmap ** pixmap, GdkBitmap ** mask)
{
	switch (type) {
	case INFO_PIXMAP:
		*pixmap = info_pixmap;
		*mask = info_pixmap_mask;
		break;

	case WARNING_PIXMAP:
		*pixmap = warning_pixmap;
		*mask = warning_pixmap_mask;
		break;

	case ERROR_PIXMAP:
		*pixmap = error_pixmap;
		*mask = error_pixmap_mask;
		break;

	case QUESTION_PIXMAP:
		*pixmap = question_pixmap;
		*mask = question_pixmap_mask;
		break;

	default:
		*pixmap = NULL;
		*mask = NULL;
		log_warning("%s: Unknown type %d.\n", __FUNCTION__, type);
		break;
	}
}

/*
 *
 *   void get_checkmark_box_pixmap (gboolean, gboolean, GdkPixmap **, GdkBitmap **)
 *   
 *   Description:
 *      This routine returns the GdkPixmap * and GdkBitmap *
 *      for one of three checkmark boxes.
 * 
 *   Entry:
 *      critical - TRUE if requesting the red checkmark box
 *      checked  - TRUE if requesting a checked box, FALSE if unchecked box
 *      pixmap   - the address of where to store the pointer to the pixmap
 *      mask     - the address of where to store the pointer to the mask
 *
 *   Exit:
 *      The pointers are set with the proper pixmap and mask pointers
 *      allocated at init time.
 *
 */
void get_checkmark_box_pixmap(gboolean critical, gboolean checked, GdkPixmap ** pixmap,
			      GdkBitmap ** mask)
{
	if (checked) {
		if (critical) {
			*pixmap = red_checkmark_pixmap;
			*mask = red_checkmark_pixmap_mask;
		} else {
			*pixmap = checkmark_pixmap;
			*mask = checkmark_pixmap_mask;
		}
	} else {
		*pixmap = unchecked_pixmap;
		*mask = unchecked_pixmap_mask;
	}
}

/*
 * The following routines are simple getter routines for
 * their respective pixmaps.
 */

void get_open_volume_pixmap(GdkPixmap ** pixmap, GdkBitmap ** mask)
{
	*pixmap = volume_open_pixmap;
	*mask = volume_open_pixmap_mask;
}

void get_refresh_pixmap(GdkPixmap ** pixmap, GdkBitmap ** mask)
{
	*pixmap = refresh_pixmap;
	*mask = refresh_pixmap_mask;
}

void get_topmost_pixmap(GdkPixmap ** pixmap, GdkBitmap ** mask)
{
	*pixmap = topmost_pixmap;
	*mask = topmost_pixmap_mask;
}

void get_ok_pixmap(GdkPixmap ** pixmap, GdkBitmap ** mask)
{
	*pixmap = ok_pixmap;
	*mask = ok_pixmap_mask;
}

void get_cancel_pixmap(GdkPixmap ** pixmap, GdkBitmap ** mask)
{
	*pixmap = cancel_pixmap;
	*mask = cancel_pixmap_mask;
}

void get_exit_pixmap(GdkPixmap ** pixmap, GdkBitmap ** mask)
{
	*pixmap = exit_pixmap;
	*mask = exit_pixmap_mask;
}

void get_exit_16_pixmap(GdkPixmap ** pixmap, GdkBitmap ** mask)
{
	*pixmap = exit_16_pixmap;
	*mask = exit_16_pixmap_mask;
}

void get_commit_pixmap(GdkPixmap ** pixmap, GdkBitmap ** mask)
{
	*pixmap = commit_pixmap;
	*mask = commit_pixmap_mask;
}

void get_commit_16_pixmap(GdkPixmap ** pixmap, GdkBitmap ** mask)
{
	*pixmap = commit_16_pixmap;
	*mask = commit_16_pixmap_mask;
}

void get_clear_pixmap(GdkPixmap ** pixmap, GdkBitmap ** mask)
{
	*pixmap = clear_pixmap;
	*mask = clear_pixmap_mask;
}

void get_help_16_pixmap(GdkPixmap ** pixmap, GdkBitmap ** mask)
{
	*pixmap = help_16_pixmap;
	*mask = help_16_pixmap_mask;
}

void get_exec_16_pixmap(GdkPixmap ** pixmap, GdkBitmap ** mask)
{
	*pixmap = exec_16_pixmap;
	*mask = exec_16_pixmap_mask;
}

void get_search_16_pixmap(GdkPixmap ** pixmap, GdkBitmap ** mask)
{
	*pixmap = search_16_pixmap;
	*mask = search_16_pixmap_mask;
}

/*
 *
 *   void set_clist_row_pixmap (GtkCList *, gint, object_type_t)
 *   
 *   Description:
 *      This routine places a pixmap in the first column of the
 *      given row of a clist. The pixmap chosen represents the
 *      the given object type.
 * 
 *   Entry:
 *      clist - the GtkCList to be updated
 *      row   - the row on which to place the pixmap
 *      type  - the object type for which we want a pixmap for
 *
 *   Exit:
 *      The pixmap representing the object is set in the first
 *      column at the row given for the clist.
 *
 */
void set_clist_row_pixmap(GtkCList * clist, gint row, object_type_t type)
{
	GdkBitmap *mask;
	GdkPixmap *pixmap;

	get_pixmap_for_object_type(type, &pixmap, &mask);
	gtk_clist_set_pixmap(clist, row, 0, pixmap, mask);
}

/*
 *
 *   GtkWidget *create_pixmap_widget (const gchar *name)
 *   
 *   Description:
 *      This routine takes a name and retrieves the Gdk pixmap and
 *      mask and creates a GtkPixmap.
 * 
 *   Entry:
 *      name - the pixmap name
 *
 *   Exit:
 *      Returns a GtkPixmap corresponding to the pixmap name or NULL
 *      if the name was not recognized.
 *
 */
GtkWidget *create_pixmap_widget(const gchar * name)
{
	GdkBitmap *mask = NULL;
	GtkWidget *gtk_pixmap = NULL;
	GdkPixmap *gdk_pixmap = NULL;

	if (g_strcasecmp(name, "refresh.xpm") == 0) {
		gdk_pixmap = refresh_pixmap;
		mask = refresh_pixmap_mask;
	} else if (g_strcasecmp(name, "commit.xpm") == 0) {
		gdk_pixmap = commit_pixmap;
		mask = commit_pixmap_mask;
	} else if (g_strcasecmp(name, "exit.xpm") == 0) {
		gdk_pixmap = exit_pixmap;
		mask = exit_pixmap_mask;
	} else if (g_strcasecmp(name, "stock_help-16.xpm") == 0) {
		gdk_pixmap = help_16_pixmap;
		mask = help_16_pixmap_mask;
	} else if (g_strcasecmp(name, "stock_cancel.xpm") == 0) {
		gdk_pixmap = cancel_pixmap;
		mask = cancel_pixmap_mask;
	} else if (g_strcasecmp(name, "stock_left.xpm") == 0) {
		gdk_pixmap = left_arrow_pixmap;
		mask = left_arrow_pixmap_mask;
	} else if (g_strcasecmp(name, "stock_right.xpm") == 0) {
		gdk_pixmap = right_arrow_pixmap;
		mask = right_arrow_pixmap_mask;
	} else if (g_strcasecmp(name, "stock_ok.xpm") == 0) {
		gdk_pixmap = ok_pixmap;
		mask = ok_pixmap_mask;
	}

	if (gdk_pixmap)
		gtk_pixmap = gtk_pixmap_new(gdk_pixmap, mask);

	return gtk_pixmap;
}
