/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libdrivelink.so
 *
 *   File: dl_missing_child.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "dl_common.h"


/*
 *  Function:  dl_alloc_missing_child_object
 *
 *  Called to malloc and setup a missing child storage object.
 */
static inline storage_object_t * dl_alloc_missing_child_object( void )
{
        int  rc;
        storage_object_t *child=NULL;

        rc = EngFncs->allocate_evms_object(NULL, &child);
        if (rc==0) {
                child->start        = 0;
                child->plugin       = dl_plugin_record;
                child->private_data = malloc(sizeof(missing_child_private_data_t));
                if (child->private_data) {
                        ((missing_child_private_data_t *)child->private_data)->signature = MISSING_CHILD_SIGNATURE;
                }
                else{
                        EngFncs->free_evms_object(child);
                        child=NULL;
                }
        }

        return child;
}

/*
 *  Function: dl_build_missing_feature_header
 *
 *  Called to build a simple feature header for a missing child object.
 */
static int dl_build_missing_feature_header( storage_object_t  *child,
                                            char              *drivelink_name )
{
        int  rc=0;
        evms_feature_header_t *fh;

        LOG_ENTRY();

        REQUIRE(child!=NULL);
        REQUIRE(strlen(drivelink_name)>0);

        fh = (evms_feature_header_t *) calloc(1, sizeof(evms_feature_header_t) );
        if (fh ){
                child->feature_header = fh;
                fh->signature   = EVMS_FEATURE_HEADER_SIGNATURE;
                fh->feature_id  = SetPluginID(EVMS_OEM_IBM, EVMS_FEATURE, EVMS_DRIVELINK_FEATURE_ID );
                strncpy( fh->object_name, drivelink_name, EVMS_VOLUME_NAME_SIZE );
        }
        else {
                rc = ENOMEM;
        }

        LOG_EXIT_INT(rc);
        return rc;
}

/*
 *  Function:  dl_build_missing_child
 *
 *  Called when a drivelink storage object is missing a child
 *  link.  This routine will build a missing child object for the
 *  specified link. The drivelink object will create a DM error
 *  mapping for the region of the drivelink that the missing child
 *  object represents.
 */
int dl_build_missing_child( storage_object_t *drivelink, int index )
{
        storage_object_t *object;
        drivelink_private_data_t *pdata;
        list_element_t iter1, iter2;
        int rc;

        LOG_ENTRY();

        REQUIRE(dl_isa_drivelink(drivelink) == TRUE);
        REQUIRE(index >= 0 && index < EVMS_DRIVELINK_MAX_ENTRIES);

        object = dl_alloc_missing_child_object();

        if (object==NULL) {
                rc = ENOMEM;
        }
        else {
                rc = dl_build_missing_feature_header(object, drivelink->name );
        }

        if (rc==0) {
                pdata = (drivelink_private_data_t *)drivelink->private_data;
                object->size = pdata->ordering_table[index].child_vsize;

                pdata->drive_link[index].sector_count    = pdata->ordering_table[index].child_vsize;
                pdata->drive_link[index].padding         = 0;
                pdata->drive_link[index].serial_number   = pdata->ordering_table[index].child_serial_number;
                pdata->drive_link[index].index           = index;
                pdata->drive_link[index].flags           = DL_FLAG_MISSING;
                pdata->drive_link[index].object          = object;

                sprintf( object->name, "%s_missing_child%d", drivelink->name, index );

                iter1 = EngFncs->insert_thing(drivelink->child_objects, object,
                                              INSERT_AFTER, NULL);

                if (iter1) {
                        iter2 = EngFncs->insert_thing(object->parent_objects,
                                                      drivelink,
                                                      INSERT_AFTER, NULL);
                        if (!iter2) {
                                rc = ENOMEM;
                        }
                } else {
                        rc = ENOMEM;
                }
        }

        if (rc) {
                dl_free_missing_child_object(object);
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function:  dl_replace_missing_child
 *
 *  Called when a drivelink storage object is missing a child
 *  link and the user wants to replace a missing child object
 *  with an available top-level object.
 */
int dl_replace_missing_child( storage_object_t *drivelink,
                              storage_object_t *missing_child,
                              storage_object_t *new_child )
{
        int  i, index, rc = EINVAL;
        drivelink_private_data_t  *pdata;

        LOG_ENTRY();

        REQUIRE( dl_isa_drivelink(drivelink) == TRUE );
        REQUIRE( missing_child != NULL);
        REQUIRE( new_child != NULL );
        REQUIRE( new_child->disk_group == drivelink->disk_group);

        LOG_DEBUG("drivelink= %s  missing_child= %s  new_child= %s\n",
                   drivelink->name, missing_child->name, new_child->name);

        pdata = (drivelink_private_data_t *) drivelink->private_data;

        // find the missing child in our drivelink
        for (i=0,index=-1; i<pdata->drive_link_count; i++) {
                if (pdata->drive_link[i].object == missing_child) {
                        index=i;
                        break;
                }
        }

        REQUIRE(index>=0);

        if (new_child->feature_header == NULL) {
                new_child->feature_header = (evms_feature_header_t *) EngFncs->engine_alloc( sizeof(evms_feature_header_t) );
        }

        if (new_child->feature_header != NULL) {
                rc = EngFncs->adopt_child(drivelink,new_child);
                if (rc==0) {
                        rc = dl_add_child_object_to_drivelink( drivelink,
                                                               new_child,
                                                               pdata->drive_link[index].serial_number,
                                                               new_child->feature_header );
                        if (rc==0) {
                                EngFncs->remove_thing(missing_child->parent_objects, drivelink);
                                dl_build_ordered_child_object_list( drivelink, &drivelink->child_objects );
                                drivelink->flags |= SOFLAG_DIRTY;
				if (drivelink->flags & SOFLAG_ACTIVE) {
					drivelink->flags |= SOFLAG_NEEDS_ACTIVATE;
				}
                        }
                        else {
                                dl_add_child_object_to_drivelink( drivelink,
                                                                  missing_child,
                                                                  pdata->drive_link[index].serial_number,
                                                                  missing_child->feature_header );
                        }

                }
        }
        else {
                rc = ENOMEM;
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function: validate_missing_child_target
 *
 *  Called with a missing child object and a replacement object so
 *  that we can validate that the replacement object is suitable
 *  for replacing the missing drivelink child.
 */
int dl_validate_missing_child_replace_target( storage_object_t *missing_child,
                                              storage_object_t *new_child )
{
        int  i, index, rc = EINVAL;
        storage_object_t *drivelink=NULL;
        drivelink_private_data_t  *pdata;
        sector_count_t minimum_size=0;

        LOG_ENTRY();

        REQUIRE( dl_isa_missing_child(missing_child) == TRUE);
        drivelink = dl_get_parent(missing_child);
        REQUIRE( drivelink != NULL );
        REQUIRE( new_child != NULL );
        REQUIRE( new_child->disk_group == drivelink->disk_group);

        LOG_DEBUG("replace object= %s  size= %"PRIu64" sectors.\n", new_child->name, new_child->size);

        pdata = (drivelink_private_data_t *) drivelink->private_data;

        // find the missing child in our drivelink
        for (i=0,index=-1; i<pdata->drive_link_count; i++) {
                if (pdata->drive_link[i].object == missing_child) {
                        index=i;
                        break;
                }
        }

        REQUIRE(index>=0);

        minimum_size = pdata->ordering_table[index].child_vsize +
                                (DRIVELINK_METADATA_SECTOR_COUNT*2) + (FEATURE_HEADER_SECTOR_COUNT*2);

        if (new_child->size >= minimum_size) {
                rc = 0;
        }
        else {
                LOG_DEBUG("replace object is too small ... need %"PRIu64" sectors\n", minimum_size);
                rc = ENOSPC;
        }

        LOG_EXIT_INT(rc);
        return rc;
}

/*
 *  Function:  can_replace_missing_child
 *
 *  Called with a missing child object to see if it is currently
 *  possible to replace it with an available object.
 */
boolean dl_can_replace_missing_child( storage_object_t *missing_child )
{
        int rc;
        list_anchor_t list = EngFncs->allocate_list();
        list_element_t iter;
        storage_object_t *object;

        LOG_ENTRY();

        if( dl_isa_missing_child(missing_child) == FALSE || list == NULL){
                LOG_EXIT_BOOL(FALSE);
                return FALSE;
        }

        rc = EngFncs->get_object_list( 0,
                                              DATA_TYPE,
                                              NULL,
                                              NULL,
                                              TOPMOST,
                                              &list );

        if (rc==0) {
                LIST_FOR_EACH(list, iter, object) {
                        if (dl_validate_missing_child_replace_target(missing_child,object)==0) {
                                LOG_EXIT_BOOL(TRUE);
                                return TRUE;
                        }
                }
        }

        LOG_EXIT_BOOL(FALSE);
        return FALSE;
}

