/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: dm-ioctl4.c
 *
 * Routines specific to version 4 of the Device-Mapper ioctl interface.
 */

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <pthread.h>
#include <sys/ioctl.h>

#include "fullengine.h"
#include "engine.h"
#include "memman.h"
#include "dm.h"
#include "dm-ioctl.h"
#include "dm-ioctl4.h"
#include "remote.h"

#define ALIGNMENT		sizeof(u_int64_t)
#define IOCTL_MIN_LENGTH	16 * 1024
#define SLASH_REPLACEMENT	'|'

typedef struct dmi_buffer {
	struct dmi_buffer *next;
	dm_ioctl_t *buffer;
	unsigned int size;
	int in_use;
} dmi_buffer_t;

static dmi_buffer_t *dmi_buffer_list = NULL;
static pthread_mutex_t dmi_buffer_mutex = PTHREAD_MUTEX_INITIALIZER;

static inline void *align(void *ptr)
{
	register unsigned long a = ALIGNMENT - 1;
	return (void *)(((unsigned long)ptr + a) & ~a);
}

/**
 * remove_slashes
 *
 * Search a string and change all slashes to a replacement character.
 **/
static void remove_slashes(char *string)
{
	for (; *string; string++)
		if (*string == '/') *string = SLASH_REPLACEMENT;
}

/**
 * add_slashes
 *
 * Search a string and change all slash-replacement-characters to slashes.
 **/
static void add_slashes(char *string)
{
	for (; *string; string++)
		if (*string == SLASH_REPLACEMENT) *string = '/';
}

/**
 * add_ioctl_target
 *
 * @target:	The target information to add to the ioctl.
 * @begin:	The location to put the target info.
 * @end:	The end of the ioctl packet, for boundary checking.
 *
 * Add information about a single target to the end of an ioctl packet.
 **/
static void *add_ioctl_target(dm_target_t *target,
			      void *begin,
			      void *end)
{
	dm_ioctl_target_t i_target;
	void *params = begin;
	unsigned long params_length = strlen(target->params);

	LOG_PROC_ENTRY();

	params += sizeof(dm_ioctl_target_t);
	if (params + params_length + 1 >= end) {
		/* Ran off the end of the ioctl packet. */
		LOG_PROC_EXIT_PTR(NULL);
		return NULL;
	}

	/* Initialize the ioctl target. */
	i_target.start = target->start;
	i_target.length = target->length;
	i_target.status = 0;
	strncpy(i_target.target_type, dm_target_type_info[target->type].name,
		sizeof(i_target.target_type));

	/* Copy the parameter string. */
	strcpy((char *)params, target->params);
	params += params_length + 1;
	params = align(params);

	i_target.next = params - begin;
	memcpy(begin, &i_target, sizeof(dm_ioctl_target_t));

	LOG_PROC_EXIT_PTR(params);
	return params;
}

/**
 * build_target_list
 *
 * @dmi: Pointer to ioctl packet.
 *
 * Convert an ioctl packet to a list of targets.
 **/
static dm_target_t *build_target_list(dm_ioctl_t *dmi)
{
	dm_target_t *target, *target_list = NULL;
	dm_ioctl_target_t *i_target;
	dm_target_type type;
	char *data_start = (char*)dmi + dmi->data_start;
	unsigned int num_devs, num_groups;
	char *params;
	int i, rc;

	LOG_PROC_ENTRY();

	i_target = (dm_ioctl_target_t*)data_start;
	for (i = 0; i < dmi->target_count; i++) {
		params = (char*)i_target + sizeof(dm_ioctl_target_t);
		num_devs = 0;
		num_groups = 0;

		/* Determine the target type. */
		for (type = 0; type <= DM_TARGET_MAX; type++) {
			if (!strncmp(i_target->target_type,
				     dm_target_type_info[type].name,
				     DM_MAX_TYPE_NAME)) {
				break;
			}
		}

		if (type > DM_TARGET_MAX) {
			/* Invalid target identifier. */
			LOG_ERROR("Invalid target type (%d) in ioctl packet.\n",
				  type);
			goto error;
		}

		/* Get the number of devs and groups from the target type. */
		rc = dm_target_type_info[type].pretranslate_params(params,
								   &num_devs,
								   &num_groups);
		if (rc) {
			LOG_ERROR("Error getting number of devices and "
				  "groups from the target type.\n");
			goto error;
		}

		/* Allocate a target structure. */
		target = dm_allocate_target(type, i_target->start,
					    i_target->length,
					    num_devs, num_groups);
		if (!target) {
			LOG_ERROR("Error allocating target for type \"%s\"\n",
				  dm_target_type_info[type].name);
			goto error;
		}
		target->params = params;

		rc = dm_target_type_info[type].translate_params(target);
		if (rc) {
			LOG_ERROR("Invalid parameter string for target type "
				  "\"%s\"\n", dm_target_type_info[type].name);
			LOG_ERROR("   Returned parameter string is: %s\n",
				  params);
			goto error;
		}

		dm_add_target(target, &target_list);

		i_target = (dm_ioctl_target_t*)(data_start + i_target->next);
	}

	/* Need to NULL the params fields in each target. */
	for (target = target_list; target; target = target->next) {
		target->params = NULL;
	}

	LOG_PROC_EXIT_PTR(target_list);
	return target_list;

error:
	LOG_ERROR("Error building target list. Name = %s\n", dmi->name);
	dm_deallocate_targets(target_list);
	LOG_PROC_EXIT_PTR(NULL);
	return NULL;
}

/**
 * build_device_list
 *
 * @dmi: Pointer to ioctl packet.
 *
 * Convert an ioctl packet to a list of devices.
 **/
static dm_device_list_t *build_device_list(dm_ioctl_t *dmi)
{
	dm_name_list_t *name, *old_name;
	dm_device_list_t *device, *device_list = NULL;
	char *data_start = (char*)dmi + dmi->data_start;

	LOG_PROC_ENTRY();

	name = (dm_name_list_t*)data_start;
	
	if (name->dev == 0) {
		goto out;
	}

	do {
		device = engine_alloc(sizeof(*device));
		if (!device) {
			dm_deallocate_device_list(device_list);
			device_list = NULL;
			goto out;
		}

		device->dev_major = major(name->dev);
		device->dev_minor = minor(name->dev);
		strncpy(device->name, name->name, EVMS_NAME_SIZE);
		add_slashes(device->name);
		device->next = device_list;
		device_list = device;

		old_name = name;
		name = (dm_name_list_t*)((char*)name + name->next);

		LOG_DEBUG("Found device %s (%x:%x)\n", device->name,
			  device->dev_major, device->dev_minor);
	} while (old_name->next);

out:
	LOG_PROC_EXIT_PTR(device_list);
	return device_list;
}

/**
 * build_module_list
 *
 * @dmi: Pointer to ioctl packet.
 *
 * Convert an ioctl packet to a list of DM target-modules.
 **/
static dm_module_list_t *build_module_list(dm_ioctl_t *dmi)
{
	dm_target_versions_t *version, *old_version;
	dm_module_list_t *module, *module_list = NULL;
	char *data_start = (char*)dmi + dmi->data_start;

	LOG_PROC_ENTRY();

	version = (dm_target_versions_t*)data_start;
	
	if (version->name[0] == '\0') {
		goto out;
	}

	do {
		module = engine_alloc(sizeof(*module));
		if (!module) {
			dm_deallocate_module_list(module_list);
			module_list = NULL;
			goto out;
		}

		module->version.major = version->version[0];
		module->version.minor = version->version[1];
		module->version.patchlevel = version->version[2];
		strncpy(module->name, version->name, EVMS_NAME_SIZE);
		module->next = module_list;
		module_list = module;

		old_version = version;
		version = (dm_target_versions_t*)((char*)version + version->next);

		LOG_DEBUG("Found module %s (%u.%u.%u)\n", module->name,
			  module->version.major, module->version.minor,
			  module->version.patchlevel);
	} while (old_version->next);

out:
	LOG_PROC_EXIT_PTR(module_list);
	return module_list;
}

/**
 * get_ioctl_packet
 *
 * @packet_length: Size (in bytes) of desired packet.
 *
 * Instead of allocating a new buffer for each ioctl, keep a list of all
 * previously allocated ioctl buffers, and whether they're in use or not. When
 * a new buffer is needed, check to see if one is available on the list. If
 * one isn't available, allocate a new buffer and add it to the list.
 **/
static dm_ioctl_t *get_ioctl_packet(unsigned int packet_size)
{
	dmi_buffer_t *entry;
	dm_ioctl_t *new_dmi = NULL;

	LOG_PROC_ENTRY();

	pthread_mutex_lock(&dmi_buffer_mutex);
	for (entry = dmi_buffer_list; entry; entry = entry->next) {
		if (!entry->in_use && packet_size <= entry->size) {
			entry->in_use = TRUE;
			new_dmi = entry->buffer;
			memset(new_dmi, 0, entry->size);
			break;
		}
	}
	pthread_mutex_unlock(&dmi_buffer_mutex);

	if (!new_dmi) {
		entry = engine_alloc(sizeof(*entry));
		if (entry) {
			new_dmi = engine_alloc(packet_size);
			if (new_dmi) {
				entry->buffer = new_dmi;
				entry->size = packet_size;
				entry->in_use = TRUE;
				pthread_mutex_lock(&dmi_buffer_mutex);
				entry->next = dmi_buffer_list;
				dmi_buffer_list = entry;
				pthread_mutex_unlock(&dmi_buffer_mutex);
			} else {
				engine_free(entry);
			}
		}
	}

	LOG_PROC_EXIT_PTR(new_dmi);
	return new_dmi;
}

/**
 * put_ioctl_packet
 *
 * Mark this ioctl packet as available.
 **/
static void put_ioctl_packet(dm_ioctl_t *dmi)
{
	dmi_buffer_t *entry;

	LOG_PROC_ENTRY();

	pthread_mutex_lock(&dmi_buffer_mutex);
	for (entry = dmi_buffer_list; entry; entry = entry->next) {
		if (entry->buffer == dmi) {
			entry->in_use = FALSE;
			break;
		}
	}
	pthread_mutex_unlock(&dmi_buffer_mutex);

	LOG_PROC_EXIT_VOID();
}

/**
 * build_ioctl_packet
 *
 * @name:	Name of device
 * @target_list:List of targets to include with this packet.
 * @new_name:	New device name. Don't specify both target_list and new_name.
 *
 * Construct a DM ioctl packet with the given information. Each DM ioctl
 * packet starts with a header (dm_ioctl_t), followed immediately by padding
 * to ensure 64-bit alignment of the remaining data. After the header and
 * the padding can be one of two things:
 * 1) An array of targets. Each target begins with a dm_ioctl_target_t
 *    structure, and is followed immediately by an ASCII string representing
 *    the target-type-specific data. After this string is enough padding to
 *    ensure 64-bit alignment of the next target.
 * 2) A new device name, as an ASCII string.
 *
 * This function returns a pointer to the beginning of the constructed
 * packet. The caller is responsible for filling in extra information
 * in the ioctl header (flags and dev) based on the command being sent.
 **/
static dm_ioctl_t *build_ioctl_packet(unsigned char *name,
				      dm_target_t *target_list,
				      unsigned char *new_name)
{
	dm_ioctl_t *dmi;
	dm_target_t *target;
	void *begin, *end;
	unsigned long packet_length = sizeof(dm_ioctl_t) + ALIGNMENT;
	int target_count = 0;

	LOG_PROC_ENTRY();

	/* Can't specify both target_list and new_name. */
	if (target_list != NULL && new_name != NULL) {
		LOG_PROC_EXIT_PTR(NULL);
		return NULL;
	}

	/* Count the number of targets. */
	for (target = target_list; target; target = target->next) {
		packet_length += sizeof(dm_ioctl_target_t);
		packet_length += strlen(target->params) + 1 + ALIGNMENT;
		target_count++;
	}

	if (new_name) {
		packet_length += strlen(new_name) + 1;
	}

	/* Make sure packet is minimum length, in case target information
	 * needs to be returned from the kernel.
	 */
	if (packet_length < IOCTL_MIN_LENGTH) {
		packet_length = IOCTL_MIN_LENGTH;
	}

	/* Allocate the ioctl packet. */
	dmi = get_ioctl_packet(packet_length);
	if (!dmi) {
		LOG_ERROR("Error allocating memory for ioctl packet. ");
		LOG_ERROR("Name = %s\n", name);
		LOG_PROC_EXIT_PTR(NULL);
		return NULL;
	}

	/* Initialize the ioctl header. */
	dmi->version[0] = DM_VERSION_MAJOR;
	dmi->version[1] = DM_VERSION_MINOR;
	dmi->version[2] = DM_VERSION_PATCHLEVEL;
	dmi->data_size = packet_length;
	dmi->data_start = (u_int32_t)(unsigned long)align((void *)sizeof(dm_ioctl_t));
	dmi->target_count = target_count;
	if (name) {
		strncpy(dmi->name, name, sizeof(dmi->name));
		remove_slashes(dmi->name);
	}

	/* Add extra information after the ioctl header. */
	begin = (void *)((char *)dmi + dmi->data_start);
	end = (void *)((char *)dmi + dmi->data_size);

	/* Add each target to the packet. */
	for (target = target_list; target; target = target->next) {
		begin = add_ioctl_target(target, begin, end);
		if (!begin) {
			LOG_PROC_EXIT_PTR(NULL);
			return NULL;
		}
	}

	/* Add the new name to the packet. */
	if (new_name) {
		strcpy(begin, new_name);
		remove_slashes(begin);
	}

	LOG_PROC_EXIT_PTR(dmi);
	return dmi;
}

/**
 * run_command_v4
 *
 * Execute the ioctl command for version-4 of DM.
 **/
int run_command_v4(void *dmi_in, unsigned long command)
{
	dm_ioctl_t *dmi = dmi_in;
	int rc = 0;

	LOG_PROC_ENTRY();
	LOG_DEBUG("Issuing DM ioctl %ld for device %s.\n",
		  _IOC_NR(command), dmi->name);

	if (dm_control_fd != 0) {
		rc = ioctl(dm_control_fd, command, dmi);
		if (rc) {
			rc = errno;
			if (!(command == DM_DEV_STATUS && rc == ENXIO)) {
				LOG_ERROR("Error returned from ioctl call: "
					  "%d: %s.\n", rc, strerror(rc));
			}
		}
	} else {
		LOG_WARNING("Device-Mapper control file not open.\n");
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_get_version_v4
 *
 * Query Device-Mapper for the current version of the ioctl interface. This
 * uses the version-4 of the DM ioctl interface.
 **/
int dm_get_version_v4(int *major, int *minor, int *patch)
{
	dm_ioctl_t *dmi;
	int rc;

	LOG_PROC_ENTRY();

	dmi = build_ioctl_packet(NULL, NULL, NULL);
	if (!dmi) {
		rc = ENOMEM;
		goto out;
	}

	/* Run the command. */
	rc = run_command_v4(dmi, DM_VERSION);
	if (rc) {
		goto out;
	}

	*major = dmi->version[0];
	*minor = dmi->version[1];
	*patch = dmi->version[2];
	rc = 0;

out:
	put_ioctl_packet(dmi);
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_suspend_v4
 *
 * @name:	Name of device to be suspended.
 * @node:	Cluster-node that owns this device.
 * @suspend:	TRUE for suspend. FALSE for resume.
 *
 * Suspend or resume the Device-Mapper device representing an active EVMS
 * object or volume. This works with version-4 of the DM ioctl interface.
 **/
int dm_suspend_v4(char *name, int suspend)
{
	dm_ioctl_t *dmi;
	int rc;

	LOG_PROC_ENTRY();

	/* Build the ioctl packet. */
	dmi = build_ioctl_packet(name, NULL, NULL);
	if (!dmi) {
		rc = ENOMEM;
		goto out;
	}

	/* Suspend and resume are the same ioctl, with different flags. */
	if (suspend) {
		dmi->flags |= DM_SUSPEND_FLAG;
	}

	/* Run the command. */
	rc = run_command_v4(dmi, DM_DEV_SUSPEND);

out:
	put_ioctl_packet(dmi);
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_activate_v4
 *
 * @name:	Name of device to be (re)activated.
 * @node:	Cluster-node that owns this device.
 * @target_list:List of targets that will make up the device's mapping.
 * @reactivate:	TRUE to reactivate an already-active device.
 * @read_only:	TRUE to make the device read-only.
 * @dev_major:	Return the device major number.
 * @dev_minor:	Return the device minor number.
 *
 * Activate (or re-activate) an EVMS object or volume as a Device-Mapper device.
 * This works with version-4 of the DM ioctl interface.
 *
 * Unlike version 3, this is not done with a single command. To fully
 * activate a device, it must first be created (without any mapping). Then
 * a mapping has to be added, and then the device can be resumed, which
 * will activate the mapping.
 **/
int dm_activate_v4(char *name,
		   dm_target_t *target_list,
		   int reactivate,
		   int read_only,
		   u_int32_t *dev_major,
		   u_int32_t *dev_minor)
{
	int rc;

	LOG_PROC_ENTRY();

	if (!reactivate) {
		rc = dm_create_v4(name, dev_major, dev_minor);
		if (rc) {
			goto out;
		}
	}

	rc = dm_load_targets_v4(name, target_list, read_only);
	if (rc) {
		goto out2;
	}

	rc = dm_suspend_v4(name, FALSE);
	if (rc) {
		goto out2;
	}

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;

out2:
	if (!reactivate) {
		dm_deactivate_v4(name);
		*dev_major = 0;
		*dev_minor = 0;
	}
	goto out;
}

/**
 * dm_deactivate_v4
 *
 * @name: Name of device to deactivate.
 * @node: Cluster-node that owns this device.
 *
 * Deactivate the Device-Mapper device representing an EVMS storage object
 * or volume.
 **/
int dm_deactivate_v4(char *name)
{
	dm_ioctl_t *dmi;
	int rc;

	LOG_PROC_ENTRY();

	/* Build the ioctl packet. */
	dmi = build_ioctl_packet(name, NULL, NULL);
	if (!dmi) {
		rc = ENOMEM;
		goto out;
	}

	/* Run the command. */
	rc = run_command_v4(dmi, DM_DEV_REMOVE);

out:
	put_ioctl_packet(dmi);
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_rename_v4
 *
 * @old_name:	The device's old name.
 * @new_name:	The device's new name.
 * @node:	Cluster-node that owns this device.
 *
 * Tell Device-Mapper to change the name for an active object or volume.
 **/
int dm_rename_v4(char *old_name, char *new_name)
{
	dm_ioctl_t *dmi;
	int rc;

	LOG_PROC_ENTRY();

	dmi = build_ioctl_packet(old_name, NULL, new_name);
	if (!dmi) {
		rc = ENOMEM;
		goto out;
	}

	/* Run the command. */
	rc = run_command_v4(dmi, DM_DEV_RENAME);

out:
	put_ioctl_packet(dmi);
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_create_v4
 *
 * @name:	Name of device to create.
 * @node:	Cluster-node that owns this device.
 * @dev_major:	Returned major number of the new device
 * @dev_minor:	Returned minor number of the new device
 *
 * Create a Device-Mapper device for an EVMS object or volume. This device will
 * have no initial mapping. Before it can be used, a mapping must be loaded with
 * dm_load_targets(), and then activated with dm_suspend().
 **/
int dm_create_v4(char *name,
		 u_int32_t *dev_major,
		 u_int32_t *dev_minor)
{
	dm_ioctl_t *dmi;
	int rc;

	LOG_PROC_ENTRY();

	/* Build the ioctl packet. */
	dmi = build_ioctl_packet(name, NULL, NULL);
	if (!dmi) {
		rc = ENOMEM;
		goto out;
	}

	rc = run_command_v4(dmi, DM_DEV_CREATE);
	if (rc) {
		goto out;
	}

	*dev_major = major(dmi->dev);
	*dev_minor = minor(dmi->dev);

out:
	put_ioctl_packet(dmi);
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_update_status_v4
 *
 * @name:	Name of device to get status for.
 * @node:	Cluster-node that owns this device.
 * @active:	Return TRUE if device is active in the kernel.
 * @read_only:	Return TRUE if device is read-only in the kernel.
 * @dev_major:	Return the device's major number.
 * @dev_minor:	Return the device's minor number.
 *
 * Query device-mapper for the status of an EVMS object or volume. If the
 * device is found in DM, return the state as active. If it is active, also
 * return the read-only state and the device major:minor number.
 **/
int dm_update_status_v4(char *name,
			int *active,
			int *read_only,
			u_int32_t *dev_major,
			u_int32_t *dev_minor)
{
	dm_ioctl_t *dmi;
	int rc;

	LOG_PROC_ENTRY();

	/* Build the ioctl packet. */
	dmi = build_ioctl_packet(name, NULL, NULL);
	if (!dmi) {
		rc = ENOMEM;
		goto out;
	}

	/* Run the command. */
	rc = run_command_v4(dmi, DM_DEV_STATUS);
	if (!rc) {
		*active = TRUE;
		*dev_major = major(dmi->dev);
		*dev_minor = minor(dmi->dev);
		*read_only = (dmi->flags & DM_READONLY_FLAG) ? TRUE : FALSE;
	} else if (rc == ENXIO) {
		/* ENXIO means DM couldn't find the device.
		 * Any other rc is a real error.
		 */
		*active = FALSE;
		*dev_major = 0;
		*dev_minor = 0;
		rc = 0;
	}

out:
	put_ioctl_packet(dmi);
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_get_targets_v4
 *
 * @name:	Name of device to get mapping for.
 * @node:	Cluster-node that owns this device.
 * @target_list:Return the list of targets.
 *
 * Query device-mapper for the set of targets that comprise an EVMS object or
 * volume. This function sets target_list to point at a list of targets. When
 * the caller is done using this list, it must be freed with a call to
 * dm_deallocate_targets().
 **/
int dm_get_targets_v4(char *name, dm_target_t **target_list)
{
	dm_ioctl_t *dmi;
	dm_target_t *targets = NULL;
	int rc;

	LOG_PROC_ENTRY();

	/* Build the ioctl packet. */
	dmi = build_ioctl_packet(name, NULL, NULL);
	if (!dmi) {
		rc = ENOMEM;
		goto out;
	}

	dmi->flags |= DM_STATUS_TABLE_FLAG;

	/* Run the command. */
	rc = run_command_v4(dmi, DM_TABLE_STATUS);
	if (rc) {
		goto out;
	}

	targets = build_target_list(dmi);
	if (!targets) {
		rc = EINVAL;
	}

out:
	*target_list = targets;
	put_ioctl_packet(dmi);
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_load_targets_v4
 *
 * @name:		Name of device to load new targets for.
 * @node:		Cluster-node that owns this device.
 * @target_list:	List of targets to load into the device.
 * @read_only:		TRUE to create the new mapping read-only.
 *
 * Load a new mapping into the device for an EVMS object or volume. This
 * mapping will initially be "inactive". To activate the map (and possibly
 * replace an existing map), call dm_suspend() on the object.
 **/
int dm_load_targets_v4(char *name,
		       dm_target_t *target_list,
		       int read_only)
{
	dm_ioctl_t *dmi;
	int rc;

	LOG_PROC_ENTRY();

	/* Build the ioctl packet. */
	dmi = build_ioctl_packet(name, target_list, NULL);
	if (!dmi) {
		rc = ENOMEM;
		goto out;
	}

	if (read_only) {
		dmi->flags |= DM_READONLY_FLAG;
	}

	/* Run the command. */
	rc = run_command_v4(dmi, DM_TABLE_LOAD);

out:
	put_ioctl_packet(dmi);
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_clear_targets_v4
 *
 * @name: Name of device to clear inactive mapping for.
 * @node: Cluster-node that owns this device.
 *
 * When dm_load_target loads a new mapping into a device, it is initially
 * "inactive". Before the device is resumed, that inactive mapping can
 * be deleted with this call.
 **/
int dm_clear_targets_v4(char *name)
{
	dm_ioctl_t *dmi;
	int rc;

	LOG_PROC_ENTRY();

	/* Build the ioctl packet. */
	dmi = build_ioctl_packet(name, NULL, NULL);
	if (!dmi) {
		rc = ENOMEM;
		goto out;
	}

	/* Run the command. */
	rc = run_command_v4(dmi, DM_TABLE_CLEAR);

out:
	put_ioctl_packet(dmi);
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_get_info_v4
 *
 * @name: Name of device to get info for.
 * @node: Cluster-node that owns this device.
 * @info: Return pointer to info string.
 *
 * Query device-mapper for the latest status info for an EVMS object or volume.
 * This function sets info to point at a newly-allocated string. When the caller
 * is done using that string, it must be freed with a call to the engine_free
 * service. Status info is target-dependent. The caller should know how to
 * interpret the returned string.

 * Currently, this function assumes the desired device has only one target.
 * Thus, only one string will be returned. May change this later to return
 * an array of strings, one for each target in the device. However, since
 * currently only snapshot provides any meaningful info for this call, there
 * is no need for multi-target info.
 **/
int dm_get_info_v4(char *name, char **info)
{
	dm_ioctl_t *dmi;
	char *info_str;
	int rc;

	LOG_PROC_ENTRY();

	/* Build the ioctl packet. */
	dmi = build_ioctl_packet(name, NULL, NULL);
	if (!dmi) {
		rc = ENOMEM;
		goto out;
	}

	/* Run the command. */
	rc = run_command_v4(dmi, DM_TABLE_STATUS);
	if (rc) {
		goto out;
	}

	info_str = (char *)dmi + dmi->data_start + sizeof(dm_ioctl_target_t);
	*info = engine_alloc(strlen(info_str)+1);
	if (!*info) {
		rc = ENOMEM;
		goto out;
	}

	strcpy(*info, info_str);

out:
	put_ioctl_packet(dmi);
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_get_devices_v4
 *
 * @device_list: Return pointer to list of active devices.
 *
 * Ask Device-Mapper for a list of all currently active devices. The name and
 * device number of each device will be returned.
 **/
int dm_get_devices_v4(dm_device_list_t **device_list)
{
	dm_ioctl_t *dmi;
	dm_device_list_t *list = NULL;
	int rc;

	LOG_PROC_ENTRY();

	/* Build the ioctl packet. */
	dmi = build_ioctl_packet(NULL, NULL, NULL);
	if (!dmi) {
		rc = ENOMEM;
		goto out;
	}

	rc = run_command_v4(dmi, DM_LIST_DEVICES);
	if (rc) {
		goto out;
	}

	list = build_device_list(dmi);

out:
	if (device_list) {
		*device_list = list;
	}
	put_ioctl_packet(dmi);
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

int dm_wait_v4(char *name, unsigned int *event_nr, char **info)
{
	dm_ioctl_t *dmi;
	char *info_str;
	int rc;

	LOG_PROC_ENTRY();

	/* Build the ioctl packet. */
	dmi = build_ioctl_packet(name, NULL, NULL);
	if (!dmi) {
		rc = ENOMEM;
		goto out;
	}

	dmi->event_nr = *event_nr;

	/* Run the command. */
	rc = run_command_v4(dmi, DM_DEV_WAIT);
	if (rc) {
		goto out;
	}

	/* Return the next event-number and the status information. */
	*event_nr = dmi->event_nr;

	info_str = (char *)dmi + dmi->data_start + sizeof(dm_ioctl_target_t);
	*info = engine_alloc(strlen(info_str)+1);
	if (!*info) {
		rc = ENOMEM;
		goto out;
	}

	strcpy(*info, info_str);

out:
	put_ioctl_packet(dmi);
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_get_modules_v4
 *
 * @module_list: Return pointer to list of currently loaded DM modules.
 *
 * Ask Device-Mapper for a list of all currently loaded target modules. The
 *  name and version number of each module will be returned.
 **/
int dm_get_modules_v4(dm_module_list_t **module_list)
{
	dm_ioctl_t *dmi;
	dm_module_list_t *list = NULL;
	int rc;

	LOG_PROC_ENTRY();

	/* Build the ioctl packet. */
	dmi = build_ioctl_packet(NULL, NULL, NULL);
	if (!dmi) {
		rc = ENOMEM;
		goto out;
	}

	/* The LIST_VERSIONS command is an addition to DM, so the ioctl
	 * version for this command must be 4.1.x instead of 4.0.x.
	 */
	dmi->version[1] = 1;

	rc = run_command_v4(dmi, DM_LIST_VERSIONS);
	if (rc) {
		goto out;
	}

	list = build_module_list(dmi);

out:
	if (module_list) {
		*module_list = list;
	}
	put_ioctl_packet(dmi);
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

