/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libgpt.so
 *
 *   File: helpers.h
 */
#ifndef GPT_HELPERS_HEADER
#define GPT_HELPERS_HEADER


int                 create_gpt_disk_private_data( LOGICALDISK  *ld );
int                 delete_gpt_disk_private_data( LOGICALDISK *ld );
DISK_PRIVATE_DATA  *get_gpt_disk_private_data( LOGICALDISK *ld );
void                delete_all_gpt_disk_private_data( void );


DISKSEG *           get_freespace_following_gpt_disk_segment( DISKSEG *seg );
DISKSEG *           find_freespace_in_seglist( list_anchor_t  seglist );
int                 find_freespace_on_gpt_disk( LOGICALDISK *ld );
int                 merge_adjacent_freedisksegs_in_gpt_seglist( list_anchor_t seglist );

DISKSEG *           allocate_gpt_freespace_disk_segment( LOGICALDISK *ld );
storage_object_t *  allocate_gpt_disk_segment( storage_object_t *object );
void                free_gpt_disk_segment( storage_object_t *seg);

storage_object_t *  create_gpt_metadata_segment( storage_object_t  *object,
                                                 lba_t              start,
                                                 sector_count_t     size,
                                                 char              *name );

storage_object_t *  build_gpt_segment_from_partition_record( storage_object_t  *obj,
                                                             gpt_partition     *part,
                                                             int                minor );

DISKSEG *           get_gpt_segment_from_minor( LOGICALDISK *ld, int minor );

int                 get_next_gpt_minor( LOGICALDISK *ld );


int                 remove_gpt_segment_from_list( list_anchor_t  seglist, storage_object_t *seg );
void *              insert_gpt_segment_into_list( list_anchor_t  seglist, storage_object_t *seg);
void *              insert_gpt_segment_into_ordered_list( list_anchor_t  seglist, storage_object_t *seg);
int                 create_protective_mbr_segment( storage_object_t *object );

char *              guid_to_string( guid_t *id );



/*    The maximum number of cylinders, heads, and sectors that a partition table entry can accomodate.
 *
 *    Cylinders are numbered 0 - 1023, for a maximum of 1024 cylinders.
 *    Heads are numbered 0 - 254, for a maximum of 255 heads.
 *    Sectors are numbered 1 - 63, for a maximum of 63 sectors per track.
 */
        #define MAX_CYLINDERS 1023
        #define MAX_HEADS     254
        #define MAX_SECTORS   63



/*
 *  Called to return the Logical Disk that a segment belongs to.
 *
 *  Can be called with any object.
 *
 *  Returns NULL on failure.
 */
static inline LOGICALDISK * get_logical_disk( storage_object_t *object )
{
        LOGICALDISK  *ld=NULL;

        if (object) {

                if (object->object_type == DISK) {
                        ld = object;
                }
                else if (object->object_type == SEGMENT && object->plugin == gpt_plugin_record_ptr) {

                        if (object->private_data) {

                                if ( ((SEG_PRIVATE_DATA *)object->private_data)->signature == GPT_SEG_MGR_PDATA_SIGNATURE ) {

                                        ld = ((SEG_PRIVATE_DATA *)object->private_data)->logical_disk;

                                }

                        }

                }

        }

        return ld;
}



/*
 *     LBA addresses are converted into CHS addressees by this formula:
 *
 *        Sector    = ( LBA MOD Sectors Per Track ) + 1
 *        Head      = ( LBA DIV Sectors Per Track ) MOD Heads Per Cylinder
 *        Cylinder  = ( LBA DIV Sectors Per Track ) DIV Heads Per Cylinder
 */
static inline int LBAtoCHS( LOGICALDISK *ld, lba_t  lba, chs_t *chs )
{
        u_int32_t          sectors_per_track;
        u_int32_t          drive_heads;
        u_int32_t          sectors_per_cylinder;
        int                rc;
        DISK_PRIVATE_DATA *disk_pdata = get_gpt_disk_private_data( ld );


        if ( (chs != NULL) &&
             (disk_pdata != NULL) &&
             (ld->geometry.sectors_per_track > 0)  ) {

                memset(chs, 0, sizeof(chs_t));

                sectors_per_track    = ld->geometry.sectors_per_track * disk_pdata->vsectors_per_block;
                drive_heads          = ld->geometry.heads;
                sectors_per_cylinder = sectors_per_track * drive_heads;

                chs->sector   = ( lba % sectors_per_track ) + 1;
                chs->head     = ( lba / sectors_per_track ) % drive_heads;
                chs->cylinder =   lba / sectors_per_cylinder;

                rc = 0;
        }
        else {
                rc = EINVAL;
        }

        return rc;
}



/*
 *     CHS addresses are converted into LBA addresses by the following formula:
 *
 *        LBA = (Sector - 1) + (Head * Sectors Per Track) + (Cylinder * Heads Per Cylinder * Sectors Per Track)
 *
 */
static inline int CHStoLBA( LOGICALDISK *ld, chs_t *chs, lba_t *callers_lba )
{
        u_int32_t          sectors_per_track;
        u_int32_t          drive_heads;
        u_int32_t          sectors_per_cylinder;
        lba_t              lba = 0;
        int                rc;
        DISK_PRIVATE_DATA *disk_pdata = get_gpt_disk_private_data( ld );


        if (  ( chs != NULL ) &&
              ( callers_lba != NULL ) &&
              ( disk_pdata != NULL ) &&
              ( ld->geometry.sectors_per_track > 0 ) ) {

                sectors_per_track    = ld->geometry.sectors_per_track * disk_pdata->vsectors_per_block;
                drive_heads          = ld->geometry.heads;
                sectors_per_cylinder = sectors_per_track * drive_heads;

                lba = (chs->sector - 1 ) + (chs->head * sectors_per_track) + (chs->cylinder * sectors_per_cylinder );

                *callers_lba = lba;

                rc = 0;
        }
        else {
                rc = EINVAL;
        }

        return rc;
}




/*
 *     A disk can be addressed by LBA or by CHS values. A partition table entry
 *     uses the following field sizes:
 *
 *     LBA - 32 bits
 *
 *     CHS - 24 bits  ( cylinders - 10 bits,  heads - 8 bits, sectors - 6 bits )
 *
 *     So, the CHS value has maximums of: 1024 cylinders, 255 heads, 63 sectors
 *
 */
static inline int LBA_to_Ptable_CHS( LOGICALDISK *ld, lba_t  lba, chs_t  *chs )
{
        int  rc;

        rc = LBAtoCHS( ld, lba, chs );

        if (rc==0) {

                /* if above 1024 cylinder limit of a partition table entry ... we need to
                 * max out the CHS values to indicate this situation. We also need to
                 * do this if we are using LBA addressing on this disk.
                 */
                if ( chs->cylinder > MAX_CYLINDERS ) {

                        if (MAX_CYLINDERS > ld->geometry.cylinders) {
                                chs->cylinder = ld->geometry.cylinders-1;
                        }
                        else {
                                chs->cylinder = MAX_CYLINDERS;
                        }

                        chs->head     = ld->geometry.heads-1;
                        chs->sector   = ld->geometry.sectors_per_track;

                }


        }

        return rc;
}



/*
 *  Returns TRUE if the LBA starts on a cylinder boundary
 */
static inline boolean  starts_on_cylinder_boundary( LOGICALDISK *ld, lba_t lba )
{
        chs_t  chs;

        if ( LBAtoCHS( ld, lba, &chs ) ) {
                return TRUE;
        }
        else {

                if ( ( chs.sector == 1 ) && ( chs.head == 0 ) ) {
                        return TRUE;
                }
                else {
                        return FALSE;
                }
        }

}


/*
 *  Returns TRUE if the LBA ends on a cylinder boundary
 */
static inline boolean  ends_on_cylinder_boundary( LOGICALDISK *ld, lba_t lba )
{
        chs_t  chs;
        DISK_PRIVATE_DATA *disk_pdata = get_gpt_disk_private_data( ld );


        if ( LBAtoCHS( ld, lba, &chs ) ) {
                return TRUE;
        }
        else {

                if ( ( chs.sector == (ld->geometry.sectors_per_track * disk_pdata->vsectors_per_block) ) &&
                     ( chs.head   == ld->geometry.heads-1 ) ) {
                        return TRUE;
                }
                else {
                        return FALSE;
                }

        }

}


/*
 *  Called to calculate the cylinder size for the specified
 *  storage object. Storage object might be either DISK or
 *  SEGMENT type of object.
 */
static inline sector_count_t get_cylinder_size( storage_object_t *ld )
{
        u_int32_t          sectors_per_track=0;
        u_int32_t          drive_heads=0;
        sector_count_t     sectors_per_cylinder=0;
        DISK_PRIVATE_DATA *disk_pdata=NULL;


        if (ld) {

                disk_pdata           = get_gpt_disk_private_data( ld );

                if (disk_pdata) {
                        sectors_per_track = ld->geometry.sectors_per_track * disk_pdata->vsectors_per_block;
                }
                else {
                        u_int32_t  vsectors_per_block = ld->geometry.bytes_per_sector >> EVMS_VSECTOR_SIZE_SHIFT;

                        sectors_per_track = ld->geometry.sectors_per_track * vsectors_per_block;
                }

                drive_heads          = ld->geometry.heads;
                sectors_per_cylinder = (sector_count_t) sectors_per_track * drive_heads;

        }

        return sectors_per_cylinder;
}


/*
 *  Returns the specified LBA rounded up to a track boundary.
 */
static inline lba_t  roundup_to_track_boundary( LOGICALDISK *ld, lba_t lba )
{
        lba_t              new_lba = lba;
        sector_count_t     extra_sectors=0;
        sector_count_t     sectors_per_track=0;
        DISK_PRIVATE_DATA *disk_pdata;

        disk_pdata = get_gpt_disk_private_data( ld );

        if ( disk_pdata ) {

                sectors_per_track = ld->geometry.sectors_per_track * disk_pdata->vsectors_per_block;

                if (sectors_per_track) {

                        extra_sectors = lba % sectors_per_track;

                        if ( extra_sectors != 0) {
                                new_lba = lba + ( sectors_per_track - extra_sectors ) - 1;
                        }

                }

        }

        return new_lba;
}



/*
 *  Returns the specified LBA rounded up to a cylinder boundary.
 */
static inline lba_t  roundup_to_cylinder_boundary( LOGICALDISK *ld, lba_t lba )
{
        lba_t              new_lba = lba;
        sector_count_t     extra_sectors=0;
        sector_count_t     sectors_per_cylinder;


        sectors_per_cylinder = get_cylinder_size(ld);

        if ( sectors_per_cylinder ) {

                extra_sectors = lba % sectors_per_cylinder;

                if ( extra_sectors != 0) {
                        new_lba = lba + ( sectors_per_cylinder - extra_sectors) - 1;
                }

        }

        return new_lba;
}


/*
 *  Returns the specified LBA rounded down to a cylinder boundary.
 */
static inline lba_t  rounddown_to_cylinder_boundary( LOGICALDISK *ld, lba_t lba )
{
        lba_t            new_lba=lba;
        sector_count_t   extra_sectors=0;
        sector_count_t   sectors_per_cylinder;



        sectors_per_cylinder = get_cylinder_size(ld);

        if (sectors_per_cylinder) {

                extra_sectors = lba % sectors_per_cylinder;

                if ( extra_sectors != 0) {
                        new_lba = lba - extra_sectors;
                }

        }

        return new_lba;
}


/*
 *  Called to test if we have private data for a disk.
 */
static inline boolean i_can_modify( storage_object_t *object )
{
        SEG_PRIVATE_DATA    *pdata;

        if (object) {

                pdata = (SEG_PRIVATE_DATA *) object->private_data;

                if (pdata) {

                        if (pdata->signature == GPT_SEG_MGR_PDATA_SIGNATURE) {
                                return TRUE;
                        }

                }
                else {
                        LOG_DEBUG("pdata is null\n");
                }
        }
        else {
                LOG_DEBUG("null object ptr\n");
        }
        return FALSE;
}



/*
 *  Returns TRUE if the callers object is a DISK storage object.
 *  This call is used to identify embedded segments, i.e. when
 *  the gpt seg mgr is partitioning a SEGMENT object ... not a
 *  DISK.
 */
static inline boolean isa_disk_object( storage_object_t  *obj )
{

        if ( obj->object_type == DISK ) {
                return TRUE;
        }
        else {
                return FALSE;
        }

}



static inline void disk_guid_to_cpu( guid_t  *g )
{
        g->time_low        = DISK_TO_CPU32(g->time_low);
        g->time_mid        = DISK_TO_CPU16(g->time_mid);
        g->time_high       = DISK_TO_CPU16(g->time_high);
        g->clock_seq_high  = DISK_TO_CPU16(g->clock_seq_high);
        g->clock_seq_low   = DISK_TO_CPU16(g->clock_seq_low);
}

static inline void cpu_guid_to_disk( guid_t  *g )
{
        g->time_low        = CPU_TO_DISK32(g->time_low);
        g->time_mid        = CPU_TO_DISK16(g->time_mid);
        g->time_high       = CPU_TO_DISK16(g->time_high);
        g->clock_seq_high  = CPU_TO_DISK16(g->clock_seq_high);
        g->clock_seq_low   = CPU_TO_DISK16(g->clock_seq_low);
}


static inline boolean disk_move_pending( storage_object_t *object )
{               
        LOGICALDISK        *ld=get_logical_disk(object);        
        DISK_PRIVATE_DATA  *disk_pdata=NULL;

        if ( ld ) {

                disk_pdata = get_gpt_disk_private_data(ld);

                if (disk_pdata) {

                        if (disk_pdata->flags & DISK_HAS_MOVE_PENDING) {
                                return TRUE;
                        }

                }

        }

        return FALSE;
}

/*
 *  Function: get device name
 *
 *  On a devfs system ... we need to change the name
 *  from ide/..../disc   to ide/..../part before appending
 *  the partition number.
 */
static inline void get_device_name(storage_object_t *ld, char *device_name)
{       
        int i;

        strcpy(device_name, ld->name);

        for (i=strlen(device_name)-1; i>=0; i--) {
                if (device_name[i] == '/') {
                        if (strncmp(&device_name[i],"/disc",5)==0) {
                                strcpy(&device_name[i+1],"");
                                return;
                        }
                }
        }

}

#endif

