/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: md_info.c
 *
 * Description: This file contains all functions related to the common creation
 *              of info for MD volumes.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>
#include <unistd.h>
#include <plugin.h>

#include "md.h"
#include "raid5_mgr.h"

#define my_plugin_record my_plugin


static int md_get_volume_info( md_volume_t            * vol,
           	               extended_info_array_t ** info_array ) {

	int rc = 0;
	storage_object_t *region;
	int info_entries;
	extended_info_array_t * info = NULL;
	extended_info_t *cur_info;
	md_member_t *member;
	list_element_t iter;

	LOG_ENTRY();
	
	region = vol->region;
	if (!region) {
		LOG_CRITICAL("No region was created for MD volume %s.\n", vol->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}
	
	if (region->flags & SOFLAG_ACTIVE) {
		rc = md_analyze_active_region(vol);
	}

	info_entries = NUM_VOLUME_INFO_ENTRIES + EngFncs->list_count(vol->members);
	if (vol->stale_disks) {
		info_entries++;
	}
	info = EngFncs->engine_alloc(sizeof(extended_info_array_t) +
				     sizeof(extended_info_t) * info_entries);
	if (info != NULL) {
		info->count = info_entries;
		cur_info = info->info;

		cur_info->name = EngFncs->engine_strdup("name");
		cur_info->title = EngFncs->engine_strdup(_("Name"));
		cur_info->desc = EngFncs->engine_strdup(_("MD volume name"));
		cur_info->type = EVMS_Type_String;
		cur_info->unit = EVMS_Unit_None;
		cur_info->format = EVMS_Format_Normal;
		cur_info->value.s = EngFncs->engine_strdup(vol->name);
		cur_info->collection_type = EVMS_Collection_None;
		cur_info->collection.list = NULL;
		cur_info->group.group_number = 0;
		cur_info->group.group_level = 0;
		cur_info->group.group_name = NULL;
		cur_info->flags = 0;

		cur_info++;
		cur_info->name = EngFncs->engine_strdup("state");
		cur_info->title = EngFncs->engine_strdup(_("State"));
		cur_info->desc = EngFncs->engine_strdup(_("State of the MD region"));
		cur_info->type = EVMS_Type_String;
		cur_info->unit = EVMS_Unit_None;
		cur_info->format = EVMS_Format_Normal;

		if (vol->flags & MD_NEW_REGION) {
			strcpy(message_buffer, _("New, Uncommitted"));
		} else {
			message_buffer[0] = '\0';
			if (vol->flags & MD_DISCOVERED) {
				strcat(message_buffer, _("Discovered"));
			}
			if (vol->flags & MD_DEGRADED) {
				if (message_buffer[0] != '\0') {
					strcat(message_buffer, ", ");
				}
				strcat(message_buffer, _("Degraded"));
			}
			if (vol->flags & MD_CORRUPT) {
				if (message_buffer[0] != '\0') {
					strcat(message_buffer, ", ");
				}
				strcat(message_buffer, _("Corrupt"));
			}
			if (vol->flags & MD_DIRTY) {
				if (message_buffer[0] != '\0') {
					strcat(message_buffer, ", ");
				}
				strcat(message_buffer, _("Dirty"));
			}
			if (region->flags & SOFLAG_ACTIVE) {
				if (message_buffer[0] != '\0') {
					strcat(message_buffer, ", ");
				}
				strcat(message_buffer, _("Active"));
			}
			if (vol->flags & MD_ARRAY_SYNCING) {
				if (message_buffer[0] != '\0') {
					strcat(message_buffer, ", ");
				}
				strcat(message_buffer, _("Syncing"));
			}
		}

		cur_info->value.s = EngFncs->engine_strdup(message_buffer);
		cur_info->collection_type = EVMS_Collection_None;
		cur_info->collection.list = NULL;
		cur_info->group.group_number = 0;
		cur_info->group.group_level = 0;
		cur_info->group.group_name = NULL;
		cur_info->flags = 0;

		cur_info++;
		cur_info->name = EngFncs->engine_strdup("personality");
		cur_info->title = EngFncs->engine_strdup(_("Personality"));
		cur_info->desc = EngFncs->engine_strdup(_("MD personality"));
		cur_info->type = EVMS_Type_String;
		cur_info->unit = EVMS_Unit_None;
		cur_info->format = EVMS_Format_Normal;
		switch (vol->personality) {
		case MD_RESERVED:
			cur_info->value.s = EngFncs->engine_strdup(_("Reserved"));
			break;
		case LINEAR:
			cur_info->value.s = EngFncs->engine_strdup(_("Linear"));
			break;
		case RAID0:
			cur_info->value.s = EngFncs->engine_strdup(_("RAID0"));
			break;
		case RAID1:
			cur_info->value.s = EngFncs->engine_strdup(_("RAID1"));
			break;
		case RAID5:
			cur_info->value.s = EngFncs->engine_strdup(_("RAID5"));
			break;
		case TRANSLUCENT:
			cur_info->value.s = EngFncs->engine_strdup(_("Translucent"));
			break;
		case HSM:
			cur_info->value.s = EngFncs->engine_strdup(_("HSM"));
			break;
		case MULTIPATH:
			cur_info->value.s = EngFncs->engine_strdup(_("Multipath"));
			break;
		}
		cur_info->collection_type = EVMS_Collection_None;
		cur_info->collection.list = NULL;
		cur_info->group.group_number = 0;
		cur_info->group.group_level = 0;
		cur_info->group.group_name = NULL;
		cur_info->flags = 0;

		cur_info++;
		cur_info->name = EngFncs->engine_strdup("superblock");
		cur_info->title = EngFncs->engine_strdup(_("Working SuperBlock"));
		cur_info->desc = EngFncs->engine_strdup(_("Copy of SuperBlock that is most up to date"));
		cur_info->type = EVMS_Type_String;
		cur_info->unit = EVMS_Unit_None;
		cur_info->format = EVMS_Format_Hex;
		cur_info->value.s = NULL;
		cur_info->collection_type = EVMS_Collection_None;
		cur_info->collection.list = NULL;
		cur_info->group.group_number = 0;
		cur_info->group.group_level = 0;
		cur_info->group.group_name = NULL;
		cur_info->flags = EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE;

		cur_info++;
		cur_info->name = EngFncs->engine_strdup("nr_disks");
		cur_info->title = EngFncs->engine_strdup(_("Number of disks"));
		cur_info->desc = EngFncs->engine_strdup(_("Number of disks found by EVMS that comprise this volume"));
		cur_info->type = EVMS_Type_Unsigned_Int32;
		cur_info->unit = EVMS_Unit_None;
		cur_info->format = EVMS_Format_Normal;
		cur_info->value.ui32 = vol->nr_disks;
		cur_info->collection_type = EVMS_Collection_None;
		cur_info->collection.list = NULL;
		cur_info->group.group_number = 0;
		cur_info->group.group_level = 0;
		cur_info->group.group_name = NULL;
		cur_info->flags = 0;

		LIST_FOR_EACH(vol->members, iter, member) {
			if (member->obj && !(member->flags & MD_MEMBER_STALE)) {
				cur_info++;
				sprintf(message_buffer,"child_object%d", member->dev_number);
				cur_info->name = EngFncs->engine_strdup(message_buffer);
				sprintf(message_buffer,_("Disk %d"), member->dev_number);
				cur_info->title = EngFncs->engine_strdup(message_buffer);
				cur_info->desc = EngFncs->engine_strdup(_("Disk that belongs to this raid volume set"));
				cur_info->type = EVMS_Type_String;
				cur_info->unit = EVMS_Unit_None;
				cur_info->format = EVMS_Format_Normal;
				cur_info->value.s = EngFncs->engine_strdup(member->obj->name);
				cur_info->collection_type = EVMS_Collection_None;
				cur_info->collection.list = NULL;
				cur_info->group.group_number = 0;
				cur_info->group.group_level = 0;
				cur_info->group.group_name = NULL;
				cur_info->flags = EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE;
			}
		}

		if (vol->stale_disks) {

			cur_info++;
			cur_info->name = EngFncs->engine_strdup("stale_disks");
			cur_info->title = EngFncs->engine_strdup(_("Number of stale disks"));
			cur_info->desc = EngFncs->engine_strdup(_("This disk has an invalid MD superblock"));
			cur_info->type = EVMS_Type_Unsigned_Int32;
			cur_info->unit = EVMS_Unit_None;
			cur_info->format = EVMS_Format_Normal;
			cur_info->value.ui32 = vol->stale_disks;
			cur_info->collection_type = EVMS_Collection_None;
			cur_info->collection.list = NULL;
			cur_info->group.group_number = 0;
			cur_info->group.group_level = 0;
			cur_info->group.group_name = NULL;
			cur_info->flags = 0;

			LIST_FOR_EACH(vol->members, iter, member) {
				if (member->obj && (member->flags & MD_MEMBER_STALE)) {
					cur_info++;
					sprintf(message_buffer,"stale_object%d", member->dev_number);
					cur_info->name = EngFncs->engine_strdup(message_buffer);
					sprintf(message_buffer,_("Stale disk %d"), member->dev_number);
					cur_info->title = EngFncs->engine_strdup(message_buffer);
					cur_info->desc = EngFncs->engine_strdup(_("This disk has an invalid MD superblock"));
					cur_info->type = EVMS_Type_String;
					cur_info->unit = EVMS_Unit_None;
					cur_info->format = EVMS_Format_Normal;
					cur_info->value.s = EngFncs->engine_strdup(member->obj->name);
					cur_info->collection_type = EVMS_Collection_None;
					cur_info->collection.list = NULL;
					cur_info->group.group_number = 0;
					cur_info->group.group_level = 0;
					cur_info->group.group_name = NULL;
					cur_info->flags = EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE;
				}
			}
		}

		info->count = info_entries;
		*info_array = info;

	} else {
		LOG_CRITICAL("Error getting memory for an extended_info_array./n");
		rc = ENOMEM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int set_disk_info(md_disk_info_t *info, extended_info_t *cur_info)
{
	int info_count = 0;
	char buf[64];

	sprintf(buf, _("number %d"), info->number);
	cur_info->name = EngFncs->engine_strdup(buf);
	cur_info->title = EngFncs->engine_strdup(_("Number"));
	cur_info->desc = EngFncs->engine_strdup(_("Disk number in the array"));
	cur_info->type = EVMS_Type_Unsigned_Int32;
	cur_info->unit = EVMS_Unit_None;
	cur_info->format = EVMS_Format_Normal;
	cur_info->value.ui32 = info->number;
	cur_info->collection_type = EVMS_Collection_None;
	cur_info->collection.list = NULL;
	cur_info->group.group_number = 0;
	cur_info->group.group_level = 0;
	cur_info->group.group_name = NULL;
	cur_info->flags = EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE;
	cur_info++;
	info_count++;

	cur_info->name = EngFncs->engine_strdup("major");
	cur_info->title = EngFncs->engine_strdup(_("Major number"));
	cur_info->desc = EngFncs->engine_strdup(_("Device major number"));
	cur_info->type = EVMS_Type_Unsigned_Int32;
	cur_info->unit = EVMS_Unit_None;
	cur_info->format = EVMS_Format_Normal;
	cur_info->value.ui32 = info->major;
	cur_info->collection_type = EVMS_Collection_None;
	cur_info->collection.list = NULL;
	cur_info->group.group_number = 0;
	cur_info->group.group_level = 0;
	cur_info->group.group_name = NULL;
	cur_info->flags = 0;
	cur_info++;
	info_count++;

	cur_info->name = EngFncs->engine_strdup("minor");
	cur_info->title = EngFncs->engine_strdup(_("Minor number"));
	cur_info->desc = EngFncs->engine_strdup(_("Device minor number"));
	cur_info->type = EVMS_Type_Unsigned_Int32;
	cur_info->unit = EVMS_Unit_None;
	cur_info->format = EVMS_Format_Normal;
	cur_info->value.ui32 = info->minor;
	cur_info->collection_type = EVMS_Collection_None;
	cur_info->collection.list = NULL;
	cur_info->group.group_number = 0;
	cur_info->group.group_level = 0;
	cur_info->group.group_name = NULL;
	cur_info->flags = 0;
	cur_info++;
	info_count++;

	cur_info->name = EngFncs->engine_strdup("raid_disk");
	cur_info->title = EngFncs->engine_strdup(_("RAID disk"));
	cur_info->desc = EngFncs->engine_strdup(_("The role of the device in the raid set"));
	cur_info->type = EVMS_Type_Int32;
	cur_info->unit = EVMS_Unit_None;
	cur_info->format = EVMS_Format_Normal;
	cur_info->value.i32 = info->raid_disk;
	cur_info->collection_type = EVMS_Collection_None;
	cur_info->collection.list = NULL;
	cur_info->group.group_number = 0;
	cur_info->group.group_level = 0;
	cur_info->group.group_name = NULL;
	cur_info->flags = 0;
	cur_info++;
	info_count++;

	cur_info->name = EngFncs->engine_strdup("state");
	cur_info->title = EngFncs->engine_strdup(_("State"));
	cur_info->desc = EngFncs->engine_strdup(_("State flags"));
	cur_info->type = EVMS_Type_String;
	cur_info->unit = EVMS_Unit_None;
	cur_info->format = EVMS_Format_Normal;
	cur_info->value.s = EngFncs->engine_strdup(info->state);
	cur_info->collection_type = EVMS_Collection_None;
	cur_info->collection.list = NULL;
	cur_info->group.group_number = 0;
	cur_info->group.group_level = 0;
	cur_info->group.group_name = NULL;
	cur_info->flags = 0;
	info_count++;

	if (info_count != NUM_DISK_INFO_ENTRIES) {
		LOG_MD_BUG();
	}
	return info_count;
}

static int get_superblock_disk_info(md_volume_t *vol, int index, extended_info_t *cur_info)
{
	mdu_disk_info_t d;
	md_disk_info_t info;
	char buffer[256];

	d.number = index;
	vol->sb_func->get_sb_disk_info_for_index(vol->sb, &d);

	info.number = d.number;
	info.major = d.major;
	info.minor = d.minor;
	info.raid_disk = d.raid_disk;
	
	buffer[0] = '\0';
	if (d.state & (1 << MD_DISK_FAULTY)) {
		strcat(buffer, _("Faulty"));
		if (d.state & (1 << MD_DISK_REMOVED)) {
			if (buffer[0] != '\0') {
				strcat(buffer, ", ");
			}
			strcat(buffer, _("Removed"));
		}	
	} else {
		if (d.state & (1 << MD_DISK_ACTIVE)) {
			if (buffer[0] != '\0') {
				strcat(buffer, ", ");
			}
			strcat(buffer, _("Active"));
		}
		if (d.state & (1 << MD_DISK_SYNC)) {
			if (buffer[0] != '\0') {
				strcat(buffer, ", ");
			}
			strcat(buffer, _("Sync"));
		}
	}
	
	// If none of the flags are set, it must be a spare disk.
	if (message_buffer[0] == '\0') {
		strcpy(buffer, _("Spare"));
	}

	info.state = buffer;

	return set_disk_info(&info, cur_info);
}

static int get_member_disk_info(md_member_t *member, extended_info_t *cur_info)
{
	md_disk_info_t info;
	char buffer[256];

	info.number = member->dev_number;
	if (member->obj && member->obj->dev_major) {
		info.major = member->obj->dev_major;
		info.minor = member->obj->dev_minor;
	} else {
		info.major = md_member_get_disk_major(member);
		info.minor = md_member_get_disk_minor(member);
	}
	info.raid_disk = md_member_get_raid_disk(member);

	buffer[0] = '\0';
	if (member->raid_disk != -1) {
		strcpy(buffer, _("Active, Sync"));
	} else {
		if (member->flags & MD_MEMBER_DISK_SPARE) {
			strcpy(buffer, _("Spare"));
		} else if (member->flags & MD_MEMBER_DISK_FAULTY) {
			strcpy(buffer, _("Faulty"));
			if (member->flags & MD_MEMBER_DISK_REMOVED) {
				if (buffer[0] != '\0') {
					strcat(buffer, ", ");
				}
				strcat(buffer, _("Removed"));
			}
		}
	}
	if (member->flags & MD_MEMBER_NEW) {
		if (buffer[0] != '\0') {
			strcat(buffer, ", ");
		}
		strcat(buffer, _("New"));
	}
	if (member->flags & MD_MEMBER_DISK_PENDING) {
		if (buffer[0] != '\0') {
			strcat(buffer, ", ");
		}
		strcat(buffer, _("Pending"));
	}
	info.state = buffer;
	
	return set_disk_info(&info, cur_info);
}

static int md_get_child_disk_info(md_member_t *member, extended_info_array_t ** info_array)
{

	int rc = 0;
	extended_info_array_t * info;

	info = EngFncs->engine_alloc(sizeof(extended_info_array_t) + sizeof(extended_info_t) * NUM_DISK_INFO_ENTRIES);
	if (info != NULL) {
		info->count = get_member_disk_info(member, info->info);
		*info_array = info;
	} else {
		LOG_CRITICAL("Error getting memory for an extended_info_array./n");
		rc = ENOMEM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int md_get_stale_disk_info(md_member_t *member, extended_info_array_t ** info_array)
{
	int rc = 0;
	extended_info_array_t * info;

	info = EngFncs->engine_alloc(sizeof(extended_info_array_t) + sizeof(extended_info_t) * NUM_DISK_INFO_ENTRIES);
	if (info != NULL) {
		info->count = get_superblock_disk_info(member->vol, member->dev_number, info->info);
		*info_array = info;
	} else {
		LOG_CRITICAL("Error getting memory for an extended_info_array./n");
		rc = ENOMEM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * md_get_superblock_info
 *
 * if child_index is -1, the master superblock will be used.
 */
static int md_get_superblock_info(
	md_volume_t *vol,
	int child_index,
	extended_info_array_t ** info_array )
{
	extended_info_array_t * info;
	extended_info_t *cur_info;
	int info_entries;
	md_super_info_t super;
	md_member_t *member;
	int i;

	if (child_index == -1) {
		// Request to display volume's master superblock information
		md_volume_get_super_info(vol, &super);
	} else {
		member = md_volume_find_member(vol, child_index);
		if (member) {
			LOG_CRITICAL("Could not locate member index %d in %s.\n",
				     child_index, vol->name);
			md_member_get_super_info(member, &super);
		} else {
			return EINVAL;
		}
	}

	info_entries = NUM_SUPER_INFO_ENTRIES + super.nr_disks*NUM_DISK_INFO_ENTRIES;
	
	info = EngFncs->engine_alloc(sizeof(extended_info_array_t) + 
				     sizeof(extended_info_t) * info_entries);
	if (info == NULL) {
		LOG_CRITICAL("Error getting memory for an extended_info_array.\n");
		return ENOMEM;
	}
	cur_info = info->info;

	cur_info->name = EngFncs->engine_strdup("md_magic");
	cur_info->title = EngFncs->engine_strdup(_("MD magic number"));
	cur_info->desc = EngFncs->engine_strdup(_("MD identifier for the volume"));
	cur_info->type = EVMS_Type_Unsigned_Int32;
	cur_info->unit = EVMS_Unit_None;
	cur_info->format = EVMS_Format_Hex;
	cur_info->value.ui32 = super.md_magic;
	cur_info->collection_type = EVMS_Collection_None;
	cur_info->collection.list = NULL;
	cur_info->group.group_number = 0;
	cur_info->group.group_level = 0;
	cur_info->group.group_name = NULL;
	cur_info->flags = 0;

	cur_info++;
	cur_info->name = EngFncs->engine_strdup("version");
	cur_info->title = EngFncs->engine_strdup(_("Version"));
	cur_info->desc = EngFncs->engine_strdup(_("Version of MD that wrote this superblock"));
	cur_info->type = EVMS_Type_String;
	cur_info->unit = EVMS_Unit_None;
	cur_info->format = EVMS_Format_Normal;
	sprintf(message_buffer, "%d.%d.%d", super.major_version,
		super.minor_version,
		super.patch_version);
	cur_info->value.s = EngFncs->engine_strdup(message_buffer);
	cur_info->collection_type = EVMS_Collection_None;
	cur_info->collection.list = NULL;
	cur_info->group.group_number = 0;
	cur_info->group.group_level = 0;
	cur_info->group.group_name = NULL;
	cur_info->flags = 0;

	cur_info++;
	cur_info->name = EngFncs->engine_strdup("set_uuid0");
	cur_info->title = EngFncs->engine_strdup(_("UUID0"));
	cur_info->desc = EngFncs->engine_strdup(_("Lowest 32 bits of the UUID"));
	cur_info->type = EVMS_Type_Unsigned_Int32;
	cur_info->unit = EVMS_Unit_None;
	cur_info->format = EVMS_Format_Hex;
	cur_info->value.ui32 = super.set_uuid0;
	cur_info->collection_type = EVMS_Collection_None;
	cur_info->collection.list = NULL;
	cur_info->group.group_number = 0;
	cur_info->group.group_level = 0;
	cur_info->group.group_name = NULL;
	cur_info->flags = 0;

	cur_info++;
	cur_info->name = EngFncs->engine_strdup("ctime");
	cur_info->title = EngFncs->engine_strdup(_("Creation time"));
	cur_info->desc = EngFncs->engine_strdup(_("The time the volume was created"));
	cur_info->type = EVMS_Type_String;
	cur_info->unit = EVMS_Unit_None;
	cur_info->format = EVMS_Format_Normal;
	strcpy(message_buffer, asctime(localtime(&super.ctime)));
	// Strip off any trailing newline.
	if (message_buffer[strlen(message_buffer) - 1] == '\n') {
		message_buffer[strlen(message_buffer) - 1] = '\0';
	}
	cur_info->value.s = EngFncs->engine_strdup(message_buffer);
	cur_info->collection_type = EVMS_Collection_None;
	cur_info->collection.list = NULL;
	cur_info->group.group_number = 0;
	cur_info->group.group_level = 0;
	cur_info->group.group_name = NULL;
	cur_info->flags = 0;

	cur_info++;
	cur_info->name = EngFncs->engine_strdup("level");
	cur_info->title = EngFncs->engine_strdup(_("RAID level"));
	cur_info->desc = EngFncs->engine_strdup(_("RAID level"));
	cur_info->type = EVMS_Type_String;
	cur_info->unit = EVMS_Unit_None;
	cur_info->format = EVMS_Format_Normal;

	switch (level_to_pers(super.level)) {
	case MD_RESERVED:
		cur_info->value.s = EngFncs->engine_strdup(_("Reserved"));
		break;
	case LINEAR:
		cur_info->value.s = EngFncs->engine_strdup(_("Linear"));
		break;
	case RAID0:
		cur_info->value.s = EngFncs->engine_strdup(_("RAID0"));
		break;
	case RAID1:
		cur_info->value.s = EngFncs->engine_strdup(_("RAID1"));
		break;
	case RAID5:
		cur_info->value.s = EngFncs->engine_strdup(_("RAID5"));
		break;
	case TRANSLUCENT:
		cur_info->value.s = EngFncs->engine_strdup(_("Translucent"));
		break;
	case HSM:
		cur_info->value.s = EngFncs->engine_strdup(_("HSM"));
		break;
	case MULTIPATH:
		cur_info->value.s = EngFncs->engine_strdup(_("Multipath"));
		break;
	}
	cur_info->collection_type = EVMS_Collection_None;
	cur_info->collection.list = NULL;
	cur_info->group.group_number = 0;
	cur_info->group.group_level = 0;
	cur_info->group.group_name = NULL;
	cur_info->flags = 0;

	cur_info++;
	cur_info->name = EngFncs->engine_strdup("size");
	cur_info->title = EngFncs->engine_strdup(_("Size"));
	cur_info->desc = EngFncs->engine_strdup(_("Apparent size of each individual disk"));
	cur_info->type = EVMS_Type_Unsigned_Int64;
	cur_info->unit = EVMS_Unit_Kilobytes;
	cur_info->format = EVMS_Format_Normal;
	cur_info->value.ui64 = super.size >> 1;
	cur_info->collection_type = EVMS_Collection_None;
	cur_info->collection.list = NULL;
	cur_info->group.group_number = 0;
	cur_info->group.group_level = 0;
	cur_info->group.group_name = NULL;
	cur_info->flags = 0;

	cur_info++;
	cur_info->name = EngFncs->engine_strdup("nr_disks");
	cur_info->title = EngFncs->engine_strdup(_("Number of disks"));
	cur_info->desc = EngFncs->engine_strdup(_("Total disks in the RAID set"));
	cur_info->type = EVMS_Type_Unsigned_Int32;
	cur_info->unit = EVMS_Unit_None;
	cur_info->format = EVMS_Format_Normal;
	cur_info->value.ui32 = super.nr_disks;
	cur_info->collection_type = EVMS_Collection_None;
	cur_info->collection.list = NULL;
	cur_info->group.group_number = 0;
	cur_info->group.group_level = 0;
	cur_info->group.group_name = NULL;
	cur_info->flags = 0;

	cur_info++;
	cur_info->name = EngFncs->engine_strdup("raid_disks");
	cur_info->title = EngFncs->engine_strdup(_("RAID disks"));
	cur_info->desc = EngFncs->engine_strdup(_("Number of disks in a fully functional RAID set"));
	cur_info->type = EVMS_Type_Unsigned_Int32;
	cur_info->unit = EVMS_Unit_None;
	cur_info->format = EVMS_Format_Normal;
	cur_info->value.ui32 = super.raid_disks;
	cur_info->collection_type = EVMS_Collection_None;
	cur_info->collection.list = NULL;
	cur_info->group.group_number = 0;
	cur_info->group.group_level = 0;
	cur_info->group.group_name = NULL;
	cur_info->flags = 0;

	cur_info++;
	cur_info->name = EngFncs->engine_strdup("md_minor");
	cur_info->title = EngFncs->engine_strdup(_("Minor number"));
	cur_info->desc = EngFncs->engine_strdup(_("Preferred MD minor device number"));
	cur_info->type = EVMS_Type_Unsigned_Int32;
	cur_info->unit = EVMS_Unit_None;
	cur_info->format = EVMS_Format_Normal;
	cur_info->value.ui32 = super.md_minor;
	cur_info->collection_type = EVMS_Collection_None;
	cur_info->collection.list = NULL;
	cur_info->group.group_number = 0;
	cur_info->group.group_level = 0;
	cur_info->group.group_name = NULL;
	cur_info->flags = 0;

	cur_info++;
	cur_info->name = EngFncs->engine_strdup("not_persistent");
	cur_info->title = EngFncs->engine_strdup(_("Persistent superblock"));
	cur_info->desc = EngFncs->engine_strdup(_("Does it have a persistent superblock?"));
	cur_info->type = EVMS_Type_String;
	cur_info->unit = EVMS_Unit_None;
	cur_info->format = EVMS_Format_Normal;
	if (super.not_persistent) {
		cur_info->value.s = EngFncs->engine_strdup(_("No"));
	} else {
		cur_info->value.s = EngFncs->engine_strdup(_("Yes"));
	}
	cur_info->collection_type = EVMS_Collection_None;
	cur_info->collection.list = NULL;
	cur_info->group.group_number = 0;
	cur_info->group.group_level = 0;
	cur_info->group.group_name = NULL;
	cur_info->flags = 0;
	
	cur_info++;
	cur_info->name = EngFncs->engine_strdup("set_uuid1");
	cur_info->title = EngFncs->engine_strdup(_("UUID1"));
	cur_info->desc = EngFncs->engine_strdup(_("Second to lowest 32 bits of the UUID"));
	cur_info->type = EVMS_Type_Unsigned_Int32;
	cur_info->unit = EVMS_Unit_None;
	cur_info->format = EVMS_Format_Hex;
	cur_info->value.ui32 = super.set_uuid1;
	cur_info->collection_type = EVMS_Collection_None;
	cur_info->collection.list = NULL;
	cur_info->group.group_number = 0;
	cur_info->group.group_level = 0;
	cur_info->group.group_name = NULL;
	cur_info->flags = 0;
	
	cur_info++;
	cur_info->name = EngFncs->engine_strdup("set_uuid2");
	cur_info->title = EngFncs->engine_strdup(_("UUID2"));
	cur_info->desc = EngFncs->engine_strdup(_("Second to highest 32 bits of the UUID"));
	cur_info->type = EVMS_Type_Unsigned_Int32;
	cur_info->unit = EVMS_Unit_None;
	cur_info->format = EVMS_Format_Hex;
	cur_info->value.ui32 = super.set_uuid2;
	cur_info->collection_type = EVMS_Collection_None;
	cur_info->collection.list = NULL;
	cur_info->group.group_number = 0;
	cur_info->group.group_level = 0;
	cur_info->group.group_name = NULL;
	cur_info->flags = 0;
	
	cur_info++;
	cur_info->name = EngFncs->engine_strdup("set_uuid3");
	cur_info->title = EngFncs->engine_strdup(_("UUID3"));
	cur_info->desc = EngFncs->engine_strdup(_("Highest 32 bits of the UUID"));
	cur_info->type = EVMS_Type_Unsigned_Int32;
	cur_info->unit = EVMS_Unit_None;
	cur_info->format = EVMS_Format_Hex;
	cur_info->value.ui32 = super.set_uuid3;
	cur_info->collection_type = EVMS_Collection_None;
	cur_info->collection.list = NULL;
	cur_info->group.group_number = 0;
	cur_info->group.group_level = 0;
	cur_info->group.group_name = NULL;
	cur_info->flags = 0;
	
	cur_info++;
	cur_info->name = EngFncs->engine_strdup("utime");
	cur_info->title = EngFncs->engine_strdup(_("Superblock update time"));
	cur_info->desc = EngFncs->engine_strdup(_("Superblock update time"));
	cur_info->type = EVMS_Type_String;
	cur_info->unit = EVMS_Unit_None;
	cur_info->format = EVMS_Format_Normal;
	strcpy(message_buffer, asctime(localtime(&super.utime)));
	// Strip off any trailing newline.
	if (message_buffer[strlen(message_buffer) - 1] == '\n') {
		message_buffer[strlen(message_buffer) - 1] = '\0';
	}
	cur_info->value.s = EngFncs->engine_strdup(message_buffer);
	cur_info->collection_type = EVMS_Collection_None;
	cur_info->collection.list = NULL;
	cur_info->group.group_number = 0;
	cur_info->group.group_level = 0;
	cur_info->group.group_name = NULL;
	cur_info->flags = 0;
	
	cur_info++;
	cur_info->name = EngFncs->engine_strdup("state");
	cur_info->title = EngFncs->engine_strdup(_("State"));
	cur_info->desc = EngFncs->engine_strdup(_("Superblock state flags"));
	cur_info->type = EVMS_Type_String;
	cur_info->unit = EVMS_Unit_None;
	cur_info->format = EVMS_Format_Normal;
	message_buffer[0] = '\0';
	if (super.state_flags & MD_SUPER_INFO_CLEAN) {
		strcat(message_buffer, _("Clean"));
	} else {
		strcat(message_buffer, _("Dirty"));
	}
	if (super.state_flags & MD_SUPER_INFO_ERRORS) {
		if (message_buffer[0] != '\0') {
			strcat(message_buffer, ", ");
		}
		strcat(message_buffer, _("Errors"));
	}
	cur_info->value.s = EngFncs->engine_strdup(message_buffer);
	cur_info->collection_type = EVMS_Collection_None;
	cur_info->collection.list = NULL;
	cur_info->group.group_number = 0;
	cur_info->group.group_level = 0;
	cur_info->group.group_name = NULL;
	cur_info->flags = 0;
	
	cur_info++;
	cur_info->name = EngFncs->engine_strdup("active_disks");
	cur_info->title = EngFncs->engine_strdup(_("Active disks"));
	cur_info->desc = EngFncs->engine_strdup(_("Number of currently active disks"));
	cur_info->type = EVMS_Type_Unsigned_Int32;
	cur_info->unit = EVMS_Unit_None;
	cur_info->format = EVMS_Format_Normal;
	cur_info->value.ui32 = super.active_disks;
	cur_info->collection_type = EVMS_Collection_None;
	cur_info->collection.list = NULL;
	cur_info->group.group_number = 0;
	cur_info->group.group_level = 0;
	cur_info->group.group_name = NULL;
	cur_info->flags = 0;
	
	cur_info++;
	cur_info->name = EngFncs->engine_strdup("working_disks");
	cur_info->title = EngFncs->engine_strdup(_("Working disks"));
	cur_info->desc = EngFncs->engine_strdup(_("Number of working disks"));
	cur_info->type = EVMS_Type_Unsigned_Int32;
	cur_info->unit = EVMS_Unit_None;
	cur_info->format = EVMS_Format_Normal;
	cur_info->value.ui32 = super.working_disks;
	cur_info->collection_type = EVMS_Collection_None;
	cur_info->collection.list = NULL;
	cur_info->group.group_number = 0;
	cur_info->group.group_level = 0;
	cur_info->group.group_name = NULL;
	cur_info->flags = 0;
	cur_info++;

	cur_info->name = EngFncs->engine_strdup("failed_disks");
	cur_info->title = EngFncs->engine_strdup(_("Failed disks"));
	cur_info->desc = EngFncs->engine_strdup(_("Number of failed disks"));
	cur_info->type = EVMS_Type_Unsigned_Int32;
	cur_info->unit = EVMS_Unit_None;
	cur_info->format = EVMS_Format_Normal;
	cur_info->value.ui32 = super.failed_disks;
	cur_info->collection_type = EVMS_Collection_None;
	cur_info->collection.list = NULL;
	cur_info->group.group_number = 0;
	cur_info->group.group_level = 0;
	cur_info->group.group_name = NULL;
	cur_info->flags = 0;
	
	cur_info++;
	cur_info->name = EngFncs->engine_strdup("spare_disks");
	cur_info->title = EngFncs->engine_strdup(_("Spare disks"));
	cur_info->desc = EngFncs->engine_strdup(_("Number of spare disks"));
	cur_info->type = EVMS_Type_Unsigned_Int32;
	cur_info->unit = EVMS_Unit_None;
	cur_info->format = EVMS_Format_Normal;
	cur_info->value.ui32 = super.spare_disks;
	cur_info->collection_type = EVMS_Collection_None;
	cur_info->collection.list = NULL;
	cur_info->group.group_number = 0;
	cur_info->group.group_level = 0;
	cur_info->group.group_name = NULL;
	cur_info->flags = 0;
	
	cur_info++;
	cur_info->name = EngFncs->engine_strdup("sb_csum");
	cur_info->title = EngFncs->engine_strdup(_("Check sum"));
	cur_info->desc = EngFncs->engine_strdup(_("Superblock check sum"));
	cur_info->type = EVMS_Type_Unsigned_Int32;
	cur_info->unit = EVMS_Unit_None;
	cur_info->format = EVMS_Format_Normal;
	cur_info->value.ui32 = super.sb_csum;
	cur_info->collection_type = EVMS_Collection_None;
	cur_info->collection.list = NULL;
	cur_info->group.group_number = 0;
	cur_info->group.group_level = 0;
	cur_info->group.group_name = NULL;
	cur_info->flags = 0;
	
	cur_info++;
	cur_info->name = EngFncs->engine_strdup("events");
	cur_info->title = EngFncs->engine_strdup(_("Update count"));
	cur_info->desc = EngFncs->engine_strdup(_("Superblock update count"));
	cur_info->type = EVMS_Type_Unsigned_Int64;
	cur_info->unit = EVMS_Unit_None;
	cur_info->format = EVMS_Format_Normal;
	cur_info->value.ui64 = super.events;
	cur_info->collection_type = EVMS_Collection_None;
	cur_info->collection.list = NULL;
	cur_info->group.group_number = 0;
	cur_info->group.group_level = 0;
	cur_info->group.group_name = NULL;
	cur_info->flags = 0;
	
	cur_info++;
	cur_info->name = EngFncs->engine_strdup("layout");
	cur_info->title = EngFncs->engine_strdup(_("Layout"));
	cur_info->desc = EngFncs->engine_strdup(_("The physical layout for a RAID5 array"));
	cur_info->type = EVMS_Type_String;
	cur_info->unit = EVMS_Unit_None;
	cur_info->format = EVMS_Format_Normal;
	if (super.level == 5) {
		switch (super.layout) {
		case ALGORITHM_LEFT_ASYMMETRIC:
			cur_info->value.s = EngFncs->engine_strdup(ALGORITHM_LEFT_ASYMMETRIC_NAME);
			break;
		case ALGORITHM_RIGHT_ASYMMETRIC:
			cur_info->value.s = EngFncs->engine_strdup(ALGORITHM_RIGHT_ASYMMETRIC_NAME);
			break;
		case ALGORITHM_LEFT_SYMMETRIC:
			cur_info->value.s = EngFncs->engine_strdup(ALGORITHM_LEFT_SYMMETRIC_NAME);
			break;
		case ALGORITHM_RIGHT_SYMMETRIC:
			cur_info->value.s = EngFncs->engine_strdup(ALGORITHM_RIGHT_SYMMETRIC_NAME);
			break;
		default:
			cur_info->value.s = EngFncs->engine_strdup(_("Unknown"));
			break;
		}
	} else {
		cur_info->value.s = NULL;
	}
	cur_info->collection_type = EVMS_Collection_None;
	cur_info->collection.list = NULL;
	cur_info->group.group_number = 0;
	cur_info->group.group_level = 0;
	cur_info->group.group_name = NULL;
	cur_info->flags = 0;
	
	cur_info++;
	cur_info->name = EngFncs->engine_strdup("chunk_size");
	cur_info->title = EngFncs->engine_strdup(_("Chunk size"));
	cur_info->desc = EngFncs->engine_strdup(_("Chunk size in bytes"));
	cur_info->type = EVMS_Type_Unsigned_Int32;
	cur_info->unit = EVMS_Unit_Bytes;
	cur_info->format = EVMS_Format_Normal;
	cur_info->value.ui32 = super.chunksize << EVMS_VSECTOR_SIZE_SHIFT;
	cur_info->collection_type = EVMS_Collection_None;
	cur_info->collection.list = NULL;
	cur_info->group.group_number = 0;
	cur_info->group.group_level = 0;
	cur_info->group.group_name = NULL;
	cur_info->flags = 0;
	
	cur_info++;
	for (i=0; i<super.nr_disks; i++) {
		int count;
		count = get_superblock_disk_info(vol, i, cur_info);
		cur_info += count;
	}

	cur_info->name = EngFncs->engine_strdup("this_disk");
	cur_info->title = EngFncs->engine_strdup(_("This disk"));
	cur_info->desc = EngFncs->engine_strdup(_("This disk's index in the array"));
	cur_info->type = EVMS_Type_Unsigned_Int32;
	cur_info->unit = EVMS_Unit_None;
	cur_info->format = EVMS_Format_Normal;
	cur_info->value.ui32 = super.this_disk_index;
	cur_info->collection_type = EVMS_Collection_None;
	cur_info->collection.list = NULL;
	cur_info->group.group_number = 0;
	cur_info->group.group_level = 0;
	cur_info->group.group_name = NULL;
	cur_info->flags = 0;

	info->count = info_entries;
	*info_array = info;
	LOG_EXIT_INT(0);
	return 0;
}


int md_get_info(md_volume_t *vol, char *name, extended_info_array_t **info_array)
{
	int rc = 0;
	int idx = -1;
	md_member_t *member;

	if (!name) {
		// Default case. Return all basic info about the region.
		rc = md_get_volume_info(vol, info_array);
	}
	else if ( ! strncmp(name, "child_object", 12) ) {
                // "Extra" information about the child's superblock
		idx = atoi(name + 12);
		member = md_volume_find_member(vol, idx);
		if (member) {
			rc = md_get_child_disk_info(member, info_array);
		} else {
			LOG_ERROR("No support for extra region information about \"%s\"\n", name);
			LOG_EXIT_INT(EINVAL);
			return EINVAL;
		}
	}
	else if ( ! strncmp(name, "stale_object", 12) ) {
                // "Extra" information about the stale disk's superblock
		idx = atoi(name + 12);
		member = md_volume_find_member(vol, idx);
		if (member) {
			rc = md_get_stale_disk_info(member, info_array);
		} else {
			LOG_ERROR("No support for extra region information about \"%s\"\n", name);
			LOG_EXIT_INT(EINVAL);
			return EINVAL;
		}
	}
	else if ( ! strncmp(name, "number", 6) ) {
                // "Extra" information about a disk in the array
		idx = atoi(name + 6);
		if (idx >= 0) {
			rc = md_get_superblock_info(vol, idx, info_array);
		}
		else {
			LOG_ERROR("No support for extra region information about \"%s\"\n", name);
			LOG_EXIT_INT(EINVAL);
			return EINVAL;
		}
	}
	else if ( ! strncmp(name, "superblock", 10) ) {
                // "Extra" information about the master superblock
		rc = md_get_superblock_info(vol, -1, info_array);
	}
	else {
		LOG_ERROR("No support for extra region information about \"%s\"\n", name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

