/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: readable.c
 */

#include <frontend.h>
#include <gtk/gtk.h>
#include "support.h"
#include "logging.h"
#include "readable.h"

#define MB_IN_KB      ((u_int64_t)1024)
#define GB_IN_KB      ((u_int64_t)1024 * MB_IN_KB)
#define TB_IN_KB      ((u_int64_t)1024 * GB_IN_KB)
#define PB_IN_KB      ((u_int64_t)1024 * TB_IN_KB)

/*
 *
 *   inline gchar* make_data_type_readable_string (data_type_t)
 *
 *   Description:
 *      This routine takes a data type code and returns
 *      the string description of the data type.
 * 
 *   Entry:
 *      type - data type enumeration value
 *
 *   Exit:
 *      Returns a pointer to a static string that describes the data type.
 *
 */
inline gchar *make_data_type_readable_string(data_type_t type)
{
	gchar *readable_string = NULL;

	switch (type) {
	case META_DATA_TYPE:
		readable_string = _("Meta Data");
		break;

	case DATA_TYPE:
		readable_string = _("Data");
		break;

	case FREE_SPACE_TYPE:
		readable_string = _("Free Space");
		break;

	default:
		readable_string = _("Unknown");
	}

	return readable_string;
}

/*
 *
 *   inline gchar* make_unit_readable_string (value_unit_t, gboolean)
 *
 *   Description:
 *      This routine takes a value unit type code and returns
 *      the string description of the data type. The caller can 
 *      request a short form if available, i.e. "KB" instead of
 *      "kilobytes".
 * 
 *   Entry:
 *      unit           - unit measurement enumeration value
 *      use_short_form - TRUE if a condensed version is preferred
 *
 *   Exit:
 *      Returns a pointer to a dynamically allocated string that
 *      describes the unit measurement.
 *
 */
inline gchar *make_unit_readable_string(value_unit_t unit, gboolean use_short_form)
{
	gchar *readable_string = NULL;

	switch (unit) {
	case EVMS_Unit_Bytes:
		readable_string = g_strdup(_("bytes"));
		break;

	case EVMS_Unit_Kilobytes:
		if (use_short_form)
			readable_string = g_strdup("KB");
		else
			readable_string = g_strdup(_("kilobytes"));
		break;

	case EVMS_Unit_Megabytes:
		if (use_short_form)
			readable_string = g_strdup("MB");
		else
			readable_string = g_strdup(_("megabytes"));
		break;

	case EVMS_Unit_Gigabytes:
		if (use_short_form)
			readable_string = g_strdup("GB");
		else
			readable_string = g_strdup(_("gigabytes"));
		break;

	case EVMS_Unit_Terabytes:
		if (use_short_form)
			readable_string = g_strdup("TB");
		else
			readable_string = g_strdup(_("terabytes"));
		break;

	case EVMS_Unit_Petabytes:
		if (use_short_form)
			readable_string = g_strdup("PB");
		else
			readable_string = g_strdup(_("petabytes"));
		break;

	case EVMS_Unit_Sectors:
		readable_string = g_strdup(_("sectors"));
		break;

	case EVMS_Unit_Percent:
		if (use_short_form)
			readable_string = g_strdup("%");
		else
			readable_string = g_strdup(_("percent"));
		break;

	default:
		log_error("%s: Unit type %d not handled.\n", __FUNCTION__, unit);
	case EVMS_Unit_None:
		readable_string = NULL;
	}

	return readable_string;
}

/*
 *
 *   inline gchar* make_sectors_readable_size_string (u_int64_t, value_unit_t *)
 *
 *   Description:
 *      This routine takes a size in sectors and returns the
 *      size as a string representation in either TB, GB,
 *      MB, KB, or bytes. The dynamically allocated string 
 *      should be freed by the caller with g_free() when no 
 *      longer needed.
 * 
 *   Entry:
 *      size_in_sectors - the size in 512 byte sector units
 *      unit            - address to write new unit measurement
 *
 *   Exit:
 *      Returns a dynamically allocated string that expresses
 *      the size in a readable form. The new unit is written
 *      to address given by the unit parameter.
 *
 */
inline gchar *make_sectors_readable_size_string(u_int64_t size_in_sectors, value_unit_t * unit)
{
	gchar *readable_string = NULL;
	u_int64_t size_in_kilobytes = size_in_sectors / 2;

	if (size_in_kilobytes >= PB_IN_KB) {
		*unit = EVMS_Unit_Petabytes;
		readable_string = g_strdup_printf("%.1f", (double) size_in_kilobytes / PB_IN_KB);
	} else if (size_in_kilobytes >= TB_IN_KB) {
		*unit = EVMS_Unit_Terabytes;
		readable_string = g_strdup_printf("%.1f", (double) size_in_kilobytes / TB_IN_KB);
	} else if (size_in_kilobytes >= GB_IN_KB) {
		*unit = EVMS_Unit_Gigabytes;
		readable_string = g_strdup_printf("%.1f", (double) size_in_kilobytes / GB_IN_KB);
	} else if (size_in_kilobytes >= MB_IN_KB) {
		*unit = EVMS_Unit_Megabytes;
		readable_string = g_strdup_printf("%.1f", (double) size_in_kilobytes / MB_IN_KB);
	} else if (size_in_sectors >= 2) {
		*unit = EVMS_Unit_Kilobytes;
		readable_string = g_strdup_printf("%" PRIu64, size_in_kilobytes);
	} else {
		*unit = EVMS_Unit_Bytes;
		readable_string = g_strdup_printf("%" PRIu64, size_in_sectors * EVMS_VSECTOR_SIZE);
	}

	return readable_string;
}

/*
 *
 *   inline gchar* make_size_readable_string (gchar *, value_unit_t)
 *
 *   Description:
 *      This routine takes a size string value and generates
 *      a new string with the unit measurement appended to it.
 * 
 *   Entry:
 *      size_string - the size value already converted to a string
 *      unit        - the size's unit measurement
 *
 *   Exit:
 *      Returns a dynamically allocated string that expresses
 *      the size in a readable form.
 *
 */
inline gchar *make_size_readable_string(gchar * size_string, value_unit_t unit)
{
	gchar *readable_string = NULL;

	if (size_string != NULL) {
		gchar *unit_string;

		unit_string = make_unit_readable_string(unit, TRUE);

		readable_string = g_strjoin(" ", size_string, unit_string, NULL);

		g_free(unit_string);
	}

	return readable_string;
}

/*
 *
 *   inline gchar* make_sectors_readable_string (u_int64_t)
 *
 *   Description:
 *      This routine takes a size in sectors and returns the
 *      size as a string representation in either TB, GB,
 *      MB, KB, or bytes. The dynamically allocated string 
 *      should be freed by the caller with g_free() when no 
 *      longer needed.
 * 
 *   Entry:
 *      size_in_sectors - the size in 512 byte sector units
 *
 *   Exit:
 *      Returns a dynamically allocated string that expresses
 *      the size in a readable form.
 *
 */
inline gchar *make_sectors_readable_string(u_int64_t size_in_sectors)
{
	gchar *size_string;
	gchar *readable_string = NULL;
	value_unit_t unit;

	size_string = make_sectors_readable_size_string(size_in_sectors, &unit);

	readable_string = make_size_readable_string(size_string, unit);

	if (readable_string == NULL)
		readable_string = size_string;
	else
		g_free(size_string);

	return readable_string;
}


/*
 *
 *   inline gchar* make_kbytes_readable_string (u_int64_t)
 *
 *   Description:
 *      This routine takes a size in kilobytes and returns the
 *      size as a string representation in either TB, GB,
 *      MB, or KB. The dynamically allocated string 
 *      should be freed by the caller with g_free() when no 
 *      longer needed.
 * 
 *   Entry:
 *      size_in_kbytes - the size in kbytes
 *
 *   Exit:
 *      Returns a dynamically allocated string that expresses
 *      the size in a readable form.
 *
 */
inline gchar *make_kbytes_readable_string(u_int64_t size_in_kbytes)
{
	return make_sectors_readable_string(size_in_kbytes * 2);
}

/*
 *
 *   inline void get_object_and_plugin_names (object_handle_t, gchar **, gchar **)
 *
 *   Description:
 *      This routine takes an object handle and returns
 *      its given name and also the short name of its
 *      plugin. If the handle is for a plugin, the 
 *      object_name will be the long descriptive name.
 * 
 *   Entry:
 *      handle      - handle to the object
 *      object_name - address to store pointer to dynamically allocated object name
 *      plugin_name - address to store pointer to dynamically allocated plugin name
 *
 *   Exit:
 *      Returns dynamically allocated strings that contain the
 *      respective names. The caller should call g_free() to free
 *      the memory once they are no longer needed.
 *
 */
inline void get_object_and_plugin_names(object_handle_t handle, gchar ** object_name,
					gchar ** plugin_name)
{
	gint rc;
	handle_object_info_t *object;

	g_return_if_fail(object_name != NULL);
	g_return_if_fail(plugin_name != NULL);

	*object_name = NULL;
	*plugin_name = NULL;

	rc = evms_get_info(handle, &object);

	if (rc == SUCCESS) {
		plugin_handle_t plugin_handle = 0;

		switch (object->type) {
		case PLUGIN:
			*object_name = g_strdup(object->info.plugin.long_name);
			plugin_handle = handle;
			break;

		case DISK:
		case SEGMENT:
		case REGION:
		case EVMS_OBJECT:
			*object_name = g_strdup(object->info.object.name);
			plugin_handle = object->info.object.plugin;
			break;

		case CONTAINER:
			*object_name = g_strdup(object->info.container.name);
			plugin_handle = object->info.container.plugin;
			break;

		case VOLUME:
			*object_name = g_strdup(object->info.volume.name);
			plugin_handle = object->info.volume.file_system_manager;
			break;

		default:
			log_debug("%s: Unknown object type %d.\n", __FUNCTION__, object->type);
			break;
		}

		*plugin_name = get_plugin_name(plugin_handle);

		evms_free(object);
	}
}

/*
 *
 *   inline gchar *get_object_name (object_handle_t)
 *
 *   Description:
 *      This routine takes an object handle and returns
 *      its given name. If the handle is for a plugin, the 
 *      object_name will be the long descriptive name.
 * 
 *   Entry:
 *      handle - handle to the object
 *
 *   Exit:
 *      Returns a dynamically allocated string that contains
 *      the objects name. The caller should call g_free() to
 *      free the memory once it is no longer needed. NULL
 *      is returned if an error retrieving the name was 
 *      encountered.
 *
 */
inline gchar *get_object_name(object_handle_t handle)
{
	gint rc;
	gchar *object_name = NULL;
	handle_object_info_t *object;

	rc = evms_get_info(handle, &object);

	if (rc == SUCCESS) {
		switch (object->type) {
		case PLUGIN:
			object_name = g_strdup(object->info.plugin.long_name);
			break;

		case DISK:
		case SEGMENT:
		case REGION:
		case EVMS_OBJECT:
			object_name = g_strdup(object->info.object.name);
			break;

		case CONTAINER:
			object_name = g_strdup(object->info.container.name);
			break;

		case VOLUME:
			object_name = g_strdup(object->info.volume.name);
			break;

		default:
			log_debug("%s: Unknown object type %d.\n", __FUNCTION__, object->type);
			break;
		}

		evms_free(object);
	}

	return object_name;
}

/*
 *
 *   inline gchar* make_object_type_readable_string (object_type_t)
 *
 *   Description:
 *      This routine takes a object type code and returns
 *      the string description of the object type.
 * 
 *   Entry:
 *      type - data type enumeration value
 *
 *   Exit:
 *      Returns a static string that describes the object type.
 *
 */
inline gchar *make_object_type_readable_string(object_type_t type)
{
	gchar *readable_string = NULL;

	switch (type) {
	case VOLUME:
		readable_string = _("Logical Volume");
		break;

	case EVMS_OBJECT:
		readable_string = _("Feature Object");
		break;

	case REGION:
		readable_string = _("Storage Region");
		break;

	case CONTAINER:
		readable_string = _("Storage Container");
		break;

	case SEGMENT:
		readable_string = _("Disk Segment");
		break;

	case DISK:
		readable_string = _("Logical Disk");
		break;

	default:
		readable_string = _("Unknown");
	}

	return readable_string;
}


/*
 *
 *   inline gchar* get_volume_fsim_name (logical_volume_info_t *)
 *
 *   Description:
 *      This routine retrieves the shortname of the
 *      plugin identified as the filesystem interface
 *      manager for the volume supplied.
 * 
 *   Entry:
 *      volume - address of the logical_volume_info_t 
 *
 *   Exit:
 *      Returns a dynamically allocated string
 *      containing the shortname of the plugin
 *      or an empty string if no FSIM exists.
 *
 */
inline gchar *get_volume_fsim_name(logical_volume_info_t * volume)
{
	gint rc;
	gchar *plugin_name;
	handle_object_info_t *plugin;

	plugin_name = g_strdup("");

	if (volume->file_system_manager != 0) {
		rc = evms_get_info(volume->file_system_manager, &plugin);

		if (rc != SUCCESS) {
			log_error("%s: evms_get_info() returned error %d.\n", __FUNCTION__, rc);
		} else {
			g_free(plugin_name);
			plugin_name = g_strdup(plugin->info.plugin.short_name);
			evms_free(plugin);
		}
	}

	return plugin_name;
}


/*
 *
 *   inline gchar *get_plugin_name (plugin_handle_t)
 *
 *   Description:
 *      This routine takes an plugin handle and returns
 *      its given short name.
 * 
 *   Entry:
 *      plugin_handle - handle to the plugin
 *
 *   Exit:
 *      Returns a dynamically allocated string that contains
 *      the plugin short name. The caller should call g_free() to
 *      free the memory once it is no longer needed. 
 *
 */
inline gchar *get_plugin_name(plugin_handle_t plugin_handle)
{
	gint rc;
	gchar *plugin_name;

	if (plugin_handle != 0) {
		handle_object_info_t *plugin;

		rc = evms_get_info(plugin_handle, &plugin);

		if (rc == SUCCESS) {
			plugin_name = g_strdup(plugin->info.plugin.short_name);
			evms_free(plugin);
		} else {
			plugin_name = g_strdup(_("Plug-in not available"));
		}
	} else {
		plugin_name = g_strdup(_("Plug-in not available"));
	}

	return plugin_name;
}

/*
 *
 *   inline gchar* make_plugin_type_readable_string (evms_plugin_code_t, gboolean)
 *
 *   Description:
 *      This routine takes a plugin type code and returns
 *      the string description of the plugin type.
 * 
 *   Entry:
 *      type      - the plugin type
 *      lowercase - TRUE if all words in the description should be lowercase
 *
 *   Exit:
 *      Returns a pointer to a static string that describes the plugin type.
 *
 */
inline gchar *make_plugin_type_readable_string(evms_plugin_code_t type, gboolean lowercase)
{
	gchar *readable_string = NULL;

	switch (type) {
	case EVMS_DEVICE_MANAGER:
		if (lowercase)
			readable_string = _("device manager");
		else
			readable_string = _("Device Manager");
		break;

	case EVMS_SEGMENT_MANAGER:
		if (lowercase)
			readable_string = _("segment manager");
		else
			readable_string = _("Segment Manager");
		break;

	case EVMS_REGION_MANAGER:
		if (lowercase)
			readable_string = _("region manager");
		else
			readable_string = _("Region Manager");
		break;

	case EVMS_FEATURE:
		if (lowercase)
			readable_string = _("feature plug-in");
		else
			readable_string = _("Feature Plug-in");
		break;

	case EVMS_ASSOCIATIVE_FEATURE:
		if (lowercase)
			readable_string = _("associative feature");
		else
			readable_string = _("Associative Feature");
		break;

	case EVMS_FILESYSTEM_INTERFACE_MODULE:
		if (lowercase)
			readable_string = _("file system interface module");
		else
			readable_string = _("File System Interface Module");
		break;

	case EVMS_CLUSTER_MANAGER_INTERFACE_MODULE:
		if (lowercase)
			readable_string = _("cluster manager interface module");
		else
			readable_string = _("Cluster Manager Interface Module");
		break;

	case EVMS_DISTRIBUTED_LOCK_MANAGER_INTERFACE_MODULE:
		if (lowercase)
			readable_string = _("distributed lock manager interface module");
		else
			readable_string = _("Distributed Lock Manager Interface Module");
		break;

	default:
		if (lowercase)
			readable_string = _("unknown plug-in");
		else
			readable_string = _("Unknown Plug-in");
	}

	return readable_string;
}

/**
 *	object_flags_to_string - produce a string with words representing object flags
 *	@flags: the bitmap of flags
 *
 *	This routine produces a dynamically allocated string that contains the words
 *	that represent a bit flag that is turned on for a storage object. Each word or
 *	words representing a flag is separated from another by a comma.
 */
gchar *object_flags_to_string(u_int32_t flags)
{
	gchar *string;
	GString *gstring;

	gstring = g_string_new("");

	if (flags & SOFLAG_ACTIVE) {
		if (flags & SOFLAG_NEEDS_ACTIVATE)
			gstring = g_string_append(gstring, _("Re-activation"));
	} else if (flags & SOFLAG_NEEDS_ACTIVATE) {
		gstring = g_string_append(gstring, _("Activation"));
	}

	if (flags & SOFLAG_NEEDS_DEACTIVATE) {
		if (*(gstring->str) != '\0')
			gstring = g_string_append_c(gstring, ',');
		gstring = g_string_append(gstring, _("Deactivation"));
	}

	if ((flags & SOFLAG_DIRTY) || (flags & SOFLAG_FEATURE_HEADER_DIRTY)) {
		if (*(gstring->str) != '\0')
			gstring = g_string_append_c(gstring, ',');
		gstring = g_string_append(gstring, _("Dirty"));
	}

	if (flags & SOFLAG_NEW) {
		if (*(gstring->str) != '\0')
			gstring = g_string_append_c(gstring, ',');
		gstring = g_string_append(gstring, _("New"));
	}

	string = gstring->str;
	g_string_free(gstring, FALSE);

	return string;
}

/**
 *	container_flags_to_string - produce a string with words representing object flags
 *	@flags: the bitmap of flags
 *
 *	This routine produces a dynamically allocated string that contains the words
 *	that represent a bit flag that is turned on for a storage container. Each word or
 *	words representing a flag is separated from another by a comma.
 */
gchar *container_flags_to_string(u_int32_t flags)
{
	gchar *string;
	GString *gstring;

	gstring = g_string_new("");

	if (flags & SCFLAG_DIRTY) {
		gstring = g_string_append(gstring, _("Dirty"));
	}

	if (flags & SCFLAG_NEW) {
		if (*(gstring->str) != '\0')
			gstring = g_string_append_c(gstring, ',');
		gstring = g_string_append(gstring, _("New"));
	}

	string = gstring->str;
	g_string_free(gstring, FALSE);

	return string;
}

/**
 *	volume_flags_to_string - produce a string with words representing object flags
 *	@flags: the bitmap of flags
 *
 *	This routine produces a dynamically allocated string that contains the words
 *	that represent a bit flag that is turned on for a logical volume. Each word or
 *	words representing a flag is separated from another by a comma.
 */
gchar *volume_flags_to_string(u_int32_t flags)
{
	gchar *string;
	GString *gstring;

	gstring = g_string_new("");

	if (flags & VOLFLAG_ACTIVE) {
		if (flags & VOLFLAG_NEEDS_ACTIVATE)
			gstring = g_string_append(gstring, _("Re-activation"));
	} else {
		gstring = g_string_append(gstring, _("Activation"));
	}

	if (flags & VOLFLAG_NEEDS_DEACTIVATE) {
		if (*(gstring->str) != '\0')
			gstring = g_string_append_c(gstring, ',');
		gstring = g_string_append(gstring, _("Deactivation"));
	}

	if (flags & VOLFLAG_DIRTY) {
		if (*(gstring->str) != '\0')
			gstring = g_string_append_c(gstring, ',');
		gstring = g_string_append(gstring, _("Dirty"));
	}

	if (flags & VOLFLAG_NEW) {
		if (*(gstring->str) != '\0')
			gstring = g_string_append_c(gstring, ',');
		gstring = g_string_append(gstring, _("New"));
	}

	if (flags & VOLFLAG_FSCK) {
		if (*(gstring->str) != '\0')
			gstring = g_string_append_c(gstring, ',');
		gstring = g_string_append(gstring, _("Fsck"));
	}

	if (flags & VOLFLAG_MKFS) {
		if (*(gstring->str) != '\0')
			gstring = g_string_append_c(gstring, ',');
		gstring = g_string_append(gstring, _("Mkfs"));
	}

	if (flags & VOLFLAG_UNMKFS) {
		if (*(gstring->str) != '\0')
			gstring = g_string_append_c(gstring, ',');
		gstring = g_string_append(gstring, _("Unmkfs"));
	}

	if (flags & VOLFLAG_DEFRAG) {
		if (*(gstring->str) != '\0')
			gstring = g_string_append_c(gstring, ',');
		gstring = g_string_append(gstring, _("Defragment"));
	}

	if (flags & VOLFLAG_PROBE_FS) {
		if (*(gstring->str) != '\0')
			gstring = g_string_append_c(gstring, ',');
		gstring = g_string_append(gstring, _("Probe Filesystem"));
	}

	if (flags & VOLFLAG_SYNC_FS) {
		if (*(gstring->str) != '\0')
			gstring = g_string_append_c(gstring, ',');
		gstring = g_string_append(gstring, _("Sync Filesystem"));
	}

	string = gstring->str;
	g_string_free(gstring, FALSE);

	return string;
}

/**
 *	changes_flag_to_string - produce a string with words representing reasons for a pending change
 *	@flags: the bitmap of changes pending reason flags
 *
 *	This routine produces a dynamically allocated string that contains the words
 *	that represent a bit flag that is turned on for reason for a change pending flag.
 *	Each word or words representing a flag is separated from another by a comma.
 */
gchar *changes_flag_to_string(u_int32_t flags)
{
	gchar *string;
	GString *gstring;

	gstring = g_string_new("");

	if (flags & CHANGE_ACTIVATE) {
		gstring = g_string_append(gstring, _("Activation"));
	}

	if (flags & CHANGE_DEACTIVATE) {
		if (*(gstring->str) != '\0')
			gstring = g_string_append_c(gstring, ',');
		gstring = g_string_append(gstring, _("Deactivation"));
	}

	if (flags & CHANGE_REACTIVATE) {
		if (*(gstring->str) != '\0')
			gstring = g_string_append_c(gstring, ',');
		gstring = g_string_append(gstring, _("Re-activation"));
	}

	if (flags & CHANGE_DIRTY) {
		if (*(gstring->str) != '\0')
			gstring = g_string_append_c(gstring, ',');
		gstring = g_string_append(gstring, _("Dirty"));
	}

	if (flags & CHANGE_FEATURE_HEADER_DIRTY) {
		if (*(gstring->str) != '\0')
			gstring = g_string_append_c(gstring, ',');
		gstring = g_string_append(gstring, _("Feature Header Dirty"));
	}

	if (flags & CHANGE_EXPAND) {
		if (*(gstring->str) != '\0')
			gstring = g_string_append_c(gstring, ',');
		gstring = g_string_append(gstring, _("Expand"));
	}

	if (flags & CHANGE_SHRINK) {
		if (*(gstring->str) != '\0')
			gstring = g_string_append_c(gstring, ',');
		gstring = g_string_append(gstring, _("Shrink"));
	}

	if (flags & CHANGE_FSCK) {
		if (*(gstring->str) != '\0')
			gstring = g_string_append_c(gstring, ',');
		gstring = g_string_append(gstring, _("Fsck"));
	}

	if (flags & CHANGE_MKFS) {
		if (*(gstring->str) != '\0')
			gstring = g_string_append_c(gstring, ',');
		gstring = g_string_append(gstring, _("Mkfs"));
	}

	if (flags & CHANGE_UNMKFS) {
		if (*(gstring->str) != '\0')
			gstring = g_string_append_c(gstring, ',');
		gstring = g_string_append(gstring, _("Unmkfs"));
	}

	if (flags & CHANGE_DEFRAG) {
		if (*(gstring->str) != '\0')
			gstring = g_string_append_c(gstring, ',');
		gstring = g_string_append(gstring, _("Defragment"));
	}

	if (flags & CHANGE_DELETE) {
		if (*(gstring->str) != '\0')
			gstring = g_string_append_c(gstring, ',');
		gstring = g_string_append(gstring, _("Deletion"));
	}

	if (flags & CHANGE_KILL_SECTORS) {
		if (*(gstring->str) != '\0')
			gstring = g_string_append_c(gstring, ',');
		gstring = g_string_append(gstring, _("Zero-fill sectors"));
	}

	string = gstring->str;
	g_string_free(gstring, FALSE);

	return string;
}
