# (C) Copyright IBM Corp. 2004
#
# This program is free software;  you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY;  without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
# the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program;  if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
#
# Perl module with common subroutines for managing MD regions.
                                                                                
package Evms::MD;
                                                                                
use strict;
use warnings;

use Evms::Common;
use Evms::Object;
                                                                                
BEGIN {
        use Exporter();
        our (@ISA, @EXPORT);
        @ISA = qw(Exporter);
        @EXPORT = qw(
		&create_md_region
		&delete_md_region
		&add_spare_to_md_region
		&remove_spare_from_md_region
		&add_active_to_md_region
		&remove_active_from_md_region
		&query_md_region
		&query_md_region_children
		&search_md_proc_stat
		);
}

# create_md_region
#	Create an MD region
#
#	PARAMETERS (4)
#	- region manager name (e.g. MDRaid1RegMgr)
#	- array of raid disks
#	- array of spares
#	- Optional: (output) the created region name.  Blank string indicates error.
#	RETURN:
#	error code if any
#
sub create_md_region($$$;\$)
{
	my $region_mgr = shift;
	my $raid_disks = shift;
	my $spare_disks = shift;
	my $region_name = shift;
	my ($command, $option, $rc);
	my @output;

	$command = "Create:Region,$region_mgr=";
	if (ref($spare_disks) && scalar(@{$spare_disks})) {
		$option = "{sparedisk=";
		$option .= join ",",@{$spare_disks};
		$option .= "},";
	} else {
		$option = "{},";
	}
	$command .= $option;
	$command .= join ",",@{$raid_disks};

	$rc = run_evms_command($command, \@output);
	if ($rc) {
		if (defined($region_name)) {
			$region_name = "";
		}
	}

	if (!defined($region_name)) {
		goto out;
	}

	# evms returns many lines of output...
	foreach (@output) {
		if ($_ =~ /created region/) {
			my @list;
			# We have found the line having "created region",
			# split it into tokens
			@list = split /\s* \s*/, $_;
			# remove trailing newline if any.
			chomp $list[-1]; 
			${$region_name} = $list[-1];
			goto out;
		}
	}
out:
	return $rc;
}

# delete_md_region
#	Delete an MD region
#
#	PARAMETERS (1)
#	- region name (e.g. md/md0)
#	RETURN:
#	- error code if any
sub delete_md_region
{
	my $region_name = shift;
	my $command;

	return undef unless defined $region_name;
	$command = "Delete:$region_name";
	return run_evms_command($command);
}

# add_spare_to_md_region
#	Add spare disks to the specified MD region.
#
#	PARAMETERS (2)
#	- region name (e.g. md/md0)
#	- array of disks.  
#	Note that MD RAID1/RAID5 plugin will add only spare at a time.
#	We use array of spare disks to test error scenarios.
#
#	RETURN:
#	- error code if any
sub add_spare_to_md_region
{
	my $region_name = shift;
	my $spare_disks = shift;
	my ($command);

	$command = "Task:addspare,$region_name,";
	$command .= join ",",@{$spare_disks};
	return  run_evms_command($command);
}


# remove_spare_from_md_region
#	Remove spare disks from the specified MD region.
#
#	PARAMETERS (2)
#	- region name (e.g. md/md0)
#	- array of disks
#
#	RETURN:
#	- error code if any
sub remove_spare_from_md_region
{
	my $region_name = shift;
	my $spare_disks = shift;
	my ($command);

	$command = "Task:remspare,$region_name,";
	$command .= join ",",@{$spare_disks};
	return  run_evms_command($command);
}

# add_active_to_md_region
#	Add active disks to the specified MD region.
#
#	PARAMETERS (2)
#	- region name (e.g. md/md0)
#	- array of disks.  
#	Note that MD RAID1 plugin will add only 1 disk at a time.
#	We use array of disks to test error scenarios.
#
#	RETURN:
#	- error code if any
sub add_active_to_md_region
{
	my $region_name = shift;
	my $active_disks = shift;
	my ($command);

	$command = "Task:addactive,$region_name,";
	$command .= join ",",@{$active_disks};
	return  run_evms_command($command);
}

# remove_active_from_md_region
#	remove active disks from the specified MD region.
#
#	PARAMETERS (2)
#	- region name (e.g. md/md0)
#	- array of disks.  
#	Note that MD RAID1 plugin will remove only 1 disk at a time.
#	We use array of disks to test error scenarios.
#
#	RETURN:
#	- error code if any
sub remove_active_from_md_region
{
	my $region_name = shift;
	my $active_disks = shift;
	my ($command);

	$command = "Task:remactive,$region_name,";
	$command .= join ",",@{$active_disks};
	return  run_evms_command($command);
}


# query_md_region
#	Query an MD region.  To determine whether the MD region exists or not,
#	call this subroutine with the region name and check the return code.
#
#	PARAMETERS (2)
#	- region name (e.g. md/md0)
#	- Optional: (output) array to hold output from evms command.
#	RETURN:
#	- error code if any
sub query_md_region($;$)
{
	my $region_name = shift;
	my $user_output = shift;
	my $command;
	my @output;
	my $rc;

	return undef unless defined $region_name;
	$command = "Query:Regions,Region=$region_name";
	$rc =  run_evms_command($command,\@output);
	if (defined($user_output)) {
		@{$user_output} = @output;
	}
	return $rc;
}

# query_md_region_children
#	Query all children of the specified MD region
#
#	PARAMETERS (2)
#	- region name (e.g. md/md0)
#	- Optional: (output) array to hold output from evms command.
#	RETURN:
#	- error code if any
sub query_md_region_children($;$)
{
	my $region_name = shift;
	my $user_output = shift;
	my $command;
	my @output;
	my $rc;

	return undef unless defined $region_name;
	$command = "Query:Children,$region_name";
	$rc =  run_evms_command($command,\@output);
	if (defined($user_output)) {
		foreach (@output) {
			chomp($_);
			next if (!length($_));
			push @{$user_output}, $_;
		}
	}
	return $rc;
}

# search_md_proc_status
#	Search in /proc/mdstat for the specified string
#
#	PARAMETERS (1)
#	- string to search
#	RETURN:
#	- First line contains the string. If not found, return blank string.
sub search_md_proc_status
{
	my $input = $_[0];
	my @lines;
	my @matches;
	my $result = "";

	open MYHANDLE, "/proc/mdstat" or 
		log_error( "Unable to open: $!.  Please enable /proc FS and MD in the kernel.\n");

	# Since we don't expect many lines in /proc/mdstat,
	# read all the lines at once.
	@lines = <MYHANDLE>;
	close MYHANDLE;

	@matches = grep(/$input/, @lines);
	foreach (@matches) {
		$result = $_;
	}
	return $result;
}

1;

