/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libs390.so
 *
 *   File: format.c
 */
/*
 *  Special thanks to dasdfmt and fdasd code.
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <plugin.h>
#include <ldm_funcs.h>

#include "vtoc.h"
#include "390segmgr.h"
#include "helpers.h"
#include "format.h"

#define BLKGETSIZE64 _IOR(0x12,114,sizeof(u_int64_t))
#define BLKSSZGET    _IO(0x12,104)
#define HDIO_GETGEO  0x0301


typedef struct bootstrap1 {
	u_int32_t key;
	u_int32_t data[6];
} __attribute__ ((packed)) bootstrap1_t;

typedef struct bootstrap2 {
	u_int32_t key;
	u_int32_t data[36];
} __attribute__ ((packed)) bootstrap2_t;


/*
C9D7D3F1 000A0000 0000000F 03000000  00000001 00000000 00000000
*/
bootstrap1_t ipl1 =
{
	0xC9D7D3F1,
	{
		0x000A0000, 0x0000000F, 0x03000000,
		0x00000001, 0x00000000, 0x00000000
	}
};

/*
C9D7D3F2 07003AB8 40000006 31003ABE  40000005 08003AA0 00000000 06000000
20000000 00000000 00000000 00000400  00000000 00000000 00000000 00000000
00000000 00000000 00000000 00000000  00000000 00000000 00000000 00000000
00000000 00000000 00000000 00000000  00000000 00000000 00000000 00000000
00000000 00000000 00000000 00000000  00000000
*/
bootstrap2_t ipl2 =
{
	0xC9D7D3F2,
	{
		0x07003AB8, 0x40000006, 0x31003ABE,
		0x40000005, 0x08003AA0, 0x00000000,
		0x06000000, 0x20000000, 0x00000000,
		0x00000000, 0x00000400, 0x00000000,
		0x00000000, 0x00000000, 0x00000000,
		0x00000000, 0x00000000, 0x00000000,
		0x00000000, 0x00000000, 0x00000000,
		0x00000000, 0x00000000, 0x00000000,
		0x00000000, 0x00000000, 0x00000000,
		0x00000000, 0x00000000, 0x00000000,
		0x00000000, 0x00000000, 0x00000000,
		0x00000000, 0x00000000, 0x00000000
	}
};


static int ldm_close( LOGICALDISK *ld )
{
	int rc;

	LOG_ENTRY();

	rc = ld->plugin->functions.plugin->plugin_function(ld, LDM_Close_Disk, NULL, NULL);

	LOG_EXIT_INT(rc);
	return rc;
}

static int ldm_open( LOGICALDISK *ld )
{
	int rc;

	LOG_ENTRY();

	rc = ld->plugin->functions.plugin->plugin_function(ld, LDM_Open_Disk, NULL, NULL);

	LOG_EXIT_INT(rc);
	return rc;
}

/*
 *  Called to issue ioctl cmd to evms logical disk when the
 *  in-use count doesn't matter.
 */
static int disk_ioctl( LOGICALDISK *ld, int cmd, void *arg )
{
	int rc = EINVAL;
	int fd;

	LOG_ENTRY();

	fd = EngFncs->open_object(ld, O_RDWR);
	if (fd >= 0) {

		rc = EngFncs->ioctl_object(ld,fd,cmd,arg);
		if (rc) {
			LOG_ERROR("ioctl failed with rc= %d  errno= %d\n", rc, errno);
			rc = errno;
		}

		EngFncs->close_object(ld,fd);
	}
	else {
		rc=errno;
		LOG_ERROR("error, open failed with rc = %d\n", rc );
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Called to update logical disk information, following
 *  a format operation which may have caused geometry
 *  changes.
 */
static int update_logical_disk_info( LOGICALDISK *ld, int fd )
{
	int                rc;
	sector_count_t     size;
	u_int32_t          hardsector_size;
	struct hd_geometry geo;
	DISK_PRIVATE_DATA *disk_pdata=get_s390_disk_private_data(ld);

	LOG_ENTRY();

#ifdef BLKGETSIZE64
	rc = EngFncs->ioctl_object(ld,fd,BLKGETSIZE64,&size);
#else
	rc = EngFncs->ioctl_object(ld,fd,BLKGETSIZE,&size);
#endif
	if (!rc) {
		size >>= EVMS_VSECTOR_SIZE_SHIFT;

		rc = EngFncs->ioctl_object(ld,fd,HDIO_GETGEO,&geo);
		if (!rc) {

			rc = EngFncs->ioctl_object(ld,fd,BLKSSZGET,&hardsector_size);
			if (!rc) {
				size >>= EVMS_VSECTOR_SIZE_SHIFT;
			}

		}

	}

	// Note ... a valid disk object starts at 0 and has some size
	if (!rc && geo.start == 0 && size > 0) {
		ld->geometry.cylinders           = geo.cylinders;
		ld->geometry.heads               = geo.heads;
		ld->geometry.sectors_per_track   = geo.sectors;

		ld->geometry.bytes_per_sector    = hardsector_size;
		//      ld->geometry.block_size          = blksize;
		ld->size                         = size;

		disk_pdata->vsectors_per_block   = hardsector_size >> EVMS_VSECTOR_SIZE_SHIFT;
		disk_pdata->geometry.cylinders   = geo.cylinders;
		disk_pdata->geometry.heads       = geo.heads;
		disk_pdata->geometry.sectors     = geo.sectors;
		disk_pdata->geometry.start       = 0;

		LOG_DEBUG("Latest Kernel Disk Info ...\n");
		LOG_DEBUG("  Cylinders  = %"PRIu64"\n", ld->geometry.cylinders);
		LOG_DEBUG("  Heads      = %d\n", ld->geometry.heads);
		LOG_DEBUG("  Sectors    = %d\n", ld->geometry.sectors_per_track);
		LOG_DEBUG("  BlockSize  = %"PRIu64"\n", ld->geometry.block_size );
		LOG_DEBUG("  SectorSize = %d\n", ld->geometry.bytes_per_sector );
		LOG_DEBUG("  TotalSize  = %"PRIu64"\n", ld->size );
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Wrapper routine for BIODASDINFO ioctl
 */
int get_390_dev_info( LOGICALDISK *ld, dasd_information_t *dasd_info )
{
	int rc;

	LOG_ENTRY();

	rc = disk_ioctl(ld,BIODASDINFO, (void *)dasd_info);
	if (rc) {
		LOG_ERROR("error, call to the dasd driver for info about device %s failed.\n", ld->name );
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Called to low level format a device
 */
static int llformat( LOGICALDISK *ld, int blocksize,  int intensity )
{
	format_data_t      format_packet;
	int                stop_track = (ld->geometry.cylinders * ld->geometry.heads) - 1;
	int                track = 1;
	int                rc = 0;
	int                cylinder_count = ld->geometry.cylinders;
	int                i;
	progress_t         progress;
	int                use_progress;
	char               title[EVMS_VOLUME_NAME_SIZE+1];
	boolean            quarter_way = FALSE;
	boolean            half_way = FALSE;
	boolean            three_quarters = FALSE;
	DISK_PRIVATE_DATA *disk_pdata=get_s390_disk_private_data(ld);
	int                 fd;

	LOG_ENTRY();
	LOG_DEBUG(" device     = %s\n", ld->name );
	LOG_DEBUG(" blocksize  = %d\n", blocksize );
	LOG_DEBUG(" intensity  = %d\n", intensity);
	LOG_DEBUG(" stop track = %d\n", stop_track );

	// close ldm fildes and reopen the device
	ldm_close(ld);

	fd = EngFncs->open_object(ld, O_RDWR);
	if (fd < 0) {
		rc = errno;
		ldm_open(ld);
		LOG_ERROR("error, failed to get a fildes\n");
		LOG_EXIT_INT(rc);
		return rc;
	}

	// disable the disk
	format_packet.start_unit = 0;
	format_packet.stop_unit  = stop_track;
	format_packet.blksize    = blocksize;
	format_packet.intensity  = intensity;

	rc = EngFncs->ioctl_object(ld,fd, BIODASDDISABLE, (void *) &format_packet );
	if (rc) {
		EngFncs->close_object(ld,fd);
		ldm_open(ld);
		LOG_ERROR("error, unable to disable disk %s.\n", ld->name);
		LOG_EXIT_INT(rc);
		return rc;
	}

	// invalidate 1st track
	format_packet.start_unit = 0;
	format_packet.stop_unit  = 0;
	format_packet.blksize    = blocksize;
	format_packet.intensity  = intensity | DASD_FMT_INT_INVAL;

	rc = EngFncs->ioctl_object(ld,fd, BIODASDFMT, (void *) &format_packet );
	if (rc) {
		EngFncs->close_object(ld,fd);
		ldm_open(ld);
		LOG_ERROR("error, unable to invalidate 1st track\n");
		LOG_EXIT_INT(rc);
		return rc;
	}

	// setup evms progress indicator
	sprintf(title, "Formatting %d cylinders on disk ... %s\n", cylinder_count, ld->name );
	progress.total_count = cylinder_count;
	progress.id = 0;
	progress.title = title;
	progress.description = "";
	progress.type = DISPLAY_PERCENT;
	progress.count = 0;
	progress.remaining_seconds = 0;
	progress.plugin_private_data = NULL;
	progress.ui_private_data = NULL;

	use_progress = EngFncs->progress(&progress) == 0;

	if (!use_progress) {
		MESSAGE("FORMATTING %d CYLINDERS ... \n", cylinder_count);
	}

	// format 1 cylinder at a time
	format_packet.blksize    = blocksize;
	format_packet.intensity  = intensity;
	i = 0;
	while ( (rc==0) && (track < stop_track) ) {

		format_packet.start_unit = track;
		format_packet.stop_unit  = track + ld->geometry.heads - 1;

		//  LOG_DEBUG("start = %d   stop = %d\n", format_packet.start_unit, format_packet.stop_unit );

		rc = EngFncs->ioctl_object(ld,fd, BIODASDFMT, (void *) &format_packet);
		if (rc) {
			MESSAGE("error, dasd driver returned error (%s) while formating tracks.\n", strerror(errno) );
		}
		else {

			++i;

			if ( use_progress ) {
				progress.count = i;
				EngFncs->progress(&progress);
			}
			else {
				if ( i >= (cylinder_count/4) && quarter_way == FALSE ) {
					MESSAGE(".... 1/4 done\n");
					quarter_way = TRUE;
				}
				else if ( i >= (cylinder_count/2) && half_way == FALSE ) {
					MESSAGE(".... 1/2 done\n");
					half_way = TRUE;
				}
				else if ( i >= (cylinder_count-(cylinder_count/4)) && three_quarters == FALSE ) {
					MESSAGE(".... 3/4 done\n");
					three_quarters = TRUE;
				}
			}

			if ( track % ld->geometry.heads == 0) {
				track += ld->geometry.heads;
			}
			else {
				track += stop_track % ld->geometry.heads;
			}

		}

	}

	if (use_progress) {
		progress.title = "Finished.";
		progress.count = progress.total_count;
		EngFncs->progress(&progress);
	}
	else {
		MESSAGE("FINISHED!\n\n");
	}


	// revalidate 1st track
	format_packet.start_unit = 0;
	format_packet.stop_unit  = 0;
	format_packet.blksize    = blocksize;
	format_packet.intensity  = intensity;

	rc = EngFncs->ioctl_object(ld,fd, BIODASDFMT, (void *) &format_packet );
	if (rc) {
		LOG_ERROR("error, revalidate track ioctl failed with error (%s)\n", strerror(errno));
	}

	// enable the disk
	format_packet.start_unit = 0;
	format_packet.stop_unit  = stop_track;
	format_packet.blksize    = blocksize;
	format_packet.intensity  = intensity;

	rc = EngFncs->ioctl_object(ld,fd, BIODASDENABLE, (void *) &format_packet );
	if (rc) {
		LOG_ERROR("error, dasd driver cant re-enable disk %s.\n", ld->name);
	}

	// update disk geometry due to any changes the format may have caused
	update_logical_disk_info(ld, fd);

	// update cdl and ldl info
	if (intensity == DASD_FMT_INT_COMPAT ) {
		disk_pdata->vtoc_record_count      = ld->geometry.sectors_per_track;
		disk_pdata->vtoc_sector_count      = disk_pdata->vtoc_record_count * disk_pdata->vsectors_per_block;
		disk_pdata->max_partitions_allowed = disk_pdata->vtoc_record_count - 2;
		disk_pdata->partition_count        = 0;
		disk_pdata->vtoc_lba               = ld->geometry.sectors_per_track * disk_pdata->vsectors_per_block;
	}
	else {
		disk_pdata->partition_count = 1;
		disk_pdata->max_partitions_allowed = 1;
	}

	EngFncs->close_object(ld,fd);
	ldm_open(ld);

	rc = 0;
	LOG_EXIT_INT(rc);
	return rc;
}




/*
 *  Called to write the 1st IPL record to the specified disk.
 */
static int write_ipl_record_1( LOGICALDISK *ld )
{
	int rc;
	char  ipl_record[EVMS_VSECTOR_SIZE];

	LOG_ENTRY();

	rc = READ( ld, IPL1_RECORD_BLOCK_NUMBER, 1, (void *) ipl_record );
	if (rc ==0) {
		memcpy( ipl_record, &ipl1, sizeof(bootstrap1_t) );
		rc = WRITE( ld, IPL1_RECORD_BLOCK_NUMBER, 1, (void *) ipl_record );
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Called to write the 2nd IPL record to the specified disk.
 */
static int write_ipl_record_2( LOGICALDISK *ld )
{
	int rc;
	char  ipl_record[EVMS_VSECTOR_SIZE];
	int blocksize;

	LOG_ENTRY();

	blocksize = ld->geometry.bytes_per_sector >> EVMS_VSECTOR_SIZE_SHIFT;

	rc = READ( ld, IPL2_RECORD_BLOCK_NUMBER*blocksize, 1, (void *) ipl_record );
	if (rc ==0) {
		memcpy( ipl_record, &ipl2, sizeof(bootstrap2_t) );
		rc = WRITE( ld, IPL2_RECORD_BLOCK_NUMBER*blocksize, 1, (void *) ipl_record );
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Called to write the volume label record to the specified disk.
 */
static int write_volume_label_record( LOGICALDISK *ld, boolean compatibility_disk, int devno  )
{
	int rc=EINVAL;
	int blocksize;
	char vlabel_record[EVMS_VSECTOR_SIZE];
	DISK_PRIVATE_DATA *disk_pdata = get_s390_disk_private_data( ld );
	char  volid[VOLSER_LENGTH+1];

	LOG_ENTRY();

	if (disk_pdata) {

		blocksize = ld->geometry.bytes_per_sector >> EVMS_VSECTOR_SIZE_SHIFT;

		// fill with ebcdic spaces
		memset( &disk_pdata->vlabel, 0x40, sizeof(volume_label_t) );

		if ( compatibility_disk ) {
			// use virtual device address as volume id
			sprintf(volid, "0X%04X", devno);
			memcpy(&disk_pdata->vlabel.volid, volid, VOLSER_LENGTH );
			atoe((char *)&disk_pdata->vlabel.volid, VOLSER_LENGTH);

			memcpy( &disk_pdata->vlabel.volkey, (&(char []){0xe5,0xd6,0xd3,0xf1}), 4);  // VOL1
			memcpy( &disk_pdata->vlabel.vollbl, (&(char []){0xe5,0xd6,0xd3,0xf1}), 4);  // VOL1
			disk_pdata->vlabel.vtoc.cc = 0;
			disk_pdata->vlabel.vtoc.hh = 1;
			disk_pdata->vlabel.vtoc.b  = 1;

			// update because the geometry may have changed during the low level format operation.
			disk_pdata->vtoc_record_count      = ld->geometry.sectors_per_track;
			disk_pdata->vtoc_sector_count      = disk_pdata->vtoc_record_count * disk_pdata->vsectors_per_block;
			disk_pdata->max_partitions_allowed = disk_pdata->vtoc_record_count - 2;
			disk_pdata->vtoc_lba               = ld->geometry.sectors_per_track * blocksize;

		}
		else {
			memcpy( &disk_pdata->vlabel.volkey, (&(char []){0xd3,0xd5,0xe7,0xf1}), 4);  // LNX1

			sprintf(volid, "0X%04X", devno);
			memcpy(&disk_pdata->vlabel.vollbl, volid, VOLSER_LENGTH );
			atoe((char *)&disk_pdata->vlabel.vollbl, VOLSER_LENGTH);

		}

		rc = READ( ld, VOLUME_LABEL_BLOCK_NUMBER*blocksize, 1, (void *) vlabel_record );
		if (rc ==0) {

			memcpy( vlabel_record, &disk_pdata->vlabel, sizeof(volume_label_t) );

			rc = WRITE( ld, VOLUME_LABEL_BLOCK_NUMBER*blocksize, 1, (void *) vlabel_record );

		}

	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Called to write default VTOC records to the specified disk. This means
 *  writing a Type 4 (vtoc descriptor) and a Type 5 (freespace descriptor)
 *  to the VTOC.
 */
static int write_vtoc( LOGICALDISK *ld, dasd_information_t *devinfo )
{
	int rc=EINVAL;
	int blocksize;
	char  vtoc_record[EVMS_VSECTOR_SIZE];
	format4_label_t  *f4 = (format4_label_t *)vtoc_record;
	format5_label_t  *f5 = (format5_label_t *)vtoc_record;
	DISK_PRIVATE_DATA *disk_pdata = get_s390_disk_private_data( ld );


	LOG_ENTRY();

	if (disk_pdata) {

		blocksize = ld->geometry.bytes_per_sector >> EVMS_VSECTOR_SIZE_SHIFT;

		rc = READ( ld, disk_pdata->vtoc_lba, 1, (void *) vtoc_record );
		if (rc ==0) {

			memset( f4, 0, sizeof(format4_label_t) );

			memset( &f4->DS4KEYCD, 0x04, 44 );

			f4->DS4IDFMT           = 0xf4;
			f4->DS4HPCHR.cc        = 0;
			f4->DS4HPCHR.hh        = 1;
			f4->DS4HPCHR.b         = 3;
			f4->DS4DSREC           = ld->geometry.sectors_per_track - 2;
			f4->DS4NOATK           = 0x0000;
			f4->DS4VTOCI           = 0x00;
			f4->DS4NOEXT           = 0x01;
			f4->DS4SMSFG           = 0x00;
			f4->DS4DEVAC           = 0x00;

			switch (devinfo->dev_type) {
			case DASD_3380_TYPE:
				f4->DS4DEVCT.DS4DEVTK = DASD_3380_VALUE;
				break;
			case DASD_3390_TYPE:
				f4->DS4DEVCT.DS4DEVTK = DASD_3390_VALUE;
				break;
			case DASD_9345_TYPE:
				f4->DS4DEVCT.DS4DEVTK = DASD_9345_VALUE;
				break;
			default:
				f4->DS4DEVCT.DS4DEVTK = ld->geometry.sectors_per_track * blocksize;
			}

			f4->DS4DEVCT.DS4DSCYL  = ld->geometry.cylinders;
			f4->DS4DEVCT.DS4DSTRK  = ld->geometry.heads;
			f4->DS4DEVCT.DS4DEVI   = 0x00;
			f4->DS4DEVCT.DS4DEVL   = 0x00;
			f4->DS4DEVCT.DS4DEVK   = 0x00;
			f4->DS4DEVCT.DS4DEVFG  = 0x30;
			f4->DS4DEVCT.DS4DEVTL  = 0x0000;
			f4->DS4DEVCT.DS4DEVDT  = ld->geometry.sectors_per_track;
			f4->DS4DEVCT.DS4DEVDB  = 0x00;

			f4->DS4VTOCE.typeind   = 1;
			f4->DS4VTOCE.seqno     = 0;
			f4->DS4VTOCE.llimit.cc = 0;
			f4->DS4VTOCE.llimit.hh = 1;
			f4->DS4VTOCE.ulimit.cc = 0;
			f4->DS4VTOCE.ulimit.hh = 1;

			rc = WRITE( ld, disk_pdata->vtoc_lba, 1, (void *) f4 );
			if (rc == 0) {

				memcpy( &disk_pdata->f4, f4, sizeof(format4_label_t) );

				rc = READ( ld, disk_pdata->vtoc_lba + blocksize, 1, (void *) vtoc_record );
				if (rc ==0) {

					u_int64_t   size=0;
					u_int16_t   fc=0;
					u_int16_t   ft=0;

					memset(f5, 0, sizeof(format5_label_t));
					memset(&f5->DS5KEYID, 0x05, 4 );
					f5->DS5FMTID = 0xF5;

					size  = ( ld->size - (2*ld->geometry.sectors_per_track*blocksize) ) / blocksize;
					fc    = size / (ld->geometry.heads*ld->geometry.sectors_per_track);
					ft    = size % (ld->geometry.heads*ld->geometry.sectors_per_track);
					ft    = ft / ld->geometry.sectors_per_track;

					f5->DS5AVEXT.t  = 3;	  // real track addr
					f5->DS5AVEXT.fc = fc;	  // full cylinders
					f5->DS5AVEXT.ft = ft;	  // full tracks

					rc = WRITE( ld, disk_pdata->vtoc_lba + blocksize, 1, (void *) vtoc_record );

				}

			}

		}

	}

	LOG_EXIT_INT(rc);
	return rc;
}



/*
 *  Called by format code to create a METADATA segment on the specified disk.
 */
static DISKSEG * create_metadata_segment( LOGICALDISK    *ld,
					  lba_t           lba,
					  sector_count_t  size )
{
	DISKSEG  *md=NULL;

	LOG_ENTRY();

	md = allocate_s390_disk_segment( ld );
	if (md) {

		md->size        = size;
		md->start       = lba;
		md->data_type   = META_DATA_TYPE;

		if ( insert_s390_segment_into_list( ld->parent_objects, md ) == NULL ) {
			free_s390_disk_segment(md);
			md = NULL;
		}

	}

	LOG_EXIT_PTR(md);
	return md;
}


/*
 *  Called by format code to create a DATA segment on the specified disk.
 */
static DISKSEG * create_data_segment( LOGICALDISK    *ld,
				      lba_t           lba,
				      sector_count_t  size )

{
	DISKSEG  *seg=NULL;

	LOG_ENTRY();

	seg = allocate_s390_disk_segment( ld );

	if (seg) {

		seg->start = lba;
		seg->size  = size;
		seg->data_type = DATA_TYPE;

		((SEG_PRIVATE_DATA *)seg->private_data)->minor = 1;

		if ( insert_s390_segment_into_list( ld->parent_objects, seg ) == NULL ) {
			free_s390_disk_segment(seg);
			seg = NULL;
		}

	}

	LOG_EXIT_PTR(seg);
	return seg;
}


/*
 *  Called to format the specified disk in CDL format.
 */
static int format_cdl_disk( LOGICALDISK *ld, int blocksize, dasd_information_t *devinfo )
{
	int                rc=EINVAL;
	DISKSEG           *md=NULL;
	sector_count_t     md_size=0;
	DISK_PRIVATE_DATA *disk_pdata=get_s390_disk_private_data(ld);


	LOG_ENTRY();

	rc = llformat( ld, blocksize, DASD_FMT_INT_COMPAT );
	if (rc == 0) {

		write_ipl_record_1(ld);
		write_ipl_record_2(ld);
		write_volume_label_record(ld,TRUE,devinfo->devno);
		write_vtoc(ld, devinfo);

		prune_s390_segments_from_list( ld->parent_objects );

		md_size = disk_pdata->vtoc_lba + disk_pdata->vtoc_sector_count;

		md = create_metadata_segment( ld, 0, md_size );
		if (md) {
			disk_pdata->md = md;
			find_freespace_on_s390_disk( ld );
			rc = 0;
		}
		else {
			LOG_ERROR("error, unable to create a metadata segment for %s\n", ld->name );
			rc = ENOMEM;
		}

	}


	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Called to format the specified disk in LDL format.
 */
static int format_ldl_disk( LOGICALDISK *ld, int blocksize, dasd_information_t *devinfo )
{
	int                rc=EINVAL;
	DISKSEG           *seg=NULL;
	DISKSEG           *md=NULL;
	sector_count_t     md_size=0;
	DISK_PRIVATE_DATA *disk_pdata=get_s390_disk_private_data(ld);


	LOG_ENTRY();

	rc  = llformat( ld, blocksize, 0 );
	if (rc == 0) {

		write_ipl_record_1(ld);
		write_ipl_record_2(ld);
		write_volume_label_record(ld,FALSE,devinfo->devno);

		prune_s390_segments_from_list( ld->parent_objects );

		// metadata is always first 3 blocks on ldl disk.
		//   block 1 = ipl record 1
		//   block 2 = ipl record 2
		//   block 3 = volume label
		md_size  = 3 * disk_pdata->vsectors_per_block;

		md  = create_metadata_segment( ld, 0, md_size );
		seg = create_data_segment( ld, md_size, ld->size - md_size );

		if (seg && md) {
			disk_pdata->md = md;
			rc = 0;
		}
		else {
			rc = ENOMEM;
		}

	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Called by SEG_Assign() to format a physical volume, using
 *  the prescribed method.  As the need arises ... you can add
 *  support here for other dasd formats, e.g. cms minidisk.
 *
 *  LNX1 - Linux
 *  VOL1 - OS/390 compatibility
 */
int  format_390_disk( LOGICALDISK *ld,  ibm_label_type_t  disk_layout, u_int32_t blocksize )
{
	int rc=0;
	dasd_information_t devinfo;
	//char cmd[128];


	LOG_ENTRY();

	rc = get_390_dev_info( ld, &devinfo );
	if (rc == 0) {

		switch ( disk_layout ) {
		
		case ibm_ldl:
			rc = format_ldl_disk(ld, blocksize, &devinfo);
			break;
		case ibm_cdl:
			rc = format_cdl_disk(ld, blocksize, &devinfo);
			//sprintf(cmd,"dasdfmt -F -b %04d -d cdl -f /dev/evms/.nodes/%s\n", blocksize, ld->name);
			//system(cmd);
			break;
		default:
			LOG_ERROR("error, unsupported partition type\n");
			rc = EINVAL;
			break;
		}

	}


	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Can you format this DISK object?
 */
int S390_can_format( LOGICALDISK * ld )
{
	int rc=EINVAL;
	int dasd_api_version=0;
	dasd_information_t dasd_info;

	LOG_ENTRY();

	// dasd driver API version
	rc = disk_ioctl(ld, DASDAPIVER, &dasd_api_version);
	if (!rc) {
		LOG_DEBUG("dasd driver API version = %d\n", dasd_api_version);
		if (dasd_api_version < 0 || dasd_api_version==1) {
			LOG_DEBUG("unsupported dasd api version, version= %d\n", dasd_api_version);
			rc=EINVAL;
		}
	}

	// check the device
	rc = get_390_dev_info(ld,&dasd_info);
	if (!rc) {

		if (dasd_info.open_count > 1) {
			LOG_DEBUG("cant format dasd %s because the in-use count > 1.\n", ld->name);
			rc = EINVAL;
		}
		else if (strncmp(dasd_info.type, "ECKD",4) != 0) {
			LOG_DEBUG("cant format this dasd because it is not E-C-K-D\n");
			rc = EINVAL;
		}
		else {
			rc=0;
		}

	}


	LOG_EXIT_INT(rc);
	return rc;
}

