
/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * 	Module:rsct_mem_info.c
 */
#include "rsct_mem_info.h" 
#include "rsct_internal.h"
#include <glib.h>
#include <stdio.h>
#include <assert.h>
#include <pthread.h>
#define MAX_NODE_NAME_LENGTH 256
#define MAX_BUF_SIZE 256

static GSList              *gl_plugin_list=NULL;
extern pthread_t            gl_appcb_thread;

static pthread_mutex_t          gl_mem_lock;
extern pthread_mutex_t          gl_ack_lock;
ece_msg_t* gl_ecemsg;
pthread_cond_t           gl_ack_cond;

static int walk_count=0;
/************************internal data structure**********************/
typedef struct info_entry_s{
  ece_nodeid_t nodeid;
  int node_number;
  boolean masterRunning;
  boolean slaveRunning;  
  boolean online;

  char name[MAX_NODE_NAME_LENGTH] ;//internal use only
}info_entry_t;


/************************internal static functions********************/
/* Internal static function don't need to get the lock to */
/* to access gl_plugin_list because the calling function must */
/* already obtain the lock                                    */


void 
RSCT_mem_print()
{
	
	int length;
	int i;
	info_entry_t* entry=NULL;
	
	LOG_ENTRY();
	
	length=g_slist_length(gl_plugin_list);
	for(i=0;i<length;i++){
		
		entry=g_slist_nth_data(gl_plugin_list,i);
		
		LOG_DEBUG("entry %d: nodeid=%s,node_number=%d,masterRunning=%d,"
			     " slaveRunning=%d,online=%d\n",
			     i,&entry->nodeid,entry->node_number,
			     entry->masterRunning,entry->slaveRunning,entry->online);
	}	

	LOG_EXIT_VOID();
	return;
}

GSList* 
get_gl_plugin_list()
{
	
	return gl_plugin_list;
	
}

info_entry_t* 
find_entry_using_name(char* name)
{
	
	int length;
	int i;
	info_entry_t* entry=NULL;
	
	LOG_ENTRY();
	
	length=g_slist_length(gl_plugin_list);
	for(i=0;i<length;i++){
		
		entry=g_slist_nth_data(gl_plugin_list,i);
		
		if(strcmp(entry->name,name)== 0 ) break;
		
	}
	
	assert(entry &&"entry not found when deciding a node's online/offline state?");
	RETURN(entry);
}

static void
get_rid_of_quota(char* str)
{
	
	int i;
	
	LOG_ENTRY();
	if(str[0] != '\"'){
		LOG_EXIT_VOID();
		return;
	}
	else{
		
		i=1;
		while(str[i] !='\"'){
			str[i-1]=str[i];
			i++;
		}
		str[i-1]=0;
		str[i]=0;
		if(str[0]=='\"')str[0]=0;
		
	}
	LOG_EXIT_VOID();
	return;
}

//this function is executed only once at the initialization
static void 
get_nodes_info()
{
	char buf[MAX_BUF_SIZE];
	char* temp;
	int i;
	FILE* fstream;
	
	LOG_ENTRY();
	
	if(system("lsrsrc -axd IBM.PeerNode > /tmp/rsct_node_info") == -1){
		LOG_ERROR("ERROR:get_nodes_info() fails ");
	}
	
	
	fstream=fopen("/tmp/rsct_node_info","r");
	
	i=0;
	while(fgets(buf, MAX_BUF_SIZE,fstream) != NULL){
		int length;
		info_entry_t* entry;

		entry=(info_entry_t*)malloc(sizeof(info_entry_t));
		memset(entry,0,sizeof(info_entry_t));
		
		temp=strtok(buf,":"); //name
		strcpy(entry->name,temp);
		get_rid_of_quota(entry->name);
		temp=strtok(NULL,":"); //nodeList
		sscanf(temp+1,"%d",&entry->node_number);
		strtok(NULL,":");   //RSCT version
		strtok(NULL,":");   //class Version
		strtok(NULL,":");   // CritRsrcProtMethod
		temp= strtok(NULL,":"); //NodeNameList in the form of {"xxx"}
		length=strlen(temp);
		strncpy((char*)&entry->nodeid,temp+2,length-4); 
		entry->nodeid.bytes[length-4]='\0';
		i++;
		
		//assert(0);
    
		//add this entry into gl_plugin_list
		entry->masterRunning=FALSE;
		entry->slaveRunning=FALSE;
		entry->online=FALSE;
		gl_plugin_list=g_slist_append(gl_plugin_list,entry);
	}
	fclose(fstream);
	
	//now set for online/offline information
	if(system("lsrpnode -dx> /tmp/rsct_node_info") == -1){
		LOG_ERROR("ERROR:get_nodes_info() fails ");
	}
	fstream=fopen("/tmp/rsct_node_info","r");
  
	i=0;
	while(fgets(buf, MAX_BUF_SIZE,fstream) != NULL){
		
		info_entry_t* entry;
		
		temp=strtok(buf,":"); //name
		
		entry=find_entry_using_name(temp);
		
		assert(entry &&"Peer Domain confiration has been changed?");
		
		temp=strtok(NULL,":"); //status about online/offline
		
		if(strcasecmp(temp,"Offline")==0)
			entry->online=FALSE;
		else if(strcasecmp(temp,"Online") == 0)
			entry->online=TRUE;
		else{
			entry->online=FALSE;
			
		}
	}
	fclose(fstream);
	
	LOG_EXIT_VOID();
	return;
}

static info_entry_t*  
find_entry_use_node_num(int node_number)
{

	int i;
	info_entry_t* entry;
	
	
	LOG_ENTRY();
	for(i=0;i< g_slist_length(gl_plugin_list);i++){
		entry=(info_entry_t*) g_slist_nth_data(gl_plugin_list,i);
		if(entry->node_number == node_number){
			LOG_EXIT_PTR(entry);
			return( entry);
		}
	}
	
	//not found , return NULL
	LOG_EXIT_PTR(NULL);
	return(NULL);
}

//return the number of online nodes
static unsigned int
get_num_online_nodes()
{
	int count=0;
	int i;
	info_entry_t* entry=NULL;
	
	LOG_ENTRY();
	
	for(i=0;i < g_slist_length(gl_plugin_list); i++){
		entry=g_slist_nth_data(gl_plugin_list,i);
		if(entry->online)count++;
	}
	
	LOG_EXIT_INT(count);
	return(count);
}

static void
set_online_offline(unsigned int node_number,
                   boolean online)
{
	info_entry_t* entry=NULL;
	
	
	LOG_ENTRY();
	
	entry=find_entry_use_node_num(node_number);
	assert(entry && "entry not found for the node number?\n");
	
	entry->online=online;
	
	LOG_EXIT_VOID();
	return;
}


/**************membership API  for plugin and node*************/

//membership API for plugin and node

void 
RSCT_mem_cleanup(void)
{
	
	LOG_ENTRY();
	
	pthread_mutex_lock(&gl_mem_lock);
	
	//all memory is freed in the following func call
	g_slist_free(gl_plugin_list);
	
	gl_plugin_list=NULL;

	pthread_mutex_unlock(&gl_mem_lock);
  
	LOG_EXIT_VOID();
	return;
}

void 
RSCT_mem_init(void)
{
	
	LOG_ENTRY();
	
	//initialize gl_send_lock 
	
	pthread_mutex_init(&gl_mem_lock,0);
	
	pthread_mutex_lock(&gl_mem_lock);
	
	assert(gl_plugin_list == NULL &&"gl_plugin_list is not NULL before init?");
	
	get_nodes_info();
	
	pthread_mutex_unlock(&gl_mem_lock);
	
	LOG_EXIT_VOID();
	return;

}


void 
RSCT_mem_nodenum_to_nodeid(ece_nodeid_t* nodeid, 
			  int node_number)
{
	info_entry_t* entry;
	
	LOG_ENTRY();
	pthread_mutex_lock(&gl_mem_lock);
	
	assert(nodeid && " pointer nodeid is NULL?!");  
	assert(node_number >0 &&"invaid node number!");
	
	entry=find_entry_use_node_num(node_number);
	if(entry == NULL){
		LOG_ERROR("node_number not found in:%d\n", node_number);  
		assert(0);
	}
	bzero(nodeid, sizeof(ece_nodeid_t));
	memcpy(nodeid, &entry->nodeid,sizeof(ece_nodeid_t));
	
	pthread_mutex_unlock(&gl_mem_lock);
	
	LOG_EXIT_VOID();
	return;
}

const char * 	
RSCT_mem_get_clusterid(void)
{
	LOG_ENTRY();
	pthread_mutex_lock(&gl_mem_lock);
	pthread_mutex_unlock(&gl_mem_lock);
	
	LOG_EXIT_PTR("RSCT");
	return("RSCT");
}


//this function does not need the mem lock
void
RSCT_mem_get_mynode(ece_nodeid_t* nodeid)
{
	
	int local_node_number=0;
	
	LOG_ENTRY();
	
	ha_gs_get_node_number(&local_node_number);
	RSCT_mem_nodenum_to_nodeid(nodeid, local_node_number);
	
	LOG_EXIT_VOID();
	return;
}




int                     
RSCT_mem_get_num_config_nodes()
{
	int number;
	
	pthread_mutex_lock(&gl_mem_lock);
	
	number = g_slist_length(gl_plugin_list);
	
	pthread_mutex_unlock(&gl_mem_lock);
	
	return number;
	
}

ece_nodeid_t* 
RSCT_mem_map_table_lookup(int node_number)
{
	
	info_entry_t* entry;
	
	ece_nodeid_t*  nodeid;

	LOG_ENTRY();
  
	pthread_mutex_lock(&gl_mem_lock);
	
	entry=find_entry_use_node_num(node_number);
	
	if(entry == NULL)
		nodeid=NULL;
	else
		nodeid=&entry->nodeid;
	
	pthread_mutex_unlock(&gl_mem_lock);
	
	LOG_EXIT_PTR(nodeid);
	
	return(nodeid);
}


//copy all nodes to the provided buffer
//the buffer should have enough space
void 
RSCT_mem_get_all_nodes(ece_nodeid_t * nodes)
{  
	
	int i;
	info_entry_t* entry;
	
	LOG_ENTRY();
	
	pthread_mutex_lock(&gl_mem_lock);
	
	for(i=0;i < g_slist_length(gl_plugin_list);i++){
		entry=(info_entry_t*)g_slist_nth_data(gl_plugin_list,i);
		memcpy(nodes+i, &entry->nodeid,sizeof(ece_nodeid_t));
	}
	
	pthread_mutex_unlock(&gl_mem_lock);
	
	LOG_EXIT_VOID();
	return;
}



/***********************membership API for plugin**********************/

void 
RSCT_plugin_add(const int node_number,
		const int instance_number,
		const ece_nodeid_t* nodeid)
{
	
	info_entry_t* entry;
	
	LOG_ENTRY();
	
	pthread_mutex_lock(&gl_mem_lock);
	
	assert((instance_number == MASTER_INSTANCE_NUM 
		||instance_number ==SLAVE_INSTANCE_NUM)
	       && "wrong instace_number");
	
	entry=find_entry_use_node_num(node_number);
	
	assert(entry &&"node number not found:%d\n");
	
	if(instance_number==MASTER_INSTANCE_NUM){
		entry->masterRunning =1;
	}else{
		entry->slaveRunning = 1;
	}
	
	pthread_mutex_unlock(&gl_mem_lock);
	
	LOG_EXIT_VOID();
	return;
}

boolean 
RSCT_plugin_is_present(const ece_nodeid_t * node, 
		       const int instance_num)
{
	int i;
	info_entry_t* temp;  
	boolean ret=FALSE;
	
	LOG_ENTRY();
	
	pthread_mutex_lock(&gl_mem_lock);
	
	assert((instance_num == MASTER_INSTANCE_NUM
		|| instance_num == SLAVE_INSTANCE_NUM)
	       && "RSCT_plugin_is_present():wrong instance_num");
	
	for(i=0;i < g_slist_length(gl_plugin_list) ;i++){
		temp=(info_entry_t*)g_slist_nth_data(gl_plugin_list,i);
		if(strcmp((char*)&temp->nodeid,(char*)node)==0){
			if(instance_num == MASTER_INSTANCE_NUM)
				ret= temp->masterRunning;
			else
				ret= temp->slaveRunning;
			break;
		}
	}
	
	pthread_mutex_unlock(&gl_mem_lock);
	//if(!ret) assert(0);
	
	LOG_EXIT_BOOL(ret);
	return(ret);
}


//get node count on which master plugin is running
const int 
RSCT_plugin_get_running_master_count(void)
{
	info_entry_t* temp;
	int count =0;
	int i;
	
	LOG_ENTRY();
  
	pthread_mutex_lock(&gl_mem_lock);
	
	for(i=0;i < g_slist_length(gl_plugin_list) ;i++){
		temp=(info_entry_t*)g_slist_nth_data(gl_plugin_list,i);
		if(temp->masterRunning)
			count++;
	}
	
	pthread_mutex_unlock(&gl_mem_lock);
	
	LOG_EXIT_INT(count);
	return(count);
	
}


//if at least one node's master plugin is running, return TRUE
//else return FALSE

boolean 
RSCT_plugin_is_filled(void)
{
	info_entry_t* temp;
	int i;
	boolean ret=FALSE;
	
	LOG_ENTRY();
	
	pthread_mutex_lock(&gl_mem_lock);
	
	if(g_slist_length(gl_plugin_list)==0)
		ret=FALSE;
	else    
		for(i=0;i < g_slist_length(gl_plugin_list) ;i++){
			temp=(info_entry_t*)g_slist_nth_data(gl_plugin_list,i);
			if(temp->masterRunning)
				ret=TRUE;
		}
	
	pthread_mutex_unlock(&gl_mem_lock);
	
	LOG_EXIT_BOOL(ret);
	return(ret);
}

void 
RSCT_plugin_init_walk()
{
	
	LOG_ENTRY();
	
	pthread_mutex_lock(&gl_mem_lock);
	
	walk_count=0;

	pthread_mutex_unlock(&gl_mem_lock);
	
	LOG_EXIT_VOID();
	return;
	
}

ece_nodeid_t* 
RSCT_plugin_get_next_nodeid()
{
	info_entry_t* temp;
	ece_nodeid_t* ret_nodeid;
	
	LOG_ENTRY();
	
	pthread_mutex_lock(&gl_mem_lock);
	
	do{
		temp=(info_entry_t*)g_slist_nth_data(gl_plugin_list,walk_count);
		if(temp==NULL) break;
		walk_count++;
	}while(! temp->masterRunning);
	
	if(temp ==NULL)
		ret_nodeid= NULL;
	else
		ret_nodeid= &temp->nodeid;
  
	pthread_mutex_unlock(&gl_mem_lock);
	
	LOG_EXIT_PTR(ret_nodeid);
	return(ret_nodeid);
}


void 
RSCT_plugin_remove(const int leave_node_number, 
		   const int instance_num)
{
  
	ece_nodeid_t* leave_nodeid;
	info_entry_t* entry=NULL;
	
	LOG_ENTRY();
	
  
	leave_nodeid=RSCT_mem_map_table_lookup(leave_node_number);
	assert(leave_nodeid);
	
	if(instance_num ==MASTER_INSTANCE_NUM)
		LOG_DEBUG("removing master plugin in node %s\n",leave_nodeid);
	else
		LOG_DEBUG("removing slave plugin in node %s\n",leave_nodeid);
	
	pthread_mutex_lock(&gl_ack_lock);
	pthread_mutex_lock(&gl_mem_lock);
	
	//the memory for leave_nodeid is freed in the following func call
	entry=find_entry_use_node_num(leave_node_number);
	assert(entry && "the entry containing the removing node is not found ?!\n");
  
	//change the entry
	if(instance_num ==MASTER_INSTANCE_NUM)
		entry->masterRunning=0;
	else
		entry->slaveRunning=0;

	
	if(gl_ecemsg &&
	   strcmp((char*)&gl_ecemsg->node,(char*)leave_nodeid) ==0)
		pthread_cond_signal(&gl_ack_cond);
	pthread_mutex_unlock(&gl_mem_lock);
	pthread_mutex_unlock(&gl_ack_lock);
  
	LOG_EXIT_VOID();
	return;
}


/************************memebership API for nodes***************/


void
RSCT_mem_online_offline_update(ha_gs_membership_t * membership_list,
                               boolean online)
{
	ha_gs_provider_t *member;
	int i;
	
	LOG_ENTRY();
	
	pthread_mutex_lock(&gl_mem_lock);

	assert(membership_list);
	assert(membership_list->gs_count >0 && "zero members?" );

	if(membership_list->gs_count) {
		member = membership_list->gs_providers;
		for(i = 0; i < membership_list->gs_count; i++, member++){
			set_online_offline(member->gs_node_number,online);
		}
	} 
	
	pthread_mutex_unlock(&gl_mem_lock);
  
	LOG_EXIT_VOID();
	return;
}

int 
RSCT_node_get_membership(ece_event_t* event)
{
	
	int node_count;
	int online_node_count;
	int i;
	info_entry_t* entry=NULL;
	int count=0;
	
	LOG_ENTRY();
	
	pthread_mutex_lock(&gl_mem_lock);
	
	node_count = g_slist_length(gl_plugin_list);
	online_node_count = get_num_online_nodes();
	if( online_node_count > event->num_entries){
		pthread_mutex_unlock(&gl_mem_lock);
		RETURN( ENOSPC);
	}
	for(i=0;i < node_count ;i++){
		entry=g_slist_nth_data(gl_plugin_list,i);
		if(entry->online)      
			memcpy(event->node+count++,&entry->nodeid, sizeof(ece_nodeid_t));
		
	} 
	
	event->num_entries=online_node_count;  
	
	pthread_mutex_unlock(&gl_mem_lock);

	LOG_EXIT_INT(0);
	return(0);
}


int 
RSCT_mem_get_max_node_number()
{
	
	int i;
	info_entry_t* entry;
	int max_node_number=0;
	
	LOG_ENTRY();
	
	pthread_mutex_lock(&gl_mem_lock);
	
	
	for(i=0;i <g_slist_length(gl_plugin_list);i++){
		entry= g_slist_nth_data(gl_plugin_list,i);
		max_node_number = max(max_node_number,
				      entry->node_number);    
	}
	
	pthread_mutex_unlock(&gl_mem_lock);
	
	LOG_EXIT_INT(max_node_number);
	return(max_node_number);
}


boolean
RSCT_mem_has_quorum(){
	
	return TRUE;
	
}
