/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: md_main.c
 *
 * Description: Global and core for MD plugins
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <plugin.h>

#include "md.h"

#define my_plugin_record my_plugin

plugin_record_t	    		* linear_plugin = &linear_plugin_record;
plugin_record_t	    		* raid1_plugin = &raid1_plugin_record;
plugin_record_t	    		* raid0_plugin = &raid0_plugin_record;
plugin_record_t                 * raid5_plugin = &raid5_plugin_record;
plugin_record_t			* mp_plugin = &multipath_plugin_record;
md_volume_t			* volume_list_head = NULL;	// List of real volume groups, indexed by vg_number.
engine_functions_t		* EngFncs;		// The Engine's internal API set.
plugin_record_t			* my_plugin = NULL;
char message_buffer[MD_MESSAGE_BUF_SIZE];
plugin_record_t * evms_plugin_records[] = {&linear_plugin_record,
	                                   &raid1_plugin_record,
	                                   &raid0_plugin_record,
	                                   &raid5_plugin_record,
					   &multipath_plugin_record,
	                                   NULL};


static storage_object_t * md_get_region_for_object(storage_object_t *object)
{
	md_volume_t * volume = volume_list_head;
	storage_object_t *region;
	int i;

	LOG_ENTRY();
	
	while (volume) {
		region = volume->region;
		if (region  && (volume->flags & MD_DISCOVERED)) {
			for (i = 0; i< MAX_MD_DEVICES; i++) {
				if (volume->child_object[i] == object) {
					LOG_DEFAULT("[%s] belongs to %s\n", object->name, region->name);
					LOG_EXIT_PTR(region);
					return region;
				}
			}
		}
		volume = volume->next;
	}
	LOG_DEFAULT("MD does not own this object [%s]\n", object->name);
	LOG_EXIT_PTR(NULL);
	return NULL;
}

/* Function: md_can_replace_child
 *
 *  Can we replace a child?
 */
int md_can_replace_child(storage_object_t *region,
			 storage_object_t *child,
			 storage_object_t *new_child)
{
	u_int32_t child_size, new_child_size;

	LOG_ENTRY();

	/*
	 * We must be able to stop the array in order to recinfigure it with
	 * a new child.
	 */
	if (!md_can_stop_array(region)) {
		LOG_EXIT_INT(EBUSY);
		return EBUSY;
	}

	LOG_DEBUG("region: %s, child:%s, new child:%s\n",
		  region->name, child->name, new_child ? new_child->name : "<unknown>");

	if (region != md_get_region_for_object(child)) {
		LOG_ERROR("[%s] does not belong to %s\n", child->name, region->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}
	if (new_child) {
		/*
		 * The Engine passed in new child object,
		 * make sure it is a data object and
		 * make sure that the sizes are the same
		 */
		if (new_child->data_type != DATA_TYPE) {
			LOG_EXIT_INT(EINVAL);
			return EINVAL;
		}

		child_size = MD_NEW_SIZE_SECTORS(child->size);
		new_child_size = MD_NEW_SIZE_SECTORS(new_child->size);
		if (child_size != new_child_size) {
			LOG_EXIT_INT(EINVAL);
			return EINVAL;
		}
	}

	LOG_EXIT_INT(0);
	return 0;
}


/* Function: md_replace_child
 *
 *
 */
int md_replace_child(storage_object_t *region,
		     storage_object_t *child,
		     storage_object_t *new_child)
{
	int rc=0;
	md_volume_t *volume;
	int i;
	u_int32_t child_size, new_child_size;

	LOG_ENTRY();

	if (region == md_get_region_for_object(child)) {
		
		child_size = MD_NEW_SIZE_SECTORS(child->size);
		new_child_size = MD_NEW_SIZE_SECTORS(new_child->size);

		LOG_DEBUG("region: %s, CHILD: %s, NEW CHILD: %s\n", region->name, child->name, new_child->name);
		LOG_DEBUG(" CHILD object size=%"PRIu64", MD size=%d and NEW CHILD: object size=%"PRIu64", MD size=%d\n",
			  child->size, child_size, new_child->size, new_child_size);

		if (child_size != new_child_size) {
			LOG_ERROR("Child size=%"PRIu64" (MD size=%d) is not equal to new child size=%"PRIu64" (MD size=%d)\n",
				  child->size, child_size, new_child->size, new_child_size);
			LOG_EXIT_INT(EINVAL);
			return EINVAL;
		}

		volume = (md_volume_t *)region->private_data;

		for (i = 0; i< MAX_MD_DEVICES; i++) {
			if (volume->child_object[i] == child) {
				
				KILL_SECTORS(child, MD_NEW_SIZE_SECTORS(child->size), MD_RESERVED_SECTORS);
				md_remove_region_from_object(region, child);
				md_append_region_to_object(region, new_child);
				volume->child_object[i] = new_child;
				volume->super_array[i]->this_disk.major = new_child->dev_major;
				volume->super_array[i]->this_disk.minor = new_child->dev_minor;
				volume->super_block->disks[i].major = new_child->dev_major;
				volume->super_block->disks[i].minor = new_child->dev_minor;
				if (md_is_region_active(region)) {
					region->flags |= SOFLAG_NEEDS_DEACTIVATE | SOFLAG_NEEDS_ACTIVATE;
				}
				region->flags |= SOFLAG_DIRTY;
				volume->flags |= MD_DIRTY;
				break;
			}
		}
	} else {
		LOG_ERROR("%s is not an MD region\n", region->name);
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

#define BLKFLSBUF _IO(0x12,97)
int md_ioctl_flush_buffer_cache(storage_object_t *obj)
{
	int rc = 0, fd;
	LOG_ENTRY();

	fd = EngFncs->open_object(obj, O_RDONLY);
	if (fd <= 0) {
		LOG_ERROR("Unable to open object %s to send ioctl\n", obj->name);
		rc = -fd;
	} else {
		rc = EngFncs->ioctl_object(obj, fd, BLKFLSBUF, NULL);
		if (rc) {
			LOG_ERROR("Unable to flush buffer cache on %s (major=%d, minor=%d)\n",
				obj->name, obj->dev_major, obj->dev_minor);
		}
		EngFncs->close_object(obj, fd);
	}
	LOG_EXIT_INT(rc);
	return rc;
}

int md_ioctl_start_array(storage_object_t *region, int dev_major, int dev_minor)
{
	int rc = 0, fd, ioctl_cmd;
	ulong arg = 0;
	LOG_ENTRY();

	fd = EngFncs->open_object(region, O_RDWR);
	if (fd <= 0) {
		LOG_ERROR("Unable to open region %s to send ioctl\n", region->name);
		rc = -fd;
	} else {
		ioctl_cmd = START_ARRAY;
		arg = makedev(dev_major, dev_minor);
		rc = EngFncs->ioctl_object(region, fd, ioctl_cmd, arg);
		if (rc) {
			LOG_ERROR("Error starting MD array %s (major=%d, minor=%d), rc=%d\n", region->name, dev_major, dev_minor, rc);
		}
		EngFncs->close_object(region, fd);
	}

	LOG_EXIT_INT(rc);
	return rc;

}

int md_ioctl_stop_array(storage_object_t *region)
{
	int rc = 0, fd, ioctl_cmd;
	ulong arg = 0;
	LOG_ENTRY();

	fd = EngFncs->open_object(region, O_RDWR);
	if (fd <= 0) {
		LOG_ERROR("Unable to open region %s to send ioctl\n", region->name);
		rc = -fd;
	} else {
		ioctl_cmd = STOP_ARRAY;
		rc = EngFncs->ioctl_object(region, fd, ioctl_cmd, arg);
		if (rc) {
			LOG_ERROR("%s: IOCTL failed, region(%s) rc(%d)\n", __FUNCTION__, region->name, rc);
		}
		EngFncs->close_object(region, fd);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

int md_ioctl_run_array(storage_object_t *region)
{
	int rc = 0, fd, ioctl_cmd;
	ulong arg = 0;
	LOG_ENTRY();

	fd = EngFncs->open_object(region, O_RDWR);
	if (fd <= 0) {
		LOG_ERROR("Unable to open region %s to send ioctl\n", region->name);
		rc = -fd;
	} else {
		ioctl_cmd = RUN_ARRAY;
		rc = EngFncs->ioctl_object(region, fd, ioctl_cmd, arg);
		if (rc) {
			LOG_ERROR("%s: IOCTL failed, region(%s) rc(%d)\n", __FUNCTION__, region->name, rc);
		}
		EngFncs->close_object(region, fd);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

int md_ioctl_get_disk_info(storage_object_t *region, mdu_disk_info_t *info)
{
	int rc = 0, fd, ioctl_cmd;
	ulong arg = 0;
	LOG_ENTRY();

	fd = EngFncs->open_object(region, O_RDWR);
	if (fd <= 0) {
		LOG_ERROR("Unable to open region %s to send ioctl\n", region->name);
		rc = -fd;
	} else {
		ioctl_cmd = GET_DISK_INFO;
		arg = (ulong)info;
		rc = EngFncs->ioctl_object(region, fd, ioctl_cmd, arg);
		if (rc) {
			LOG_ERROR("%s: IOCTL failed, region(%s) rc(%d)\n", __FUNCTION__, region->name, rc);
		}
		EngFncs->close_object(region, fd);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

int md_ioctl_get_array_info(storage_object_t *region, mdu_array_info_t *md_info)
{
	int rc = 0, fd, ioctl_cmd;
	ulong arg = 0;
	LOG_ENTRY();

	fd = EngFncs->open_object(region, O_RDWR);
	if (fd <= 0) {
		LOG_ERROR("Unable to open region %s to send ioctl\n", region->name);
		rc = -fd;
	} else {
		ioctl_cmd = GET_ARRAY_INFO;
		arg = (ulong)md_info;
		rc = EngFncs->ioctl_object(region, fd, ioctl_cmd, arg);
		if (rc) {
			LOG_ERROR("%s: IOCTL failed, region(%s) rc(%d)\n", __FUNCTION__, region->name, rc);
		}
		EngFncs->close_object(region, fd);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

int md_ioctl_set_array_info(storage_object_t *region, mdu_array_info_t *md_info)
{
	int rc = 0, fd, ioctl_cmd;
	ulong arg = 0;
	LOG_ENTRY();

	fd = EngFncs->open_object(region, O_RDWR);
	if (fd <= 0) {
		LOG_ERROR("Unable to open region %s to send ioctl\n", region->name);
		rc = -fd;
	} else {
		ioctl_cmd = SET_ARRAY_INFO;
		arg = (ulong)md_info;
		rc = EngFncs->ioctl_object(region, fd, ioctl_cmd, arg);
		if (rc) {
			LOG_ERROR("%s: IOCTL failed, region(%s) rc(%d)\n", __FUNCTION__, region->name, rc);
		}
		EngFncs->close_object(region, fd);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

int md_ioctl_add_new_disk(storage_object_t *region, mdu_disk_info_t *disk_info)
{
	int rc = 0, fd, ioctl_cmd;
	ulong arg = 0;
	LOG_ENTRY();

	fd = EngFncs->open_object(region, O_RDWR);
	if (fd <= 0) {
		LOG_ERROR("Unable to open region %s to send ioctl\n", region->name);
		rc = -fd;
	} else {
		ioctl_cmd = ADD_NEW_DISK;
		arg = (ulong)disk_info;
		rc = EngFncs->ioctl_object(region, fd, ioctl_cmd, arg);
		if (rc) {
			LOG_ERROR("%s: IOCTL failed, region(%s), disk(%d:%d) rc(%d)\n", __FUNCTION__, region->name, disk_info->major, disk_info->minor, rc);
		}
		EngFncs->close_object(region, fd);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

#define BLKGETSIZE64 _IOR(0x12,114,size_t)
int md_ioctl_get_blk_size(storage_object_t *region, u_int64_t *size)
{
	int rc = 0, fd, ioctl_cmd;
	ulong arg = 0;
	LOG_ENTRY();

	fd = EngFncs->open_object(region, O_RDWR);
	if (fd <= 0) {
		LOG_ERROR("Unable to open md object %s to send ioctl\n", region->name);
		rc = -fd;
	} else {
		ioctl_cmd = BLKGETSIZE64;
		arg = (ulong)size;
		rc = EngFncs->ioctl_object(region, fd, ioctl_cmd, arg);
		if (rc) {
			LOG_ERROR("Error getting size of md region %s, rc=%d size=%"PRIu64".\n", region->name, rc, *size);
		}
		EngFncs->close_object(region, fd);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int md_ioctl_hot_add_remove(storage_object_t *region, int ioctl_cmd, int kdev)
{
	int rc = 0, fd;
	ulong arg;
	LOG_ENTRY();

	fd = EngFncs->open_object(region, O_RDWR);
	if (fd <= 0) {
		LOG_ERROR("Unable to open md object %s to send ioctl\n", region->name);
		rc = -fd;
	} else {
		arg = (ulong)kdev;
		rc = EngFncs->ioctl_object(region, fd, ioctl_cmd, arg);
		if (rc) {
			LOG_ERROR("Unable to hot %s the spare (major=%d, minor=%d) to [%s]\n",
				  (ioctl_cmd == HOT_ADD_DISK) ? "add" : "remove",
                                  major(kdev), minor(kdev), region->name);
		}
		EngFncs->close_object(region, fd);
	}
	LOG_EXIT_INT(rc);
	return rc;
}

int md_ioctl_hot_add_disk(storage_object_t *region, int kdev)
{
	int rc;
	rc = md_ioctl_hot_add_remove(region, HOT_ADD_DISK, kdev);
	LOG_EXIT_INT(rc);
	return rc;
}

int md_ioctl_hot_remove_disk(storage_object_t *region, int kdev)
{
	int rc;
	rc = md_ioctl_hot_add_remove(region, HOT_REMOVE_DISK, kdev);
	LOG_EXIT_INT(rc);
	return rc;
}

int md_ioctl_set_disk_faulty(storage_object_t *region, int kdev)
{
	int rc = 0, fd;
	ulong arg;
	LOG_ENTRY();

	fd = EngFncs->open_object(region, O_RDWR);
	if (fd <= 0) {
		LOG_ERROR("Unable to open md object %s to send ioctl\n", region->name);
		rc = -fd;
	} else {
		arg = (ulong)kdev;
		rc = EngFncs->ioctl_object(region, fd, SET_DISK_FAULTY, arg);
		if (rc) {
			LOG_ERROR("Unable to set disk faulty (major=%d, minor=%d) to [%s]\n",
                                  major(kdev), minor(kdev), region->name);
		}
		EngFncs->close_object(region, fd);
	}
	LOG_EXIT_INT(rc);
	return rc;
}

int md_get_kernel_info(storage_object_t * region, mdu_array_info_t* md_info)
{
	int rc=0;
	u_int64_t size = 0;
	md_volume_t *vol = (md_volume_t *)region->private_data;

	LOG_ENTRY();

	rc = md_ioctl_get_array_info(region, md_info);
	if (rc) {
		region->flags &= ~SOFLAG_ACTIVE;
	} else {
		rc = md_ioctl_get_blk_size(region, &size);
		if (!rc && size != 0) {
			if (vol) vol->flags |= MD_ACTIVE;
			region->flags |= SOFLAG_ACTIVE;
		} else {
			if (vol) vol->flags &= ~MD_ACTIVE;
			region->flags &= ~SOFLAG_ACTIVE;
			rc = EBUSY;
		}
	}
	LOG_EXIT_INT(rc);
	return rc;
}


int md_activate_region(storage_object_t * region)
{
	int rc=0;
	u_int32_t i;
	md_volume_t  *volume = (md_volume_t *)region->private_data;
	mdp_super_t *sb;
	mdu_array_info_t md_info;

	LOG_ENTRY();

	if (volume->flags & MD_USE_OLD_DEV) {
		LOG_DEFAULT("The original dev nodes have been restored for %s, will not re-activate\n",region->name);
		LOG_EXIT_INT(0);
		return 0;
	}
	
	if (region->flags & SOFLAG_CORRUPT) {
		LOG_WARNING("Region %s is corrupt.  It can not be activated.\n", region->name);
		LOG_EXIT_INT(EINVAL);
		return 0;
	}

	if (region->flags & SOFLAG_ACTIVE) {
		LOG_DEFAULT("%s is currently active, deactivating...\n", region->name);
		rc = md_deactivate_region(region);
		if (rc) {
			LOG_EXIT_INT(rc);
			return 0;
		}
	}

	sb = volume->super_block;
	rc = md_ioctl_set_array_info(region, NULL);
	if (!rc) {
		int found;
		for (i=0, found=0; i< MAX_MD_DEVICES && !rc && found < sb->nr_disks; i++) {
			if (!volume->child_object[i])
				continue;
			if (!disk_removed(&sb->disks[i]) &&
			    !disk_faulty(&sb->disks[i]) &&
			    !disk_new_pending_active(&sb->disks[i]) ) {
					mdu_disk_info_t disk_info;
					memset(&disk_info, 0, sizeof(disk_info));
					disk_info.major = volume->child_object[i]->dev_major;
					disk_info.minor = volume->child_object[i]->dev_minor;
					rc = md_ioctl_add_new_disk(region, &disk_info);
					found++;
			}
		}
		if (!rc)
			rc = md_ioctl_run_array(region);
	}

	if (!rc) {
		// activate succeeded, turn on active flag by calling md_get_kernel_info
		rc = md_get_kernel_info(region, &md_info);
	}

	if (!rc) {
		sb->state = md_info.state;
		region->flags &= ~SOFLAG_NEEDS_ACTIVATE;
	}

	LOG_EXIT_INT(rc);
	return rc;

}

int md_deactivate_region(storage_object_t * region)
{
	int rc=0;
	md_volume_t *volume;

	LOG_ENTRY();
	if (!region) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}
	
	/* If the region is not active, just clear flags and return success */
	if (!md_is_region_active(region)) {
		region->flags &= ~(SOFLAG_ACTIVE | SOFLAG_NEEDS_DEACTIVATE);
		LOG_EXIT_INT(rc);
		return rc;
	}

	rc = md_ioctl_stop_array(region);
	if (!rc) {
		if (md_is_region_active(region)) {
			rc = EBUSY;
			LOG_ERROR("BUG: %s, %s, %d\n", __FILE__, __FUNCTION__, __LINE__);
		} else {
			/*
			 * Reload all superblocks ???
			 * For now just mark the array clean.
			 */
			volume = (md_volume_t *)region->private_data;
			if (volume) {
				volume->super_block->state |= (1 << MD_SB_CLEAN);
			} else {
				LOG_DETAILS("MD volume is gone, this must be a delete region (%s)\n", region->name);
			}
		}
	}
	
	if (!rc || (rc == ENODEV)) {
		region->flags &= ~(SOFLAG_ACTIVE | SOFLAG_NEEDS_DEACTIVATE);
		rc = 0;
	}

	LOG_EXIT_INT(rc);
	return rc;

}

boolean md_is_region_active(storage_object_t * region)
{
	MD_CHECK_ACTIVE(region);
	return (region->flags & SOFLAG_ACTIVE) ? TRUE : FALSE;
}


static boolean mdstat_check_recovery(int mddev_minor)
{
	FILE *fp;
	char string[256];
	char mddev_str[10];
	boolean recovery_running = FALSE;

	LOG_ENTRY();

	sprintf(mddev_str, "md%d :", mddev_minor);

	fp = fopen("/proc/mdstat", "r");
	if (fp) {
		while (fgets(string, 256, fp) != NULL) {
			if (!strncmp(string, mddev_str, strlen(mddev_str))) {

				if (fgets(string, 256, fp) == NULL)
					break;
				if (fgets(string, 256, fp) == NULL)
					break;
				if (strstr(string, "recovery"))
					recovery_running = TRUE;
				else if (strstr(string, "resync"))
						recovery_running = TRUE;
				
				/* found mdx, break out of while() loop */
				break;
			}
		}
		fclose(fp);
	} else {
		LOG_WARNING("Could not open /proc/mdstat for reading.\n");
	}
	LOG_EXIT_INT(recovery_running);
	return recovery_running;
}

boolean md_is_recovery_running(storage_object_t * region)
{
	int rc;
	boolean recovery_running = FALSE;
	mdu_array_info_t md_info;
	md_volume_t *volume;

	LOG_ENTRY();

	rc = md_ioctl_get_array_info(region, &md_info);
	if (rc) {
		LOG_EXIT_INT(FALSE);
		return FALSE;
	}

	recovery_running = (md_info.state & (1<<MD_ARRAY_RECOVERY_RUNNING)) ? TRUE : FALSE;
	if (recovery_running == FALSE) {
		/*
		 * Just in case the kernel does not provide the information
		 * about resync/recovery, try scanning /proc/mdstat
		 */
		volume = (md_volume_t *)region->private_data;
		recovery_running = mdstat_check_recovery(volume->super_block->md_minor);
	}

	LOG_EXIT_INT(recovery_running);
	return recovery_running;
}

int md_fix_dev_major_minor(md_volume_t *vol, boolean do_msg)
{
	int rc=0;
	int i, major, minor;
	mdp_super_t *sb = vol->super_block;
	storage_object_t *obj;

	LOG_ENTRY();

	if (!sb) {
		md_log_internal_bug(__FILE__, __FUNCTION__, __LINE__);
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}
	if (!vol->region) {
		md_log_internal_bug(__FILE__, __FUNCTION__, __LINE__);
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (vol->commit_flag & MD_COMMIT_RESTORE_SAVED_SB) {
		md_log_internal_bug(__FILE__, __FUNCTION__, __LINE__);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	for (i=0; i<MAX_MD_DEVICES; i++) {
		obj = vol->child_object[i];
		if (!obj)
			continue;
		
		major = obj->dev_major;
		minor = obj->dev_minor;
		if ((sb->disks[i].major != major ||
		     sb->disks[i].minor != minor) &&
			(major != 0 || minor !=0)) {
			LOG_DEFAULT("Region %s object (%s) index (%d) : changing from major/minor (%d:%d) to (%d:%d)\n",
			    vol->name, obj->name, i,
			    sb->disks[i].major,
			    sb->disks[i].minor,
			    major,
			    minor);
			MESSAGE("Region %s: The MD superblock has old device "
				"major/minor number for object (%s). The old "
				"dev [%d:%d] has been replaced with new dev "
				"[%d:%d]. The MD superblock will be modified "
				"when the MD region is activated.\n",
				vol->region->name, obj->name,
				sb->disks[i].major, sb->disks[i].minor,
				major, minor );
			
			if (sb->disks[i].major != major)
				vol->commit_flag |= MD_COMMIT_SAVE_SB;
			
			sb->disks[i].major = major;
			sb->disks[i].minor = minor;
			vol->flags |= MD_DIRTY;
			vol->region->flags |= (SOFLAG_DIRTY | SOFLAG_NEEDS_DEACTIVATE);
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

boolean follow_up_mark_faulty(md_volume_t *volume, storage_object_t *faulty)
{
	int    answer = 0;
	char * choice_text[3] = {  "Don't remove faulty", "Remove faulty", NULL };
	EngFncs->user_message(
		my_plugin, &answer,choice_text,
		"The object [%s] has been marked faulty.  "
		"Would you also like to remove [%s] from %s region?\n\n"
		"If you elect not to remove [%s] at this time, "
		"you should consider removing it later via the region's option menu.\n",
		faulty->name, faulty->name, volume->name, faulty->name);
	
	return answer ? TRUE : FALSE;
}

int md_sync_sbs(md_volume_t *volume, mdu_array_info_t *array_info)
{
	int i, j, rc=0;
	mdp_super_t *sb;
	mdp_disk_t disk;
	LOG_ENTRY();

	if (!volume) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	sb = volume->super_block;

	sb->nr_disks = array_info->nr_disks;
	sb->raid_disks = array_info->raid_disks;
	sb->md_minor = array_info->md_minor;
	sb->utime = array_info->utime;
	sb->state = array_info->state;
	sb->active_disks = array_info->active_disks;
	sb->working_disks = array_info->working_disks;
	sb->failed_disks = array_info->failed_disks;
	sb->spare_disks = array_info->spare_disks;
	for (i=0; i < MAX_MD_DEVICES; i++) {
		mdu_disk_info_t disk_info;
		disk_info.number = i;
		rc = md_ioctl_get_disk_info(volume->region, &disk_info);
		if (!rc) {
			/* save a copy before copying */
			disk = sb->disks[i];

//			sb->disks[i].number = disk_info.number;
			sb->disks[i].major = disk_info.major;
			sb->disks[i].minor = disk_info.minor;
			sb->disks[i].raid_disk = disk_info.raid_disk;
			sb->disks[i].state = disk_info.state;

			if (memcmp(&disk, &sb->disks[i], sizeof(mdp_disk_t)) == 0)
				continue;	// the same

			LOG_EXTRA("[%s] number(%d) major(%d) minor(%d) raid_disk(%d) state(0x%08X)\n",	
				volume->name, disk_info.number,
				disk_info.major, disk_info.minor,
				disk_info.raid_disk, disk_info.state);
			
			/*
			 * Unless it had been removed,
			 * a disk should be associated with a child object.
			 * If child_object is NULL or the major/minor don't match, 
			 * the kernel probably swapped disk entries after a resync.
			 *
			 * One example is a 2-way raid1 array with a spare and the
			 * operations are:
			 *  - mark disk faulty
			 *  - remove faulty
			 *
			 */

			if (!disk_removed(&sb->disks[i])) {
				if ((volume->child_object[i] == NULL) ||
					((volume->child_object[i]->dev_major != disk_info.major ||
						volume->child_object[i]->dev_minor != disk_info.minor)) ) {
					
					j = find_disk_in_volume(volume, disk_info.major, disk_info.minor);
					
					if (j == MAX_MD_DEVICES) {
						LOG_ERROR("%s: Could not find child object for disk index=%d.\n",
							volume->name, i);
						md_log_internal_bug(__FILE__, __FUNCTION__, __LINE__);
					} else {
						void *tmp_ptr;
						char *name1 = NULL, *name2 = NULL;

						if (volume->child_object[i])
							name1 = volume->child_object[i]->name;
						if (volume->child_object[j])
							name2 = volume->child_object[j]->name;
						
						/* swap entries */
						LOG_DEFAULT("[%s] swapping entries obj[i=%d]=%s obj[j=%d]=%s.\n",
							volume->name, i, name1, j, name2);

						tmp_ptr = volume->child_object[i];
						volume->child_object[i] = volume->child_object[j];
						volume->child_object[j] = (storage_object_t *)tmp_ptr;
						tmp_ptr = volume->super_array[i];
						volume->super_array[i] = volume->super_array[j];
						volume->super_array[j] = (mdp_super_t *)tmp_ptr;
					}
				}
			}
		} else {
			LOG_EXIT_INT(rc);
			return rc;
		}
	}
	sb->this_disk = sb->disks[sb->this_disk.number];

	for (i = 0; i < MAX_MD_DEVICES; i++) {

		if (!volume->child_object[i])
			continue;

		sb = volume->super_array[i];
		if (!sb)
			continue;

		memcpy(sb, volume->super_block, MD_SB_BYTES);
		sb->this_disk = sb->disks[i];
	}

	LOG_EXIT_INT(0);
	return 0;
}

/*
 * Function:  md_can_stop_array
 *
 * Note : MD driver will fail MD_STOP ioctl if the "open" count is not 0.
 */
boolean md_can_stop_array( storage_object_t *region )
{
	boolean rc = TRUE;

	LOG_ENTRY();
	if (md_is_region_active(region)) {
		if (region->volume) {
			if (!(region->volume->flags & VOLFLAG_COMPATIBILITY)) {
				LOG_DETAILS("Region %s is part of EVMS volume %s.\n", 
					    region->name, region->volume->name);
				rc = FALSE;
			}
			if (EngFncs->is_mounted(region->volume->name, NULL)) {
				LOG_DETAILS("Region %s is part of volume %s which is mounted on %s.\n",
					    region->name, region->volume->name, region->volume->mount_point);
				rc = FALSE;
			}
		}
	
		if (EngFncs->list_count(region->parent_objects)) {
			LOG_DETAILS("Region %s has parent(s).\n", region->name);
			rc = FALSE;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

int child_to_index(md_volume_t *vol, storage_object_t *obj)
{
	int i;

	for (i=0; i<MAX_MD_DEVICES; i++) {
		if (vol->child_object[i] == obj) {
			break;
		}
		if (vol->stale_object[i] == obj) {
			break;
		}
	}
	if (i==MAX_MD_DEVICES)
		md_log_internal_bug(__FILE__, __FUNCTION__, __LINE__);
	return i;
}

int find_disk_in_volume(md_volume_t *volume, int major, int minor)
{
	int i;
	for (i=0; i < MAX_MD_DEVICES; i++) {
		if (volume->child_object[i] &&
		    volume->child_object[i]->dev_major == major &&
		    volume->child_object[i]->dev_minor == minor) {
		    break;
		}
	}
	if (i==MAX_MD_DEVICES)
		LOG_WARNING("Could not find disk[%d:%d] in %s.\n",
			major, minor, volume->name);
	return i;
}

int find_disk_in_active_region(storage_object_t *region, int major, int minor)
{
	int i, rc;
	mdu_disk_info_t info;

	for (i=0; i<MAX_MD_DEVICES; i++) {
		info.number = i;
		rc = md_ioctl_get_disk_info(region, &info);
		if (!rc && (info.major == major) && (info.minor == minor))
			break;
	}
	if (i==MAX_MD_DEVICES)
		LOG_WARNING("Could not find disk[%d:%d] in MD region %s.\n",
			major, minor, region->name);
	return i;
}

int find_empty_slot(mdp_super_t *sb)
{
	int i;
	int start_idx, stop_idx;
	boolean found = FALSE;

	LOG_ENTRY();

	/*
	 * Try to find an empty slot starting from raid_disks,
	 * (see kernel md driver code (md.c)
	 * If we reach MAX_MD_DEVICES, don't give up,
	 * go back and try from the beginning.
	 */
	start_idx = sb->raid_disks;
	stop_idx = MAX_MD_DEVICES;
	while (found == FALSE) {
		for (i=start_idx; i < stop_idx; i++) {
			if (descriptor_removed(&sb->disks[i])) {
				found = TRUE;
				break;
			}
			if (descriptor_empty(&sb->disks[i])) {
				found = TRUE;
				break;
			}
		}

		if (found == FALSE) {  
			if (start_idx != 0) {
				start_idx = 0; // Try from the beginning
				stop_idx = sb->raid_disks;
			} else {
				LOG_CRITICAL("md%d array is full.\n", sb->md_minor);
				i = MAX_MD_DEVICES;
				break;
			}
		}
	}
	
	LOG_EXIT_INT(i);
	return i;
}

inline void md_log_internal_bug(const char * file, const char * func, int line)
{
	LOG_CRITICAL(" Internal error from %s, in %s function, at line %d\n", file, func, line);
}
