/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: mdregmgr.h
 */

#ifndef _MD_REGION_MANAGER_H_
#define _MD_REGION_MANAGER_H_ 1

/*
 * RAID superblock.
 *
 * The RAID superblock maintains some statistics on each RAID configuration.
 * Each real device in the RAID set contains it near the end of the device.
 * Some of the ideas are copied from the ext2fs implementation.
 *
 * We currently use 4096 bytes as follows:
 *
 *	word offset	function
 *
 *	   0  -    31	Constant generic RAID device information.
 *        32  -    63   Generic state information.
 *	  64  -   127	Personality specific information.
 *	 128  -   511	12 32-words descriptors of the disks in the raid set.
 *	 512  -   911	Reserved.
 *	 912  -  1023	Disk specific descriptor.
 */

/*
 * If x is the real device size in bytes, we return an apparent size of:
 *
 *	y = (x & ~(MD_RESERVED_BYTES - 1)) - MD_RESERVED_BYTES
 *
 * and place the 4kB superblock at offset y.
 */
#define MD_RESERVED_BYTES		(64 * 1024)
#define MD_RESERVED_SECTORS		(MD_RESERVED_BYTES / 512)
#define MD_RESERVED_BLOCKS		(MD_RESERVED_BYTES / BLOCK_SIZE)

#define MD_NEW_SIZE_SECTORS(x)		(((x) & ~(MD_RESERVED_SECTORS - 1)) - MD_RESERVED_SECTORS)
#define MD_NEW_SIZE_BLOCKS(x)		(((x) & ~(MD_RESERVED_BLOCKS - 1)) - MD_RESERVED_BLOCKS)

#define MD_SB_BYTES			4096
#define MD_SB_WORDS			(MD_SB_BYTES / 4)
#define MD_SB_BLOCKS			(MD_SB_BYTES / BLOCK_SIZE)
#define MD_SB_SECTORS			(MD_SB_BYTES / 512)

/*
 * The following are counted in 32-bit words
 */
#define	MD_SB_GENERIC_OFFSET		0
#define MD_SB_PERSONALITY_OFFSET	64
#define MD_SB_DISKS_OFFSET		128
#define MD_SB_DESCRIPTOR_OFFSET		992

#define MD_SB_GENERIC_CONSTANT_WORDS	32
#define MD_SB_GENERIC_STATE_WORDS	32
#define MD_SB_GENERIC_WORDS		(MD_SB_GENERIC_CONSTANT_WORDS + MD_SB_GENERIC_STATE_WORDS)
#define MD_SB_PERSONALITY_WORDS		64
#define MD_SB_DESCRIPTOR_WORDS		32
#define MD_SB_DISKS			27
#define MD_SB_DISKS_WORDS		(MD_SB_DISKS*MD_SB_DESCRIPTOR_WORDS)
#define MD_SB_RESERVED_WORDS		(1024 - MD_SB_GENERIC_WORDS - MD_SB_PERSONALITY_WORDS - MD_SB_DISKS_WORDS - MD_SB_DESCRIPTOR_WORDS)
#define MD_SB_EQUAL_WORDS		(MD_SB_GENERIC_WORDS + MD_SB_PERSONALITY_WORDS + MD_SB_DISKS_WORDS)


/*
 * Device "operational" state bits
 */
#define MD_DISK_FAULTY		0 /* disk is faulty / operational */
#define MD_DISK_ACTIVE		1 /* disk is running or spare disk */
#define MD_DISK_SYNC		2 /* disk is in sync with the raid set */
#define MD_DISK_REMOVED		3 /* disk has kind of been removed, but not really or it would not be here */

#define MD_DISK_NEW		4 /* disk has just been added to the raid set */
#define MD_DISK_PENDING_ACTIVE	5 /* disk was spare, but should be activated */
#define MD_DISK_MISSING		6 /* disk is missing */
#define MD_DISK_STALE		7 /* disk with event < master sb event */

typedef struct mdp_device_descriptor_s {
	u_int32_t number;	/* 0 Device number in the entire set	      */
	u_int32_t major;	/* 1 Device major number		      */
	u_int32_t minor;	/* 2 Device minor number		      */
	u_int32_t raid_disk;	/* 3 The role of the device in the raid set   */
	u_int32_t state;	/* 4 Operational state			      */
	u_int32_t reserved[MD_SB_DESCRIPTOR_WORDS - 5];
} mdp_disk_t;

static inline int disk_faulty(mdp_disk_t * d)
{
	return d->state & (1 << MD_DISK_FAULTY);
}

static inline int disk_active(mdp_disk_t * d)
{
	return d->state & (1 << MD_DISK_ACTIVE);
}

static inline int disk_sync(mdp_disk_t * d)
{
	return d->state & (1 << MD_DISK_SYNC);
}

static inline int disk_spare(mdp_disk_t * d)
{
	return !disk_sync(d) && !disk_active(d) && !disk_faulty(d);
}

static inline int disk_removed(mdp_disk_t * d)
{
	return d->state & (1 << MD_DISK_REMOVED);
}

static inline int disk_new(mdp_disk_t * d)
{
	return d->state & (1 << MD_DISK_NEW);
}

static inline int disk_pending_active(mdp_disk_t * d)
{
	return (disk_new(d) && disk_pending_active(d));
}

static inline int disk_new_pending_active(mdp_disk_t * d)
{
	return d->state & (1 << MD_DISK_PENDING_ACTIVE);
}


static inline void mark_disk_faulty(mdp_disk_t * d)
{
	d->state |= (1 << MD_DISK_FAULTY);
}

static inline void mark_disk_active(mdp_disk_t * d)
{
	d->state |= (1 << MD_DISK_ACTIVE);
}

static inline void mark_disk_sync(mdp_disk_t * d)
{
	d->state |= (1 << MD_DISK_SYNC);
}

static inline void mark_disk_spare(mdp_disk_t * d)
{
	d->state = 0;
}

static inline void mark_disk_removed(mdp_disk_t * d)
{
	d->state = (1 << MD_DISK_FAULTY) | (1 << MD_DISK_REMOVED);
}

static inline void mark_disk_inactive(mdp_disk_t * d)
{
	d->state &= ~(1 << MD_DISK_ACTIVE);
}

static inline void mark_disk_nonsync(mdp_disk_t * d)
{
	d->state &= ~(1 << MD_DISK_SYNC);
}

static inline void mark_disk_missing(u_int32_t *state)
{
	*state |= (1 << MD_DISK_MISSING);
}

static inline void mark_disk_stale(u_int32_t *state)
{
	*state |= (1 << MD_DISK_STALE);
}

static inline int descriptor_removed(mdp_disk_t *disk) {
	if ( (disk->major == 0) &&  (disk->minor == 0) &&
	     (disk->state & (1 << MD_DISK_REMOVED)))
		return 1;
	else
		return 0;

}

static inline void remove_descriptor(mdp_disk_t *d) {
	d->major = d->minor = 0;
	d->state = (1 << MD_DISK_REMOVED);
}

static inline int descriptor_empty(mdp_disk_t *d) {
	return ( d->number == 0 && d->major == 0 && d->minor == 0 && d->raid_disk == 0 && !disk_faulty(d) && !disk_active(d));
}


#define MD_SB_MAGIC		0xa92b4efc

/*
 * Superblock state bits
 */
#define MD_SB_CLEAN		0
#define MD_SB_ERRORS		1

typedef struct mdp_superblock_s {
	/*
	 * Constant generic information
	 */
	u_int32_t md_magic;		/*  0 MD identifier 			      */
	u_int32_t major_version;	/*  1 major version to which the set conforms */
	u_int32_t minor_version;	/*  2 minor version ...			      */
	u_int32_t patch_version;	/*  3 patchlevel version ...		      */
	u_int32_t gvalid_words;		/*  4 Number of used words in this section    */
	u_int32_t set_uuid0;		/*  5 Raid set identifier		      */
	u_int32_t ctime;		/*  6 Creation time			      */
	u_int32_t level;		/*  7 Raid personality			      */
	u_int32_t size;			/*  8 Apparent size of each individual disk   */
	u_int32_t nr_disks;		/*  9 total disks in the raid set	      */
	u_int32_t raid_disks;		/* 10 disks in a fully functional raid set    */
	u_int32_t md_minor;		/* 11 preferred MD minor device number	      */
	u_int32_t not_persistent;	/* 12 does it have a persistent superblock    */
	u_int32_t set_uuid1;		/* 13 Raid set identifier #2		      */
	u_int32_t set_uuid2;		/* 14 Raid set identifier #3		      */
	u_int32_t set_uuid3;		/* 15 Raid set identifier #4		      */
	u_int32_t gstate_creserved[MD_SB_GENERIC_CONSTANT_WORDS - 16];

	/*
	 * Generic state information
	 */
	u_int32_t utime;		/*  0 Superblock update time		      */
	u_int32_t state;		/*  1 State bits (clean, ...)		      */
	u_int32_t active_disks;		/*  2 Number of currently active disks	      */
	u_int32_t working_disks;	/*  3 Number of working disks		      */
	u_int32_t failed_disks;		/*  4 Number of failed disks		      */
	u_int32_t spare_disks;		/*  5 Number of spare disks		      */
	u_int32_t sb_csum;		/*  6 checksum of the whole superblock        */
#if __BYTE_ORDER == __BIG_ENDIAN
	u_int32_t events_hi;		/*  7 high-order of superblock update count   */
	u_int32_t events_lo;		/*  8 low-order of superblock update count    */
#else
	u_int32_t events_lo;		/*  7 low-order of superblock update count    */
	u_int32_t events_hi;		/*  8 high-order of superblock update count   */
#endif
	u_int32_t gstate_sreserved[MD_SB_GENERIC_STATE_WORDS - 9];

	/*
	 * Personality information
	 */
	u_int32_t layout;		/*  0 the array's physical layout	      */
	u_int32_t chunk_size;		/*  1 chunk size in bytes		      */
	u_int32_t root_pv;		/*  2 LV root PV */
	u_int32_t root_block;		/*  3 LV root block */
	u_int32_t pstate_reserved[MD_SB_PERSONALITY_WORDS - 4];

	/*
	 * Disks information
	 */
	mdp_disk_t disks[MD_SB_DISKS];

	/*
	 * Reserved
	 */
	u_int32_t reserved[MD_SB_RESERVED_WORDS];

	/*
	 * Active descriptor
	 */
	mdp_disk_t this_disk;

}mdp_super_t;

#define MD_SAVED_SB_INFO_BYTES		1024
#define MD_SAVED_SB_INFO_WORDS		(MD_SAVED_SB_INFO_BYTES / 4)
#define MD_SAVED_SB_INFO_SECTS		(MD_SAVED_SB_INFO_BYTES / 512)
#define MD_SAVED_SB_SECTOR_OFFSET	(MD_RESERVED_SECTORS - MD_SAVED_SB_INFO_SECTS)

typedef struct mdp_saved_super_s {
	/*
	 * Constant generic information
	 */
	u_int32_t md_magic;		/*  0 MD identifier			*/
	u_int32_t major_version;	/*  1 major version when created	*/
	u_int32_t minor_version;	/*  2 minor version when created	*/
	u_int32_t patch_version;	/*  3 patchlevel version		*/
	u_int32_t set_uuid0;		/*  4 MD set identifier #1		*/
	u_int32_t set_uuid1;		/*  5 MD set identifier #2		*/
	u_int32_t set_uuid2;		/*  6 MD set identifier #3		*/
	u_int32_t set_uuid3;		/*  7 MD set identifier #4		*/
	u_int32_t ctime;		/*  8 Creation time			      */
	u_int32_t level;		/*  9 Raid personality			      */
	u_int32_t size;			/* 10 Apparent size of each individual disk   */
	u_int32_t md_minor;		/* 11 preferred MD minor device number	      */

	u_int32_t this_disk_number;	/* 12 original this disk number		*/
	u_int32_t this_disk_major;	/* 13 original this disk major		*/
	u_int32_t this_disk_minor;	/* 14 original this disk minor		*/
	u_int32_t sb_csum;		/* 15 checksum				*/
	u_int32_t reserved[MD_SAVED_SB_INFO_WORDS - 16];
}mdp_saved_super_t;

static inline u_int64_t md_event(mdp_super_t *sb) {
	u_int64_t ev = sb->events_hi;
	return (ev<<32)| sb->events_lo;
}

// Datatypes and Defs used in mdregmgr.c
#define SECTOR_SIZE_BITS            9
#define PAGE_SIZE_SECTORS           16      // for 8k pages
#define PE_SIZE_ALIGNMENT_SECTORS   128     // for 64k aligned PEs
#define MD_NAME_SPACE               EVMS_DEV_NODE_PATH "md"
#define MD_DEV_DIRECTORY            "md/"
#define MAX_MD_DEVICES              27
#define MAX_MD_MINORS               255
#define MD_MIN_CHUNK_SIZE           4       // in kilobytes
#define MD_MAX_CHUNK_SIZE           4096    // in kilobytes = 4MB
#define MD_DEFAULT_CHUNK_SIZE       32      // in kilobytes
#define BLOCK_SIZE                  1024    // needed by md size macros

#define MD_RESERVED       0UL
#define LINEAR            1UL
#define RAID0             2UL
#define RAID1             3UL
#define RAID5             4UL
#define TRANSLUCENT       5UL
#define HSM               6UL
#define MULTIPATH         7UL
#define MAX_PERSONALITY   8UL

#define INVALID_LEVEL    99UL

static inline int pers_to_level (int pers) {
	switch (pers) {
	case MULTIPATH:     return -4;
	case HSM:           return -3;
	case TRANSLUCENT:   return -2;
	case LINEAR:        return -1;
	case RAID0:         return 0;
	case RAID1:         return 1;
	case RAID5:         return 5;
	}
	return INVALID_LEVEL;
}

static inline int level_to_pers (int level) {
	switch (level) {
	case -4: return MULTIPATH;
	case -3: return HSM;
	case -2: return TRANSLUCENT;
	case -1: return LINEAR;
	case 0:  return RAID0;
	case 1:  return RAID1;
	case 4:
	case 5:  return RAID5;
	}
	return MD_RESERVED;
}

static inline char * level_to_string (int level) {
	switch (level) {
	case -4: return "MULTIPATH";
	case -3: return "HSM";
	case -2: return "TRANSLUCENT";
	case -1: return "LINEAR";
	case 0:  return "RAID0";
	case 1:  return "RAID1";
	case 4:
	case 5:  return "RAID5";
	}
	return "INVALID LEVEL";
}

#define MD_CHUNK_ALIGN_NEW_SIZE_SECTORS(chunk_size_in_sectors, size_in_sectors) \
	((chunk_size_in_sectors) ? (MD_NEW_SIZE_SECTORS(size_in_sectors) & ~((chunk_size_in_sectors) - 1)) : MD_NEW_SIZE_SECTORS(size_in_sectors))
#define MD_CHUNK_ALIGN_NEW_SIZE_BLOCKS(chunk_size_in_blocks, size_in_blocks) \
	((chunk_size_in_blocks) ? (MD_NEW_SIZE_BLOCKS(size_in_blocks) & ~((chunk_size_in_blocks) - 1)) : MD_NEW_SIZE_BLOCKS(size_in_blocks))

#define VSECTORS_TO_BLOCKS(sectors) ((sector_count_t) sectors / (BLOCK_SIZE >> EVMS_VSECTOR_SIZE_SHIFT))
#define BLOCKS_TO_VSECTORS(blocks)  ((sector_count_t) blocks  * (BLOCK_SIZE >> EVMS_VSECTOR_SIZE_SHIFT))

/* ioctls */
#define MD_MAJOR 9

/* status */
#define RAID_VERSION		_IOR (MD_MAJOR, 0x10, mdu_version_t)
#define GET_ARRAY_INFO		_IOR (MD_MAJOR, 0x11, mdu_array_info_t)
#define GET_DISK_INFO		_IOR (MD_MAJOR, 0x12, mdu_disk_info_t)
#define PRINT_RAID_DEBUG	_IO (MD_MAJOR, 0x13)
#define RAID_AUTORUN		_IO (MD_MAJOR, 0x14)

/* configuration */
#define CLEAR_ARRAY		_IO (MD_MAJOR, 0x20)
#define ADD_NEW_DISK		_IOW (MD_MAJOR, 0x21, mdu_disk_info_t)
#define HOT_REMOVE_DISK		_IO (MD_MAJOR, 0x22)
#define SET_ARRAY_INFO		_IOW (MD_MAJOR, 0x23, mdu_array_info_t)
#define SET_DISK_INFO		_IO (MD_MAJOR, 0x24)
#define WRITE_RAID_INFO		_IO (MD_MAJOR, 0x25)
#define UNPROTECT_ARRAY		_IO (MD_MAJOR, 0x26)
#define PROTECT_ARRAY		_IO (MD_MAJOR, 0x27)
#define HOT_ADD_DISK		_IO (MD_MAJOR, 0x28)
#define SET_DISK_FAULTY		_IO (MD_MAJOR, 0x29)
#define HOT_GENERATE_ERROR	_IO (MD_MAJOR, 0x2a)

/* usage */
#define RUN_ARRAY		_IOW (MD_MAJOR, 0x30, mdu_param_t)
#define START_ARRAY		_IO (MD_MAJOR, 0x31)
#define STOP_ARRAY		_IO (MD_MAJOR, 0x32)
#define STOP_ARRAY_RO		_IO (MD_MAJOR, 0x33)
#define RESTART_ARRAY_RW	_IO (MD_MAJOR, 0x34)

typedef struct mdu_version_s {
	int major;
	int minor;
	int patchlevel;
} mdu_version_t;


#define MD_ARRAY_CLEAN			0
#define MD_ARRAY_ERRORS			1
#define MD_ARRAY_RECOVERY_RUNNING	2

typedef struct mdu_array_info_s {
	/*
	 * Generic constant information
	 */
	int major_version;
	int minor_version;
	int patch_version;
	int ctime;
	int level;
	int size;
	int nr_disks;
	int raid_disks;
	int md_minor;
	int not_persistent;

	/*
	 * Generic state information
	 */
	int utime;		/*  0 Superblock update time		      */
	int state;		/*  1 State bits (clean, ...)		      */
	int active_disks;	/*  2 Number of currently active disks	      */
	int working_disks;	/*  3 Number of working disks		      */
	int failed_disks;	/*  4 Number of failed disks		      */
	int spare_disks;	/*  5 Number of spare disks		      */

	/*
	 * Personality information
	 */
	int layout;		/*  0 the array's physical layout	      */
	int chunk_size;	/*  1 chunk size in bytes		      */

} mdu_array_info_t;

typedef struct mdu_disk_info_s {
	/*
	 * configuration/status of one particular disk
	 */
	int number;
	int major;
	int minor;
	int raid_disk;
	int state;

} mdu_disk_info_t;

typedef struct mdu_start_info_s {
	/*
	 * configuration/status of one particular disk
	 */
	int major;
	int minor;
	int raid_disk;
	int state;

} mdu_start_info_t;

typedef struct mdu_param_s
{
	int			personality;	/* 1,2,3,4 */
	int			chunk_size;	/* in bytes */
	int			max_fault;	/* unused for now */
} mdu_param_t;


// md_volume_t flags
#define MD_DISCOVERED		(1 << 0)
#define MD_DEGRADED		(1 << 1)
#define MD_CORRUPT		(1 << 2)
#define MD_PROBLEMATIC_SB	(1 << 3)
#define MD_DIRTY		(1 << 4)
#define MD_ACTIVE		(1 << 5)  // This md volume is active in the kernel
#define MD_SAVED_SB_EXISTS	(1 << 6)	// Indicates that saved SB info block exists
#define MD_USE_OLD_DEV		(1 << 7)	// The user has chosen to restore the original dev maj/minor
#define MD_ARRAY_SYNCING	(1 << 8)	// Resync/recovery is running
#define MD_MP_ACTIVATE_REGION	(1 << 9)	// Multipath region needs to be activated.

// commit_flags
#define MD_COMMIT_USE_DISK	(1 << 0)	// Set this flag in modifies if you want commit
						// to preserve the existing on disk SB.state
#define MD_COMMIT_SAVE_SB	(1 << 1)  	// need to write a backup SB info because the array
						// was created by other MD tools

#define MD_COMMIT_RESTORE_SAVED_SB (1 << 2)	// Set this flag in modifies to restore original SB info
						// per user's request

#define MD_COMMIT_DONT_CHECK_ACTIVE (1 << 3)	// Don't check if the region is active before writing
						// the superblock.

typedef struct md_volume_s {
	storage_object_t * region;
	storage_object_t * child_object[MAX_MD_DEVICES];
        storage_object_t * stale_object[MAX_MD_DEVICES];
	mdp_super_t      * super_array[MAX_MD_DEVICES];
	mdp_saved_super_t * saved_super_array[MAX_MD_DEVICES];
	char               name[128];
	u_int32_t          commit_flag;	 //if set, use SB state from disk, else use from Master SB
	u_int32_t          uuid;

	/* analyze code uses the following fields */
	u_int32_t          nr_disks;
	u_int32_t          active_disks;
	u_int32_t          working_disks;
	u_int32_t          spare_disks;
	u_int32_t          failed_disks;
	u_int32_t          stale_disks;

	u_int32_t          region_mgr_flags;
	u_int32_t          personality;	 // uses the positive defines, not level
	mdp_super_t *      super_block;	 // master superblock
	u_int32_t          flags;
	u_int32_t          daemon_pid;
	struct md_volume_s * next;

	/* commands to perform during SETUP phase of commit process */
	list_anchor_t	   setup_funcs;

	/* ioctl packages to perform during POST_ACTIVATE phase of commit process */
	list_anchor_t      ioctl_pkgs;
	list_anchor_t      ioctl_cleanup;

	void             * private_data;
} md_volume_t;


// Global data used by mdregmgr
extern engine_functions_t   * EngFncs;
extern plugin_record_t      * linear_plugin;
extern plugin_record_t        linear_plugin_record;
extern plugin_record_t      * raid1_plugin;
extern plugin_record_t        raid1_plugin_record;
extern plugin_record_t      * raid0_plugin;
extern plugin_record_t        raid0_plugin_record;
extern plugin_record_t      * raid5_plugin;
extern plugin_record_t        raid5_plugin_record;
extern plugin_record_t      * mp_plugin;
extern plugin_record_t        multipath_plugin_record;
extern plugin_record_t      * my_plugin;
extern md_volume_t          * volume_list_head;	// List of real volume groups, indexed by vg_number.
#define MD_MESSAGE_BUF_SIZE	4096
extern char                   message_buffer[MD_MESSAGE_BUF_SIZE];

extern u_int32_t md_find_first_active_disk(md_volume_t *volume);
extern int md_can_replace_child(storage_object_t *region,
			    storage_object_t *child,
			    storage_object_t *new_child);
extern int md_replace_child(storage_object_t *region,
			    storage_object_t *child,
			    storage_object_t *new_child);
extern int md_ioctl_flush_buffer_cache(storage_object_t *obj);
extern int md_ioctl_start_array(storage_object_t *region, int dev_major, int dev_minor);
extern int md_ioctl_stop_array(storage_object_t *region);
extern int md_ioctl_run_array(storage_object_t *region);
extern int md_ioctl_get_disk_info(storage_object_t *region, mdu_disk_info_t* disk_info);
extern int md_ioctl_get_array_info(storage_object_t *region, mdu_array_info_t* info);
extern int md_ioctl_set_array_info(storage_object_t *region, mdu_array_info_t* info);
extern int md_ioctl_add_new_disk(storage_object_t *region, mdu_disk_info_t* disk_info);
extern int md_ioctl_get_blk_size(storage_object_t *region, u_int64_t *size);
extern int md_ioctl_hot_add_disk(storage_object_t *region, int kdev);
extern int md_ioctl_hot_remove_disk(storage_object_t *region, int kdev);
extern int md_ioctl_set_disk_faulty(storage_object_t *region, int kdev);
extern int md_deactivate_region(storage_object_t * region);
extern int md_activate_region(storage_object_t * region);
extern int md_get_kernel_info(storage_object_t * region, mdu_array_info_t * info);
extern boolean md_is_region_active(storage_object_t *region);
extern boolean md_is_recovery_running(storage_object_t *region);
extern int md_sync_sbs(md_volume_t *vol, mdu_array_info_t *info);
extern boolean md_can_stop_array(storage_object_t *region);
extern int child_to_index(md_volume_t *vol, storage_object_t *obj);
extern int find_disk_in_volume(md_volume_t *vol, int major, int minor);
extern int find_disk_in_active_region(storage_object_t *region, int major, int minor);
extern int find_empty_slot(mdp_super_t *sb);
extern int md_fix_dev_major_minor(md_volume_t * vol, boolean do_msg);
extern boolean follow_up_mark_faulty(md_volume_t *volume, storage_object_t *faulty);
extern inline void md_log_internal_bug(const char *file, const char *func, int line);
#define MD_CHECK_ACTIVE(region)	\
do {				\
	mdu_array_info_t info;	\
	md_get_kernel_info(region, &info);	\
} while (0)



#include "md_discover.h"
#include "md_dlist.h"
#include "md_io.h"
#include "md_create.h"
#include "md_info.h"

#endif

