/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: LVM Region Manager
 * File: evms2/engine/plugins/lvm/lvmregmgr.h
 */ 

/* Datatypes and Defs used in lvmregmgr.c. */
#define SECTOR_SIZE_BITS		9
#define PAGE_SIZE_SECTORS		16	/* for 8k pages. */
#define PE_SIZE_ALIGNMENT_SECTORS	128	/* for 64k aligned PEs. */
#define LVM_DEV_DIRECTORY		"lvm/"
#define EVMS_LVM_PV_REMOVE_IOCTL	0x01
#define EVMS_LVM_SNAPSHOT_STAT_IOCTL	0x02	/* no longer used */
#define EVMS_LVM_MAX_LV_METADATA_SIZE	round_up((MAX_LV+2)*sizeof(lv_disk_t), EVMS_VSECTOR_SIZE) /* in bytes. */
#define LVM_COPY_CHUNK_SIZE_DEFAULT	64 * 1024 / EVMS_VSECTOR_SIZE

/* The following definitions and data structures are from lvm.h and liblvm.h
 * LVM 0.9.1beta8 distribution (Copyright Heinz Mauelshagen and Sistina
 * Software (www.sistina.com)). Since the metadata format changed in beta8,
 * lvm.h (in the Linux kernel source) changed significantly enough that this
 * module would no longer compile. Instead of requiring EVMS users to install
 * the latest lvm release, the required definitions and data structures will
 * now be included in this header file.
 */

/* Absolute limits for VGs, PVs per VG and LVs per VG. */
#ifndef	SECTOR_SIZE
#define SECTOR_SIZE	512
#endif
#ifndef PAGE_SIZE
#define PAGE_SIZE	4096
#endif
#define MAX_VG  	99
#define MAX_LV		256
#define	MAX_PV		256	
#define	NAME_LEN	128
#define	UUID_LEN	32
#define LVM_BLK_MAJOR	58

/* The following are all sector-based. */
#define	LVM_METADATA_VERSION	1
#define	LVM_PE_T_MAX		((1 << (sizeof(uint16_t)* 8)) - 2)
#define	LVM_LV_SIZE_MAX(pe_size) ((long long)LVM_PE_T_MAX * pe_size > (long long) 1024*1024/SECTOR_SIZE*1024*1024 ? (long long) 1024*1024/SECTOR_SIZE*1024*1024 : (long long) LVM_PE_T_MAX * pe_size)
#define LVM_MIN_PV_SIZE		(200L * 1024L / SECTOR_SIZE)		// 200k in sectors
#define	LVM_MIN_PE_SIZE		(8L * 1024L / SECTOR_SIZE)		// 8 kB in sectors
#define	LVM_MAX_PE_SIZE		(16L * 1024L * 1024L / SECTOR_SIZE * 1024L)	// 16 GB in sectors
#define	LVM_DEFAULT_PE_SIZE	(16L * 1024L * 1024L / SECTOR_SIZE)	// 16 MB in sectors
#define	LVM_DEFAULT_STRIPE_SIZE	(16L * 1024L / SECTOR_SIZE)		// 16 kB  in sectors
#define	LVM_MIN_STRIPE_SIZE	(PAGE_SIZE / SECTOR_SIZE)		// PAGESIZE in sectors 
#define	LVM_MAX_STRIPE_SIZE	(512L * 1024 / SECTOR_SIZE)		// 512 kB in sectors
#define	LVM_MAX_STRIPES		128					// max # of stripes
#define	LVM_MAX_SIZE            (1024L * 1024L * 1024L * 1024L / SECTOR_SIZE)	// 1 TB in sectors
#define	LVM_PE_SIZE_PV_SIZE_REL	5					// max relation PV size and PE size
#define	LVM_SNAPSHOT_MAX_CHUNK	(1024L * 1024L / SECTOR_SIZE)		// 1 MB in sectors
#define	LVM_SNAPSHOT_DEF_CHUNK	(64L * 1024L / SECTOR_SIZE)		// 64 kB in sectors
#define	LVM_SNAPSHOT_MIN_CHUNK	(PAGE_SIZE / SECTOR_SIZE)		// PAGESIZE in sectors
#define	LVM_MAX_READ_AHEAD	120					// maximum read ahead sectors

/* The following are all byte-based. */
#define LVM_VGDA_ALIGN		4096UL			// some metadata on the disk need to be aligned
#define	LVM_PV_DISK_BASE	0L			// base of PV structure in disk partition
#define	LVM_PV_DISK_SIZE	1024L			// size reserved for PV structure on disk
#define	LVM_VG_DISK_BASE	round_up(LVM_PV_DISK_BASE + LVM_PV_DISK_SIZE, LVM_VGDA_ALIGN)
							// base of VG structure in disk partition
#define	LVM_VG_DISK_SIZE  	(8*512L)		// size reserved for VG structure
#define	LVM_PV_UUIDLIST_DISK_BASE round_up(LVM_VG_DISK_BASE + LVM_VG_DISK_SIZE, LVM_VGDA_ALIGN)
							// name list of physical volumes on disk
#define	LVM_LV_DISK_OFFSET(a,b)	((a)->lv_on_disk.base + sizeof(lv_disk_t) * b)
#define	LVM_VGDA_SIZE(pv)	((pv)->pe_on_disk.base + (pv)->pe_on_disk.size)
#define	LVM_PE_ALIGN		(65536UL / SECTOR_SIZE)

/* Status flags */

/* vg_disk_t */
#define	VG_ACTIVE            0x01	/* vg_status */
#define	VG_EXPORTED          0x02
#define	VG_EXTENDABLE        0x04

#define	VG_READ              0x01	/* vg_access */
#define	VG_WRITE             0x02
#define	VG_CLUSTERED         0x04
#define	VG_SHARED            0x08

/* lv_disk_t */
#define	LV_ACTIVE            0x01	/* lv_status */
#define	LV_SPINDOWN          0x02

#define	LV_READ              0x01	/* lv_access */
#define	LV_WRITE             0x02
#define	LV_SNAPSHOT          0x04
#define	LV_SNAPSHOT_ORG      0x08

#define	LV_BADBLOCK_ON       0x01	/* lv_badblock */

#define	LV_STRICT            0x01	/* lv_allocation */
#define	LV_CONTIGUOUS        0x02

/* pv_disk_t */
#define	PV_ACTIVE            0x01	/* pv_status */
#define	PV_ALLOCATABLE       0x02	/* pv_allocatable */

/* misc */
#define LVM_SNAPSHOT_DROPPED_SECTOR 1

/**
 * struct lv_COW_table_disk_v1
 *
 * Copy on write tables in disk format.
 */
typedef struct lv_COW_table_disk_v1 {
	uint64_t pv_org_number;
	uint64_t pv_org_rsector;
	uint64_t pv_snap_number;
	uint64_t pv_snap_rsector;
} lv_COW_table_disk_t;

/**
 * struct pe_disk_t
 *
 * Disk stored pe information.
 */
typedef struct pe_disk {
	uint16_t lv_num;
	uint16_t le_num;
} pe_disk_t;

/**
 * struct lvm_disk_data_t
 *
 * Disk stored PV, VG, LV and PE size and offset information.
 */
typedef struct lvm_disk_data {
	uint32_t base;
	uint32_t size;
} lvm_disk_data_t;

/**
 * struct pv_disk_v2
 *
 * PV disk structure.
 */
typedef struct pv_disk_v2 {
	uint8_t id[2];			/* Identifier == HM */
	uint16_t version;		/* LVM version */
	lvm_disk_data_t pv_on_disk;
	lvm_disk_data_t vg_on_disk;
	lvm_disk_data_t pv_uuidlist_on_disk;
	lvm_disk_data_t lv_on_disk;
	lvm_disk_data_t pe_on_disk;
	uint8_t pv_uuid[NAME_LEN];
	uint8_t vg_name[NAME_LEN];
	uint8_t system_id[NAME_LEN];	/* for vgexport/vgimport */
	uint32_t pv_major;
	uint32_t pv_number;
	uint32_t pv_status;
	uint32_t pv_allocatable;
	uint32_t pv_size;
	uint32_t lv_cur;
	uint32_t pe_size;
	uint32_t pe_total;
	uint32_t pe_allocated;
	
	/* New in struct version 2 */
	uint32_t pe_start;	        /* in sectors */

} pv_disk_t;

/**
 * struct lv_disk_v3
 *
 * LV disk structure.
 */
typedef struct lv_disk_v3 {
	uint8_t lv_name[NAME_LEN];
	uint8_t vg_name[NAME_LEN];
	uint32_t lv_access;
	uint32_t lv_status;
	uint32_t lv_open;	
	uint32_t lv_dev;
	uint32_t lv_number;
	uint32_t lv_mirror_copies;
	uint32_t lv_recovery;
	uint32_t lv_schedule;
	uint32_t lv_size;
	uint32_t lv_snapshot_minor;	/* minor number of original */
	uint16_t lv_chunk_size;		/* chunk size of snapshot */
	uint16_t dummy;
	uint32_t lv_allocated_le;
	uint32_t lv_stripes;
	uint32_t lv_stripesize;
	uint32_t lv_badblock;
	uint32_t lv_allocation;
	uint32_t lv_io_timeout;
	uint32_t lv_read_ahead;
} lv_disk_t;

/**
 * struct vg_disk_v2
 *
 * VG disk structure.
 */
typedef struct vg_disk_v2 {
	uint8_t vg_uuid[UUID_LEN];	/* volume group UUID */
	uint8_t vg_name_dummy[NAME_LEN-UUID_LEN];
	uint32_t vg_number;		/* volume group number */
	uint32_t vg_access;		/* read/write */
	uint32_t vg_status;		/* active or not */
	uint32_t lv_max;		/* maximum logical volumes */
	uint32_t lv_cur;		/* current logical volumes */
	uint32_t lv_open;		/* open logical volumes */
	uint32_t pv_max;		/* maximum physical volumes */
	uint32_t pv_cur;		/* current physical volumes */
	uint32_t pv_act;		/* active physical volumes */
	uint32_t dummy;
	uint32_t vgda;			/* volume group descriptor arrays */
	uint32_t pe_size;		/* physical extent size in sectors  */
	uint32_t pe_total;		/* total of physical extents */
	uint32_t pe_allocated;		/* allocated physical extents */
	uint32_t pvg_total;		/* physical volume groups */
} vg_disk_t;

/* Useful inlines
 * The "size" paramaters MUST be power-of-2.
 */
static inline ulong round_down(ulong n, ulong size) {
	size--;
	return(n & ~size);
}

static inline ulong round_up(ulong n, ulong size) {
	size--;
	return((n + size) & ~size);
}

static inline ulong div_up(ulong n, ulong size) {
	return(round_up(n, size) / size);
}

/* End of lvm.h/liblvm.h imported data structures. */


/**
 * struct lvm_physical_extent
 * @pv:		PV on which this PE resides.
 * @pe:		Copy of on-disk metadata for this PE.
 * @number:	Index of this PE within the PV.
 * @sector:	Starting sector of this PE within the PV.
 * @le:		The logical extent currently mapped to this PE.
 * @new_le:	The LE being moved onto this PE.
 *
 * Physical Extents (PEs)
 */
typedef struct lvm_physical_extent {
	struct lvm_physical_volume	* pv;
	struct pe_disk			pe;
	u_int32_t			number;
	u_int64_t			sector;
	struct lvm_logical_extent	* le;
	struct lvm_logical_extent	* new_le;
} lvm_physical_extent_t;

/**
 * struct lvm_physical_volume
 * @pv:			Copy of LVM on-disk PV struct.
 * @segment:		EVMS segment representing this PV.
 * @group:		Pointer back to the parent volume group.
 * @pe_map:		Entire map of PEs on this PV.
 * @move_extents:	Number of PEs that are the target of a move.
 * @number:		Number of this PV (pv->pv_number)
 * @flags:		LVM_PV_FLAG_*
 *
 * Entries in the array of physical volumes (PV)
 * in a volume group (VG)
 */
typedef struct lvm_physical_volume {
	pv_disk_t		* pv;
	storage_object_t	* segment;
	struct lvm_volume_group	* group;
	lvm_physical_extent_t	* pe_map;
	u_int32_t		move_extents;
	long			number;
	long			flags;       
} lvm_physical_volume_t;

#define LVM_PV_FLAG_INVALID_UUID	(1 << 0)
#define LVM_PV_FLAG_LV_CUR_UPDATED	(1 << 1)

/**
 * struct lvm_logical_extent
 * @volume:	The volume in which this LE resides.
 * @number:	Index of this LE within the volume.
 * @pe:		The PE that this LE maps to.
 * @new_pe:	The PE that this LE will be moving to.
 *
 * Table for mapping logical extents (LE) to physical extents (PE)
 * This structure is used within the lvm_logical_volume_t
 */
typedef struct lvm_logical_extent {
	struct lvm_logical_volume	* volume;
	u_int32_t			number;
	struct lvm_physical_extent	* pe;
	struct lvm_physical_extent	* new_pe;
	copy_job_t			* copy_job;
} lvm_logical_extent_t;

/**
 * struct lvm_logical_volume
 * @lv:		The corresponding entry in the group's lv_array.
 * @region:	The EVMS region representing this volume.
 * @group:	Pointer back to the parent volume group.
 * @le_map:	Mapping of logical-to-physical extents.
 * @number:	The number of this volume (lv->lv_number + 1)
 * @minor:	MINOR(lv->lv_dev)
 * @flags:	LVM_LV_FLAG_*
 *
 * Logical volumes (LV) in a volume group (VG)
 */
typedef struct lvm_logical_volume {
	lv_disk_t		* lv;
	storage_object_t	* region;
	struct lvm_volume_group	* group;
	lvm_logical_extent_t	* le_map;
	int			number;
	int			minor;
	int			flags;
} lvm_logical_volume_t;

#define LVM_LV_FLAG_EXPORTED		(1 << 0)
#define LVM_LV_FLAG_INCOMPLETE		(1 << 1)
#define LVM_LV_FLAG_DIRTY		(1 << 2)
#define LVM_LV_FLAG_MOVE_PENDING	(1 << 3)

/**
 * struct lvm_volume_group
 * @vg:			Copy of the LVM on-disk VG struct.
 * @container:		The EVMS storage container representing this group.
 * @pv_list:		Array of physical volumes that make up this group.
 * @uuid_list:		Array of UUIDs of the PVs in this group.
 * @volume_list:	Array of logical volumes exported from this group.
 * @lv_array:		The entire array of LVs stored on each PV in this group.
 * @freespace:		The volume representing the freespace in this group.
 * @geometry:		Keep track of max sector and block sizes for this group.
 * @move_extents:	Number of extents waiting to be moved in this group.
 * @pv_count:		Number of PVs/segments found on this group.
 * @volume_count:	Number of LVs/regions found on this group.
 * @flags:		LVM_VG_FLAG_*
 *
 * Volume groups (VG)
 */
typedef struct lvm_volume_group {
	vg_disk_t		* vg;
	storage_container_t	* container;
	lvm_physical_volume_t	* pv_list[MAX_PV+1];
	unsigned char		* uuid_list[MAX_PV+1];
	lvm_logical_volume_t	* volume_list[MAX_LV+1];
	lv_disk_t		* lv_array;
	lv_disk_t		* lv_array_disk;
	lvm_logical_volume_t	* freespace;
	geometry_t		geometry;
	u_int32_t		move_extents;
	int			pv_count;
	int			volume_count;
	int			flags;
} lvm_volume_group_t;

#define LVM_VG_FLAG_UUID_LIST_PRESENT	(1 << 0)
#define LVM_VG_FLAG_LV_LIST_PRESENT	(1 << 1)
#define LVM_VG_FLAG_INVALID_VG_NUMBER	(1 << 2)

/* Global data used by lvmregmgr. */
extern engine_functions_t	* EngFncs;
extern plugin_record_t		* my_plugin_record;
extern list_anchor_t		lvm_group_list;

/* These need to follow all of the structure definitions. */
#include "lvm_discover.h"
#include "lvm_groups.h"
#include "lvm_info.h"
#include "lvm_io.h"
#include "lvm_list.h"
#include "lvm_move.h"
#include "lvm_names.h"
#include "lvm_options.h"
#include "lvm_pv.h"
#include "lvm_uuid.h"
#include "lvm_volumes.h"

