/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: LVM Region Manager
 * File: evms2/engine/plugins/lvm/lvm_volumes.c
 *
 * Description: This file contains all functions related to the discovery,
 *              creation and management of logical volumes in the LVM region
 *              manager.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "lvmregmgr.h"

/* Array for keeping track of which minor numbers are used by LVM LVs. */
static int minor_in_use[MAX_LV] = {FALSE};


/****** Volume Memory Allocation/Deallocation Functions ******/


/**
 * lvm_allocate_le_map
 *
 * Allocates memory to hold a volume's LE-to-PE mapping. The
 * lv_allocated_le field of the lv must be filled in.
 **/
static int lvm_allocate_le_map(lvm_logical_volume_t * volume)
{
	u_int32_t i;
	int rc = 0;

	LOG_ENTRY();

	/* A freespace region could potentially have no allocated LEs. */
	if (volume->lv->lv_allocated_le == 0) {
		volume->le_map = NULL;
		goto out;
	}

	volume->le_map = EngFncs->engine_alloc(volume->lv->lv_allocated_le *
					       sizeof(lvm_logical_extent_t));
	if (!volume->le_map) {
		rc = ENOMEM;
		goto out;
	}

	/* Initialize the constant fields in the LE map. */
	for (i = 0; i < volume->lv->lv_allocated_le; i++) {
		volume->le_map[i].volume = volume;
		volume->le_map[i].number = i;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_deallocate_le_map
 *
 * Releases memory used for a volume's LE-to-PE mapping.
 **/
static void lvm_deallocate_le_map(lvm_logical_volume_t * volume)
{
	LOG_ENTRY();

	if (volume->le_map) {
		EngFncs->engine_free(volume->le_map);
		volume->le_map = NULL;
	}

	LOG_EXIT_VOID();
}

/**
 * lvm_allocate_logical_volume
 *
 * Allocate all necessary memory structures for a new logical volume.
 * Initialize based on the lv and group parameters.
 **/
lvm_logical_volume_t * lvm_allocate_logical_volume(lv_disk_t * lv,
						   lvm_volume_group_t * group)
{
	lvm_logical_volume_t * new_volume;
	char region_name[EVMS_NAME_SIZE+1] = {0};
	char * dg_name = group->container->disk_group ?
			 group->container->disk_group->name : NULL;
	int rc;

	LOG_ENTRY();

	/* Allocate the logical volume itself. */
	new_volume = EngFncs->engine_alloc(sizeof(lvm_logical_volume_t));
	if (!new_volume) {
		LOG_CRITICAL("Memory error creating new logical volume %s.\n",
			     lv->lv_name);
		goto out;
	}

	/* Initialize the basic fields. */
	new_volume->lv		= lv;
	new_volume->group	= group;
	new_volume->number	= lv->lv_number + 1;	/* Need the +1 to match the PE Map entries on the PV. */
	new_volume->minor	= minor(lv->lv_dev);
	new_volume->flags	= LVM_LV_FLAG_DIRTY;

	/* Space for the LE-to-PE mapping table. */
	rc = lvm_allocate_le_map(new_volume);
	if (rc) {
		goto out2;
	}

	/* Translate the LV name to the region name. Should have checked for
	 * uniqueness when initializing the LV.
	 */
	rc = lvm_translate_lv_name_to_region_name(lv->lv_name,
						  dg_name, region_name);
	if (rc) {
		goto out2;
	}

	/* An EVMS region to represent this volume. Add this new region
	 * to the "produced" list in the container.
	 */
	rc = EngFncs->allocate_region(region_name, &new_volume->region);
	if (rc) {
		goto out2;
	}

	rc = lvm_append_region_to_container(new_volume->region, group->container);
	if (rc) {
		goto out2;
	}

	snprintf(new_volume->region->uuid, EVMS_NAME_SIZE, "lvm-%s-%d",
		 group->vg->vg_uuid, new_volume->lv->lv_number);
	new_volume->region->object_type		= REGION;
	new_volume->region->data_type		= DATA_TYPE;
	new_volume->region->plugin		= my_plugin_record;
	new_volume->region->flags		= (!(lv->lv_access & LV_WRITE)) ? SOFLAG_READ_ONLY : 0;
	new_volume->region->size		= lv->lv_size;
	new_volume->region->geometry		= group->geometry;
	new_volume->region->private_data	= new_volume;

	minor_in_use[new_volume->minor]		= TRUE;

	LOG_DEFAULT("Created region %s\n", new_volume->region->name);
	goto out;

out2:
	lvm_deallocate_logical_volume(new_volume);
	new_volume = NULL;
	
out:
	LOG_EXIT_PTR(new_volume);
	return new_volume;
}

/**
 * lvm_deallocate_logical_volume
 *
 * Free all memory usage for this volume. Remove it from its group's
 * list of volumes if necessary.
 **/
void lvm_deallocate_logical_volume(lvm_logical_volume_t * volume)
{
	lvm_volume_group_t * group = volume->group;
	storage_object_t * region = volume->region;

	LOG_ENTRY();

	/* Delete the LE map. */
	lvm_deallocate_le_map(volume);

	/* Remove the region from the container and delete. */
	if (region) {
		lvm_clear_child_list(region);
		lvm_remove_region_from_container(region);
		EngFncs->free_region(region);
		volume->region = NULL;
	}

	/* Only the freespace volume has an lv field that was allocated. All
	 * other volumes point to entries in the group's lv_array.
	 */
	if (volume->number == 0 && volume->lv) {
		EngFncs->engine_free(volume->lv);
	}

	/* Remove this volume from the group's list. */
	if (group && group->volume_list[volume->number] == volume) {
		group->volume_list[volume->number] = NULL;
		group->volume_count--;
	}

	/* Delete the volume itself. */
	minor_in_use[volume->minor]	= FALSE;
	volume->number			= 0;
	volume->minor			= 0;
	volume->flags			= 0;
	volume->group			= NULL;
	volume->lv			= NULL;

	EngFncs->engine_free(volume);

	LOG_EXIT_VOID();
}

/**
 * lvm_mark_volume_dirty
 *
 * When an LV is newly created or modified, it will need to be marked
 * dirty to indicate it requires some special processing. This will
 * include deleting all associations to its underlying segments, and
 * setting the dirty flag so its LE maps will be rebuilt.
 **/
static void lvm_mark_volume_dirty(lvm_logical_volume_t * volume)
{
	LOG_ENTRY();

	lvm_clear_child_list(volume->region);
	volume->flags |= LVM_LV_FLAG_DIRTY;

	LOG_EXIT_VOID();
}

/**
 * lvm_create_freespace_volume
 *
 * Every group must have one volume that represents the unused space
 * in that group. This is done by having a "dummy" volume that is
 * separate from the group's regular volume list. The size of the
 * freespace volume is determined by subtracting the number of allocated
 * PEs in the entire group from the total number of PEs in the group. Then
 * when the LE maps are built, any PE found with an LV number of zero will
 * be added to the LE map of the freespace volume.
 **/
int lvm_create_freespace_volume(lvm_volume_group_t * group)
{
	lv_disk_t * free_lv;
	lvm_logical_volume_t * free_volume;
	int rc = 0;

	LOG_ENTRY();

	if (group->freespace) {
		goto out;
	}

	/* Create a dummy lv_disk_t structure and initialize. */
	free_lv = EngFncs->engine_alloc(sizeof(lv_disk_t));
	if (!free_lv) {
		LOG_CRITICAL("Memory error creating LV structure for Freespace region for container %s.\n",
			     group->container->name);
		rc = ENOMEM;
		goto out;
	}

	/* Need to make the freespace's name look like a legal LVM lv name. */
	lvm_make_lv_name("Freespace", group, free_lv->lv_name);

	free_lv->lv_allocated_le = group->vg->pe_total - group->vg->pe_allocated - group->move_extents;
	free_lv->lv_size	= free_lv->lv_allocated_le * group->vg->pe_size;
	free_lv->lv_number	= -1;

	/* Create the volume structure. */
	free_volume = lvm_allocate_logical_volume(free_lv, group);
	if (!free_volume) {
		LOG_CRITICAL("Memory error creating region %s.\n",
			     free_lv->lv_name);
		rc = ENOMEM;
		goto out;
	}
	free_volume->region->data_type = FREE_SPACE_TYPE;

	group->freespace = free_volume;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_update_freespace_volume
 *
 * Update the freespace volume after a change in the volume group. Reset
 * the freespace size accordingly, delete and recreate the LE map, and
 * rebuild the LE maps for the group.
 **/
int lvm_update_freespace_volume(lvm_volume_group_t * group)
{
	lvm_logical_volume_t * freespace = group->freespace;
	int rc;

	LOG_ENTRY();

	freespace->lv->lv_allocated_le	= group->vg->pe_total - group->vg->pe_allocated - group->move_extents;
	freespace->lv->lv_size		= freespace->lv->lv_allocated_le * group->vg->pe_size;
	freespace->region->size		= freespace->lv->lv_size;

	lvm_mark_volume_dirty(freespace);

	lvm_deallocate_le_map(freespace);
	rc = lvm_allocate_le_map(freespace);
	if (rc) {
		LOG_CRITICAL("Memory error creating LE map for region %s\n",
			     freespace->region->name);
	} else {
		lvm_build_le_maps(group);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_update_expanded_volume
 **/
int lvm_update_expanded_volume(lvm_logical_volume_t * volume,
			       lvm_lv_expand_options_t * lv_opts)
{
	int rc;

	LOG_ENTRY();

	volume->lv->lv_allocated_le += lv_opts->add_extents;
	volume->lv->lv_size += lv_opts->add_size;
	volume->region->size = volume->lv->lv_size;

	lvm_mark_volume_dirty(volume);

	lvm_deallocate_le_map(volume);
	rc = lvm_allocate_le_map(volume);
	if (rc) {
		LOG_CRITICAL("Memory error creating LE map for region %s\n",
			     volume->region->name);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_update_shrunk_volume
 **/
int lvm_update_shrunk_volume(lvm_logical_volume_t * volume,
			     u_int32_t remove_extents)
{
	int rc;

	LOG_ENTRY();

	volume->lv->lv_allocated_le -= remove_extents;
	volume->lv->lv_size -= remove_extents * volume->group->vg->pe_size;
	volume->region->size = volume->lv->lv_size;

	lvm_mark_volume_dirty(volume);

	lvm_deallocate_le_map(volume);
	rc = lvm_allocate_le_map(volume);
	if (rc) {
		LOG_CRITICAL("Memory error creating LE map for region %s\n",
			     volume->region->name);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_build_le_maps
 *
 * After all logical volumes have been discovered, the mappings from
 * logical extents to physical extents must be constructed. Each PV
 * contains an on-disk-map of its PEs. Each PE map entry contains the
 * logical volume number and the logical extent number on that volume.
 * The maps in the logical volumes are the reverse. Each entry contains
 * a pointer to an entry in the group's pv_list, and a sector offset of
 * the PE that the LE maps to.
 * In the PE map, an LV number of zero indicates an unused PE. This PE
 * will be added as an LE for the freespace volume. A running count is
 * kept of the current LE for the freespace volume to figure out where
 * each free PE should be mapped.
 **/
void lvm_build_le_maps(lvm_volume_group_t * group)
{
	pv_disk_t * pv;
	lvm_physical_volume_t * pv_entry;
	lvm_physical_extent_t * pe_map;
	u_int32_t lv_number, le_number;
	u_int32_t freespace_current_le = 0;
	int i, j;

	LOG_ENTRY();

	LOG_DETAILS("Building LE maps for container %s\n", group->container->name);

	/* For every PV in this group. */
	for (i = 1; i <= MAX_PV; i++) {
		pv_entry = group->pv_list[i];
		if (!pv_entry) {
			continue;
		}
		pv	= pv_entry->pv;
		pe_map	= pv_entry->pe_map;

		/* For every entry in the PE map, update the correct LV's LE
		 * map. LV number of 0 marks an unused PE. Also, use this
		 * opportunity to set up the child and parent lists in the
		 * segments and regions.
		 */
		for (j = 0; j < pv->pe_total; j++) {
			lv_number = pe_map[j].pe.lv_num;
			if (lv_number) {
				if (group->volume_list[lv_number]) {
					le_number = pe_map[j].pe.le_num;
					group->volume_list[lv_number]->le_map[le_number].pe = &(pe_map[j]);
					pe_map[j].le = &(group->volume_list[lv_number]->le_map[le_number]);
					lvm_append_region_to_segment(group->volume_list[lv_number]->region,
								     pv_entry->segment);
				} else {
					/* Can't find the volume. */
					pe_map[j].le = NULL;
				}

				/* If this PE is already mapped to a volume, it
				 * better not be the target of a move.
				 */
				pe_map[j].new_le = NULL;
			} else  {
				if (pe_map[j].new_le) {
					/* This PE is the target of a move.
					 * Don't map it anywhere.
					 */
					pe_map[j].le = NULL;
				} else {
					/* Map this PE to the freespace. */
					if (freespace_current_le >= group->freespace->lv->lv_allocated_le) {
						LOG_SERIOUS("Found a free PE, but the freespace map is full!\n");
						LOG_SERIOUS("Container %s, PV %s, PE %d\n",
							    group->container->name,
							    pv_entry->segment->name, j);
					} else {
						group->freespace->le_map[freespace_current_le].pe = &(pe_map[j]);
						pe_map[j].le = &(group->freespace->le_map[freespace_current_le]);
						lvm_append_region_to_segment(group->freespace->region,
									     pv_entry->segment);
						freespace_current_le++;
					}
				}
			}
		}
	}

	LOG_EXIT_VOID();
}

/**
 * lvm_check_le_maps
 **/
void lvm_check_le_maps(lvm_volume_group_t * group,
		       boolean final_call)
{
	lvm_logical_volume_t * volume;
	int i, j, count;

	LOG_ENTRY();
	LOG_DETAILS("Verifying LE maps for container %s.\n",
		    group->container->name);

	for (i = 1; i <= MAX_LV; i++) {
		volume = group->volume_list[i];
		if (!volume) {
			continue;
		}

		for (j = 0, count = 0; j < volume->lv->lv_allocated_le; j++) {
			if (!volume->le_map[j].pe) {
				count++;
			}
		}
		if (count) {
			if (final_call) {
				MESSAGE("Region %s has an incomplete LE map.\n"
					"Missing %d out of %d LEs.\n",
					volume->region->name, count,
					volume->lv->lv_allocated_le);
			}
			volume->flags |= LVM_LV_FLAG_INCOMPLETE;
		} else {
			volume->flags &= (~LVM_LV_FLAG_INCOMPLETE & ~LVM_LV_FLAG_DIRTY);
		}
	}

	LOG_EXIT_VOID();
}

/**
 * lvm_compare_volume_targets
 *
 * Compare the engine's current view of the mapping for this region with
 * the kernel's current view. If they are different, mark the region as
 * needing to be activated.
 **/
static void lvm_compare_volume_targets(lvm_logical_volume_t * volume,
				       boolean final_call)
{
	dm_target_t *dm_target_list = NULL, *lvm_target_list = NULL;
	dm_target_t *dm_target, *lvm_target;
	dm_device_t *dm_linear, *lvm_linear;
	dm_target_stripe_t *dm_stripe = NULL, *lvm_stripe = NULL;
	u_int32_t i, num_stripes;
	int rc, needs_reactivate = TRUE;

	LOG_ENTRY();

	/* Get the kernel's mapping for this region. */
	rc = EngFncs->dm_get_targets(volume->region, &dm_target_list);
	if (rc) {
		LOG_ERROR("Error getting kernel mapping for region %s.\n",
			  volume->region->name);
		goto compare_done;
	}

	/* Get the engine's mapping for this region. */
	lvm_target_list = lvm_build_volume_targets(volume);
	if (!lvm_target_list) {
		LOG_ERROR("Error building current mapping for region %s.\n",
			  volume->region->name);
		goto compare_done;
	}

	/* Compare the kernel's mapping with the current mapping. */
	for (dm_target = dm_target_list, lvm_target = lvm_target_list;
	     dm_target && lvm_target;
	     dm_target = dm_target->next, lvm_target = lvm_target->next) {
		num_stripes = 1;

		if (dm_target->start != lvm_target->start ||
		    dm_target->length != lvm_target->length ||
		    dm_target->type != lvm_target->type) {
			goto compare_done;
		}

		if (dm_target->type == DM_TARGET_ERROR) {
			/* "Error" targets have no more information to check. */
			continue;
		}

		if (dm_target->type == DM_TARGET_STRIPE) {
			dm_stripe = dm_target->data.stripe;
			lvm_stripe = lvm_target->data.stripe;
			if (dm_stripe->num_stripes != lvm_stripe->num_stripes ||
			    dm_stripe->chunk_size != lvm_stripe->chunk_size) {
				goto compare_done;
			}
			num_stripes = dm_stripe->num_stripes;
		}

		for (i = 0; i < num_stripes; i++) {
			dm_linear = (dm_target->type == DM_TARGET_STRIPE) ?
				    &dm_stripe->devices[i] : dm_target->data.linear;
			lvm_linear = (lvm_target->type == DM_TARGET_STRIPE) ?
				     &lvm_stripe->devices[i] : lvm_target->data.linear;
			if (dm_linear->major != lvm_linear->major ||
			    dm_linear->minor != lvm_linear->minor ||
			    dm_linear->start != lvm_linear->start) {
				goto compare_done;
			}
		}
	}

	if (!dm_target && !lvm_target) {
		needs_reactivate = FALSE;
	}

	if ((volume->lv->lv_access & LV_WRITE) &&
	    (volume->region->flags & SOFLAG_READ_ONLY) &&
	    !(volume->flags & LVM_LV_FLAG_INCOMPLETE)) {
		/* If the kernel has this region as read-only, but all missing
		 * extents have been fixed, we need to reactivate to make the
		 * kernel device read-write (even though it may have the same
		 * mapping).
		 */
		volume->region->flags &= ~SOFLAG_READ_ONLY;
		needs_reactivate = TRUE;
	}

compare_done:
	if (final_call && needs_reactivate) {
		MESSAGE("Error comparing kernel mappings for active region %s.\n",
			volume->region->name);
		MESSAGE("Region %s will be reactivated at the next commit.\n",
			volume->region->name);
		volume->region->flags |= SOFLAG_NEEDS_ACTIVATE;
	}

	EngFncs->dm_deallocate_targets(dm_target_list);
	EngFncs->dm_deallocate_targets(lvm_target_list);

	LOG_EXIT_VOID();
}

/**
 * lvm_check_for_active_volumes
 **/
void lvm_check_for_active_volumes(lvm_volume_group_t * group,
				  boolean final_call)
{
	lvm_logical_volume_t * volume;
	int i, rc;

	LOG_ENTRY();
	LOG_DETAILS("Checking for active regions in container %s.\n",
		    group->container->name);

	for (i = 1; i <= MAX_LV; i++) {
		volume = group->volume_list[i];
		if (!volume) {
			continue;
		}

		/* Check if this region is active in the kernel. */
		rc = EngFncs->dm_update_status(volume->region);
		if (rc) {
			LOG_ERROR("Error checking status for region %s.\n",
				  volume->region->name);
			continue;
		}

		if (!(volume->region->flags & SOFLAG_ACTIVE)) {
			volume->region->flags |= SOFLAG_NEEDS_ACTIVATE;
			continue;
		}

		lvm_compare_volume_targets(volume, final_call);
	}

	LOG_EXIT_VOID();
}

/**
 * lvm_export_logical_volumes
 *
 * Find any new LVM logical volumes and place them on the discovery
 * output list. If the final_call flag is set, also place any incomplete
 * volumes on the discovery list, and mark them as read-only.
 **/
int lvm_export_logical_volumes(list_anchor_t regions,
			       boolean final_call)
{
	lvm_volume_group_t * group;
	lvm_logical_volume_t * volume;
	list_element_t itr, itr2;
	int count = 0;
	int i;

	LOG_ENTRY();

	/* Examine every volume group. */
	LIST_FOR_EACH(lvm_group_list, itr, group) {

		/* First export the freespace volumes. */
		volume = group->freespace;
		if (! (volume->flags & LVM_LV_FLAG_EXPORTED)) {
			itr2 = EngFncs->insert_thing(regions, volume->region,
						     INSERT_AFTER, NULL);
			if (itr2) {
				volume->flags |= LVM_LV_FLAG_EXPORTED;
				count++;
				LOG_DEFAULT("Exporting region %s\n",
					    volume->region->name);
			}
		}

		/* Now examine every regular LV in this group. */
		for (i = 0; i <= MAX_LV; i++) {
			volume = group->volume_list[i];

			/* Only export volumes without the EXPORTED flag set.
			 * Only export incomplete volumes on final discovery.
			 */
			if (!volume ||
			    volume->flags & LVM_LV_FLAG_EXPORTED ||
			    (!final_call && volume->flags & LVM_LV_FLAG_INCOMPLETE)) {
				continue;
			}

			itr2 = EngFncs->insert_thing(regions, volume->region,
						     INSERT_AFTER, NULL);
			if (!itr2) {
				continue;
			}

			volume->flags |= LVM_LV_FLAG_EXPORTED;
			count++;

			if (volume->flags & LVM_LV_FLAG_INCOMPLETE) {
				volume->region->flags |= SOFLAG_READ_ONLY;
			}

			LOG_DEFAULT("Exporting region %s\n",
				    volume->region->name);
		}
	}

	LOG_EXIT_INT(count);
	return count;
}

/**
 * lvm_get_freespace_volume
 *
 * Get the first (and only) item from the input list, which should be
 * the freespace region specified for the create. Verify that it is
 * owned by LVM and is actually freespace.
 **/
int lvm_get_freespace_volume(list_anchor_t freespace_region_list,
			     lvm_logical_volume_t ** freespace_volume)
{
	storage_object_t * freespace_region;
	lvm_logical_volume_t * free_vol;
	int count;
	int rc = 0;

	LOG_ENTRY();

	/* Extract the specified freespace region from the input list. There
	 * must be only a single item on this list.
	 */
	count = EngFncs->list_count(freespace_region_list);
	if (count <= 0) {
		LOG_ERROR("Must specify exactly one freespace region\n");
		rc = EINVAL;
		goto out;
	}

	freespace_region = EngFncs->first_thing(freespace_region_list, NULL);

	/* The selected object must be owned by LVM,
	 * and must be a freespace region.
	 */
	if (freespace_region->plugin != my_plugin_record) {
		LOG_ERROR("Region %s does not belong to LVM\n",
			  freespace_region->name);
		rc = EINVAL;
		goto out;
	}

	free_vol = freespace_region->private_data;
	if (free_vol->group->freespace != free_vol) {
		LOG_ERROR("Region %s is not a freespace region\n",
			  freespace_region->name);
		rc = EINVAL;
		goto out;
	}

	*freespace_volume = free_vol;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_check_lv_name
 *
 * Run through the specified volume group and check if the specified LV
 * name is already in use. It is assumed that "name" is ONLY the volume
 * name (no lvm/group_name).
 **/
int lvm_check_lv_name(char * name,
		      lvm_volume_group_t * group)
{
	char vg_name[NAME_LEN] = {0};
	char lv_name[NAME_LEN] = {0};
	int i, rc = 0;

	LOG_ENTRY();

	/* No empty strings allowed. */
	if (name[0] == 0) {
		LOG_ERROR("Must specify a name for the new region\n");
		rc = EINVAL;
		goto out;
	}

	/* Convert "name" to "/dev/vg_name/name". */
	lvm_translate_container_name_to_vg_name(group, vg_name);
	snprintf(lv_name, NAME_LEN, "/dev/%s/%s", vg_name, name);

	/* Search through all LVs in this group. */
	for (i = 1; i <= MAX_LV; i++) {
		if (group->volume_list[i] &&
		    ! strncmp(lv_name, group->volume_list[i]->lv->lv_name, NAME_LEN)) {
			MESSAGE("LV name %s already exists in container %s\n",
				name, group->container->name);
			rc = EEXIST;
			goto out;
		}
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_check_lv_size
 *
 * Make sure that lv_size is a multiple of the group's PE size. If it
 * isn't, round up to the next PE size multiple.
 **/
int lvm_check_lv_size(u_int32_t * lv_size,
		      u_int32_t pe_size)
{
	int rc = 0;

	LOG_ENTRY();

	if (*lv_size % pe_size) {
		LOG_WARNING("LV Size (%d) is not a multiple of the PE size %d\n",
			    *lv_size, pe_size);
		*lv_size = round_up(*lv_size, pe_size);
		LOG_WARNING("Rounding LV Size up to %d\n", *lv_size);
		rc = -1;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_compare_lv_size_and_extents
 *
 * Verify that the specified lv_size and extents match. If one is not
 * set, calculate it from the other. If neither are set, return error.
 *
 * The lvm_check_lv_size function should be called before this to
 * make sure lv_size is a multiple of the group's pe_size.
 **/
int lvm_compare_lv_size_and_extents(u_int32_t * lv_size,
				    u_int32_t * extents,
				    u_int32_t pe_size)
{
	int rc = 0;

	LOG_ENTRY();

	if (*lv_size) {
		if (*extents) {
			if (*lv_size != *extents * pe_size) {
				LOG_ERROR("Mismatch in LV Size and Extents\n");
				LOG_ERROR("LV Size: %d\n", *lv_size);
				LOG_ERROR("Extents: %d (total size: %d)\n",
					  *extents, *extents * pe_size);
				LOG_ERROR("Please specify only LV Size or only Extents\n");
				rc = EINVAL;
			}
		} else {
			*extents = *lv_size / pe_size;
		}
	} else if (*extents) {
		*lv_size = *extents * pe_size;
	} else {
		LOG_ERROR("Must specify either LV Size or Extents\n");
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_check_stripe_size
 *
 * During creation of a striped LV, the user-specified stripe size must
 * be a power of 2 betweeen 8k and 512k, and also less than the PE size
 * for that group. If it is out of range, reset to the nearest correct
 * value. If it is not a power of 2, round down until a suitable value
 * is found.
 **/
void lvm_check_stripe_size(u_int32_t * stripe_size,
			   u_int32_t pe_size)
{
	unsigned long mask = 1;

	LOG_ENTRY();

	if (*stripe_size) {
		if (*stripe_size < LVM_MIN_STRIPE_SIZE) {
			/* Check lower bound. */
			LOG_WARNING("Stripe size %d is below lower limit.\n", *stripe_size);
			*stripe_size = LVM_MIN_STRIPE_SIZE;
			LOG_WARNING("Resetting stripe size to %d.\n", *stripe_size);
		} else if (*stripe_size > LVM_MAX_STRIPE_SIZE) {
			/* Check upper bound. */
			LOG_WARNING("Stripe size %d is above upper limit.\n", *stripe_size);
			*stripe_size = LVM_MAX_STRIPE_SIZE;
			LOG_WARNING("Resetting stripe size to %d.\n", *stripe_size);
		} else if ((*stripe_size & (*stripe_size - 1))) {
			/* Check that stripe_size is a power of 2. */
			LOG_WARNING("Stripe size %d not a power of 2.\n", *stripe_size);
			while ((*stripe_size & (*stripe_size - 1))) {
				*stripe_size =  *stripe_size & ~mask;
				mask = mask << 1;
			}
			LOG_WARNING("Rounding stripe size down to %d.\n", *stripe_size);
		}
	} else {
		/* Default stripe-size is 16k. */
		*stripe_size = LVM_DEFAULT_STRIPE_SIZE;
	}

	/* Check against PE size. */
	if (*stripe_size > pe_size) {
		LOG_WARNING("Stripe size %d is greater than PE size %d.\n", *stripe_size, pe_size);
		*stripe_size = pe_size;
		LOG_WARNING("Resetting stripe size to %d.\n", *stripe_size);
	}

	LOG_EXIT_VOID();
}

/**
 * lvm_find_free_lv_number
 *
 * Search through the list of LVs in this group and return the first
 * unused LV number.
 **/
static int lvm_find_free_lv_number(lvm_volume_group_t * group)
{
	int i;

	LOG_ENTRY();

	for (i = 1; i <= MAX_LV; i++) {
		if (! group->volume_list[i]) {
			LOG_EXIT_INT(i);
			return i;
		}
	}

	LOG_ERROR("Container %s has maximum number of regions.\n",
		  group->container->name);
	LOG_EXIT_INT(-1);
	return -1;
}

/**
 * lvm_find_free_minor_number
 *
 * Determine the next minor number that is available for an LV.
 **/
static int lvm_find_free_minor_number(void)
{
	int minor = 0;

	LOG_ENTRY();

	for (minor = 0; minor < MAX_LV; minor++) {
		if (!minor_in_use[minor]) {
			LOG_EXIT_INT(minor);
			return minor;
		}
	}

	LOG_ERROR("All LVM minor numbers in use.\n");
	LOG_EXIT_INT(-1);
	return -1;
}

/**
 * lvm_initialize_new_lv
 *
 * During creation of a new logical volume, we need to fill in an LV disk
 * structure with appropriate values based on the user's input.
 **/
int lvm_initialize_new_lv(lvm_lv_create_options_t * lv_opts,
			  lvm_volume_group_t * group,
			  lv_disk_t ** lv)
{
	int lv_number, minor;
	int rc = 0;

	LOG_ENTRY();

	/* Get an LV number. */
	lv_number = lvm_find_free_lv_number(group);
	if (lv_number <= 0) {
		LOG_ERROR("Could not initialize LV metadata\n");
		rc = ENOSPC;
		goto out;
	}

	/* Get a minor number. */
	minor = lvm_find_free_minor_number();
	if (minor < 0) {
		LOG_ERROR("Could not initialize LV metadata\n");
		rc = ENOSPC;
		goto out;
	}

	*lv = &(group->lv_array[lv_number - 1]);
	lvm_clear_lv(*lv);

	/* Copy the LV and VG names. */
	lvm_make_lv_name(lv_opts->lv_name, group, (*lv)->lv_name);
	lvm_translate_container_name_to_vg_name(group, (*lv)->vg_name);

	/* Fill in remaining fields. */
	(*lv)->lv_access		= LV_READ | LV_WRITE;
	(*lv)->lv_status		= LV_ACTIVE;
	(*lv)->lv_open			= 0;
	(*lv)->lv_dev			= makedev(LVM_BLK_MAJOR, minor);
	(*lv)->lv_number		= lv_number - 1;
	(*lv)->lv_mirror_copies		= 0;
	(*lv)->lv_recovery		= 0;
	(*lv)->lv_schedule		= 0;
	(*lv)->lv_size			= lv_opts->lv_size;
	(*lv)->lv_snapshot_minor	= 0;
	(*lv)->lv_chunk_size		= 0;
	(*lv)->dummy			= 0;
	(*lv)->lv_allocated_le		= lv_opts->extents;
	(*lv)->lv_stripes		= lv_opts->stripes;
	(*lv)->lv_stripesize		= lv_opts->stripe_size;
	(*lv)->lv_badblock		= 0;
	(*lv)->lv_allocation		= 0;
	(*lv)->lv_io_timeout		= 0;
	(*lv)->lv_read_ahead		= LVM_MAX_READ_AHEAD;

out:
	LOG_EXIT_INT(rc);
	return rc;
}


/**
 * lvm_clear_lv
 *
 * Erases the contents of the specified lv_disk_t structure.
 **/
void lvm_clear_lv(lv_disk_t * lv)
{
	LOG_ENTRY();
	memset(lv, 0, sizeof(lv_disk_t));
	LOG_EXIT_VOID();
}

/**
 * lvm_can_expand_volume
 *
 * Determine if the specified volume can be expanded.
 **/
int lvm_can_expand_volume(lvm_logical_volume_t * volume)
{
	int rc = 0;

	LOG_ENTRY();

	if (volume->region->data_type != DATA_TYPE) {
		/* Can't expand the freespace region. */
		LOG_DETAILS("Cannot expand freespace region %s.\n",
			    volume->region->name);
		rc = EINVAL;
	} else if (volume->flags & LVM_LV_FLAG_INCOMPLETE) {
		/* Can't expand regions that have missing extents. */
		LOG_DETAILS("Region %s is currently missing PVs. Cannot expand.\n",
			    volume->region->name);
		rc = EINVAL;
	} else if (volume->flags & LVM_LV_FLAG_MOVE_PENDING) {
		/* Can't expand regions that have extents that are being moved. */
		LOG_DETAILS("Region %s has extents that are waiting to be moved. Cannot expand.\n",
			    volume->region->name);
		rc = EINVAL;
	} else if (volume->group->freespace->lv->lv_allocated_le == 0) {
		/* Must be space available in the group. */
		LOG_DETAILS("No freespace left in container %s.\n",
			    volume->group->container->name);
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_can_shrink_volume
 *
 * Determine if the specified volume can be shrunk.
 **/
int lvm_can_shrink_volume(lvm_logical_volume_t * volume)
{
	int rc = 0;

	LOG_ENTRY();

	if (volume->region->data_type != DATA_TYPE) {
		/* Can't shrink the freespace region. */
		LOG_DETAILS("Cannot shrink freespace region %s.\n",
			    volume->region->name);
		rc = EINVAL;
	} else if (volume->flags & LVM_LV_FLAG_INCOMPLETE) {
		/* Can't shrink regions that have missing extents. */
		LOG_DETAILS("Region %s is currently missing PVs. Cannot shrink.\n",
			    volume->region->name);
		rc = EINVAL;
	} else if (volume->flags & LVM_LV_FLAG_MOVE_PENDING) {
		/* Can't shrink regions that have extents that are being moved. */
		LOG_DETAILS("Region %s has extents that are waiting to be moved. Cannot shrink.\n",
			    volume->region->name);
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/****** Checking Available Extents for Volumes ******/


/**
 * lvm_check_available_extents_simple
 *
 * Check that the specified group has the specified number of unallocated
 * extents. If any PVs have been specified, only check available PEs on
 * those PVs. The pv_entries parameter must be a NULL-terminated array of
 * PV pointers.
 **/
static int lvm_check_available_extents_simple(lvm_volume_group_t * group,
					      u_int32_t desired_extents,
					      lvm_physical_volume_t * pv_entries[])
{
	int available_extents = 0;
	int rc = 0;
	int i = 0;

	LOG_ENTRY();

	if (!pv_entries[0]) {
		/* No PVs specified. */
		available_extents = group->freespace->lv->lv_allocated_le;
	} else {
		/* Sum up available PEs from specified PVs. */
		while (pv_entries[i]) {
			available_extents += pv_entries[i]->pv->pe_total -
					     pv_entries[i]->pv->pe_allocated -
					     pv_entries[i]->move_extents;
			i++;
		}
	}

	if (available_extents < desired_extents) {
		LOG_ERROR("Requested %d extents.\n", desired_extents);
		LOG_ERROR("Container %s only has %d extents available.\n",
			  group->container->name, available_extents);
		rc = ENOSPC;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_check_available_extents_striped
 *
 * Check that the specified group has enough available extents to
 * allocate to a striped volume with the specified number of stripes.
 * The options must have already been verified before calling this
 * function, so that extents is a multiple of stripes.
 **/
static int lvm_check_available_extents_striped(lvm_volume_group_t * group,
					       u_int32_t extents,
					       u_int32_t stripes,
					       lvm_physical_volume_t * pv_list[])
{
	u_int32_t extents_per_stripe = extents / stripes;
	u_int32_t stripes_found = 0;
	int i, rc = 0;

	LOG_ENTRY();

	if (!pv_list[0]) {
		/* No PVs specified. Use the group's list. */
		pv_list = group->pv_list;
	}

	/* Check each PV for sufficient extents_per_stripe. If found, increment
	 * stripes_found. When stripes_found == stripes, we know we have enough
	 * extents.
	 */
	for (i = 0; i <= MAX_PV && stripes_found < stripes; i++) {
		if (pv_list[i] &&
		    (pv_list[i]->pv->pe_total - pv_list[i]->pv->pe_allocated -
		     pv_list[i]->move_extents >= extents_per_stripe)) {
			stripes_found++;
		}
	}

	if (stripes_found < stripes) {
		LOG_ERROR("Requested %d extents on %d stripes (%d extents per stripe)\n",
			  extents, stripes, extents_per_stripe);
		LOG_ERROR("Only have %d stripes available with %d extents each.\n",
			  stripes_found, extents_per_stripe);
		rc = ENOSPC;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_check_available_expand_extents_striped
 *
 * Striped volumes can only be expanded if all of the underlying PVs
 * each have the appropriate number of extents available.
 **/
static int lvm_check_available_expand_extents_striped(lvm_logical_volume_t * volume,
						      u_int32_t add_extents)
{
	storage_object_t * segment;
	lvm_physical_volume_t * pv_entry;
	list_element_t itr;
	u_int32_t extents_per_stripe = add_extents / volume->lv->lv_stripes;
	int rc = 0;

	LOG_ENTRY();

	/* Check each PV in this volume's child-list to see if it has
	 * enough free extents for the expand.
	 */
	LIST_FOR_EACH(volume->region->child_objects, itr, segment) {
		pv_entry = lvm_get_pv_for_segment(segment);
		if (pv_entry->pv->pe_total - pv_entry->pv->pe_allocated -
		    pv_entry->move_extents < extents_per_stripe) {
			LOG_ERROR("Not enough available extents on PV %s to expand region %s.\n",
				  pv_entry->segment->name, volume->region->name);
			rc = EINVAL;
			goto out;
		}
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_check_available_extents
 *
 * Determine which kind of allocation is going to be used, and call the
 * appropriate function to check for available space.
 **/
int lvm_check_available_extents(lvm_volume_group_t * group,
				lvm_lv_create_options_t * lv_opts)
{
	int rc = 0;

	LOG_ENTRY();

	if (lv_opts->stripe_size > 1) {
		rc = lvm_check_available_extents_striped(group,
							 lv_opts->extents,
							 lv_opts->stripes,
							 lv_opts->pv_entries);
	} else {
		rc = lvm_check_available_extents_simple(group,
							lv_opts->extents,
							lv_opts->pv_entries);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_check_available_expand_extents
 *
 * Determine which kind of allocation the existing volume uses, and call
 * the appropriate function to check for available space.
 **/
int lvm_check_available_expand_extents(lvm_logical_volume_t * volume,
				       lvm_lv_expand_options_t * lv_opts)
{
	int rc;

	LOG_ENTRY();

	if (volume->lv->lv_stripes > 1) {
		rc = lvm_check_available_expand_extents_striped(volume,
								lv_opts->add_extents);
	} else {
		rc = lvm_check_available_extents_simple(volume->group,
							lv_opts->add_extents,
							lv_opts->pv_entries);
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/****** Allocating Extents to Volumes ******/


/**
 * lvm_allocate_extents_simple
 * @volume:	Volume to allocate extents to.
 * @le_total:	The number of extents the final volume should have.
 * @pv_list:	NULL-terminated array of PV pointers, or NULL to try to
 *		allocate from all available PVs.
 *
 * Run through each PV in this volume's group and look for unused PEs.
 * Assign unused PEs to this volume until total is reached. This scheme
 * implements a simple first-available allocation of PEs.
 *
 * This function must NOT be called unless lvm_check_available_extents has
 * been called to verify that ample extents are available.
 **/
static int lvm_allocate_extents_simple(lvm_logical_volume_t * volume,
				       u_int32_t le_total,
				       lvm_physical_volume_t * pv_list[])
{
	lvm_volume_group_t * group = volume->group;
	lvm_physical_volume_t * pv_entry;
	u_int32_t current_le = 0;
	int used_this_pv, i, j, rc = 0;

	LOG_ENTRY();

	if (!pv_list[0]) {
		/* No PVs specified. Use the group's list. */
		pv_list = group->pv_list;
	}

	/* Go through the list of PVs. For each PV, go through the list of PEs.
	 * For every PE that does not map to an LV, record the LV and LE
	 * numbers, and updated the appropriate PV entries.
	 */
	for (i = 0; i <= MAX_PV && current_le < le_total; i++) {
		pv_entry = pv_list[i];
		if (pv_entry) {
			used_this_pv = FALSE;
			for (j = 0; j < pv_entry->pv->pe_total && current_le < le_total; j++) {
				if (!pv_entry->pe_map[j].pe.lv_num &&
				    !pv_entry->pe_map[j].new_le) {
					pv_entry->pe_map[j].pe.lv_num = volume->number;
					pv_entry->pe_map[j].pe.le_num = current_le;
					pv_entry->pv->pe_allocated++;
					current_le++;
					if (!used_this_pv) {
						pv_entry->pv->lv_cur++;
						used_this_pv = TRUE;
					}
				}
			}
		}
	}

	if (current_le != le_total) {
		/* This should never happen as long as we've checked for
		 * available space ahead of time.
		 */
		LOG_SERIOUS("Could not allocate enough extents for region %s\n",
			    volume->region->name);
		rc = ENOSPC;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_allocate_expand_extents_simple
 *
 * Expanding simple volumes now has its own function for allocating
 * the extents.
 **/
static int lvm_allocate_expand_extents_simple(lvm_logical_volume_t * volume,
					      u_int32_t expand_extents,
					      lvm_physical_volume_t * pv_list[])
{
	lvm_volume_group_t * group = volume->group;
	lvm_physical_volume_t * pv_entry;
	u_int32_t current_le = volume->lv->lv_allocated_le;
	u_int32_t le_total = current_le + expand_extents;
	int used_this_pv, i, j, rc = 0;

	LOG_ENTRY();

	if (!pv_list[0]) {
		/* No PVs specified. Use the group's list. */
		pv_list = group->pv_list;
	}

	/* Go through the list of PVs. For each PV, go through the list of PEs.
	 * For every PE that does not map to an LV, record the LV and LE
	 * numbers, and updated the appropriate PV entries.
	 */
	for (i = 0; i <= MAX_PV && current_le < le_total; i++) {
		pv_entry = pv_list[i];
		if (pv_entry) {
			/* Determine if the volume already has extents on
			 * this PV.
			 */
			used_this_pv = FALSE;
			if (lvm_volume_is_on_pv(volume, pv_entry)) {
				used_this_pv = TRUE;
			}

			for (j = 0; j < pv_entry->pv->pe_total && current_le < le_total; j++) {
				if (!pv_entry->pe_map[j].pe.lv_num &&
				    !pv_entry->pe_map[j].new_le) {
					pv_entry->pe_map[j].pe.lv_num = volume->number;
					pv_entry->pe_map[j].pe.le_num = current_le;
					pv_entry->pv->pe_allocated++;
					current_le++;
					if (!used_this_pv) {
						pv_entry->pv->lv_cur++;
						used_this_pv = TRUE;
					}
				}
			}
		}
	}

	if (current_le != le_total) {
		/* This should never happen as long as we've checked for
		 * available space ahead of time.
		 */
		LOG_SERIOUS("Could not allocate enough extents for region %s\n",
			    volume->region->name);
		rc = ENOSPC;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_allocate_extents_striped
 *
 * This function implements a striped allocation scheme.
 *
 * This function must NOT be called unless lvm_check_available_extents has
 * been called to verify that ample extents are available.
 **/
static int lvm_allocate_extents_striped(lvm_logical_volume_t * volume,
					u_int32_t extents,
					u_int32_t stripes,
					lvm_physical_volume_t * pv_list[])
{
	lvm_volume_group_t * group = volume->group;
	lvm_physical_volume_t * pv_entry;
	u_int32_t extents_per_stripe = extents / stripes;
	u_int32_t stripes_found = 0;
	u_int32_t current_le = 0;
	int i, j, k, rc = 0;

	LOG_ENTRY();

	if (!pv_list[0]) {
		/* No PVs specified. Use the group's list. */
		pv_list = group->pv_list;
	}

	/* Run through the list of PVs. For each one that has enough extents
	 * available, look for extents to allocate. When one is found, assign
	 * it to the specified volume.
	 */
	for (i = 0; i <= MAX_PV && stripes_found < stripes; i++) {
		pv_entry = pv_list[i];
		if (pv_entry &&
		    (pv_entry->pv->pe_total - pv_entry->pv->pe_allocated -
		     pv_entry->move_extents >= extents_per_stripe)) {
			for (j = 0, k = 0;
			     j < pv_entry->pv->pe_total && k < extents_per_stripe;
			     j++) {
				if (!pv_entry->pe_map[j].pe.lv_num &&
				    !pv_entry->pe_map[j].new_le) {
					pv_entry->pe_map[j].pe.lv_num = volume->number;
					pv_entry->pe_map[j].pe.le_num = current_le;
					pv_entry->pv->pe_allocated++;
					current_le++;
					k++;
				}
			}
			pv_entry->pv->lv_cur++;
			stripes_found++;
		}
	}

	if (stripes_found < stripes) {
		/* This should never happen as long as we've checked for
		 * available space ahead of time.
		 */
		LOG_SERIOUS("Requested %d extents on %d stripes (%d extents per stripe)\n",
			    extents, stripes, extents_per_stripe);
		LOG_SERIOUS("Only have %d stripes available with %d extents each.\n",
			    stripes_found, extents_per_stripe);
		rc = ENOSPC;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_allocate_expand_extents_striped
 *
 * Striped volumes can only be expanded if all underlying PVs for the
 * volume have enough available extents.
 **/
static int lvm_allocate_expand_extents_striped(lvm_logical_volume_t * volume,
					       u_int32_t add_extents)
{
	lvm_physical_volume_t * pv_entry;
	u_int32_t extents_per_stripe = add_extents / volume->lv->lv_stripes;
	u_int32_t le, new_le;
	int i, j, rc = 0;

	LOG_ENTRY();

	/* Rewrite all PE map entries in all the PVs for this LV. */
	for (le = new_le = 0; le < volume->lv->lv_allocated_le; le++) {
		volume->le_map[le].pe->pe.le_num = new_le;
		new_le++;

		/* When we get to the last PE for each PV, allocate new
		 * extents on that PV.
		 */
		if (le == volume->lv->lv_allocated_le - 1 ||
		    volume->le_map[le].pe->pv != volume->le_map[le+1].pe->pv) {
			pv_entry = volume->le_map[le].pe->pv;
			for (i = j = 0;
			     i < pv_entry->pv->pe_total &&
			     j < extents_per_stripe;
			     i++ ) {
				if (!pv_entry->pe_map[i].pe.lv_num &&
				    !pv_entry->pe_map[i].new_le) {
					pv_entry->pe_map[i].pe.lv_num = volume->number;
					pv_entry->pe_map[i].pe.le_num = new_le;
					pv_entry->pv->pe_allocated++;
					new_le++;
					j++;
				}
			}
		}
	}

	if (new_le < volume->lv->lv_allocated_le + add_extents) {
		/* This should never happen as long as we check for
		 * available space before trying to allocate.
		 */
		LOG_SERIOUS("Not enough extents to extend striped region %s\n",
			    volume->region->name);
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_allocate_extents_to_volume
 *
 * Decide what kind of allocation is needed, and call the appropriate
 * function to perform the allocation of extents.
 **/
int lvm_allocate_extents_to_volume(lvm_logical_volume_t * volume,
				   lvm_lv_create_options_t * lv_opts)
{
	int rc = 0;

	LOG_ENTRY();

	if ( lv_opts->stripes > 1 ) {
		rc = lvm_allocate_extents_striped(volume, lv_opts->extents,
						  lv_opts->stripes,
						  lv_opts->pv_entries);
	} else {
		rc = lvm_allocate_extents_simple(volume, lv_opts->extents,
						 lv_opts->pv_entries);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_allocate_expand_extents_to_volume
 *
 * Decide what kind of allocation is needed, and call the appropriate
 * function to perform the allocation of extents.
 **/
int lvm_allocate_expand_extents_to_volume(lvm_logical_volume_t * volume,
					  lvm_lv_expand_options_t * lv_opts)
{
	int rc;

	LOG_ENTRY();

	if (volume->lv->lv_stripes > 1) {
		rc = lvm_allocate_expand_extents_striped(volume,
							 lv_opts->add_extents);
	} else {
		rc = lvm_allocate_expand_extents_simple(volume,
							lv_opts->add_extents,
							lv_opts->pv_entries );
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_deallocate_extents_from_volume
 *
 * Traverse the LE map of the specified volume, and free each extent by
 * clearing the appropriate PE map entry on the appropriate PV.
 **/
void lvm_deallocate_extents_from_volume(lvm_logical_volume_t * volume)
{
	lvm_physical_volume_t * pv_entry;
	u_int32_t pe_number;
	int i;

	LOG_ENTRY();

	/* For every logical extent. */
	for (i = 0; i < volume->lv->lv_allocated_le; i++) {
		/* Check that this LE points to a valid PE. */
		if (volume->le_map[i].pe) {
			pv_entry = volume->le_map[i].pe->pv;
			pe_number = volume->le_map[i].pe->number;
			/* Check that the LE entry and the PE entry match. */
			if (pv_entry->pe_map[pe_number].pe.lv_num == volume->number) {
				pv_entry->pe_map[pe_number].pe.lv_num = 0;
				pv_entry->pe_map[pe_number].pe.le_num = 0;
				pv_entry->pe_map[pe_number].le = NULL;
				pv_entry->pe_map[pe_number].new_le = NULL;
				pv_entry->pv->pe_allocated--;
				if (! (pv_entry->flags & LVM_PV_FLAG_LV_CUR_UPDATED)) {
					/* Only update the lv_cur variable once
					 * per PV.
					 */
					pv_entry->pv->lv_cur--;
					pv_entry->flags |= LVM_PV_FLAG_LV_CUR_UPDATED;
				}
			} else {
				LOG_SERIOUS("LE map inconsistency in region %s (%d)\n",
					    volume->region->name, volume->number);
				LOG_SERIOUS("LE %d maps to PV %s:PE %d\n", i,
					    pv_entry->segment->name, pe_number);
				LOG_SERIOUS("PV %s:PE %d maps to LV %d:LE %d\n",
					    pv_entry->segment->name, pe_number,
					    pv_entry->pe_map[pe_number].pe.lv_num,
					    pv_entry->pe_map[pe_number].pe.le_num);
			}
		}

		/* Check for extents that this volume was moving to. */
		if (volume->le_map[i].new_pe) {
			volume->le_map[i].new_pe->new_le = NULL;
		}
	}

	/* Clear all the LV_CUR flags in the PVs. */
	for (i = 1; i <= MAX_PV; i++) {
		if (volume->group->pv_list[i]) {
			volume->group->pv_list[i]->flags &= ~LVM_PV_FLAG_LV_CUR_UPDATED;
		}
	}

	LOG_EXIT_VOID();
}

/**
 * lvm_discard_extents_from_volume
 *
 * A lot like deallocate_extents_from_volume, but doesn't update any metadata.
 * Only called from lvm_discard.
 **/
void lvm_discard_extents_from_volume(lvm_logical_volume_t * volume)
{
	int i;

	LOG_ENTRY();

	/* For every logical extent. */
	for (i = 0; i < volume->lv->lv_allocated_le; i++) {
		/* Check that this LE points to a valid PE. */
		if (volume->le_map[i].pe) {
			volume->le_map[i].pe->le = NULL;
		}

		/* Check for extents that this volume was moving to. */
		if (volume->le_map[i].new_pe) {
			volume->le_map[i].new_pe->new_le = NULL;
		}
	}

	LOG_EXIT_VOID();
}

/**
 * lvm_deallocate_shrink_extents_striped
 **/
static void lvm_deallocate_shrink_extents_striped(lvm_logical_volume_t * volume,
						  u_int32_t remove_extents)
{
	u_int32_t current_extents_per_stripe;
	u_int32_t new_extents_per_stripe;
	u_int32_t le, new_le, i;
	
	LOG_ENTRY();

	current_extents_per_stripe = volume->lv->lv_allocated_le / volume->lv->lv_stripes;
	new_extents_per_stripe = current_extents_per_stripe - remove_extents / volume->lv->lv_stripes;

	for (le = new_le = i = 0; le < volume->lv->lv_allocated_le; ) {
		if (i < new_extents_per_stripe) {
			volume->le_map[le].pe->pe.le_num = new_le;
			new_le++;
			le++;
			i++;

			/* Check for LEs that were being moved. */
			if (volume->le_map[le].new_pe) {
				volume->le_map[le].new_pe->new_le = NULL;
			}
		} else {
			for ( ; i < current_extents_per_stripe; i++, le++) {
				volume->le_map[le].pe->pe.lv_num = 0;
				volume->le_map[le].pe->pe.le_num = 0;
				volume->le_map[le].pe->pv->pv->pe_allocated--;

				/* Check for LEs that were being moved. */
				if (volume->le_map[le].new_pe) {
					volume->le_map[le].new_pe->new_le = NULL;
				}
			}
			i = 0;
		}

	}

	LOG_EXIT_VOID();
}

/**
 * lvm_deallocate_shrink_extents_simple
 **/
static void lvm_deallocate_shrink_extents_simple(lvm_logical_volume_t * volume,
						 u_int32_t remove_extents)
{
	lvm_physical_volume_t * pv_entry;
	lvm_physical_extent_t * pe;
	u_int32_t le = volume->lv->lv_allocated_le - 1;
	u_int32_t i;

	LOG_ENTRY();

	for ( i = 0; i < remove_extents; i++, le-- ) {
		/* Check that this LE points to a valid PV. */
		pe = volume->le_map[le].pe;
		if (pe) {
			pv_entry = pe->pv;

			/* Check that the LE entry and the PE entry match. */
			if (pe->pe.lv_num == volume->number) {
				pe->pe.lv_num = 0;
				pe->pe.le_num = 0;
				pv_entry->pv->pe_allocated--;

				/* See if we need to decrement lv_cur for
				 * this PV.
				 */
				if (!lvm_volume_is_on_pv(volume, pv_entry)) {
					pv_entry->pv->lv_cur--;
				}
			} else {
				LOG_SERIOUS("LE map inconsistency in region %s (%d)\n",
					    volume->region->name, volume->number);
				LOG_SERIOUS("LE %d maps to PV %s:PE %d\n",
					    le, pv_entry->segment->name, pe->number);
				LOG_SERIOUS("PV %s:PE %d maps to LV %d:LE %d\n",
					    pv_entry->segment->name, pe->number,
					    pe->pe.lv_num, pe->pe.le_num);
			}
		}

		if (volume->le_map[le].new_pe) {
			volume->le_map[le].new_pe->new_le = NULL;
		}
	}

	LOG_EXIT_VOID();
}

/**
 * lvm_deallocate_shrink_extents_from_volume
 *
 * Traverse the LE map of the specified volume, and free the specified
 * number of extents by clearing the appropriate PE map entry on the
 * appropriate PV.
 **/
void lvm_deallocate_shrink_extents_from_volume(lvm_logical_volume_t * volume,
					       u_int32_t remove_extents)
{
	LOG_ENTRY();

	if ( volume->lv->lv_stripes > 1 ) {
		lvm_deallocate_shrink_extents_striped(volume, remove_extents);
	} else {
		lvm_deallocate_shrink_extents_simple(volume, remove_extents);
	}

	LOG_EXIT_VOID();
}

/**
 * lvm_build_volume_targets
 *
 * Build a list of DM targets to represent this volume.
 **/
dm_target_t * lvm_build_volume_targets(lvm_logical_volume_t * volume)
{
	dm_target_t *target, *target_list = NULL;
	dm_device_t *linear;
	dm_target_stripe_t *stripe = NULL;
	storage_object_t *object;
	lvm_logical_extent_t *le;
	unsigned long pe_size = volume->group->vg->pe_size;
	unsigned long stripes = volume->lv->lv_stripes;
	unsigned long les_per_stripe = volume->lv->lv_allocated_le / stripes;
	unsigned long start_le = 0;
	unsigned long num_le = stripes;
	dm_target_type type = (stripes==1) ?
			      DM_TARGET_LINEAR : DM_TARGET_STRIPE;
	int i, j, k, rc = 0;

	LOG_ENTRY();

	for (i = 0; i < les_per_stripe; i++) {
		for (j = 0; j < stripes; j++) {
			if (! is_next_le_consecutive(volume, i + j*les_per_stripe)) {
				if (is_le_missing(volume, i + j*les_per_stripe)) {
					/* Allocate an error target for this one LE. */
					target = EngFncs->dm_allocate_target(DM_TARGET_ERROR, start_le*pe_size, num_le*pe_size, 0, 0);
					if (!target) {
						rc = ENOMEM;
						goto out;
					}
				} else {
					/* At the end of a run of consecutive LEs. */
					target = EngFncs->dm_allocate_target(type, start_le*pe_size, num_le*pe_size, stripes, 0);
					if (!target) {
						rc = ENOMEM;
						goto out;
					}
					if (type == DM_TARGET_STRIPE) {
						stripe = target->data.stripe;
						stripe->num_stripes = stripes;
						stripe->chunk_size = volume->lv->lv_stripesize;
					}
					for (k = 0; k < stripes; k++) {
						linear = (type == DM_TARGET_STRIPE) ? &stripe->devices[k] : target->data.linear;
						le = &(volume->le_map[start_le/stripes + k*les_per_stripe]);
						object = (le->copy_job) ? le->copy_job->mirror : le->pe->pv->segment;
						linear->major = object->dev_major;
						linear->minor = object->dev_minor;
						linear->start = (le->copy_job) ? 0 : le->pe->sector;
					}
				}

				EngFncs->dm_add_target(target, &target_list);
				start_le += num_le;
				num_le = 0;
				break;
			}
		}
		num_le += stripes;
	}

out:
	if (rc) {
		EngFncs->dm_deallocate_targets(target_list);
		target_list = NULL;
	}
	LOG_EXIT_PTR(target_list);
	return target_list;
}

/**
 * lvm_le_is_valid
 *
 * Check if the specified LE index is valid for the specified volume.
 **/
inline int lvm_le_is_valid(lvm_logical_volume_t * volume, u_int32_t le)
{
	return (le >= 0 && le < volume->lv->lv_allocated_le);
}

/**
 * lvm_le_is_scheduled_for_move
 **/
inline int lvm_le_is_scheduled_for_move(lvm_logical_extent_t * le)
{
	return (le->new_pe != NULL);
}

/**
 * lvm_volume_is_on_pv
 *
 * Does the specified volume have any extents mapped to the specified PV?
 **/
int lvm_volume_is_on_pv(lvm_logical_volume_t * volume,
			lvm_physical_volume_t * pv_entry)
{
	u_int32_t i;

	LOG_ENTRY();

	for (i = 0; i < pv_entry->pv->pe_total; i++) {
		if (pv_entry->pe_map[i].pe.lv_num == volume->number) {
			LOG_EXIT_INT(TRUE);
			return TRUE;
		}
	}

	LOG_EXIT_INT(FALSE);
	return FALSE;
}

/**
 * lvm_volume_is_online
 *
 * Is the specified volume in use/open/mounted?
 **/
boolean lvm_volume_is_online(lvm_logical_volume_t * volume)
{
	int online;

	LOG_ENTRY();

	online = ! EngFncs->is_offline(volume->region, NULL);

	LOG_EXIT_BOOL(online);
	return online;
}

