/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: LVM Region Manager
 * File: evms2/engine/plugins/lvm/lvm_uuid.c
 *
 * Description: This file contains all functions related to creation and
 *              checking of UUIDs in the LVM region manager.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <plugin.h>
#include "lvmregmgr.h"

/* String containing allowable characters for a UUID. */
static unsigned char c[] = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";

/**
 * lvm_check_for_uuid
 *
 * Check all existing volume groups and physical volumes to see if the
 * specified UUID is already in use.
 **/
int lvm_check_for_uuid(char * uuid)
{
	lvm_volume_group_t * group;
	list_element_t itr;
	int i, rc = 0;

	LOG_ENTRY();

	LIST_FOR_EACH(lvm_group_list, itr, group) {
		/* Check each group's UUID. */
		if ( ! memcmp(uuid, group->vg->vg_uuid, UUID_LEN) ) {
			LOG_ERROR("UUID %s already in use by VG %s\n", uuid, group->container->name);
			rc = EINVAL;
			goto out;
		}

		/* Check each group's PVs' UUIDs. */
		for ( i = 1; i <= MAX_PV; i++ ) {
			if ( group->pv_list[i] ) {
				if ( ! memcmp(uuid, group->pv_list[i]->pv->pv_uuid, UUID_LEN) ) {
					LOG_ERROR("UUID %s already in use by PV %s\n", uuid, group->pv_list[i]->segment->name);
					rc = EINVAL;
					goto out;
				}
			}
		}
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_create_uuid
 *
 * Create a new UUID string. The uuid parameter must point to a char array
 * at least 128 bytes in length. Make sure the generated UUID does not
 * conflict with any UUIDs in the system. This function is based on the
 * "lvm_create_uuid" function from the LVM tools library.
 **/
int lvm_create_uuid(char * uuid)
{
	int random, i, rc = 0;

	LOG_ENTRY();

	memset( uuid, 0, UUID_LEN );
	random = open("/dev/urandom", O_RDONLY);
	if (random < 0) {
		LOG_ERROR("Error opening /dev/urandom\n");
		rc = EIO;
		goto out;
	}

	while (1) {
		i = read(random, uuid, UUID_LEN);
		if ( i < 0 ) {
			LOG_ERROR("Read error from /dev/urandom\n");
			rc = EIO;
			goto out;
		}
		for ( i = 0; i < UUID_LEN; i++ ) {
			uuid[i] = c[uuid[i] % (sizeof(c)-1)];
		}
		if ( ! lvm_check_for_uuid(uuid) ) {
			break;
		}
	}

out:
	if (random > 0) {
		close(random);
	}
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_print_uuid
 *
 * Return a printable string of the specified UUID. This function is
 * based on the "lvm_show_uuid" function from the LVM tools library.
 **/
char * lvm_print_uuid (char * uuid)
{
	int i, j;
	static char ret[NAME_LEN] = { 0, };

	LOG_ENTRY();

	memset(ret, 0, NAME_LEN);

	i = 6;
	memcpy(ret, uuid, i);
	uuid += i;

	for ( j = 0; j < 6; j++ ) {
		ret[i++] = '-';
		memcpy(&ret[i], uuid, 4);
		uuid += 4;
		i += 4;
	}

	memcpy(&ret[i], uuid, 2);

	LOG_EXIT_PTR(ret);
	return ret;
}

/**
 * lvm_clear_uuid_list_entry
 *
 * Erase the specified UUID entry in the specified group.
 **/
int lvm_clear_uuid_list_entry(lvm_volume_group_t * group, u_int32_t number)
{
	int rc = 0;

	LOG_ENTRY();

	if ( number < 1 || number > MAX_PV ) {
		rc = EINVAL;
	} else if ( group->uuid_list[number] ) {
		EngFncs->engine_free(group->uuid_list[number]);
		group->uuid_list[number] = NULL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_set_uuid_list_entry
 *
 * Set the specified UUID entry in the specified group to the
 * specified string.
 **/
int lvm_set_uuid_list_entry(lvm_volume_group_t * group,
			    u_int32_t number,
			    unsigned char * uuid)
{
	int rc = 0;

	LOG_ENTRY();

	if ( number < 1 || number > MAX_PV ) {
		rc = EINVAL;
		goto out;
	}

	if ( ! group->uuid_list[number] ) {
		group->uuid_list[number] = EngFncs->engine_alloc(UUID_LEN);
		if ( ! group->uuid_list[number] ) {
			LOG_CRITICAL("Memory error creating string for UUID entry %d in container %s\n",
				number, group->container->name);
			rc = ENOMEM;
			goto out;
		}
	}
	memcpy(group->uuid_list[number], uuid, UUID_LEN);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_verify_pv_uuid
 *
 * pv_entry : The PV whose UUID we are verifying
 * group : The group whose UUID list we are searching
 *
 * Verify that the specified PV belongs to the specified group by
 * searching for the PV's UUID in the group's list of PV UUIDs. If it is
 * found, but in the wrong location, update the PV info to reflect the
 * new location.
 **/
int lvm_verify_pv_uuid(lvm_physical_volume_t * pv_entry,
		       lvm_volume_group_t * group)
{
	int i, rc = 0;

	LOG_ENTRY();

	/* Obviously the UUID list must be present in order to search. */
	if ( ! (group->flags & LVM_VG_FLAG_UUID_LIST_PRESENT) ) {
		LOG_ERROR("UUID list is missing from container %s\n", group->container->name);
		LOG_ERROR("Cannot verify UUID for PV %s\n", pv_entry->segment->name);
		goto out;
	}

	/* Start with the UUID entry for this PV's number. */
	if ( group->uuid_list[pv_entry->number] &&
	     ! memcmp(pv_entry->pv->pv_uuid, group->uuid_list[pv_entry->number], UUID_LEN) ) {
		goto out;
	}

	/* If it wasn't found there, then search the entire group's list. */
	for ( i = 1; i <= MAX_PV; i++ ) {
		if ( group->uuid_list[i] &&
		     ! memcmp(pv_entry->pv->pv_uuid, group->uuid_list[i], UUID_LEN) ) {
			/* Found the UUID. Update the PV to reflect the
			 * correct PV number.
			 */
			MESSAGE("Detected UUID mismatch for PV %s\n", pv_entry->segment->name);
			MESSAGE("Moving PV %s from number %ld to %d\n", pv_entry->segment->name, pv_entry->number, i);
			pv_entry->number = i;
			pv_entry->pv->pv_number = i;
			group->flags |= SCFLAG_DIRTY;
			goto out;
		}
	}

	LOG_SERIOUS("Could not find UUID for PV %s in container %s\n", pv_entry->segment->name, group->container->name);
	rc = EINVAL;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

