/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: LVM Region Manager
 * File: evms2/engine/plugins/lvm/lvm_options.c
 *
 * Description: This file contains all functions related to the user-available
 *              options in the LVM region manager.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "lvmregmgr.h"


/****** Volume Group Creation Option Handlers ******/


/**
 * lvm_create_container_parse_option_array
 *
 * Parse the options array to find options pertaining to volume group
 * creation. Each option can either be keyed by value or string, so we
 * have to check for both. Acceptable options for group creation are:
 * LVM_OPTION_VG_NAME
 * LVM_OPTION_PE_SIZE
 * Incorrect options are simply ignored.
 **/
int lvm_create_container_parse_option_array(option_array_t * options,
					    char * vg_name,
					    u_int32_t * pe_size)
{
	int i, rc = 0;

	LOG_ENTRY();

	/* Default values. */
	*pe_size = LVM_DEFAULT_PE_SIZE;

	for (i = 0; i < options->count; i++) {

		/* If only the name-based index is specified, get the number. */
		if (!options->option[i].is_number_based) {
			if (!strcmp(options->option[i].name, LVM_OPTION_VG_NAME_STR)) {
				options->option[i].number = LVM_OPTION_VG_NAME_INDEX;
			}
			else if (!strcmp(options->option[i].name, LVM_OPTION_PE_SIZE_STR)) {
				options->option[i].number = LVM_OPTION_PE_SIZE_INDEX;
			}
			else {
				continue;
			}
		}

		LOG_EXTRA("Parsing option %d\n", options->option[i].number);

		/* Use the number-based index to record the option. */
		switch (options->option[i].number) {
		case LVM_OPTION_VG_NAME_INDEX:
			strncpy(vg_name, options->option[i].value.s, NAME_LEN);
			break;
		case LVM_OPTION_PE_SIZE_INDEX:
			*pe_size = options->option[i].value.ui32;
			break;
		default:
			break;
		}
	}

	rc = lvm_create_container_verify_options(vg_name, pe_size);

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_create_container_verify_options
 *
 * Run through the options that have been collected, and verify that they
 * are all valid for creating a new group. Correct any bad options if
 * possible, or return an error.
 **/
int lvm_create_container_verify_options(char * vg_name,
					u_int32_t * pe_size)
{
	int rc = 0;

	LOG_ENTRY();

	rc = lvm_check_vg_name(vg_name);
	if (!rc) {
		lvm_check_pe_size(pe_size);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_create_container_allocate_options
 *
 * This is a helper function for lvm_init_options. It sets up the initial
 * information in the option descriptor for a Create_Container task.
 **/
int lvm_create_container_allocate_option_descriptor(option_desc_array_t * od)
{
	LOG_ENTRY();

	od->count = LVM_OPTION_PE_SIZE_INDEX + 1;

	/* Option 0 is VG name. */
	SET_STRING(od->option[LVM_OPTION_VG_NAME_INDEX].name, LVM_OPTION_VG_NAME_STR);
	SET_STRING(od->option[LVM_OPTION_VG_NAME_INDEX].title, "Name for new LVM container");
	od->option[LVM_OPTION_VG_NAME_INDEX].type = EVMS_Type_String;
	od->option[LVM_OPTION_VG_NAME_INDEX].min_len = 1;
	od->option[LVM_OPTION_VG_NAME_INDEX].max_len = EVMS_VOLUME_NAME_SIZE;
	od->option[LVM_OPTION_VG_NAME_INDEX].flags = EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
	od->option[LVM_OPTION_VG_NAME_INDEX].value.s = EngFncs->engine_alloc(EVMS_VOLUME_NAME_SIZE);

	/* Option 1 is PE size. */
	SET_STRING(od->option[LVM_OPTION_PE_SIZE_INDEX].name, LVM_OPTION_PE_SIZE_STR);
	SET_STRING(od->option[LVM_OPTION_PE_SIZE_INDEX].title, "PE size for new container");
	SET_STRING(od->option[LVM_OPTION_PE_SIZE_INDEX].tip, "Acceptable range: 8kB to 16GB. Must be a power of 2.");
	od->option[LVM_OPTION_PE_SIZE_INDEX].type = EVMS_Type_Unsigned_Int32;
	od->option[LVM_OPTION_PE_SIZE_INDEX].unit = EVMS_Unit_Sectors;
	od->option[LVM_OPTION_PE_SIZE_INDEX].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED|EVMS_OPTION_FLAGS_AUTOMATIC;
	od->option[LVM_OPTION_PE_SIZE_INDEX].constraint_type = EVMS_Collection_List;
	SET_POWER2_LIST(od->option[LVM_OPTION_PE_SIZE_INDEX].constraint.list, LVM_MIN_PE_SIZE, LVM_MAX_PE_SIZE);
	od->option[LVM_OPTION_PE_SIZE_INDEX].value.ui32 = LVM_DEFAULT_PE_SIZE;

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * lvm_create_container_get_acceptable
 *
 * This is a helper function for lvm_init_task. It determines which
 * objects in EVMS are appropriate for a Create_Container task.
 */
int lvm_create_container_get_acceptable(list_anchor_t acceptable_segments)
{
	list_anchor_t object_list;
	int rc;

	LOG_ENTRY();

	/* Find all top-level disks, segments, and regions in the system. This
	 * will also retrieve LVM regions, so let the user beware. :)
	 */
	rc = EngFncs->get_object_list(DISK|SEGMENT|REGION, DATA_TYPE, NULL, NULL,
				      VALID_INPUT_OBJECT, &object_list);
	if (!rc) {
		rc = EngFncs->merge_lists(acceptable_segments, object_list, NULL);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_create_container_set_objects
 *
 * Examine the selected_objects list in the task, and verify that each
 * object is valid for inclusion in a new container. If any are not valid,
 * remove them from the selected_objects and add them to the
 * declined_objects. In create_container, no options will change based
 * on which objects are selected.
 **/
int lvm_create_container_set_objects(task_context_t * context,
				     list_anchor_t declined_objects,
				     task_effect_t * effect)
{
	storage_object_t * segment;
	list_element_t itr;
	int rc;

	LOG_ENTRY();

	/* Check that all of the segments are valid for use in a new group. */
	LIST_FOR_EACH(context->selected_objects, itr, segment) {

		rc = lvm_check_segment_for_group_inclusion(segment, NULL);
		if (rc) {
			/* FIXME: Add this segment to declined objects. */
			LOG_ERROR("One or more objects are invalid for container creation\n");
			goto out;
		}

		/* If any of the segments are too small for the default PE
		 * size, try to reset the PE size.
		 */
		rc = lvm_check_segment_for_pe_size(segment, &(context->option_descriptors->option[LVM_OPTION_PE_SIZE_INDEX].value.ui32));
		if (rc) {
			LOG_DEBUG("Object %s is too small\n", segment->name);
			LOG_DEBUG("Resetting initial PE size value to %d sectors\n",
				  context->option_descriptors->option[LVM_OPTION_PE_SIZE_INDEX].value.ui32);
			*effect |= EVMS_Effect_Reload_Options;
		}
	}

	rc = 0;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_create_container_set_option
 *
 * This is a helper function for lvm_set_option. It sets one particular
 * option in the option descriptor for a Create_Container task, and
 * updates other option descriptor information accordingly.
 **/
int lvm_create_container_set_option(task_context_t * context,
				    u_int32_t index,
				    value_t * value,
				    task_effect_t * effect)
{
	option_desc_array_t * od = context->option_descriptors;
	storage_object_t * segment;
	list_element_t itr;
	int rc = 0;

	LOG_ENTRY();
	LOG_EXTRA("Setting option %d\n", index);

	switch (index) {

	case LVM_OPTION_VG_NAME_INDEX:
		if (strlen(value->s) >= EVMS_VOLUME_NAME_SIZE) {
			LOG_ERROR("Container name too long\n");
			rc = ENOSPC;
			break;
		}
		rc = lvm_check_vg_name(value->s);
		if (rc) {
			break;
		}
		strcpy(od->option[index].value.s, value->s);
		od->option[index].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
		break;

	case LVM_OPTION_PE_SIZE_INDEX:
		if (lvm_check_pe_size(&(value->ui32))) {
			*effect |= EVMS_Effect_Inexact;
		}

		/* Make sure every segment in the "selected" list is large
		 * enough for the specified pe_size.
		 */
		LIST_FOR_EACH(context->selected_objects, itr, segment) {
			rc = lvm_check_segment_for_pe_size(segment,
							   &(value->ui32));
			if (rc) {
				MESSAGE("One or more objects too small for PE size.\n");
				rc = ENOSPC;
				break;
			}
		}

		if (rc != ENOSPC) {
			od->option[index].value.ui32 = value->ui32;
			od->option[index].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
			rc = 0;
		}
		break;
	default:
		rc = EINVAL;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/****** Logical Volume Creation Option Handlers ******/

/**
 * lvm_parse_pv_list_option
 * @list:	A value-list of strings of PV names.
 * @pv_entries:	The array in which to return the translated pointers.
 * @group:	The group consuming the desired PVs.
 *
 * Several LVM tasks use options that are array's of PV names. These arrays of
 * names need to be transformed into arrays of pointers to PV entries.
 **/
int lvm_parse_pv_list_option(value_list_t * list,
			     lvm_physical_volume_t * pv_entries[],
			     lvm_volume_group_t * group)
{
	int i, rc = 0;

	LOG_ENTRY();

	for (i = 0; i < list->count; i++) {
		pv_entries[i] = lvm_get_pv_for_name(list->value[i].s, group);
		if (!pv_entries[i]) {
			LOG_ERROR("%s is not an object in container %s\n",
				  list->value[i].s, group->container->name);
			rc = EINVAL;
			break;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_create_region_parse_option_array
 *
 * Parse the option_array_t to find options pertaining to logical volume
 * creation. Each option can either be keyed by value or string, so we
 * have to check for both. Acceptable options for volume creation are:
 * LVM_OPTION_LV_NAME
 * LVM_OPTION_EXTENTS
 * LVM_OPTION_LV_SIZE
 * LVM_OPTION_STRIPES
 * LVM_OPTION_STRIPE_SIZE
 * LVM_OPTION_PV_NAMES
 * Incorrect options are simply ignored.
 **/
int lvm_create_region_parse_option_array(option_array_t * options,
					 lvm_volume_group_t * group,
					 lvm_lv_create_options_t * lv_opts)
{
	int i, rc;

	LOG_ENTRY();

	/* Initialize the options. */
	memset(lv_opts, 0, sizeof(lvm_lv_create_options_t));
	lv_opts->stripes = 1;

	for (i = 0; i < options->count; i++) {

		/* If only the name-based index is specified, get the number. */
		if (!options->option[i].is_number_based) {
			if (!strcmp(options->option[i].name, LVM_OPTION_LV_NAME_STR)) {
				options->option[i].number = LVM_OPTION_LV_NAME_INDEX;
			}
			else if (!strcmp(options->option[i].name, LVM_OPTION_EXTENTS_STR)) {
				options->option[i].number = LVM_OPTION_EXTENTS_INDEX;
			}
			else if (!strcmp(options->option[i].name, LVM_OPTION_LV_SIZE_STR)) {
				options->option[i].number = LVM_OPTION_LV_SIZE_INDEX;
			}
			else if (!strcmp(options->option[i].name, LVM_OPTION_STRIPES_STR)) {
				options->option[i].number = LVM_OPTION_STRIPES_INDEX;
			}
			else if (!strcmp(options->option[i].name, LVM_OPTION_STRIPE_SIZE_STR)) {
				options->option[i].number = LVM_OPTION_STRIPE_SIZE_INDEX;
			}
			else if (!strcmp(options->option[i].name, LVM_OPTION_PV_NAMES_STR)) {
				options->option[i].number = LVM_OPTION_PV_NAMES_INDEX;
			}
			else {
				continue;
			}
		}

		LOG_EXTRA("Parsing option %d\n", options->option[i].number);

		/* Use the number-based index to record the option. */
		switch (options->option[i].number) {
		case LVM_OPTION_LV_NAME_INDEX:
			strncpy(lv_opts->lv_name,
				options->option[i].value.s, NAME_LEN);
			break;
		case LVM_OPTION_EXTENTS_INDEX:
			lv_opts->extents = options->option[i].value.ui32;
			break;
		case LVM_OPTION_LV_SIZE_INDEX:
			lv_opts->lv_size = options->option[i].value.ui32;
			break;
		case LVM_OPTION_STRIPES_INDEX:
			lv_opts->stripes = options->option[i].value.ui32;
			break;
		case LVM_OPTION_STRIPE_SIZE_INDEX:
			lv_opts->stripe_size = options->option[i].value.ui32;
			break;
		case LVM_OPTION_PV_NAMES_INDEX:
			rc = lvm_parse_pv_list_option(options->option[i].value.list,
						      lv_opts->pv_entries, group);
			if (rc) {
				goto out;
			}
			break;
		default:
			break;
		}
	}

	rc = lvm_create_region_verify_options(lv_opts, group);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_create_region_verify_options
 *
 * Run through the options that have been collected, and verify that they
 * are all valid for using in the specified group. Correct any bad options
 * if possible, or return an error.
 **/
int lvm_create_region_verify_options(lvm_lv_create_options_t * lv_opts,
				     lvm_volume_group_t * group)
{
	int rc;

	LOG_ENTRY();

	/* Check the name. */
	rc = lvm_check_lv_name(lv_opts->lv_name, group);
	if (rc) {
		LOG_ERROR("Error verifying region creation options\n");
		goto out;
	}

	/* Make sure lv_size is a multiple of the PE size. If it is
	 * currently zero, this won't change anything.
	 */
	lvm_check_lv_size(&lv_opts->lv_size, group->vg->pe_size);

	/* Check/set extents and lv_size. This only checks that lv_size and
	 * extents match. It does not yet check available space.
	 */
	rc = lvm_compare_lv_size_and_extents(&lv_opts->lv_size,
					     &lv_opts->extents,
					     group->vg->pe_size);
	if (rc) {
		LOG_ERROR("Error verifying region creation options\n");
		goto out;
	}

	/* Check striping options. */
	if (lv_opts->stripes > 1) {

		/* Check for enough PVs for the specified number of stripes. */
		if (lv_opts->stripes > lvm_get_available_stripes(group)) {
			LOG_ERROR("%d stripes more than %d available objects in container %s\n",
				  lv_opts->stripes, group->pv_count,
				  group->container->name);
			rc = EINVAL;
			goto out;
		}

		/* Make sure extents is divisible by the number of stripes.
		 * We want to use an equal number of PEs on each PV.
		 */
		rc = lv_opts->extents % lv_opts->stripes;
		if (rc) {
			lv_opts->extents += (lv_opts->stripes - rc);
			lv_opts->lv_size = lv_opts->extents * group->vg->pe_size;
			MESSAGE("Rounding size up to stripes boundary: %d\n",
				lv_opts->lv_size);
			rc = 0;
		}

		/* Check the stripe size. */
		lvm_check_stripe_size(&lv_opts->stripe_size, group->vg->pe_size);
	} else {
		/* No striping. Ignore stripe size. */
		lv_opts->stripes = 1;
		lv_opts->stripe_size = 0;
	}

	/* 16-bit extent numbers mean we can't have more than 65536 extents
	 * in a volume.
	 */
	if (lv_opts->extents > LVM_PE_T_MAX ) {
		LOG_ERROR("Desired region size (%d extents) too large\n",
			  lv_opts->extents);
		LOG_ERROR("Maximum of %d extents per region allowed\n",
			  LVM_PE_T_MAX);
		rc = ENOSPC;
		goto out;
	}

	/* Check for sufficient space in the group. This is only a rough check.
	 * Space constraints on striping will happen later.
	 */
	if (lv_opts->extents > group->freespace->lv->lv_allocated_le) {
		LOG_ERROR("Not enough freespace in container %s\n",
			  group->container->name);
		LOG_ERROR("Specified size: %d sectors\n", lv_opts->lv_size);
		LOG_ERROR("Available space: %d sectors\n",
			  group->freespace->lv->lv_size);
		rc = ENOSPC;
		goto out;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_create_region_allocate_option_descriptor
 *
 * This is a helper function for lvm_init_options. It sets up the initial
 * information in the option descriptor for a Create task.
 **/
int lvm_create_region_allocate_option_descriptor(option_desc_array_t * od)
{
	LOG_ENTRY();

	od->count = LVM_OPTION_PV_NAMES_INDEX + 1;

	/* Option 0 is LV Name. */
	SET_STRING(od->option[LVM_OPTION_LV_NAME_INDEX].name, LVM_OPTION_LV_NAME_STR);
	SET_STRING(od->option[LVM_OPTION_LV_NAME_INDEX].title, "Name for new LVM Region (LV)");
	od->option[LVM_OPTION_LV_NAME_INDEX].type	= EVMS_Type_String;
	od->option[LVM_OPTION_LV_NAME_INDEX].min_len	= 1;
	od->option[LVM_OPTION_LV_NAME_INDEX].max_len	= EVMS_VOLUME_NAME_SIZE;
	od->option[LVM_OPTION_LV_NAME_INDEX].flags	= EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
	od->option[LVM_OPTION_LV_NAME_INDEX].value.s	= EngFncs->engine_alloc(EVMS_VOLUME_NAME_SIZE);

	/* Option 1 is Extents. */
	SET_STRING(od->option[LVM_OPTION_EXTENTS_INDEX].name, LVM_OPTION_EXTENTS_STR);
	SET_STRING(od->option[LVM_OPTION_EXTENTS_INDEX].title, "Number of logical extents");
	SET_STRING(od->option[LVM_OPTION_EXTENTS_INDEX].tip, "Extents are the unit of allocation of space for LVM regions. Specify number of Extents, and Size will be adjusted accordingly.");
	od->option[LVM_OPTION_EXTENTS_INDEX].type	= EVMS_Type_Unsigned_Int32;
	od->option[LVM_OPTION_EXTENTS_INDEX].flags	= EVMS_OPTION_FLAGS_NOT_REQUIRED |
							  EVMS_OPTION_FLAGS_NO_INITIAL_VALUE |
							  EVMS_OPTION_FLAGS_AUTOMATIC;

	/* Option 2 is LV Size. */
	SET_STRING(od->option[LVM_OPTION_LV_SIZE_INDEX].name, LVM_OPTION_LV_SIZE_STR);
	SET_STRING(od->option[LVM_OPTION_LV_SIZE_INDEX].title, "Size of new region");
	SET_STRING(od->option[LVM_OPTION_LV_SIZE_INDEX].tip, "Specify Size, and the number of Extents will be adjusted accordingly.");
	od->option[LVM_OPTION_LV_SIZE_INDEX].type	= EVMS_Type_Unsigned_Int32;
	od->option[LVM_OPTION_LV_SIZE_INDEX].unit	= EVMS_Unit_Sectors;
	od->option[LVM_OPTION_LV_SIZE_INDEX].flags	= EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

	/* Option 3 is Stripes. */
	SET_STRING(od->option[LVM_OPTION_STRIPES_INDEX].name, LVM_OPTION_STRIPES_STR);
	SET_STRING(od->option[LVM_OPTION_STRIPES_INDEX].title, "Number of stripes");
	SET_STRING(od->option[LVM_OPTION_STRIPES_INDEX].tip, "Number of objects (PVs) to use in a striped volume. Leave at 1 for a linear volume.");
	od->option[LVM_OPTION_STRIPES_INDEX].type	= EVMS_Type_Unsigned_Int32;
	od->option[LVM_OPTION_STRIPES_INDEX].flags	= EVMS_OPTION_FLAGS_NOT_REQUIRED |
							  EVMS_OPTION_FLAGS_AUTOMATIC;
	od->option[LVM_OPTION_STRIPES_INDEX].value.ui32	= 1;

	/* Option 4 is Stripe Size. */
	SET_STRING(od->option[LVM_OPTION_STRIPE_SIZE_INDEX].name, LVM_OPTION_STRIPE_SIZE_STR);
	SET_STRING(od->option[LVM_OPTION_STRIPE_SIZE_INDEX].title, "Stripe Size");
	SET_STRING(od->option[LVM_OPTION_STRIPE_SIZE_INDEX].tip, "Granularity at which data is striped across the underlying objects.");
	od->option[LVM_OPTION_STRIPE_SIZE_INDEX].type	= EVMS_Type_Unsigned_Int32;
	od->option[LVM_OPTION_STRIPE_SIZE_INDEX].unit	= EVMS_Unit_Sectors;
	od->option[LVM_OPTION_STRIPE_SIZE_INDEX].flags	= EVMS_OPTION_FLAGS_NOT_REQUIRED |
							  EVMS_OPTION_FLAGS_NO_INITIAL_VALUE |
							  EVMS_OPTION_FLAGS_AUTOMATIC |
							  EVMS_OPTION_FLAGS_INACTIVE;

	/* Option 5 is PV Names. */
	SET_STRING(od->option[LVM_OPTION_PV_NAMES_INDEX].name, LVM_OPTION_PV_NAMES_STR);
	SET_STRING(od->option[LVM_OPTION_PV_NAMES_INDEX].title, "Objects (PVs) to place the region on");
	SET_STRING(od->option[LVM_OPTION_PV_NAMES_INDEX].tip, "Region will be allocated on only these objects. Leave blank for automatic allocation.");
	od->option[LVM_OPTION_PV_NAMES_INDEX].type		= EVMS_Type_String;
	od->option[LVM_OPTION_PV_NAMES_INDEX].min_len		= 1;
	od->option[LVM_OPTION_PV_NAMES_INDEX].max_len		= EVMS_VOLUME_NAME_SIZE;
	od->option[LVM_OPTION_PV_NAMES_INDEX].flags		= EVMS_OPTION_FLAGS_NOT_REQUIRED |
								  EVMS_OPTION_FLAGS_AUTOMATIC |
								  EVMS_OPTION_FLAGS_VALUE_IS_LIST;
	od->option[LVM_OPTION_PV_NAMES_INDEX].value.list	= EngFncs->engine_alloc(sizeof(value_list_t) + MAX_PV*sizeof(value_t));
	od->option[LVM_OPTION_PV_NAMES_INDEX].value.list->count = 0;
	/* Don't allocate the memory for the list strings yet.
	 * Wait until set_option to do that.
	 */

	LOG_EXIT_INT(0);
	return 0;
}


/**
 * lvm_create_region_get_acceptable
 *
 * This is a helper function for lvm_get_acceptable_objects. It determines
 * which objects in EVMS are appropriate for a Create task.
 **/
void lvm_create_region_get_acceptable(list_anchor_t acceptable_regions)
{
	lvm_volume_group_t * group;
	list_element_t itr;

	LOG_ENTRY();

	/* Add all freespace regions to the acceptable regions list. */
	LIST_FOR_EACH(lvm_group_list, itr, group) {
		if (group->freespace->region->size != 0) {
			EngFncs->insert_thing(acceptable_regions,
					      group->freespace->region,
					      INSERT_AFTER, NULL);
		}
	}

	LOG_EXIT_VOID();
}

/**
 * lvm_create_region_set_objects
 *
 * Examine the selected_objects list in the task, and verify that each
 * object is valid for using to create a new region. If any are not valid,
 * remove them from the selected_objects and add them to the
 * declined_objects.
 **/
int lvm_create_region_set_objects(task_context_t * context,
				  list_anchor_t declined_objects,
				  task_effect_t * effect)
{
	lvm_logical_volume_t * freespace;
	lvm_volume_group_t * group;
	option_desc_array_t * od = context->option_descriptors;
	u_int32_t pe_size, extents, stripes;
	int i, j, rc = 0;

	LOG_ENTRY();

	/* Only one freespace region can be selected, so just grab the
	 * first item on the list.
	 */
	rc = lvm_get_freespace_volume(context->selected_objects, &freespace);
	if (rc) {
		goto out;
	}

	group = freespace->group;
	pe_size = group->vg->pe_size;
	extents = (freespace->lv->lv_allocated_le < LVM_PE_T_MAX) ?
		  freespace->lv->lv_allocated_le : LVM_PE_T_MAX;
	stripes = lvm_get_available_stripes(group);

	LOG_EXTRA("Setting object %s\n", freespace->region->name);

	/* Now that we have a selected freespace, we can update the option
	 * descriptor appropriately.
	 */

	/* Update range of extents and set initial value to maximum. */
	od->option[LVM_OPTION_EXTENTS_INDEX].constraint_type = EVMS_Collection_Range;
	SET_RANGE32(od->option[LVM_OPTION_EXTENTS_INDEX].constraint.range, 1, extents, 1);
	od->option[LVM_OPTION_EXTENTS_INDEX].value.ui32 = extents;
	od->option[LVM_OPTION_EXTENTS_INDEX].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

	/* Update range of LV size and set initial value to maximum. */
	od->option[LVM_OPTION_LV_SIZE_INDEX].constraint_type = EVMS_Collection_Range;
	SET_RANGE32(od->option[LVM_OPTION_LV_SIZE_INDEX].constraint.range, pe_size, pe_size * extents, pe_size);
	od->option[LVM_OPTION_LV_SIZE_INDEX].value.ui32 = extents * pe_size;
	od->option[LVM_OPTION_LV_SIZE_INDEX].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

	/* Update number of available stripes. Initial value is already set. */
	od->option[LVM_OPTION_STRIPES_INDEX].constraint_type = EVMS_Collection_Range;
	SET_RANGE32(od->option[LVM_OPTION_STRIPES_INDEX].constraint.range, 1, stripes, 1);

	/* Update range of stripe sizes and initial value. */
	od->option[LVM_OPTION_STRIPE_SIZE_INDEX].constraint_type = EVMS_Collection_List;
	SET_POWER2_LIST(od->option[LVM_OPTION_STRIPE_SIZE_INDEX].constraint.list, LVM_MIN_STRIPE_SIZE, ((LVM_MAX_STRIPE_SIZE < pe_size) ? LVM_MAX_STRIPE_SIZE : pe_size));
	od->option[LVM_OPTION_STRIPE_SIZE_INDEX].value.ui32 = (LVM_DEFAULT_STRIPE_SIZE < pe_size) ? LVM_DEFAULT_STRIPE_SIZE : pe_size;
	od->option[LVM_OPTION_STRIPE_SIZE_INDEX].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

	/* Setup list of possible PVs. */
	od->option[LVM_OPTION_PV_NAMES_INDEX].constraint_type = EVMS_Collection_List;
	od->option[LVM_OPTION_PV_NAMES_INDEX].constraint.list = EngFncs->engine_alloc(sizeof(value_list_t) + group->pv_count*sizeof(value_t));
	for (i = 1, j = 0; i < MAX_PV; i++) {
		if (group->pv_list[i] &&
		    lvm_pv_has_available_extents(group->pv_list[i])) {
			SET_STRING(od->option[LVM_OPTION_PV_NAMES_INDEX].constraint.list->value[j].s, group->pv_list[i]->segment->name);
			j++;
		}
	}
	od->option[LVM_OPTION_PV_NAMES_INDEX].constraint.list->count = j;

	*effect |= EVMS_Effect_Reload_Options;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_create_region_set_option
 *
 * This is a helper function for lvm_set_option. It sets one particular
 * option in the option descriptor for a Create task, and updates other
 * option descriptor information accordingly.
 **/
int lvm_create_region_set_option(task_context_t * context,
				 u_int32_t index,
				 value_t * value,
				 task_effect_t * effect)
{
	option_desc_array_t * od = context->option_descriptors;
	lvm_logical_volume_t * freespace;
	lvm_volume_group_t * group;
	u_int32_t extents = 0;
	u_int32_t lv_size = 0;
	u_int32_t stripes = 1;
	int i, rc = 0;

	LOG_ENTRY();

	/* Only one freespace region should be selected, so just grab the
	 * first item on the list. Anything else on the list will be ignored.
	 */
	rc = lvm_get_freespace_volume(context->selected_objects, &freespace);
	if (rc) {
		goto out;
	}
	group = freespace->group;

	LOG_EXTRA("Setting option %d\n", index);

	switch (index) {

	case LVM_OPTION_LV_NAME_INDEX:
		/* Make sure this name isn't in use already. */
		rc = lvm_check_lv_name(value->s, group);
		if (rc) {
			LOG_ERROR("Invalid name: %s\n", value->s);
		} else {
			strncpy(od->option[index].value.s, value->s, NAME_LEN);
			od->option[index].flags	&= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
		}
		break;

	case LVM_OPTION_EXTENTS_INDEX:
		/* Make sure there are enough extents. */
		extents = (group->freespace->lv->lv_allocated_le < LVM_PE_T_MAX) ?
			  group->freespace->lv->lv_allocated_le : LVM_PE_T_MAX;
		if (value->ui32 > extents) {
			LOG_ERROR("%d extents chosen. Only %d available.\n",
				  value->ui32, extents);
			value->ui32 = extents;
			*effect |= EVMS_Effect_Inexact;
		}
		od->option[index].value.ui32 = value->ui32;
		od->option[index].flags &= ~(EVMS_OPTION_FLAGS_NOT_REQUIRED|EVMS_OPTION_FLAGS_NO_INITIAL_VALUE);

		/* Recalculate size based on extents. */
		lv_size = value->ui32 * group->vg->pe_size;
		od->option[LVM_OPTION_LV_SIZE_INDEX].value.ui32 = lv_size;
		od->option[LVM_OPTION_LV_SIZE_INDEX].flags |= EVMS_OPTION_FLAGS_NOT_REQUIRED;
		od->option[LVM_OPTION_LV_SIZE_INDEX].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

		*effect |= EVMS_Effect_Reload_Options;
		break;

	case LVM_OPTION_LV_SIZE_INDEX:
		/* Make sure lv_size is a multiple of the PE size. */
		if (lvm_check_lv_size(&value->ui32, group->vg->pe_size)) {
			*effect |= EVMS_Effect_Inexact;
		}

		/* Make sure there is enough space. */
		extents = (group->freespace->lv->lv_allocated_le < LVM_PE_T_MAX) ?
			  group->freespace->lv->lv_allocated_le : LVM_PE_T_MAX;
		lv_size = extents * group->vg->pe_size;
		if (value->ui32 > lv_size) {
			LOG_ERROR("%d sectors chosen for size. Only %d available.\n",
				  value->ui32, lv_size);
			value->ui32 = lv_size;
			*effect |= EVMS_Effect_Inexact;
		}
		od->option[index].value.ui32 = value->ui32;
		od->option[index].flags &= ~(EVMS_OPTION_FLAGS_NOT_REQUIRED|EVMS_OPTION_FLAGS_NO_INITIAL_VALUE);

		/* Recalculate extents based on lv_size. */
		extents = value->ui32 / group->vg->pe_size;
		od->option[LVM_OPTION_EXTENTS_INDEX].value.ui32 = extents;
		od->option[LVM_OPTION_EXTENTS_INDEX].flags |= EVMS_OPTION_FLAGS_NOT_REQUIRED;
		od->option[LVM_OPTION_EXTENTS_INDEX].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

		*effect |= EVMS_Effect_Reload_Options;
		break;

	case LVM_OPTION_STRIPES_INDEX:
		/* Make sure there are enough PVs available. */
		stripes = lvm_get_available_stripes(group);
		if (value->ui32 > stripes) {
			LOG_ERROR("%d stripes chosen. Only %d available.\n",
				  value->ui32, stripes);
			value->ui32 = stripes;
			*effect |= EVMS_Effect_Inexact;
		}
		od->option[index].value.ui32 = value->ui32;

		/* Turn on/off related options. */
		if ( value->ui32 > 1 ) {
			od->option[LVM_OPTION_STRIPE_SIZE_INDEX].flags &= ~EVMS_OPTION_FLAGS_INACTIVE;
		} else {
			od->option[LVM_OPTION_STRIPE_SIZE_INDEX].flags |= EVMS_OPTION_FLAGS_INACTIVE;
		}

		*effect |= EVMS_Effect_Reload_Options;
		break;

	case LVM_OPTION_STRIPE_SIZE_INDEX:
		/* Make sure stripe_size is within range. */
		lvm_check_stripe_size(&(value->ui32), group->vg->pe_size);
		od->option[index].value.ui32 = value->ui32;
		break;

	case LVM_OPTION_PV_NAMES_INDEX:
		for (i = 0; i < value->list->count; i++) {
			if (od->option[index].value.list->value[i].s) {
				EngFncs->engine_free(od->option[index].value.list->value[i].s);
				od->option[index].value.list->value[i].s = NULL;
			}
			SET_STRING(od->option[index].value.list->value[i].s, value->list->value[i].s);
		}
		for (; i < od->option[index].value.list->count; i++) {
			if (od->option[index].value.list->value[i].s) {
				EngFncs->engine_free(od->option[index].value.list->value[i].s);
				od->option[index].value.list->value[i].s = NULL;
			}
		}
		od->option[index].value.list->count = value->list->count;
		break;

	default:
		rc = EINVAL;
		break;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}


/****** Logical Volume Expand Option Handlers ******/


/**
 * lvm_expand_region_parse_option_array
 *
 * Parse the option_array_t to find options pertaining to logical volume
 * expansion. Each option can either be keyed by value or string, so we
 * have to check for both. Acceptable options for volume creation are:
 * LVM_OPTION_EXPAND_SIZE
 * LVM_OPTION_EXPAND_EXTENTS
 * LVM_OPTION_EXPAND_PV_NAMES
 * Incorrect options are simply ignored.
 **/
int lvm_expand_region_parse_option_array(option_array_t * options,
					 lvm_volume_group_t * group,
					 lvm_logical_volume_t * volume,
					 lvm_lv_expand_options_t * lv_opts)
{
	int i, j, rc = 0;

	LOG_ENTRY();

	/* Initialize the options. */
	memset(lv_opts, 0, sizeof(lvm_lv_expand_options_t));

	for (i = 0; !rc && i < options->count; i++) {

		/* If only the name-based index is specified, get the number. */
		if (! options->option[i].is_number_based) {
			if (! strcmp(options->option[i].name, LVM_OPTION_EXPAND_EXTENTS_STR)) {
				options->option[i].number = LVM_OPTION_EXPAND_EXTENTS_INDEX;
			} else if (! strcmp(options->option[i].name, LVM_OPTION_EXPAND_SIZE_STR)) {
				options->option[i].number = LVM_OPTION_EXPAND_SIZE_INDEX;
			} else if (! strcmp(options->option[i].name, LVM_OPTION_EXPAND_PV_NAMES_STR)) {
				options->option[i].number = LVM_OPTION_EXPAND_PV_NAMES_INDEX;
			} else {
				continue;
			}
		}

		LOG_EXTRA("Parsing option %d\n", options->option[i].number);

		/* Use the number-based index to record the option. */
		switch (options->option[i].number) {
		case LVM_OPTION_EXPAND_EXTENTS_INDEX:
			lv_opts->add_extents = options->option[i].value.ui32;
			break;
		case LVM_OPTION_EXPAND_SIZE_INDEX:
			lv_opts->add_size = options->option[i].value.ui32;
			break;
		case LVM_OPTION_EXPAND_PV_NAMES_INDEX:
			for (j = 0; j < options->option[i].value.list->count; j++) {
				lv_opts->pv_entries[j] = lvm_get_pv_for_name(options->option[i].value.list->value[j].s, group);
				if (!lv_opts->pv_entries[j]) {
					LOG_ERROR("%s is not an object in container %s\n",
						  options->option[i].value.list->value[j].s,
						  group->container->name);
					rc = EINVAL;
				}
			}
			break;
		default:
			break;
		}
	}

	if (!rc) {
		rc = lvm_expand_region_verify_options(lv_opts, group, volume);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_expand_region_verify_options
 *
 * Run through the options that have been collected, and verify that they
 * are all valid for using in the specified group. Correct any bad options
 * if possible, or return an error.
 **/
int lvm_expand_region_verify_options(lvm_lv_expand_options_t * lv_opts,
				     lvm_volume_group_t * group,
				     lvm_logical_volume_t * volume)
{
	int rc = 0;

	LOG_ENTRY();

	/* Make sure add_size is a multiple of the PE size. If it is
	 * currently zero, this won't change anything.
	 */
	lvm_check_lv_size(&lv_opts->add_size, group->vg->pe_size);

	/* Check/set extents and size. This only checks that size and
	 * extents match. It does not yet check available space.
	 */
	rc = lvm_compare_lv_size_and_extents(&lv_opts->add_size,
					     &lv_opts->add_extents,
					     group->vg->pe_size);
	if (rc) {
		LOG_ERROR("Error verifying region expansion options\n");
		goto out;
	}

	/* Make sure extents is divisible by the number of stripes. We need to
	 * use an equal number of PEs on each PV. On a non-striped volume this
	 * test will always fail.
	 */
	rc = lv_opts->add_extents % volume->lv->lv_stripes;
	if (rc) {
		lv_opts->add_extents += (volume->lv->lv_stripes - rc);
		lv_opts->add_size = lv_opts->add_extents * group->vg->pe_size;
		LOG_WARNING("Rounding size up to stripes boundary: %d\n",
			    lv_opts->add_size);
		rc = 0;
	}

	/* 16-bit extent numbers mean we can't have more than
	 * 65536 extents in a volume.
	 */
	if (lv_opts->add_extents + volume->lv->lv_allocated_le > LVM_PE_T_MAX) {
		LOG_ERROR("Desired final region size (%d extents) too large\n",
			  lv_opts->add_extents + volume->lv->lv_allocated_le);
		LOG_ERROR("Maximum of %d extents per region allowed\n",
			  LVM_PE_T_MAX);
		rc = ENOSPC;
		goto out;
	}

	/* Check for sufficient space in the group. This is only a
	 * rough check. Space constraints on striping 
	 * will happen later.
	 */
	if (lv_opts->add_extents > group->freespace->lv->lv_allocated_le) {
		LOG_ERROR("Not enough freespace in container %s\n",
			  group->container->name);
		LOG_ERROR("Specified additional size of: %d sectors\n",
			  lv_opts->add_size);
		LOG_ERROR("Available space: %d sectors\n",
			  group->freespace->lv->lv_size);
		rc = ENOSPC;
		goto out;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_expand_region_allocate_options
 *
 * This is a helper function for lvm_init_options. It sets up the initial
 * information in the option descriptor for an Expand task.
 **/
int lvm_expand_region_allocate_option_descriptor(option_desc_array_t * od)
{
	LOG_ENTRY();

	od->count = LVM_OPTION_EXPAND_PV_NAMES_INDEX + 1;

	/* Option 0 is expansion extents. */
	SET_STRING(od->option[LVM_OPTION_EXPAND_EXTENTS_INDEX].name, LVM_OPTION_EXPAND_EXTENTS_STR);
	SET_STRING(od->option[LVM_OPTION_EXPAND_EXTENTS_INDEX].title, "Additional Extents");
	SET_STRING(od->option[LVM_OPTION_EXPAND_EXTENTS_INDEX].tip, "Number of extents to add to the selected LVM region. Only specify extents or size!");
	od->option[LVM_OPTION_EXPAND_EXTENTS_INDEX].type = EVMS_Type_Unsigned_Int32;
	od->option[LVM_OPTION_EXPAND_EXTENTS_INDEX].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED |
							    EVMS_OPTION_FLAGS_NO_INITIAL_VALUE |
							    EVMS_OPTION_FLAGS_AUTOMATIC;

	/* Option 1 is expansion size. */
	SET_STRING(od->option[LVM_OPTION_EXPAND_SIZE_INDEX].name, LVM_OPTION_EXPAND_SIZE_STR);
	SET_STRING(od->option[LVM_OPTION_EXPAND_SIZE_INDEX].title, "Additional Size");
	SET_STRING(od->option[LVM_OPTION_EXPAND_SIZE_INDEX].tip, "Amount of space to add to the selected LVM region");
	od->option[LVM_OPTION_EXPAND_SIZE_INDEX].type = EVMS_Type_Unsigned_Int32;
	od->option[LVM_OPTION_EXPAND_SIZE_INDEX].unit = EVMS_Unit_Sectors;
	od->option[LVM_OPTION_EXPAND_SIZE_INDEX].flags = EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

	/* Option 2 is PV Names. */
	SET_STRING(od->option[LVM_OPTION_EXPAND_PV_NAMES_INDEX].name, LVM_OPTION_EXPAND_PV_NAMES_STR);
	SET_STRING(od->option[LVM_OPTION_EXPAND_PV_NAMES_INDEX].title, "Objects (PVs) to expand the region onto");
	SET_STRING(od->option[LVM_OPTION_EXPAND_PV_NAMES_INDEX].tip, "Region will be expanded only onto these objects. Leave blank for automatic allocation.");
	od->option[LVM_OPTION_EXPAND_PV_NAMES_INDEX].type = EVMS_Type_String;
	od->option[LVM_OPTION_EXPAND_PV_NAMES_INDEX].min_len = 1;
	od->option[LVM_OPTION_EXPAND_PV_NAMES_INDEX].max_len = EVMS_VOLUME_NAME_SIZE;
	od->option[LVM_OPTION_EXPAND_PV_NAMES_INDEX].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED |
							     EVMS_OPTION_FLAGS_AUTOMATIC |
							     EVMS_OPTION_FLAGS_VALUE_IS_LIST;
	od->option[LVM_OPTION_EXPAND_PV_NAMES_INDEX].value.list = EngFncs->engine_alloc(sizeof(value_list_t) +
											MAX_PV*sizeof(value_t));
	od->option[LVM_OPTION_EXPAND_PV_NAMES_INDEX].value.list->count	= 0;
	/* Don't allocate the memory for the list strings yet.
	 * Wait until set_option to do that.
	 */

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * lvm_expand_region_init_options
 *
 * Examine the selected_objects list in the task, and verify that each
 * object is valid for using to expand a region. If any are not valid,
 * remove them from the selected_objects and add them to the
 * declined_objects.
 **/
int lvm_expand_region_init_options(task_context_t * context)
{
	lvm_logical_volume_t * volume = context->object->private_data;
	lvm_volume_group_t * group = volume->group;
	lvm_logical_volume_t * freespace = group->freespace;
	option_desc_array_t * od = context->option_descriptors;
	u_int32_t pe_size = group->vg->pe_size;
	u_int32_t add_extents;
	sector_count_t add_sectors;
	int i, j, rc = 0;

	LOG_ENTRY();

	/* Calculate maximum number of extents that can be added. */
	add_extents = freespace->lv->lv_allocated_le;
	if (add_extents + volume->lv->lv_allocated_le > LVM_PE_T_MAX) {
		add_extents = LVM_PE_T_MAX - volume->lv->lv_allocated_le;
	}

	/* Round down to the nearest multiple of the stripe size. */
	if (volume->lv->lv_stripes > 1) {
		add_extents -= add_extents % volume->lv->lv_stripes;
	}

	/* Check the additional space with the engine. */
	add_sectors = add_extents * pe_size;
	rc = EngFncs->can_expand_by(context->object, &add_sectors);
	if (rc == EAGAIN) {
		if (add_sectors < pe_size) {
			LOG_ERROR("Unable to expand region %s.\n",
				  volume->region->name);
			LOG_ERROR("The Engine will only allow expanding by %"PRIu64" sectors,\n",
				  add_sectors);
			LOG_ERROR("but LVM must expand the region by at least %d sectors.\n",
				  pe_size);
			rc = ENOSPC;
			goto out;
		} else if (add_sectors < add_extents * pe_size) {
			add_extents = add_sectors / pe_size;
			rc = 0;
		}
	} else if (rc) {
		LOG_ERROR("A parent object or fsim has disallowed the expand of region %s\n",
			  context->object->name);
		goto out;
	}

	LOG_EXTRA("Setting selected object %s\n", freespace->region->name);

	/* Now that we have a selected freespace, we can update the option
	 * descriptor appropriately.
	 */

	/* Update range of extents and set initial value. */
	od->option[LVM_OPTION_EXPAND_EXTENTS_INDEX].constraint_type = EVMS_Collection_Range;
	SET_RANGE32(od->option[LVM_OPTION_EXPAND_EXTENTS_INDEX].constraint.range, volume->lv->lv_stripes, add_extents, volume->lv->lv_stripes);
	od->option[LVM_OPTION_EXPAND_EXTENTS_INDEX].value.ui32 = add_extents;
	od->option[LVM_OPTION_EXPAND_EXTENTS_INDEX].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

	/* Update range of LV size and set initial value. */
	od->option[LVM_OPTION_EXPAND_SIZE_INDEX].constraint_type = EVMS_Collection_Range;
	SET_RANGE32(od->option[LVM_OPTION_EXPAND_SIZE_INDEX].constraint.range, pe_size * volume->lv->lv_stripes, pe_size * add_extents, pe_size * volume->lv->lv_stripes);
	od->option[LVM_OPTION_EXPAND_SIZE_INDEX].value.ui32 = add_extents * pe_size;
	od->option[LVM_OPTION_EXPAND_SIZE_INDEX].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

	/* Setup list of possible PVs. */
	od->option[LVM_OPTION_EXPAND_PV_NAMES_INDEX].constraint_type = EVMS_Collection_List;
	od->option[LVM_OPTION_EXPAND_PV_NAMES_INDEX].constraint.list = EngFncs->engine_alloc(sizeof(value_list_t) + group->pv_count*sizeof(value_t));
	for (i = 1, j = 0; i < MAX_PV; i++) {
		if (group->pv_list[i] &&
		    lvm_pv_has_available_extents(group->pv_list[i])) {
			SET_STRING(od->option[LVM_OPTION_EXPAND_PV_NAMES_INDEX].constraint.list->value[j].s, group->pv_list[i]->segment->name);
			j++;
		}
	}
	od->option[LVM_OPTION_EXPAND_PV_NAMES_INDEX].constraint.list->count = j;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_expand_region_set_option
 *
 * This is a helper function for lvm_set_option. It sets one particular
 * option in the option descriptor for an Expand task, and updates other
 * option descriptor information accordingly.
 **/
int lvm_expand_region_set_option(task_context_t * context,
				 u_int32_t index,
				 value_t * value,
				 task_effect_t * effect)
{
	option_desc_array_t * od = context->option_descriptors;
	lvm_logical_volume_t * volume = context->object->private_data;
	lvm_volume_group_t * group = volume->group;
	lvm_logical_volume_t * freespace = group->freespace;
	u_int32_t pe_size = group->vg->pe_size;
	u_int32_t add_extents = 0;
	u_int32_t add_size = 0;
	sector_count_t add_sectors;
	int i, rc = 0;

	LOG_ENTRY();

	/* Calculate maximum number of extents that can be added. */
	add_extents = freespace->lv->lv_allocated_le;
	if ( add_extents + volume->lv->lv_allocated_le > LVM_PE_T_MAX ) {
		add_extents = LVM_PE_T_MAX - volume->lv->lv_allocated_le;
	}

	LOG_EXTRA("Setting option %d\n", index);

	switch (index) {

	case LVM_OPTION_EXPAND_EXTENTS_INDEX:
		/* Make sure there are enough extents. */
		if (value->ui32 > add_extents) {
			LOG_ERROR("%d extents chosen. Only %d available.\n",
				  value->ui32, add_extents);
			value->ui32 = add_extents;
			*effect |= EVMS_Effect_Inexact;
		}

		/* Check the additional space with the engine. */
		add_sectors = value->ui32 * pe_size;
		rc = EngFncs->can_expand_by(context->object, &add_sectors);
		if (rc == EAGAIN) {
			if (add_sectors < pe_size * volume->lv->lv_stripes) {
				LOG_ERROR("Unable to expand region %s.\n",
					  context->object->name);
				LOG_ERROR("The Engine will only allow expanding by %"PRIu64" sectors,\n",
					  add_sectors);
				LOG_ERROR("but LVM must expand the region by at least %d sectors.\n",
					  pe_size);
				rc = ENOSPC;
				goto out;
			} else if (add_sectors < value->ui32 * pe_size) {
				value->ui32 = add_sectors / pe_size;
				if (volume->lv->lv_stripes > 1) {
					value->ui32 -= value->ui32 % volume->lv->lv_stripes;
				}
				*effect |= EVMS_Effect_Inexact;
				LOG_ERROR("A parent object or fsim has restricted the expand size for region %s.\n",
					  context->object->name);
				LOG_ERROR("Rounding down to %d extents.\n",
					  value->ui32);
			}
			rc = 0;
		} else if (rc) {
			LOG_ERROR("A parent object or fsim has disallowed the expand of region %s\n",
				  context->object->name);
			goto out;
		}

		od->option[index].value.ui32 = value->ui32;
		od->option[index].flags &= ~(EVMS_OPTION_FLAGS_NOT_REQUIRED |
					     EVMS_OPTION_FLAGS_NO_INITIAL_VALUE);

		/* Recalculate size based on extents. */
		add_size = value->ui32 * pe_size;
		od->option[LVM_OPTION_EXPAND_SIZE_INDEX].value.ui32 = add_size;
		od->option[LVM_OPTION_EXPAND_SIZE_INDEX].flags |= EVMS_OPTION_FLAGS_NOT_REQUIRED;
		od->option[LVM_OPTION_EXPAND_SIZE_INDEX].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

		*effect |= EVMS_Effect_Reload_Options;
		break;

	case LVM_OPTION_EXPAND_SIZE_INDEX:
		/* Make sure lv_size is a multiple of the PE size. */
		if ( lvm_check_lv_size(&value->ui32, pe_size) ) {
			*effect |= EVMS_Effect_Inexact;
		}

		/* Make sure there is enough space. */
		add_size = add_extents * pe_size;
		if ( value->ui32 > add_size ) {
			LOG_ERROR("%d sectors chosen for size. Only %d available.\n",
				  value->ui32, add_size);
			value->ui32 = add_size;
			*effect |= EVMS_Effect_Inexact;
		}

		/* Check the additional space with the engine. */
		add_sectors = value->ui32;
		rc = EngFncs->can_expand_by(context->object, &add_sectors);
		if ( rc == EAGAIN ) {
			if ( add_sectors < pe_size * volume->lv->lv_stripes) {
				LOG_ERROR("Unable to expand region %s.\n",
					  context->object->name);
				LOG_ERROR("The Engine will only allow expanding by %"PRIu64" sectors,\n",
					  add_sectors);
				LOG_ERROR("but LVM must expand the region by at least %d sectors.\n",
					  pe_size);
				rc = ENOSPC;
				goto out;
			} else if ( add_sectors < value->ui32 ) {
				value->ui32 = add_sectors;
				if (volume->lv->lv_stripes > 1) {
					value->ui32 -= value->ui32 %
						       (pe_size * volume->lv->lv_stripes);
				}
				lvm_check_lv_size(&value->ui32, pe_size);
				*effect |= EVMS_Effect_Inexact;
				LOG_ERROR("A parent object or fsim has restricted the expand size for region %s.\n",
					  context->object->name);
				LOG_ERROR("Rounding down to %d sectors.\n",
					  value->ui32);
			}
			rc = 0;
		} else if (rc) {
			LOG_ERROR("A parent object or fsim has disallowed the expand of region %s\n",
				  context->object->name);
			goto out;
		}

		od->option[index].value.ui32 = value->ui32;
		od->option[index].flags &= ~(EVMS_OPTION_FLAGS_NOT_REQUIRED|EVMS_OPTION_FLAGS_NO_INITIAL_VALUE);

		/* Recalculate extents based on lv_size. */
		add_extents = value->ui32 / pe_size;
		od->option[LVM_OPTION_EXPAND_EXTENTS_INDEX].value.ui32 = add_extents;
		od->option[LVM_OPTION_EXPAND_EXTENTS_INDEX].flags |= EVMS_OPTION_FLAGS_NOT_REQUIRED;
		od->option[LVM_OPTION_EXPAND_EXTENTS_INDEX].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

		*effect |= EVMS_Effect_Reload_Options;
		break;

	case LVM_OPTION_EXPAND_PV_NAMES_INDEX:
		for ( i = 0; i < value->list->count; i++ ) {
			SET_STRING(od->option[index].value.list->value[i].s, value->list->value[i].s);
		}
		for ( ; i < od->option[index].value.list->count; i++ ) {
			if ( od->option[index].value.list->value[i].s ) {
				EngFncs->engine_free(od->option[index].value.list->value[i].s);
				od->option[index].value.list->value[i].s = NULL;
			}
		}
		od->option[index].value.list->count = value->list->count;
		break;

	default:
		rc = EINVAL;
		break;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}



/****** Logical Volume Shrink Option Handlers ******/


/**
 * lvm_shrink_region_parse_option_array
 *
 * Parse the option_array_t to find options pertaining to logical volume
 * shrink. Each option can either be keyed by value or string, so we
 * have to check for both. Acceptable options for volume shrink are:
 * LVM_OPTION_SHRINK_SIZE
 * LVM_OPTION_SHRINK_EXTENTS
 * Incorrect options are simply ignored.
 **/
int lvm_shrink_region_parse_option_array(option_array_t *options,
					 lvm_volume_group_t *group,
					 lvm_logical_volume_t *volume,
					 u_int32_t *remove_extents )
{
	u_int32_t extents = 0;
	u_int32_t size = 0;
	int i, rc;

	LOG_ENTRY();

	/* Initialize the options. */
	*remove_extents = 0;

	for ( i = 0; i < options->count; i++ ) {

		/* If only the name-based index is specified, get the number. */
		if ( ! options->option[i].is_number_based ) {
			if ( ! strcmp(options->option[i].name, LVM_OPTION_SHRINK_EXTENTS_STR) ) {
				options->option[i].number = LVM_OPTION_SHRINK_EXTENTS_INDEX;
			} else if ( ! strcmp(options->option[i].name, LVM_OPTION_SHRINK_SIZE_STR) ) {
				options->option[i].number = LVM_OPTION_SHRINK_SIZE_INDEX;
			} else {
				continue;
			}
		}

		LOG_EXTRA("Parsing option %d\n", options->option[i].number);

		/* Use the number-based index to record the option. */
		switch ( options->option[i].number ) {
		case LVM_OPTION_SHRINK_EXTENTS_INDEX:
			extents = options->option[i].value.ui32;
			break;
		case LVM_OPTION_SHRINK_SIZE_INDEX:
			size = options->option[i].value.ui32;
			break;
		default:
			break;
		}
	}

	rc = lvm_shrink_region_verify_options(&extents, &size, group, volume);

	*remove_extents = extents;

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_shrink_region_verify_options
 *
 * Run through the options that have been collected, and verify that they
 * are all valid for using in the specified group. Correct any bad options
 * if possible, or return an error.
 **/
int lvm_shrink_region_verify_options(u_int32_t * extents,
				     u_int32_t * size,
				     lvm_volume_group_t * group,
				     lvm_logical_volume_t * volume)
{
	int rc = 0;

	LOG_ENTRY();

	/* Make sure size is a multiple of the PE size. If it is
	 * currently zero, this won't change anything.
	 */
	lvm_check_lv_size(size, group->vg->pe_size);

	/* Check/set extents and size. This only checks that size and
	 * extents match.
	 */
	rc = lvm_compare_lv_size_and_extents(size, extents, group->vg->pe_size);
	if (rc) {
		LOG_ERROR("Error verifying region shrink options\n");
		goto out;
	}

	/* Make sure extents is divisible by the number of stripes. We need to
	 * use an equal number of PEs on each PV. On a non-striped volume this
	 * test will always fail.
	 */
	rc = *extents % volume->lv->lv_stripes;
	if (rc) {
		*extents -= rc;
		*size = *extents * group->vg->pe_size;
		LOG_ERROR("Rounding size down to stripes boundary: %d sectors\n", *size);
		rc = 0;
	}

	/* Make sure we are shrinking by at least one extent.
	 * Kind of silly otherwise.
	 */
	if ( *extents == 0 ) {
		LOG_ERROR("Specified zero extents to remove. Please shrink by a non-zero amount.\n");
		rc = EINVAL;
		goto out;
	}

	/* Make sure there is at least one extent left in the volume. To
	 * remove all extents, use the delete function.
	 */
	if ( *extents >= volume->lv->lv_allocated_le ) {
		LOG_ERROR("Cannot shrink region %s to zero size\n", volume->region->name);
		rc = EINVAL;
		goto out;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_shrink_region_allocate_option_descriptor
 *
 * This is a helper function for lvm_init_options. It sets up the initial
 * information in the option descriptor for a Shrink task.
 **/
int lvm_shrink_region_allocate_option_descriptor(option_desc_array_t * od)
{
	LOG_ENTRY();

	od->count = LVM_OPTION_SHRINK_SIZE_INDEX + 1;

	/* Option 0 is shrink extents. */
	SET_STRING(od->option[LVM_OPTION_SHRINK_EXTENTS_INDEX].name, LVM_OPTION_SHRINK_EXTENTS_STR);
	SET_STRING(od->option[LVM_OPTION_SHRINK_EXTENTS_INDEX].title, "Shrink by Extents");
	SET_STRING(od->option[LVM_OPTION_SHRINK_EXTENTS_INDEX].tip, "Number of extents to remove from the selected LVM region. Only specify extents or size!");
	od->option[LVM_OPTION_SHRINK_EXTENTS_INDEX].type	= EVMS_Type_Unsigned_Int32;
	od->option[LVM_OPTION_SHRINK_EXTENTS_INDEX].flags	= EVMS_OPTION_FLAGS_NOT_REQUIRED |
								  EVMS_OPTION_FLAGS_NO_INITIAL_VALUE |
								  EVMS_OPTION_FLAGS_AUTOMATIC;

	/* Option 1 is shrink size. */
	SET_STRING(od->option[LVM_OPTION_SHRINK_SIZE_INDEX].name, LVM_OPTION_SHRINK_SIZE_STR);
	SET_STRING(od->option[LVM_OPTION_SHRINK_SIZE_INDEX].title, "Shrink by Size");
	SET_STRING(od->option[LVM_OPTION_SHRINK_SIZE_INDEX].tip, "Amount of space to remove from the selected LVM region");
	od->option[LVM_OPTION_SHRINK_SIZE_INDEX].type	= EVMS_Type_Unsigned_Int32;
	od->option[LVM_OPTION_SHRINK_SIZE_INDEX].unit	= EVMS_Unit_Sectors;
	od->option[LVM_OPTION_SHRINK_SIZE_INDEX].flags	= EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * lvm_shrink_region_init_options
 **/
int lvm_shrink_region_init_options(task_context_t * context)
{
	lvm_logical_volume_t	* volume = context->object->private_data;
	option_desc_array_t	* od = context->option_descriptors;
	u_int32_t		pe_size = volume->group->vg->pe_size;
	u_int32_t		remove_extents = volume->lv->lv_allocated_le - 1;
	sector_count_t		remove_sectors;
	int			rc;

	LOG_ENTRY();

	/* Round down to the nearest multiple of the stripe size. */
	if (volume->lv->lv_stripes > 1) {
		remove_extents -= remove_extents % volume->lv->lv_stripes;
	}

	/* Check the removal space with the engine. */
	remove_sectors = remove_extents * pe_size;
	rc = EngFncs->can_shrink_by(context->object, &remove_sectors);
	if ( rc == EAGAIN ) {
		if ( remove_sectors < pe_size ) {
			LOG_ERROR("Unable to shrink region %s.\n", context->object->name);
			LOG_ERROR("The Engine will only allow shrinking by %"PRIu64" sectors,\n", remove_sectors);
			LOG_ERROR("but LVM must shrink the region by at least %d sectors.\n", pe_size);
			rc = ENOSPC;
			goto out;
		} else if ( remove_sectors < remove_extents * pe_size ) {
			remove_extents = remove_sectors / pe_size;
		}
		rc = 0;
	} else if (rc) {
		LOG_ERROR("A parent object or fsim has disallowed the shrink of region %s\n", context->object->name);
		goto out;
	}

	/* Update range of extents and set initial value to minimum. */
	od->option[LVM_OPTION_SHRINK_EXTENTS_INDEX].constraint_type = EVMS_Collection_Range;
	SET_RANGE32(od->option[LVM_OPTION_SHRINK_EXTENTS_INDEX].constraint.range, volume->lv->lv_stripes, remove_extents, volume->lv->lv_stripes);
	od->option[LVM_OPTION_SHRINK_EXTENTS_INDEX].value.ui32 = volume->lv->lv_stripes;
	od->option[LVM_OPTION_EXPAND_EXTENTS_INDEX].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

	/* Update range of LV size and set initial value to minimum. */
	od->option[LVM_OPTION_SHRINK_SIZE_INDEX].constraint_type = EVMS_Collection_Range;
	SET_RANGE32(od->option[LVM_OPTION_SHRINK_SIZE_INDEX].constraint.range, pe_size * volume->lv->lv_stripes, pe_size * remove_extents, pe_size * volume->lv->lv_stripes);
	od->option[LVM_OPTION_SHRINK_SIZE_INDEX].value.ui32 = pe_size;
	od->option[LVM_OPTION_SHRINK_SIZE_INDEX].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_shrink_region_set_option
 *
 * This is a helper function for lvm_set_option. It sets one particular
 * option in the option descriptor for a Shrink task, and updates other
 * option descriptor information accordingly.
 **/
int lvm_shrink_region_set_option(task_context_t * context,
				 u_int32_t index,
				 value_t * value,
				 task_effect_t * effect)
{
	option_desc_array_t * od = context->option_descriptors;
	lvm_logical_volume_t * volume = context->object->private_data;
	lvm_volume_group_t * group = volume->group;
	u_int32_t pe_size = group->vg->pe_size;
	u_int32_t remove_extents = 0;
	u_int32_t remove_size = 0;
	sector_count_t remove_sectors = 0;
	int rc = 0;

	LOG_ENTRY();

	/* Calculate maximum number of extents that can be removed. */
	remove_extents = volume->lv->lv_allocated_le - 1;

	LOG_EXTRA("Setting option %d\n", index);

	switch (index) {

	case LVM_OPTION_SHRINK_EXTENTS_INDEX:
		/* Make sure there are enough extents. */
		if ( value->ui32 > remove_extents ) {
			LOG_ERROR("%d extents chosen. Only %d allowed.\n", value->ui32, remove_extents);
			value->ui32 = remove_extents;
			*effect |= EVMS_Effect_Inexact;
		}

		/* Check the removal space with the engine. */
		remove_sectors = value->ui32 * pe_size;
		rc = EngFncs->can_shrink_by(context->object, &remove_sectors);
		if ( rc == EAGAIN ) {
			if ( remove_sectors < pe_size * volume->lv->lv_stripes) {
				LOG_ERROR("Unable to shrink region %s.\n", context->object->name);
				LOG_ERROR("The Engine will only allow shrinking by %"PRIu64" sectors,\n", remove_sectors);
				LOG_ERROR("but LVM must shrink the region by at least %d sectors.\n", pe_size);
				rc = ENOSPC;
				goto out;
			} else if ( remove_sectors < value->ui32 * pe_size ) {
				value->ui32 = remove_sectors / pe_size;
				if (volume->lv->lv_stripes > 1) {
					value->ui32 -= value->ui32 % volume->lv->lv_stripes;
				}
				*effect |= EVMS_Effect_Inexact;
				LOG_ERROR("A parent object or fsim has restricted the shrink size for region %s.\n", context->object->name);
				LOG_ERROR("Rounding down to %d extents.\n", value->ui32);
			}
			rc = 0;
		} else if (rc) {
			LOG_ERROR("A parent object or fsim has disallowed the shrink of region %s\n", context->object->name);
			break;
		}

		od->option[index].value.ui32 = value->ui32;
		od->option[index].flags &= ~(EVMS_OPTION_FLAGS_NOT_REQUIRED|EVMS_OPTION_FLAGS_NO_INITIAL_VALUE);

		/* Recalculate size based on extents. */
		remove_size = value->ui32 * pe_size;
		od->option[LVM_OPTION_SHRINK_SIZE_INDEX].value.ui32 = remove_size;
		od->option[LVM_OPTION_SHRINK_SIZE_INDEX].flags |= EVMS_OPTION_FLAGS_NOT_REQUIRED;
		od->option[LVM_OPTION_SHRINK_SIZE_INDEX].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

		*effect |= EVMS_Effect_Reload_Options;
		break;

	case LVM_OPTION_SHRINK_SIZE_INDEX:
		/* Make sure the size is a multiple of the PE size. */
		if ( lvm_check_lv_size(&value->ui32, pe_size) ) {
			*effect |= EVMS_Effect_Inexact;
		}

		/* Make sure there is enough space. */
		remove_size = remove_extents * pe_size;
		if ( value->ui32 > remove_size ) {
			LOG_ERROR("%d sectors chosen for size. Only %d allowed.\n", value->ui32, remove_size);
			value->ui32 = remove_size;
			*effect |= EVMS_Effect_Inexact;
		}

		/* Check the removal space with the engine. */
		remove_sectors = value->ui32;
		rc = EngFncs->can_shrink_by(context->object, &remove_sectors);
		if ( rc == EAGAIN ) {
			if ( remove_sectors < pe_size * volume->lv->lv_stripes) {
				LOG_ERROR("Unable to shrink region %s.\n", context->object->name);
				LOG_ERROR("The Engine will only allow shrinking by %"PRIu64" sectors,\n", remove_sectors);
				LOG_ERROR("but LVM must shrink the region by at least %d sectors.\n", pe_size);
				rc = ENOSPC;
				goto out;
			} else if ( remove_sectors < value->ui32 ) {
				value->ui32 = remove_sectors;
				if (volume->lv->lv_stripes > 1) {
					value->ui32 -= value->ui32 % (pe_size * volume->lv->lv_stripes);
				}
				lvm_check_lv_size(&value->ui32, pe_size);
				*effect |= EVMS_Effect_Inexact;
				LOG_ERROR("A parent object or fsim has restricted the shrink size for region %s.\n", context->object->name);
				LOG_ERROR("Rounding down to %d sectors.\n", value->ui32);
			}
			rc = 0;
		} else if (rc) {
			LOG_ERROR("A parent object or fsim has disallowed the shrink of region %s\n", context->object->name);
			break;
		}

		od->option[index].value.ui32 = value->ui32;
		od->option[index].flags &= ~(EVMS_OPTION_FLAGS_NOT_REQUIRED|EVMS_OPTION_FLAGS_NO_INITIAL_VALUE);

		/* Recalculate extents based on lv_size. */
		remove_extents = value->ui32 / pe_size;
		od->option[LVM_OPTION_SHRINK_EXTENTS_INDEX].value.ui32 = remove_extents;
		od->option[LVM_OPTION_SHRINK_EXTENTS_INDEX].flags |= EVMS_OPTION_FLAGS_NOT_REQUIRED;
		od->option[LVM_OPTION_SHRINK_EXTENTS_INDEX].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

		*effect |= EVMS_Effect_Reload_Options;
		break;

	default:
		rc = EINVAL;
		break;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}



/****** Volume Group Expand Option Handlers ******/



/**
 * lvm_expand_container_allocate_option_descriptor
 *
 * This is a helper function for lvm_init_options. It sets up the initial
 * information in the option descriptor for a Expand_Container task.
 **/
void lvm_expand_container_allocate_option_descriptor(option_desc_array_t * od)
{
	LOG_ENTRY();

	od->count = 0;

	LOG_EXIT_VOID();
}

/**
 * lvm_expand_container_get_acceptable
 *
 * This is a helper function for lvm_get_acceptable_objects. It determines
 * which objects in EVMS are appropriate for an Expand_Container task. This
 * is very similar to get_acceptable for create_container, but this
 * function will check each segment for the correct PE size ratio before
 * declaring it as acceptable.
 **/
int lvm_expand_container_get_acceptable(storage_container_t * container,
					list_anchor_t acceptable_segments)
{
	lvm_volume_group_t * group = container->private_data;
	storage_object_t * segment;
	object_search_flags_t flags;
	list_anchor_t object_list;
	list_element_t itr;
	u_int32_t pe_size;
	int rc;

	LOG_ENTRY();

	/* Find all top-level disks, segments, and regions in the system. This
	 * will also retrieve LVM regions, so let the user beware. :)
	 */
	flags = VALID_INPUT_OBJECT | (container->disk_group ? 0 : NO_DISK_GROUP);
	rc = EngFncs->get_object_list(DISK|SEGMENT|REGION, DATA_TYPE, NULL,
				      container->disk_group, flags, &object_list);
	if (rc) {
		goto out;
	}

	LIST_FOR_EACH(object_list, itr, segment) {
		pe_size = group->vg->pe_size;
		rc = lvm_check_segment_for_pe_size(segment, &pe_size);
		if (!rc) {
			rc = lvm_check_segment_for_group_inclusion(segment, group);
			if (!rc) {
				EngFncs->insert_thing(acceptable_segments, segment,
						      INSERT_AFTER, NULL);
			}
		}
	}
	EngFncs->destroy_list(object_list);
	rc = 0;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_expand_container_set_objects
 *
 * Examine the selected_objects list in the task, and verify that each
 * object is valid for inclusion in the selected container. If any are not
 * valid, remove them from the selected_objects and add them to the
 * declined_objects.
 **/
int lvm_expand_container_set_objects(task_context_t * context,
				     list_anchor_t declined_objects,
				     task_effect_t * effect)
{
	lvm_volume_group_t * group = context->container->private_data;
	storage_object_t * segment;
	list_element_t itr;
	u_int32_t pe_size;
	int rc;

	LOG_ENTRY();

	/* Check that all of the segments are valid for use in the
	 * selected group.
	 */
	LIST_FOR_EACH(context->selected_objects, itr, segment) {

		rc = lvm_check_segment_for_group_inclusion(segment, group);
		if (rc) {
			/* FIXME: Add this segment to declined objects. */
			LOG_ERROR("One or more objects are invalid for container expansion\n");
			goto out;
		}

		/* If any of the segments are too small for the default PE
		 * size, try to reset the PE size.
		 */
		pe_size = group->vg->pe_size;
		rc = lvm_check_segment_for_pe_size(segment, &pe_size);
		if (rc) {
			/* FIXME: Add this segment to declined objects. */
			LOG_ERROR("One or more objects are invalid for container expansion\n");
			goto out;
		}
	}
	rc = 0;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_expand_container_set_option
 *
 * This is a helper function for lvm_set_option. Since there are no options
 * for container expansion, this function is mostly a no-op.
 **/
void lvm_expand_container_set_option(task_context_t * context,
				    u_int32_t index,
				    value_t * value,
				    task_effect_t * effect)
{
	LOG_ENTRY();
	LOG_EXIT_VOID();
}




/****** Volume Group Shrink Option Handlers ******/



/**
 * lvm_shrink_container_allocate_option_descriptor
 *
 * This is a helper function for lvm_init_options. It sets up the initial
 * information in the option descriptor for a Shrink_Container task.
 **/
void lvm_shrink_container_allocate_option_descriptor(option_desc_array_t * od)
{
	LOG_ENTRY();

	od->count = 0;

	LOG_EXIT_VOID();
}

/**
 * lvm_shrink_container_get_acceptable
 *
 * This is a helper function for lvm_get_acceptable_objects. It determines
 * which objects in EVMS are appropriate for a Shrink_Container task.
 **/
void lvm_shrink_container_get_acceptable(storage_container_t * container,
					 list_anchor_t acceptable_segments)
{
	lvm_volume_group_t * group = container->private_data;
	int rc, i;

	LOG_ENTRY();

	/* Run through the list of PVs in this container, and determine which
	 * ones can be removed.
	 */
	for ( i = 1; i <= MAX_PV; i++ ) {
		if ( group->pv_list[i] ) {
			rc = my_plugin_record->container_functions->can_remove_object(group->pv_list[i]->segment);
			if (!rc) {
				EngFncs->insert_thing(acceptable_segments,
						      group->pv_list[i]->segment,
						      INSERT_AFTER, NULL);
			}
		}
	}

	LOG_EXIT_VOID();
}

/**
 * lvm_shrink_container_set_objects
 *
 * Examine the selected_objects list in the task, and verify that each
 * object is valid for inclusion in the selected container. If any are not
 * valid, remove them from the selected_objects and add them to the
 * declined_objects.
 **/
int lvm_shrink_container_set_objects(task_context_t * context,
				     list_anchor_t declined_objects,
				     task_effect_t * effect)
{
	lvm_volume_group_t * group = context->container->private_data;
	storage_object_t * segment;
	list_element_t itr;
	int rc, count;

	LOG_ENTRY();

	count = EngFncs->list_count(context->selected_objects);
	if ( count >= group->vg->pv_cur ) {
		LOG_ERROR("Cannot select all PVs in group %s for removal.\n", group->container->name);
		rc = EINVAL;
		goto out;
	}

	/* Check that all of the selected segments can be removed from
	 * the selected group.
	 */
	LIST_FOR_EACH(context->selected_objects, itr, segment) {
		rc = my_plugin_record->container_functions->can_remove_object(segment);
		if (rc) {
			/* FIXME: Add this segment to declined objects. */
			LOG_ERROR("One or more objects are invalid for container shrink.\n");
			goto out;
		}
	}
	rc = 0;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_shrink_container_set_option
 *
 * This is a helper function for lvm_set_option. Since there are no options
 * for container shrink, this function is mostly a no-op.
 **/
void lvm_shrink_container_set_option(task_context_t * context,
				     u_int32_t index,
				     value_t * value,
				     task_effect_t * effect)
{
	LOG_ENTRY();
	LOG_EXIT_VOID();
}

