/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libgpt.so
 *
 *   File: discovery.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "gpt.h"
#include "discovery.h"
#include "checks.h"
#include "helpers.h"
#include "dm.h"


static int   DisplayDiskSeg( storage_object_t *seg )
{
        char       Type[21];
        int        rc;
        u_int32_t  StartLBA;
        u_int32_t  EndLBA;
        u_int32_t  SectorCount;
        char       Name[128];



        if (seg) {

                StartLBA    = seg->start;
                EndLBA      = seg->start + seg->size -1;
                SectorCount = seg->size;

                if (seg->name) {
                        strncpy(Name, seg->name, EVMS_NAME_SIZE );
                }
                else {
                        strcpy(Name, "n/a");
                }

                switch (seg->data_type) {
                case   META_DATA_TYPE:
                        strcpy(Type, "Meta Data");
                        break;

                case   DATA_TYPE:
                        strcpy(Type, "Data Seg");
                        break;

                case   FREE_SPACE_TYPE:
                        strcpy(Type, "Free Space");
                        break;

                default:
                        strcpy(Type, "Unknown ");
                        break;
                }

                LOG_DEBUG(" %08d   %08d   %08d   %s\n", StartLBA, EndLBA, SectorCount, Name);

                rc = 0;
        }
        else {
                rc = EPERM;
        }

        return rc;
}

static void  DisplayDiskSegmentList(storage_object_t *object)
{
        list_element_t iter;
        storage_object_t *seg;

        LOG_DEBUG("\tGPT Segment List ... \n");
        LOG_DEBUG("Start LBA    End LBA    Sectors   SegName\n");

        LIST_FOR_EACH( object->parent_objects,iter, seg ) {
                DisplayDiskSeg(seg);
        }

}



/*
 *  Called to convert a GPT header from disk format to the arch specific
 *  format.
 */
static void disk_gpt_header_to_cpu( gpt_header *gh )
{
        if (gh) {

                gh->signature              = DISK_TO_CPU64(gh->signature);
                gh->version                = DISK_TO_CPU32(gh->version);
                gh->size                   = DISK_TO_CPU32(gh->size);
                gh->crc                    = DISK_TO_CPU32(gh->crc);
                gh->reserve                = DISK_TO_CPU32(gh->reserve);
                gh->my_lba                 = DISK_TO_CPU64(gh->my_lba);
                gh->alternate_lba          = DISK_TO_CPU64(gh->alternate_lba);
                gh->start_useable          = DISK_TO_CPU64(gh->start_useable);
                gh->end_useable            = DISK_TO_CPU64(gh->end_useable);

                gh->disk_id.time_low       = DISK_TO_CPU32(gh->disk_id.time_low);
                gh->disk_id.time_mid       = DISK_TO_CPU16(gh->disk_id.time_mid);
                gh->disk_id.time_high      = DISK_TO_CPU16(gh->disk_id.time_high);

                gh->ptable_lba             = DISK_TO_CPU64(gh->ptable_lba);
                gh->ptable_count           = DISK_TO_CPU32(gh->ptable_count);
                gh->ptable_entry_size      = DISK_TO_CPU32(gh->ptable_entry_size);
                gh->ptable_crc             = DISK_TO_CPU32(gh->ptable_crc);

        }

}


static void Display_GPT_Header( gpt_header *gh )
{
        if (gh) {

                LOG_DEBUG("GPT HEADER INFO ...\n");
                LOG_DEBUG("     Signature         = 0x%"PRIX64"\n", gh->signature );
                LOG_DEBUG("     Version           = 0x%X\n", gh->version );
                LOG_DEBUG("     Header Size       = %d\n", gh->size);
                LOG_DEBUG("     CRC               = 0x%X\n", gh->crc);
                LOG_DEBUG("     My LBA            = %"PRIu64"\n", gh->my_lba);
                LOG_DEBUG("     Alternate         = %"PRIu64"\n", gh->alternate_lba);
                LOG_DEBUG("     Start Useable     = %"PRIu64"\n", gh->start_useable);
                LOG_DEBUG("     End Useable       = %"PRIu64"\n", gh->end_useable);

                /*
                gh->disk_id.time_low       = DISK_TO_CPU32(gh->disk_id.time_low);
                gh->disk_id.time_mid       = DISK_TO_CPU16(gh->disk_id.time_mid);
                gh->disk_id.time_high      = DISK_TO_CPU16(gh->disk_id.time_high);
                */

                LOG_DEBUG("     Ptable LBA        = %"PRIu64"\n", gh->ptable_lba);
                LOG_DEBUG("     Ptable Count      = %d\n", gh->ptable_count);
                LOG_DEBUG("     Ptable Entry Size = %d\n", gh->ptable_entry_size);
                LOG_DEBUG("     Ptable CRC        = 0x%X\n", gh->ptable_crc);
        }

}


static void Display_GPT_Partition_Record( gpt_partition *part)
{
        if (part) {

                LOG_DEBUG("GPT PARTITION INFO ...\n");
                LOG_DEBUG("     Type ... \n");
                LOG_DEBUG("        Time  Low   = 0x%X\n", DISK_TO_CPU32(part->type.time_low) );
                LOG_DEBUG("        Time  Mid   = 0x%X\n", DISK_TO_CPU16(part->type.time_mid) );
                LOG_DEBUG("        Time High   = 0x%X\n", DISK_TO_CPU16(part->type.time_high) );
                LOG_DEBUG("        Clk  High   = 0x%X\n", part->type.clock_seq_high );
                LOG_DEBUG("        Clk   Low   = 0x%X\n", part->type.clock_seq_low );
                LOG_DEBUG("        Node        = %X %X %X %X %X %X\n",
                          part->type.node[0], part->type.node[1], part->type.node[2],
                          part->type.node[3], part->type.node[4], part->type.node[5] );
                LOG_DEBUG("     ID ... \n");
                LOG_DEBUG("        Time  Low   = 0x%X\n", DISK_TO_CPU32(part->id.time_low) );
                LOG_DEBUG("        Time  Mid   = 0x%X\n", DISK_TO_CPU16(part->id.time_mid) );
                LOG_DEBUG("        Time High   = 0x%X\n", DISK_TO_CPU16(part->id.time_high) );
                LOG_DEBUG("        Clk  High   = 0x%X\n", part->id.clock_seq_high );
                LOG_DEBUG("        Clk   Low   = 0x%X\n", part->id.clock_seq_low );
                LOG_DEBUG("        Node        = %X %X %X %X %X %X\n",
                          part->id.node[0], part->id.node[1], part->id.node[2],
                          part->id.node[3], part->id.node[4], part->id.node[5] );
                LOG_DEBUG("     Start LBA         = %"PRIu64"\n", DISK_TO_CPU64(part->start));
                LOG_DEBUG("     End   LBA         = %"PRIu64"\n", DISK_TO_CPU64(part->end));
        }
}


/*
 *  Called to read the specified partition table into memory and validate it.
 */
static gpt_partition *  get_gpt_partition_table( storage_object_t *object, gpt_header *gh)
{
        int rc;
        gpt_partition             *pt;
        struct plugin_functions_s *funcs;
        lsn_t                      lsn;
        sector_count_t             sector_count;
        u_int32_t                  calculated_crc;


        LOG_ENTRY();

        // get objects function table
        funcs = (struct plugin_functions_s *)object->plugin->functions.plugin;
        if (funcs==NULL) {
                LOG_ERROR("error ... no function table found for parent storage object\n");
                LOG_EXIT_PTR(NULL);
                return NULL;
        }

        // our i/o is relative to the start of the evms storage object
        lsn = gh->ptable_lba; // - object->start;

        // get partition table size in sectors
        sector_count = (gh->ptable_count * gh->ptable_entry_size) / EVMS_VSECTOR_SIZE;

        if ( ((gh->ptable_count * gh->ptable_entry_size) % EVMS_VSECTOR_SIZE) != 0) {
                ++sector_count;
        }

        // malloc i/o buffer
        pt = malloc( sector_count * EVMS_VSECTOR_SIZE );

        // now read the partition table into memory and crc it
        if ( pt ) {

                rc = funcs->read( object, lsn, sector_count, pt );
                if ( rc == 0 ) {

                        calculated_crc = EngFncs->calculate_CRC( EVMS_INITIAL_CRC,
                                                                 pt,
                                                                 gh->ptable_count * gh->ptable_entry_size);

                        if ( ~calculated_crc != gh->ptable_crc ) {
                                rc = ENODATA;
                        }

                }

        }
        else {
                rc = ENOMEM;
        }

        // free partition table i/o buffer on failure
        if (rc) {
                if (pt) free (pt);
                pt = NULL;
        }

        LOG_DEBUG("returning ptable ptr = %p\n", pt );
        LOG_EXIT_PTR(pt);
        return pt;
}


/*
 *  Called to see if the specified GPT header seems Ok. This
 *  is just a quick sanity check of the header.
 */
static boolean isa_valid_gpt_header( storage_object_t *object, lba_t lba, gpt_header *gh )
{
        u_int32_t       crc;
        u_int32_t       calculated_crc;
        sector_count_t  sector_count;
        lba_t           object_end_lba = object->start + object->size - 1;

        LOG_ENTRY();

        // check for valid signature
        if ( DISK_TO_CPU64(gh->signature) != GPT_DISKMAGIC ) {
                LOG_DEBUG("header contains invalid signature\n");
                LOG_EXIT_BOOL(FALSE);
                return FALSE;
        }

        // crc the header
        crc = DISK_TO_CPU32(gh->crc);

        gh->crc = 0;

        calculated_crc = ~( EngFncs->calculate_CRC( EVMS_INITIAL_CRC, gh, DISK_TO_CPU32(gh->size) ) );

        gh->crc = CPU_TO_DISK32(crc);

        if ( calculated_crc != crc) {
                LOG_DEBUG("header contains invalid crc\n");
                LOG_EXIT_BOOL(FALSE);
                return FALSE;
        }

        // check that header LBA matches actual LBA
        if ( lba != DISK_TO_CPU64(gh->my_lba) ) {
                LOG_DEBUG("gpt header on %s contains invalid MyLba field\n", object->name);
                LOG_EXIT_BOOL(FALSE);
                return FALSE;
        }

        // check the alternate header LBA against size of object
        if ( DISK_TO_CPU64(gh->end_useable) > object_end_lba  ) {
                LOG_DEBUG("gpt header on %s contains invalid alternate header LBA\n", object->name);
                LOG_EXIT_BOOL(FALSE);
                return FALSE;
        }

        // check on the partition table info
        if ( gh->ptable_count == 0 || gh->ptable_entry_size == 0) {
                LOG_DEBUG("gpt header on %s is missing partition table info\n", object->name);
                LOG_EXIT_BOOL(FALSE);
                return FALSE;
        }

        // get the partition table sector count
        sector_count = (DISK_TO_CPU32(gh->ptable_count) * DISK_TO_CPU32(gh->ptable_entry_size) ) / EVMS_VSECTOR_SIZE;
        if ( (DISK_TO_CPU32(gh->ptable_count) * DISK_TO_CPU32(gh->ptable_entry_size) ) % EVMS_VSECTOR_SIZE ) {
                ++sector_count;
        }

        LOG_EXIT_BOOL(TRUE);
        return TRUE;
}



/*
 *  Called to read the specified sector into memory, verify that it
 *  is a gpt header and return the gpt header to the caller.
 */
static gpt_header *  get_gpt_header( storage_object_t *object, lsn_t lsn )
{
        int                         rc=EINVAL;
        gpt_header                 *gh = NULL;
        struct plugin_functions_s  *funcs;


        LOG_ENTRY();
        LOG_DEBUG("reading gpt header off %s at lsn %"PRIu64"\n", object->name, lsn);

        // get objects function table
        funcs = (struct plugin_functions_s *)object->plugin->functions.plugin;
        if (funcs==NULL) {
                LOG_ERROR("error ... no function table found for parent storage object\n");
                LOG_EXIT_INT(0);
                return 0;
        }

        // get specified gpt header if we can
        gh = (gpt_header *) malloc( EVMS_VSECTOR_SIZE );
        if ( gh ) {

                rc = funcs->read( object, lsn, 1, (void *) gh );
                if ( rc == 0 ) {

                        if ( isa_valid_gpt_header(object, lsn, gh)==FALSE ) {  //(lba_t) object->start+lsn,  gh ) == FALSE ) {
                                rc = ENODATA;
                        }

                }

        }

        if (rc) {
                if (gh) free(gh);
                gh = NULL;
        }

        LOG_EXIT_PTR(gh);
        return gh;
}


/*
 *  Called to probe the storage object to see if it has a
 *  valid GPT header.
 */
static boolean isa_object_with_valid_gpt_header( LOGICALDISK  *ld )
{
        gpt_header  *gh  = NULL;
        char * choices[] = {"Yes", "No", NULL};
        int answer = 0;     // Initialize to Revert


        LOG_ENTRY();

        // get primary gpt header
        gh = get_gpt_header(ld, 1 );

        // if we failed to get primary then ... get alternate gpt header - last sector on object
        if (gh == NULL) {
                gh = get_gpt_header(ld, ld->size - 1 );
                if (gh) {

                        QUESTION( &answer, choices,
                                  "\nErrors were found with the partition information on drive %s.\n\n"
                                  "The primary GPT header is missing or invalid but the alternate GPT header was discovered on the drive.\n\n"
                                  "The GPT segment manager can be assigned to the drive and attempt to restore missing GPT metadata, using "
                                  "the alternate GPT header to provide information about the drive.\n\n"
                                  "Question: Would you like to GPT segment manager to be assigned to this drive?\n",
                                  ld->name);

                        if (answer == 1) {
                                free(gh);
                                gh=NULL;
                        }

                }
        }

        if (gh) {
                free(gh);
                LOG_EXIT_BOOL(TRUE);
                return TRUE;
        }
        else {
                LOG_EXIT_BOOL(FALSE);
                return FALSE;
        }
}



/*
 *  Called by segment discovery code.
 *
 *  After learning that there is a gpt partitioning scheme on the specified
 *  object, we now need to walk the partition tables and create DISKSEG
 *  structs for each partition record that we find.
 *
 */
static int get_segments(  storage_object_t *object )
{
        gpt_header                 *gh=NULL;
        gpt_header                 *gh1=NULL;
        gpt_header                 *gh2=NULL;
        storage_object_t           *seg;
        storage_object_t           *metadata1;
        storage_object_t           *metadata2;
        int                         i;
        int                         rc=0;
        gpt_partition              *part;
        gpt_partition              *ptable;
        char                       *cptr=NULL;
        DISK_PRIVATE_DATA          *disk_pdata = get_gpt_disk_private_data(object);
        boolean                     md1_dirty=FALSE;
        boolean                     md2_dirty=FALSE;
        char                       *uuid_string;
        sector_count_t              md_sector_count=0;


        LOG_ENTRY();


        // Get 1st copy of gpt header ... always at LBA 1
        gh1 = get_gpt_header(object, 1 );
        disk_gpt_header_to_cpu( gh1 );

        // Get 2nd copy of gpt header ... 1st copy of gpt header
        // points to 2nd copy.  If the 1st copy is bad ... then
        // the 2nd copy will be found in the last sector on the disk.
        if (gh1) {
                gh2 = get_gpt_header(object, gh1->alternate_lba );
                disk_gpt_header_to_cpu( gh2 );
        }
        else {
                gh2 = get_gpt_header(object, object->size - 1 ); //(object->start+object->size-1) );
                disk_gpt_header_to_cpu( gh2 );
        }

        // Check that we got at least 1 copy of the header
        if (gh1==NULL && gh2==NULL) {
                rc = ENODATA;
                LOG_DEBUG("error, both copies of the gpt header are missing.\n");
                LOG_EXIT_INT(rc);
                return rc;
        }

        // Make sure that we have both copies of the GPT header by reproducing
        // a missing header from information obtained from the only good gpt header.
        if (gh1==NULL) {
                gh1 = (gpt_header *)calloc(1,EVMS_VSECTOR_SIZE);
                if (gh1) {
                        memcpy(gh1, gh2, EVMS_VSECTOR_SIZE);
                        gh1->my_lba        = 1;
                        gh1->alternate_lba = gh2->my_lba;
                        gh1->ptable_lba    = 2;

                        md1_dirty = TRUE;   // mark 1st copy of metadata dirty so the new
                                            // gpt header will be committed.

                        gh = gh2;           // use gh2 for rest of discovery

                        rc = 0;
                }
                else {
                        rc = ENOMEM;
                        LOG_ERROR("error, unable to malloc a gpt header\n");
                }

        }
        else if (gh2==NULL) {
                gh2 = (gpt_header *)calloc(1,EVMS_VSECTOR_SIZE);
                if (gh2) {
                        memcpy(gh2, gh1, EVMS_VSECTOR_SIZE);
                        gh2->my_lba        = object->size-1;  //object->start+object->size-1;
                        gh2->alternate_lba = gh1->my_lba;
                        gh2->ptable_lba    = gh1->end_useable + 1;

                        md2_dirty = TRUE;   // mark 2nd copy of metadata dirty so the new
                        // gpt header will be committed.

                        gh = gh1;           // use gh1 for rest of discovery

                        rc = 0;
                }
                else {
                        rc = ENOMEM;
                        LOG_ERROR("error, unable to malloc a gpt header\n");
                }

        }
        else {
                gh = gh1; // both copies are good but use the primary copy
                rc = 0;
        }

        // Exit if we were unable to produce both copies of the GPT
        // header needed by the GPT segment manager.
        if (rc) {
                if (gh1) free(gh1);
                if (gh2) free(gh2);
                LOG_EXIT_INT(rc);
                return rc;
        }

        Display_GPT_Header( gh );

        // Register disk guid identifier - convert uuid to ascii string and register it.
        uuid_string = guid_to_string( &gh->disk_id );
        if (uuid_string) {
                rc = EngFncs->register_name( uuid_string );
                free(uuid_string);
        }
        else {
                rc = ENOMEM;
        }

        if (rc) {
                free(gh1);
                free(gh2);
                LOG_ERROR("error, unable to convert DISK uuid identifier to ascii string for registering.\n");
                LOG_EXIT_INT(rc);
                return rc;
        }

        // create PMBR
        rc = create_protective_mbr_segment( object );
        if (rc) {
                free(gh1);
                free(gh2);
                LOG_EXIT_INT(rc);
                return rc;
        }

        // Need to calculate the size of the metadata segments.  Metadata segments
        // are EQUAL in size. They each have a copy of the GPT header and a copy of
        // the partition table.

        md_sector_count = (gh->ptable_count * gh->ptable_entry_size) / EVMS_VSECTOR_SIZE;

        if ( ((gh->ptable_count * gh->ptable_entry_size) % EVMS_VSECTOR_SIZE) != 0) {
                ++md_sector_count;
        }

        ++md_sector_count;   // add in 1 sector for the GPT header

        // create 1st gpt metadata segment
        metadata1 = create_gpt_metadata_segment( object,
                                                 1,
                                                 md_sector_count,
                                                 "metadata1" );
        if (metadata1 == NULL) {
                free(gh1);
                free(gh2);
                LOG_EXIT_INT(ENOMEM);
                return ENOMEM;
        }

        if ( insert_gpt_segment_into_list(object->parent_objects, metadata1) == NULL ) {
                free(gh1);
                free(gh2);
                free_gpt_disk_segment(metadata1);
                LOG_EXIT_INT(ENOMEM);
                return ENOMEM;
        }

        // if primary gpt header is missing then mark it dirty so engine will commit it.
        if (md1_dirty == TRUE) {
                MESSAGE("\nPrimary GPT Header is missing or corrupt. Marking %s dirty to correct the problem\n",
                        metadata1->name );
                metadata1->flags |= SOFLAG_DIRTY;
        }

        // create 2nd gpt metadata segment
        metadata2 = create_gpt_metadata_segment( object,
                                                 gh->end_useable+1,
                                                 md_sector_count,
                                                 "metadata2" );
        if (metadata2 == NULL) {
                free(gh1);
                free(gh2);
                LOG_EXIT_INT(ENOMEM);
                return ENOMEM;
        }

        if ( insert_gpt_segment_into_list(object->parent_objects, metadata2) == NULL ) {
                free(gh1);
                free(gh2);
                free_gpt_disk_segment(metadata2);
                LOG_EXIT_INT(ENOMEM);
                return ENOMEM;
        }

        // if secondary gpt header is missing then mark it dirty so engine will commit it.
        if (md2_dirty == TRUE) {
                MESSAGE("\nAlternate GPT Header is missing or corrupt. Marking %s dirty to correct the problem\n",
                        metadata2->name );
                metadata2->flags |= SOFLAG_DIRTY;
        }

        // read gpt partition table into memory
        ptable = get_gpt_partition_table( object, gh);
        if (ptable == NULL) {
                free(gh1);
                free(gh2);
                LOG_EXIT_INT(ENODATA);
                return ENODATA;
        }

        // Save gpt headers in their respective metadata segments
        ((SEG_PRIVATE_DATA *)metadata1->private_data)->gh = gh1;
        ((SEG_PRIVATE_DATA *)metadata2->private_data)->gh = gh2;

        // save metadata segments in logical disk private data for quick searching
        disk_pdata->md1 = metadata1;
        disk_pdata->md2 = metadata2;

        // create a segment storage object for each gpt partition ...
        for (i=1, cptr = (char *)ptable; i<=gh->ptable_count; i++) {

                part = (gpt_partition *) cptr;

                if ( isa_unused_gpt_partition_record( part ) == FALSE ) {

                        Display_GPT_Partition_Record(part);

                        seg = build_gpt_segment_from_partition_record(object, part, i );
                        if (seg) {

                                // gpt header in every seg we create
                                ((SEG_PRIVATE_DATA *)seg->private_data)->gh = gh;

                                // insert into segment list
                                if ( insert_gpt_segment_into_list(object->parent_objects, seg) == NULL ) {
                                        free_gpt_disk_segment(seg);
                                        rc = ENOMEM;
                                        break;
                                }

                        }
                }

                cptr += gh->ptable_entry_size;
        }

        // lastly reveal any freespace on the disk
        if (rc==0) {
                find_freespace_on_gpt_disk( object );
        }
        else {
                free(gh1);
                free(gh2);
                disk_pdata->md1 = NULL;
                disk_pdata->md2 = NULL;
        }

        if (ptable) free(ptable);
        LOG_EXIT_INT(rc);
        return rc;
}

/*
 *  Called when we are aborting discovery.  Called with a list
 *  of segment objects we have created, the purpose of this routine
 *  is simply to provide a convenient means of discarding the work
 *  done prior to aborting discovery.  By returning TRUE for any
 *  object we created, the object will be pruned from the list.
 */
static void prune_gpt_segments_from_list( list_anchor_t list )
{
        list_element_t iter,iter2;
        storage_object_t *seg;
        boolean prune;

        LIST_FOR_EACH_SAFE( list, iter, iter2, seg ) {

                prune = FALSE;

                if ( seg->plugin == gpt_plugin_record_ptr ) {
                        free_gpt_disk_segment(seg);
                        prune = TRUE;               
                }

                if (prune==TRUE) EngFncs->delete_element(iter);
        }

}


/*
 *  Function: gpt_get_devmap_info
 *
 *  Called to test if the segment has an active device mapper
 *  node in the kernel and set the object info accordingly.
 */
static int gpt_get_devmap_info( DISKSEG *seg )
{
        int rc;
        dm_target_t *targets=NULL;
        dm_device_t *dev = NULL;


        LOG_ENTRY();

        if (seg->data_type == DATA_TYPE) {

                rc = EngFncs->dm_update_status(seg);
                if (!rc) {

                        if (seg->flags & SOFLAG_ACTIVE) {

                                rc = EngFncs->dm_get_targets(seg, &targets);
                                if (  (!rc) &&                   // RC==Zero
                                      (targets) &&               // got a target list
                                      (targets->next == NULL) && // with 1 target only
                                      (targets->data.linear) &&  // with device data
                                      (targets->start == 0)) {   // and target starts at Zero

                                        dev = (dm_device_t *) targets->data.linear;

                                        if ( ( seg->start != dev->start )||
                                             ( seg->size  != targets->length )) {
                                                LOG_DEBUG("this segment is being marked needs_activate\n");
                                                seg->flags |= SOFLAG_NEEDS_ACTIVATE;
                                        }
                                        else {
                                                seg->flags &= ~SOFLAG_NEEDS_ACTIVATE;
                                        }

                                }

                                if (targets) EngFncs->dm_deallocate_targets(targets);
                        }
                        else {
                                seg->flags |= SOFLAG_NEEDS_ACTIVATE;
                        }

                }

        }

        LOG_EXIT_INT(0);
        return 0;
}



/*
 *   Called by GPT_Discover() ... once for each item in the input_object list
 *
 *   Probes the storage object, looking for a valid GPT header.
 *
 */
int gpt_segment_discovery( storage_object_t *obj, list_anchor_t  output_objects, uint *count )
{
        boolean                     success=FALSE;
        int                         rc;
        uint                        segcount;
        SEG_PRIVATE_DATA           *pdata=NULL;
        DISK_PRIVATE_DATA          *disk_pdata=NULL;
        list_element_t              iter;
        storage_object_t           *seg;

        LOG_ENTRY();
        LOG_DEBUG("examining object %s\n", obj->name );

        pdata            = (SEG_PRIVATE_DATA *) obj->private_data;

        // test if the object is acceptable
        if ( ( obj->plugin == gpt_plugin_record_ptr ) ||
             ( obj->data_type != DATA_TYPE ) ||
             ( obj->object_type == SEGMENT && (pdata->cflags&SEG_CFLAG_TOP_SEGMENT) )) {

                if (EngFncs->insert_thing(output_objects,obj,INSERT_AFTER,NULL)) {
                        rc = 0;
                }
                else {
                        rc = EPERM;
                }

                LOG_DEBUG("object is not acceptable\n" );
                LOG_EXIT_INT(0);
                return 0;
        }


        if ( isa_object_with_valid_gpt_header( obj ) == TRUE) {

                // create disk private data
                create_gpt_disk_private_data( obj );
                disk_pdata = get_gpt_disk_private_data(obj);

                if (disk_pdata) {

                        rc = get_segments(obj);

                        if (rc == 0) {

                                segcount = EngFncs->list_count( obj->parent_objects );

                                LOG_DEBUG( "Discovery, Success ...I created %d segments\n", segcount );

                                if (segcount > 0) {

                                        DisplayDiskSegmentList(obj);

                                        LIST_FOR_EACH(obj->parent_objects, iter, seg ) {
                                                gpt_get_devmap_info(seg);
                                        }

                                        rc = EngFncs->concatenate_lists( output_objects, obj->parent_objects );
                                }

                                // mark segment discovery as successful
                                if (rc==0) {

                                        // return number of segments found
                                        *count += segcount;

                                        success = TRUE;
                                }

                        }
                }

        }

        if (success == FALSE) {

                LOG_DEBUG("Discovery, Failure ... not my disk or no partitions\n");

                // remove any storage objects we may have created
                prune_gpt_segments_from_list(obj->parent_objects);

                if (EngFncs->insert_thing(output_objects,obj,INSERT_AFTER,NULL)) {
                        rc = 0;
                }
                else {
                        rc = EPERM;
                }

                // free disk private data if we allocated it
                if (disk_pdata) delete_gpt_disk_private_data(obj);

        }


        LOG_EXIT_INT(0);
        return 0;
}

