/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libdos.so
 *
 *   File: discovery.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "ptables.h"
#include "defsegmgr.h"
#include "os2dlat.h"
#include "segs.h"
#include "checks.h"
#include "commit.h"
#include "display.h"



static void DisplayDlatEntry(DLA_Entry *dla, int index)
{

	char        DriveLetter[8];
	char        VolName[EVMS_VOLUME_NAME_SIZE+1];
	char        PartName[EVMS_NAME_SIZE+1];


	if ((dla->Partition_Size==0)&&(dla->Partition_Start==0)) {
		LOG_DEBUG("Entry[%d]- unused ...\n", index);
	}
	else {

		memset(VolName, 0, EVMS_VOLUME_NAME_SIZE+1);
		memset(PartName, 0, EVMS_NAME_SIZE+1);

		DriveLetter[0] = dla->Drive_Letter;
		DriveLetter[1] = '\0';
		if (strlen(DriveLetter)==0) {
			strcpy(DriveLetter, "n/a");
		}

		strncpy(VolName, dla->Volume_Name, EVMS_VOLUME_NAME_SIZE);
		if (strlen(VolName)==0) {
			strcpy(VolName, "n/a");
		}

		strncpy(PartName, dla->Partition_Name, EVMS_NAME_SIZE);
		if (strlen(PartName)==0) {
			strcpy(PartName, "n/a");
		}

		LOG_DEBUG("Entry[%d]- Pname(%s)  Start(%08d)  Size(%08d)  Psn(%X)  Vname(%s) Vsn(%X) Drive(%s)\n",
			  index,
			  PartName,
			  dla->Partition_Start,
			  dla->Partition_Size,
			  dla->Partition_Serial_Number,
			  VolName,
			  dla->Volume_Serial_Number,
			  DriveLetter );

	}

}

static void DisplayDlatTable( DLA_Table_Sector *dlat , lba_t  lba)
{
	int i;

	LOG_DEBUG("\t\tOS/2 Drive Letter Assignment Table ... LBA= %"PRIu64"\n", lba);
	LOG_DEBUG("Disk Name    %s\n", dlat->Disk_Name);
	LOG_DEBUG("Disk SN      0x%x\n", dlat->Disk_Serial_Number );
	LOG_DEBUG("BootDisk SN  0x%x\n", dlat->Boot_Disk_Serial_Number);
	LOG_DEBUG("Geometry     C(%d)  H(%d)  S(%d)\n", dlat->Cylinders, dlat->Heads_Per_Cylinder, dlat->Sectors_Per_Track);
	LOG_DEBUG("Flags        0x%x\n", dlat->Install_Flags);
	for (i=0; i<4; i++) {
		DisplayDlatEntry(&dlat->DLA_Array[i], i);
	}

}

/*
 *  Called when we are creating a logical drive or being assigned to
 *  a disk ... because these events require that we allocate a DLAT
 *  and initialize it.
 */
DLA_Table_Sector *   Allocate_Dlat( LOGICALDISK *ld )
{
	DLA_Table_Sector   *dlat_buffer = (DLA_Table_Sector *) calloc( 1, EVMS_VSECTOR_SIZE );

	LOG_ENTRY();

	if ( dlat_buffer  ) {

		strncpy( dlat_buffer->Disk_Name, ld->name, DISK_NAME_SIZE );

		dlat_buffer->DLA_Signature1     = DLA_TABLE_SIGNATURE1;
		dlat_buffer->DLA_Signature2     = DLA_TABLE_SIGNATURE2;
		dlat_buffer->Cylinders          = ld->geometry.cylinders;
		dlat_buffer->Heads_Per_Cylinder = ld->geometry.heads;
		dlat_buffer->Sectors_Per_Track  = ld->geometry.sectors_per_track;

	}
	else {
		LOG_ERROR("unable to malloc a DLAT buffer\n");
	}

	LOG_EXIT_PTR(dlat_buffer);
	return dlat_buffer;
}


/*
 *  Called to find the DLAT entry that matches the DISKSEG. The EBR or MBR meta data
 *  segment will have a ptr to the DLA table sector. All we need to do is to look at
 *  the DLA entries found in the DLA table ... find the matching entry ... and return it.
 */
DLA_Entry *  Get_Dlat_Entry_Matching_DiskSegment( DISKSEG *ebr, DISKSEG *seg )
{
	DLA_Entry         *dla;
	SEG_PRIVATE_DATA  *ebr_pdata;
	SEG_PRIVATE_DATA  *seg_pdata;
	int                i;
	LOGICALDISK       *ld;

	ld = get_logical_disk(seg);

	LOG_ENTRY();

	if (ebr) {

		ebr_pdata        = (SEG_PRIVATE_DATA *)ebr->private_data;
		seg_pdata        = (SEG_PRIVATE_DATA *)seg->private_data;

		/* all DISKSEG structs have a ptr set to dla table the partition is found in */
		seg_pdata->dlat  = ebr_pdata->dlat;

		/*
		 * find the DLAT entry corresponding to this data segment
		 */
		if ( ( seg->data_type != FREE_SPACE_TYPE ) &&
		     ( (seg_pdata->flags & SEG_IS_MBR) == 0) ) {

			for (i=0; i<4; i++) {

				dla = &ebr_pdata->dlat->DLA_Array[i];

				if ( ( dla->Partition_Size  == seg->size ) &&
				     ( dla->Partition_Start == seg->start) ) {
					LOG_EXIT_PTR(dla);
					return dla;
				}
			}

		}
	}

	LOG_EXIT_PTR(NULL);
	return NULL;
}


/*
 * Called to convert a DLAT from disk format to CPU format
 */
static void Disk_Dlat_To_CPU_Dlat( DLA_Table_Sector *dlat )
{
	int i;

	dlat->DLA_Signature1          = DISK_TO_CPU32(dlat->DLA_Signature1);
	dlat->DLA_Signature2          = DISK_TO_CPU32(dlat->DLA_Signature2);
	dlat->DLA_CRC                 = DISK_TO_CPU32(dlat->DLA_CRC);
	dlat->Disk_Serial_Number      = DISK_TO_CPU32(dlat->Disk_Serial_Number);
	dlat->Boot_Disk_Serial_Number = DISK_TO_CPU32(dlat->Boot_Disk_Serial_Number);
	dlat->Install_Flags           = DISK_TO_CPU32(dlat->Install_Flags);
	dlat->Cylinders               = DISK_TO_CPU32(dlat->Cylinders);
	dlat->Heads_Per_Cylinder      = DISK_TO_CPU32(dlat->Heads_Per_Cylinder);
	dlat->Sectors_Per_Track       = DISK_TO_CPU32(dlat->Sectors_Per_Track);

	for (i=0; i<4; i++) {
		dlat->DLA_Array[i].Volume_Serial_Number    = DISK_TO_CPU32(dlat->DLA_Array[i].Volume_Serial_Number);
		dlat->DLA_Array[i].Partition_Serial_Number = DISK_TO_CPU32(dlat->DLA_Array[i].Partition_Serial_Number);
		dlat->DLA_Array[i].Partition_Size          = DISK_TO_CPU32(dlat->DLA_Array[i].Partition_Size);
		dlat->DLA_Array[i].Partition_Start         = DISK_TO_CPU32(dlat->DLA_Array[i].Partition_Start);
	}

}

/*
 * Called to convert a DLAT from CPU format to DISK format
 */
static void CPU_Dlat_To_Disk_Dlat( DLA_Table_Sector *dlat )
{
	int i;

	dlat->DLA_Signature1          = CPU_TO_DISK32(dlat->DLA_Signature1);
	dlat->DLA_Signature2          = CPU_TO_DISK32(dlat->DLA_Signature2);
	dlat->DLA_CRC                 = CPU_TO_DISK32(dlat->DLA_CRC);
	dlat->Disk_Serial_Number      = CPU_TO_DISK32(dlat->Disk_Serial_Number);
	dlat->Boot_Disk_Serial_Number = CPU_TO_DISK32(dlat->Boot_Disk_Serial_Number);
	dlat->Install_Flags           = CPU_TO_DISK32(dlat->Install_Flags);
	dlat->Cylinders               = CPU_TO_DISK32(dlat->Cylinders);
	dlat->Heads_Per_Cylinder      = CPU_TO_DISK32(dlat->Heads_Per_Cylinder);
	dlat->Sectors_Per_Track       = CPU_TO_DISK32(dlat->Sectors_Per_Track);

	for (i=0; i<4; i++) {
		dlat->DLA_Array[i].Volume_Serial_Number    = CPU_TO_DISK32(dlat->DLA_Array[i].Volume_Serial_Number);
		dlat->DLA_Array[i].Partition_Serial_Number = CPU_TO_DISK32(dlat->DLA_Array[i].Partition_Serial_Number);
		dlat->DLA_Array[i].Partition_Size          = CPU_TO_DISK32(dlat->DLA_Array[i].Partition_Size);
		dlat->DLA_Array[i].Partition_Start         = CPU_TO_DISK32(dlat->DLA_Array[i].Partition_Start);
	}

}


/*
 *  Partition records must be recorded in the DLAT. This routine
 *  is called with a segment object so that the partition info
 *  can be saved in a dla entry where OS/2 expects to find it.
 */
int Add_DiskSegment_To_Dlat( LOGICALDISK *ld, DLA_Table_Sector *dlat, DISKSEG *seg )
{
	int                rc=EINVAL;
	int                i;
	SEG_PRIVATE_DATA  *pdata = (SEG_PRIVATE_DATA *)seg->private_data;
	DLA_Entry         *dla = NULL;
	DLA_Entry         *dla_entry;


	LOG_ENTRY();

	if (seg==NULL || dlat==NULL) {
		rc=EINVAL;
	}
	else { // find an empty dla entry in the dla array

		for (i=0; i<4; i++) {

			dla_entry = &dlat->DLA_Array[i];

			if ( (dla_entry->Partition_Size == 0) &&
			     (dla_entry->Partition_Start == 0)) {
				dla = dla_entry;
				break;
			}
		}
	}

	// copy evms segment info to the dla entry
	if ( dla ) {

		memcpy(dla, pdata->dla_entry, sizeof(DLA_Entry));

		rc = 0;
	}
	else {
		LOG_ERROR("no empty dla entries in the DLA Table\n");
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Called to read a DLAT sector from disk. This routine is responsible
 *  for converting the DLAT to CPU format, verifying the DLAT signature
 *  and doing the I/O.
 */
DLA_Table_Sector * Read_Dlat_Sector( LOGICALDISK *ld,  lba_t ebr_lba  )
{
	int                rc;
	lba_t              lba;
	DLA_Table_Sector  *buffer = NULL;
	struct plugin_functions_s *dft = (struct plugin_functions_s *)ld->plugin->functions.plugin;
	DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);

	LOG_ENTRY();

	if (disk_pdata) {

		// malloc buffer so we can read in the DLAT sector
		buffer = (DLA_Table_Sector *) malloc( EVMS_VSECTOR_SIZE );
		if (buffer) {

			// calc the LBA of the DLA table ... always last sector of track
			lba =  (lba_t) ebr_lba + disk_pdata->geometry.sectors_per_track - 1;

			rc = dft->read( ld, lba, 1, buffer);
			if (rc == 0) {

				/* verify the DLAT sector */
				if ( ( DISK_TO_CPU32(buffer->DLA_Signature1) == DLA_TABLE_SIGNATURE1 ) &&
				     ( DISK_TO_CPU32(buffer->DLA_Signature2) == DLA_TABLE_SIGNATURE2 ) ) {

					// Ok ... so convert the DLAT to CPU format
					Disk_Dlat_To_CPU_Dlat(buffer);

				}
				else {

					free(buffer);
					buffer = NULL;

				}

			}

		}

	}

	LOG_EXIT_PTR(buffer);
	return buffer;
}


/*
 *  Called to write a DLAT sector to disk. This routine is responsible
 *  for converting the DLAT to DISK format, updating the CRC and doing
 *  the I/O.
 */
int Write_Dlat_Sector( LOGICALDISK *ld, DISKSEG *boot_seg )
{
	DLA_Table_Sector           *dlat_buffer=NULL;
	u_int32_t                   Calculated_CRC;
	lba_t                       dlat_lba;
	struct plugin_functions_s  *Fncs;
	int                         rc = ENODEV;
	DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);


	LOG_ENTRY();

	if (disk_pdata) {

		dlat_buffer = ((SEG_PRIVATE_DATA *) boot_seg->private_data)->dlat;
		if ( dlat_buffer ) {

			if ( ( dlat_buffer->DLA_Signature1 == DLA_TABLE_SIGNATURE1 ) &&
			     ( dlat_buffer->DLA_Signature2 == DLA_TABLE_SIGNATURE2 ) ) {

				CPU_Dlat_To_Disk_Dlat( dlat_buffer );

				dlat_buffer->DLA_CRC = 0;
				Calculated_CRC       = EngFncs->calculate_CRC( EVMS_INITIAL_CRC, (void *)dlat_buffer, EVMS_VSECTOR_SIZE);
				dlat_buffer->DLA_CRC = CPU_TO_DISK32(Calculated_CRC);

				dlat_lba = (lba_t) (disk_pdata->geometry.sectors_per_track - 1) + boot_seg->start;

				Fncs = (struct plugin_functions_s *)ld->plugin->functions.plugin;
				rc = Fncs->write( ld, dlat_lba, 1,(void *) dlat_buffer);

				rc = 0;
			}
			else {
				rc = EINVAL;
			}

		}

	}

	LOG_EXIT_INT(rc);
	return rc;
}




/*
 *  Called to determine if the DLA entries map to the entries in the
 *  partition table.
 *
 *  Some rules ...
 *
 *  (1) DLA entries and partition table entries may be out of order and so
 *      you need to search the entire table every time.
 *  (2) An extended partition record (from the mbr table) will not appear in the
 *      corresponding DLAT table.  The DLAT table will have a NULL entry for
 *      this ebr partition record.
 *  (3) Partition records, found in the ebr table, will have dlat entries in the
 *      corresponding DLAT table. However, the starting LBA of the partition will
 *      differ between the DLAT entry and the EBR entry.  The DLAT entry will have
 *      the starting LBA of the extended partition added to the starting LBA of
 *      the partition.
 *
 *  Returns:  TRUE if there is a mapping between the DLAT and PTABLE
 */
static boolean dla_matches_ptable( Master_Boot_Record *mbr,
				   DLA_Table_Sector   *DLA_Table,
				   lba_t               mbr_lba )
{
	int                 i;
	int                 j;
	lba_t               PartStartLba;
	u_int32_t           PartSize;
	boolean             partition_mapped[4] = {FALSE, FALSE, FALSE, FALSE};
	boolean             dla_maps_a_partition;
	DLA_Entry          *dla;
	Partition_Record   *part;


	LOG_ENTRY();
/*
printf("     validating that the dla entries match the partition table, dlat follows \n");
for (i=0; i<4; i++) {
    dla = &DLA_Table->DLA_Array[i];
    OS2_DisplayDlatEntry(dla, i );
}
*/

	for (i=0; i<4; i++) {

		dla = &DLA_Table->DLA_Array[i];

		for (j=0, dla_maps_a_partition=FALSE; (j<4 && dla_maps_a_partition==FALSE); j++) {

			part = &mbr->Partition_Table[j];

			if ( partition_mapped[j] == FALSE ) {

				if ( ( isa_null_partition_record(part)==TRUE ) ||
				     ( isa_ebr_partition_record(part)==TRUE ) ) {
					PartStartLba = 0;
					PartSize     = 0;
				}
				else {
					PartStartLba = DISK_TO_CPU32(START_LBA(part)) + mbr_lba;
					PartSize     = DISK_TO_CPU32(NR_SECTS(part));
				}

/*
printf("     DLA->Start  = %d\n", (u_int32_t) dla->Partition_Start);
printf("     DLA->Size   = %d\n", (u_int32_t) dla->Partition_Size);
printf("    Part->Start  = %d\n", (u_int32_t) PartStartLba );
printf("    Part->Size   = %d\n", (u_int32_t) PartSize );
*/

				if ( (dla->Partition_Start == PartStartLba ) &&
				     (dla->Partition_Size  == PartSize ) ) {

					dla_maps_a_partition  = TRUE;
					partition_mapped[j]   = TRUE;
				}

			} // end if partition mapped

		} // end looping through ptable


		if ( dla_maps_a_partition == FALSE ) {

/*
printf("     the following dla entry does not match any partition\n");
OS2_DisplayDlatEntry(dla, i );
*/
			LOG_EXIT_BOOL(FALSE);
			return FALSE;
		}

	} // end looking at all dla entries

	LOG_EXIT_BOOL(TRUE);
	return TRUE;
}



/*
 *  It is a valid OS2 DLAT if ...
 *
 *  (1) has dlat signature
 *  (2) crc matches crc stored in dlat
 *  (3) each dlat entry matches an entry in the partition table
 *
 *  Notes ...
 *
 *  Make sure every dla table entry maps to a unique partition table entry.  This
 *  makes sure we dont map multiple dla entries to the same partition table entry.
 *  The partition_mapped[] array keeps track of partition table entries we have
 *  already associated with a dla table entry.
 *
 *  The dla table and partition table may have a different number of entries. So,
 *  we need to see if the partition table maps to the dla table ... AND ... that
 *  the dla table maps back to the partition table.  This will account for differnt
 *  table entrie and also different number of entries.
 */
static boolean isa_valid_OS2_DLAT( LOGICALDISK        *ld,
				   Master_Boot_Record *mbr,
				   DLA_Table_Sector   *DLA_Table,
				   lba_t               mbr_lba,
				   lba_t               extd_partition_lba )
{
	u_int32_t          Actual_CRC;
	u_int32_t          Calculated_CRC;

	LOG_ENTRY();
	LOG_DEBUG("validating DLAT for mbr/ebr at addr %d\n", (u_int32_t) mbr_lba );

	if ( ( DLA_Table->DLA_Signature1 == DLA_TABLE_SIGNATURE1 ) &&
	     ( DLA_Table->DLA_Signature2 == DLA_TABLE_SIGNATURE2 ) ) {

		Actual_CRC         = DLA_Table->DLA_CRC;
		DLA_Table->DLA_CRC = 0;

		Calculated_CRC = EngFncs->calculate_CRC( EVMS_INITIAL_CRC, DLA_Table, ld->geometry.bytes_per_sector );

		if ( Calculated_CRC == Actual_CRC ) {

			if ( dla_matches_ptable( mbr, DLA_Table, mbr_lba ) == TRUE ) {

				DisplayDlatTable( DLA_Table, (mbr_lba+ld->geometry.sectors_per_track-1));
				LOG_EXIT_BOOL(TRUE);
				return TRUE;

			}
			else {
				LOG_DEBUG("bad dla mapping to partition table\n");
				LOG_EXIT_BOOL(FALSE);
				return FALSE;
			}
		}
		else {
			LOG_DEBUG("bad CRC ... Calculated CRC= %d   Actual CRC= %d\n", Calculated_CRC, Actual_CRC );
			LOG_EXIT_BOOL(FALSE);
			return FALSE;
		}
	}
	else {
		LOG_DEBUG("bad DLA signature\n");
		LOG_EXIT_BOOL(FALSE);
		return FALSE;
	}
}



/*
 *  Rules ...
 *
 *  (1) There will be an MSDOS signature in the usual place ... the last couple of
 *      bytes in the MBR sector.
 *  (2) The MBR track will contain a dlat table in the last sector of the track.
 *  (3) The EBR track will also contain a dlat table in the last sector of the track.
 *  (4) Dlat table entries map disk partitions but the ordering of the dlat entries
 *      may differ from the ordering of the entries in the corresponding partition table.
 *
 */
boolean isa_os2_partitioned_disk( LOGICALDISK        *ld,
				  Master_Boot_Record *mbr,
				  lba_t               mbr_lba,
				  lba_t               extd_partition_lba )
{
	int                         i;
	lba_t                       dlat_lba;
	lba_t                       ebr_lba;
	Extended_Boot_Record       *ebr = NULL;
	DLA_Table_Sector           *dlat_buffer = NULL;
	boolean                     rc;
	struct plugin_functions_s  *dft;
	DISK_PRIVATE_DATA          *disk_pdata = get_disk_private_data(ld);

	LOG_ENTRY();
	LOG_DEBUG("disk= %s\n", ld->name );

	if ( ( disk_pdata ) &&
	     ( has_msdos_signature( mbr )==TRUE )) {

		dlat_buffer = (DLA_Table_Sector *) malloc(ld->geometry.bytes_per_sector);
		if (dlat_buffer==NULL) return FALSE;

		dlat_lba = (lba_t) mbr_lba + disk_pdata->geometry.sectors_per_track - 1;

		dft = (struct plugin_functions_s *)ld->plugin->functions.plugin;
		if (dft==NULL) {
			LOG_ERROR("error, no disk plugin function table\n");
			LOG_EXIT_BOOL(FALSE);
			free(dlat_buffer);
			return FALSE;
		}

		if ( dft->read( ld, dlat_lba, 1, dlat_buffer) ) {
			LOG_ERROR("error, i/o error reading DLA Table off disk at LBA= %"PRIu64"\n", dlat_lba);
			LOG_EXIT_BOOL(FALSE);
			free(dlat_buffer);
			return FALSE;
		}
		else {
			Disk_Dlat_To_CPU_Dlat( dlat_buffer );
		}

		if (isa_valid_OS2_DLAT( ld, mbr, dlat_buffer, mbr_lba, extd_partition_lba )==FALSE) {
			LOG_EXIT_BOOL(FALSE);
			free(dlat_buffer);
			return FALSE;
		}

		for (i=0; i<4; i++) {

			if (isa_ebr_partition_record(&mbr->Partition_Table[i])==TRUE) {

				ebr_lba = START_LBA( (&mbr->Partition_Table[i]) ) + extd_partition_lba;

				ebr = (Extended_Boot_Record *) malloc(ld->geometry.bytes_per_sector);
				if (ebr==NULL) {
					LOG_ERROR("error, failed to malloc boot sector buffer\n");
					LOG_EXIT_BOOL(FALSE);
					free(dlat_buffer);
					return FALSE;
				}

				if ( dft->read( ld, ebr_lba, 1, (void *) ebr )) {
					LOG_ERROR("error, i/o error reading boot sector off disk at LBA= %"PRIu64"\n", ebr_lba);
					LOG_EXIT_BOOL(FALSE);
					free(ebr);
					free(dlat_buffer);
					return FALSE;
				}

				/* did we find the start of the extended partition yet ? */
				if (extd_partition_lba == 0) {
					rc = isa_os2_partitioned_disk( ld, (Master_Boot_Record *) ebr, ebr_lba, ebr_lba );
				}
				else {
					rc = isa_os2_partitioned_disk( ld, (Master_Boot_Record *) ebr, ebr_lba, extd_partition_lba );
				}

				LOG_EXIT_BOOL(rc);
				return rc;
			}

		}

		LOG_EXIT_BOOL(TRUE);
		return TRUE;
	}
	else {	    // bad msdos signature
		LOG_EXIT_BOOL(FALSE);
		return FALSE;
	}

}
