/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libcsm.so
 *
 *   File: helpers.c
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <uuid/uuid.h>

#include <plugin.h>

#include "csm_plugin.h"



storage_object_t *  allocate_csm_segment( storage_object_t *object )
{
        int   rc;
        DISKSEG  *seg=NULL;
        seg_private_data_t  *pdata=NULL;
        list_element_t e;

        LOG_ENTRY();

        rc = EngFncs->allocate_segment( NULL, &seg );
        if (rc) {
                LOG_EXIT_PTR(NULL);
                return NULL;
        }

        e=EngFncs->insert_thing( seg->child_objects,
                                 object,
                                 INSERT_AFTER,
                                 NULL );
        if (e!=NULL) {
                rc = 0;
        }
        else {
                rc = ENOMEM;
        }

        if (rc == 0) {

                seg->plugin      = csm_plugin_record_ptr;
                seg->object_type = SEGMENT;

                memcpy(&seg->geometry, &object->geometry, sizeof(geometry_t) );

                seg->private_data = calloc(1, sizeof(seg_private_data_t) );
                if (seg->private_data) {

                        pdata = (seg_private_data_t *)seg->private_data;

                        pdata->signature    = CSM_SEGMENT_PDATA_SIGNATURE ;
                        pdata->logical_disk = object;
                }
                else {
                        LOG_ERROR("call to malloc segment private storage area failed\n");
                        EngFncs->free_segment( seg );
                        seg = NULL;
                }
        }
        else {
                LOG_ERROR("call to insert DISK storage object in segment child_objects list failed, RC= %d\n", rc );
                EngFncs->free_segment( seg );
                seg = NULL;
        }

        LOG_EXIT_PTR(seg);
        return seg;
}


void free_csm_segment( DISKSEG *seg )
{
        LOG_ENTRY();
        LOG_DEBUG("segment name= %s\n", seg->name );

        if (seg->private_data) free(seg->private_data);

        EngFncs->free_segment( seg );

        LOG_EXIT_VOID();
}


DISKSEG * create_csm_metadata_segment( LOGICALDISK         *ld,
                                       storage_container_t *container,
                                       lba_t                start,
                                       sector_count_t       size,
                                       char                *name,
                                       u_int32_t            object_flags )
{
        DISKSEG *metadata;
        seg_private_data_t  *pdata=NULL;
//      char devname[EVMS_VOLUME_NAME_SIZE+1];

        LOG_ENTRY();

        metadata = allocate_csm_segment( ld );
        if (metadata) {

                metadata->size        = size;
                metadata->start       = start;
                metadata->data_type   = META_DATA_TYPE;
                metadata->flags       = object_flags;

                pdata = (seg_private_data_t *)metadata->private_data;
                if (start==0) {
                        pdata->commit_phase = 1;   // 1st copy of metadata committed at phase 1
                }
                else {
                        pdata->commit_phase = 2;   // 2nd copy of metadata committed at phase 2
                }

                set_segment_storage_type(metadata,container);

                metadata->disk_group = container;
/*
                get_device_name(ld, devname);

                if ( ld->object_type != DISK ) {
                        sprintf(metadata->name, "%s.ece_%s", devname, name );
                }
                else {
                        if (devname[strlen(devname)-1] == '/') {
                                sprintf(metadata->name, "%sece_%s", devname, name );
                        }
                        else {
                                sprintf(metadata->name, "%s_ece_%s", devname, name );
                        }
                }
*/
                sprintf(metadata->name, "%s/%s_%s", container->name, ld->name, name );
        }

        LOG_EXIT_PTR(metadata);
        return metadata;
}


DISKSEG * create_csm_data_segment( LOGICALDISK          *ld,
                                   storage_container_t  *container,
                                   lba_t                 start,
                                   sector_count_t        size,
                                   u_int32_t             object_flags)
{
        DISKSEG *seg;
//      char devname[EVMS_VOLUME_NAME_SIZE+1];

        LOG_ENTRY();

        seg = allocate_csm_segment( ld );

        if (seg) {

                seg->size        = size;
                seg->start       = start;
                seg->data_type   = DATA_TYPE;
                seg->flags       = object_flags;

                set_segment_storage_type(seg,container);

                seg->disk_group = container;
/*
                get_device_name(ld, devname);

                if ( ld->object_type != DISK ) {
                        sprintf(seg->name, "%s.ece_data", devname );
                }
                else {
                        if (devname[strlen(devname)-1] == '/') {
                                sprintf(seg->name, "%sece_data", devname );
                        }
                        else {
                                sprintf(seg->name, "%s_ece_data", devname );
                        }
                }
*/
                sprintf(seg->name, "%s/%s", container->name, ld->name );

        }

        LOG_EXIT_PTR(seg);
        return seg;
}


int  remove_csm_segment_from_list( list_anchor_t seglist, DISKSEG *seg )
{
        LOG_ENTRY();
        LOG_DEBUG("segment name= %s\n", seg->name );

        EngFncs->remove_thing( seglist, seg );

        EngFncs->unregister_name( seg->name );

        LOG_EXIT_INT(0);
        return 0;
}


int insert_csm_segment_into_list( list_anchor_t seglist, DISKSEG *seg)
{
        int rc;

        LOG_ENTRY();

        rc = EngFncs->register_name( seg->name );

        if (rc==0) {
                rc = insert_csm_segment_into_ordered_list( seglist, seg );
                if (rc) {
                        EngFncs->unregister_name( seg->name );
                }
        }

        LOG_EXIT_INT(rc);
        return rc;
}


int insert_csm_segment_into_ordered_list( list_anchor_t seglist, DISKSEG *seg )
{
        int         rc=-1;
        DISKSEG    *seg2;
        lba_t       seg2_end_lba;
        boolean     overlapping=FALSE;
        list_element_t  iter, e;

        LOG_ENTRY();
        LOG_DEBUG("seg name= %s   seg start= %"PRIu64"  ends= %"PRIu64"  size= %"PRIu64"\n", seg->name, seg->start, seg->start+seg->size-1, seg->size );

        LIST_FOR_EACH( seglist, iter, seg2 ) {

                seg2_end_lba = seg2->start + seg2->size - 1;

                // test and set ... overlapping segments flag
                if ( (  seg->start >= seg2->start )&&
                     (  seg->start <= seg2_end_lba)) {
                        overlapping = TRUE;
                }
                else if ( ( seg->start  <  seg2->start ) &&
                          ( seg2->start <= (seg->start + seg->size - 1)) ) {
                        overlapping = TRUE;
                }
                else {
                        overlapping = FALSE;
                }

                if ( overlapping == TRUE ) {

                        LOG_DEBUG("Error ... Overlapping Segments ...\n");
                        LOG_DEBUG("seg2:   name: %s\n",   seg2->name );
                        LOG_DEBUG("       start: %"PRIu64"\n", seg2->start );
                        LOG_DEBUG("        size: %"PRIu64"\n", seg2->size );
                        LOG_DEBUG("         end: %"PRIu64"\n", seg2_end_lba );
                        LOG_DEBUG(" overlap lba: %"PRIu64"\n", seg->start );

                        rc = EINVAL;    // must be genuine partition overlap
                        break;          // break out of loop ... looking for insertion pt

                }

                // test for ... valid insertion point
                if (seg2->start > seg->start ) {
                        rc=0;
                        break;
                }
        }

        switch (rc) {

        case 0:  /* Ok, found a segment we should insert in front of */
                e = EngFncs->insert_thing( seglist,
                                           seg,
                                           INSERT_BEFORE|EXCLUSIVE_INSERT,
                                           EngFncs->find_in_list(seglist, seg2, NULL) );
                if (e != NULL) {
                        rc = 0;
                }
                else {
                        rc = ENOMEM;
                }
                break;

        case -1:/* This new segment lays out on the disk at a higher */
                /* LBA than any other existing segment.  So, just    */
                /* insert at end of the segment list.                */
                e = EngFncs->insert_thing( seglist,
                                           seg,
                                           EXCLUSIVE_INSERT,
                                           NULL);
                if (e) {
                        rc = 0;
                }
                else {
                        rc = ENOMEM;
                }
                break;

        default:     /* REAL ERROR ...  */
                LOG_ERROR("error, insertion failed ... RC= %d\n",  rc);
                break;
        }


        LOG_EXIT_INT(rc);
        return rc;
}


char * guid_to_string( guid_t *id )
{
        char *guid_string=NULL;

        if (id) {

                guid_string = (char *) malloc(64);
                if (guid_string) {

                        sprintf( guid_string, "%08X-%04X-%04X-%02X%02X-%02X%02X%02X%02X%02X%02X",
                                 id->time_low,
                                 id->time_mid,
                                 id->time_high,
                                 id->clock_seq_high,
                                 id->clock_seq_low,
                                 id->node[0],
                                 id->node[1],
                                 id->node[2],
                                 id->node[3],
                                 id->node[4],
                                 id->node[5] );

                }

        }

        return guid_string;
}

static u_int32_t  str_to_ulong( char *str, int length )
{
        char *endptr = &str[length-1];

        return( strtoul(str, &endptr, 16) );
}

int string_to_guid( char *str, guid_t *id )
{
        int rc;

        REQUIRE(str != NULL);
        REQUIRE(id != NULL);
        REQUIRE(strlen(str)==36);
        REQUIRE(str[8]=='-');
        REQUIRE(str[13]=='-');
        REQUIRE(str[18]=='-');
        REQUIRE(str[23]=='-');

        id->time_low       = str_to_ulong( &str[0],  8 );
        id->time_mid       = str_to_ulong( &str[9],  4 );
        id->time_high      = str_to_ulong( &str[14], 4 );
        id->clock_seq_high = str_to_ulong( &str[19], 2 );
        id->clock_seq_low  = str_to_ulong( &str[21], 2 );
        id->node[0]        = str_to_ulong( &str[24], 2 );
        id->node[1]        = str_to_ulong( &str[26], 2 );
        id->node[2]        = str_to_ulong( &str[28], 2 );
        id->node[3]        = str_to_ulong( &str[30], 2 );
        id->node[4]        = str_to_ulong( &str[32], 2 );
        id->node[5]        = str_to_ulong( &str[34], 2 );

        rc = 0;

        return rc;
}


boolean isa_valid_csm_header( storage_object_t *object, csm_header_t *hdr )
{
        u_int32_t       crc;
        u_int32_t       calculated_crc;


        LOG_ENTRY();

        // check for valid signature
        if ( DISK_TO_CPU64(hdr->signature) != CSM_DISK_MAGIC ) {
                LOG_DEBUG("Invalid signature\n");
                LOG_EXIT_BOOL(FALSE);
                return FALSE;
        }

        // crc the header
        crc = DISK_TO_CPU32(hdr->crc);

        hdr->crc = 0;

        calculated_crc = ~( EngFncs->calculate_CRC( EVMS_INITIAL_CRC, hdr, DISK_TO_CPU32(hdr->size) ) );

        hdr->crc = CPU_TO_DISK32(crc);

        if ( calculated_crc != crc) {
                LOG_DEBUG("Invalid crc, result = FALSE\n");
                LOG_EXIT_BOOL(FALSE);
                return FALSE;
        }

        if ( DISK_TO_CPU64(hdr->start_useable) >= DISK_TO_CPU64(hdr->end_useable) ) {
                LOG_DEBUG("Invalid start_useable lba, result = FALSE\n");
                LOG_EXIT_BOOL(FALSE);
                return FALSE;
        }

        if ( DISK_TO_CPU64(hdr->end_useable) >=  (object->start + object->size -1)  ) {
                LOG_DEBUG("Invalid end useable lba, result = FALSE\n");
                LOG_EXIT_BOOL(FALSE);
                return FALSE;
        }

        if ( DISK_TO_CPU64(hdr->alternate_lba) >  (object->start + object->size -1)  ) {
                LOG_DEBUG("Invalid alternate lba, result = TRUE\n");
                LOG_EXIT_BOOL(FALSE);
                return FALSE;
        }

        LOG_DEBUG("success, result = TRUE\n");
        LOG_EXIT_BOOL(TRUE);
        return TRUE;
}

/*
 *  Function:  prune csm segment objects from LIST
 *
 *  Called when we need to remove our objects from a LIST.
 *  Typically used to back out changes or to destruct a
 *  CSM owned disk.
 */
void prune_csm_seg_objects_from_list( list_anchor_t list )
{
        list_element_t  iter,iter2;
        storage_object_t *seg;
        boolean prune;

        LIST_FOR_EACH_SAFE( list, iter, iter2, seg ) {

                prune = FALSE;

                if ( seg->plugin == csm_plugin_record_ptr ) {
                        free_csm_segment(seg);
                        prune = TRUE;
                }

                if (prune==TRUE) {
                        EngFncs->delete_element(iter);
                }
        }

}


/*
 *  Function:  csm_nodeid_to_string
 *
 *  Called to convert a Cluster Unique Identifier (CUID) to an
 *  asciiz string. CUID == ece_nodeid_t
 */
char * csm_nodeid_to_string( ece_nodeid_t *id )
{
        char *nodeid_string=NULL;
        int rc=0;
        uint buffer_size = NODEID_STRING_SIZE+1;

        LOG_ENTRY();

        if (id) {

                nodeid_string = (char *) calloc(1,buffer_size);
                if (nodeid_string) {

                        rc = EngFncs->nodeid_to_string( id,
                                                        nodeid_string,
                                                        &buffer_size);

                        if (rc == ENOSPC) {
                                free(nodeid_string);
                                nodeid_string = (char *) calloc(1,buffer_size);

                                rc = EngFncs->nodeid_to_string( id,
                                                                nodeid_string,
                                                                &buffer_size);
                        }

                        if (rc) {
                                free(nodeid_string);
                                nodeid_string=NULL;
                        }

                }

        }

        LOG_EXIT_PTR(nodeid_string);
        return nodeid_string;
}


/*
 *  Function:  csm_string_to_nodeid
 *
 *  Called to convert an asciiz string representing a nodeid to a
 *  Cluster Unique Identifier (CUID) a.k.a. ece_nodeid_t
 */
int csm_string_to_nodeid( char *str, ece_nodeid_t *nodeid )
{
        int rc=0;

        LOG_ENTRY();

        REQUIRE(str != NULL);
        REQUIRE(nodeid != NULL);

        rc = EngFncs->string_to_nodeid(str,nodeid);

        LOG_EXIT_INT(rc);
        return rc;
}


static boolean valid_cluster_node( ece_nodeid_t  *nodeid  )
{
        boolean  result=FALSE;
        char    *nodeid_string=NULL;

        LOG_ENTRY();

        if (nodeid) {

                nodeid_string = csm_nodeid_to_string(nodeid);
                if (nodeid_string) {
                        free(nodeid_string);
                        result=TRUE;
                }

        }

        LOG_EXIT_BOOL(result);
        return result;
}


/*
 *  Function: isa_accessible_container
 *
 *  Tests if a container is accessible by looking at the container
 *  flag bits.  Must be either cluster shared storage or
 *  else my cluster private storage -AND- we must be an active
 *  cluster member ... we must have quorum.
 */
boolean isa_accessible_container( storage_container_t *container )
{
        container_private_data_t *c_pdata;
        boolean result=FALSE;

        LOG_ENTRY();

        if (csm_has_quorum==TRUE)
                LOG_DEBUG("quorum        : yes\n");
        else
                LOG_DEBUG("quorum        : no\n");
        if (csm_admin_mode==TRUE)
                LOG_DEBUG("admin mode    : yes\n");
        else
                LOG_DEBUG("admin mode    : no\n");
        if (container==NULL)
                LOG_DEBUG("container     : NULL ptr\n");

        if (container) {

                c_pdata = (container_private_data_t *) container->private_data;

                if (memcmp(&csm_clusterid,&c_pdata->clusterid,sizeof(ece_clusterid_t))==0) {

                        LOG_DEBUG("clusterid     : Ok\n");

                        if (valid_cluster_node(&c_pdata->nodeid)==TRUE) {

                                LOG_DEBUG("nodeid        : Ok\n");

                                if ( (csm_has_quorum==TRUE || csm_admin_mode==TRUE) && container != NULL ) {

                                        if (csm_admin_mode == TRUE) {
                                                result = TRUE;
                                        }
                                        else if (container->flags & SCFLAG_CLUSTER_SHARED) {
                                                result = TRUE;
                                        }
                                        else if ( (container->flags & SCFLAG_CLUSTER_PRIVATE) &&
                                                  (memcmp(&c_pdata->nodeid, &csm_nodeid, sizeof(ece_nodeid_t))==0) ) {
                                                result = TRUE;
                                        }

                                }
                        }
                        else {
                                LOG_DEBUG("nodeid        : unknown to cluster\n");
                        }
                }
                else {
                        LOG_DEBUG("clusterid     : bad or diff cluster\n");
                }

        }

        if (result==TRUE) {
                LOG_DEBUG("container is accessible\n");
        }
        else {
                LOG_DEBUG("container not accessible\n");
        }

        LOG_EXIT_BOOL(result);
        return result;
}


