/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libcsm.so
 *
 *   File: discovery.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "csm_plugin.h"


/*
 *  Function:  display_segment_list
 *
 *  Called by discovery code, after successfully creating csm
 *  segment objects, to dump the information to the engine
 *  log for debugging purposes.
 */
static void  display_segment_list( LOGICALDISK  *ld)
{
        DISKSEG *seg;
        list_element_t iter;

        LOG_DEBUG("\n");
        LOG_DEBUG("  CSM Segment List ... logical disk= %s\n", ld->name);

        LIST_FOR_EACH( ld->parent_objects, iter, seg ){

                LOG_DEBUG("   start= %"PRIu64"  end= %"PRIu64"  size= %"PRIu64" name=%s\n",
                          seg->start,
                          seg->start+seg->size-1,
                          seg->size,
                          seg->name );
        }

        LOG_DEBUG("\n");
}


/*
 *  Function:  read_csm_header_sector
 *
 *  Called to read and validate a CSM header ... at a specific LBA
 *  on a logical disk. If a valid CSM header is found at the LBA
 *  it should be returned to the caller. Otherwise, return NULL.
 */
static csm_header_t *  read_csm_header_sector( storage_object_t *object,
                                               lba_t             lba )
{
        int rc=EINVAL;
        csm_header_t  *hdr = NULL;

        LOG_ENTRY();
        LOG_DEBUG("reading header off %s at lba %"PRIu64"\n", object->name, lba);

        hdr = (csm_header_t *) malloc( EVMS_VSECTOR_SIZE );
        if ( hdr ) {

                rc = READ( object, lba, 1, (void *) hdr );
                if ( rc == 0 ) {

                        if ( isa_valid_csm_header(object, hdr)==FALSE ) {
                                rc = ENODATA;
                        }

                }

        }

        if (rc) {
                if (hdr) free(hdr);
                hdr = NULL;
        }

        LOG_EXIT_PTR(hdr);
        return hdr;
}


/*
 *  Function:  get_csm_header
 *
 *  Called to probe a logical disk object for a CSM header and
 *  return a CSM header if one is found.  This routine is called
 *  by discovery code for 2 reasons:
 *
 *      (1) to determine if the logical disk has CSM metadata
 *          at all.
 *
 *      (2) to get a copy of the metadata without any regard to
 *          which copy is obtained.
 *
 *  Its job is to return whatever csm header it can find on the disk.
 *  Doesn't matter if its the primary or alternate header.
 */
static csm_header_t * get_csm_header( LOGICALDISK  *ld )
{
        csm_header_t  *hdr  = NULL;
        char * choices[] = {"Yes", "No", NULL};
        int answer = 0;     // Initialize to Revert

        LOG_ENTRY();

        // get primary header
        hdr = read_csm_header_sector(ld, 0 );

        // if we failed to get primary then ... get alternate header - last sector on object
        if (hdr == NULL) {

                hdr = read_csm_header_sector(ld, ld->size - 1 );

                if (hdr) {

                        QUESTION( &answer, choices,
                                  "\nErrors were found with the cluster segment manager metadata on drive %s.\n\n"
                                  "The primary header is missing or invalid but the alternate header was discovered on the drive.\n\n"
                                  "The cluster segment manager can be assigned to the drive and attempt to restore missing metadata, using "
                                  "the alternate header to provide information about the drive.\n\n"
                                  "Question: Would you like to cluster segment manager to be assigned to this drive?\n",
                                  ld->name);

                        if (answer == 1) {
                                free(hdr);
                                hdr=NULL;
                        }

                }
        }

        LOG_EXIT_PTR(hdr);
        return hdr;
}

/*
 *  Function:  get_segments
 *
 *  Called to create csm segment storage objects that map the
 *  specified logical disk.  The job of this routine is to build
 *  3 segment storage objects:
 *
 *      (1) Metadata    - always the 1st sector ... but also addressed
 *                        by the 2nd copy of metadata ... this
 *                        segment maps the 1st copy of metadata and
 *                        holds a ptr to the 1st CSM header. It also
 *                        prevents other plugins from successfully
 *                        discoverying their metadata on this storage
 *                        object.
 *      (2) Metadata    - usually the last sector ... but also addressed
 *                        by the 1st copy of metadata ... this segment
 *                        maps the 2nd copy of metadata and holds a ptr
 *                        to the 2nd CSM header.
 *      (3) Data        - maps the useable area of the logical disk ...
 *                        the sectors between the 2 copies of metadata.
 *
 */
int get_segments( storage_object_t     *object,
		  storage_container_t  *container )
{
        csm_header_t          *hdr=NULL;
        csm_header_t          *hdr1=NULL;
        csm_header_t          *hdr2=NULL;
        lba_t                  hdr1_lba=0;
        lba_t                  hdr2_lba=0;
        DISKSEG               *metadata1=NULL;
        DISKSEG               *metadata2=NULL;
        DISKSEG               *dataseg=NULL;
        int                    rc=0,i;
        disk_private_data_t   *disk_pdata = get_csm_disk_private_data(object);
        boolean                md1_dirty=FALSE;
        boolean                md2_dirty=FALSE;
        char                  *uuid_string;


        LOG_ENTRY();


        // Get 1st copy of csm header ... always at LBA 0
        hdr1 = read_csm_header_sector(object, 0);
        if (hdr1) {
                disk_csm_header_to_cpu( hdr1 );
                hdr1_lba = 0;
        }

        // Get 2nd copy of csm header ... 1st copy of csm header
        // points to 2nd copy.  If the 1st copy is bad ... then
        // the 2nd copy might be found in the last sector on the disk.
        if (hdr1) {
                hdr2 = read_csm_header_sector(object, hdr1->alternate_lba );
                if (hdr2) {
                        disk_csm_header_to_cpu( hdr2 );
                }
                hdr2_lba = hdr1->alternate_lba;
        }
        else {
                hdr2_lba = object->size - 1;

                // look in last 8 sectors for a csm header
                for (i=0; hdr2==NULL && i<8; i++, --hdr2_lba) {
                        hdr2 = read_csm_header_sector(object, hdr2_lba );
                }

                if (hdr2) {
                        disk_csm_header_to_cpu( hdr2 );
                }
        }

        LOG_DEBUG("hdr1= %p  hdr2= %p\n", hdr1, hdr2);

        // Check that we got at least 1 copy of the header
        if (hdr1==NULL && hdr2==NULL) {
                LOG_DEBUG("error, both copies of the csm header are missing.\n");
                LOG_EXIT_INT(0);
                return 0;
        }

        // Make sure that we have both copies of the CSM header by reproducing
        // a missing header from information obtained from the only good header.
        if (hdr1==NULL) {
                LOG_DEBUG("hdr1 is null\n");
                hdr1 = (csm_header_t *)calloc(1,EVMS_VSECTOR_SIZE);
                if (hdr1) {
                        memcpy(hdr1, hdr2, EVMS_VSECTOR_SIZE);

                        hdr1->alternate_lba = hdr2_lba;

                        md1_dirty = TRUE;   // new header is dirty of course

                        hdr = hdr2;
                }
                else {
                        if (hdr2) free(hdr2);
                        LOG_ERROR("error, unable to malloc a csm header\n");
                        LOG_EXIT_INT(0);
                        return 0;
                }

        }
        else if (hdr2==NULL) {
                LOG_DEBUG("hdr2 is null\n");
                hdr2 = (csm_header_t *)calloc(1,EVMS_VSECTOR_SIZE);
                if (hdr2) {
                        memcpy(hdr2, hdr1, EVMS_VSECTOR_SIZE);

                        hdr2->alternate_lba = hdr1_lba;

                        md2_dirty = TRUE;   // new header is dirty of course

                        hdr = hdr1;
                }
                else {
                        if (hdr1) free(hdr1);
                        LOG_ERROR("error, unable to malloc a csm header\n");
                        LOG_EXIT_INT(0);
                        return 0;
                }

        }
        else {
                LOG_DEBUG("hdr1 == hdr2 == Ok\n");
                hdr = hdr1; // both copies are Ok
        }

        // Register disk guid identifier - convert uuid to ascii string and register it.
        uuid_string = guid_to_string( &hdr->disk_id );
        if (uuid_string) {
                rc = EngFncs->register_name( uuid_string );
                free(uuid_string);
        }
        else {
                rc = ENOMEM;
        }

        if (rc) {
                if (hdr1) free(hdr1);
                if (hdr2) free(hdr2);
                LOG_ERROR("error, unable to convert DISK uuid identifier to ascii string for registering.\n");
                LOG_EXIT_INT(0);
                return 0;
        }

        // create 1st metadata segment
        metadata1 = create_csm_metadata_segment( object,
                                                 container,
                                                 hdr1_lba,
                                                 1,
                                                 "metadata1",
                                                 0 );

        if (metadata1 == NULL) {
                if (hdr1) free(hdr1);
                if (hdr2) free(hdr2);
                LOG_EXIT_INT(0);
                return 0;
        }

        if ( insert_csm_segment_into_list(object->parent_objects, metadata1) ) {
                if (hdr1) free(hdr1);
                if (hdr2) free(hdr2);
                free_csm_segment(metadata1);
                LOG_EXIT_INT(0);
                return 0;
        }

        // if primary header is missing then mark it dirty so engine will commit it.
        if (md1_dirty == TRUE) {
                MESSAGE("\nPrimary CSM Header is missing or corrupt. Marking %s dirty to correct the problem\n",
                        metadata1->name );
                metadata1->flags |= SOFLAG_DIRTY;
        }

        // create 2nd metadata segment
        metadata2 = create_csm_metadata_segment( object,
                                                 container,
                                                 hdr2_lba,
                                                 1,
                                                 "metadata2",
                                                 0 );

        if (metadata2 == NULL) {
                if (hdr1) free(hdr1);
                if (hdr2) free(hdr2);
                if (metadata1) free_csm_segment(metadata1);
                LOG_EXIT_INT(0);
                return 0;
        }

        if ( insert_csm_segment_into_list(object->parent_objects, metadata2) != 0 ) {
                if (hdr1) free(hdr1);
                if (hdr2) free(hdr2);
                if (metadata1) free_csm_segment(metadata1);
                if (metadata2) free_csm_segment(metadata2);
                LOG_EXIT_INT(0);
                return 0;
        }

        // if secondary header is missing then mark it dirty so engine will commit it.
        if (md2_dirty == TRUE) {
                MESSAGE("\nAlternate CSM Header is missing or corrupt. Marking %s dirty to correct the problem\n",
                        metadata2->name );
                metadata2->flags |= SOFLAG_DIRTY;
        }

        if (isa_accessible_container(container)==TRUE) {

                dataseg = create_csm_data_segment( object,
                                                   container,
                                                   hdr->start_useable,
                                                   hdr->end_useable - hdr->start_useable + 1,
                                                   0 );

                if (dataseg == NULL) {
                        if (hdr1) free(hdr1);
                        if (hdr2) free(hdr2);
                        if (metadata1) free_csm_segment(metadata1);
                        if (metadata2) free_csm_segment(metadata2);
                        LOG_EXIT_INT(0);
                        return 0;
                }

                if ( insert_csm_segment_into_list(object->parent_objects, dataseg) != 0 ) {
                        if (hdr1) free(hdr1);
                        if (hdr2) free(hdr2);
                        if (metadata1) free_csm_segment(metadata1);
                        if (metadata2) free_csm_segment(metadata2);
                        if (dataseg) free_csm_segment(dataseg);
                        LOG_EXIT_INT(0);
                        return 0;
                }

        }

        // Save headers in their respective metadata segments
        ((seg_private_data_t *)metadata1->private_data)->hdr = hdr1;
        ((seg_private_data_t *)metadata2->private_data)->hdr = hdr2;

        // save metadata segments in logical disk private data for quick searching
        disk_pdata->md1 = metadata1;
        disk_pdata->md2 = metadata2;

        LOG_EXIT_INT(3);
        return 3;
}


/*
 *  Function: get_csm_segment_devmap_info
 *
 *  Called to test if the segment has an active device mapper
 *  node in the kernel and set the object info accordingly.
 */
void get_csm_segment_devmap_info( storage_object_t *seg, storage_container_t *container )
{
        seg_private_data_t *pdata;
        int rc=0;
        dm_target_t *targets=NULL;
        dm_device_t *dev=NULL;
        boolean ok_to_activate = FALSE;

        LOG_ENTRY();


        // Test ... if it is Ok to mark this segment as needing
        // activate.  Also, determine if we should deactivate it
        // at all by examining: quorum & container ownership.
        if ( isa_accessible_container( container )==TRUE) {
                ok_to_activate = TRUE;
        }

        pdata = (seg_private_data_t *) seg->private_data;

        // all data segments of course but also any data segments we are
        // currently hiding but may need to deactivate in the kernel.
        if ( seg->data_type == DATA_TYPE ) {

                rc = EngFncs->dm_update_status(seg);
                if (!rc) {

                        if (seg->flags & SOFLAG_ACTIVE) {

                                if (ok_to_activate) {

                                        rc = EngFncs->dm_get_targets(seg, &targets);
                                        if (  (!rc) &&                     // RC==Zero
                                              (targets) &&                // got a target list
                                              (targets->next == NULL) &&  // with 1 target only
                                              (targets->data.linear) &&     // with device data
                                              (targets->start == 0)) {    // and target starts at Zero

                                                dev = targets->data.linear;

                                                // test if dm map is incorrect and we need to reactivate
                                                if ( seg->start != dev->start ||
                                                     seg->size  != targets->length ) {
                                                        LOG_DEBUG("this segment is being marked needs_activate\n");
                                                        seg->flags |= SOFLAG_NEEDS_ACTIVATE;
                                                }
                                                else {
                                                        seg->flags &= ~SOFLAG_NEEDS_ACTIVATE;
                                                }

                                        }

                                        if (targets) EngFncs->dm_deallocate_targets(targets);
                                }
                                else {
                                        LOG_DEBUG("seg is active but needs to be deactivated.\n");
                                        seg->flags |= SOFLAG_NEEDS_DEACTIVATE;
                                }

                        }
                        else {
                                // SEGMENT IS NOT ACTIVE ...

                                if ( ok_to_activate == TRUE ) {
                                        seg->flags |= SOFLAG_NEEDS_ACTIVATE;
                                }
                                else {
                                        LOG_DEBUG("seg is not active but not this nodeid so not activating\n");
                                }

                        }

                }

        }

        LOG_EXIT_VOID();
}



/*
 *  Function:  discover_cluster_segments
 *
 *  Called by the csm discover api ... once for each storage object
 *  that is found in the discovery list of objects.  Our job is to
 *  probe a storage object, looking for CSM metadata.  If we find our
 *  metadata:
 *
 *      - create csm segment storage objects
 *      - add the logical disk object to the consuming container
 *      - add the data segment object to the container's produced object list
 *
 *  Return: the number of segment objects created 0-n and place new
 *          segment objects on the output_object list.
 */
int discover_cluster_segments( storage_object_t *obj,
                               list_anchor_t     output_objects,
                               uint             *count )
{
        boolean                     success=FALSE;
        int                         rc=EINVAL;
        uint                        seg_count=0;
        seg_private_data_t         *pdata=NULL;
        disk_private_data_t        *disk_pdata=NULL;
        csm_header_t               *hdr=NULL;
        storage_container_t        *container=NULL;
        boolean                     created_container=FALSE;
        boolean                     created_disk_pdata=FALSE;
        u_int32_t                   flags;
        list_element_t              iter;
        DISKSEG                    *seg;

        LOG_ENTRY();
        LOG_DEBUG("examining object %s\n", obj->name );

        pdata = (seg_private_data_t *) obj->private_data;

        // check if object is acceptable and probe for CSM header
        if ( ( obj->plugin == csm_plugin_record_ptr ) ||
             ( obj->data_type != DATA_TYPE ) ||
             ( obj->object_type == SEGMENT && (pdata->cflags&SEG_CFLAG_TOP_SEGMENT))) {

                LOG_DEBUG("object is not acceptable or \n" );
                hdr = NULL;

        }
        else {
                hdr = get_csm_header( obj );
        }

        if (hdr) {

                // convert from disk format to cpu format
                disk_csm_header_to_cpu( hdr );

                // for debugging ...
                display_csm_header( hdr );

                // make sure ...
                //
                // (1) its not a multipath to a disk we have already seen
                // (2) its not a disk belonging to another cluster
                //
                if ( isa_known_disk(&hdr->disk_id) == FALSE ) {

                        // find the container
                        rc = csm_find_container( hdr->container_name, &container );
                        if (rc) {

                                // not found ... must be 1st time weve seen a CSM hdr
                                // identifying this container ... so create it.
                                flags = 0;
                                if (hdr->flags & CSM_CLUSTER_PRIVATE) {
                                        flags |= SCFLAG_CLUSTER_PRIVATE;
                                }
                                else if (hdr->flags & CSM_CLUSTER_SHARED) {
                                        flags |= SCFLAG_CLUSTER_SHARED;
                                }
                                else if (hdr->flags & CSM_CLUSTER_DEPORTED) {
                                        flags |= SCFLAG_CLUSTER_DEPORTED;
                                }
                                else {
                                        LOG_DEBUG("csm header flags are not set to any known storage type\n");
                                }

                                rc = csm_create_container_storage_object( hdr->container_name,
                                                                          &container,
                                                                          &hdr->nodeid,
                                                                          &hdr->clusterid,
                                                                          flags );
                                if (!rc) {
                                        created_container=TRUE;
                                }

                        }

                }
                else {
                        LOG_DEBUG("tossing multipath disk\n");
                }

        }

        if (rc==0) {

                create_csm_disk_private_data( obj );

                disk_pdata = get_csm_disk_private_data(obj);

                if (disk_pdata) {

                        created_disk_pdata=TRUE;

                        seg_count = get_segments(obj, container);

                        if (seg_count > 0) {

                                display_segment_list(obj);

                                rc = add_disk_to_container(obj, container);

                                if (rc==0) {

                                        container->flags &= ~SCFLAG_DIRTY; // cuz add object doesnt know
                                                                           // the diff between discovery
                                                                           // and assign.

                                        rc=EngFncs->concatenate_lists( output_objects, obj->parent_objects );

                                        if (rc==0) {
                                                LIST_FOR_EACH( obj->parent_objects, iter, seg ) {
                                                        get_csm_segment_devmap_info(seg,container);
                                                }
                                                success = TRUE;
                                        }

                                }

                        }
                        else {
                                LOG_ERROR("error, CSM object but no segments produced\n");
                        }

                }
                else {
                        LOG_ERROR("error, unable to create logical disk private data\n");
                }

        }

        if (success == FALSE) {

                // toss any segments we created
                if (seg_count > 0) {
                        prune_csm_seg_objects_from_list(obj->parent_objects);
                }

                // place the disk storage object back onto the discovery list
                EngFncs->insert_thing( output_objects,
                                       obj,
                                       INSERT_AFTER,
                                       NULL );

                // toss our disk private data
                if (created_disk_pdata==TRUE) {
                        delete_csm_disk_private_data(obj);
                }

                // remove consuming container info from disk
                obj->consuming_container = NULL;

                // toss container if we just created one
                if (created_container==TRUE) {
                        free_csm_container(container);
                }

                // return - no objects created
                seg_count = 0;

        }

        LOG_DEBUG( "discovery ...resulted in %d new segments\n", seg_count );

        // return count of objects we created
        *count += seg_count;

        if (hdr) free(hdr);

        LOG_EXIT_INT(0);
        return 0;
}

