/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libcsm.so
 *
 *   File: commit.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "csm_plugin.h"



/*
 *  Function:  write_metadata
 *
 *  Called with a CSM metadata storage object to commit to disk. A CSM metadata
 *  storage object holds a ptr in its private data to a CSM header that will be
 *  committed to disk.  The size and location of the CSM header on disk?  It is
 *  always equal to the size and location of the metadata storage object!
 */
static int  write_metadata( DISKSEG  *md, LOGICALDISK  *ld )
{
	int  rc=0;
	csm_header_t *hdr=NULL;
	u_int32_t     crc;
	u_int32_t     size = 0;


	LOG_ENTRY();

	LOG_DEBUG("writing metadata on object %s\n", ld->name);

	hdr = (void *) calloc(1, (EVMS_VSECTOR_SIZE*md->size) );
	if (hdr) {
		memcpy( hdr,
			((seg_private_data_t *)md->private_data)->hdr,
			((seg_private_data_t *)md->private_data)->hdr->size );
		size = hdr->size;
		cpu_csm_header_to_disk( hdr );
		hdr->crc = 0;
		crc = EngFncs->calculate_CRC( EVMS_INITIAL_CRC,hdr,size );
		hdr->crc = CPU_TO_DISK32(~crc);

		rc = WRITE( ld, md->start, md->size, (void *) hdr );
		if (rc) {
			rc = WRITE( ld, md->start, md->size, (void *) hdr );
		}

		if (rc) {
			LOG_ERROR("ERROR--> WRITE METADATA TO LBA %"PRIu64" FAILED WITH RC= %d\n", md->start,rc);
		}

		free(hdr);
	}
	else {
		rc = ENOMEM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Function:  commit_csm_metadata
 *
 *  Called by the CSM plug-in commit api with:
 *
 *      (1) a dirty segment storage object
 *      (2) the logical disk the segment is found on
 *      (3) the current commit phase
 *
 *  A CSM commit follows this logic:
 *
 *  IF   ... dirty_segment == a CSM metadata segment
 *  THEN ... IF metadata segments commit phase matches current commit phase
 *           THEN commit the metdata segment and mark it clean
 *           ELSE leave the metadata segment dirty for subsequent commit phases
 *  ELSE ... simply mark the storage object clean
 *
 *  The result is that the all 3 CSM segments will be committed at an
 *  appropriate time. The data segment ... whenever it first appears. The
 *  1st metadata segment during phase 1.  The 2nd metadata segment during
 *  phase 2.
 */
int  commit_csm_metadata( DISKSEG *seg, LOGICALDISK *ld,  uint commit_phase )
{
	int rc=0;
	seg_private_data_t *pdata;


	LOG_ENTRY();

	REQUIRE(ld != NULL);
	REQUIRE(seg != NULL);

	pdata = (seg_private_data_t *) seg->private_data;

	REQUIRE(pdata != NULL);

	if ( seg->data_type == META_DATA_TYPE ) {

		if ( pdata->commit_phase == commit_phase ) {

			rc = write_metadata( seg, ld );
			if (!rc) {
				seg->flags &= ~SOFLAG_DIRTY;
			}

		}

	}
	else {
		seg->flags &= ~SOFLAG_DIRTY;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

