/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libbsd.so
 *
 *   File: segoptions.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "bsd_plugin.h"





int bsd_init_task(task_context_t * context)
{
        LOG_ENTRY();
        REQUIRE(context != NULL);
        LOG_EXIT_INT(ENOSYS);
        return ENOSYS;
}


int bsd_set_objects( task_context_t * context,
                     list_anchor_t          declined_objects,
                     task_effect_t  * effect )
{
        LOG_ENTRY();
        REQUIRE(context != NULL);
        REQUIRE(declined_objects != NULL);
        REQUIRE(effect != NULL);
        LOG_EXIT_INT(ENOSYS);
        return ENOSYS;
}


int bsd_get_option_count(task_context_t * task)
{
        int count=0;

        LOG_ENTRY();

        REQUIRE(task != NULL);

        switch (task->action) {
        
        case EVMS_Task_Create:
                count = BSD_CREATE_OPTION_COUNT;
                break;

        case EVMS_Task_Assign_Plugin:
                count = BSD_ASSIGN_OPTION_COUNT;
                break;

        case EVMS_Task_Expand:
                count = BSD_EXPAND_OPTION_COUNT;
                break;

        case EVMS_Task_Shrink:
                count = BSD_SHRINK_OPTION_COUNT;
                break;

        default:
                LOG_DEBUG("unsupported task ... task->action = %d\n", task->action);                    
                count = 0;
                break;
        }

        LOG_EXIT_INT(count);
        return count;
}


int bsd_set_option( task_context_t * context,
                    u_int32_t        index,
                    value_t        * value,
                    task_effect_t  * effect )
{
        LOG_ENTRY();

        REQUIRE(context != NULL);
        REQUIRE(value != NULL);
        REQUIRE(effect != NULL);

        LOG_EXIT_INT(ENOSYS);
        return ENOSYS;
}



/*
 * Returns segment specific information
 */
int bsd_get_segment_info( storage_object_t  * object, extended_info_array_t * * info)
{
        int rc = EINVAL;
        extended_info_array_t   *Info;


        LOG_ENTRY();

        rc    = ENOMEM;  // init to failed calloc
        *info = NULL;     // init to no info returned

        Info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + ( SEG_INFO_COUNT * sizeof(extended_info_t) ) );
        if (Info) {

                Info->count = SEG_INFO_COUNT;

                SET_STRING( Info->info[SEG_INFO_NAME_INDEX].name, "Name" );
                SET_STRING( Info->info[SEG_INFO_NAME_INDEX].title, "Name" );
                SET_STRING( Info->info[SEG_INFO_NAME_INDEX].desc, "This is the partition name. It must be unique on the system.");
                Info->info[SEG_INFO_NAME_INDEX].type               = EVMS_Type_String;
                Info->info[SEG_INFO_NAME_INDEX].unit               = EVMS_Unit_None;
                SET_STRING( Info->info[SEG_INFO_NAME_INDEX].value.s, object->name );
                Info->info[SEG_INFO_NAME_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_INFO_NAME_INDEX].group, 0, sizeof(group_info_t));

                SET_STRING( Info->info[SEG_INFO_SIZE_INDEX].name, "Size" );
                SET_STRING( Info->info[SEG_INFO_SIZE_INDEX].title, "Size" );
                SET_STRING( Info->info[SEG_INFO_SIZE_INDEX].desc, "This is the size of the partition in sectors.");
                Info->info[SEG_INFO_SIZE_INDEX].type               = EVMS_Type_Unsigned_Int64;
                Info->info[SEG_INFO_SIZE_INDEX].unit               = EVMS_Unit_Sectors;
                Info->info[SEG_INFO_SIZE_INDEX].value.ui64         = object->size;
                Info->info[SEG_INFO_SIZE_INDEX].format             = EVMS_Format_Normal;
                Info->info[SEG_INFO_SIZE_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_INFO_SIZE_INDEX].group, 0, sizeof(group_info_t));
                Info->info[SEG_INFO_SIZE_INDEX].flags |= EVMS_EINFO_FLAGS_NO_UNIT_CONVERSION;

                SET_STRING( Info->info[SEG_INFO_START_INDEX].name, "Start" );
                SET_STRING( Info->info[SEG_INFO_START_INDEX].title, "Start LBA" );
                SET_STRING( Info->info[SEG_INFO_START_INDEX].desc, "This is the sector offset of the partition on the disk, i.e., the logical block address of the first sector of the partition.");
                Info->info[SEG_INFO_START_INDEX].type               = EVMS_Type_Unsigned_Int64;
                Info->info[SEG_INFO_START_INDEX].unit               = EVMS_Unit_None;
                Info->info[SEG_INFO_START_INDEX].value.ui64         = object->start;
                Info->info[SEG_INFO_START_INDEX].format             = EVMS_Format_Normal;
                Info->info[SEG_INFO_START_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_INFO_START_INDEX].group, 0, sizeof(group_info_t));


                SET_STRING( Info->info[SEG_INFO_TYPE_INDEX].name, "Type" );
                SET_STRING( Info->info[SEG_INFO_TYPE_INDEX].title, "Type" );
                SET_STRING( Info->info[SEG_INFO_TYPE_INDEX].desc, "Partition type - data, metadata, free.");
                Info->info[SEG_INFO_TYPE_INDEX].type               = EVMS_Type_String;
                Info->info[SEG_INFO_TYPE_INDEX].unit               = EVMS_Unit_None;
                Info->info[SEG_INFO_TYPE_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_INFO_TYPE_INDEX].group, 0, sizeof(group_info_t));

                if ( object->data_type == DATA_TYPE ) {
                        SET_STRING( Info->info[SEG_INFO_TYPE_INDEX].value.s, "data" );
                }
                else if ( object->data_type == META_DATA_TYPE ) {
                        SET_STRING( Info->info[SEG_INFO_TYPE_INDEX].value.s, "metadata" );
                        Info->info[SEG_INFO_TYPE_INDEX].flags           = EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE;
                }
                else {
                        SET_STRING( Info->info[SEG_INFO_TYPE_INDEX].value.s, "unknown" );
                }

                *info = Info;

                rc = 0;
        }
        else {
                LOG_ERROR("unable to malloc memory for extended info array\n");
        }


        LOG_EXIT_INT(rc);
        return rc;
}



/*
 * Returns BSD Disk Label (metadata) information
 */
int bsd_get_metadata_info(storage_object_t * object, extended_info_array_t **info)
{
        int rc = EINVAL;
        extended_info_array_t   *Info;
        LOGICALDISK             *ld;
        bsd_disklabel_t         *disk_label;
        char                     geometry_string[128];

        LOG_ENTRY();

        ld         = get_logical_disk( object );
        disk_label = get_bsd_disk_label( ld );

        REQUIRE(ld != NULL);
        REQUIRE(disk_label != NULL);

        rc    = ENOMEM;  // init to failed calloc
        *info = NULL;     // init to no info returned

        Info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + ( BSD_LABEL_INFO_COUNT * sizeof(extended_info_t) ) );
        if (Info) {

                Info->count = BSD_LABEL_INFO_COUNT;

                sprintf(geometry_string,"C= %d   H= %d  S= %d", 
                        DISK_TO_CPU32(disk_label->d_ncylinders), 
                        DISK_TO_CPU32(disk_label->d_ntracks), 
                        DISK_TO_CPU32(disk_label->d_nsectors));

                SET_STRING( Info->info[BSD_INFO_GEO_INDEX].name, "Geometry" );
                SET_STRING( Info->info[BSD_INFO_GEO_INDEX].title, "Disk Geometry" );
                SET_STRING( Info->info[BSD_INFO_GEO_INDEX].desc, "This is the disk geometry reported in the label.");
                Info->info[BSD_INFO_GEO_INDEX].type               = EVMS_Type_String;
                Info->info[BSD_INFO_GEO_INDEX].unit               = EVMS_Unit_None;
                Info->info[BSD_INFO_GEO_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[BSD_INFO_GEO_INDEX].group, 0, sizeof(group_info_t));
                SET_STRING( Info->info[BSD_INFO_GEO_INDEX].value.s, geometry_string );                          

                SET_STRING( Info->info[BSD_INFO_SECTORSIZE_INDEX].name, "Sector Size" );
                SET_STRING( Info->info[BSD_INFO_SECTORSIZE_INDEX].title, "Sector Size" );
                SET_STRING( Info->info[BSD_INFO_SECTORSIZE_INDEX].desc, "Sector size reported by disk label.");
                Info->info[BSD_INFO_SECTORSIZE_INDEX].type               = EVMS_Type_Unsigned_Int32;
                Info->info[BSD_INFO_SECTORSIZE_INDEX].unit               = EVMS_Unit_None;
                Info->info[BSD_INFO_SECTORSIZE_INDEX].value.ui64         = DISK_TO_CPU32(disk_label->d_secsize);
                Info->info[BSD_INFO_SECTORSIZE_INDEX].format             = EVMS_Format_Normal;
                Info->info[BSD_INFO_SECTORSIZE_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[BSD_INFO_SECTORSIZE_INDEX].group, 0, sizeof(group_info_t));

                SET_STRING( Info->info[BSD_INFO_PCOUNT_INDEX].name, "Partitions" );
                SET_STRING( Info->info[BSD_INFO_PCOUNT_INDEX].title, "Partition Count" );
                SET_STRING( Info->info[BSD_INFO_PCOUNT_INDEX].desc, "The number of partitions reported by the disk label.");
                Info->info[BSD_INFO_PCOUNT_INDEX].type               = EVMS_Type_Unsigned_Int16;
                Info->info[BSD_INFO_PCOUNT_INDEX].unit               = EVMS_Unit_None;
                Info->info[BSD_INFO_PCOUNT_INDEX].value.ui64         = DISK_TO_CPU16(disk_label->d_npartitions);
                Info->info[BSD_INFO_PCOUNT_INDEX].format             = EVMS_Format_Normal;
                Info->info[BSD_INFO_PCOUNT_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[BSD_INFO_PCOUNT_INDEX].group, 0, sizeof(group_info_t));

                SET_STRING( Info->info[BSD_INFO_BOOTSIZE_INDEX].name, "Boot Size" );
                SET_STRING( Info->info[BSD_INFO_BOOTSIZE_INDEX].title, "Boot Area Size" );
                SET_STRING( Info->info[BSD_INFO_BOOTSIZE_INDEX].desc, "Size of boot area that is reported by the disk label.");
                Info->info[BSD_INFO_BOOTSIZE_INDEX].type               = EVMS_Type_Unsigned_Int32;
                Info->info[BSD_INFO_BOOTSIZE_INDEX].unit               = EVMS_Unit_None;
                Info->info[BSD_INFO_BOOTSIZE_INDEX].value.ui64         = DISK_TO_CPU32(disk_label->d_bbsize);
                Info->info[BSD_INFO_BOOTSIZE_INDEX].format             = EVMS_Format_Normal;
                Info->info[BSD_INFO_BOOTSIZE_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[BSD_INFO_BOOTSIZE_INDEX].group, 0, sizeof(group_info_t));

                SET_STRING( Info->info[BSD_INFO_SBSIZE_INDEX].name, "SB Size" );
                SET_STRING( Info->info[BSD_INFO_SBSIZE_INDEX].title, "Superblock Size" );
                SET_STRING( Info->info[BSD_INFO_SBSIZE_INDEX].desc, "Size of file system super block that is reported by the disk label.");
                Info->info[BSD_INFO_SBSIZE_INDEX].type               = EVMS_Type_Unsigned_Int32;
                Info->info[BSD_INFO_SBSIZE_INDEX].unit               = EVMS_Unit_None;
                Info->info[BSD_INFO_SBSIZE_INDEX].value.ui64         = DISK_TO_CPU32(disk_label->d_sbsize);
                Info->info[BSD_INFO_SBSIZE_INDEX].format             = EVMS_Format_Normal;
                Info->info[BSD_INFO_SBSIZE_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[BSD_INFO_SBSIZE_INDEX].group, 0, sizeof(group_info_t));

                *info = Info;

                rc = 0;

        }
        else {
                LOG_ERROR("unable to malloc memory for extended info array\n");
        }


        LOG_EXIT_INT(rc);
        return rc;
}


/*
 * Returns plugin specific information
 */
int bsd_get_plugin_info( char * descriptor_name, extended_info_array_t * * info )
{
        int rc = EINVAL;
        extended_info_array_t   *Info;
        char                     version_string[64];
        char                     required_engine_api_version_string[64];
        char                     required_plugin_api_version_string[64];


        LOG_ENTRY();

        // a measure of protection ...
        if (info == NULL) {
                LOG_EXIT_INT(rc);
                return rc;
        }

        rc    = ENOMEM;  // init to failed calloc
        *info = NULL;     // init to no info returned

        Info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + (PLUGIN_INFO_COUNT*sizeof(extended_info_t))  );
        if (Info) {

                Info->count = PLUGIN_INFO_COUNT;

                sprintf(version_string, "%d.%d.%d",
                        MAJOR_VERSION,
                        MINOR_VERSION,
                        PATCH_LEVEL );

                sprintf(required_engine_api_version_string, "%d.%d.%d",
                        bsd_plugin->required_engine_api_version.major,
                        bsd_plugin->required_engine_api_version.minor,
                        bsd_plugin->required_engine_api_version.patchlevel );

                sprintf(required_plugin_api_version_string, "%d.%d.%d",
                        bsd_plugin->required_plugin_api_version.plugin.major,
                        bsd_plugin->required_plugin_api_version.plugin.minor,
                        bsd_plugin->required_plugin_api_version.plugin.patchlevel );

                SET_STRING( Info->info[PLUGIN_INFO_SNAME_INDEX].name, "ShortName" );
                SET_STRING( Info->info[PLUGIN_INFO_SNAME_INDEX].title, "Short Name" );
                SET_STRING( Info->info[PLUGIN_INFO_SNAME_INDEX].desc, "A short name given to this plugin.");
                Info->info[PLUGIN_INFO_SNAME_INDEX].type               = EVMS_Type_String;
                Info->info[PLUGIN_INFO_SNAME_INDEX].unit               = EVMS_Unit_None;
                SET_STRING( Info->info[PLUGIN_INFO_SNAME_INDEX].value.s, bsd_plugin->short_name );
                Info->info[PLUGIN_INFO_SNAME_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[PLUGIN_INFO_SNAME_INDEX].group, 0, sizeof(group_info_t));

                SET_STRING( Info->info[PLUGIN_INFO_LNAME_INDEX].name, "LongName" );
                SET_STRING( Info->info[PLUGIN_INFO_LNAME_INDEX].title, "Long Name" );
                SET_STRING( Info->info[PLUGIN_INFO_LNAME_INDEX].desc, "A longer and more descriptive name for this plugin.");
                Info->info[PLUGIN_INFO_LNAME_INDEX].type               = EVMS_Type_String;
                Info->info[PLUGIN_INFO_LNAME_INDEX].unit               = EVMS_Unit_None;
                SET_STRING( Info->info[PLUGIN_INFO_LNAME_INDEX].value.s, bsd_plugin->long_name );
                Info->info[PLUGIN_INFO_LNAME_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[PLUGIN_INFO_LNAME_INDEX].group, 0, sizeof(group_info_t));

                SET_STRING( Info->info[PLUGIN_INFO_TYPE_INDEX].name, "Type" );
                SET_STRING( Info->info[PLUGIN_INFO_TYPE_INDEX].title, "Plugin Type" );
                SET_STRING( Info->info[PLUGIN_INFO_TYPE_INDEX].desc, "There are various types of plugins; each responsible for some kind of storage object.");
                Info->info[PLUGIN_INFO_TYPE_INDEX].type               = EVMS_Type_String;
                Info->info[PLUGIN_INFO_TYPE_INDEX].unit               = EVMS_Unit_None;
                SET_STRING( Info->info[PLUGIN_INFO_TYPE_INDEX].value.s, "Partition Manager" );
                Info->info[PLUGIN_INFO_TYPE_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[PLUGIN_INFO_TYPE_INDEX].group, 0, sizeof(group_info_t));

                SET_STRING( Info->info[PLUGIN_INFO_VERSION_INDEX].name, "Version" );
                SET_STRING( Info->info[PLUGIN_INFO_VERSION_INDEX].title, "Plugin Version" );
                SET_STRING( Info->info[PLUGIN_INFO_VERSION_INDEX].desc, "This is the version number of the plugin.");
                Info->info[PLUGIN_INFO_VERSION_INDEX].type               = EVMS_Type_String;
                Info->info[PLUGIN_INFO_VERSION_INDEX].unit               = EVMS_Unit_None;
                SET_STRING( Info->info[PLUGIN_INFO_VERSION_INDEX].value.s, version_string );
                Info->info[PLUGIN_INFO_VERSION_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[PLUGIN_INFO_VERSION_INDEX].group, 0, sizeof(group_info_t));

                SET_STRING( Info->info[PLUGIN_INFO_REQENGSVCVERSION_INDEX].name, "Required Engine Services Version" );
                SET_STRING( Info->info[PLUGIN_INFO_REQENGSVCVERSION_INDEX].title, "Required Engine Services Version" );
                SET_STRING( Info->info[PLUGIN_INFO_REQENGSVCVERSION_INDEX].desc, "This is the version of the Engine services that this plugin requires. It will not run on older versions of the Engine services.");
                Info->info[PLUGIN_INFO_REQENGSVCVERSION_INDEX].type               = EVMS_Type_String;
                Info->info[PLUGIN_INFO_REQENGSVCVERSION_INDEX].unit               = EVMS_Unit_None;
                SET_STRING( Info->info[PLUGIN_INFO_REQENGSVCVERSION_INDEX].value.s, required_engine_api_version_string );
                Info->info[PLUGIN_INFO_REQENGSVCVERSION_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[PLUGIN_INFO_REQENGSVCVERSION_INDEX].group, 0, sizeof(group_info_t));

                SET_STRING( Info->info[PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].name, "Required Engine Plug-in API Version" );
                SET_STRING( Info->info[PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].title, "Required Engine Plug-in API Version" );
                SET_STRING( Info->info[PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].desc, "This is the version of the Engine plug-in API that this plugin requires. It will not run on older versions of the Engine plug-in API.");
                Info->info[PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].type               = EVMS_Type_String;
                Info->info[PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].unit               = EVMS_Unit_None;
                SET_STRING( Info->info[PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].value.s, required_plugin_api_version_string );
                Info->info[PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].group, 0, sizeof(group_info_t));

                *info = Info;

                rc = 0;
        }


        LOG_EXIT_INT(rc);
        return rc;
}

