/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: task.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

#include "fullengine.h"
#include "task.h"
#include "lists.h"
#include "engine.h"
#include "handlemgr.h"
#include "common.h"
#include "option.h"
#include "memman.h"
#include "internalAPI.h"
#include "expand.h"
#include "shrink.h"
#include "remote.h"

typedef struct task_name_s {
	task_action_t task_action;
	char        * task_name;
} task_name_t;

static task_name_t task_names[] = {
	{EVMS_Task_Create,             "EVMS_Task_Create"},
	{EVMS_Task_Create_Container,   "EVMS_Task_Create_Container"},
	{EVMS_Task_Assign_Plugin,      "EVMS_Task_Assign_Plugin"},
	{EVMS_Task_Expand_Container,   "EVMS_Task_Expand_Container"},
	{EVMS_Task_Set_Info,           "EVMS_Task_Set_Info"},
	{EVMS_Task_Expand,             "EVMS_Task_Expand"},
	{EVMS_Task_Shrink,             "EVMS_Task_Shrink"},
	{EVMS_Task_Slide,              "EVMS_Task_Slide"},
	{EVMS_Task_Move,               "EVMS_Task_Move"},
	{EVMS_Task_mkfs,               "EVMS_Task_mkfs"},
	{EVMS_Task_fsck,               "EVMS_Task_fsck"},
	{EVMS_Task_defrag,             "EVMS_Task_defrag"},
	{EVMS_Task_Message,            "EVMS_Task_Message"},
	{EVMS_Task_Add_Feature,        "EVMS_Task_Add_Feature"},
	{EVMS_Task_Shrink_Container,   "EVMS_Task_Shrink_Container"},
	{EVMS_Task_Set_Container_Info, "EVMS_Task_Set_Container_Info"}
};

static char task_msg[32];

static char * get_task_name(task_action_t action) {

	int i;

	if (action >= EVMS_Task_Plugin_Function) {
		sprintf(task_msg, "Plug-in function index %d", action - EVMS_Task_Plugin_Function);
		return task_msg;
	}

	for (i = 0; i < (sizeof(task_names) / sizeof(task_name_t)); i++) {
		if (task_names[i].task_action == action) {
			return task_names[i].task_name;
		}
	}

	return "Unknown task action";
}


/*
 *
 *   static inline void FreeTaskMemory(task_handle_t, task_context_t *)
 *
 *   Description:
 *      This routine frees all memory dynamically allocated by the task
 *      API referenced in the task context structure. This specifically
 *      includes the parameters, option descriptors and array, and the
 *      option values and array.
 *
 *   Entry:
 *      task - address of task context structure created by evms_create_task()
 *
 *   Exit:
 *      All task context memory is freed
 *
 */
static inline void FreeTaskMemory(task_context_t * task) {

	LOG_PROC_ENTRY();

	if (task != NULL) {

		int i;
		/*
		 * Avoiding memory leaks is of great importance here.
		 */
		if (task->selected_objects != NULL) {
			/*
			 * Destroy the parameter list. Don't free the
			 * objects in the list as they are on someone
			 * else's list.
			 */
			destroy_list(task->selected_objects);
		}
		if (task->option_descriptors != NULL) {
			/*
			 * All strings in the descriptor structure should be dynamically
			 * allocated.  This includes, constraint lists containing strings
			 * and range structures for numeric ranges.  We free all of these
			 * before freeing the option descriptor array.
			 */
			for (i=0; i < task->option_descriptors->count; i++) {
				free_option_descriptor_contents(&(task->option_descriptors->option[i]));
			}

			engine_free(task->option_descriptors);
		}

		engine_free(task);
	}

	LOG_PROC_EXIT_VOID();
	return;
}


/*
 *
 *   static inline int GetTaskOptionsCount(task_context_t *)
 *
 *   Description:
 *      This routine calls the GetOptionsCount plug-in function to
 *      retrieve the maximum backen options available for a
 *      specific operation/action.
 *
 *   Entry:
 *      task - address of task context structure created by evms_create_task()
 *
 *   Exit:
 *      Returns count of maximum back end options for a specific task
 *
 */
static inline int GetTaskOptionsCount(task_context_t * task) {
	int count;

	LOG_PROC_ENTRY();

	switch (GetPluginType(task->plugin->id)) {
		case EVMS_DEVICE_MANAGER:
		case EVMS_SEGMENT_MANAGER:
		case EVMS_REGION_MANAGER:
		case EVMS_FEATURE:
		case EVMS_ASSOCIATIVE_FEATURE:
			count = task->plugin->functions.plugin->get_option_count(task);
			break;

		case EVMS_FILESYSTEM_INTERFACE_MODULE:
			count = task->plugin->functions.fsim->get_option_count(task);
			break;

		case EVMS_CLUSTER_MANAGER_INTERFACE_MODULE:
		case EVMS_DISTRIBUTED_LOCK_MANAGER_INTERFACE_MODULE:
		default:
			count = 0;
			break;
	}

	LOG_PROC_EXIT_INT(count);
	return count;
}


/*
 *
 *   static inline int InitTaskOptions(task_context_t *)
 *
 *   Description:
 *      This routine calls the InitOptions plug-in function to allow
 *      the back end to initialize the option descriptor and option
 *      values arrays.
 *
 *   Entry:
 *      task - address of task context structure created by evms_create_task()
 *
 *   Exit:
 *      On success, returns a return code of 0 and option values and
 *      option descriptor arrays are initialized.
 *      On error, returns an error code < 0
 *
 */
static inline int InitTaskOptions(task_context_t * task) {
	int rc = 0;

	LOG_PROC_ENTRY();

	switch (GetPluginType(task->plugin->id)) {
		case EVMS_DEVICE_MANAGER:
		case EVMS_SEGMENT_MANAGER:
		case EVMS_REGION_MANAGER:
		case EVMS_FEATURE:
		case EVMS_ASSOCIATIVE_FEATURE:
			rc = task->plugin->functions.plugin->init_task(task);
			break;

		case EVMS_FILESYSTEM_INTERFACE_MODULE:
			rc = task->plugin->functions.fsim->init_task(task);
			break;

		case EVMS_CLUSTER_MANAGER_INTERFACE_MODULE:
		case EVMS_DISTRIBUTED_LOCK_MANAGER_INTERFACE_MODULE:
		default:
			rc = 0;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int is_valid_expand_object(storage_object_t * obj) {

	int rc = 0;
	STATIC_LIST_DECL(expand_points);

	LOG_PROC_ENTRY();

	if (obj->volume != NULL) {

		rc = get_volume_expand_points(obj->volume, &expand_points);

	} else {
		storage_object_t * top_object = obj;
		uint parent_count = 0;

		parent_count = list_count(top_object->parent_objects);

		while (parent_count > 0) {
			top_object = first_thing(top_object->parent_objects, NULL);

			if (top_object != NULL) {
				parent_count = list_count(top_object->parent_objects);
			}
		}

		if (parent_count <= 1) {
			sector_count_t max_delta_size = -1;

			rc = get_object_expand_points(top_object, &max_delta_size, &expand_points);

		} else {
			if (parent_count > 1) {
				LOG_ERROR("Object %s cannot be expanded because it has multiple parents.\n", obj->name);
				rc = EINVAL;
			}
		}
	}

	if (rc == 0) {
		/*
		 * We have a list of expand points for the volume or top level object.
		 * Check if the object given appears in the list of expand points.
		 */
		list_element_t iter;
		expand_object_info_t * expand_point = NULL;
		
		LIST_FOR_EACH(&expand_points, iter, expand_point) {
			if (expand_point->object == obj) {
				break;
			}
		}

		if (expand_point != NULL) {
			rc = 0;

		} else {
			rc = EINVAL;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int is_valid_shrink_object(storage_object_t * obj) {

	int rc = 0;
	STATIC_LIST_DECL(shrink_points);

	LOG_PROC_ENTRY();

	if (obj->volume != NULL) {

		rc = get_volume_shrink_points(obj->volume, &shrink_points);

	} else {
		storage_object_t * top_object = obj;
		uint parent_count = 0;

		parent_count = list_count(top_object->parent_objects);

		while (parent_count > 0) {
			top_object = first_thing(top_object->parent_objects, NULL);

			if (top_object != NULL) {
				parent_count = list_count(top_object->parent_objects);
			}
		}

		if (parent_count <= 1) {
			sector_count_t max_delta_size = -1;

			rc = get_object_shrink_points(top_object, &max_delta_size, &shrink_points);

		} else {
			if (parent_count > 1) {
				LOG_ERROR("Object %s cannot be shrunk because it has multiple parents.\n", obj->name);
				rc = EINVAL;
			}
		}
	}

	if (rc == 0) {
		/*
		 * We have a list of shrink points for the volume or top level object.
		 * Check if the object given appears in the list of shrink points.
		 */
		list_element_t iter;
		expand_object_info_t * shrink_point = NULL;
		
		LIST_FOR_EACH(&shrink_points, iter, shrink_point) {
			if (shrink_point->object == obj) {
				break;
			}
		}

		if (shrink_point != NULL) {
			rc = 0;

		} else {
			rc = EINVAL;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int isa_valid_task(task_action_t           action,
			  void                  * object,
			  object_type_t           obj_type,
			  plugin_record_t     * * pplugin,
			  storage_object_t    * * pobj,
			  storage_container_t * * pcon,
			  logical_volume_t    * * pvol) {

	int rc = 0;

	plugin_record_t * plugin = NULL;
	storage_object_t * obj = NULL;
	storage_container_t * con = NULL;
	logical_volume_t * vol = NULL;

	LOG_PROC_ENTRY();

	/* Given the plug-in type, make sure the action is allowed. */
	switch (obj_type) {
		case PLUGIN:
			if ((action == EVMS_Task_Create) ||
			    (action == EVMS_Task_Create_Container) ||
			    (action == EVMS_Task_Assign_Plugin) ||
			    (action == EVMS_Task_mkfs) ||
			    (action == EVMS_Task_Add_Feature) ||
			    (action >= EVMS_Task_Plugin_Function)) {

				plugin = (plugin_record_t *) object;

			} else {
				LOG_ERROR("Command %d cannot be targeted at a plug-in.\n", action);
				rc = EINVAL;
			}
			break;

		case EVMS_OBJECT:
		case REGION:
		case SEGMENT:
		case DISK:
			if ((action == EVMS_Task_Set_Info) ||
			    (action == EVMS_Task_Expand) ||
			    (action == EVMS_Task_Shrink) ||
			    (action >= EVMS_Task_Plugin_Function)) {

				obj = (storage_object_t *) object;
				plugin = obj->plugin;

			} else {
				LOG_ERROR("Command %d cannot be targeted at an object.\n", action);
				rc = EINVAL;
			}
			break;

		case CONTAINER:
			if ((action == EVMS_Task_Expand_Container) ||
			    (action == EVMS_Task_Shrink_Container) ||
			    (action == EVMS_Task_Set_Container_Info) ||
			    (action >= EVMS_Task_Plugin_Function)) {

				con = (storage_container_t *) object;
				plugin = con->plugin;

			} else {
				LOG_ERROR("Command %d cannot be targeted at a container.\n", action);
				rc = EINVAL;
			}
			break;

		case VOLUME:
			if ((action == EVMS_Task_Expand) ||
			    (action == EVMS_Task_fsck) ||
			    (action == EVMS_Task_defrag) ||
			    (action == EVMS_Task_Set_Info) ||
			    (action >= EVMS_Task_Plugin_Function)) {

				vol = (logical_volume_t *) object;

				if (vol->file_system_manager != NULL) {
					plugin = vol->file_system_manager;

				} else {
					LOG_ERROR("Command %d cannot be executed on volume %s because the volume has no File System Interface Module assigned to it.\n", action, vol->name);
					rc = EINVAL;
				}

			} else {
				LOG_ERROR("Command %d cannot be targeted at a volume.\n", action);
				rc = EINVAL;
			}
			break;

		default:
			LOG_ERROR("A task cannot be created for an object of type %#x.\n", obj_type);
			rc = EINVAL;
			break;
	}

	if (rc == 0) {

		/* Given the action, make sure the object is allowed. */
		switch (action) {
			case EVMS_Task_Create:
				break;

			case EVMS_Task_Create_Container:
				break;

			case EVMS_Task_Assign_Plugin:
				break;

			case EVMS_Task_Set_Container_Info:
			case EVMS_Task_Expand_Container:
			case EVMS_Task_Shrink_Container:
				break;

			case EVMS_Task_Set_Info:
				break;

			case EVMS_Task_Expand:
				if (obj != NULL) {
					rc = is_valid_expand_object(obj);
				}
				break;

			case EVMS_Task_Shrink:
				rc = is_valid_shrink_object(obj);
				break;

			case EVMS_Task_Slide:
			case EVMS_Task_Move:
				/* We don't do slide and move yet. */
				rc = ENOSYS;
				break;

			case EVMS_Task_mkfs:
				break;

			case EVMS_Task_fsck:
				break;

			case EVMS_Task_defrag:
				break;

			case EVMS_Task_Message:
				break;

			case EVMS_Task_Add_Feature:
				break;

			default:
				if (action < EVMS_Task_Plugin_Function) {
					LOG_ERROR("%d is not a valid task action.\n", action);
					rc = EINVAL;
				}
				break;
		}
	}

	*pplugin = plugin;
	*pobj = obj;
	*pcon = con;
	*pvol = vol;

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 *
 *   int evms_create_task(plugin_handle_t, object_handle_t, task_action_t *, task_handle_t *)
 *
 *   Description:
 *      This routine creates the task context structure used to bind parameters
 *      and option data for a specific operation, e.g. CreateObject().
 *
 *   Entry:
 *      plugin             - handle of plug-in to communicate task to
 *      referential_object - handle of object used in decision of acceptable
 *                           parameters
 *      action             - operation/low-level API to eventually get invoked
 *      new_task_context   - address for task context handle to be returned
 *
 *   Exit:
 *      On success, returns a return code of 0 and *new_task_context is updated.
 *      On error, returns an error code < 0
 *
 */
int evms_create_task(engine_handle_t thing,
		     task_action_t   action,
		     task_handle_t * new_task_context) {
	int rc;
	void * object;
	object_type_t type;

	LOG_PROC_ENTRY();
	LOG_DEBUG("Request to create task with action %d: %s.\n", action, get_task_name(action));

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_create_task(thing, action, new_task_context);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = translate_handle(thing,
			      &object,
			      &type);

	if (rc == HANDLE_MANAGER_NO_ERROR) {
		plugin_record_t * plugin;
		storage_object_t * obj;
		storage_container_t * con;
		logical_volume_t * vol;

		rc = isa_valid_task(action, object, type,
				    &plugin, &obj, &con, &vol);

		if (rc == 0) {
			if (new_task_context != NULL) {
				task_context_t * task = engine_alloc(sizeof(task_context_t));

				if (task != NULL) {
					int size;
					int count;

					task->type = TASK;

					/*
					 * Initialize static members of the task context
					 * structure
					 */

					task->plugin = plugin;
					task->object = obj;
					task->container = con;
					task->volume = vol;
					task->action = action;

					/*
					 * Determine option count and allocate option
					 * descriptor and option value arrays.
					 */

					count = GetTaskOptionsCount(task);

					if (count > 1) {
						size = sizeof(option_desc_array_t) +
						       (sizeof(option_descriptor_t) * (count - 1));
					} else {
						size = sizeof(option_desc_array_t);
					}

					task->option_descriptors = engine_alloc(size);

					if (task->option_descriptors != NULL) {

						/* Initialize task lists. */
						if (action == EVMS_Task_Add_Feature) {
							if (GetPluginType(task->plugin->id) == EVMS_FEATURE) {
								list_element_t iter1;
								list_element_t iter2;
								logical_volume_t * vol;

								/*
								 * The Engine builds a list of volumes for the
								 * acceptable_objects list for
								 * EVMS_Task_Add_Feature.
								 */
								engine_get_volume_list(NULL, NULL, 0, &task->acceptable_objects);
								LIST_FOR_EACH_SAFE(task->acceptable_objects, iter1, iter2, vol) {
									if (vol->flags & VOLFLAG_COMPATIBILITY) {
										delete_element(iter1);
									}
								}
								task->min_selected_objects = 1;
								task->max_selected_objects = 1;

							} else {
								LOG_ERROR("Plug-in %s is not of type EVMS_FEATURE.\n", task->plugin->short_name);
								rc = EINVAL;
							}

						} else {
							task->acceptable_objects = allocate_list();
						}

						task->selected_objects = allocate_list();
						rc = InitTaskOptions(task);

						if (rc == 0) {
							rc = create_handle(task,
									   TASK,
									   new_task_context);

							if (rc < 0) {
								LOG_WARNING("create_handle() error!\n");
								FreeTaskMemory(task);
							}

						} else {
							LOG_WARNING("Error initializing options.\n");
							FreeTaskMemory(task);
						}

					} else {
						LOG_WARNING("Allocation of option descriptor array failed.\n");
						FreeTaskMemory(task);
						rc = ENOMEM;
					}

				} else {
					LOG_WARNING("Memory allocation of task_context_t failed.\n");
					rc = ENOMEM;
				}

			} else {
				LOG_ERROR("Address for context handle can not be NULL.\n");
				rc = EINVAL;
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * This function does the work for the EVMS_Task_Expand_Container task.
 * The code is broken out into a separate function to keep the switch
 * statement in evms_ivoke_task() tidy.
 */
static int do_expand_container_task(handle_array_t * selected_ha,
				    task_context_t * task,
				    option_array_t * option_array) {
	int rc = 0;
	int i;
	int final_rc = 0;

	LOG_PROC_ENTRY();


	/*
	 * evms_transfer() can only handle transferring one object at a time.
	 * Issue an evms_transfer() for each object.
	 */

	for (i = 0; i < selected_ha->count; i++) {
		rc = evms_transfer(selected_ha->handle[i],
				   task->plugin->app_handle,
				   task->container->app_handle,
				   option_array);

		if (rc != 0) {

			/* Save the first non-zero error code. */
			if (final_rc == 0) {
				final_rc = rc;
			}
		}
	}

	rc = final_rc;

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * This function does the work for the EVMS_Task_Shrink_Container task.
 * The code is broken out into a separate function to keep the switch
 * statement in evms_invoke_task() tidy.
 */
static int do_shrink_container_task(handle_array_t * selected_ha,
				    task_context_t * task,
				    option_array_t * option_array) {
	int rc = 0;
	int i;
	int final_rc = 0;

	LOG_PROC_ENTRY();


	/*
	 * evms_transfer() can only handle transferring one object at a time.
	 * Issue an evms_transfer() for each object.
	 */

	for (i = 0; i < selected_ha->count; i++) {
		rc = evms_transfer(selected_ha->handle[i],
				   0,
				   0,
				   option_array);

		if (rc != 0) {

			/* Save the first non-zero error code. */
			if (final_rc == 0) {
				final_rc = rc;
			}
		}
	}

	rc = final_rc;

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 *
 *   int evms_invoke_task(task_handle_t handle, handle_array_t * * resulting_objects)
 *
 *   Description:
 *      This routine uses the current option values and parameters in
 *      the task context to make the appropriate call to the low-level
 *      API for the task defined action/operation.
 *
 *   Entry:
 *      handle        - handle to task context created by evms_create_task()
 *      returned_data - pointer to any data returned as a result of
 *                      low-level API call
 *
 *   Exit:
 *      On success, returns a return code of 0 and possibly *returned_data
 *      is updated.
 *      On error, returns an error code < 0
 *
 */
int evms_invoke_task(task_handle_t      handle,
		     handle_array_t * * resulting_objects) {

	int rc, tmp_rc, i;
	void * object;
	object_type_t type;
	option_array_t * option_array;

	LOG_PROC_ENTRY();

	*resulting_objects = NULL;

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_invoke_task(handle, resulting_objects);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = translate_handle(handle,
			      &object,
			      &type);

	if (rc != HANDLE_MANAGER_NO_ERROR) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (type == TASK) {
		task_context_t * task = (task_context_t *) object;

		LOG_DEBUG("Request to invoke task with action %d: %s.\n", task->action, get_task_name(task->action));

		if (task->option_descriptors->count > 0) {
			option_array = create_option_array_from_descriptors(task->option_descriptors);
		} else {
			option_array = engine_alloc(sizeof(option_array_t));
		}

		if (option_array != NULL) {
			handle_array_t * selected_ha;

			rc = make_handle_array(task->selected_objects, &selected_ha);

			if (rc == 0) {
				switch (task->action) {
					case EVMS_Task_Create:

						rc = evms_create(task->plugin->app_handle,
								 selected_ha,
								 option_array,
								 resulting_objects);
						break;

					case EVMS_Task_Create_Container:
						{
							object_handle_t object_handle = 0;

							rc = evms_create_container(task->plugin->app_handle,
										   selected_ha,
										   option_array,
										   &object_handle);

							if (rc == 0) {
								if (resulting_objects != NULL) {
									*resulting_objects = alloc_app_struct(sizeof(handle_array_t), NULL);

									if (*resulting_objects != NULL) {
										(*resulting_objects)->count = 1;
										(*resulting_objects)->handle[0] = object_handle;

									} else {
										LOG_CRITICAL("Error getting memory for a return handle array.\n");
										rc = ENOMEM;
									}
								}
							}
						}
						break;

					case EVMS_Task_Assign_Plugin:

						rc = evms_assign(selected_ha->handle[0],
								 task->plugin->app_handle,
								 option_array);
						break;

					case EVMS_Task_Set_Container_Info:

						if (task->container != NULL) {
							rc = evms_set_info(task->container->app_handle,
									   option_array);
						} else {
							LOG_ERROR("Task has no target container.\n");
							rc = EINVAL;
						}
						break;

					case EVMS_Task_Expand_Container:

						rc = do_expand_container_task(selected_ha,
									      task,
									      option_array);
						break;

					case EVMS_Task_Shrink_Container:

						rc = do_shrink_container_task(selected_ha,
									      task,
									      option_array);
						break;

					case EVMS_Task_Set_Info:

						if (task->object != NULL) {
							rc = evms_set_info(task->object->app_handle,
									   option_array);
						} else if (task->volume != NULL) {
							rc = evms_set_info(task->volume->app_handle,
									   option_array);
						} else {
							LOG_ERROR("Task has no target object, nor volume.\n");
							rc = EINVAL;
						}
						break;

					case EVMS_Task_Expand:

						if (task->object != NULL) {
							rc = evms_expand(task->object->app_handle,
									 selected_ha,
									 option_array);

						} else if (task->volume!= NULL) {
							rc = evms_expand(task->volume->app_handle,
									 selected_ha,
									 option_array);
						
						} else {
							rc = EINVAL;
						}
						break;

					case EVMS_Task_Shrink:

						rc = evms_shrink(task->object->app_handle,
								 selected_ha,
								 option_array);
						break;

					case EVMS_Task_mkfs:

						/* The FSIM may allow multiple selections of the
						 * volumes to be mkfs'd, but the evms_mkfs() API
						 * only takes a single volume.  Run through the
						 * list of selected volumes and issue an
						 * evms_mkfs() for it.
						 */
						for (i = 0; i < selected_ha->count; i++) {
							tmp_rc = evms_mkfs(selected_ha->handle[i],
									   task->plugin->app_handle,
									   option_array);

							if ((tmp_rc != 0) && (rc == 0)) {
								rc = tmp_rc;
							}
						}
						break;

					case EVMS_Task_fsck:

						rc = evms_fsck(task->volume->app_handle,
							       option_array);
						break;

					case EVMS_Task_defrag:

						rc = evms_defrag(task->volume->app_handle,
								 option_array);
						break;

					case EVMS_Task_Slide:
					case EVMS_Task_Move:
						LOG_WARNING("Task action %d is not supported yet.\n", task->action);
						break;

					case EVMS_Task_Add_Feature:

						rc = evms_add_feature_to_volume(selected_ha->handle[0],
										task->plugin->app_handle,
										option_array);
						break;

					default:
						if (task->action >= EVMS_Task_Plugin_Function) {
							if (task->object != NULL) {
								rc = evms_do_plugin_function(task->object->app_handle,
											     task->action,
											     selected_ha,
											     option_array);

							} else if (task->container != NULL) {
								rc = evms_do_plugin_function(task->container->app_handle,
											     task->action,
											     selected_ha,
											     option_array);

							} else if (task->volume != NULL) {
								rc = evms_do_plugin_function(task->volume->app_handle,
											     task->action,
											     selected_ha,
											     option_array);
							} else {
								rc = evms_do_plugin_function(task->plugin->app_handle,
											     task->action,
											     selected_ha,
											     option_array);
							}

						} else {
							LOG_ERROR("Unknown task action %d.\n", task->action);
							rc = EINVAL;
						}
						break;
				}
			}

			free_option_array(option_array);

		} else {
			LOG_CRITICAL("Error allocating memory for building the option array.\n");
			rc = ENOMEM;
		}

	} else {
		LOG_ERROR("%d is not a task context handle.\n", handle);
		rc = EINVAL;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 *
 *   int evms_destroy_task(task_handle_t)
 *
 *   Description:
 *      This routine frees up all resources/memory associated with the
 *      task handle and invalidates the task handle upon successful
 *      completion.
 *
 *   Entry:
 *      handle - handle to task context created by evms_create_task()
 *
 *   Exit:
 *      On success, returns a return code of 0 and task context is no more
 *      On error, returns an error code < 0
 *
 */
int evms_destroy_task(task_handle_t handle) {
	int rc;
	void * object;
	object_type_t type;

	/*
	 * Question: What mechanism do we have to avoid
	 * race conditions between two or more threads
	 * calling us with the same handle?
	 */

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_destroy_task(handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = translate_handle(handle,
			      &object,
			      &type);

	if (rc == HANDLE_MANAGER_NO_ERROR) {

		if (type == TASK) {
			task_context_t * task = (task_context_t *) object;

			LOG_DEBUG("Request to destroy task with action %d: %s.\n", task->action, get_task_name(task->action));

			/*
			 * Empty out the parameter list and deallocate all
			 * memory, including the task_context_t and finally
			 * destroy the task context handle.
			 */

			FreeTaskMemory(task);

			rc = destroy_handle(handle);

		} else {
			LOG_ERROR("%d is not a task context handle.\n", handle);
			rc = EINVAL;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 *
 *   int evms_get_task_action (task_handle_t, task_action_t *)
 *
 *   Description:
 *      This routine
 *
 *   Entry:
 *      handle - handle to task context created by evms_create_task()
 *      action -
 *
 *   Exit:
 *      On success, returns a return code of 0 and action is returned
 *      On error, returns an error code != 0
 *
 */
int evms_get_task_action(task_handle_t   handle,
			 task_action_t * action) {
	int rc;
	void * object;
	object_type_t type;

	LOG_PROC_ENTRY();

	rc = check_engine_read_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_get_task_action(handle, action);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = translate_handle(handle,
			      &object,
			      &type);

	if (rc == HANDLE_MANAGER_NO_ERROR) {

		if (type == TASK) {
			task_context_t * task = (task_context_t *) object;

			LOG_DEBUG("Task action is %d: %s.\n", task->action, get_task_name(task->action));

			if (action != NULL)
				*action = task->action;
			else
				rc = EINVAL;

		} else {
			LOG_ERROR("%d is not a task context handle.\n", handle);
			rc = EINVAL;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Add the declined_handle for a given object to a declined_handle_array_t.
 */
static int make_declined_handle_entry(declined_object_t       * pDecObj,
				      declined_handle_array_t * dha) {
	int rc = 0;
	storage_object_t * pObj = pDecObj->object;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Number of entries in declined handle array:  %d.\n", dha->count);

	rc = ensure_app_handle(pObj);
	if (rc == 0) {
		dha->declined[dha->count].handle = pObj->app_handle;
		dha->declined[dha->count].reason = pDecObj->reason;
		dha->count++;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Make a user (it has a Engine memory block header) array of handles
 * (handle_array_t) for the objects in a list_anchor_t.
 */
static int make_user_declined_handle_array(list_anchor_t               declined_objects_list,
					   declined_handle_array_t * * dha) {

	int rc = 0;
	uint count;
	uint size;

	LOG_PROC_ENTRY();

	count = list_count(declined_objects_list);

	LOG_DEBUG("Number of objects in the list:  %d\n", count);
	if (count > 1) {
		size = sizeof(declined_handle_array_t) + ((count -1) * sizeof(declined_handle_t));
	} else {
		size = sizeof(declined_handle_array_t);
	}

	*dha = alloc_app_struct(size, NULL);
	if (*dha != NULL) {
		list_element_t iter;
		declined_object_t * pdo;

		LIST_FOR_EACH(declined_objects_list, iter, pdo) {
			make_declined_handle_entry(pdo, *dha);
		}
	} else {
		rc = ENOMEM;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 *
 *   int evms_get_acceptable_objects(task_handle_t, handle_array_t **)
 *
 *   Description:
 *      This routine retrieves handles to objects, e.g. regions,
 *      segments, containers, etc., deemed "acceptable" to the
 *      back end for the specified task context.
 *
 *   Entry:
 *      handle          - handle to task context created by evms_create_task()
 *      acceptable_list - address of pointer to contain acceptable parameter
 *                        handles
 *
 *   Exit:
 *      On success, returns a return code of 0 and **acceptable_list is updated.
 *      On error, returns an error code < 0
 *
 */
int evms_get_acceptable_objects(task_handle_t      handle,
				handle_array_t * * acceptable_object_list) {
	int rc;
	void * object;
	object_type_t type;

	LOG_PROC_ENTRY();

	rc = check_engine_read_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_get_acceptable_objects(handle, acceptable_object_list);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	/* The caller must provide an acceptable_object_list. */
	if (acceptable_object_list != NULL) {

		*acceptable_object_list = NULL;

		rc = translate_handle(handle,
				      &object,
				      &type);

		if (rc == HANDLE_MANAGER_NO_ERROR) {

			if (type == TASK) {
				task_context_t * task = (task_context_t *) object;

				LOG_DEBUG("Get acceptable objects for task with action %d: %s.\n", task->action, get_task_name(task->action));

				rc = make_user_handle_array(task->acceptable_objects, acceptable_object_list);

			} else {
				LOG_ERROR("The handle given is not for a task context.\n");
				rc = EINVAL;
			}
		} else {
			LOG_ERROR("translate_handle() could not find the task context for handle %d.\n", handle);
			rc = EINVAL;
		}

	} else {
		LOG_ERROR("The pointer to the acceptable objects list cannot be NULL.\n");
		rc = EINVAL;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 *
 *   int evms_get_selected_objects(task_handle_t, handle_array_t **)
 *
 *   Description:
 *      This routine retrieves handles to objects, e.g. regions,
 *      segments, containers, etc., that are currently selected by the user.
 *
 *   Entry:
 *      handle          - handle to task context created by evms_create_task()
 *      acceptable_list - address of pointer to contain selected object
 *                        handles
 *
 *   Exit:
 *      On success, returns a return code of 0 and **acceptable_list is updated.
 *      On error, returns an error code < 0
 *
 */
int evms_get_selected_objects(task_handle_t      handle,
			      handle_array_t * * selected_object_list) {
	int rc;
	void * object;
	object_type_t type;

	LOG_PROC_ENTRY();

	rc = check_engine_read_access();

	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_get_selected_objects(handle, selected_object_list);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	/* The caller must provide a selected_object_list. */
	if (selected_object_list != NULL) {

		*selected_object_list = NULL;

		rc = translate_handle(handle,
				      &object,
				      &type);

		if (rc == HANDLE_MANAGER_NO_ERROR) {

			if (type == TASK) {
				task_context_t * task = (task_context_t *) object;

				LOG_DEBUG("Get selected objects for task with action %d: %s.\n", task->action, get_task_name(task->action));

				rc = make_user_handle_array(task->selected_objects, selected_object_list);

			} else {
				LOG_ERROR("The handle given is not for a task context.\n");
				rc = EINVAL;
			}
		} else {
			LOG_ERROR("translate_handle() could not find the task context for handle %d.\n", handle);
			rc = EINVAL;
		}

	} else {
		LOG_ERROR("The pointer to the selected objects list cannot be NULL.\n");
		rc = EINVAL;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 *
 *   int evms_get_selected_object_limits(task_handle_t, u_int32_t *, u_int32_t *)
 *
 *   Description:
 *      This routine retrieves limits, minimum and maximum, for the number of
 *      objects that the user can select.
 *
 *   Entry:
 *      handle          - handle to task context created by evms_create_task()
 *      pMininum        - address of where to put the minimum
 *      pMaximum        - address of where to put the maximum
 *
 *   Exit:
 *      On success, returns a return code of 0 and *minimum and *maximum are
 *      filled in.
 *      On error, returns an error code < 0
 *
 */
int evms_get_selected_object_limits(task_handle_t handle,
				    u_int32_t   * pMinimum,
				    u_int32_t   * pMaximum) {
	int rc;
	void * object;
	object_type_t type;

	LOG_PROC_ENTRY();

	rc = check_engine_read_access();

	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_get_selected_object_limits(handle, pMinimum, pMaximum);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	/* The caller must provide valid pointers for the limits */
	if ((pMinimum != NULL) && (pMaximum != NULL)) {

		*pMinimum = 0;
		*pMaximum = 0;

		rc = translate_handle(handle,
				      &object,
				      &type);

		if (rc == HANDLE_MANAGER_NO_ERROR) {

			if (type == TASK) {
				task_context_t * task = (task_context_t *) object;

				LOG_DEBUG("Get selected object limits for task with action %d: %s.\n", task->action, get_task_name(task->action));

				*pMinimum = task->min_selected_objects;
				*pMaximum = task->max_selected_objects;

			} else {
				LOG_ERROR("The handle given is not for a task context.\n");
				rc = EINVAL;
			}
		} else {
			LOG_ERROR("translate_handle() could not find the task context for handle %d.\n", handle);
			rc = EINVAL;
		}

	} else {
		if (pMinimum == NULL) {
			LOG_ERROR("The address for the minimum value cannot be NULL.\n");
		}
		if (pMaximum == NULL) {
			LOG_ERROR("The address for the maximum value cannot be NULL.\n");
		}
		rc = EINVAL;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int set_volume_for_add_feature(task_context_t * task,
				      list_anchor_t    declined_objects,
				      task_effect_t  * effect) {
	int rc = 0;

	LOG_PROC_ENTRY();

	if (list_empty(task->selected_objects)) {
		LOG_ERROR("Must select a volume for EVMS_Task_Add_Feature.\n");
		rc = EINVAL;

	} else {
		list_element_t iter1;
		list_element_t iter2;
		common_header_t * thing;
		boolean found_volume = FALSE;

		LIST_FOR_EACH_SAFE(task->selected_objects, iter1, iter2, thing) {

			if ((thing->type == VOLUME) && !found_volume) {
				found_volume = TRUE;
			} else {
				declined_object_t * declined_object;

				switch (thing->type) {
					case VOLUME:
						LOG_ERROR("Only one volume can be specified.  Volume %s is declined.\n", ((logical_volume_t *) thing)->name);
						break;
					case EVMS_OBJECT:
					case REGION:
					case SEGMENT:
					case DISK:
						LOG_ERROR("Object %s is not a volume.\n", ((storage_object_t *) thing)->name);
						break;
					case PLUGIN:
						LOG_ERROR("Plug-in %s is not a volume.\n", ((plugin_record_t *) thing)->short_name);
						break;
					default:
						LOG_ERROR("Object is not a volume.\n");
						break;
				}

				declined_object = engine_alloc(sizeof(*declined_object));

				if (declined_object != NULL) {
					list_element_t el;

					declined_object->object = thing;
					declined_object->reason = EINVAL;

					el = insert_thing(declined_objects,
							  declined_object,
							  INSERT_AFTER,
							  NULL);

					if (el != NULL) {
						delete_element(iter1);
					}

				} else {
					LOG_CRITICAL("Error allocating memory for a declined object.\n");
					rc = ENOMEM;
				}
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * boolean same_disk_group(list_anchor_t           list,
 *                         storage_container_t * * p_disk_group,
 *                         list_anchor_t           declined_objects,
 *                         task_effect_t         * effect)
 *
 * Description:
 *      This is a helper function for validate_selected_objects() below.  This
 *      function is called to fileter out objects that are not in the same disk
 *      goup.  If an object is not in the same disk group, it creates a
 *      declined_object_t for the object and puts the declined_object_t on the
 *      declined_object list.
 *
 * Entry:
 *     list            	list to filter
 *     p_disk_group     pointer to the disk group the things should be in.
 *     declined_objects where to put the rejected things
 *     effect		the EVMS_Effect_Reload_Objects flag is set in *effect
 *      		if the list was changed.
 *
 * Exit:
 *      EINVAL if one of the list things is bogus.
 */

static int filter_same_disk_group(list_anchor_t           list,
				  storage_container_t * * p_disk_group,
				  list_anchor_t           declined_objects,
				  task_effect_t         * effect) {

	storage_container_t * thing_disk_group = NULL;
	common_header_t * thing;
	char * thing_name;
	list_element_t iter1;
	list_element_t iter2;
	storage_container_t * disk_group = *p_disk_group;

	LOG_PROC_ENTRY();

	LIST_FOR_EACH_SAFE(list, iter1, iter2, thing) {

		switch (thing->type) {
			case DISK:
			case SEGMENT:
			case REGION:
			case EVMS_OBJECT:
				{
					storage_object_t * obj = (storage_object_t *) thing;

					LOG_DEBUG("Initialize disk group to %s.\n", (obj->disk_group != NULL) ? obj->disk_group->name : "(local)");
					thing_disk_group = obj->disk_group;
					thing_name = obj->name;
				}
				break;
			case VOLUME:
				{
					logical_volume_t * vol = (logical_volume_t *) thing;

					LOG_DEBUG("Initialize disk group to %s.\n", (vol->disk_group != NULL) ? vol->disk_group->name : "(local)");
					thing_disk_group = vol->disk_group;
					thing_name = vol->name;
				}
				break;
			default:
				LOG_SERIOUS("Things of type %#x don't have a disk group.\n", thing->type);
				/* Remove it from the list. */
				delete_element(iter1);
				LOG_PROC_EXIT_INT(EINVAL);
				return EINVAL;
		}

		if (disk_group == (storage_container_t *) -1) {
			/* Unspecified disk group. */
			/* First object determines the disk group. */
			disk_group = thing_disk_group;

		} else {
			if (thing_disk_group != disk_group) {
				LOG_DEBUG("%s is in disk group %s, not in disk group %s.\n",
					  thing_name,
					  (thing_disk_group != NULL) ? thing_disk_group->name : "(local)",
					  (disk_group != NULL) ? disk_group->name : "(local)");

				*effect |= EVMS_Effect_Reload_Objects;

				if (declined_objects != NULL) {
					declined_object_t * declined_object = engine_alloc(sizeof(declined_object_t));

					if (declined_object != NULL) {
						list_element_t el;

						declined_object->reason = EINVAL;
						declined_object->object = thing;

						el = insert_thing(declined_objects,
								  declined_object,
								  INSERT_AFTER,
								  NULL);

						if (el == NULL) {
							LOG_CRITICAL("Error inserting declined object into declined_objects list.\n");
						}

					} else {
						LOG_CRITICAL("Error allocating memory for a declined object.\n");
					}
				}

				delete_element(iter1);
			}
		}
	}

	*p_disk_group = disk_group;

	LOG_PROC_EXIT_INT(0);
	return 0;
}



/*
 * boolean is_acceptable_object(common_engine_header_t * thing,
 *                              list_anchor_t            acceptable_objects,
 *                              list_anchor_t            declined_objects)
 *
 * Description:
 *      This is a helper function for validate_selected_objects() below.  to
 *      process a list of selected_objects.  This function checks to see if the
 *      object is in the acceptable_objects list.  If not, it creates a
 *      declined_object_t for the object and puts the declined_object_t on the
 *      declined_object list.
 *
 * Entry:
 *      thing,             - thing to examine
 *      acceptable_objects - list it must be on
 *      declined_objects   - where to put it if it's not in the list
 *
 * Exit:
 *      TRUE   - thing is in the acceptable_objects list
 *      FALSE  - thing is not in the acceptable_objects list
 */

static boolean is_acceptable_object(common_header_t * thing,
				    list_anchor_t     acceptable_objects,
				    list_anchor_t     declined_objects) {
	boolean result = TRUE;
	int rc = 0;
	list_element_t el;

	LOG_PROC_ENTRY();

	/* Check if the object is in the acceptable_objects list. */
	el = find_in_list(acceptable_objects, thing, NULL);

	/*
	 * If the object is not found in the acceptable_object list, build a
	 * declined object for it.
	 */
	if (el == NULL) {
		declined_object_t * declined_object = engine_alloc(sizeof(declined_object_t));

		if (declined_object != NULL) {

			declined_object->object = thing;
			declined_object->reason = EINVAL;

			el = insert_thing(declined_objects,
					  declined_object,
					  INSERT_AFTER,
					  NULL);

			if (el == NULL) {
				LOG_CRITICAL("Error %d inserting declined object into declined_objects list.\n", rc);
			}

		} else {
			LOG_CRITICAL("Error allocating memory for a declined object.\n");
		}

		result = FALSE;
	}

	LOG_PROC_EXIT_BOOLEAN(result);
	return result;

}


/*
 * int validate_selected_objects(task_context_t *, list_anchor_t, task_effect *)
 *
 * Description:
 *      This is a helper function for evms_set_selected_objects() below.
 *      It makes sure that all the objects in the selected_objects list in
 *      the task context appear in the acceptable_objects list in the task
 *      context.  Any selected objects that do not appear in the
 *      acceptable_objects list are removed from the selected_objects list.
 *      A declined_object_t is created for the removed object and is placed
 *      on the declined_objects list.
 *
 * Entry:
 *      task             - a pointer to a task context
 *      declined_objects - a list for declined objects
 *      effect           - where to set any effect flags
 *
 * Exit:
 *      0        - All selected objects are in the acceptable_objects list
 *      EINVAL   - One or more selected objects is not in the acceptable_objects
 *                 list
 *      error    - some other error occurred during processing
 */
static int validate_selected_objects(task_context_t * task,
				     list_anchor_t    declined_objects,
				     task_effect_t *  effect) {

	int rc = 0;

	storage_container_t * disk_group = (storage_container_t *) -1;

	list_element_t iter1;
	list_element_t iter2;
	common_header_t * thing;

	LOG_PROC_ENTRY();

	rc = filter_same_disk_group(task->selected_objects, &disk_group, declined_objects, effect);

	if (rc != 0) {
		/* Unexpected error.  Abort. */
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	/*
	 * Filter the acceptable_objects list to make sure they are only things
	 * in the same disk group.  disk_group contains the disk group of the
	 * first selected thing, left over from the above call to
	 * filter_same_disk_group().  Pass a declined_objects list of NULL so
	 * that filter_same_disk_group() will not put any rejected objects on
	 * the declined objects list.
	 */
	rc = filter_same_disk_group(task->acceptable_objects, &disk_group, NULL, effect);

	if (rc != 0) {
		/* Unexpected error.  Abort. */
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	LIST_FOR_EACH_SAFE(task->selected_objects, iter1, iter2, thing) {
		/*
		 * is_acceptable_object() will put any unacceptable objects
		 * on the delined_objects_list.
		 */
		is_acceptable_object(thing, task->acceptable_objects, declined_objects);
	}

	/*
	 * To determine if any selected objects were declined, check if there
	 * are any objects on the declined_objects list.
	 */
	if (!list_empty(declined_objects)) {
		rc = EINVAL;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 *
 *   int evms_set_selected_objects(task_handle_t,
 *                                 handle_array_t *,
 *                                 declined_handle_array_t **,
 *                                 task_effect_t)
 *
 *   Description:
 *      This routine allows the application to associate/set the choice of
 *      objects from the initial acceptable objects. If the choices
 *      are not acceptable to the back end, the handles of the declined
 *      objects and reason codes are returned in the declined_list.
 *
 *   Entry:
 *      handle               - handle to task context created by evms_create_task()
 *      selected_object_list - address of handle_array_t containing selected
 *                             objects
 *      declined_list        - address of pointer to a declined_handle_array_t
 *                             containing any declined handles and reason codes
 *      effect               - flags indicating side effects from this command,
 *                             e.g., object list(s) changed, options changed
 *
 *   Exit:
 *      On success, returns a return code of 0 and possibly
 *      **declined_list and *effect are updated.
 *      On error, returns an error code < 0
 *
 */
int evms_set_selected_objects(task_handle_t               handle,
			      handle_array_t            * selected_object_list,
			      declined_handle_array_t * * declined_list,
			      task_effect_t             * effect) {
	int rc;
	void * object;
	task_context_t * task;
	object_type_t type;
	list_anchor_t declined_objects;
	task_effect_t local_effect;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_set_selected_objects(handle, selected_object_list, declined_list, effect);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	/* The caller must provide a selected_object_list. */
	if (selected_object_list == NULL) {
		LOG_ERROR("The pointer to the source list cannot be NULL.\n");
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}
	rc = translate_handle(handle,
			      &object,
			      &type);

	if (rc != HANDLE_MANAGER_NO_ERROR) {
		LOG_ERROR("translate_handle() could not find the task context for handle %d.\n", handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (type != TASK) {
		LOG_ERROR("The handle given is not for a task context.\n");
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	task = (task_context_t *) object;

	LOG_DEBUG("Set selected objects for task with action %d: %s.\n", task->action, get_task_name(task->action));

	/* Reset the lists in the context record */
	delete_all_elements(task->selected_objects);

	make_list(selected_object_list, task->selected_objects);
	declined_objects = allocate_list();

	if (declined_objects == NULL) {
		LOG_CRITICAL("Error allocating memory for the declined objects list.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	/*
	 * If the user did not specify an address for the effect, point the
	 * effect to a local variable so that other code does not have to
	 * worry about handling a NULL pointer.
	 */
	if (effect == NULL) {
		effect = &local_effect;
	}

	*effect = 0;

	/*
	 * If the caller selected no objects, re-initialize the task.
	 * Previous calls to evms_set_selected_objects() may have trimmed
	 * the acceptable_objects list to contain only objects in the same
	 * disk group.  With no objects selected, the acceptable_objects
	 * list should be reset to all valid accptable objects.
	 */
	if (list_empty(task->selected_objects)) {
		LOG_DEBUG("No objects selected.  Reinitialize the task.\n");
		delete_all_elements(task->acceptable_objects);
		InitTaskOptions(task);

		*effect = EVMS_Effect_Reload_Options | EVMS_Effect_Reload_Objects;

	} else {
		/*
		 * Do some pre-validation for the plug-in.
		 * Make sure all of the selected objects appear in
		 * the acceptable_objects list.
		 */
		rc = validate_selected_objects(task, declined_objects, effect);

		if (rc == 0) {

			/*
			 * If the number of selected objects is below the minimum
			 * needed, don't bother calling the plug-in, because it
			 * will just fail the request anyway.
			 */
			uint object_count;

			object_count = list_count(task->selected_objects);

			if (object_count >= task->min_selected_objects) {
				list_element_t iter;
				declined_object_t * declined_object;

				evms_plugin_code_t plugin_type = GetPluginType(task->plugin->id);


				switch (task->action) {
					case EVMS_Task_Add_Feature:
						rc = set_volume_for_add_feature(task, declined_objects, effect);
						break;

					default:
						switch (plugin_type) {
							case EVMS_DEVICE_MANAGER:
							case EVMS_SEGMENT_MANAGER:
							case EVMS_REGION_MANAGER:
							case EVMS_FEATURE:
							case EVMS_ASSOCIATIVE_FEATURE:
								rc = task->plugin->functions.plugin->set_objects(task, declined_objects, effect);
								break;

							case EVMS_FILESYSTEM_INTERFACE_MODULE:
								rc = task->plugin->functions.fsim->set_volumes(task, declined_objects, effect);
								break;

							default:
								rc = ENOSYS;
						}
						break;
				}

				/*
				 * Remove any declined objects from the
				 * selected_objects list in the task context,
				 * just in case the plug-in didn't do it.
				 */
				LIST_FOR_EACH(declined_objects, iter, declined_object) {
					remove_thing(task->selected_objects, declined_object->object);
				}
			}
		}

		/* Does the user want a declined handle array? */
		if (declined_list != NULL) {

			/*
			 * Make a declined handle array only if we have
			 * declined objects.
			 */
			if (!list_empty(declined_objects)) {
				int err;

				err = make_user_declined_handle_array(declined_objects, declined_list);

				/*
				 * If we don't already have a bad error
				 * code to return, return any error from
				 * make_user_declined_handle_array().
				 */
				if (rc == 0) {
					rc = err;
				}

			} else {
				*declined_list = NULL;
			}
		}
	}

	destroy_list(declined_objects);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

