/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003, 2004
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: flatten.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <errno.h>

#include "fullengine.h"
#include "dmonmsg.h"
#include "engine.h"
#include "memman.h"


/*
 * Utility functions
 */

static char * valid_format_characters = "bcCdeghiIklopstvx[]{}";

static int validate_format(char * format) {

	int rc = 0;
	char * pch = format;
	int i, j;
	char stack[32];

	LOG_PROC_ENTRY();

	while (*pch != '\0') {
		char * p;

		p = strchr(valid_format_characters, *pch);
		if (p == NULL) {
			LOG_ERROR("Invalid format character %c at offset %d in format string \"%s\".\n", *pch, (int)(pch - format), format);
			rc = EINVAL;
			pch++;

		} else {
			switch (*pch) {
				case 'c':
				case 'C':
					/* The next character must be a '['. */
					if (*(pch + 1) != '[') {
						LOG_ERROR("A bracket ('[') is required after the %c at offset %d in the format string \"%s\".\n", *pch, (int)(pch - format), format);
						pch++;

					} else {
						/*
						 * Skip over the 'c' or 'C' and
						 * the following bracket.
						 */
						pch +=2;
					}
					break;

				case 'x':
					{
						/* Skip over the digits that follow. */
						int num_digits;

						pch++;
						num_digits = strspn(pch, "0123456789");
						pch += num_digits;
					}
					break;

				default:
					pch++;
			}
		}
	}

	/*
	 * Make sure the number of open brackets/braces matches the number of
	 * close brackets/braces and that they are nested properly.
	 */
	pch = format;
	i = 0;
	while (*pch != '\0') {
		switch (*pch) {
			case '[':
				stack[i] = ']';
				i++;
				break;

			case ']':
				if (i <= 0) {
					LOG_ERROR("Format \"%s\" has a close bracket (']') at offset %d without a matching open bracket ('[).\n",
						  format, (int)(pch - format));
					rc = EINVAL;

				} else {
					if (stack[i-1] != ']') {
						LOG_ERROR("Expecting a '%c' before the close bracket (']') at offset %d in format \"%s\".\n",
							  stack[i-1], (int)(pch - format), format);
						rc = EINVAL;

					} else {
						i--;
					}
				}
				break;

			case '{':
				stack[i] = '}';
				i++;
				break;

			case '}':
				if (i <= 0) {
					LOG_ERROR("Format \"%s\" has a close brace ('}') at offset %d without a matching open brace ('{').\n",
						  format, (int)(pch - format));
					rc = EINVAL;

				} else {
					if (stack[i-1] != '}') {
						LOG_ERROR("Expecting a '%c' before the close brace ('}') at offset %d in format \"%s\".\n",
							  stack[i-1], (int)(pch - format), format);
						rc = EINVAL;

					} else {
						i--;
					}
				}
				break;
		}
		pch++;
	}

	if (i > 0) {
		for (j = 0; j < i; j++) {
			LOG_ERROR("Format \"%s\" is missing a '%c'.\n", format, stack[j]);
		}
		rc = EINVAL;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static char * closing_bracket(char * format) {

	char * pch;
	uint bracket_count;

	for (pch = format, bracket_count = 0;
	    (*pch != '\0') && ((*pch != ']') || (bracket_count != 0));
	    pch++) {
		if (*pch == '[') {
			bracket_count ++;

		} else if (*pch == ']') {
			bracket_count--;
		}
	}

	return pch;
}


static char * closing_brace(char * format) {

	char * pch;
	uint brace_count;

	for (pch = format, brace_count = 0;
	    (*pch != '\0') && ((*pch != '}') || (brace_count != 0));
	    pch++) {
		if (*pch == '{') {
			brace_count ++;

		} else if (*pch == '}') {
			brace_count--;
		}
	}

	return pch;
}


/*
 * Functions for calculating the size needed for a buffer to contain
 * the network stream for given data structures.
 */

static size_t sizeof_netbuf_value(value_t value, value_type_t type, boolean value_is_list) {

	size_t size = sizeof(u_int8_t);	/* boolean that says if the value is a list */
	uint count;
	value_t * p_value;
	int i;

	LOG_PROC_ENTRY();

	if (value_is_list) {
		/* First byte says if the list pointer is not NULL. */
		size += sizeof(u_int8_t);

		if (value.list != NULL) {
			value_list_t * vl = (value_list_t *) value.list;
			count = vl->count;
			p_value = &vl->value[0];
			size += sizeof(u_int32_t);	/* Room for the count */

		} else {
			/* That's all folks. */

			LOG_PROC_EXIT_INT((int)size);
			return size;
		}

	} else {
		count = 1;
		p_value = &value;
	}

	for (i = 0; i < count; i++, p_value++) {
		switch (type) {
			case EVMS_Type_String:
				/*
				 * One for the indicator that says if the string pointer
				 * is NULL.
				 */
				size += sizeof(u_int8_t);

				if ((*p_value).s != NULL) {
					size +=  strlen(value.s) + sizeof(char);
				}
				break;

			case EVMS_Type_Boolean:
			case EVMS_Type_Char:
			case EVMS_Type_Unsigned_Char:
			case EVMS_Type_Int8:
			case EVMS_Type_Unsigned_Int8:
				size += sizeof(u_int8_t);
				break;

			case EVMS_Type_Int16:
			case EVMS_Type_Unsigned_Int16:
				size += sizeof(u_int16_t);
				break;

			case EVMS_Type_Real32:
			case EVMS_Type_Int:
			case EVMS_Type_Int32:
			case EVMS_Type_Unsigned_Int:
			case EVMS_Type_Unsigned_Int32:
				size += sizeof(u_int32_t);
				break;

			case EVMS_Type_Real64:
			case EVMS_Type_Int64:
			case EVMS_Type_Unsigned_Int64:
				size += sizeof(u_int64_t);
				break;
		}
	}

	LOG_PROC_EXIT_INT((int)size);
	return size;
}


static size_t sizeof_netbuf_collection(value_collection_t coll,
				       collection_type_t  coll_type,
				       value_type_t       value_type) {

	size_t size = 0;
	int i;

	LOG_PROC_ENTRY();

	switch (coll_type) {
		case EVMS_Collection_None:
			break;

		case EVMS_Collection_List:
			size += sizeof(u_int32_t);	/* od->constraint.list->count */
			if (value_type == EVMS_Type_String) {
				for (i = 0; i < coll.list->count; i++) {
					size += sizeof_netbuf_value(coll.list->value[i], value_type, FALSE);
				}

			} else {
				size += coll.list->count * sizeof_netbuf_value(coll.list->value[0], value_type, FALSE);
			}
			break;

		case EVMS_Collection_Range:
			/*
			 * Don't have to worry about the value being of type
			 * EVMS_Type_String.  String cannot be used as ranges.
			 */
			size += sizeof_netbuf_value(coll.range->min, value_type, FALSE)
				* 3;  /* coll->range->min       */
				      /* coll->range->max       */
				      /* coll->range->increment */
			break;
	}

	LOG_PROC_EXIT_INT((int)size);
	return size;
}


static void * sizeof_hostbuf_to_netbuf(char * format,
				       void * host_buf,
				       size_t * psize) {

	char * pf = format;
	char curr_f;
	size_t size = 0;
	size_t curr_size;
	u_int32_t count;
	int i;
	char * pch;
	option_descriptor_t * od;
	extended_info_t * ei;
	key_value_pair_t * kvp;
	char * string;
	char * * strings;
	char tmp_format[64];
	handle_object_info_t * hoi;
	uint len;
	boolean value_is_list;
	value_type_t value_type;

	LOG_PROC_ENTRY();

	LOG_EXTRA("Format is: %s\n", format);

	while (*pf != '\0') {
		curr_size = size;
		curr_f = *pf;

		switch (*pf) {
			case 'b':
				size += sizeof(boolean);
				pf++;
				break;

			case 'c':
				/* First is the int of the count. */
				count = (u_int32_t) *((int *) host_buf);

				size += sizeof(u_int32_t);
				host_buf += sizeof(int);

				/* Skip over "c[". */
				pf += 2;

				/* Parse out the substring between the brackets. */
				strcpy(tmp_format, pf);
				pch = closing_bracket(tmp_format);
				*pch = '\0';

				/* tmp_format now has the substring between the brackets. */

				for (i = 0; i < count; i++) {
					/* call myself recursively to process the substring. */
					host_buf = sizeof_hostbuf_to_netbuf(tmp_format, host_buf, &size);
				}

				pf += strlen(tmp_format) + sizeof(char);
				break;

			case 'C':
				/* First is the int of the count. */
				count = *((u_int32_t *) host_buf);

				size += sizeof(u_int32_t);
				host_buf += sizeof(u_int32_t);

				/* Skip over "C[". */
				pf += 2;

				/* Parse out the substring between the brackets. */
				strcpy(tmp_format, pf);
				pch = closing_bracket(tmp_format);
				*pch = '\0';

				/* tmp_format now has the substring between the brackets. */

				for (i = 0; i < count; i++) {
					/* call myself recursively to process the substring. */
					host_buf = sizeof_hostbuf_to_netbuf(tmp_format, host_buf, &size);
				}

				pf += strlen(tmp_format) + sizeof(char);
				break;

			case 'd':
				od = (option_descriptor_t *) host_buf;

				if (od->name != NULL) {
					size += strlen(od->name) + sizeof(char);
				} else {
					size += sizeof(char);
				}
				if (od->title != NULL) {
					size += strlen(od->title) + sizeof(char);
				} else {
					size += sizeof(char);
				}
				if (od->tip != NULL) {
					size += strlen(od->tip) + sizeof(char);
				} else {
					size += sizeof(char);
				}
				if (od->help != NULL) {
					size += strlen(od->help) + sizeof(char);
				} else {
					size += sizeof(char);
				}

				size += sizeof(u_int32_t)	/* type		   */
					+ sizeof(u_int32_t)	/* unit		   */
					+ sizeof(u_int32_t)	/* format	   */
					+ sizeof(u_int32_t)	/* min_len	   */
					+ sizeof(u_int32_t)	/* max_len	   */
					+ sizeof(u_int32_t)	/* flags	   */
					+ sizeof(u_int32_t);	/* constraint_type */

				size += sizeof_netbuf_collection(od->constraint,
								 od->constraint_type,
								 od->type);

				size += sizeof_netbuf_value(od->value, od->type, od->flags & EVMS_OPTION_FLAGS_VALUE_IS_LIST);

				size += sizeof(u_int32_t)	/* od->group.group_number */
					+ sizeof(u_int32_t);	/* od->group.group_level  */
				if (od->group.group_name != NULL) {
					size += strlen(od->group.group_name) + sizeof(char);

				} else {
					size += sizeof(char);
				}

				host_buf += sizeof(option_descriptor_t);
				pf++;
				break;

			case 'e':
				ei = (extended_info_t *) host_buf;

				if (ei->name != NULL) {
					size += strlen(ei->name) + sizeof(char);
				} else {
					size += sizeof(char);
				}
				if (ei->title != NULL) {
					size += strlen(ei->title) + sizeof(char);
				} else {
					size += sizeof(char);
				}
				if (ei->desc != NULL) {
					size += strlen(ei->desc) + sizeof(char);
				} else {
					size += sizeof(char);
				}

				size += sizeof(u_int32_t)	/* type		   */
					+ sizeof(u_int32_t)	/* unit		   */
					+ sizeof(u_int32_t);	/* format	   */

				size += sizeof_netbuf_value(ei->value, ei->type, FALSE);

				size += sizeof(u_int32_t);	/* collection_type */

				size += sizeof_netbuf_collection(ei->collection,
								 ei->collection_type,
								 ei->type);

				size += sizeof(u_int32_t)	/* od->group.group_number */
					+ sizeof(u_int32_t);	/* od->group.group_level  */
				if (ei->group.group_name != NULL) {
					size += strlen(ei->group.group_name) + sizeof(char);

				} else {
					size += sizeof(char);
				}

				size += sizeof(u_int16_t);	/* flags */

				host_buf += sizeof(extended_info_t);
				pf++;
				break;

			case 'g':
				size += sizeof(u_int8_t);
				host_buf += sizeof(u_int8_t);
				pf++;
				break;

			case 'h':
				size += sizeof(u_int16_t);
				host_buf += sizeof(u_int16_t);
				pf++;
				break;

			case 'i':
				size += sizeof(u_int32_t);
				host_buf += sizeof(int);
				pf++;
				break;

			case 'I':
				size += sizeof(u_int32_t);
				host_buf += sizeof(u_int32_t);
				pf++;
				break;

			case 'k':
				kvp = (key_value_pair_t *) host_buf;

				if (kvp->name != NULL) {
					size += strlen(kvp->name) + sizeof(char);
				} else {
					size += sizeof(char);
				}

				size += sizeof(u_int16_t)	/* number          */
					+ sizeof(u_int8_t)	/* is_number_based */
					+ sizeof(u_int32_t)	/* type            */
					+ sizeof(u_int16_t);	/* flags           */

				size += sizeof_netbuf_value(kvp->value, kvp->type, kvp->flags & EVMS_KEY_VALUE_IS_LIST);

				host_buf += sizeof(key_value_pair_t);
				pf++;
				break;

			case 'l':
				size += sizeof(u_int64_t);
				host_buf += sizeof(u_int64_t);
				pf++;
				break;

			case 'o':
				hoi = (handle_object_info_t *) host_buf;
				/* Space for the object type */
				size += sizeof(u_int32_t);

				switch (hoi->type) {
					case DISK:
					case SEGMENT:
					case REGION:
					case EVMS_OBJECT:
						host_buf = sizeof_hostbuf_to_netbuf(storage_object_info_f, &hoi->info, &size);
						break;

					case CONTAINER:
						host_buf = sizeof_hostbuf_to_netbuf(storage_container_info_f, &hoi->info, &size);
						break;

					case VOLUME:
						host_buf = sizeof_hostbuf_to_netbuf(logical_volume_info_f, &hoi->info, &size);
						break;

					case PLUGIN:
						host_buf = sizeof_hostbuf_to_netbuf(plugin_info_f, &hoi->info, &size);
						break;

					default:
						break;
				}

				host_buf += sizeof(handle_object_info_t);
				pf++;
				break;

			case 'p':
				if (*(pf + 1) == '{') {
					void * sub_buf = *((int * *) host_buf);

					size += sizeof(u_int8_t);

					if (sub_buf != NULL) {
					}
					/* Skip over "p{". */
					pf += 2;

					/* Parse out the substring between the braces. */
					strcpy(tmp_format, pf);
					pch = closing_brace(tmp_format);
					*pch = '\0';

					/* tmp_format now has the substring between the braces. */

					if (sub_buf != NULL) {
						/* Call myself recursively to process the substring. */
						sizeof_hostbuf_to_netbuf(tmp_format, sub_buf, &size);
					}

					pf += strlen(tmp_format) + sizeof(char);

				} else {
					/*
					 * Size for the largest pointer on
					 * any architecture.
					 */
					size += sizeof(u_int64_t);
					pf++;
				}

				host_buf += sizeof(void *);
				break;

			case 's':
				/*
				 * One for the indicator that says if the string
				 * pointer is NULL.
				 */
				size += sizeof(u_int8_t);

				string = *((char * *) host_buf);
				if (string != NULL) {
					size += strlen(string) + sizeof(char);
				}

				host_buf += sizeof(char *);
				pf++;
				break;

			case 't':
				/*
				 * One for the indicator that says if the
				 * strings pointer is NULL.
				 */
				size += sizeof(u_int8_t);

				strings = *((char * * *) host_buf);
				if (strings != NULL) {
					for (i = 0; strings[i] != NULL; i++) {
						size += strlen(strings[i]) + sizeof(char);
					}

					/* one more for the final terminating nul. */
					size += sizeof(char);
				}

				host_buf += sizeof(char *);
				pf++;
				break;

			case 'v':
				value_type = *((value_type_t *) host_buf);
				host_buf += sizeof(value_type_t);

				value_is_list = *((boolean *) host_buf);
				host_buf += sizeof(boolean);

				size += sizeof(u_int8_t) +
					sizeof(u_int32_t) +
					sizeof_netbuf_value(*((value_t *) host_buf), value_type, value_is_list);
				host_buf += sizeof(value_t);
				pf++;
				break;

			case 'x':
				pf++;
				len = atoi(pf);
				size += len;

				host_buf += len;
				pf += strspn(pf, "0123456789");
				break;

			default:
				LOG_SERIOUS("Format character '%c' slipped past the validater. "
					    "I don't know how to handle it.  I'm skipping it.\n", *pf);
				pf++;
		}

		LOG_EXTRA("Size for format '%c' is %zd.\n", curr_f, size - curr_size);
	}

	LOG_EXTRA("Total size for format \"%s\" is %zd.\n", format, size);

	*psize += size;

	LOG_PROC_EXIT_PTR(host_buf);
	return host_buf;
}


int evms_sizeof_host_to_net(size_t * psize, char * format, ...) {

	int rc;
	va_list args;
	char * pf = format;
	char curr_f;
	size_t size = 0;
	size_t curr_size;
	int i;
	int x;
	u_int64_t u64;
	void * ptr;
	char * pch;
	char * string;
	char * * strings;
	char tmp_format[64];
	boolean value_is_list;
	value_type_t value_type;

	LOG_PROC_ENTRY();

	LOG_EXTRA("Format is: %s\n", format);

	rc = validate_format(format);
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	va_start(args, format);

	while (*pf != '\0') {
		curr_size = size;
		curr_f = *pf;

		switch (*pf) {
			case 'b':
				x = va_arg(args, int);
				size += sizeof(u_int8_t);
				pf++;
				break;

			case 'c':
			case 'C':
				/*
				 * Arrays are not passed as args.
				 * Pointers to arrays are, but not
				 * arrays directly themsleves.
				 */
				LOG_ERROR("Arrays are not passed as arguments. "
					  "Try using \"p{c[...]}\" for the array arguement.\n");
				LOG_PROC_EXIT_INT(EINVAL);
				return EINVAL;
				break;

			case 'd':
				/* Option descriptors are not passed as args. */
				LOG_ERROR("Option descriptors are not passed as arguments.\n");
				LOG_PROC_EXIT_INT(EINVAL);
				return EINVAL;
				break;

			case 'e':
				/* Extended info is not passed as args. */
				LOG_ERROR("Extended info is not passed as an argument.\n");
				LOG_PROC_EXIT_INT(EINVAL);
				return EINVAL;
				break;

			case 'g':
				x = va_arg(args, int);
				size += sizeof(u_int8_t);
				pf++;
				break;

			case 'h':
				x = va_arg(args, int);
				size += sizeof(u_int16_t);
				pf++;
				break;

			case 'i':
				x = va_arg(args, int);
				size += sizeof(u_int32_t);
				pf++;
				break;

			case 'I':
				x = va_arg(args, int);
				size += sizeof(u_int32_t);
				pf++;
				break;

			case 'k':
				/* Key value pairs are not passed as args. */
				LOG_ERROR("Key value pairs are not passed as arguments.\n");
				LOG_PROC_EXIT_INT(EINVAL);
				return EINVAL;
				break;

			case 'l':
				u64 = va_arg(args, u_int64_t);
				size += sizeof(u_int64_t);
				pf++;
				break;

			case 'o':
				/*
				 * handle_object_info_t structures are not
				 * passed as args.  Pointers to them are,
				 * but not structures directly themsleves.
				 */
				LOG_ERROR("handle_object_info_t structures are not passed as arguments. "
					  "Try using \"p{o}\" for the handle_object_info_t arguement.\n");
				LOG_PROC_EXIT_INT(EINVAL);
				return EINVAL;
				break;

			case 'p':
				if (*(pf + 1) == '{') {
					void * host_buf;

					host_buf = va_arg(args, void *);

					size += sizeof(u_int8_t);

					/* Skip over "p{". */
					pf += 2;

					/* Parse out the substring between the braces. */
					strcpy(tmp_format, pf);
					pch = closing_brace(tmp_format);
					*pch = '\0';

					/* tmp_format now has the substring between the braces. */

					if (host_buf != NULL) {
						/* Call myself recursively to process the substring. */
						sizeof_hostbuf_to_netbuf(tmp_format, host_buf, &size);
					}

					pf += strlen(tmp_format) + sizeof(char);

				} else {
					/*
					 * Size for the largest pointer on
					 * any architecture.
					 */
					ptr = va_arg(args, void *);
					size += sizeof(u_int64_t);
					pf++;
				}
				break;

			case 's':
				/*
				 * One for the indicator that says if the string
				 * pointer is NULL.
				 */
				size += sizeof(u_int8_t);

				string = va_arg(args, char *);
				if (string != NULL) {
					size += strlen(string) + sizeof(char);
				}
				pf++;
				break;

			case 't':
				/*
				 * One for the indicator that says if the string
				 * pointer is NULL.
				 */
				size += sizeof(u_int8_t);

				strings = va_arg(args, char * *);
				if (strings != NULL) {
					for (i = 0; strings[i] != NULL; i++) {
						size += strlen(strings[i]) + sizeof(char);
					}

					/* one more for the final terminating nul. */
					size++;
				}

				pf++;
				break;

			case 'v':
				value_type = va_arg(args, value_type_t);
				
				/*
				 * The following line should be:
				 * value_is_list = va_arg(args, boolean);
				 * but the compiler warns:
				 * "'boolean' is promoted to `int' when passed through `...'"
				 * so we do the casting manually.
				 */
				value_is_list = (boolean) va_arg(args, int);

				size += sizeof(u_int8_t) +
					sizeof(u_int32_t) +
					sizeof_netbuf_value(va_arg(args, value_t), value_type, value_is_list);
				pf++;
				break;

			case 'x':
				/*
				 * Byte strings are not passed as args.
				 * Pointers to them are, but not the
				 * bytes directly themsleves.
				 */
				LOG_ERROR("Byte strings are not passed as arguments. "
					  "Try using \"pxnnn\" for the byte string arguement.\n");
				LOG_PROC_EXIT_INT(EINVAL);
				return EINVAL;
				break;

			default:
				LOG_SERIOUS("Format character %c slipped past the validater. "
					    "I don't know how to handle it.  I'm skipping it.\n", *pf);
				pf++;
		}

		LOG_EXTRA("Size for format '%c' is %zd.\n", curr_f, size - curr_size);
	}

	va_end(args);

	LOG_EXTRA("Total size for format \"%s\" is %zd.\n", format, size);

	*psize = size;

	LOG_PROC_EXIT_INT(0);
	return 0;
}


/*
 * Functions for converting data into a network stream.
 */

static void * value_to_netbuf(void       * net_buf,
			      value_t      value,
			      value_type_t type,
			      boolean      value_is_list) {

	void * orig_net_buf = net_buf;
	uint count;
	value_t * p_value;
	int i;

	LOG_PROC_ENTRY();

	if (value_is_list) {
		/* First byte says if it is a NULL pointer or not. */
		*((u_int8_t *) net_buf) = (value.list != NULL);
		net_buf += sizeof(u_int8_t);

		if (value.list != NULL) {
			value_list_t * vl = (value_list_t *) value.list;
			count = vl->count;
			p_value = &vl->value[0];
			*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) count);
			net_buf += sizeof(u_int32_t);

		} else {
			/* That's all folks. */
			LOG_PROC_EXIT_PTR(net_buf);
			return net_buf;
		}

	} else {
		count = 1;
		p_value = &value;
	}

	for (i = 0; i < count; i++, p_value++) {
		switch (type) {
			case EVMS_Type_String:
				if ((*p_value).s != NULL) {
					/* A valid string follows. */
					*((u_int8_t *) net_buf) = TRUE;
					net_buf += sizeof(u_int8_t);

					strcpy(net_buf, (*p_value).s);
					net_buf += strlen((char *) net_buf) + sizeof(char);

				} else {
					/* String is NULL. */
					*((u_int8_t *) net_buf) = FALSE;
					net_buf += sizeof(u_int8_t);
				}
				break;

			case EVMS_Type_Boolean:
			case EVMS_Type_Char:
			case EVMS_Type_Unsigned_Char:
			case EVMS_Type_Int8:
			case EVMS_Type_Unsigned_Int8:
				*((u_int8_t *) net_buf) = (*p_value).uc;
				net_buf += sizeof(u_int8_t);
				break;

			case EVMS_Type_Int16:
			case EVMS_Type_Unsigned_Int16:
				*((u_int16_t *) net_buf) = HOST_TO_NET16((*p_value).ui16);
				net_buf += sizeof(u_int16_t);
				break;

			case EVMS_Type_Real32:
			case EVMS_Type_Int:
			case EVMS_Type_Int32:
			case EVMS_Type_Unsigned_Int:
			case EVMS_Type_Unsigned_Int32:
				*((u_int32_t *) net_buf) = HOST_TO_NET32((*p_value).ui32);
				net_buf += sizeof(u_int32_t);
				break;

			case EVMS_Type_Real64:
			case EVMS_Type_Int64:
			case EVMS_Type_Unsigned_Int64:
				*((u_int64_t *) net_buf) = HOST_TO_NET64((*p_value).ui64);
				net_buf += sizeof(u_int64_t);
				break;
		}
	}

	LOG_EXTRA("Size of output is %d.\n", (int)(net_buf - orig_net_buf));

	LOG_PROC_EXIT_PTR(net_buf);
	return net_buf;
}


static void * collection_to_netbuf(void             * net_buf,
				   value_collection_t coll,
				   collection_type_t  coll_type,
				   value_type_t       value_type) {
	int i;
	void * orig_net_buf = net_buf;

	LOG_PROC_ENTRY();

	switch (coll_type) {
		case EVMS_Collection_None:
			break;

		case EVMS_Collection_List:
			*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) coll.list->count);
			net_buf += sizeof(u_int32_t);

			for (i = 0; i < coll.list->count; i++) {
				net_buf = value_to_netbuf(net_buf, coll.list->value[i], value_type, FALSE);
			}
			break;

		case EVMS_Collection_Range:
			net_buf = value_to_netbuf(net_buf, coll.range->min,       value_type, FALSE);
			net_buf = value_to_netbuf(net_buf, coll.range->max,       value_type, FALSE);
			net_buf = value_to_netbuf(net_buf, coll.range->increment, value_type, FALSE);
			break;
	}

	LOG_EXTRA("Size of output is %d.\n", (int)(net_buf - orig_net_buf));

	LOG_PROC_EXIT_PTR(net_buf);
	return net_buf;
}


static void hostbuf_to_netbuf(char * format,
			      void * host_buf,       void * net_buf,
			      uint * p_host_buf_inc, uint * p_net_buf_inc) {

	void * orig_host_buf = host_buf;
	void * orig_net_buf  = net_buf;
	void * curr_net_buf;
	uint host_buf_inc;
	uint net_buf_inc;
	char * pf = format;
	char curr_f;
	u_int8_t  u8;
	u_int16_t u16;
	u_int32_t u32;
	u_int64_t u64;
	u_int32_t count;
	option_descriptor_t * od;
	extended_info_t * ei;
	key_value_pair_t * kvp;
	int i;
	char * pch;
	char tmp_format[64];
	handle_object_info_t * hoi;
	void * ptr;
	char * string;
	char * * strings;
	uint len;
	boolean value_is_list;
	value_type_t value_type;

	LOG_PROC_ENTRY();

	LOG_EXTRA("Format is: %s\n", format);

	while (*pf != '\0') {
		curr_net_buf = net_buf;
		curr_f = *pf;

		switch (*pf) {
			case 'b':
				u8 = (u_int8_t) *((boolean *) host_buf);
				*((u_int8_t *) net_buf) = u8;
				host_buf += sizeof(boolean);
				net_buf += sizeof(u_int8_t);
				pf++;
				break;

			case 'c':
				/* First is the int of the count. */
				count = (u_int32_t) *((int *) host_buf);
				*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) count);
				host_buf += sizeof(u_int32_t);
				net_buf += sizeof(u_int32_t);

				/* Skip over "c[". */
				pf += 2;

				/* Parse out the substring between the brackets. */
				strcpy(tmp_format, pf);
				pch = closing_bracket(tmp_format);
				*pch = '\0';

				/* tmp_format now has the substring between the brackets. */

				for (i = 0; i < count; i++) {
					uint host_buf_inc;
					uint net_buf_inc;

					/* call myself recursively to process the substring. */
					hostbuf_to_netbuf(tmp_format,
							  host_buf, net_buf,
							  &host_buf_inc, &net_buf_inc);

					host_buf += host_buf_inc;
					net_buf  += net_buf_inc;
				}

				pf += strlen(tmp_format) + sizeof(char);
				break;

			case 'C':
				/* First is the int of the count. */
				count = (u_int32_t) *((u_int32_t *) host_buf);
				*((u_int32_t *) net_buf) = HOST_TO_NET32(count);
				host_buf += sizeof(int);
				net_buf += sizeof(u_int32_t);

				/* Skip over "C[". */
				pf += 2;

				/* Parse out the substring between the brackets. */
				strcpy(tmp_format, pf);
				pch = closing_bracket(tmp_format);
				*pch = '\0';

				/* tmp_format now has the substring between the brackets. */

				for (i = 0; i < count; i++) {
					uint host_buf_inc;
					uint net_buf_inc;

					/* call myself recursively to process the substring. */
					hostbuf_to_netbuf(tmp_format,
							  host_buf, net_buf,
							  &host_buf_inc, &net_buf_inc);

					host_buf += host_buf_inc;
					net_buf  += net_buf_inc;
				}

				pf += strlen(tmp_format) + sizeof(char);
				break;

			case 'd':
				od = (option_descriptor_t *) host_buf;

				if (od->name != NULL) {
					len = strlen(od->name) + sizeof(char);
					memcpy(net_buf, od->name, len);
					net_buf += len;
				} else {
					*((char *) net_buf) = '\0';
					net_buf += sizeof(char);
				}
				if (od->title != NULL) {
					len = strlen(od->title) + sizeof(char);
					memcpy(net_buf, od->title, len);
					net_buf += len;
				} else {
					*((char *) net_buf) = '\0';
					net_buf += sizeof(char);
				}
				if (od->tip != NULL) {
					len = strlen(od->tip) + sizeof(char);
					memcpy(net_buf, od->tip, len);
					net_buf += len;
				} else {
					*((char *) net_buf) = '\0';
					net_buf += sizeof(char);
				}
				if (od->help != NULL) {
					len = strlen(od->help) + sizeof(char);
					memcpy(net_buf, od->help, len);
					net_buf += len;
				} else {
					*((char *) net_buf) = '\0';
					net_buf += sizeof(char);
				}

				*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) od->type);
				net_buf += sizeof(u_int32_t);
				*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) od->unit);
				net_buf += sizeof(u_int32_t);
				*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) od->format);
				net_buf += sizeof(u_int32_t);
				*((u_int32_t *) net_buf) = HOST_TO_NET32(            od->min_len);
				net_buf += sizeof(u_int32_t);
				*((u_int32_t *) net_buf) = HOST_TO_NET32(            od->max_len);
				net_buf += sizeof(u_int32_t);
				*((u_int32_t *) net_buf) = HOST_TO_NET32(            od->flags);
				net_buf += sizeof(u_int32_t);
				*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) od->constraint_type);
				net_buf += sizeof(u_int32_t);

				net_buf = collection_to_netbuf(net_buf,
							       od->constraint,
							       od->constraint_type,
							       od->type);

				net_buf = value_to_netbuf(net_buf,
							  od->value,
							  od->type,
							  od->flags & EVMS_OPTION_FLAGS_VALUE_IS_LIST);

				*((u_int32_t *) net_buf) = HOST_TO_NET32(od->group.group_number);
				net_buf += sizeof(u_int32_t);
				*((u_int32_t *) net_buf) = HOST_TO_NET32(od->group.group_level);
				net_buf += sizeof(u_int32_t);

				if (od->group.group_name != NULL) {
					len = strlen(od->group.group_name) + sizeof(char);
					memcpy(net_buf, od->group.group_name, len);
					net_buf += len;
				} else {
					*((char *) net_buf) = '\0';
					net_buf += sizeof(char);
				}

				host_buf += sizeof(option_descriptor_t);
				pf++;
				break;

			case 'e':
				ei = (extended_info_t *) host_buf;

				if (ei->name != NULL) {
					len = strlen(ei->name) + sizeof(char);
					memcpy(net_buf, ei->name, len);
					net_buf += len;
				} else {
					*((char *) net_buf) = '\0';
					net_buf += sizeof(char);
				}
				if (ei->title != NULL) {
					len = strlen(ei->title) + sizeof(char);
					memcpy(net_buf, ei->title, len);
					net_buf += len;
				} else {
					*((char *) net_buf) = '\0';
					net_buf += sizeof(char);
				}
				if (ei->desc != NULL) {
					len = strlen(ei->desc) + sizeof(char);
					memcpy(net_buf, ei->desc, len);
					net_buf += len;
				} else {
					*((char *) net_buf) = '\0';
					net_buf += sizeof(char);
				}

				*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) ei->type);
				net_buf += sizeof(u_int32_t);
				*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) ei->unit);
				net_buf += sizeof(u_int32_t);
				*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) ei->format);
				net_buf += sizeof(u_int32_t);

				net_buf = value_to_netbuf(net_buf,
							  ei->value,
							  ei->type,
							  FALSE);

				*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) ei->collection_type);
				net_buf += sizeof(u_int32_t);

				net_buf = collection_to_netbuf(net_buf,
							       ei->collection,
							       ei->collection_type,
							       ei->type);

				*((u_int32_t *) net_buf) = HOST_TO_NET32(ei->group.group_number);
				net_buf += sizeof(u_int32_t);
				*((u_int32_t *) net_buf) = HOST_TO_NET32(ei->group.group_level);
				net_buf += sizeof(u_int32_t);

				if (ei->group.group_name != NULL) {
					len = strlen(ei->group.group_name) + sizeof(char);
					memcpy(net_buf, ei->group.group_name, len);
					net_buf += len;
				} else {
					*((char *) net_buf) = '\0';
					net_buf += sizeof(char);
				}

				*((u_int16_t *) net_buf) = HOST_TO_NET16(ei->flags);
				net_buf += sizeof(u_int16_t);

				host_buf += sizeof(extended_info_t);
				pf++;
				break;

			case 'g':
				*((u_int8_t *) net_buf) = *((u_int8_t *) host_buf);
				host_buf += sizeof(u_int8_t);
				net_buf += sizeof(u_int8_t);
				pf++;
				break;

			case 'h':
				u16 = *((u_int16_t *) host_buf);
				*((u_int16_t *) net_buf) = HOST_TO_NET16(u16);
				host_buf += sizeof(u_int16_t);
				net_buf += sizeof(u_int16_t);
				pf++;
				break;

			case 'i':
				u32 = (u_int32_t) *((int *) host_buf);
				*((u_int32_t *) net_buf) = HOST_TO_NET32(u32);
				host_buf += sizeof(int);
				net_buf += sizeof(u_int32_t);
				pf++;
				break;

			case 'I':
				u32 = *((u_int32_t *) host_buf);
				*((u_int32_t *) net_buf) = HOST_TO_NET32(u32);
				host_buf += sizeof(u_int32_t);
				net_buf += sizeof(u_int32_t);
				pf++;
				break;

			case 'k':
				kvp = (key_value_pair_t *) host_buf;

				if (kvp->name != NULL) {
					len = strlen(kvp->name) + sizeof(char);
					memcpy(net_buf, kvp->name, len);
					net_buf += len;
				} else {
					*((char *) net_buf) = '\0';
					net_buf += sizeof(char);
				}

				*((u_int16_t *) net_buf) = HOST_TO_NET16(kvp->number);
				net_buf += sizeof(u_int16_t);
				*((u_int8_t *)  net_buf) = kvp->is_number_based;
				net_buf += sizeof(u_int8_t);
				*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) kvp->type);
				net_buf += sizeof(u_int32_t);
				*((u_int16_t *) net_buf) = HOST_TO_NET16(kvp->flags);
				net_buf += sizeof(u_int16_t);

				net_buf = value_to_netbuf(net_buf, kvp->value, kvp->type, kvp->flags & EVMS_KEY_VALUE_IS_LIST);

				host_buf += sizeof(key_value_pair_t);
				pf++;
				break;

			case 'l':
				u64 = *((u_int64_t *) host_buf);
				*((u_int64_t *) net_buf) = HOST_TO_NET64(u64);
				host_buf += sizeof(u_int64_t);
				net_buf += sizeof(u_int64_t);
				pf++;
				break;

			case 'o':
				hoi = (handle_object_info_t *) host_buf;
				/* Space for the object type */
				*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) hoi->type);
				net_buf += sizeof(u_int32_t);

				switch (hoi->type) {
					case DISK:
					case SEGMENT:
					case REGION:
					case EVMS_OBJECT:
						hostbuf_to_netbuf(storage_object_info_f,
								  &hoi->info, net_buf,
								  &host_buf_inc, &net_buf_inc);
						break;

					case CONTAINER:
						hostbuf_to_netbuf(storage_container_info_f,
								  &hoi->info, net_buf,
								  &host_buf_inc, &net_buf_inc);
						break;

					case VOLUME:
						hostbuf_to_netbuf(logical_volume_info_f,
								  &hoi->info, net_buf,
								  &host_buf_inc, &net_buf_inc);
						break;

					case PLUGIN:
						hostbuf_to_netbuf(plugin_info_f,
								  &hoi->info, net_buf,
								  &host_buf_inc, &net_buf_inc);
						break;

					default:
						break;
				}

				net_buf += net_buf_inc;
				host_buf += sizeof(handle_object_info_t);
				pf++;
				break;

			case 'p':
				if (*(pf + 1) == '{') {
					/* It's a pointer to a structure. */
					ptr = *((int * *) host_buf);

					/* Skip over "p{". */
					pf += 2;

					/* Parse out the substring between the braces. */
					strcpy(tmp_format, pf);
					pch = closing_brace(tmp_format);
					*pch = '\0';

					/* tmp_format now has the substring between the braces. */

					if (ptr != NULL) {
						/* Set the marker to say we have a valid pointer. */
						*((u_int8_t *) net_buf) = TRUE;
						net_buf += sizeof(u_int8_t);

						/* Call myself recursively to process the substring. */
						hostbuf_to_netbuf(tmp_format,
								  ptr, net_buf,
								  &host_buf_inc, &net_buf_inc);

						net_buf  += net_buf_inc;

					} else {
						/* Set the marker to say the pointer is NULL. */
						*((u_int8_t *) net_buf) = FALSE;
						net_buf += sizeof(u_int8_t);

					}

					host_buf += sizeof(void *);
					pf += strlen(tmp_format) + sizeof(char);

				} else {
					/* It's a generic pointer. */
					/*
					 * Use the largest archectctural size
					 * for a pointer.
					 */
					u64 = (u_int64_t) (unsigned long) *((unsigned long * *) host_buf);
					*((u_int64_t *) net_buf) = HOST_TO_NET64(u64);
					host_buf += sizeof(void *);
					net_buf += sizeof(u_int64_t);
					pf++;
				}
				break;

			case 's':
				string = *((char * *) host_buf);
				if (string != NULL) {
					/* A valid string follows. */
					*((u_int8_t *) net_buf) = TRUE;
					net_buf += sizeof(u_int8_t);

					strcpy(net_buf, string);
					net_buf += strlen(string) + sizeof(char);

				} else {
					/* String is NULL. */
					*((u_int8_t *) net_buf) = FALSE;
					net_buf += sizeof(u_int8_t);
				}

				host_buf += sizeof(char *);
				pf++;
				break;

			case 't':
				strings = *((char * * *) host_buf);
				if (strings != NULL) {
					/* Valid strings follow. */
					*((u_int8_t *) net_buf) = TRUE;
					net_buf += sizeof(u_int8_t);

					for (i = 0; strings[i] != NULL; i++) {
						strcpy(net_buf, strings[i]);
						net_buf += strlen(strings[i]) + FALSE;
					}

					/* Slap on the terminating nul. */
					*((char *) net_buf) = '\0';
					net_buf += sizeof(char);

				} else {
					/* Strings is NULL. */
					*((u_int8_t *) net_buf) = FALSE;
					net_buf += sizeof(u_int8_t);
				}

				host_buf += sizeof(char *);
				pf++;
				break;

			case 'v':
				value_type = *((value_type_t *) host_buf);
				*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) value_type);
				net_buf += sizeof(u_int32_t);
				host_buf += sizeof(value_type_t);

				value_is_list = *((boolean *) host_buf);
				*((u_int8_t *) net_buf) = (u_int8_t) value_is_list;
				net_buf += sizeof(u_int8_t);
				host_buf += sizeof(boolean);

				net_buf = value_to_netbuf(net_buf, *((value_t *) host_buf), value_type, value_is_list);
				host_buf += sizeof(value_t);
				pf++;
				break;

			case 'x':
				pf++;
				len = atoi(pf);

				memcpy(net_buf, host_buf, len);

				net_buf += len;
				host_buf += len;
				pf += strspn(pf, "0123456789");
				break;

			default:
				LOG_SERIOUS("Format character %c slipped past the validater. "
					    "I don't know how to handle it.  I'm skipping it.\n", *pf);
				pf++;
		}

		LOG_EXTRA("Size of output for format '%c' is %d.\n", curr_f, (int)(net_buf - curr_net_buf));
	}

	*p_host_buf_inc = host_buf - orig_host_buf;
	*p_net_buf_inc  = net_buf  - orig_net_buf;

	LOG_EXTRA("Total size of output for format \"%s\" is %d.\n", format, (int)(net_buf  - orig_net_buf));

	LOG_PROC_EXIT_VOID();
	return;
}


int evms_host_to_net(void * net_buf, char * format, ...) {

	int rc;
	char * pf = format;
	char curr_f;
	void * orig_net_buf = net_buf;
	void * curr_net_buf;
	va_list args;
	u_int16_t u16;
	u_int32_t u32;
	u_int64_t u64;
	int i;
	char * pch;
	char tmp_format[64];
	void * ptr;
	char * string;
	char * * strings;
	boolean value_is_list;
	value_type_t value_type;

	LOG_PROC_ENTRY();

	LOG_EXTRA("Format is: %s\n", format);

	rc = validate_format(format);
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	va_start(args, format);

	while (*pf != '\0') {
		curr_net_buf = net_buf;
		curr_f = *pf;

		switch (*pf) {
			case 'b':
				*((u_int8_t *) net_buf) = (u_int8_t) va_arg(args, int);
				net_buf += sizeof(u_int8_t);
				pf++;
				break;

			case 'c':
			case 'C':
				/*
				 * Arrays are not passed as args.
				 * Pointers to arrays are, but not
				 * arrays directly themsleves.
				 */
				LOG_ERROR("Arrays are not passed as arguments. "
					  "Try using \"p{c[...]}\" for the array arguement.\n");
				LOG_PROC_EXIT_INT(EINVAL);
				return EINVAL;
				break;

			case 'd':
				/* Option descriptors are not passed as args. */
				LOG_ERROR("Option descriptors are not passed as arguments.\n");
				LOG_PROC_EXIT_INT(EINVAL);
				return EINVAL;
				break;

			case 'e':
				/* Extended info is not passed as args. */
				LOG_ERROR("Extended info is not passed as an argument.\n");
				LOG_PROC_EXIT_INT(EINVAL);
				return EINVAL;
				break;

			case 'g':
				*((u_int8_t *) net_buf) = (u_int8_t) va_arg(args, int);
				net_buf += sizeof(u_int8_t);
				pf++;
				break;

			case 'h':
				u16 = (u_int16_t) va_arg(args, int);
				*((u_int16_t *) net_buf) = HOST_TO_NET16(u16);
				net_buf += sizeof(u_int16_t);
				pf++;
				break;

			case 'i':
				u32 = (u_int32_t) va_arg(args, int);
				*((u_int32_t *) net_buf) = HOST_TO_NET32(u32);
				net_buf += sizeof(u_int32_t);
				pf++;
				break;

			case 'I':
				u32 = (u_int32_t) va_arg(args, int);
				*((u_int32_t *) net_buf) = HOST_TO_NET32(u32);
				net_buf += sizeof(u_int32_t);
				pf++;
				break;

			case 'k':
				/* Key value pairs are not passed as args. */
				LOG_ERROR("Key value pairs are not passed as arguments.\n");
				LOG_PROC_EXIT_INT(EINVAL);
				return EINVAL;
				break;

			case 'l':
				u64 = (u_int64_t) va_arg(args, u_int64_t);
				*((u_int64_t *) net_buf) = HOST_TO_NET64(u64);
				net_buf += sizeof(u_int64_t);
				pf++;
				break;

			case 'o':
				/*
				 * handle_object_info_t structures are not
				 * passed as args.  Pointers to them are,
				 * but not structures directly themsleves.
				 */
				LOG_ERROR("handle_object_info_t structures are not passed as arguments. "
					  "Try using \"p{o}\" for the handle_object_info_t arguement.\n");
				LOG_PROC_EXIT_INT(EINVAL);
				return EINVAL;
				break;

			case 'p':
				if (*(pf + 1) == '{') {
					uint host_buf_inc;
					uint net_buf_inc;

					/* It's a pointer to a structure. */
					ptr = va_arg(args, void *);

					/* Skip over "p{". */
					pf += 2;

					/* Parse out the substring between the braces. */
					strcpy(tmp_format, pf);
					pch = closing_brace(tmp_format);
					*pch = '\0';

					/* tmp_format now has the substring between the braces. */

					if (ptr != NULL) {
						/* Set the marker to say we have a valid pointer. */
						*((u_int8_t *) net_buf) = TRUE;
						net_buf += sizeof(u_int8_t);

						/* Call myself recursively to process the substring. */
						hostbuf_to_netbuf(tmp_format,
								  ptr, net_buf,
								  &host_buf_inc, &net_buf_inc);

						net_buf += net_buf_inc;

					} else {
						/* Set the marker to say the pointer is NULL. */
						*((u_int8_t *) net_buf) = FALSE;
						net_buf += sizeof(u_int8_t);

					}

					pf += strlen(tmp_format) + sizeof(char);

				} else {
					/* It's a generic pointer. */
					/*
					 * Use the largest archectctural size
					 * for a pointer.
					 */
					u64 = (u_int64_t) (unsigned long) va_arg(args, void *);
					*((u_int64_t *) net_buf) = HOST_TO_NET64(u64);
					net_buf += sizeof(u_int64_t);
					pf++;
				}
				break;

			case 's':
				string = va_arg(args, char *);
				if (string != NULL) {
					/* A valid string follows. */
					*((u_int8_t *) net_buf) = TRUE;
					net_buf += sizeof(u_int8_t);

					strcpy(net_buf, string);
					net_buf += strlen(string) + sizeof(char);

				} else {
					/* String is NULL. */
					*((u_int8_t *) net_buf) = FALSE;
					net_buf += sizeof(u_int8_t);
				}

				pf++;
				break;

			case 't':
				strings = va_arg(args, char * *);
				if (strings != NULL) {
					/* Valid strings follows. */
					*((u_int8_t *) net_buf) = TRUE;
					net_buf += sizeof(u_int8_t);

					for (i = 0; strings[i] != NULL; i++) {
						strcpy(net_buf, strings[i]);
						net_buf += strlen(strings[i]) + sizeof(char);
					}

					/* Slap on the terminating nul. */
					*((char *) net_buf) = '\0';
					net_buf += sizeof(char);

				} else {
					/* Strings is NULL. */
					*((u_int8_t *) net_buf) = FALSE;
					net_buf += sizeof(u_int8_t);
				}
				pf++;
				break;

			case 'v':
				value_type = va_arg(args, value_type_t);
				*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) value_type);
				net_buf += sizeof(u_int32_t);

				/*
				 * The following line should be:
				 * value_is_list = va_arg(args, boolean);
				 * but the compiler warns:
				 * "'boolean' is promoted to `int' when passed through `...'"
				 * so we do the casting manually.
				 */
				value_is_list = (boolean) va_arg(args, int);
				*((u_int8_t *) net_buf) = (u_int8_t) value_is_list;
				net_buf += sizeof(u_int8_t);

				net_buf = value_to_netbuf(net_buf, va_arg(args, value_t), value_type, value_is_list);
				pf++;
				break;

			case 'x':
				/*
				 * Byte strings are not passed as args.
				 * Pointers to them are, but not the
				 * bytes directly themsleves.
				 */
				LOG_ERROR("Byte strings are not passed as arguments. "
					  "Try using \"pxnnn\" for the byte string arguement.\n");
				LOG_PROC_EXIT_INT(EINVAL);
				return EINVAL;
				break;

			default:
				LOG_SERIOUS("Format character %c slipped past the validater. "
					    "I don't know how to handle it.  I'm skipping it.\n", *pf);
				pf++;
		}

		LOG_EXTRA("Size of output for format '%c' is %d.\n", curr_f, (int)(net_buf - curr_net_buf));
	}

	va_end(args);

	LOG_EXTRA("Total size of output buffer for format \"%s\" is %d.\n", format, (int)(net_buf - orig_net_buf));

	LOG_PROC_EXIT_INT(0);
	return 0;
}


/*
 * Functions for calculating the size needed for a local buffer to receive
 * the data in a network stream.
 */

static void * skip_netbuf_value(void       * net_buf,
				value_type_t type,
				boolean      value_is_list) {
	uint count;
	int i;

	LOG_PROC_ENTRY();

	if (value_is_list) {
		/* First byte says if we have a NUL pointer or not. */
		boolean is_null = *((u_int8_t *) net_buf) == 0;;
		net_buf += sizeof(u_int8_t);

		if (is_null) {
			LOG_PROC_EXIT_PTR(net_buf);
			return net_buf;
		}

		count = NET_TO_HOST32(*((u_int32_t *) net_buf));
		net_buf += sizeof(u_int32_t);

	} else {
		count = 1;
	}

	for (i = 0; i < count; i++) {
		switch (type) {
			case EVMS_Type_String:
				if (*((u_int8_t *) net_buf) != 0) {
					/* A valid string pointer follows. */
					net_buf += sizeof(u_int8_t);

					net_buf += strlen((char *) net_buf) + sizeof(char);

				} else {
					/* The string pointer is NULL. */
					net_buf += sizeof(u_int8_t);
				}
				break;

			case EVMS_Type_Boolean:
			case EVMS_Type_Char:
			case EVMS_Type_Unsigned_Char:
			case EVMS_Type_Int8:
			case EVMS_Type_Unsigned_Int8:
				net_buf += sizeof(u_int8_t);
				break;

			case EVMS_Type_Int16:
			case EVMS_Type_Unsigned_Int16:
				net_buf += sizeof(u_int16_t);
				break;

			case EVMS_Type_Real32:
			case EVMS_Type_Int:
			case EVMS_Type_Int32:
			case EVMS_Type_Unsigned_Int:
			case EVMS_Type_Unsigned_Int32:
				net_buf += sizeof(u_int32_t);
				break;

			case EVMS_Type_Real64:
			case EVMS_Type_Int64:
			case EVMS_Type_Unsigned_Int64:
				net_buf += sizeof(u_int64_t);
				break;
		}
	}

	LOG_PROC_EXIT_PTR(net_buf);
	return net_buf;
}


static void * skip_netbuf_collection(void            * net_buf,
				     collection_type_t coll_type,
				     value_type_t      value_type) {
	
	uint count;
	int i;

	LOG_PROC_ENTRY();

	switch (coll_type) {
		case EVMS_Collection_None:
			break;

		case EVMS_Collection_List:
			/* First 32-bit value in the net_buf is the count */
			count = (uint) NET_TO_HOST32(*((u_int32_t *) net_buf));
			net_buf += sizeof(u_int32_t);

			for (i = 0; i < count; i++) {
				net_buf = skip_netbuf_value(net_buf, value_type, FALSE);
			}
			break;

		case EVMS_Collection_Range:
			net_buf = skip_netbuf_value(net_buf, value_type, FALSE);
			net_buf = skip_netbuf_value(net_buf, value_type, FALSE);
			net_buf = skip_netbuf_value(net_buf, value_type, FALSE);
			break;
	}

	LOG_PROC_EXIT_PTR(net_buf);
	return net_buf;
}


static void * skip_netbuf_option_descriptor(void * net_buf) {

	option_descriptor_t od;

	LOG_PROC_ENTRY();

	/* Skip name */
	net_buf += strlen((char *) net_buf) + sizeof(char);

	/* Skip title */
	net_buf += strlen((char *) net_buf) + sizeof(char);

	/* Skip tip */
	net_buf += strlen((char *) net_buf) + sizeof(char);

	/* Skip help */
	net_buf += strlen((char *) net_buf) + sizeof(char);

	od.type = (value_type_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

//	od.unit = (value_unit_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

//	od.format = (value_format_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

//	od.min_len = NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

//	od.max_len = NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

	od.flags = NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

	od.constraint_type = (collection_type_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

	net_buf = skip_netbuf_collection(net_buf,
					 od.constraint_type,
					 od.type);

	net_buf = skip_netbuf_value(net_buf,
				    od.type,
				    od.flags & EVMS_OPTION_FLAGS_VALUE_IS_LIST);

//	od.group.group_number = NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

//	od.group.group_level = NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

	/* Skip group.group_name */
	net_buf += strlen((char *) net_buf) + sizeof(char);

	LOG_PROC_EXIT_PTR(net_buf);
	return net_buf;
}


static void * skip_netbuf_extended_info(void * net_buf) {

	value_type_t      type;
	collection_type_t collection_type;

	LOG_PROC_ENTRY();

	/* Skip name */
	net_buf += strlen((char *) net_buf) + sizeof(char);

	/* Skip title */
	net_buf += strlen((char *) net_buf) + sizeof(char);

	/* Skip desc */
	net_buf += strlen((char *) net_buf) + sizeof(char);

	type = (value_type_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

//	ei->unit = (value_unit_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

//	ei->format = (value_format_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

	net_buf = skip_netbuf_value(net_buf,
				    type,
				    FALSE);

	collection_type = (collection_type_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

	net_buf = skip_netbuf_collection(net_buf,
					 collection_type,
					 type);

//	ei->group.group_number = NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

//	ei->group.group_level = NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

	/* Skip group.group_name */
	net_buf += strlen((char *) net_buf) + sizeof(char);

//	ei->flags = NET_TO_HOST16(*((u_int16_t *) net_buf));
	net_buf += sizeof(u_int16_t);

	LOG_PROC_EXIT_PTR(net_buf);
	return net_buf;
}


static void * skip_netbuf_key_value_pair(void * net_buf) {

	value_type_t type;
	u_int16_t    flags;

	LOG_PROC_ENTRY();

	/* Skip name */
	net_buf += strlen((char *) net_buf) + sizeof(char);


//	kvp->is_number_based = *((u_int8_t *) net_buf);
	net_buf += sizeof(u_int8_t);

	type = (value_type_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

	flags = NET_TO_HOST16(*((u_int16_t *) net_buf));
	net_buf += sizeof(u_int16_t);

	net_buf = skip_netbuf_value(net_buf,
				    type,
				    flags & EVMS_KEY_VALUE_IS_LIST);

	LOG_PROC_EXIT_PTR(net_buf);
	return net_buf;
}


static void * sizeof_netbuf_to_hostbuf(char * format, void * net_buf, size_t * psize) {

	char * pf = format;
	u_int32_t count;
	int i;
	char * pch;
	char tmp_format[64];
	uint len;

	LOG_PROC_ENTRY();

	LOG_EXTRA("Format is: %s\n", format);

	while (*pf != '\0') {
		switch (*pf) {
			case 'b':
				*psize += sizeof(boolean);
				net_buf += sizeof(u_int8_t);
				pf++;
				break;

			case 'c':
				/* First is the int of the count. */
				count = NET_TO_HOST32(*((u_int32_t *) net_buf));
				*psize += sizeof(uint);

				/* Counts are always sent on the wire as 32-bint ints. */
				net_buf += sizeof(u_int32_t);

				/* Skip over "c[". */
				pf += 2;

				/* Parse out the substring between the brackets. */
				strcpy(tmp_format, pf);
				pch = closing_bracket(tmp_format);
				*pch = '\0';

				/* tmp_format now has the substring between the brackets. */

				for (i = 0; i < count; i++) {
					/* call myself recursively to process the substring. */
					net_buf = sizeof_netbuf_to_hostbuf(tmp_format, net_buf, psize);
				}

				pf += strlen(tmp_format) + sizeof(char);
				break;

			case 'C':
				/* First is the int of the count. */
				count = NET_TO_HOST32(*((u_int32_t *) net_buf));
				*psize += sizeof(u_int32_t);

				/* Skip over "C[". */
				pf += 2;

				/* Parse out the substring between the brackets. */
				strcpy(tmp_format, pf);
				pch = closing_bracket(tmp_format);
				*pch = '\0';

				/* tmp_format now has the substring between the brackets. */

				for (i = 0; i < count; i++) {
					/* call myself recursively to process the substring. */
					net_buf = sizeof_netbuf_to_hostbuf(tmp_format, net_buf, psize);
				}

				pf += strlen(tmp_format) + sizeof(char);
				break;

			case 'd':
				*psize += sizeof(option_descriptor_t);
				net_buf = skip_netbuf_option_descriptor(net_buf);
				pf++;
				break;

			case 'e':
				*psize += sizeof(extended_info_t);
				net_buf = skip_netbuf_extended_info(net_buf);
				pf++;
				break;

			case 'g':
				*psize += sizeof(u_int8_t);
				net_buf += sizeof(u_int8_t);
				pf++;
				break;

			case 'h':
				*psize += sizeof(u_int16_t);
				net_buf += sizeof(u_int16_t);
				pf++;
				break;

			case 'i':
				*psize += sizeof(int);
				/* ints are sent on the wire as 32-bit ints. */
				net_buf += sizeof(u_int32_t);
				pf++;
				break;

			case 'I':
				*psize += sizeof(u_int32_t);
				net_buf += sizeof(u_int32_t);
				pf++;
				break;

			case 'k':
				*psize += sizeof(key_value_pair_t);
				net_buf = skip_netbuf_key_value_pair(net_buf);
				pf++;
				break;

			case 'l':
				*psize += sizeof(u_int64_t);
				net_buf += sizeof(u_int64_t);
				pf++;
				break;

			case 'o':
				*psize += sizeof(handle_object_info_t);
				pf++;
				break;

			case 'p':
				/*
				 * No need to delve into the structure that the
				 * pointer may point to.  As a pointer to a
				 * structure, the structure will be allocated
				 * separately.  It is not part of the base
				 * size being calculated.
				 */
				*psize += sizeof(void *);
				pf++;
				break;

			case 's':
			case 't':
				/*
				 * No need to get the lengths of the strings.
				 * The strings will be allocated separately.
				 * They will not be part of the base size
				 * being calculated.
				 */
				*psize += sizeof(char * *);
				pf++;
				break;

			case 'v':
				*psize += sizeof(value_t);
				pf++;
				break;

			case 'x':
				pf++;
				len = atoi(pf);
				*psize += len;

				net_buf += len;
				pf += strspn(pf, "0123456789");
				break;

			default:
				LOG_SERIOUS("Format character %c slipped past the validater. "
					    "I don't know how to handle it.  I'm skipping it.\n", *pf);
				pf++;
		}
	}

	LOG_PROC_EXIT_PTR(net_buf);
	return net_buf;
}


/*
 * Functions for converting a network stream into local buffers.
 */

static void * netbuf_to_value(void       * net_buf,
			      value_t    * pvalue,
			      value_type_t type,
			      boolean      value_is_list) {
	uint count;
	value_t * p_value;
	int i;

	LOG_PROC_ENTRY();

	if (value_is_list) {
		value_list_t * vl;

		/* First byte says if we have a NUL pointer or not. */
		boolean is_null = *((u_int8_t *) net_buf) == 0;;
		net_buf += sizeof(u_int8_t);

		if (is_null) {
			(*pvalue).list = NULL;

			LOG_PROC_EXIT_PTR(net_buf);
			return net_buf;
		}

		count = NET_TO_HOST32(*((u_int32_t *) net_buf));
		net_buf += sizeof(u_int32_t);

		vl = engine_alloc(sizeof(value_list_t) + (count - 1) * sizeof(value_t));
		(*pvalue).list = vl;
		if (vl != NULL) {
			vl->count = count;
			p_value = &vl->value[0];

		} else {
			/*
			 * Couldn't get memory for the value list.
			 * Skip over the value list in the net_buf and quit.
			 */
			int i;

			for (i = 0; i < count; i++) {
				switch (type) {
					case EVMS_Type_String:
						if (*((u_int8_t *) net_buf) != 0) {
							/* A valid string pointer follows. */
							net_buf += sizeof(u_int8_t);
							net_buf += strlen((char *) net_buf) + sizeof(char);

						} else {
							/* The string pointer is NULL. */
							net_buf += sizeof(u_int8_t);
						}
						break;

					case EVMS_Type_Boolean:
					case EVMS_Type_Char:
					case EVMS_Type_Unsigned_Char:
					case EVMS_Type_Int8:
					case EVMS_Type_Unsigned_Int8:
						net_buf += sizeof(u_int8_t);
						break;

					case EVMS_Type_Int16:
					case EVMS_Type_Unsigned_Int16:
						net_buf += sizeof(u_int16_t);
						break;

					case EVMS_Type_Real32:
					case EVMS_Type_Int:
					case EVMS_Type_Int32:
					case EVMS_Type_Unsigned_Int:
					case EVMS_Type_Unsigned_Int32:
						net_buf += sizeof(u_int32_t);
						break;

					case EVMS_Type_Real64:
					case EVMS_Type_Int64:
					case EVMS_Type_Unsigned_Int64:
						net_buf += sizeof(u_int64_t);
						break;
				}
			}

			LOG_PROC_EXIT_PTR(net_buf);
			return net_buf;
		}

	} else {
		count = 1;
		p_value = pvalue;
	}

	for (i = 0; i < count; i++, p_value++) {
		switch (type) {
			case EVMS_Type_String:
				if (*((u_int8_t *) net_buf) != 0) {
					/* A valid string pointer follows. */
					net_buf += sizeof(u_int8_t);

					(*p_value).s = engine_strdup((const char *) net_buf);
					net_buf += strlen((char *) net_buf) + sizeof(char);

				} else {
					/* The string pointer is NULL. */
					net_buf += sizeof(u_int8_t);

					(*p_value).s = NULL;
				}
				break;

			case EVMS_Type_Boolean:
			case EVMS_Type_Char:
			case EVMS_Type_Unsigned_Char:
			case EVMS_Type_Int8:
			case EVMS_Type_Unsigned_Int8:
				(*p_value).uc = *((u_int8_t *) net_buf);
				net_buf += sizeof(u_int8_t);
				break;

			case EVMS_Type_Int16:
			case EVMS_Type_Unsigned_Int16:
				(*p_value).ui16 = NET_TO_HOST16(*((u_int16_t *) net_buf));
				net_buf += sizeof(u_int16_t);
				break;

			case EVMS_Type_Real32:
			case EVMS_Type_Int:
			case EVMS_Type_Int32:
			case EVMS_Type_Unsigned_Int:
			case EVMS_Type_Unsigned_Int32:
				(*p_value).ui32 = NET_TO_HOST32(*((u_int32_t *) net_buf));
				net_buf += sizeof(u_int32_t);
				break;

			case EVMS_Type_Real64:
			case EVMS_Type_Int64:
			case EVMS_Type_Unsigned_Int64:
				(*p_value).ui64 = NET_TO_HOST64(*((u_int64_t *) net_buf));
				net_buf += sizeof(u_int64_t);
				break;
		}
	}

	LOG_PROC_EXIT_PTR(net_buf);
	return net_buf;
}


static void * netbuf_to_collection(void               * net_buf,
				   value_collection_t * pcoll,
				   collection_type_t    coll_type,
				   value_type_t         value_type) {

	uint count;
	value_list_t * list;
	value_range_t * range;
	int i;

	LOG_PROC_ENTRY();

	switch (coll_type) {
		case EVMS_Collection_None:
			(*(pcoll)).list = NULL;
			break;

		case EVMS_Collection_List:
			/* First 32-bit value in the net_buf is the count */
			count = (uint) NET_TO_HOST32(*((u_int32_t *) net_buf));
			net_buf += sizeof(u_int32_t);

			list = engine_alloc(sizeof(value_list_t) + (count - 1) * sizeof(value_t));

			if (list != NULL) {
				list->count = count;

				for (i = 0; i < count; i++) {
					net_buf = netbuf_to_value(net_buf, &list->value[i], value_type, FALSE);
				}
			}

			(*(pcoll)).list = list;
			break;

		case EVMS_Collection_Range:
			range = engine_alloc(sizeof(value_range_t));

			if (range != NULL) {
				net_buf = netbuf_to_value(net_buf, &range->min, value_type, FALSE);
				net_buf = netbuf_to_value(net_buf, &range->max, value_type, FALSE);
				net_buf = netbuf_to_value(net_buf, &range->increment, value_type, FALSE);
			}

			(*(pcoll)).range = range;
			break;
	}

	LOG_PROC_EXIT_PTR(net_buf);
	return net_buf;
}


static void netbuf_to_hostbuf(char * format,
			      void * net_buf, void * host_buf,
			      uint * p_net_buf_inc, uint * p_host_buf_inc) {

	void * orig_net_buf  = net_buf;
	void * orig_host_buf = host_buf;
	void * sub_buf;
	uint net_buf_inc;
	uint host_buf_inc;
	uint sub_buf_inc;
	char * pf = format;
	u_int16_t u16;
	u_int32_t u32;
	u_int64_t u64;
	u_int32_t count;
	option_descriptor_t * od;
	extended_info_t * ei;
	key_value_pair_t * kvp;
	int i;
	handle_object_info_t * hoi;
	char * pch;
	char tmp_format[64];
	char * string;
	char * * strings;
	size_t alloc_size;
	uint len;
	boolean value_is_list;
	value_type_t value_type;

	LOG_PROC_ENTRY();

	LOG_EXTRA("Format is: %s\n", format);

	while (*pf != '\0') {
		switch (*pf) {
			case 'b':
				*((boolean *) host_buf) = (boolean) *((u_int8_t *) net_buf);
				net_buf += sizeof(u_int8_t);
				host_buf += sizeof(boolean);
				pf++;
				break;

			case 'c':
				/* First is the int of the count. */
				count = NET_TO_HOST32(*((u_int32_t *) net_buf));
				*((uint *) host_buf) = (uint) count;
				net_buf += sizeof(u_int32_t);
				host_buf += sizeof(uint);

				/* Skip over "c[". */
				pf += 2;

				/* Parse out the substring between the brackets. */
				strcpy(tmp_format, pf);
				pch = closing_bracket(tmp_format);
				*pch = '\0';

				/* tmp_format now has the substring between the brackets. */

				for (i = 0; i < count; i++) {
					/* call myself recursively to process the substring. */
					netbuf_to_hostbuf(tmp_format,
							  net_buf, host_buf,
							  &net_buf_inc, &host_buf_inc);

					net_buf  += net_buf_inc;
					host_buf += host_buf_inc;
				}

				pf += strlen(tmp_format) + sizeof(char);
				break;

			case 'C':
				/* First is the int of the count. */
				count = NET_TO_HOST32(*((u_int32_t *) net_buf));
				*((u_int32_t *) host_buf) = count;
				net_buf += sizeof(u_int32_t);
				host_buf += sizeof(u_int32_t);

				/* Skip over "C[". */
				pf += 2;

				/* Parse out the substring between the brackets. */
				strcpy(tmp_format, pf);
				pch = closing_bracket(tmp_format);
				*pch = '\0';

				/* tmp_format now has the substring between the brackets. */

				for (i = 0; i < count; i++) {
					/* call myself recursively to process the substring. */
					netbuf_to_hostbuf(tmp_format,
							  net_buf, host_buf,
							  &net_buf_inc, &host_buf_inc);

					net_buf  += net_buf_inc;
					host_buf += host_buf_inc;
				}

				pf += strlen(tmp_format) + sizeof(char);
				break;

			case 'd':
				od = (option_descriptor_t *) host_buf;

				if (*((char *) net_buf) != '\0') {
					od->name = engine_strdup((const char *) net_buf);
				} else {
					od->name = NULL;
				}
				net_buf += strlen((char *) net_buf) + sizeof(char);

				if (*((char *) net_buf) != '\0') {
					od->title = engine_strdup((const char *) net_buf);
				} else {
					od->title = NULL;
				}
				net_buf += strlen((char *) net_buf) + sizeof(char);

				if (*((char *) net_buf) != '\0') {
					od->tip = engine_strdup((const char *) net_buf);
				} else {
					od->tip = NULL;
				}
				net_buf += strlen((char *) net_buf) + sizeof(char);

				if (*((char *) net_buf) != '\0') {
					od->help = engine_strdup((const char *) net_buf);
				} else {
					od->help = NULL;
				}
				net_buf += strlen((char *) net_buf) + sizeof(char);

				od->type = (value_type_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
				net_buf += sizeof(u_int32_t);

				od->unit = (value_unit_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
				net_buf += sizeof(u_int32_t);

				od->format = (value_format_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
				net_buf += sizeof(u_int32_t);

				od->min_len = NET_TO_HOST32(*((u_int32_t *) net_buf));
				net_buf += sizeof(u_int32_t);

				od->max_len = NET_TO_HOST32(*((u_int32_t *) net_buf));
				net_buf += sizeof(u_int32_t);

				od->flags = NET_TO_HOST32(*((u_int32_t *) net_buf));
				net_buf += sizeof(u_int32_t);

				od->constraint_type = (collection_type_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
				net_buf += sizeof(u_int32_t);

				net_buf = netbuf_to_collection(net_buf,
							       &od->constraint,
							       od->constraint_type,
							       od->type);

				net_buf = netbuf_to_value(net_buf,
							  &od->value,
							  od->type,
							  od->flags & EVMS_OPTION_FLAGS_VALUE_IS_LIST);

				od->group.group_number = NET_TO_HOST32(*((u_int32_t *) net_buf));
				net_buf += sizeof(u_int32_t);

				od->group.group_level = NET_TO_HOST32(*((u_int32_t *) net_buf));
				net_buf += sizeof(u_int32_t);


				if (*((char *) net_buf) != '\0') {
					od->group.group_name = engine_strdup((const char *) net_buf);
				} else {
					od->group.group_name = NULL;
				}
				net_buf += strlen((char *) net_buf) + sizeof(char);

				host_buf += sizeof(option_descriptor_t);
				pf++;
				break;

			case 'e':
				ei = (extended_info_t *) host_buf;

				if (*((char *) net_buf) != '\0') {
					ei->name = engine_strdup((const char *) net_buf);
				} else {
					ei->name = NULL;
				}
				net_buf += strlen((char *) net_buf) + sizeof(char);

				if (*((char *) net_buf) != '\0') {
					ei->title = engine_strdup((const char *) net_buf);
				} else {
					ei->title = NULL;
				}
				net_buf += strlen((char *) net_buf) + sizeof(char);

				if (*((char *) net_buf) != '\0') {
					ei->desc = engine_strdup((const char *) net_buf);
				} else {
					ei->desc = NULL;
				}
				net_buf += strlen((char *) net_buf) + sizeof(char);

				ei->type = (value_type_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
				net_buf += sizeof(u_int32_t);

				ei->unit = (value_unit_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
				net_buf += sizeof(u_int32_t);

				ei->format = (value_format_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
				net_buf += sizeof(u_int32_t);

				net_buf = netbuf_to_value(net_buf,
							  &ei->value,
							  ei->type,
							  FALSE);

				ei->collection_type = (collection_type_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
				net_buf += sizeof(u_int32_t);

				net_buf = netbuf_to_collection(net_buf,
							       &ei->collection,
							       ei->collection_type,
							       ei->type);

				ei->group.group_number = NET_TO_HOST32(*((u_int32_t *) net_buf));
				net_buf += sizeof(u_int32_t);

				ei->group.group_level = NET_TO_HOST32(*((u_int32_t *) net_buf));
				net_buf += sizeof(u_int32_t);


				if (*((char *) net_buf) != '\0') {
					ei->group.group_name = engine_strdup((const char *) net_buf);
				} else {
					ei->group.group_name = NULL;
				}
				net_buf += strlen((char *) net_buf) + sizeof(char);

				ei->flags = NET_TO_HOST16(*((u_int16_t *) net_buf));
				net_buf += sizeof(u_int16_t);

				host_buf += sizeof(extended_info_t);
				pf++;
				break;

			case 'g':
				*((u_int8_t *) host_buf) = *((u_int8_t *) net_buf);
				net_buf += sizeof(u_int8_t);
				host_buf += sizeof(u_int8_t);
				pf++;
				break;

			case 'h':
				u16 = *((u_int16_t *) net_buf);
				*((u_int16_t *) host_buf) = NET_TO_HOST16(u16);
				net_buf += sizeof(u_int16_t);
				host_buf += sizeof(u_int16_t);
				pf++;
				break;

			case 'i':
				u32 = *((u_int32_t *) net_buf);
				*((uint *) host_buf) = (int) NET_TO_HOST32(u32);
				net_buf += sizeof(u_int32_t);
				host_buf += sizeof(uint);
				pf++;
				break;

			case 'I':
				u32 = *((u_int32_t *) net_buf);
				*((u_int32_t *) host_buf) = NET_TO_HOST32(u32);
				net_buf += sizeof(u_int32_t);
				host_buf += sizeof(u_int32_t);
				pf++;
				break;

			case 'k':
				kvp = (key_value_pair_t *) host_buf;

				if (*((char *) net_buf) != '\0') {
					kvp->name = engine_strdup((const char *) net_buf);
				} else {
					kvp->name = NULL;
				}
				net_buf += strlen((char *) net_buf) + sizeof(char);


				kvp->is_number_based = *((u_int8_t *) net_buf);
				net_buf += sizeof(u_int8_t);

				kvp->type = (value_type_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
				net_buf += sizeof(u_int32_t);

				kvp->flags = NET_TO_HOST16(*((u_int16_t *) net_buf));
				net_buf += sizeof(u_int16_t);

				net_buf = netbuf_to_value(net_buf,
							  &kvp->value,
							  kvp->type,
							  kvp->flags & EVMS_KEY_VALUE_IS_LIST);

				host_buf += sizeof(key_value_pair_t);
				pf++;
				break;

			case 'l':
				u64 = *((u_int64_t *) net_buf);
				*((u_int64_t *) host_buf) = NET_TO_HOST64(u64);
				net_buf += sizeof(u_int64_t);
				host_buf += sizeof(u_int64_t);
				pf++;
				break;

			case 'o':
				hoi = (handle_object_info_t *) host_buf;
				hoi->type = (object_type_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
				net_buf += sizeof(u_int32_t);
				switch (hoi->type) {
					case DISK:
					case SEGMENT:
					case REGION:
					case EVMS_OBJECT:
						netbuf_to_hostbuf(storage_object_info_f,
								  net_buf, &hoi->info,
								  &net_buf_inc, &host_buf_inc);
						break;

					case CONTAINER:
						netbuf_to_hostbuf(storage_container_info_f,
								  net_buf, &hoi->info,
								  &net_buf_inc, &host_buf_inc);
						break;

					case VOLUME:
						netbuf_to_hostbuf(logical_volume_info_f,
								  net_buf, &hoi->info,
								  &net_buf_inc, &host_buf_inc);
						break;

					case PLUGIN:
						netbuf_to_hostbuf(plugin_info_f,
								  net_buf, &hoi->info,
								  &net_buf_inc, &host_buf_inc);
						break;

					default:
						break;
				}

				net_buf += net_buf_inc;
				host_buf += sizeof(handle_object_info_t);
				pf++;
				break;

			case 'p':
				if (*(pf + 1) == '{') {
					u_int8_t valid_pointer;

					/* Skip over "p{". */
					pf += 2;

					/* Parse out the substring between the braces. */
					strcpy(tmp_format, pf);
					pch = closing_brace(tmp_format);
					*pch = '\0';

					valid_pointer = *((u_int8_t *) net_buf);
					net_buf += sizeof(u_int8_t);

					if (valid_pointer) {
						alloc_size = 0;
						sizeof_netbuf_to_hostbuf(tmp_format, net_buf, &alloc_size);
						sub_buf = engine_alloc(alloc_size);

						/* tmp_format now has the substring between the braces. */

						/* Call myself recursively to process the substring. */
						netbuf_to_hostbuf(tmp_format,
								  net_buf, sub_buf,
								  &sub_buf_inc, &net_buf_inc);

						*((int * *) host_buf) = (int *) sub_buf;

						net_buf  += net_buf_inc;

					} else {
						*((int * *) host_buf) = NULL;
					}

					host_buf += sizeof(void *);
					pf += strlen(tmp_format) + sizeof(char);

				} else {
					/* It's a generic pointer. */
					/*
					 * Use the largest archectctural size
					 * for a pointer.
					 */
					u64 = (u_int64_t) (unsigned long) *((unsigned long * *) net_buf);
					*((u_int64_t *) host_buf) = NET_TO_HOST64(u64);
					net_buf += sizeof(u_int64_t);
					host_buf += sizeof(void *);
					pf++;
				}
				break;

			case 's':
				if (*((u_int8_t *) net_buf)) {
					/* A valid string follows. */
					net_buf += sizeof(u_int8_t);

					*((char * *) host_buf) = engine_strdup(net_buf);
					net_buf += strlen(net_buf) + sizeof(char);

				} else {
					/* The string pointer is NULL. */
					net_buf += sizeof(u_int8_t);

					*((char * *) host_buf) = NULL;
				}

				host_buf += sizeof(char *);
				pf++;
				break;

			case 't':
				if (*((u_int8_t *) net_buf)) {
					/* Valid strings follow. */
					net_buf += sizeof(u_int8_t);

					/*
					 * Count the number of strings so that
					 * we know how big to allocate the array
					 * of pointers to the strings.
					 */
					string = (char *) net_buf;
					count = 0;
					while (*string != '\0') {
						count++;
						string += strlen(string) +1;
					}

					strings = engine_alloc((count + 1) * sizeof(char *));
					if (strings != NULL) {
						*((char * * *) host_buf) = strings;

						string = (char *) net_buf;
						for (i = 0; *string != '\0'; i++) {
							strings[i] = engine_strdup(string);
							net_buf += strlen(string) + sizeof(char);
							string = (char *) net_buf;
						}
						strings[count] = NULL;
						net_buf += sizeof(char);
					}

				} else {
					/*
					 * No strings passed.  Set the
					 * string array pointer to NULL.
					 */
					net_buf += sizeof(u_int8_t);

					*((char * * *) host_buf) = NULL;
				}

				host_buf += sizeof(char * *);
				pf++;
				break;

			case 'v':
				value_type = (value_type_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
				net_buf += sizeof(u_int32_t);

				value_is_list = (boolean) *((u_int8_t *) net_buf);
				net_buf += sizeof(u_int8_t);

				net_buf = netbuf_to_value(net_buf,
							  (value_t *) host_buf,
							  value_type,
							  value_is_list);
				host_buf += sizeof(value_t);
				pf++;
				break;

			case 'x':
				pf++;
				len = atoi(pf);

				memcpy(host_buf, net_buf, len);

				net_buf += len;
				host_buf += len;
				pf += strspn(pf, "0123456789");
				break;

			default:
				LOG_SERIOUS("Format character %c slipped past the validater. "
					    "I don't know how to handle it.  I'm skipping it.\n", *pf);
				pf++;
		}
	}

	*p_net_buf_inc  = net_buf  - orig_net_buf;
	*p_host_buf_inc = host_buf - orig_host_buf;

	LOG_PROC_EXIT_VOID();
	return;
}


int evms_net_to_host(void * net_buf, char * format, ...) {

	int rc;
	char * pf = format;
	va_list args;
	void * var_ptr;
	u_int16_t u16;
	u_int32_t u32;
	u_int64_t u64;
	int i;
	uint count;
	char * pch;
	char tmp_format[64];
	char * string;
	char * * strings;
	size_t alloc_size;
	boolean value_is_list;
	value_type_t value_type;

	LOG_PROC_ENTRY();

	LOG_EXTRA("Format is: %s\n", format);

	rc = validate_format(format);
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	va_start(args, format);

	while (*pf != '\0') {
		var_ptr = va_arg(args, void *);

		switch (*pf) {
			case 'b':
				*((boolean *) var_ptr) = (boolean) *((u_int8_t *) net_buf);
				net_buf += sizeof(u_int8_t);
				pf++;
				break;

			case 'c':
			case 'C':
				/*
				 * Arrays are not passed as args.
				 * Pointers to arrays are, but not
				 * arrays directly themsleves.
				 */
				LOG_ERROR("Arrays are not passed as arguments. "
					  "Try using \"p{c[...]}\" for the array arguement.\n");
				LOG_PROC_EXIT_INT(EINVAL);
				return EINVAL;
				break;

			case 'd':
				/* Option descriptors are not passed as args. */
				LOG_ERROR("Option descriptors are not passed as arguments.\n");
				LOG_PROC_EXIT_INT(EINVAL);
				return EINVAL;
				break;

			case 'e':
				/* Extended info is not passed as args. */
				LOG_ERROR("Extended info is not passed as an argument.\n");
				LOG_PROC_EXIT_INT(EINVAL);
				return EINVAL;
				break;

			case 'g':
				*((u_int8_t *) var_ptr) = *((u_int8_t *) net_buf);
				net_buf += sizeof(u_int8_t);
				pf++;
				break;

			case 'h':
				u16 = NET_TO_HOST16(*((u_int16_t *) net_buf));
				*((u_int16_t *) var_ptr) = u16;
				net_buf += sizeof(u_int16_t);
				pf++;
				break;

			case 'i':
				u32 = NET_TO_HOST32(*((u_int32_t *) net_buf));
				*((int *) var_ptr) = (int) u32;
				net_buf += sizeof(u_int32_t);
				pf++;
				break;

			case 'I':
				u32 = NET_TO_HOST32(*((u_int32_t *) net_buf));
				*((u_int32_t *) var_ptr) = u32;
				net_buf += sizeof(u_int32_t);
				pf++;
				break;

			case 'l':
				u64 = NET_TO_HOST64(*((u_int64_t *) net_buf));
				*((u_int64_t *) var_ptr) = u64;
				net_buf += sizeof(u_int64_t);
				pf++;
				break;

			case 'k':
				/* Key value pairs are not passed as args. */
				LOG_ERROR("Key value pairs are not passed as arguments.\n");
				LOG_PROC_EXIT_INT(EINVAL);
				return EINVAL;
				break;

			case 'o':
				/*
				 * handle_object_info_t structures are not
				 * passed as args.  Pointers to them are,
				 * but not structures directly themsleves.
				 */
				LOG_ERROR("handle_object_info_t structures are not passed as arguments. "
					  "Try using \"p{o}\" for the handle_object_info_t arguement.\n");
				LOG_PROC_EXIT_INT(EINVAL);
				return EINVAL;
				break;

			case 'p':
				if (*(pf + 1) == '{') {
					/* It's a pointer to a structure. */
					void * sub_buf;
					uint net_buf_inc;
					uint sub_buf_inc;
					u_int8_t valid_pointer;

					/* Skip over "p{". */
					pf += 2;

					/* Parse out the substring between the braces. */
					strcpy(tmp_format, pf);
					pch = closing_brace(tmp_format);
					*pch = '\0';

					valid_pointer = *((u_int8_t *) net_buf);
					net_buf += sizeof(u_int8_t);

					if (valid_pointer) {
						alloc_size = 0;
						sizeof_netbuf_to_hostbuf(tmp_format, net_buf, &alloc_size);
						sub_buf = engine_alloc(alloc_size);

						/* tmp_format now has the substring between the braces. */

						/* Call myself recursively to process the substring. */
						netbuf_to_hostbuf(tmp_format,
								  net_buf, sub_buf,
								  &sub_buf_inc, &net_buf_inc);

						net_buf += net_buf_inc;

						*((int * *) var_ptr) = (int *) sub_buf;

					} else {
						*((int * *) var_ptr) = NULL;
					}

					pf += strlen(tmp_format) + sizeof(char);

				} else {
					/* It's a generic pointer. */
					/*
					 * Pointers are passed in the largest
					 * architectural size for a pointer.
					 */
					u64 = NET_TO_HOST64(*((u_int64_t *) net_buf));
					*((unsigned long * *) var_ptr) = (unsigned long *) (unsigned long) u64;
					net_buf += sizeof(u_int64_t);
					pf++;
				}
				break;

			case 's':
				if (*((u_int8_t *) net_buf)) {
					/* A valid string follows. */
					net_buf += sizeof(u_int8_t);

					*((char * *) var_ptr) = engine_strdup(net_buf);
					net_buf += strlen(net_buf) + sizeof(char);

				} else {
					/* The string pointer is NULL. */
					net_buf += sizeof(u_int8_t);

					*((char * *) var_ptr) = NULL;
				}
				pf++;
				break;

			case 't':
				if (*((u_int8_t *) net_buf)) {
					/* Valid strings follow. */
					net_buf += sizeof(u_int8_t);

					/*
					 * Count the number of strings so that
					 * we know how big to allocate the array
					 * of pointers to the strings.
					 */
					string = (char *) net_buf;
					count = 0;
					while (*string != '\0') {
						count++;
						string += strlen(string) +1;
					}

					strings = engine_alloc((count + 1) * sizeof(char *));
					if (strings != NULL) {
						*((char * * *) var_ptr) = strings;

						string = (char *) net_buf;
						for (i = 0; *string != '\0'; i++) {
							strings[i] = engine_strdup(string);
							net_buf += strlen(string) + sizeof(char);
							string = (char *) net_buf;
						}
						strings[count] = NULL;
						net_buf += sizeof(char);
					}

				} else {
					/*
					 * No strings passed.  Set the
					 * string array pointer to NULL.
					 */
					net_buf += sizeof(u_int8_t);

					*((char * * *) var_ptr) = NULL;
				}

				pf++;
				break;

			case 'v':
				value_type = (value_type_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
				net_buf += sizeof(u_int32_t);

				value_is_list = (boolean) *((u_int8_t *) net_buf);
				net_buf += sizeof(u_int8_t);

				net_buf = netbuf_to_value(net_buf,
							  (value_t *) var_ptr,
							  value_type,
							  value_is_list);
				pf++;
				break;

			case 'x':
				/*
				 * Byte strings are not passed as args.
				 * Pointers to them are, but not the
				 * bytes directly themsleves.
				 */
				LOG_ERROR("Byte strings are not passed as arguments. "
					  "Try using \"p{xnnn}\" for the byte string argument.\n");
				LOG_PROC_EXIT_INT(EINVAL);
				return EINVAL;
				break;

			default:
				LOG_SERIOUS("Format character %c slipped past the validater. "
					    "I don't know how to handle it.  I'm skipping it.\n", *pf);
				pf++;
		}
	}

	va_end(args);

	LOG_PROC_EXIT_INT(0);
	return 0;
}


