# -*- coding: utf-8 -*-
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Tests for utilities for the base directory implementation."""
import os
import sys

from dirspec import basedir
from dirspec.utils import get_env_path, get_special_folders, user_home
from dirspec.tests import BaseTestCase


class UtilsTestCase(BaseTestCase):
    """Test for the multiplatform directory utilities."""

    def test_user_home_is_utf8_bytes(self):
        """The returned path is bytes."""
        actual = user_home
        self.assert_utf8_bytes(actual)


class FakeShellConModule(object):
    """Override CSIDL_ constants."""

    CSIDL_PROFILE = 0
    CSIDL_LOCAL_APPDATA = 1
    CSIDL_COMMON_APPDATA = 2


class FakeShellModule(object):

    """Fake Shell Module."""

    def __init__(self):
        """Set the proper mapping between CSIDL_ consts."""
        self.values = {
            0: u'c:\\path\\to\\users\\home',
            1: u'c:\\path\\to\\users\\home\\appData\\local',
            2: u'c:\\programData',
        }

    # pylint: disable=C0103
    def SHGetFolderPath(self, dummy0, shellconValue, dummy2, dummy3):
        """Override SHGetFolderPath functionality."""
        return self.values[shellconValue]
    # pylint: enable=C0103


class TestBaseDirectoryWindows(BaseTestCase):
    """Tests for the BaseDirectory module."""

    def test_get_special_folders(self):
        """Make sure we can import the platform module."""
        import win32com.shell
        shell_module = FakeShellModule()
        self.patch(win32com.shell, "shell", shell_module)
        self.patch(win32com.shell, "shellcon", FakeShellConModule())
        special_folders = get_special_folders()
        self.assertTrue('Personal' in special_folders)
        self.assertTrue('Local AppData' in special_folders)
        self.assertTrue('AppData' in special_folders)
        self.assertTrue('Common AppData' in special_folders)

        self.assertTrue(special_folders['Personal'] == \
            shell_module.values[FakeShellConModule.CSIDL_PROFILE])
        self.assertTrue(special_folders['Local AppData'] == \
            shell_module.values[FakeShellConModule.CSIDL_LOCAL_APPDATA])
        self.assertTrue(special_folders['Local AppData'].startswith(
            special_folders['AppData']))
        self.assertTrue(special_folders['Common AppData'] == \
            shell_module.values[FakeShellConModule.CSIDL_COMMON_APPDATA])

        for val in special_folders.itervalues():
            self.assertIsInstance(val, str)
            val.decode('utf8')
        # Should not raise exceptions

    if sys.platform != 'win32':
        test_get_special_folders.skip = 'Only works on Win32'

    def test_get_data_dirs(self):
        """Check thet get_data_dirs uses pathsep correctly."""
        bad_sep = filter(lambda x: x not in os.pathsep, ":;")
        dir_list = ["A", "B", bad_sep, "C"]
        self.patch(os, "environ",
            dict(XDG_DATA_DIRS=os.pathsep.join(
                dir_list)))
        dirs = basedir.get_xdg_data_dirs()[1:]
        self.assertEqual(dirs, dir_list)

    def test_get_config_dirs(self):
        """Check thet get_data_dirs uses pathsep correctly."""
        bad_sep = filter(lambda x: x not in os.pathsep, ":;")
        dir_list = ["A", "B", bad_sep, "C"]
        self.patch(os, "environ",
            dict(XDG_CONFIG_DIRS=os.pathsep.join(
                dir_list)))
        dirs = basedir.get_xdg_config_dirs()[1:]
        self.assertEqual(dirs, dir_list)

    def set_fake_environ(self, key, value):
        """Set (and restore) a fake environ variable."""
        if key in os.environ:
            prev = os.environ[key]
            self.addCleanup(os.environ.__setitem__, key, prev)
        else:
            self.addCleanup(os.environ.__delitem__, key)
        os.environ[key] = value

    def unset_fake_environ(self, key):
        """Unset (and restore) a fake environ variable."""
        if key in os.environ:
            current_value = os.environ[key]
            self.addCleanup(os.environ.__setitem__, key, current_value)
            del(os.environ[key])

    def test_get_env_path_var(self):
        """Test that get_env_path transforms an env var."""
        fake_path = u"C:\\Users\\Ñandú"
        fake_env_var = "FAKE_ENV_VAR"

        mbcs_path = fake_path.encode(sys.getfilesystemencoding())
        utf8_path = fake_path.encode("utf-8")

        self.set_fake_environ(fake_env_var, mbcs_path)
        self.assertEqual(get_env_path(fake_env_var, "unexpected"), utf8_path)

    def test_get_env_path_no_var(self):
        """Test that get_env_path returns the default when env var not set."""
        fake_path = u"C:\\Users\\Ñandú"
        fake_env_var = "fake_env_var"
        default = fake_path.encode(sys.getfilesystemencoding())

        self.unset_fake_environ(fake_env_var)
        self.assertEqual(get_env_path(fake_env_var, default), default)
    test_get_env_path_var.skip = 'UnicodeEncodeError: bug #907053'
    test_get_env_path_no_var.skip = test_get_env_path_var.skip
