/*
 * Copyright (C) 2009 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 */

#if !defined (_DEE_H_INSIDE) && !defined (DEE_COMPILATION)
#error "Only <dee.h> can be included directly."
#endif

#ifndef _HAVE_DEE_MODEL_H
#define _HAVE_DEE_MODEL_H

#include <glib.h>
#include <glib-object.h>

G_BEGIN_DECLS

#define DEE_TYPE_MODEL (dee_model_get_type ())

#define DEE_MODEL(obj) \
        (G_TYPE_CHECK_INSTANCE_CAST ((obj), DEE_TYPE_MODEL, DeeModel))

#define DEE_IS_MODEL(obj) \
       (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DEE_TYPE_MODEL))

#define DEE_MODEL_GET_IFACE(obj) \
       (G_TYPE_INSTANCE_GET_INTERFACE(obj, dee_model_get_type (), DeeModelIface))

typedef struct _DeeModelIface DeeModelIface;
typedef struct _DeeModel DeeModel;

/**
 * DeeModelIter:
 *
 * The DeeModelIter structure is private and should only be used with the
 * provided #DeeModel API. It is owned by DeeModel and should not be freed.
 **/
typedef struct _DeeModelIter DeeModelIter;

struct _DeeModelIface
{
  GTypeInterface g_iface;

  /* Signals */
  void           (*row_added)       (DeeModel     *self,
                                     DeeModelIter *iter);

  void           (*row_removed)     (DeeModel     *self,
                                     DeeModelIter *iter);

  void           (*row_changed)     (DeeModel     *self,
                                     DeeModelIter *iter);

  /*< public >*/
  void           (* set_column_type)(DeeModel     *self,
                                     guint         column,
                                     GType         type);

  void           (* set_n_columns)  (DeeModel     *self,
                                     guint         n_columns);
  
  guint          (*get_n_columns)   (DeeModel *self);

  guint          (*get_n_rows)      (DeeModel *self);

  GType          (*get_column_type) (DeeModel *self,
                                     guint      column);

  void           (*clear)           (DeeModel *self);

  DeeModelIter*  (*append_valist)    (DeeModel *self,
                                      va_list    args);

  DeeModelIter*  (*prepend_valist)   (DeeModel *self,
                                      va_list    args);

  DeeModelIter*  (*insert_valist)    (DeeModel *self,
                                      gint       pos,
                                      va_list    args);

  DeeModelIter*  (*insert_before_valist) (DeeModel     *self,
                                          DeeModelIter *iter,
                                          va_list    args);

  void           (*remove)          (DeeModel     *self,
                                     DeeModelIter *iter);

  void           (*set_valist)      (DeeModel       *self,
                                     DeeModelIter   *iter,
                                     va_list          args);

  void           (*set_value)       (DeeModel       *self,
                                     DeeModelIter   *iter,
                                     guint            column,
                                     const GValue    *value);

  void           (*set_value_silently) (DeeModel       *self,
                                        DeeModelIter   *iter,
                                        guint            column,
                                        const GValue    *value);
  
  void           (*get_valist)      (DeeModel     *self,
                                     DeeModelIter *iter,
                                     va_list        args);

  void           (*get_value)       (DeeModel     *self,
                                     DeeModelIter *iter,
                                     guint          column,
                                     GValue        *value);
  
  DeeModelIter* (*get_first_iter)  (DeeModel     *self);

  DeeModelIter* (*get_last_iter)   (DeeModel     *self);

  DeeModelIter* (*get_iter_at_row) (DeeModel     *self,
                                     guint          row);

  gboolean       (*get_bool)        (DeeModel     *self,
                                     DeeModelIter *iter,
                                     guint          column);

  guchar         (*get_uchar)       (DeeModel     *self,
                                     DeeModelIter *iter,
                                     guint          column);

  gint           (*get_int)         (DeeModel     *self,
                                     DeeModelIter *iter,
                                     guint          column);

  guint          (*get_uint)        (DeeModel     *self,
                                     DeeModelIter *iter,
                                     guint          column);

  gint64         (*get_int64)       (DeeModel     *self,
                                     DeeModelIter *iter,
                                     guint          column);

  guint64        (*get_uint64)      (DeeModel     *self,
                                     DeeModelIter *iter,
                                     guint          column);

  gdouble        (*get_double)      (DeeModel     *self,
                                     DeeModelIter *iter,
                                     guint          column);

  const gchar*   (*get_string)      (DeeModel     *self,
                                     DeeModelIter *iter,
                                     guint          column);

  DeeModelIter*  (*next)            (DeeModel     *self,
                                     DeeModelIter *iter);

  DeeModelIter*  (*prev)            (DeeModel     *self,
                                     DeeModelIter *iter);

  gboolean       (*is_first)        (DeeModel     *self,
                                     DeeModelIter *iter);

  gboolean       (*is_last)         (DeeModel     *self,
                                     DeeModelIter *iter);

  gint           (*get_position)    (DeeModel     *self,
                                     DeeModelIter *iter);

  void           (*freeze_signals)  (DeeModel     *self);
  
  void           (*thaw_signals)    (DeeModel     *self);
         
};

/**
 * dee_model_get_type:
 *
 * The GType of #DeeModel
 *
 * Return value: the #GType of #DeeModel
 **/
gboolean        dee_model_check_type             (GType     type);

GType           dee_model_get_type_for_signature (gchar     chr);

gchar*          dee_model_build_col_spec  (DeeModel *self);

void            dee_model_set_column_type (DeeModel *self,
                                           guint     column,
                                           GType     type);

void            dee_model_set_n_columns   (DeeModel *self,
                                           guint     n_columns);

GType           dee_model_get_type        (void) G_GNUC_CONST;

guint           dee_model_get_n_columns   (DeeModel *self);

guint           dee_model_get_n_rows      (DeeModel *self);

GType           dee_model_get_column_type (DeeModel *self,
                                            guint      column);

void            dee_model_clear           (DeeModel *self);

DeeModelIter*   dee_model_append          (DeeModel *self,
                                           ...);

DeeModelIter*   dee_model_append_valist   (DeeModel *self,
                                           va_list   args);

DeeModelIter*   dee_model_prepend         (DeeModel *self,
                                           ...);

DeeModelIter*   dee_model_prepend_valist  (DeeModel *self,
                                           va_list   args);

DeeModelIter*   dee_model_insert           (DeeModel *self,
                                            gint       pos,
                                            ...);

DeeModelIter*   dee_model_insert_valist    (DeeModel *self,
                                            gint       pos,
                                            va_list    args);

DeeModelIter*   dee_model_insert_before    (DeeModel     *self,
                                             DeeModelIter *iter,
                                             ...);

DeeModelIter*   dee_model_insert_before_valist (DeeModel     *self,
                                                DeeModelIter *iter,
                                                va_list    args);

void            dee_model_remove          (DeeModel     *self,
                                            DeeModelIter *iter);

void            dee_model_set             (DeeModel     *self,
                                            DeeModelIter *iter,
                                            ...);

void            dee_model_set_valist      (DeeModel       *self,
                                            DeeModelIter   *iter,
                                            va_list          args);

void            dee_model_set_value       (DeeModel       *self,
                                            DeeModelIter   *iter,
                                            guint            column,
                                            const GValue    *value);

void            dee_model_set_value_silently (DeeModel       *self,
                                              DeeModelIter   *iter,
                                              guint            column,
                                              const GValue    *value);

void            dee_model_get             (DeeModel     *self,
                                            DeeModelIter *iter,
                                            ...);

void            dee_model_get_valist      (DeeModel     *self,
                                            DeeModelIter *iter,
                                            va_list        args);

void            dee_model_get_value       (DeeModel     *self,
                                            DeeModelIter *iter,
                                            guint          column,
                                            GValue        *value);

DeeModelIter * dee_model_get_first_iter  (DeeModel     *self);

DeeModelIter * dee_model_get_last_iter   (DeeModel     *self);

DeeModelIter * dee_model_get_iter_at_row (DeeModel     *self,
                                            guint          row);

gboolean        dee_model_get_bool        (DeeModel     *self,
                                            DeeModelIter *iter,
                                            guint          column);

guchar          dee_model_get_uchar       (DeeModel     *self,
                                            DeeModelIter *iter,
                                            guint          column);

gint            dee_model_get_int         (DeeModel     *self,
                                            DeeModelIter *iter,
                                            guint          column);

guint           dee_model_get_uint        (DeeModel     *self,
                                            DeeModelIter *iter,
                                            guint          column);

gint64          dee_model_get_int64       (DeeModel     *self,
                                            DeeModelIter *iter,
                                            guint          column);

guint64         dee_model_get_uint64      (DeeModel     *self,
                                            DeeModelIter *iter,
                                            guint          column);

gdouble         dee_model_get_double      (DeeModel     *self,
                                            DeeModelIter *iter,
                                            guint          column);

const gchar *   dee_model_get_string      (DeeModel     *self,
                                            DeeModelIter *iter,
                                            guint          column);

DeeModelIter * dee_model_next            (DeeModel     *self,
                                            DeeModelIter *iter);

DeeModelIter * dee_model_prev            (DeeModel     *self,
                                            DeeModelIter *iter);

gboolean        dee_model_is_first        (DeeModel     *self,
                                            DeeModelIter *iter);

gboolean        dee_model_is_last         (DeeModel     *self,
                                            DeeModelIter *iter);

gint            dee_model_get_position    (DeeModel     *self,
                                            DeeModelIter *iter);

void            dee_model_freeze_signals  (DeeModel     *self);

void            dee_model_thaw_signals    (DeeModel     *self);

G_END_DECLS

#endif /* _HAVE_DEE_MODEL_H */
