/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef DECIBEL_DAEMON_ACCOUNTCONNECTORBASE_H
#define DECIBEL_DAEMON_ACCOUNTCONNECTORBASE_H

#include <Decibel/AccountData>

#include <QtCore/QVariantMap>

/**
 * @brief A API for connectors to password management systems.
 *
 *
 * The AccountConnectorBase is a abstract base class defining an interface
 * used by the AccountManager to interact with the users preferred
 * password management system.
 *
 * @author Tobias Hunger <info@basyskom.de>
 */
class AccountConnectorBase : public QObject
{
    Q_OBJECT
public:
    AccountConnectorBase(QObject * parent);
    /** @brief Destructor. */
    virtual ~AccountConnectorBase();

    /**
     * @brief Open account data storage.
     */
    virtual void openStorage() = 0;

    /**
     * @brief Close the account storage.
     */
    virtual void closeStorage() = 0;

    /**
     * @brief Does a account with the given Id exist?
     * @param id The id of the account.
     * @returns true if the account exists, false otherwise.
     *
     * Checks whether an account of the given Id exists.
     */
    virtual bool hasAccount(const uint id) const = 0;
    /**
     * @brief Store an account.
     * @param nv_pairs The account data.
     * @returns The id of the new account.
     *
     * Stores an account with the given set of data and returns the id of
     * that account.
     */
    virtual uint storeAccount(const QVariantMap & nv_pairs) = 0;
    /**
     * @brief Update the data stored on an account.
     * @param id The id of the account to update.
     * @param nv_pairs The new data.
     * @returns true if the update was successful.
     *
     * Updates the data on an account. It replaces the the stored data
     * with the given data.
     */
    virtual void updateAccount(const uint id, const QVariantMap & nv_pairs) = 0;
    /**
     * @brief Get the data stored for an account of the given id.
     * @param id The id of the account queried.
     * @returns The name value pairs stored for the account. The QVariantMap
     * is empty if the account was not stored.
     *
     * Get the data stored for an account.
     */
    virtual QVariantMap getAccount(const uint id) const = 0;
    /**
     * @brief Delete an account.
     * @param id The id of the account to delete.
     * @returns true if the account was deleted and false otherwise.
     *
     * Delete an account.
     */
    virtual bool deleteAccount(const uint id) = 0;
    /**
     * @brief Get a list of defined accounts.
     * @returns A list of defined account ids.
     *
     * Get a list of defined accounts.
     */
    virtual QList<uint> accountIds() const = 0;
    /**
     * @brief Find accounts matching a criteria.
     * @param nv_pairs The criteria to search for
     * @returns A list of ids with accounts matching the search criteria. This
     *          list is empty if no account matches all criteria.
     *
     * Search all accounts matching all given criteria.
     */
    virtual QList<uint> findAccounts(const QVariantMap & nv_pairs) const = 0;

    /**
     * @brief Set a key to value on one account.
     * @param id The account id
     * @param key The key to set.
     * @param value The value to set.
     *
     * Sets the given key to the given value on the account id.
     *
     * Removes the key when a Null QVariant is given.
     */
    virtual bool setValue(const uint id,
                          const QString & key, const QVariant & value) = 0;

    /**
     * @brief Find the protocol used by an account.
     * @param id the id of the account.
     * @returns The protocol of the account.
     *
     * Find the protocol used by an account.
     */
    virtual QString protocol(const uint id)
    {
        QVariantMap account_data = getAccount(id);
        return account_data[Decibel::name_protocol].toString();
    }

Q_SIGNALS:
    /**
     * @brief Signal account data availability.
     * @param available True if the account data is available and false
     *     otherwise.
     *
     * Account data may not be available right on start up, e.g. due
     * to the user not having opened their password management system yet.
     * This signal is used to notify Decibel of the account data becoming
     * (un-) available.
     */
    void accountDataAvailable(const bool available);

protected:
private:
};

#endif // header guard
