# Copyright (C) 2008-2011  Canonical, Ltd.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Command line user interface."""

from __future__ import absolute_import, unicode_literals

__metaclass__ = type
__all__ = [
    'main',
    ]


import sys
import dbus
import gobject
import argparse
import textwrap

from dbus.mainloop.glib import DBusGMainLoop
from operator import mod

from computerjanitorapp import __version__, setup_gettext
from computerjanitorapp.terminalsize import get_terminal_size
from computerjanitorapp.utilities import format_size
from computerjanitord.service import DBUS_INTERFACE_NAME
_ = setup_gettext()


class Options:
    """Command line option parser."""
    def __init__(self, runner):
        """Parse the command line options.

        :param runner: The class implementing the sub-commands.
        :type runner: `Runner`
        """
        self.runner = runner
        # Parse the arguments.
        self.parser = argparse.ArgumentParser(
            description=_("""\
            Find and remove cruft from your system.

            Cruft is anything that shouldn't be on your system, but is.
            Stretching the definition, cruft is also things that should be on
            your system but aren't."""))
        self.parser.add_argument(
            '--version', action='version',
            # XXX barry 2010-08-23: LP: #622720
            version=mod(_('Computer Janitor %(version)s'),
                        dict(version=__version__)))
        subparser = self.parser.add_subparsers(title='Commands')
        # The 'find' subcommand.
        command = subparser.add_parser(
            'find', help=_('Find and display all cruft found on your system.'))
        command.add_argument(
            '-v', '--verbose', action='store_true',
            help=_("""\
            Display a detailed explanation for each piece of cruft found."""))
        command.add_argument(
            '-i', '--ignored', action='store_true',
            help=_('Find and display only the ignored cruft.'))
        command.add_argument(
            '-r', '--removable', action='store_true',
            help=_('Find and display only the removable cruft.'))
        command.add_argument(
            '-s', '--short', action='store_true',
            help=_('Display only the package names.  Do not use with -v.'))
        command.set_defaults(func=self.runner.find)
        # The 'ignore' subcommand.
        command = subparser.add_parser(
            'ignore', help=_("""\
            Ignore a piece of cruft so that it is not cleaned up."""))
        command.add_argument(
            'cruft', nargs=1,
            help=_('The name of the cruft to ignore.'))
        command.set_defaults(func=self.runner.ignore)
        # The 'unignore' subcommand.
        command = subparser.add_parser(
            'unignore', help=_("""\
            Unignore a piece of cruft so that it will be cleaned up."""))
        command.add_argument(
            'cruft', nargs=1,
            help=_('The name of the cruft to unignore.'))
        command.set_defaults(func=self.runner.unignore)
        # The 'clean' subcommand.
        command = subparser.add_parser(
            'clean',
            help=_('Remove the selected cruft from the system.'))
        command.add_argument(
            '-a', '--all', action='store_true',
            help=_('Clean up all unignored cruft.'))
        command.add_argument(
            '-v', '--verbose', action='store_true',
            help=_("Provide more details on what's being cleaned."))
        command.add_argument(
            'cruft', nargs='?',
            help=_("""\
            The name of the cruft to clean up.  Do not use if specifying
            --all."""))
        command.set_defaults(func=self.runner.clean)
        # Parse the arguments and execute the subcommand.
        self.arguments = self.parser.parse_args()


class Runner:
    """Implementations of subcommands."""

    def __init__(self):
        # Connect to the dbus service.
        system_bus = dbus.SystemBus()
        proxy = system_bus.get_object(DBUS_INTERFACE_NAME, '/')
        self.janitord = dbus.Interface(
            proxy, dbus_interface=DBUS_INTERFACE_NAME)
        # Connect to the signal the server will emit when cleaning up.
        self.janitord.connect_to_signal('cleanup_status', self._clean_working)
        # This will get backpatched by __main__.  We need it to produce error
        # messages from the argparser.
        self.options = None
        # The main loop for asynchronous calls and signal reception.
        self.loop = gobject.MainLoop()

    def _error(self, message):
        """Generate a parser error and exit.

        :param message: The error message.
        :type message: string
        """
        self.options.parser.error(message)
        # No return.

    def find(self, arguments):
        """Find and display all cruft.

        :param arguments: Command line options.
        """
        # Cruft will be prefixed by 'removable' if it is not being ignored.
        ignored = set(self.janitord.ignored())
        cruft_names = set(self.janitord.find())
        # Filter names based on option flags.
        if arguments.ignored:
            cruft = sorted(cruft_names & ignored)
        elif arguments.removable:
            cruft = sorted(cruft_names - ignored)
        else:
            cruft = sorted(cruft_names)
        # The prefix will either be 'ignored' or 'removable' however this
        # string will be translated, so calculate the prefix size in the
        # native language, then add two columns of separator, followed by the
        # cruft name.
        prefixi = _('ignored')
        prefixr = _('removable')
        prefix_width = max(len(prefixi), len(prefixr))
        # Long, short, shorter display.
        if arguments.verbose and arguments.short:
            self._error('Use either -s or -v but not both.')
        if arguments.short:
            # This is the shorter output.
            for name in cruft:
                print name
        elif not arguments.verbose:
            # This is the short output.
            for name in cruft:
                prefix = (prefixi if name in ignored else prefixr)
                # 10 spaces for the prefix
                print '{0:{1}}  {2}'.format(prefix, prefix_width, name)
        else:
            # This is the verbose output.  Start by getting the terminal's
            # size, though all we care about is the width.
            rows, columns = get_terminal_size()
            width = ((80 if columns in (None, 0) else columns)
                     - prefix_width # space for prefix
                     - 2            # separator
                     - 1            # avoid last column, some terminals wrap
                     )
            margin = ' ' * (prefix_width + 2)
            for name in cruft:
                prefix = (prefixi if name in ignored else prefixr)
                # 10 spaces for the prefix
                print '{0:{1}}  {2}'.format(prefix, prefix_width, name)
                # Print some details about the cruft.
                cruft_type, disk_usage = self.janitord.get_details(name)
                print '{0}{1} of type {2}'.format(
                    margin, format_size(disk_usage), cruft_type)
                # Get the description, wrap it to the available columns, and
                # display it line-by-line with the proper amount of leading
                # spaces (prefix_width + 2).
                description = self.janitord.get_description(name)
                if len(description) == 0:
                    print
                    continue
                paragraphs = description.split('\n\n')
                for paragraph in paragraphs:
                    for line in textwrap.wrap(paragraph, width):
                        # 2010-02-09 barry: The original code forced the
                        # output to utf-8, claiming that was necessary to
                        # "write out stuff even when it's going to somewhere
                        # like a pipe [...] ignor[ing] the user's desired
                        # charset, which is bad, bad, bad...".  This makes me
                        # pretty uncomfortable, so I'd like to see a bug
                        # report before I copy that from the previous
                        # implementation.
                        print '{0}{1}'.format(margin, line)
                    # Paragraph separator.
                    print

    def ignore(self, arguments):
        """Ignore some cruft.

        :param arguments: Command line options.
        """
        assert len(arguments.cruft) == 1, 'Unexpected arguments'
        self.janitord.ignore(arguments.cruft[0])
        self.janitord.save()

    def unignore(self, arguments):
        """Unignore some cruft.

        :param arguments: Command line options.
        """
        assert len(arguments.cruft) == 1, 'Unexpected arguments'
        self.janitord.unignore(arguments.cruft[0])
        self.janitord.save()

    def _clean_reply(self):
        """The 'clean' operation has completed successfully."""
        self.loop.quit()
        print 'done.'

    def _clean_error(self, exception):
        """The 'clean' operation has failed."""
        self.loop.quit()
        print 'dbus service error:', exception

    def _clean_working(self, cruft):
        """The 'clean' operation is in progress.

        :param cruft: The cruft that is being cleaned up.
        :type cruft: string
        """
        verbose = self.options.arguments.verbose
        if cruft == '':
            # We're done.
            if not verbose:
                sys.stdout.write(' ')
                sys.stdout.flush()
        else:
            if verbose:
                print 'Working on', cruft
            else:
                sys.stdout.write('.')
                sys.stdout.flush()

    def clean(self, arguments):
        """Clean up the cruft.

        :param arguments: Command line options.
        """
        if arguments.all:
            # You can't specify both --all and a cruft name.
            if arguments.cruft is not None:
                self._error('Specify a cruft name or --all, but not both')
                # No return.
            all_cruft = set(self.janitord.find())
            ignored = set(self.janitord.ignored())
            cleanable_cruft = tuple(all_cruft - ignored)
        else:
            if arguments.cruft is None:
                self._error('You must specify a cruft name, or use --all')
                # No return.
            cleanable_cruft = (arguments.cruft,)
        # Make the asynchronous call because this can take a long time.  We'll
        # get status updates periodically.  Note however that even though this
        # is asynchronous, dbus still expects a response within a certain
        # amount of time.  We have no idea how long it will take to clean up
        # the cruft though, so just crank the timeout up to some insanely huge
        # number (of seconds).
        self.janitord.clean(cleanable_cruft,
                            reply_handler=self._clean_reply,
                            error_handler=self._clean_error,
                            # If it takes longer than an hour, we're screwed.
                            timeout=3600)
        # Start the main loop.  This will exit when the remote operation is
        # complete.
        if not arguments.verbose:
            sys.stdout.write('processing')
            sys.stdout.flush()
        self.loop.run()


def main():
    # We'll need a main loop to receive status signals from the dbus service.
    # Don't start the main loop yet though, since we only need it for 'clean'
    # commands.
    DBusGMainLoop(set_as_default=True)
    runner = Runner()
    options = Options(runner)
    # Backpatch runner because of circular references.
    runner.options = options
    # Execute the subcommand.
    options.arguments.func(options.arguments)
