#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""Tests the debconf forwarding"""

import logging
import os
import subprocess
import tempfile
import unittest

import debian.deb822
import gobject

import sys
sys.path.insert(0, "../..")

DEBUG=False

from aptdaemon.debconf import DebconfProxy

class DebconfTestBasic(unittest.TestCase):

    def _stop(self):
        self.proxy.stop()
        self.loop.quit()

    def setUp(self):
        self.loop = gobject.MainLoop()
        self.debconf_socket_path = tempfile.mktemp(prefix="debconf-socket-")
        editor = r"sed -ie 's/\(aptdaemon\/test=\).*/\1\"lalelu\"/'"
        os.environ["EDITOR"] = editor
        self.proxy = DebconfProxy("editor", self.debconf_socket_path)
        self.proxy.start()

    def _spawn_config_script(self, config_db_path):
        env = {}
        env["DEBCONF_DB_REPLACE"] = "File{%s}" % config_db_path
        env["DEBIAN_FRONTEND"] = "passthrough"
        env["DEBCONF_PIPE"] = self.debconf_socket_path
        if DEBUG:
            env["DEBCONF_DEBUG"] = ".*"
        proc = subprocess.Popen("debconf/aptdaemon.config", env=env)
        return proc.pid

    def testBasic(self):
        def config_done(pid, cond):
            self.assertEquals(cond, 0,
                              "Config script failed: %s" % os.WEXITSTATUS(cond))
            self._stop()
        debconf_db_path = tempfile.mktemp(suffix=".dat", prefix="config-basic-")
        pid = self._spawn_config_script(debconf_db_path)
        gobject.child_watch_add(pid, config_done)
        self.loop.run()
        # Check the results
        self._check_database(debconf_db_path)

    def testSerial(self):
        """Run several config scripts in a row."""
        def config_done(pid, cond):
            self.assertEquals(cond, 0,
                              "Config script failed: %s" % os.WEXITSTATUS(cond))
            self.config_scripts -= 1
            if self.config_scripts <= 0:
                self._stop()
            else:
                pid = self._spawn_config_script(debconf_db_path)
                gobject.child_watch_add(pid, config_done)
        debconf_db_path = tempfile.mktemp(suffix=".dat", prefix="config-row-")
        self.config_scripts = 10
        pid = self._spawn_config_script(debconf_db_path)
        gobject.child_watch_add(pid, config_done)
        self.loop.run()
        # Check the results
        self._check_database(debconf_db_path)

    def testRace(self):
        def config_done(pid, cond):
            self.assertEquals(cond, 0,
                              "Config script failed: %s" % os.WEXITSTATUS(cond))
            self.workers -= 1
            if self.workers <= 0:
                self._stop()
        debconf_dbs = []
        self.workers = 0
        for i in range(10):
            debconf_db_path = tempfile.mktemp(suffix=".dat",
                                              prefix="config-race-")
            pid = self._spawn_config_script(debconf_db_path)
            gobject.child_watch_add(pid, config_done)
            debconf_dbs.append(debconf_db_path)
            self.workers += 1
        self.loop.run()
        # Check the results
        for db_path in debconf_dbs:
            self._check_database(db_path)

    def _check_database(self, path):
        db_file = open(path)
        db = debian.deb822.Deb822(db_file)
        db_file.close()
        self.assertEquals(db["Value"], "lalelu")
        os.remove(path)

    def tearDown(self):
        os.remove(self.debconf_socket_path)
        self.proxy = None
        self.loop.quit()
        self.loop = None


if __name__ == "__main__":
    if DEBUG:
        logging.basicConfig(level=logging.DEBUG)
    unittest.main()

# vim: ts=4 et sts=4
