// -*- C++ -*- (c) 2007, 2008 Petr Rockai <me@mornfall.net>

#include <adept/util.h>
#include <adept/packagedata.h>

#include <QtCore/QTimer>
#include <QtGui/QApplication>
#include <kcmdlineargs.h>

#include <errno.h>

#include <apt-pkg/acquire.h>
#include <adept/pkgsystem.h>
#include <ept/core/desktop.h>

#include <adept/processevents.h>

// package display
#include <adept/packagelist.h>
#include <adept/tokenmodel.h>
#include <adept/packagedetails.h>
#include <adept/previewwidget.h>
#include <adept/applicationswidget.h>

// download/commit-time widgets
#include <adept/progresswidget.h>
#include <adept/downloadprogress.h>
#include <adept/guidpkgpm.h>
#include <adept/commitwidget.h>

// search/filtering
#include <adept/statefilter.h>

// general-purpose widgets
#include <adept/sidebar.h>
#include <QtGui/QPushButton>
#include <QtGui/QStackedWidget>
#include <KXmlGuiWindow>
#include <KIcon>
#include <KVBox>
#include <KLineEdit>
#include <KStandardAction>
#include <KActionCollection>
#include <KAction>

#ifndef ADEPT_MANAGER_H
#define ADEPT_MANAGER_H

using namespace ept::core;
using namespace adept;

class TagSearch : public KVBox
{
    Q_OBJECT
public:
    typedef std::map< std::string, int > Tags;
    typedef QMap< QLabel *, std::string > LabelMap;
    typedef QMap< std::string, QLabel * > AntiLabelMap;

protected:
    PackageData &d;
    QSet< QWidget * > m_widgets;
    LabelMap m_tag, m_facet, m_not;
    AntiLabelMap m_tagLabel, m_notLabel;
    Tags m_tags;
    std::set< std::string > m_excluded, m_included;
    QWidget *m_spacer;
    QLabel *m_clear;

public:

    TagSearch( PackageData &_d, QWidget *p ) : KVBox( p ), d( _d ), m_spacer( 0 )
    {
        layout()->setMargin( 5 );
        QFontMetrics fm( font() );
        setMinimumWidth( fm.averageCharWidth() * 35 );

        KHBox *title = new KHBox( this );
        new QWidget( title ); // spacer
        QLabel *l = new QLabel( title );
        l->setText( "<b>Relevant tags</b>" );
        l->setSizePolicy( QSizePolicy::Preferred , QSizePolicy::Fixed );
        m_clear = new QLabel( title );
        m_clear->setText( "<qt>&nbsp;&nbsp;[clear]</qt>" );
        m_clear->setSizePolicy( QSizePolicy::Preferred , QSizePolicy::Fixed );
        m_clear->installEventFilter( this );
        m_clear->setEnabled( false );
        new QWidget( title ); // spacer
        setSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed );
        fill();
    }

    // FIXME ignores colour scheme...
    QColor relevanceColor( std::string t ) {
        int rel = m_tags[ t ];
        // std::cerr << t << " -- " << rel << std::endl;
        int shade = 200 - 43 * log( rel + 1 );
        if ( shade < 0 ) shade = 0;
        if ( shade > 200 ) shade = 200;
        return QColor::fromRgb( shade, shade, shade );
    }

    bool eventFilter( QObject *o, QEvent *e )
    {
        // mouseover highlight + active tag logic
        if ( QLabel *l = dynamic_cast< QLabel * >( o ) ) {
            std::string t;
            bool non;
            QPalette p = l->palette();

            if ( l == m_clear ) {
                if ( active() && e->type() == QEvent::Enter ) {
                    qDebug() << "activating clear button...";
                    p.setColor( QPalette::WindowText, Qt::red );
                } else if ( e->type() == QEvent::Leave ) {
                    p.setColor( QPalette::WindowText,
                                palette().color( QPalette::WindowText ) );
                } else if ( e->type() == QEvent::MouseButtonRelease ) {
                    QTimer::singleShot( 0, this, SLOT( runSourcesEditor() ) );
                }
                l->setPalette( p );
                return false;
            }

            if ( m_tag.find( l ) == m_tag.end() ) {
                t = m_not[ l ];
                non = true;
            } else {
                t = m_tag[ l ];
                non = false;
            }

            if ( e->type() == QEvent::Enter ) {
                p.setColor( QPalette::WindowText, Qt::blue );
            } else if ( e->type() == QEvent::Leave ) {
                p.setColor( QPalette::WindowText, relevanceColor( t ) );
            } else if ( e->type() == QEvent::MouseButtonRelease ) {
                // FIXME hardcoded XT prefix... Aaargs my eyes!
                if ( non ) {
                    m_included.erase( "XT" + t );
                    if ( m_excluded.count( "XT" + t ) )
                        m_excluded.erase( "XT" + t );
                    else
                        m_excluded.insert( "XT" + t );
                } else {
                    m_excluded.erase( "XT" + t );
                    if ( m_included.count( "XT" + t ) )
                        m_included.erase( "XT" + t );
                    else
                        m_included.insert( "XT" + t );
                }
                QTimer::singleShot( 0, this, SIGNAL( changed() ) );
                return false;
            }
            l->setPalette( p );
        }
        return false;
    }

    void clear() {
        delete m_spacer;
        m_spacer = 0;
        QSet< QWidget * >::iterator i;
        for ( i = m_widgets.begin(); i != m_widgets.end(); ++i ) {
            delete *i;
        }
        m_widgets.clear();
        m_tag.clear();
        m_facet.clear();
    }

    Tags initialTags() {
        std::set< std::string > t;
        Tags tags;
        t.insert( "role::program" );
        t.insert( "role::shared-lib" );
        t.insert( "role::devel-lib" );
        t.insert( "role::plugin" );

        t.insert( "use::browsing" );
        t.insert( "use::editing" );
        t.insert( "use::configuring" );
        t.insert( "use::converting" );
        t.insert( "use::learning" );
        t.insert( "use::gameplaying" );

        t.insert( "interface::x11" );
        t.insert( "interface::web" );
        t.insert( "interface::commandline" );
        t.insert( "interface::daemon" );

        for ( std::set< std::string >::iterator i = t.begin();
              i != t.end(); ++ i) {
            tags.insert( std::make_pair( *i, -1 ) );
        }
        return tags;
    }

    std::string facet( std::string t ) {
        return std::string( t, 0, t.find( "::" ) );
    }

    std::string tag( std::string t ) {
        return std::string( t, t.find( "::" ) + 2, std::string::npos );
    }

    void fill( Tags tags = Tags() ) {
        clear();
        if ( tags.empty() )
            tags = initialTags();

        std::set< std::string >::iterator i;
        Tags fixed;

        // FIXME inconsistencies about XT prefix
        for ( i = m_included.begin(); i != m_included.end(); ++i )
            fixed[ std::string( *i, 2, std::string::npos ) ] = -1;
        for ( i = m_excluded.begin(); i != m_excluded.end(); ++i )
            fixed[ std::string( *i, 2, std::string::npos ) ] = -1;

        m_tags.clear();
        std::set_union( tags.begin(), tags.end(),
                        fixed.begin(), fixed.end(),
                        std::inserter( m_tags, m_tags.begin() ) );

        m_clear->setEnabled( active() );
        QPalette p = palette();
        std::string lastFacet = "";
        for ( Tags::const_iterator i = m_tags.begin(); i != m_tags.end(); ++i ) {
            std::string t = i->first;
            if ( lastFacet != facet( t ) ) {
                QLabel *f = new QLabel( this );
                f->setText( u8( facet( t ) + ":" ) );
                lastFacet = facet( t );
                m_facet[ f ] = facet( t );
                p.setColor( QPalette::WindowText, 
                            QColor::fromRgb( 120, 50, 50 ) );
                f->setPalette( p );
                m_widgets.insert( f );
            }

            KHBox *b = new KHBox( this );
            b->setSizePolicy( QSizePolicy::Preferred, QSizePolicy::Fixed );
            QLabel *l = new QLabel( b );
            l->setSizePolicy( QSizePolicy::Preferred, QSizePolicy::Fixed );
            QLabel *no = new QLabel( b );
            no->setSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed );

            m_tag[ l ] = t;
            m_not[ no ] = t;
            m_tagLabel[ t ] = l;
            m_notLabel[ t ] = no;
            m_widgets.insert( b );

            QString txt = "<qt>&nbsp;&nbsp;&nbsp;";
            if ( m_included.count( "XT" + t ) )
                txt += "<b>";
            txt += u8( tag( t ) );
            if ( m_included.count( "XT" + t ) )
                txt += "</b>";
            txt += "</qt>";

            l->setText( txt );
            l->installEventFilter( this );

            txt = "<qt>&nbsp;";
            if ( m_excluded.count( "XT" + t ) )
                txt += "<b>";
            txt += "[not]";
            if ( m_excluded.count( "XT" + t ) )
                txt += "</b>";
            no->setText( txt );

            no->setAlignment( Qt::AlignRight );
            no->installEventFilter( this );
            no->setToolTip( i18n( "Exclude packages tagged %1 from search.",
                                  u8( t ) ) );
            p.setColor( QPalette::WindowText, relevanceColor( t ) );
            if ( !m_excluded.count( "XT" + t )
                 && !m_included.count( "XT" + t ) ) {
                l->setPalette( p );
                no->setPalette( p );
            }
        }

        m_spacer = new QWidget();
    }

    bool active() {
        return ! ( m_included.empty() && m_excluded.empty() );
    }

    void updateQuery( xapian::Query &q ) {
        q.addTerms( m_included );
        q.addTerms( m_excluded, true );
    }

    void update( xapian::Query q ) {
        fill( q.relevantTags( 10 ) );
    }
public Q_SLOTS:

    void runSourcesEditor() {
        m_included.clear();
        m_excluded.clear();
        changed();
    }

Q_SIGNALS:
    void changed();
};

class SearchWidget : public KVBox
{
    Q_OBJECT
protected:
    bool partial;
    PackageData &d;
    KLineEdit *m_searchLine;
    StateFilterWidget *m_states;
    TagSearch *m_tags;
    QTimer *m_searchTimer;
    bool m_longTimeout;
    bool m_sort;
    int m_cut;

public:
    SearchWidget( PackageData &_d, QWidget *p ) : KVBox( p ), d( _d )
    {
        layout()->setSpacing( 5 );
        m_searchLine = new KLineEdit( this );
        m_searchLine->setSizePolicy( QSizePolicy::Preferred,
                                     QSizePolicy::Fixed );
        m_searchLine->setClearButtonShown( true );
        m_searchLine->setClickMessage( i18n( "Search packages" ) );
        setFocusProxy( m_searchLine );
        connect( m_searchLine, SIGNAL( returnPressed() ),
                 this, SIGNAL( searchChanged() ) );
        connect( m_searchLine, SIGNAL( textChanged( const QString & ) ),
                 this, SLOT( searchTextChanged() ) );

        partial = false;
        m_longTimeout = false;
        m_searchTimer = new QTimer();
        m_searchTimer->setSingleShot( true );
        connect( m_searchTimer, SIGNAL( timeout() ),
                 this, SLOT( searchTextTimeout() ) );

        m_states = new StateFilterWidget( this );
        connect( m_states, SIGNAL( changed() ),
                 this, SIGNAL( searchChanged() ) );

        m_tags = new TagSearch( d, this );
        connect( m_tags, SIGNAL( changed() ),
                 this, SIGNAL( searchChanged() ) );

        new QWidget( this ); // filler
    }

    bool haveFilters() {
        return m_states->stateMask() != StateFilter::fullMask;
    }

    template< typename L >
    void fillFilter( TokenModel *m, L l ) {
        if ( m_sort ) {
            if ( m_cut ) {
                m->fillFrom( list::sort( list::take( m_cut, list::filter( l, StateFilter( d, m_states->stateMask() ) ) ) ), Validator( d ) );
            } else {
                m->fillFrom( list::sort( list::filter( l, StateFilter( d, m_states->stateMask() ) ) ), Validator( d ) );
            }
            } else {
            if ( m_cut ) {
                m->fillFrom( list::take( m_cut, list::filter( l, StateFilter( d, m_states->stateMask() ) ) ), Validator( d ) );
            } else {
                m->fillFrom( list::filter( l, StateFilter( d, m_states->stateMask() ) ), Validator( d ) );
            }
        }
    }

Q_SIGNALS:
    void searchChanged();

public Q_SLOTS:
    void searchTextChanged() {
        m_longTimeout = false;
        m_searchTimer->start( 200 );
    }

    void searchTextTimeout() {
        if ( m_longTimeout ) {
            partial = false;
            searchChanged();
        } else {
            m_longTimeout = true;
            m_searchTimer->start( 300 );
            partial = true;
            searchChanged();
        }
    }

    void fill( TokenModel *m ) {
        QString txt = m_searchLine->text();
	int minQuerySz = 4; // query strings need to be 4 chars or bigger
        m_cut = 200;
        if ( txt.length() < minQuerySz && !m_tags->active() ) {
            m_tags->fill(); // initial tags
            if ( haveFilters() ) {
                // FIXME how to propagate a warning that we only show
                // first 1000 hits?
                m_cut = 1000;
                m_sort = true;
                fillFilter( m, d.pkgs.list< package::Name >() );
            } else {
                m->clear();
            }
            return;
        }
        m_sort = false;
        // maybe use toUtf8 intsead?
        std::string query = txt.length() < minQuerySz ? "" : txt.toLocal8Bit().data();
        xapian::Query q = d.xap.query( query, true );
        m_tags->updateQuery( q );
        if ( partial && !q.results().empty() )
            partial = false;
        if ( !partial )
            m_tags->update( q );
        fillFilter(
            m, partial ? d.xap.partialQuery( query ).results() : q.results() );
    }
};

struct DownloadFailedException : AptException
{
    DownloadFailedException( std::string ctx ) : AptException( ctx ) {}
};

class SourcesWidget : public KVBox
{
    Q_OBJECT
protected:
    QLabel *m_label;
    QPushButton *m_update;
    QPushButton *m_sourcesEditorLauncher;
public:
    SourcesWidget( QWidget *p = 0 )
        : KVBox( p )
    {
        setMargin( 10 );
        layout()->setSpacing( 3 );

        new QWidget(); // spacer

        // TODO if software-properties-kde is installed, offer to run it?
        // TODO implement apt-get update at least, it's not that hard
        if (QFile::exists("/usr/bin/software-properties-kde")) {
            KHBox* box = new KHBox( this );
            new QWidget( box );
            m_sourcesEditorLauncher = new QPushButton( box );
            m_sourcesEditorLauncher->setText( i18n( "Edit Software Sources" ) );
            connect( m_sourcesEditorLauncher, SIGNAL( clicked() ),
                    this, SLOT( runSourcesEditor() ) );
            new QWidget( box );
        } else {
            m_label = new QLabel( this );
            m_label->setText(
                i18n( "The editor for package sources is <b>not yet implemented</b>."
                    " Please use the <b>Software Properties</b> tool in the"
                    " meantime. (You need to close Adept first and re-start it"
                    " after setting up your sources, though. To refresh the list"
                    " of available software, click the button below." ) );
            m_label->setWordWrap( true );
            m_label->setAlignment( Qt::AlignCenter );
        }


        m_update = new QPushButton( this );
        m_update->setText( i18n( "Fetch current package lists" ) );
        connect( m_update, SIGNAL( clicked() ),
                 this, SIGNAL( fetchUpdates() ) );

        new QWidget(); // spacer
    }

    QPushButton *updateButton() {
        return m_update;
    }

public Q_SLOTS:

    void runSourcesEditor() {
        KProcess* proc = new KProcess(this);
        QStringList arguments;
        int winID = effectiveWinId();
        arguments << "software-properties-kde" << "--dont-update" << "--attach" << QString::number(winID);
        proc->setProgram(arguments);
        find(winID)->setEnabled(false);
        proc->start();
        connect( proc, SIGNAL( finished(int, QProcess::ExitStatus) ),
                 this, SLOT( sourcesEditorFinished(int) ) );
    }

    void sourcesEditorFinished(int reload) {
        find(effectiveWinId())->setEnabled(true);
        if (reload == 1) {
            emit fetchUpdates();
        }
    }

Q_SIGNALS:
    void fetchUpdates();
};

class AdeptManager : public KXmlGuiWindow
{
    Q_OBJECT

    PackageData d;

    TokenModel *m_searchModel;
    QStackedWidget *m_stack;

    KVBox *m_transient;
    QLabel *m_transientLabel;
    QPushButton *m_transientOk;

    Sidebar *m_sidebar;
    KHBox *m_search;
    SearchWidget *m_searchWidget;
    PackageListView *m_searchList;

    PreviewWidget *m_preview;
    
    ApplicationsWidget *m_apps;
    SourcesWidget *m_sources;
    PackageDetails *m_details;
    ProgressWidget *m_progress;

    QList< KAction * > m_sidebarActs;
    KAction *m_commitAct, *m_updateAct;

    int m_interruptedWidget;
    bool m_actionRunning;

    enum OperationMode { Updater, Installer, Manager };
    OperationMode m_opMode;

    struct ActionRunning {
        AdeptManager *m;
        void setActionsEnabled( bool w ) {
            for ( QList< KAction * >::iterator i = m->m_sidebarActs.begin();
                  i != m->m_sidebarActs.end(); ++ i )
            {
                (*i)->setEnabled( w );
            }
            m->m_commitAct->setEnabled( w ? m->d.db.writeable() : false );
            m->m_updateAct->setEnabled( w ? m->d.db.writeable() : false );
        }

        ActionRunning( AdeptManager *_m ) : m( _m ) {
            if ( m->m_actionRunning )
                throw wexcept::Consistency(
                    "Internal Error: Operation already in progress." );
            m->m_actionRunning = true;
            m->m_sidebar->setSidebarEnabled( false );
            setActionsEnabled( false );
        }

        ~ActionRunning() {
            m->m_actionRunning = false;
            m->m_sidebar->setSidebarEnabled( true );
            setActionsEnabled( true );
        }
    };

public:

    void transient( QString txt, bool justify = true ) {
        m_transientLabel->setAlignment(
            Qt::AlignVCenter |
            ( justify ? Qt::AlignJustify : Qt::AlignHCenter ) );
        m_stack->setCurrentWidget( m_transient );
        /* Recommend a safer graphical sudo frontend such as kdesudo or
           gksu to prevent beginners who know how to sorta use the console
           from frying the permissions of various vital files in their /home.
        */
        if (QFile::exists("/usr/bin/kdesudo")) {
            txt.replace("sudo", "kdesudo");
        } else if (QFile::exists("/usr/bin/gksudo")) {
            txt.replace("sudo", "gksudo");
        }
        m_transientLabel->setText( txt );
    }

    // FIXME tell the user something maybe?
    virtual bool queryClose() {
        if ( ProcessEvents::busy() ) {
            QTimer::singleShot( 1000, this, SLOT( close() ) );
            return false;
        }
        return true;
    }

    void forceDatabases()
    {
        ept::Token t( "adept" ); // ...
        d.pkgs.get< package::State >( t );
    }

    void setupItem( QWidget *w, KIcon i, QString txt, const char *act, const char *slot )
    {
        m_sidebar->addItem( w, i, txt );
        KAction *action = actionCollection()->addAction( act );
        action->setText( txt );
        action->setIcon( i );
        connect( action, SIGNAL( triggered() ), this, slot );
        m_sidebarActs.append( action );
    }

    void setupManager() {
        m_opMode = Manager;
        setupItem( m_sources, KIcon( "applications-internet" ),
                   i18n( "Sources" ), "goto-sources", SLOT( gotoSources() ) );
        setupItem( m_search, KIcon( "edit-find" ), i18n( "Search" ),
                   "goto-search", SLOT( gotoSearch() ) );
        setupItem( m_details, KIcon( "edit-copy" ), i18n( "Details" ),
                   "goto-details", SLOT( gotoDetails() ) );
        setupItem( m_preview, KIcon( "dialog-ok-apply" ), i18n( "Changes" ),
                   "goto-preview", SLOT( gotoPreview() ) );
        setupItem( m_progress, KIcon( "system-run" ), i18n( "Progress" ),
                   "goto-progress", SLOT( gotoProgress() ) );
    }


    void setupUpdater() {
        m_opMode = Updater;
        m_sidebar->addItem( m_sources, KIcon( "applications-internet" ),
                            "Fetch Lists" );
        m_sidebar->addItem( m_preview, KIcon( "dialog-ok-apply" ), "Upgrade" );
        m_sidebar->addItem( m_progress, KIcon( "system-run" ), "Progress" );
        m_preview->setEmptyText(
            i18n( "There are <b>no updates</b> available for your system. "
                  "You may want to fetch list of updates if you think "
                  "that there may be updates available that I do not know "
                  "about." ) );
        m_preview->upgradeButton()->hide();
        m_preview->revertButton()->hide();
    }

    void setupInstaller() {
        m_opMode = Installer;

        m_apps = new ApplicationsWidget( d );
        connect( m_apps, SIGNAL( refreshSig() ),
                 this, SLOT( refresh() ) );
        m_sidebar->addItem( m_sources, KIcon( "applications-internet" ),
                            "Sources" );
        m_sidebar->addItem( m_apps, KIcon( "edit-find" ), "Browse" );
        m_sidebar->addItem( m_preview, KIcon( "dialog-ok-apply" ), "Preview" );
        m_sidebar->addItem( m_progress, KIcon( "system-run" ), "Progress" );
    }

    QWidget *defaultWidget() {
        if ( m_opMode == Updater )
            return m_preview;
        if ( m_opMode == Installer )
            return m_apps;
        return m_search;
    }

    AdeptManager( KCmdLineArgs *a, QWidget *p = 0 )
        : KXmlGuiWindow( p ),
          m_actionRunning( false )
    {
        QApplication::setQuitOnLastWindowClosed( true );
        new PkgSystem( d.db );

        m_stack = new QStackedWidget( this );

        m_transient = new KVBox( m_stack );
        m_transient->setMargin( 10 );
        m_stack->addWidget( m_transient );

        new QWidget( m_transient );
        m_transientLabel = setupErrorWidget( m_transient );

        m_transientOk = new QPushButton( m_transient );
        m_transientOk->setText( i18n( "Ok" ) );
        connect( m_transientOk, SIGNAL( clicked() ),
                 this, SLOT( transientClose() ) );
        new QWidget( m_transient );

        m_sidebar = new Sidebar( m_stack );
        m_stack->addWidget( m_sidebar );
        m_stack->setCurrentWidget( m_sidebar );

        m_apps = 0;
        m_search = new KHBox();
        m_preview = new PreviewWidget( d );

        m_details = new PackageDetails( d );
        connect( m_details, SIGNAL( changed() ),
                 this, SLOT( refresh() ) );

        m_searchModel = new TokenModel();
        m_searchWidget = new SearchWidget( d, m_search );
        m_searchList = setupList( this, m_search, m_searchModel, d );
        connect( m_searchWidget, SIGNAL( searchChanged() ),
                 this, SLOT( fill() ) );

        m_progress = new ProgressWidget();
        d.db.setProgress( m_progress->progress() );

        m_sources = new SourcesWidget();

        m_commitAct = actionCollection()->addAction( "commit" );
        m_commitAct->setText( i18n( "Apply Changes" ) );
        connect( m_commitAct, SIGNAL( triggered() ), this, SLOT( apply() ) );
        m_updateAct = actionCollection()->addAction( "update" );
        m_updateAct->setText( i18n( "Fetch package list" ) );
        connect( m_updateAct, SIGNAL( triggered() ), this, SLOT( update() ) );

        if ( a->count() > 0 && a->arg( 0 ) == "updater" ) {
            setupUpdater();
        } else if ( a->count() > 0 && a->arg( 0 ) == "installer" ) {
            setupInstaller();
        } else {
            setupManager();
        }

        m_sidebar->setCurrentWidget( defaultWidget() );
        setCentralWidget( m_stack );
        m_search->setFocusProxy( m_searchWidget );
        m_search->setFocus( Qt::OtherFocusReason );

        QTimer::singleShot( 0, this, SLOT( fill() ) );

        connectList( m_searchList );
        connectList( m_preview->list() );

        // set up search-as-you-type

        connect( m_progress->progress(), SIGNAL( activated() ),
                 this, SLOT( progressStart() ) );
        connect( m_progress->progress(), SIGNAL( done() ),
                 this, SLOT( progressDone() ) );

        connect( m_sources, SIGNAL( fetchUpdates() ),
                 this, SLOT( update() ) );

        connect( m_preview, SIGNAL( upgrade() ), this, SLOT( upgrade() ) );
        connect( m_preview, SIGNAL( apply() ), this, SLOT( apply() ) );
        // Update the package list after the package actions have actually been taken
        connect( m_preview, SIGNAL( revert() ), this, SLOT( revert() ) );

        actionCollection()->addAction(
            KStandardAction::Quit, QApplication::instance(), SLOT( quit() ) );

        createGUI();
        QTimer::singleShot( 0, this, SLOT( delayed() ) );
    }

    static bool isInteresting( ept::Token, PackageState s )
    {
        return s.upgradable() || s.held() || s.modify();
    }

    void connectList( PackageListView *l ) {
        connect( l, SIGNAL( selected(ept::Token) ),
                 m_details, SLOT(setPackage(ept::Token)));
        connect( l->itemDelegate(),
                 SIGNAL( detailsRequested( ept::Token ) ),
                 this, SLOT( gotoDetails() ) );
        connect( l->itemDelegate(),
                 SIGNAL( detailsRequested( ept::Token ) ),
                 m_details, SLOT( setPackage( ept::Token ) ) );
        connect( l->itemDelegate(), SIGNAL( changed() ),
                 this, SLOT( refresh() ) );
    }

public slots:
    void gotoSources() { m_sidebar->setCurrentWidget( m_sources ); }
    void gotoSearch() { m_sidebar->setCurrentWidget( m_search ); }
    void gotoDetails() { m_sidebar->setCurrentWidget( m_details ); }
    void gotoPreview() { m_sidebar->setCurrentWidget( m_preview ); }
    void gotoProgress() { m_sidebar->setCurrentWidget( m_progress ); }

    void progressStart() {
        m_interruptedWidget = m_sidebar->currentIndex();
        m_sidebar->setCurrentWidget( m_progress );
    }

    void progressDone() {
        m_sidebar->setCurrentIndex( m_interruptedWidget );
    }

    void delayed() {
        checkAptDatabase();
        revert();
        if ( m_opMode == Updater )
            upgrade();
        refresh();
        if ( m_apps )
            m_apps->delayed();
        m_details->setPackage( ept::Token( "adept" ) );
    }

    void checkAptDatabase() {
        bool updateXapian = true;
        disconnect( this, SIGNAL( transientClosed() ),
                    this, SLOT( checkAptDatabase() ) );
        d.invalidate();
        try {
            d.db.cache(); // this here forces cacheOpen()
        } catch ( RecoveryNeededException &e ) {
            connect( this, SIGNAL( transientClosed() ),
                     this, SLOT( runRecovery() ) );
            transient(
                i18n( "It appears that dpkg has been interrupted and there"
                      " are some operations that need to be finished before"
                      " normal operation can be resumed. If you say OK here,"
                      " I can try to replay the failed operations. If this"
                      " fails, it is advisable that you investigate the"
                      " situation manually, using <b>dpkg --configure -a</b>"
                      " as root, in a terminal. If you prefer that Adept"
                      " does not attempt recovery, you can safely close the"
                      " program now." ) );
        } catch ( LockingFailureException &e ) {
            bool noperm = e.code() == EACCES || e.code() == EPERM;
            if ( noperm )
                updateXapian = false;
            transient(
                i18n( "Could not obtain a write lock on the cache, "
                      " falling back to <b>read-only mode</b>. "
                      " You won't be able to install, remove or upgrade"
                      " packages. However, you can still search in"
                      " the package database and browse packages.<br> %1",
                      noperm ?
                      i18n(
                          "You apparently do not have sufficient permissions to"
                          " install or remove programs. You may need to run this"
                          " program as user <b>root</b> (or through <b>sudo</b>)"
                          " to gain write access. " ) :
                      i18n( 
                          "It appears that another process is running, which"
                          " holds the write lock on the database. You first need"
                          " to close that program and then restart Adept to gain"
                          " write access." ) ) );
            assert( !d.db.writeable() );
        } catch ( std::exception &e ) {
            connect( this, SIGNAL( transientClosed() ),
                     QApplication::instance(), SLOT( quit() ) );
            transient(
                i18n( "Something broke. Adept will exit when you click OK. "
                      "Please recover manually. "
                      "<br><br><b>The exception says:</b> %1",
                      describeExcept( e ) ) );
        }
        try {
            if ( updateXapian ) {
                d.xap.updateLeniently( d.db, m_progress->progress() );
            }
        } catch ( std::exception &e ) {
            transient(
                i18n( "Updating Xapian index failed. If it does not exist, it"
                      " will be impossible to search for packages. If it is out"
                      " of date, package search might be inaccurate.<br>"
                      " <b>The error is:</b> %1", describeExcept( e ) ) );
        }

        putenv( const_cast< char * >( "APT_LISTCHANGES_FRONTEND=xterm-pager" ) );
        putenv( const_cast< char * >( "APT_LISTBUGS_FRONTEND=none" ) );
        forceDatabases(); // FIXME work around broken laziness
        m_progress->progress()->Done();

        refresh();
    }

    void runRecovery() {
        ActionRunning act( this );
        disconnect( this, SIGNAL( transientClosed() ),
                    this, SLOT( runRecovery() ) );
        DpkgGui *dpkg = new DpkgGui( d.pkgs );
        dpkg->setDebconf( m_progress->debconf() );
        m_progress->commit()->setPM( dpkg );
        m_progress->commit()->setStatus( 0, i18n( "Starting recovery..." ) );
        m_progress->raiseCommit();
        m_sidebar->setCurrentWidget( m_progress );
        try {
            dpkg->recover();
            connect( this, SIGNAL( transientClosed() ),
                     this, SLOT( checkAptDatabase() ) );
            transient(
                i18n( "It seems that the recovery has been successful. We will"
                      " now try to re-open the cache..." ), false );
        } catch ( std::exception &e ) {
            connect( this, SIGNAL( transientClosed() ),
                     QApplication::instance(), SLOT( quit() ) );
            transient(
                i18n( "It seems that the recovery has failed. Please"
                      " fix manually (try <b>dpkg --configure -a</b>"
                      " and/or <b>apt-get -f install</b>) in terminal..."
                      " <br><br><b>The error was:</b> %1",
                      describeExcept( e ) ) );
        }
    }

    void transientClose() {
        m_stack->setCurrentWidget( m_sidebar );
        emit transientClosed();
    }

    void refresh() {
        m_searchList->viewport()->repaint();
        dynamic_cast< PackageDelegate *> (
            m_searchList->itemDelegate() )->refresh();
        m_preview->refresh();
        m_details->refresh();
        fillPreview();
        bool enabled = !d.pkgs.changedList().empty();
        m_preview->applyButton()->setEnabled( enabled && d.db.writeable() );
        m_commitAct->setEnabled( enabled && d.db.writeable() );
        m_preview->revertButton()->setEnabled( enabled );
        m_preview->upgradeButton()->setEnabled(
            d.actl.empty() ||
            d.actl.latest().type() != package::Action::SystemUpgrade );
        m_updateAct->setEnabled( d.db.writeable() );
    }

    void actionListChanged() {
        d.pkgs.revertStates();
        d.actl.replay( d.pkgs );
        refresh();
    }

    void revert() {
        d.actl.clear();
        m_preview->setEmptyText(
            i18n( "You have not asked for any changes." ) );
        actionListChanged();
    }

    void upgrade() {
        d.actl.add( package::Action(
                      ept::Token(), package::Action::SystemUpgrade ) );
        m_preview->setEmptyText(
            i18n( "Sorry, there are no packages that you could upgrade at this"
                  " time. If you think otherwise, you might try to fetch"
                  " updated package lists, on the \"Sources\" tab. Moreover,"
                  " there might be upgradable packages that you are holding"
                  " -- try searching for them." ) );
        actionListChanged();
    }

    void update() {
        pkgSourceList SList;
        ActionRunning act( this );

        if(!SList.ReadMainList())
            abort();

        m_sidebar->setCurrentWidget( m_progress );

        FileFd Lock;
        Lock.Fd(GetLock(_config->FindDir("Dir::State::Lists") + "lock"));
        // TODO check for failure

        std::auto_ptr< pkgAcquire > fetcher(
            new pkgAcquire( m_progress->download() ) );
        SList.GetIndexes( fetcher.get() );

        pkgAcquire::RunResult res = runFetcher( fetcher.get() );

       if( res == pkgAcquire::Failed )
           abort();

       if( res == pkgAcquire::Cancelled ) {
           m_progress->updateCancelled();
           return;
       }

        bool Failed = false;
        string FailedURI;
        for (pkgAcquire::ItemIterator I = fetcher->ItemsBegin();
             I != fetcher->ItemsEnd(); I++)
        {
            if ((*I)->Status == pkgAcquire::Item::StatDone)
                continue;

            (*I)->Finished();

            if((*I)->ErrorText.empty())
                FailedURI += (*I)->DescURI() + "\n";
            else
                FailedURI += (*I)->DescURI() + ": " + (*I)->ErrorText + "\n";
            Failed = true;
        }

        // Clean out any old list files
        if (_config->FindB("APT::Get::List-Cleanup",true) == true)
        {
            if ( fetcher->Clean(_config->FindDir("Dir::State::lists")) == false ||
                 fetcher->Clean(_config->FindDir("Dir::State::lists")
                                  + "partial/") == false )
            {
                // return false;
            }
        }

        m_progress->done();
        checkAptDatabase(); // invalidates everything as well
        if ( m_opMode == Updater ) {
            m_interruptedWidget = 1; // FIXME
            m_sidebar->setCurrentWidget( m_preview );
            upgrade();
        }
    }

    void apply() {
        try {
            doApply();
        } catch ( DownloadFailedException &e ) {
            m_progress->error(
                i18n( "<b>Download failed.</b><br> The list of errors is"
                      " attached. The recovery for this case is currently not"
                      " implemented. If you see 404 errors, it might be"
                      " useful to try fetching package lists (see the"
                      " Sources tab) and retrying the operation.<br><br>"
                      "<b>The error was:</b> %1",
                      describeExcept( e ) ) );
        } catch ( std::exception &e ) {
            connect( this, SIGNAL( transientClosed() ),
                     this, SLOT( runRecovery() ) );
            transient(
                i18n( "<b>Commit failed.</b><br>"
                      "To continue, hit ok and we will try"
                      " to recover. If you close the application now, we will"
                      " not do anything and you may try to resolve the problem"
                      " manually.<br>"
                      " (If you suspect this is a bug in Adept, please also"
                      " provide the following exception description in the"
                      " report).<br><br> <b>The error was:</b> %1",
                      describeExcept( e ) ) );
        }
    }

    pkgAcquire::RunResult runFetcher( pkgAcquire *fetcher )
    {
#ifndef RPM
        return fetcher->Run( 50000 );
#else
        return fetcher->Run();
#endif
    }

    pkgPackageManager::OrderResult
    runPm( pkgPackageManager *pm, PkgSystem::TemporaryUnlock &unlock )
    {
        pkgPackageManager::OrderResult res;
        try {
#ifndef RPM
            res = pm->DoInstall(-1);
#else
            res = pm->DoInstall();
#endif
        } catch ( ... ) {
            // we need to give up on trying to reacquire the lock, recovery
            // will run anyway
            unlock.dropLock();
            throw;
        }
        return res;
    }

    void doApply() {
        bool invalidated = false;
        ActionRunning act( this );
        pkgPackageManager *_pm = _system->CreatePM( &d.db.state() );
        std::auto_ptr< pkgAcquire > fetcher(
            new pkgAcquire( m_progress->download() ) );
        pkgSourceList SList;

        std::auto_ptr< GuiDPkgPM >
            pm( dynamic_cast< GuiDPkgPM * >( _pm ) );
        assert( pm.get() );
        pm->harness().setDebconf( m_progress->debconf() );

        m_progress->commit()->setPM( &pm->harness() );
        m_progress->commit()->setStatus( 0, i18n( "Initializing..." ) );
        m_sidebar->setCurrentWidget( m_progress );

        if(!SList.ReadMainList())
            abort();

        std::auto_ptr< pkgRecords > r( new pkgRecords( d.db.cache() ) );

        while (1) {

            // TODO we need to check _error here, possibly (as well as
            // in some other places?)
            fetcher->Shutdown(); // reload
            if ( !pm->GetArchives( fetcher.get(), &SList, r.get() ) )
                abort();

            {
                pkgAcquire::RunResult res = runFetcher( fetcher.get() );
                
                if ( res == pkgAcquire::Cancelled ) {
                    m_progress->downloadCancelled();
                    break;
                }

                bool failed = false;
                for ( pkgAcquire::ItemIterator I = fetcher->ItemsBegin();
                      I != fetcher->ItemsEnd(); I++ ) {
                    if ((*I)->Status == pkgAcquire::Item::StatDone &&
                        (*I)->Complete == true)
                    continue; // downloaded just fine
                    failed = true;
                    _error->Error( "%s: %s",
                                   (*I)->DescURI().c_str(),
                                   (*I)->ErrorText.c_str() );
                }
                if ( failed || res == pkgAcquire::Failed ) {
                    throw DownloadFailedException( "Package download failed" );
                }
            }

            {
                pkgPackageManager::OrderResult res;
                {
                    PkgSystem::TemporaryUnlock unlock( _system );
                    res = runPm( pm.get(), unlock );
                }
                invalidated = true;

                if (res == pkgPackageManager::Completed) {
                    m_progress->done();
                    break;
                } else if (res == pkgPackageManager::Failed) {
                    throw AptException( "Commit failed." );
                    break;
                }
            }

        }

        if ( invalidated ) {
            checkAptDatabase();
            d.actl.prune( d.pkgs );
            actionListChanged();
        }
    }

    // TODO in both fill methods, we need to prevent db.invalidate()
    // from happening (we will get crashes otherwise)
    // ... OK, probably not crashes in the fill() case, but still...

    void fillPreview() {
        if ( m_opMode == Updater ) {
            m_preview->fill(
                d.pkgs.propertyFilter< package::State >(
                    isInteresting ) );
        } else {
            m_preview->fill( d.pkgs.changedList() );
        }
    }

    void fill() {
        try {
            m_searchList->closeAllEditors(); // FIXME, maybe needlessly strict?
            m_searchWidget->fill( m_searchModel );
        } catch ( std::exception &e ) {
            transient(
                i18n( "Error executing search."
                      "<br><br><b>The exception says:</b> %1",
                      describeExcept( e ) ) );
        } catch ( Xapian::Error &e ) {
            transient(
                i18n( "Error executing search."
                      "<br><br><b>The exception says:</b> %1",
                      describeExcept( e ) ) );
        }
    }

Q_SIGNALS:
    void transientClosed();
};

#endif
